// $URL: http://subversion:8080/svn/gsc/trunk/drivers/LINUX/SIO4%20and%20SIO8/SIO4_Linux_1.x.x_GSC_DN/driver/close.c $
// $Rev: 53095 $
// $Date: 2023-06-13 10:42:41 -0500 (Tue, 13 Jun 2023) $

// SIO4: Device Driver: source file

#include "main.h"



/******************************************************************************
*
*	Function:	close_channel
*
*	Purpose:
*
*		Perform cleanup as a device is being closed. THE CHANNEL SEMAPHORE MUST
*		BE LOCKED WHEN THIS SERVICE IS CALLED.
*
*	Arguments.
*
*		chan	The structure to cleanup.
*
*	Returned:
*
*		None.
*
******************************************************************************/

void close_channel(chan_data_t* chan)
{
	channel_reset_ioctl(chan, NULL);
	dev_io_close(&chan->rx);
	dev_io_close(&chan->tx);
	dev_irq_close(chan->dev);
	osc_close(chan);
	mp_close(chan);
	chan->in_use	= 0;
}



/******************************************************************************
*
*	Function:	os_close
*
*	Purpose:
*
*		Implement the device close functionality.
*
*	Arguments:
*
*		inode	The inode structure used to access the device.
*
*		filp	The file pointer.
*
*	Returned:
*
*		0		All went well.
*		< 0		An appropriate error status.
*
******************************************************************************/

s32 os_close(struct inode* inode, struct file* filp)
{
	chan_data_t*	chan;
	dev_data_t*		dev;
	s32				fw_type;
	u8*				pu8Addr;
	s32				ret;

	for (;;)	// A convenience loop.
	{
		ret	= dev_lock_inode(inode, 0, &chan);

		if (ret)
			break;

		fasync_helper(-1, filp, 0, &chan->irq.fasync);
		filp->private_data	= NULL;

		close_channel(chan);

		// Cleanup the Z16C30 resources.
		ret	= fw_type_config_get(chan, &fw_type);

		if ((ret == 0) && (fw_type == SIO4_FW_TYPE_Z16C30))
		{
			// This is a ZILOG device.
			dev		= chan->dev;
			pu8Addr = (u8*) dev->vaddr.gsc_regs;

			// clear the USC interrupt control register
			os_reg_mem_tx_u8(dev, (VADDR_T) (pu8Addr + ZILOG_INT_CTRL_LOW_REG(chan)), 0);
			os_reg_mem_tx_u8(dev, (VADDR_T) (pu8Addr + ZILOG_INT_CTRL_HIGH_REG(chan)), 0);

			// clear the receive interrupt ctl/stat reg
			os_reg_mem_tx_u8(dev, (VADDR_T) (pu8Addr + ZILOG_RX_INTR_CTRL_LOW_REG(chan)), 0);
			os_reg_mem_tx_u8(dev, (VADDR_T) (pu8Addr + ZILOG_RX_CMD_STAT_LOW_REG(chan)), 0xff);

			// clear the transmit interrupt ctl/stat reg
			os_reg_mem_tx_u8(dev, (VADDR_T) (pu8Addr + ZILOG_TX_INTR_CTRL_LOW_REG(chan)), 0);
			os_reg_mem_tx_u8(dev, (VADDR_T) (pu8Addr + ZILOG_TX_CMD_STAT_LOW_REG(chan)), 0xff);

			// clear the status interrupt ctl, misc stat reg
			os_reg_mem_tx_u8(dev, (VADDR_T) (pu8Addr + ZILOG_STAT_INT_CTRL_LOW_REG(chan)), 0);
			os_reg_mem_tx_u8(dev, (VADDR_T) (pu8Addr + ZILOG_MISC_INT_STAT_LOW_REG(chan)), 0xff);
			os_reg_mem_tx_u8(dev, (VADDR_T) (pu8Addr + ZILOG_MISC_INT_STAT_HIGH_REG(chan)), 0xff);

			// clear the DCCR
			os_reg_mem_tx_u8(dev, (VADDR_T) (pu8Addr + ZILOG_DAISY_CH_CTRL_LOW_REG(chan)), 0xbf);
		}

		// We're done.
		os_module_count_dec();
		os_sem_unlock(&chan->sem);
		break;
	}

	return(ret);
}



