// $URL: http://subversion:8080/svn/gsc/trunk/drivers/LINUX/SIO4%20and%20SIO8/SIO4_Linux_1.x.x_GSC_DN/async/samples/txasync/perform.c $
// $Rev: 53084 $
// $Date: 2023-06-13 10:11:19 -0500 (Tue, 13 Jun 2023) $

// SIO4: Asynchronous Protocol Library: Sample Application: source file

#include "main.h"



// variables ******************************************************************

static	__u8	_xfer_buf[256];		// Must be 256!!!



//*****************************************************************************
static int _fw_type_check(const args_t* args, int* skip)
{
	int		errs	= 0;
	int		i;
	__s32	set;

	gsc_label_index("Firmware Type", args->index);

	set	= SIO4_FW_TYPE_CONFIG_READ;
	i	= ioctl(args->fd, SIO4_FW_TYPE_CONFIG, &set);

	if (i == -1)
	{
		printf("FAIL <---\n");
		errs++;
	}
	else
	{
		switch (set)
		{
			default:

				errs++;
				printf("FAIL <---  (Unknown firmware type: %ld)\n", (long) set);
				break;

			case SIO4_FW_TYPE_CONFIG_SYNC:

				skip[0]	= 1;
				printf("SKIP <---  (SYNC, wrong firmware type)\n");
				break;

			case SIO4_FW_TYPE_CONFIG_Z16C30:

				printf("PASS  (Z16C30)\n");
				break;
		}
	}

	return(errs);
}



//*****************************************************************************
static long _xfer_data(const args_t* args)
{
	int		errs;
	int		i;
	size_t	qty;
	long	xfered;

	for (i = 0; i < sizeof(_xfer_buf); i++)
		_xfer_buf[i]	= i;

	// Write out a single block of data.
	errs	= sio4_write(args->fd, _xfer_buf, sizeof(_xfer_buf), &qty);
	xfered	= (long) qty;

	if (errs)
	{
		printf(	"FAIL <---  (sio4_write())\n");
		xfered	= -1;
	}

	return(xfered);
}



//*****************************************************************************
static int _data_xfer_test(const args_t* args)
{
	char	buf[64]		= "Tx Data";
	time_t	end;
	int		errs		= 0;
	time_t	now;
	long	xfer_qty;
	float	xfer_total	= 0;

	for (;;)	// A convenience loop.
	{
		gsc_label(buf);

		// Make sure the FIFO is empty.
		os_sleep_ms(10);
		errs	+= fifo_reset(args->fd, TX_FIFO, 0);
		errs	+= channel_command_send(args->fd, TX_FIFO_PURGE_CMD, 0);

		// Initialization.

		end	= time(NULL) + args->seconds;

		for (;;)
		{
			xfer_qty	= _xfer_data(args);

			if (xfer_qty < 0)
			{
				errs++;
				break;
			}

			if (xfer_qty == 0)
				os_sleep_ms(1);

			xfer_total	+= xfer_qty;
			now			= time(NULL);

			if (now > end)
				break;
		}

		if (errs == 0)
			printf("PASS");

		printf(	"  (Tx Total %.0f bytes)\n", xfer_total);
		break;
	}

	return(errs);
}



//*****************************************************************************
static int _setup_channel(const args_t* args)
{
	sio4_async_t	async;
	int				errs	= 0;

	gsc_label("Rx Configuration");

	memset(&async, 0, sizeof(async));

	async.cable.dcedte.enable	= 1;
	async.cable.dcedte.dce		= 0;
	async.cable.legacy.config	= SIO4_CABLE_CONFIG_TXLWR_RXUPR;
	async.cable.cts				= SIO4_CTS_CABLE_CONFIG_CTS_IN;
	async.cable.dcd				= SIO4_DCD_CABLE_CONFIG_DCD_IN;
	async.cable.xceiver			= args->mp;

	async.loopback.enable		= 0;
	async.loopback.internal		= 0;

	async.osc.ref_comp			= 0;	// This will be updated.
	async.osc.ref_default		= SIO4_OSC_FREQ_REF_DEFAULT;
	async.osc.ref_got			= 0;	// This will be updated.

	async.rx.ae					= 16;
	async.rx.af					= 16;
	async.rx.baud_comp			= 0;	// This will be updated.
	async.rx.baud_want			= args->baud;
	async.rx.char_size			= SIO4_USC_CHAR_LENGTH_8;
	async.rx.enable				= 0;
	async.rx.encoding			= SIO4_USC_ENCODING_NRZ;
	async.rx.fifo_full_chan		= SIO4_RX_FIFO_FULL_CFG_CHAN_OVER;
	async.rx.fifo_full_glb		= SIO4_RX_FIFO_FULL_CFG_GLB_OVER;
	async.rx.io_mode			= SIO4_IO_MODE_PIO;
	async.rx.io_mode_force		= 0;
	async.rx.parity_enable		= 0;
	async.rx.parity_type		= SIO4_USC_PARTYPE_EVEN;
	async.rx.timeout			= (async.rx.io_mode == SIO4_IO_MODE_PIO) ? 1 : 5;
	async.rx.usc_rxc_config		= SIO4_RXC_USC_CONFIG_IN_PRG_CLK;
	async.rx.usc_rxc_rate		= SIO4_OSC_FREQ_REF_DEFAULT;

	async.tx.ae					= 16;
	async.tx.af					= 16;
	async.tx.baud_comp			= 0;	// This will be updated.
	async.tx.baud_want			= args->baud;
	async.tx.cable_clock		= SIO4_TX_CABLE_CLOCK_CONFIG_PRG_CLK;
	async.tx.cable_data			= SIO4_TX_CABLE_DATA_CONFIG_USC_TXD;
	async.tx.char_size			= SIO4_USC_CHAR_LENGTH_8;
	async.tx.enable				= 1;
	async.tx.encoding			= SIO4_USC_ENCODING_NRZ;
	async.tx.idle_output		= SIO4_USC_TCSR_TXIDLE_DEFAULT;
	async.tx.io_mode			= SIO4_IO_MODE_PIO;
	async.tx.io_mode_force		= 0;
	async.tx.parity_enable		= 0;
	async.tx.parity_type		= SIO4_USC_PARTYPE_EVEN;
	async.tx.shave_bits			= 0;
	async.tx.stop_bits			= SIO4_USC_CMR_AT_STOPBITS_1;
	async.tx.timeout			= (async.tx.io_mode == SIO4_IO_MODE_PIO) ? 1 : 5;
	async.tx.usc_txc_config		= SIO4_TXC_USC_CONFIG_IN_PRG_CLK;
	async.tx.usc_txc_rate		= SIO4_OSC_FREQ_REF_DEFAULT;

	if (args->pc)
	{
		async.cable.dcedte.enable	= 1;
		async.cable.dcedte.dce		= 1;
		async.cable.xceiver			= SIO4_MP_PROT_RS_232;

		async.rx.baud_want			= 19200;
		async.rx.char_size			= SIO4_USC_CHAR_LENGTH_8;
		async.rx.parity_enable		= 0;

		async.tx.baud_want			= 19200;
		async.tx.char_size			= SIO4_USC_CHAR_LENGTH_8;
		async.tx.parity_enable		= 0;
		async.tx.stop_bits			= SIO4_USC_CMR_AT_STOPBITS_1;
	}

	errs	= sio4_async_config(args->fd, &async, errs, 0);

	if (errs)
		;
	else if (args->pc)
		printf("PASS  (PC: DCE, 19200 baud, 8 data bits, No Parity, 1 stop bit, NRZ)\n");
	else
		printf("PASS  (%ld baud)\n", (long) async.tx.baud_comp);

	return(errs);
}



//*****************************************************************************
static int _io_test(const args_t* args)
{
	int	errs	= 0;

	gsc_label("Setup");
	printf("\n");
	gsc_label_level_inc();

	if (errs == 0)
		errs	= _setup_channel(args);

	gsc_label_level_dec();
	gsc_label("Data Write");

	if (errs)
		printf("SKIPPED  (setup errors)\n");

	if (errs == 0)
		printf("(%ld Seconds)\n", (long) args->seconds);

	gsc_label_level_inc();

	if (errs == 0)
		errs	= _data_xfer_test(args);

	gsc_label_level_dec();
	return(errs);
}



//*****************************************************************************
int perform_tests(const args_t* args, int* skip)
{
	int	errs;

	errs	= _fw_type_check(args, skip);

	if (errs)
	{
		printf("Test sequence skipped due to pre-test errors.\n");
	}
	else if (skip[0])
	{
		printf("Test sequence skipped due to firmware type.\n");
	}
	else
	{
		errs	= _io_test(args);
	}

	return(errs);
}


