// $URL: http://subversion:8080/svn/gsc/trunk/drivers/LINUX/SIO4%20and%20SIO8/SIO4_Linux_1.x.x_GSC_DN/utils/util_reg.c $
// $Rev: 53094 $
// $Date: 2023-06-13 10:21:23 -0500 (Tue, 13 Jun 2023) $

// SIO4: Utilities: source file

#include "main.h"



// macros *********************************************************************

#define	_GSC_REG(a)							"GSC " #a, SIO4_GSC_##a, 0, 0
#define	_PCI_REG(a)							"PCI " #a, SIO4_PCI_##a, 0, 0
#define	_PLX_REG(a)							"PLX " #a, SIO4_PLX_##a, 0, 0
#define	_USC_REG(a)							"USC " #a, (int) SIO4_USC_##a, 0, 0
#define	WIDTH								22

#define	FIELD_DECODE						SIO4_DECODE

#define	FIELD_SHOW(beg,end,eol,lst,dsc)		\
			gsc_reg_field_show(width+10,WIDTH,value,(beg),(end),(eol),(lst),(dsc))

#define	QUERY_BOARD_RESET(fd, ptr)			feature_test((fd), SIO4_FEATURE_BOARD_RESET,	(ptr), 1)
#define	QUERY_FORM_FACTOR(fd, ptr)			feature_test((fd), SIO4_FEATURE_FORM_FACTOR,	(ptr), 1)
#define	QUERY_LED_MAIN(fd,ptr)				feature_test((fd), SIO4_FEATURE_LED_MAIN,		(ptr), 1)
#define	QUERY_MODEL_BASE(fd,ptr)			feature_test((fd), SIO4_FEATURE_MODEL_BASE,		(ptr), 1)
#define	QUERY_OSC_CHIP(fd, ptr)				feature_test((fd), SIO4_FEATURE_OSC_CHIP,		(ptr), 1)
#define	QUERY_REG_BSR(fd, ptr)				feature_test((fd), SIO4_FEATURE_REG_BSR,		(ptr), 1)
#define	QUERY_REG_CCR(fd, ptr)				feature_test((fd), SIO4_FEATURE_REG_CCR,		(ptr), 1)
#define	QUERY_REG_FCR(fd, ptr)				feature_test((fd), SIO4_FEATURE_REG_FCR,		(ptr), 1)
#define	QUERY_REG_FR(fd, ptr)				feature_test((fd), SIO4_FEATURE_REG_FR,			(ptr), 1)
#define	QUERY_REG_FSR(fd, ptr)				feature_test((fd), SIO4_FEATURE_REG_FSR,		(ptr), 1)
#define	QUERY_REG_FTR(fd, ptr)				feature_test((fd), SIO4_FEATURE_REG_FTR,		(ptr), 1)
#define	QUERY_REG_GPIOSR(fd, ptr)			feature_test((fd), SIO4_FEATURE_REG_GPIOSR,		(ptr), 1)
#define	QUERY_REG_IELR(fd, ptr)				feature_test((fd), SIO4_FEATURE_REG_IELR,		(ptr), 1)
#define	QUERY_REG_IHLR(fd, ptr)				feature_test((fd), SIO4_FEATURE_REG_IHLR,		(ptr), 1)
#define	QUERY_REG_IOCR(fd, ptr)				feature_test((fd), SIO4_FEATURE_REG_IOCR,		(ptr), 1)
#define	QUERY_REG_PSRCR(fd, ptr)			feature_test((fd), SIO4_FEATURE_REG_PSRCR,		(ptr), 1)
#define	QUERY_REG_PSTSR(fd, ptr)			feature_test((fd), SIO4_FEATURE_REG_PSTSR,		(ptr), 1)
#define	QUERY_REG_RCR(fd, ptr)				feature_test((fd), SIO4_FEATURE_REG_RCR,		(ptr), 1)
#define	QUERY_REG_SBR(fd, ptr)				feature_test((fd), SIO4_FEATURE_REG_SBR,		(ptr), 1)
#define	QUERY_REG_TCR(fd, ptr)				feature_test((fd), SIO4_FEATURE_REG_TCR,		(ptr), 1)
#define	QUERY_REG_TSR(fd, ptr)				feature_test((fd), SIO4_FEATURE_REG_TSR,		(ptr), 1)
#define	QUERY_SIO4_TYPE(fd,ptr)				feature_test((fd), SIO4_FEATURE_SIO4_TYPE,		(ptr), 1)



// prototypes *****************************************************************

static int	_is_tx_mode_async(void);



// variables ******************************************************************

static gsc_reg_def_t	_pci[25]	=
{
	{ _PCI_REG(VID),	0, NULL,	"Vendor ID Register"						},
	{ _PCI_REG(DID),	0, NULL,	"Device ID Register"						},
	{ _PCI_REG(CR),		0, NULL,	"Command Register"							},
	{ _PCI_REG(SR),		0, NULL,	"Status Register"							},
	{ _PCI_REG(REV),	0, NULL,	"Revision ID Register"						},
	{ _PCI_REG(CCR),	0, NULL,	"Class Code Register"						},
	{ _PCI_REG(CLSR),	0, NULL,	"Cache Line Size Register"					},
	{ _PCI_REG(LTR),	0, NULL,	"Latency Timer Register"					},
	{ _PCI_REG(HTR),	0, NULL,	"Header Type Register"						},
	{ _PCI_REG(BISTR),	0, NULL,	"Built-In Self-Test Register"				},
	{ _PCI_REG(BAR0),	0, NULL,	"Base Address Register 0"					},
	{ _PCI_REG(BAR1),	0, NULL,	"Base Address Register 1"					},
	{ _PCI_REG(BAR2),	0, NULL,	"Base Address Register 2"					},
	{ _PCI_REG(BAR3),	0, NULL,	"Base Address Register 3"					},
	{ _PCI_REG(BAR4),	0, NULL,	"Base Address Register 4"					},
	{ _PCI_REG(BAR5),	0, NULL,	"Base Address Register 4"					},
	{ _PCI_REG(CIS),	0, NULL,	"Cardbus Information Structure Register"	},
	{ _PCI_REG(SVID),	0, NULL,	"Subsystem Vendor ID Register"				},
	{ _PCI_REG(SID),	0, NULL,	"Subsystem ID Register"						},
	{ _PCI_REG(ERBAR),	0, NULL,	"Expansion ROM Base Address Register"		},
	{ _PCI_REG(ILR),	0, NULL,	"Interrupt Line Register"					},
	{ _PCI_REG(IPR),	0, NULL,	"Interrupt Pin Register"					},
	{ _PCI_REG(MGR),	0, NULL,	"Minimum Grant Register"					},
	{ _PCI_REG(MLR),	0, NULL,	"Maximum Latency Register"					},
	{ NULL, 0, 0, 0, 0, NULL, NULL }
};

static gsc_reg_def_t	_plx[44]	=
{
	{ _PLX_REG(LAS0RR),		0, NULL, "Local Address Space 0 Range Register"					},
	{ _PLX_REG(LAS0BA),		0, NULL, "Local Address Space 0 Base Address Register"			},
	{ _PLX_REG(MARBR),		0, NULL, "Mode/Arbitration Register"							},
	{ _PLX_REG(BIGEND),		0, NULL, "Big/Little Endian Descriptor Register"				},
	{ _PLX_REG(EROMRR),		0, NULL, "Expansion ROM Range Register"							},
	{ _PLX_REG(EROMBA),		0, NULL, "Expansion ROM Base Address Register"					},
	{ _PLX_REG(LBRD0),		0, NULL, "Local BUS Region Descriptor Register"					},
	{ _PLX_REG(DMRR),		0, NULL, "Direct Master Range Register"							},
	{ _PLX_REG(DMLBAM),		0, NULL, "Direct Master Local Base Address Memory Register"		},
	{ _PLX_REG(DMLBAI),		0, NULL, "Direct Master Local Base Address I/O Register"		},
	{ _PLX_REG(DMPBAM),		0, NULL, "Direct Master PCI Base Address Memory Register"		},
	{ _PLX_REG(DMCFGA),		0, NULL, "Direct Master Config Base Address Memory Register"	},
	{ _PLX_REG(LAS1RR),		0, NULL, "Local Address Space 1 Range Register"					},
	{ _PLX_REG(LAS1BA),		0, NULL, "Local Address Space 1 Base Address Register"			},
	{ _PLX_REG(LBRD1),		0, NULL, "Local Address Space 1 Region Descriptor Register"		},
	{ _PLX_REG(MBOX0),		0, NULL, "Mailbox Register 0"									},
	{ _PLX_REG(MBOX1),		0, NULL, "Mailbox Register 1"									},
	{ _PLX_REG(MBOX2),		0, NULL, "Mailbox Register 2"									},
	{ _PLX_REG(MBOX3),		0, NULL, "Mailbox Register 3"									},
	{ _PLX_REG(MBOX4),		0, NULL, "Mailbox Register 4"									},
	{ _PLX_REG(MBOX5),		0, NULL, "Mailbox Register 5"									},
	{ _PLX_REG(MBOX6),		0, NULL, "Mailbox Register 6"									},
	{ _PLX_REG(MBOX7),		0, NULL, "Mailbox Register 7"									},
	{ _PLX_REG(P2LDBELL),	0, NULL, "PCI-to-Local Doorbell Register"						},
	{ _PLX_REG(L2PDBELL),	0, NULL, "Local-to-PCI Doorbell Register"						},
	{ _PLX_REG(INTCSR),		0, NULL, "Interrupt Control/Status Register"					},
	{ _PLX_REG(CNTRL),		0, NULL, "Control Register"										},
	{ _PLX_REG(PCIHIDR),	0, NULL, "PCI Perminent Configuration ID Register"				},
	{ _PLX_REG(PCIHREV),	0, NULL, "PCI Perminent Revision ID Register"					},
	{ _PLX_REG(DMAMODE0),	0, NULL, "DMA Channel 0 Mode Register"							},
	{ _PLX_REG(DMAPADR0),	0, NULL, "DMA Channel 0 PCI Address Register"					},
	{ _PLX_REG(DMALADR0),	0, NULL, "DMA Channel 0 Local Address Register"					},
	{ _PLX_REG(DMASIZ0),	0, NULL, "DMA Channel 0 Size Register"							},
	{ _PLX_REG(DMADPR0),	0, NULL, "DMA Channel 0 Descriptor Pointer Register"			},
	{ _PLX_REG(DMACSR0),	0, NULL, "DMA Channel 0 Command/Status Register"				},
	{ _PLX_REG(DMAMODE1),	0, NULL, "DMA Channel 1 Mode Register"							},
	{ _PLX_REG(DMAPADR1),	0, NULL, "DMA Channel 1 PCI Address Register"					},
	{ _PLX_REG(DMALADR1),	0, NULL, "DMA Channel 1 Local Address Register"					},
	{ _PLX_REG(DMASIZ1),	0, NULL, "DMA Channel 1 Size Register"							},
	{ _PLX_REG(DMADPR1),	0, NULL, "DMA Channel 1 Descriptor Register"					},
	{ _PLX_REG(DMACSR1),	0, NULL, "DMA Channel 1 Command/Status Register"				},
	{ _PLX_REG(DMAARB),		0, NULL, "DMA Arbitration Register"								},
	{ _PLX_REG(DMATHR),		0, NULL, "DMA Threshold Register"								},
	{ NULL, 0, 0, 0, 0, NULL, NULL }
};



//*****************************************************************************
static const gsc_reg_def_t* _find_reg(u32 reg, const gsc_reg_def_t* list)
{
	const gsc_reg_def_t*	def	= NULL;
	int						i;

	for (i = 0; list[i].name; i++)
	{
		if (reg == list[i].reg)
		{
			def	= &list[i];
			break;
		}
	}

	return(def);
}


//*****************************************************************************
static int _get_fw_type(int fd, s32* fw_type)
{
	int	errs;
	int	ret;
	s32	set		= -1;

	ret		= ioctl(fd, SIO4_FW_TYPE_CONFIG, &set);
	errs	= (ret < 0) ? 1 : 0;

	if (fw_type)
		fw_type[0]	= set;

	return(errs);
}



//*****************************************************************************
static int _is_zilog_fw(int fd)
{
	s32	fw_type;
	int	zilog;

	_get_fw_type(fd, &fw_type);
	zilog	= (fw_type == SIO4_FW_TYPE_CONFIG_Z16C30) ? 1 : 0;
	return(zilog);
}



//*****************************************************************************
static int _gsc_frr(int fd, int supported, u32 value, int width)
{
	static const char*	_form[]	=
	{
		"RESERVED",
		"PCI",
		"PMC",
		"cPCI",
		"PC/104+",
		"PCIe",
		"XMC",
		"RESERVED",
		"RESERVED",
		"RESERVED",
		"RESERVED",
		"RESERVED",
		"RESERVED",
		"RESERVED",
		"RESERVED",
		"RESERVED"
	};

	static const char*	_32bit[]	= { "32-bit",	"64-bit"	};
	static const char*	_33mhz[]	= { "33MHz",	"66MHz"		};
	static const char*	_no[]		= { "No",		"Yes"		};

	int	errs;
	u32	field;
	s32	type;

	// FRR Verification
	// SIO4_MODEL_SIO8BX2		verified 2013/6/24

	errs	= QUERY_SIO4_TYPE(fd, &type);

	if (errs == 0)
	{
		if (type == SIO4_TYPE_SIO4)
		{
			FIELD_SHOW	(31, 16, 1, NULL,	"Reserved"			);

			field	= FIELD_DECODE(value, 14, 8);

			if (field == 0)
				field	= SIO4_FW_TYPE_Z16C30;
		}
		else if ((value & 0x40000000) == 0)
		{
			FIELD_SHOW	(31, 16, 1, NULL,	"Format Unknown"	);

			field	= FIELD_DECODE(value, 15, 8);
		}
		else
		{
			FIELD_SHOW	(31, 31, 1, _no,	"Features Reg"		);
			FIELD_SHOW	(30, 30, 1, _no,	"Spec Compliant"	);
			FIELD_SHOW	(29, 29, 1, _33mhz,	"Bus Speed"			);
			FIELD_SHOW	(28, 28, 1, _32bit,	"Bus Width"			);
			FIELD_SHOW	(27, 24, 1, _form,	"Form Factor"		);
			FIELD_SHOW	(23, 20, 1, NULL,	"Board Type"		);

			FIELD_SHOW	(19, 16, 0, NULL,	"HW Rev"			);
			field	= FIELD_DECODE(value, 19, 16);

			if (field)
				printf("Rev %c\n", (char) ('A' + field - 1));
			else
				printf("NR\n");

			field	= FIELD_DECODE(value, 15, 8);
		}

		FIELD_SHOW		(15,  8, 0, NULL,	"Firmware Type"		);

		if (field == SIO4_FW_TYPE_Z16C30)
			printf("Zilog\n");
		else if (field == SIO4_FW_TYPE_SYNC)
			printf("SYNC\n");
		else
			printf("\n");

		FIELD_SHOW		(7,  0, 0, NULL,	"Firmware Rev"		);
		field	= FIELD_DECODE(value, 7, 0);
		printf("%02lX\n", (long) field);
	}

	return(errs);
}



//*****************************************************************************
static int _gsc_bcr(int fd, int supported, u32 value, int width)
{
	static const char*	_dmdma[]	=
	{
		"Channel 1 Rx FIFO",
		"Channel 3 Rx FIFO",
		"Channel 2 Rx FIFO",
		"Channel 4 Rx FIFO",
		"Channel 1 Tx FIFO",
		"Channel 3 Tx FIFO",
		"Channel 2 Tx FIFO",
		"Channel 4 Tx FIFO"
	};

	static const char*	_enable[]	= { "Enable",	"Disable"		};
	static const char*	_idle[]		= { "Idle",		"Active"		};
	static const char*	_led[]		= { "Off",		"On"			};
	static const char*	_over[]		= { "Overrun",	"Rx Disable"	};
	static const char*	_time[]		= { "Enable",	"Reset"			};
	static const char*	_time_src[]	= { "Internal",	"External"		};

	int	errs		= 0;
	s32	led_main;
	s32	reset;
	s32	type;

	errs	+= QUERY_BOARD_RESET(fd, &reset);
	errs	+= QUERY_LED_MAIN(fd, &led_main);
	errs	+= QUERY_SIO4_TYPE(fd, &type);

	for (;;)	// A convenience loop.
	{
		if (errs)
			break;

		if (type == SIO4_TYPE_SIO8BX2)
		{
			// SIO8BX2 - verified 2013/6/24
			FIELD_SHOW	(31, 31, 1, _idle,		"Board Reset"			);
			FIELD_SHOW	(30, 26, 1, NULL,		"Reserved"				);
			FIELD_SHOW	(25, 25, 1, _led,		"Red LED D1/D6"			);
			FIELD_SHOW	(24, 24, 1, _led,		"Green LED D1/D6"		);
			FIELD_SHOW	(23, 23, 1, _time,		"Timestamp"				);
			FIELD_SHOW	(22, 22, 1, _time_src,	"Timestamp Source"		);
			FIELD_SHOW	(21,  9, 1, NULL,		"Reserved"				);
			FIELD_SHOW	( 8,  8, 1, _over,		"Rx FIFO Full Config"	);
			FIELD_SHOW	( 7,  7, 1, _enable,	"DMDMA 1 SCD"			);
			FIELD_SHOW	( 6,  4, 1, _dmdma,		"DMDMA 1"				);
			FIELD_SHOW	( 3,  3, 1, _enable,	"DMDMA 0 SCD"			);
			FIELD_SHOW	( 2,  0, 1, _dmdma,		"DMDMA 0"				);
			break;
		}

		if (type < SIO4_TYPE_SIO4B)
		{
			// SIO4
			// SIO4A
			// SIO4AR
			// SIO4AHRM
			FIELD_SHOW	(31,  7, 1, NULL,		"Reserved"		);
			FIELD_SHOW	( 6,  4, 1, _dmdma,		"DMDMA 1"		);
			FIELD_SHOW	( 3,  3, 1, NULL,		"Reserved"		);
			FIELD_SHOW	( 2,  0, 1, _dmdma,		"DMDMA 0"		);
			break;
		}

		if (type <= SIO4_TYPE_SIO4BX)
		{
			// SIO4B
			// SIO4BX
			FIELD_SHOW	(31, 31, 1, _idle,		"Board Reset"			);
			FIELD_SHOW	(30,  9, 1, NULL,		"Reserved"				);
			FIELD_SHOW	( 8,  8, 1, _over,		"Rx FIFO Full Config"	);
			FIELD_SHOW	( 7,  7, 1, _enable,	"DMDMA 1 SCD"			);
			FIELD_SHOW	( 6,  4, 1, _dmdma,		"DMDMA 1"				);
			FIELD_SHOW	( 3,  3, 1, _enable,	"DMDMA 0 SCD"			);
			FIELD_SHOW	( 2,  0, 1, _dmdma,		"DMDMA 0"				);
			break;
		}

		if (type == SIO4_TYPE_SIO4BXR)
		{
			// SIO4BXR
			FIELD_SHOW	(31, 31, 1, _idle,		"Board Reset"			);
			FIELD_SHOW	(30, 27, 1, NULL,		"Reserved"				);
			FIELD_SHOW	(26, 26, 1, _led,		"Green LED D3"			);
			FIELD_SHOW	(25, 25, 1, _led,		"Green LED D2"			);
			FIELD_SHOW	(24, 24, 1, _led,		"Green LED D1"			);
			FIELD_SHOW	(23,  9, 1, NULL,		"Reserved"				);
			FIELD_SHOW	( 8,  8, 1, _over,		"Rx FIFO Full Config"	);
			FIELD_SHOW	( 7,  7, 1, _enable,	"DMDMA 1 SCD"			);
			FIELD_SHOW	( 6,  4, 1, _dmdma,		"DMDMA 1"				);
			FIELD_SHOW	( 3,  3, 1, _enable,	"DMDMA 0 SCD"			);
			FIELD_SHOW	( 2,  0, 1, _dmdma,		"DMDMA 0"				);
			break;
		}

		if (type == SIO4_TYPE_SIO8BXS)
		{
			// SIO8BXS
			FIELD_SHOW	(31, 31, 1, _idle,		"Board Reset"			);
			FIELD_SHOW	(30,  9, 1, NULL,		"Reserved"				);
			FIELD_SHOW	( 8,  8, 1, _over,		"Rx FIFO Full Config"	);
			FIELD_SHOW	( 7,  7, 1, _enable,	"DMDMA 1 SCD"			);
			FIELD_SHOW	( 6,  4, 1, _dmdma,		"DMDMA 1"				);
			FIELD_SHOW	( 3,  3, 1, _enable,	"DMDMA 0 SCD"			);
			FIELD_SHOW	( 2,  0, 1, _dmdma,		"DMDMA 0"				);
			break;
		}

		break;
	}

	return(errs);
}



//*****************************************************************************
static int _gsc_bsr(int fd, int supported, u32 value, int width)
{
	static const char*	_size[]		= { "48KLC", "64K", "256K", "RESERVED"	};

	static const char*	_0_on[]		= { "Jumper On",	"Jumper Off"	};
	static const char*	_1_on[]		= { "Jumper Off",	"Jumper On"		};
	static const char*	_chans[]	= { "Channels 1-4",	"Channels 5-8"	};
	static const char*	_fifo_ei[]	= { "External",		"Internal"		};
	static const char*	_fifo_ie[]	= { "Internal",		"External"		};
	static const char*	_fw[]		= { "Standard",		"SYNC"			};
	static const char*	_idle[]		= { "Idle",			"Active"		};
	static const char*	_pcb[]		= { "Standard",		"L3RIO"			};
	static const char*	_subid[]	= { "Subdevice 0",	"Subdevice 1"	};
	static const char*	_tt[]		= { "Standard",		"External"		};

	int	errs	= 0;
	s32	fw;
	int	ret;
	s32	type;
	s32	val;

	for (;;)	// A convenience loop.
	{
		if (supported)
		{
			ret	= QUERY_REG_BSR(fd, &val);
			ret	= (ret < 0) ? 0 : (val ? 1 : 0);
			break;
		}

		errs	+= _get_fw_type(fd, &fw);
		errs	+= QUERY_SIO4_TYPE(fd, &type);
		ret		= errs;

		if (errs)
			break;

		if (type == SIO4_TYPE_SIO8BX2)
		{
			// PCIE4-SIO8BX2 - verified 2013/6/24
			FIELD_SHOW	(31,  9, 1, NULL,		"Reserved"			);
			FIELD_SHOW	( 8,  8, 1, _fw,		"FW Type"			);
			FIELD_SHOW	( 7,  6, 1, NULL,		"Reserved"			);
			FIELD_SHOW	( 5,  4, 1, _size,		"FIFO Size"			);
			FIELD_SHOW	( 3,  3, 1, _0_on,		"Board ID 4"		);
			FIELD_SHOW	( 2,  2, 1, _0_on,		"Board ID 3"		);
			FIELD_SHOW	( 1,  1, 1, _0_on,		"Board ID 2"		);
			FIELD_SHOW	( 0,  0, 1, _subid,		"Board ID 1"		);
			break;
		}

		if (type < SIO4_TYPE_SIO4B)
		{
			// SIO4 - not applicable
			// SIO4A
			// SIO4AR
			// SIO4AHRM
			FIELD_SHOW	(31,  2, 1, NULL,		"Reserved"		);
			FIELD_SHOW	( 1,  1, 1, _0_on,		"Board ID 1"	);
			FIELD_SHOW	( 0,  0, 1, _0_on,		"Board ID 0"	);
			break;
		}

		if (type <= SIO4_TYPE_SIO4BX)
		{
			// SIO4B
			// SIO4BX
			FIELD_SHOW	(31, 31, 1, _idle,		"Board Reset"		);
			FIELD_SHOW	(30, 16, 1, NULL,		"Reserved"			);
			FIELD_SHOW	(15, 15, 1, _fifo_ie,	"Ch4 Rx FIFO Type"	);
			FIELD_SHOW	(14, 14, 1, _fifo_ie,	"Ch4 Tx FIFO Type"	);
			FIELD_SHOW	(13, 13, 1, _fifo_ie,	"Ch3 Rx FIFO Type"	);
			FIELD_SHOW	(12, 12, 1, _fifo_ie,	"Ch3 Tx FIFO Type"	);
			FIELD_SHOW	(11, 11, 1, _fifo_ie,	"Ch2 Rx FIFO Type"	);
			FIELD_SHOW	(10, 10, 1, _fifo_ie,	"Ch2 Tx FIFO Type"	);
			FIELD_SHOW	( 9,  9, 1, _fifo_ie,	"Ch1 Rx FIFO Type"	);
			FIELD_SHOW	( 8,  8, 1, _fifo_ie,	"Ch1 Tx FIFO Type"	);
			FIELD_SHOW	( 7,  2, 1, NULL,		"Reserved"			);
			FIELD_SHOW	( 1,  1, 1, _0_on,		"Board ID 1"		);
			FIELD_SHOW	( 0,  0, 1, _0_on,		"Board ID 0"		);
			break;
		}

		if (type == SIO4_TYPE_SIO4BXR)
		{
			// SIO4BXR
			FIELD_SHOW	(31,  9, 1, NULL,		"Reserved"			);
			FIELD_SHOW	( 8,  8, 1, _fw,		"FW Type"			);
			FIELD_SHOW	( 7,  7, 1, _tt,		"Timetag Option"	);
			FIELD_SHOW	( 6,  6, 1, _pcb,		"PCB Option"		);
			FIELD_SHOW	( 5,  4, 1, _size,		"FIFO Size"			);
			FIELD_SHOW	( 3,  3, 1, _0_on,		"Board ID 4"		);
			FIELD_SHOW	( 2,  2, 1, _0_on,		"Board ID 3"		);
			FIELD_SHOW	( 1,  1, 1, _0_on,		"Board ID 2"		);
			FIELD_SHOW	( 0,  0, 1, _0_on,		"Board ID 1"		);
			break;
		}

		if (type == SIO4_TYPE_SIO8BXS)
		{
			// SIO8BXS
			FIELD_SHOW	(31, 31, 1, _idle,		"Board Reset"		);
			FIELD_SHOW	(30, 16, 1, NULL,		"Reserved"			);
			FIELD_SHOW	(15, 15, 1, _fifo_ei,	"Ch4 Rx FIFO Type"	);
			FIELD_SHOW	(14, 14, 1, _fifo_ei,	"Ch4 Tx FIFO Type"	);
			FIELD_SHOW	(13, 13, 1, _fifo_ei,	"Ch3 Rx FIFO Type"	);
			FIELD_SHOW	(12, 12, 1, _fifo_ei,	"Ch3 Tx FIFO Type"	);
			FIELD_SHOW	(11, 11, 1, _fifo_ei,	"Ch2 Rx FIFO Type"	);
			FIELD_SHOW	(10, 10, 1, _fifo_ei,	"Ch2 Tx FIFO Type"	);
			FIELD_SHOW	( 9,  9, 1, _fifo_ei,	"Ch1 Rx FIFO Type"	);
			FIELD_SHOW	( 8,  8, 1, _fifo_ei,	"Ch1 Tx FIFO Type"	);
			FIELD_SHOW	( 7,  6, 1, NULL,		"Reserved"			);
			FIELD_SHOW	( 5,  5, 1, _1_on,		"Board ID 4"		);
			FIELD_SHOW	( 4,  4, 1, _1_on,		"Board ID 3"		);
			FIELD_SHOW	( 3,  3, 1, _1_on,		"Board ID 2"		);
			FIELD_SHOW	( 2,  2, 1, NULL,		"Reserved"			);
			FIELD_SHOW	( 1,  1, 1, _1_on,		"Board ID 1"		);
			FIELD_SHOW	( 0,  0, 1, _chans,		"Board ID 0"		);
			break;
		}

		break;
	}

	return(ret);
}



//*****************************************************************************
static int _gsc_ccr(int fd, int supported, u32 value, int width)
{
	static const char*	_lower[]	= { "Disable",	"Enable Lower"	};
	static const char*	_upper[]	= { "Disable",	"Enable Upper"	};

	int	errs;
	int	ret		= 0;
	s32	val;

	if (supported)
	{
		errs	= QUERY_REG_CCR(fd, &val);
		ret		= errs ? 0 : (val ? 1 : 0);
	}
	else
	{
		FIELD_SHOW(31, 16, 1, NULL,		"Reserved"		);
		FIELD_SHOW(15, 15, 1, _lower,	"RxClk4LwrEn"	);
		FIELD_SHOW(14, 14, 1, _upper,	"RxClk4UprEn"	);
		FIELD_SHOW(13, 13, 1, _lower,	"TxClk4LwrEn"	);
		FIELD_SHOW(12, 12, 1, _upper,	"TxClk4UprEn"	);
		FIELD_SHOW(11, 11, 1, _lower,	"RxClk3LwrEn"	);
		FIELD_SHOW(10, 10, 1, _upper,	"RxClk3UprEn"	);
		FIELD_SHOW( 9,  9, 1, _lower,	"TxClk3LwrEn"	);
		FIELD_SHOW( 8,  8, 1, _upper,	"TxClk3UprEn"	);
		FIELD_SHOW( 7,  7, 1, _lower,	"RxClk2LwrEn"	);
		FIELD_SHOW( 6,  6, 1, _upper,	"RxClk2UprEn"	);
		FIELD_SHOW( 5,  5, 1, _lower,	"TxClk2LwrEn"	);
		FIELD_SHOW( 4,  4, 1, _upper,	"TxClk2UprEn"	);
		FIELD_SHOW( 3,  3, 1, _lower,	"RxClk1LwrEn"	);
		FIELD_SHOW( 2,  2, 1, _upper,	"RxClk1UprEn"	);
		FIELD_SHOW( 1,  1, 1, _lower,	"TxClk1LwrEn"	);
		FIELD_SHOW( 0,  0, 1, _upper,	"TxClk1UprEn"	);
	}

	return(ret);
}



//*****************************************************************************
static int _gsc_tsr(int fd, int supported, u32 value, int width)
{
	int	errs;
	int	ret		= 0;
	s32	val;

	if (supported)
	{
		errs	= QUERY_REG_TSR(fd, &val);
		ret		= errs ? 0 : (val ? 1 : 0);
	}
	else
	{
		FIELD_SHOW(31, 24, 1, NULL,	"Reserved"	);
		FIELD_SHOW(23,  0, 1, NULL,	"Timestamp"	);
	}

	return(ret);
}



//*****************************************************************************
static int _gsc_tar(int fd, int supported, u32 value, int width)
{
	FIELD_SHOW(31, 16, 1, NULL,	"Almost Full"	);
	FIELD_SHOW(15,  0, 1, NULL,	"Almost Empty"	);
	return(0);
}



//*****************************************************************************
static int _gsc_fdr(int fd, int supported, u32 value, int width)
{
	FIELD_SHOW(31,  8, 1, NULL,	"Reserved"	);
	FIELD_SHOW( 7,  0, 1, NULL,	"Rx Data"	);
	return(0);
}



//*****************************************************************************
static int _gsc_csr(int fd, int supported, u32 value, int width)
{
	static const char*	_disable[]	= { "Disabled",		"Enabled"		};
	static const char*	_dual[]		= { "Normal",		"Dual"			};
	static const char*	_fall[]		= { "Falling Edge",	"Rising Edge"	};
	static const char*	_hi[]		= { "High",			"Low"			};
	static const char*	_idle[]		= { "Idle",			"Active"		};
	static const char*	_int[]		= { "No",			"Internal"		};
	static const char*	_l_rxd[]	= { "Disable",		"Lower RxD"		};
	static const char*	_l_txd[]	= { "Disable",		"Lower TxD"		};
	static const char*	_msb[]		= { "MSB",			"LSB"			};
	static const char*	_no[]		= { "No",			"Yes"			};
	static const char*	_remain[]	= { "No Change",	"Disable"		};
	static const char*	_txclksrc[]	= { "Internal",		"External"		};
	static const char*	_u_rxd[]	= { "Disable",		"Upper RxD"		};
	static const char*	_u_txd[]	= { "Disable",		"Upper TxD"		};
	static const char*	_yes[]		= { "Yes",			"No"			};

	int	errs;
	s32	fw;

	errs	= _get_fw_type(fd, &fw);

	if (errs)
	{
	}
	else if (fw == SIO4_FW_TYPE_SYNC)
	{
		FIELD_SHOW(31, 31, 1, _no,			"Rx Count Err"			);
		FIELD_SHOW(30, 30, 1, NULL,			"Reserved"				);
		FIELD_SHOW(29, 29, 1, _dual,		"Sync Mode"				);
		FIELD_SHOW(28, 28, 1, _int,			"Int Loopback"			);
		FIELD_SHOW(27, 27, 1, _hi,			"Tx Env Pol"			);
		FIELD_SHOW(26, 26, 1, _hi,			"Rx Env Pol"			);
		FIELD_SHOW(25, 25, 1, _msb,			"Tx First Bit"			);
		FIELD_SHOW(24, 24, 1, _msb,			"Rx First Bit"			);
		FIELD_SHOW(23, 23, 1, _idle,		"Rx Count Clr"			);
		FIELD_SHOW(22, 22, 1, _txclksrc,	"Tx Clk Source"			);
		FIELD_SHOW(21, 21, 1, _idle,		"Tx Clk Idle"			);
		FIELD_SHOW(20, 20, 1, _fall,		"Tx Clk Pol"			);
		FIELD_SHOW(19, 19, 1, _fall,		"Rx Clk Pol"			);
		FIELD_SHOW(18, 18, 1, _remain,		"Tx FIFO Full Config"	);
		FIELD_SHOW(17, 17, 1, _disable,		"Tx Enable"				);
		FIELD_SHOW(16, 16, 1, _disable,		"Rx Enable"				);
		FIELD_SHOW(15, 15, 1, _yes,			"Rx FIFO Full"			);
		FIELD_SHOW(14, 14, 1, _yes,			"Rx FIFO A Full"		);
		FIELD_SHOW(13, 13, 1, _yes,			"Rx FIFO A Empty"		);
		FIELD_SHOW(12, 12, 1, _yes,			"Rx FIFO Empty"			);
		FIELD_SHOW(11, 11, 1, _yes,			"Tx FIFO Full"			);
		FIELD_SHOW(10, 10, 1, _yes,			"Tx FIFO A Full"		);
		FIELD_SHOW( 9,  9, 1, _yes,			"Tx FIFO A Empty"		);
		FIELD_SHOW( 8,  8, 1, _yes,			"Tx FIFO Empty"			);
		FIELD_SHOW( 7,  6, 1, NULL,			"Reserved"				);
		FIELD_SHOW( 5,  5, 1, _u_rxd,		"RxDatUprEn"			);
		FIELD_SHOW( 4,  4, 1, _l_rxd,		"RxDatLwrEn"			);
		FIELD_SHOW( 3,  3, 1, _u_txd,		"TxDatUprEn"			);
		FIELD_SHOW( 2,  2, 1, _l_txd,		"TxDatLwrEn"			);
		FIELD_SHOW( 1,  1, 1, _idle,		"Rx FIFO Reset"			);
		FIELD_SHOW( 0,  0, 1, _idle,		"Tx FIFO Reset"			);
	}
	else  if (fw == SIO4_FW_TYPE_Z16C30)
	{
		FIELD_SHOW(31, 16, 1, NULL,			"Reserved"			);
		FIELD_SHOW(15, 15, 1, _yes,			"Rx FIFO Full"		);
		FIELD_SHOW(14, 14, 1, _yes,			"Rx FIFO A Full"	);
		FIELD_SHOW(13, 13, 1, _yes,			"Rx FIFO A Empty"	);
		FIELD_SHOW(12, 12, 1, _yes,			"Rx FIFO Empty"		);
		FIELD_SHOW(11, 11, 1, _yes,			"Tx FIFO Full"		);
		FIELD_SHOW(10, 10, 1, _yes,			"Tx FIFO A Full"	);
		FIELD_SHOW( 9,  9, 1, _yes,			"Tx FIFO A Empty"	);
		FIELD_SHOW( 8,  8, 1, _yes,			"Tx FIFO Empty"		);
		FIELD_SHOW( 7,  7, 1, _idle,		"USC Reset"			);
		FIELD_SHOW( 6,  6, 1, NULL,			"Reserved"			);
		FIELD_SHOW( 5,  5, 1, _l_rxd,		"RxDDCDLwrEn"		);
		FIELD_SHOW( 4,  4, 1, _u_rxd,		"RxDDCDUprEn"		);
		FIELD_SHOW( 3,  3, 1, _l_txd,		"TxDCTSLwrEn"		);
		FIELD_SHOW( 2,  2, 1, _u_txd,		"TxDCTSUprEn"		);
		FIELD_SHOW( 1,  1, 1, _idle,		"Rx FIFO Reset"		);
		FIELD_SHOW( 0,  0, 1, _idle,		"Tx FIFO Reset"		);
	}

	return(errs);
}



//*****************************************************************************
static int _gsc_sbr(int fd, int supported, u32 value, int width)
{
	int	errs;
	int	ret		= 0;
	s32	val;

	if (supported)
	{
		errs	= QUERY_REG_SBR(fd, &val);
		ret		= errs ? 0 : (val ? 1 : 0);
	}
	else
	{
		FIELD_SHOW(31,  8, 1, NULL,	"Reserved"	);
		FIELD_SHOW( 7,  0, 1, NULL,	"Sync Byte"	);
	}

	return(ret);
}



//*****************************************************************************
static int _gsc_irq(int fd, int supported, u32 value, int width, const char** state)
{
	int	errs	= 0;
	s32	fw;
	s32	type;

	errs	+= _get_fw_type(fd, &fw);
	errs	+= QUERY_SIO4_TYPE(fd, &type);

	if (errs == 0)
	{
		if (type == SIO4_TYPE_SIO4)
		{
			FIELD_SHOW(31, 16, 1, NULL,		"Reserved"			);
		}
		else
		{
			FIELD_SHOW(31, 31, 1, state,	"Ch4 Rx FIFO Full"	);
			FIELD_SHOW(30, 30, 1, state,	"Ch4 Rx FIFO Empty"	);
			FIELD_SHOW(29, 29, 1, state,	"Ch4 Tx FIFO Full"	);
			FIELD_SHOW(28, 28, 1, state,	"Ch4 Tx FIFO Empty"	);
			FIELD_SHOW(27, 27, 1, state,	"Ch3 Rx FIFO Full"	);
			FIELD_SHOW(26, 26, 1, state,	"Ch3 Rx FIFO Empty"	);
			FIELD_SHOW(25, 25, 1, state,	"Ch3 Tx FIFO Full"	);
			FIELD_SHOW(24, 24, 1, state,	"Ch3 Tx FIFO Empty"	);
			FIELD_SHOW(23, 23, 1, state,	"Ch2 Rx FIFO Full"	);
			FIELD_SHOW(22, 22, 1, state,	"Ch2 Rx FIFO Empty"	);
			FIELD_SHOW(21, 21, 1, state,	"Ch2 Tx FIFO Full"	);
			FIELD_SHOW(20, 20, 1, state,	"Ch2 Tx FIFO Empty"	);
			FIELD_SHOW(19, 19, 1, state,	"Ch1 Rx FIFO Full"	);
			FIELD_SHOW(18, 18, 1, state,	"Ch1 Rx FIFO Empty"	);
			FIELD_SHOW(17, 17, 1, state,	"Ch1 Tx FIFO Full"	);
			FIELD_SHOW(16, 16, 1, state,	"Ch1 Tx FIFO Empty"	);
		}

		if (fw == SIO4_FW_TYPE_Z16C30)
		{
			FIELD_SHOW(15, 15, 1, state,	"Ch4 USC"			);
			FIELD_SHOW(14, 14, 1, state,	"Ch4 Rx FIFO AF"	);
			FIELD_SHOW(13, 13, 1, state,	"Ch4 Tx FIFO AE"	);
			FIELD_SHOW(12, 12, 1, state,	"Ch4 Sync Detect"	);
			FIELD_SHOW(11, 11, 1, state,	"Ch3 USC"			);
			FIELD_SHOW(10, 10, 1, state,	"Ch3 Rx FIFO AF"	);
			FIELD_SHOW( 9,  9, 1, state,	"Ch3 Tx FIFO AE"	);
			FIELD_SHOW( 8,  8, 1, state,	"Ch3 Sync Detect"	);
			FIELD_SHOW( 7,  7, 1, state,	"Ch2 USC"			);
			FIELD_SHOW( 6,  6, 1, state,	"Ch2 Rx FIFO AF"	);
			FIELD_SHOW( 5,  5, 1, state,	"Ch2 Tx FIFO AE"	);
			FIELD_SHOW( 4,  4, 1, state,	"Ch2 Sync Detect"	);
			FIELD_SHOW( 3,  3, 1, state,	"Ch1 USC"			);
			FIELD_SHOW( 2,  2, 1, state,	"Ch1 Rx FIFO AF"	);
			FIELD_SHOW( 1,  1, 1, state,	"Ch1 Tx FIFO AE"	);
			FIELD_SHOW( 0,  0, 1, state,	"Ch1 Sync Detect"	);
		}
		else if ((fw != SIO4_FW_TYPE_SYNC) || (type <= SIO4_TYPE_SIO4A))
		{
			FIELD_SHOW(15,  0, 1, NULL,		"Reserved"			);
		}
		else
		{
			FIELD_SHOW(15, 15, 1, state,	"Ch4 Rx Spare"		);
			FIELD_SHOW(14, 14, 1, state,	"Ch4 Rx FIFO AF"	);
			FIELD_SHOW(13, 13, 1, state,	"Ch4 Tx FIFO AE"	);
			FIELD_SHOW(12, 12, 1, state,	"Ch4 Rx Envelope"	);
			FIELD_SHOW(11, 11, 1, state,	"Ch3 Rx Spare"		);
			FIELD_SHOW(10, 10, 1, state,	"Ch3 Rx FIFO AF"	);
			FIELD_SHOW( 9,  9, 1, state,	"Ch3 Tx FIFO AE"	);
			FIELD_SHOW( 8,  8, 1, state,	"Ch3 Rx Envelope"	);
			FIELD_SHOW( 7,  7, 1, state,	"Ch2 Rx Spare"		);
			FIELD_SHOW( 6,  6, 1, state,	"Ch2 Rx FIFO AF"	);
			FIELD_SHOW( 5,  5, 1, state,	"Ch2 Tx FIFO AE"	);
			FIELD_SHOW( 4,  4, 1, state,	"Ch2 Rx Envelope"	);
			FIELD_SHOW( 3,  3, 1, state,	"Ch1 Rx Spare"		);
			FIELD_SHOW( 2,  2, 1, state,	"Ch1 Rx FIFO AF"	);
			FIELD_SHOW( 1,  1, 1, state,	"Ch1 Tx FIFO AE"	);
			FIELD_SHOW( 0,  0, 1, state,	"Ch1 Rx Envelope"	);
		}
	}

	return(errs);
}



//*****************************************************************************
static int _gsc_icr(int fd, int supported, u32 value, int width)
{
	static const char*	_disable[]	= { "Disabled",	"Enabled"	};

	int	errs;

	errs	= _gsc_irq(fd, supported, value, width, _disable);
	return(errs);
}



//*****************************************************************************
static int _gsc_isr(int fd, int supported, u32 value, int width)
{
	static const char*	_idle[]	= { "Idle",	"Active"	};

	int	errs;

	errs	= _gsc_irq(fd, supported, value, width, _idle);
	return(errs);
}



//*****************************************************************************
static int _gsc_ielr(int fd, int supported, u32 value, int width)
{
	static const char*	_level[]	= { "Level",	"Edge"	};

	int	errs;
	int	ret;
	s32	val;

	if (supported)
	{
		errs	= QUERY_REG_IELR(fd, &val);
		ret		= errs ? 0 : (val ? 1 : 0);
	}
	else
	{
		errs	= _gsc_irq(fd, supported, value, width, _level);
		ret		= errs;
	}

	return(ret);
}



//*****************************************************************************
static int _gsc_ihlr(int fd, int supported, u32 value, int width)
{
	static const char*	_low[]	= { "Low",	"High"	};

	int	errs;
	int	ret;
	s32	val;

	if (supported)
	{
		errs	= QUERY_REG_IHLR(fd, &val);
		ret		= errs ? 0 : (val ? 1 : 0);
	}
	else
	{
		errs	= _gsc_irq(fd, supported, value, width, _low);
		ret		= errs;
	}

	return(ret);
}



//*****************************************************************************
static int _gsc_psrcr(int fd, int supported, u32 value, int width)
{
	static const char*	_xcvr[]	=
	{
		"RS-422/RS-485",
		"RS-423",
		"RS-232",
		"RESERVED",
		"RS-530",
		"RS-530A",
		"V.35 Mode 1",
		"V.35 Mode 2",
		"RS-422/RS-423 Mixed Mode 1",
		"RS-422/RS-423 Mixed Mode 2",
		"RESERVED",
		"RESERVED",
		"RESERVED",
		"RESERVED",
		"RESERVED",
		"Disabled"
	};

	static const char*	_disable[]	= { "Disable",	"Enable"	};
	static const char*	_dte[]		= { "DTE",		"DCE"		};
	static const char*	_enable[]	= { "Enable",	"Disable"	};

	int	errs	= 0;
	s32	factor;
	s32	fw;
	s32	model;
	int	ret;
	s32	type;
	s32	val;

	for (;;)	// A convenience loop.
	{
		if (supported)
		{
			errs	= QUERY_REG_PSRCR(fd, &val);
			ret		= errs ? 0 : (val ? 1 : 0);
			break;
		}

		errs	+= _get_fw_type(fd, &fw);
		errs	+= QUERY_FORM_FACTOR(fd, &factor);
		errs	+= QUERY_MODEL_BASE(fd, &model);
		errs	+= QUERY_SIO4_TYPE(fd, &type);
		ret		= errs;

		if (errs)
			break;

		if ((type == SIO4_TYPE_SIO8BX2) && (fw == SIO4_FW_TYPE_Z16C30))
		{
			// PCIE4-SIO8BX2	verified 2013/6/24

			static const char*	_cbl_cts_src[]	= { "USC CTS Output", "RTS Output", "Output 0", "Output 1" };
			static const char*	_cbl_dcd_src[]	= { "USC DCD Output", "RTS Output", "Output 0", "Output 1" };
			static const char*	_cbl_txaux[]	= { "Tri-state", "Prog. Clock", "Output 0", "Output 1" };
			static const char*	_cbl_txc_src[]	= { "Prog. Clock", "Invert Prog. Clock", "Output 0", "Output 1", "USC TxC", "USC RxC", "Cable RxC Input", "Cable RxAuxC" };
			static const char*	_cbl_txd_src2[]	= { "USC TxD", "USC TxD", "Output 0", "Output 1", "Diff. Biphase Mark", "Diff. Biphase Space", "Level", "Diff. Biphase Level" };
			static const char*	_usc_cts_dir[]	= { "Tri-state", "Input Cable CTS", "Reserved", "Output to Cable" };
			static const char*	_usc_dcd_dir[]	= { "Disable", "Input Cable DCD", "Reserved", "Output to Cable" };
			static const char*	_usc_rxc_src[]	= { "Prog. Clock", "Invert Prog. Clock", "Output 0", "Output 1", "Cable RxC", "Cable RxAuxC", "Reserved", "USC RxC" };
			static const char*	_usc_txc_src[]	= { "Prog. Clock", "Invert Prog. Clock", "Output 0", "Output 1", "Cable RxC", "Cable RxAuxC", "Reserved", "USC RxC" };

			FIELD_SHOW	(31, 31, 1, _disable,		"Cable Xcvr"		);
			FIELD_SHOW	(30, 30, 1, _enable,		"Termination"		);
			FIELD_SHOW	(29, 29, 1, _disable,		"External Loopback"	);
			FIELD_SHOW	(28, 28, 1, _dte,			"DCE/DTE Select"	);
			FIELD_SHOW	(27, 24, 1, _xcvr,			"Transceivers"		);
			FIELD_SHOW	(23, 23, 1, _disable,		"Internal Loopback"	);
			FIELD_SHOW	(22, 22, 1, NULL,			"Reserved"			);
			FIELD_SHOW	(21, 19, 1, _cbl_txd_src2,	"Cable TxD Source"	);
			FIELD_SHOW	(18, 17, 1, _cbl_txaux,		"Cable TxAuxC Src"	);
			FIELD_SHOW	(16, 15, 1, _cbl_dcd_src,	"Cable DCD Source"	);
			FIELD_SHOW	(14, 13, 1, _cbl_cts_src,	"Cable RTS Source"	);
			FIELD_SHOW	(12, 11, 1, _usc_dcd_dir,	"USC DCD Direction"	);
			FIELD_SHOW	(10,  9, 1, _usc_cts_dir,	"USC CTS Direction"	);
			FIELD_SHOW	( 8,  6, 1, _cbl_txc_src,	"Cable TxC Source"	);
			FIELD_SHOW	( 5,  3, 1, _usc_rxc_src,	"USC RxC Source"	);
			FIELD_SHOW	( 2,  0, 1, _usc_txc_src,	"USC TxC Source"	);
			break;
		}

		if (type < SIO4_TYPE_SIO4B)
		{
			// SIO4 - not applicable
			// SIO4A
			// SIO4AR
			// SIO4AHRM

			static const char*	_txclk_src[]	= { "Prog. Clock", "Prog. Clock", "Output 0", "Output 1", "USC TxC", "USC RxC", "Cable RxClk", "Reserved" };
			static const char*	_usc_cts_dir[]	= { "Disable", "Cable Input", "Cable Input", "Cable Output" };
			static const char*	_usc_rxc_src[]	= { "Prog. Clock", "Prog. Clock", "Output 0", "Output 1", "Cable RxClk", "Reserved", "Reserved", "USC Output" };
			static const char*	_usc_txc_src[]	= { "Prog. Clock", "Prog. Clock", "Output 0", "Output 1", "Cable RxClk", "Reserved", "Reserved", "USC Output" };

			FIELD_SHOW	(31, 11, 1, NULL,			"Reserved"			);
			FIELD_SHOW	(10,  9, 1, _usc_cts_dir,	"USC CTS Direction"	);
			FIELD_SHOW	( 8,  6, 1, _txclk_src,		"TxClk Source"		);
			FIELD_SHOW	( 5,  3, 1, _usc_rxc_src,	"USC RxC Source"	);
			FIELD_SHOW	( 2,  0, 1, _usc_txc_src,	"USC TxC Source"	);
			break;
		}

		if (fw == SIO4_FW_TYPE_Z16C30)
		{
			// SIO4B
			// SIO4BX
			// SIO4BXR
			// SIO8BXS-SD

			static const char*	_cbl_cts_src[]	= { "USC CTS Output", "RTS Output", "Output 0", "Output 1" };
			static const char*	_cbl_dcd_src[]	= { "USC DCD Output", "RTS Output", "Output 0", "Output 1" };
			static const char*	_cbl_dsr[]		= { "Tri-state", "Input DSR", "Output 0", "Output 1" };
			static const char*	_cbl_txaux[]	= { "Tri-state", "Prog. Clock", "Output 0", "Output 1" };
			static const char*	_cbl_txc_src[]	= { "Prog. Clock", "Invert Prog. Clock", "Output 0", "Output 1", "USC TxC", "USC RxC", "Cable RxC Input", "Cable RxAuxC" };
			static const char*	_cbl_txd_src[]	= { "USC TxD", "USC TxD", "Output 0", "Output 1" };
			static const char*	_usc_cts_dir[]	= { "Tri-state", "Input Cable CTS", "Reserved", "Output to Cable" };
			static const char*	_usc_dcd_dir[]	= { "Disable", "Input Cable DCD", "Reserved", "Output to Cable" };
			static const char*	_usc_rxc_src[]	= { "Prog. Clock", "Invert Prog. Clock", "Output 0", "Output 1", "Cable RxC", "Cable RxAuxC", "Reserved", "USC RxC" };
			static const char*	_usc_txc_src[]	= { "Prog. Clock", "Invert Prog. Clock", "Output 0", "Output 1", "Cable RxC", "Cable RxAuxC", "Reserved", "USC RxC" };

			FIELD_SHOW	(31, 31, 1, _disable,		"Cable Xcvr"		);

			if (type == SIO4_TYPE_SIO4B)
				FIELD_SHOW	(30, 30, 1, NULL,		"Reserved"			);
			else
				FIELD_SHOW	(30, 30, 1, _enable,	"Termination"		);

			FIELD_SHOW	(29, 29, 1, _disable,		"External Loopback"	);
			FIELD_SHOW	(28, 28, 1, _dte,			"DCE/DTE Select"	);
			FIELD_SHOW	(27, 24, 1, _xcvr,			"Transceivers"		);
			FIELD_SHOW	(23, 23, 1, _disable,		"Internal Loopback"	);

			if (type == SIO4_TYPE_SIO8BXS)
			{
				FIELD_SHOW	(22, 21, 1, _cbl_dsr,		"DTR/DST Control"	);
				FIELD_SHOW	(20, 19, 1, _cbl_txd_src,	"Cable TxD Source"	);
			}
			else
			{
				FIELD_SHOW	(22, 21, 1, NULL,			"Reserved"			);
				FIELD_SHOW	(20, 19, 1, _cbl_txd_src,	"Cable TxD Source"	);
			}

			if (type == SIO4_TYPE_SIO4B)
				FIELD_SHOW	(18, 17, 1, NULL,		"Reserved"			);
			else
				FIELD_SHOW	(18, 17, 1, _cbl_txaux,	"Cable TxAuxC Src"	);

			FIELD_SHOW	(16, 15, 1, _cbl_dcd_src,	"Cable DCD Source"	);
			FIELD_SHOW	(14, 13, 1, _cbl_cts_src,	"Cable RTS Source"	);
			FIELD_SHOW	(12, 11, 1, _usc_dcd_dir,	"USC DCD Direction"	);
			FIELD_SHOW	(10,  9, 1, _usc_cts_dir,	"USC CTS Direction"	);
			FIELD_SHOW	( 8,  6, 1, _cbl_txc_src,	"Cable TxC Source"	);
			FIELD_SHOW	( 5,  3, 1, _usc_rxc_src,	"USC RxC Source"	);
			FIELD_SHOW	( 2,  0, 1, _usc_txc_src,	"USC TxC Source"	);
			break;
		}
		else
		{
			// SIO4B-SYNC
			// SIO4BX-SYNC
			// SIO4BXR-SYNC

			static const char*	_rxc_src[]	= { "Falling Edge", "Rising Edge" };
			static const char*	_rxd_src[]	= { "RxD Active Hi (NRZ)", "RxD Active Low (NRZB)", "Reserved", "Reserved" };
			static const char*	_rxe_src[]	= { "RxE Active Hi", "RxE Active Low", "RxE Disabled", "RxE Disabled" };
			static const char*	_txax_src[]	= { "Tri-state", "Prog. Clock/2", "Drive 0", "Drive 1" };
			static const char*	_txc_idle[]	= { "Always Active", "No TxC While Idle" };
			static const char*	_txc_src[]	= { "Rising Edge, Prog. Clock/2", "Falling Edge, Prog. Clock/2", "Rising Edge, Ext. Clock", "Falling Edge, Ext. Clock", "Reserved", "Reserved", "Drive 0", "Drive 1" };
			static const char*	_txd_idle[]	= { "Low While Idle", "High While Idle" };
			static const char*	_txd_src[]	= { "TxD Active Hi (NRZ)", "TxD Active Low (NRZB)", "Reserved", "Reserved", "Reserved", "Reserved", "Drive 0", "Drive 1" };
			static const char*	_txe_src[]	= { "TxE Active Hi", "TxE Active Low", "Drive 0", "Drive 1" };
			static const char*	_txsp_src[]	= { "Tri-state", "Tri-state", "Drive 0", "Drive 1" };

			FIELD_SHOW	(31, 31, 1, _disable,		"Cable Xcvr"			);
			FIELD_SHOW	(30, 30, 1, NULL,			"Reserved"				);
			FIELD_SHOW	(29, 29, 1, _disable,		"External Loopback"		);
			FIELD_SHOW	(28, 28, 1, _dte,			"DCE/DTE Select"		);
			FIELD_SHOW	(27, 24, 1, _xcvr,			"Transceivers"			);
			FIELD_SHOW	(23, 23, 1, _disable,		"Internal Loopback"		);

			if (type == SIO4_TYPE_SIO4B)
			{
				FIELD_SHOW	(22, 20, 1, NULL,		"Reserved"				);
			}
			else
			{
				FIELD_SHOW	(22, 21, 1, NULL,		"Reserved"				);
				FIELD_SHOW	(20, 20, 1, _txd_idle,	"TxC Idle"				);
			}

			FIELD_SHOW	(19, 19, 1, _txd_idle,		"TxD Idle"				);

			if (type == SIO4_TYPE_SIO4B)
			{
				FIELD_SHOW	(18, 18, 1, NULL,		"Reserved"				);
				FIELD_SHOW	(17, 16, 1, _rxd_src,	"RxD Source"			);
			}
			else
			{
				FIELD_SHOW	(18, 17, 1, NULL,		"Reserved"				);
				FIELD_SHOW	(16, 16, 1, _rxd_src,	"RxD Source"			);
			}

			FIELD_SHOW	(15, 14, 1, _rxe_src,		"RxE Source"			);
			FIELD_SHOW	(13, 13, 1, _rxc_src,		"RxC Source"			);

			if ((type == SIO4_TYPE_SIO4B) && (factor == SIO4_FORM_FACTOR_PCI))
			{
				FIELD_SHOW	(12,  9, 1, NULL,		"Reserved"				);
			}
			else
			{
				FIELD_SHOW	(12, 11, 1, _txsp_src,	"TxSpare Source"		);
				FIELD_SHOW	(10,  9, 1, _txax_src,	"TxAuxC Source"			);
			}

			FIELD_SHOW	( 8,  6, 1, _txd_src,		"TxD Source"			);
			FIELD_SHOW	( 5,  4, 1, _txe_src,		"TxE Source"			);
			FIELD_SHOW	( 3,  3, 1, _txc_idle,		"TxC Idle"				);
			FIELD_SHOW	( 2,  0, 1, _txc_src,		"TxC Source"			);
		}

		break;
	}

	return(ret);
}



//*****************************************************************************
static int _gsc_pstsr(int fd, int supported, u32 value, int width)
{
	static const char*	_low[]	= { "0, Low",	"1, High"	};

	int	errs	= 0;
	s32	fw;
	int	ret		= 0;
	s32	type;
	s32	val;

	errs	+= _get_fw_type(fd, &fw);
	errs	+= QUERY_SIO4_TYPE(fd, &type);

	if (supported)
	{
		errs	= QUERY_REG_PSTSR(fd, &val);
		ret		= errs ? 0 : (val ? 1 : 0);
	}
	else if (errs)
	{
	}
	else if ((type == SIO4_TYPE_SIO8BX2) && (fw == SIO4_FW_TYPE_Z16C30))
	{
		FIELD_SHOW(31, 10, 1, NULL,	"Reserved"		);
		FIELD_SHOW( 9,  9, 1, _low,	"TxAuxC Output"	);
		FIELD_SHOW( 8,  8, 1, _low,	"RxAuxC Input"	);
		FIELD_SHOW( 7,  7, 1, _low,	"DCD Output"	);
		FIELD_SHOW( 6,  6, 1, _low,	"RTS Output"	);
		FIELD_SHOW( 5,  5, 1, _low,	"TxD Output"	);
		FIELD_SHOW( 4,  4, 1, _low,	"TxC Output"	);
		FIELD_SHOW( 3,  3, 1, _low,	"DCD Input"		);
		FIELD_SHOW( 2,  2, 1, _low,	"CTS Input"		);
		FIELD_SHOW( 1,  1, 1, _low,	"RxD Input"		);
		FIELD_SHOW( 0,  0, 1, _low,	"RxC Input"		);
	}
	else if (fw == SIO4_FW_TYPE_SYNC)
	{
		FIELD_SHOW(31, 10, 1, NULL,	"Reserved"		);
		FIELD_SHOW( 9,  9, 1, _low,	"TxSp Output"	);
		FIELD_SHOW( 8,  8, 1, _low,	"RxSp Input"	);
		FIELD_SHOW( 7,  7, 1, _low,	"TxAuxC Output"	);
		FIELD_SHOW( 6,  6, 1, _low,	"TxE Output"	);
		FIELD_SHOW( 5,  5, 1, _low,	"TxD Output"	);
		FIELD_SHOW( 4,  4, 1, _low,	"TxC Output"	);
		FIELD_SHOW( 3,  3, 1, _low,	"RxAuxC Input"	);
		FIELD_SHOW( 2,  2, 1, _low,	"RxE Input"		);
		FIELD_SHOW( 1,  1, 1, _low,	"RxD Input"		);
		FIELD_SHOW( 0,  0, 1, _low,	"RxC Input"		);
	}
	else
	{
		FIELD_SHOW(31, 12, 1, NULL,	"Reserved"		);
		FIELD_SHOW(11, 11, 1, _low,	"DTR Output"	);
		FIELD_SHOW(10, 10, 1, _low,	"DSR Input"		);
		FIELD_SHOW( 9,  9, 1, _low,	"TxAuxC Output"	);
		FIELD_SHOW( 8,  8, 1, _low,	"RxAuxC Input"	);
		FIELD_SHOW( 7,  7, 1, _low,	"DCD Output"	);
		FIELD_SHOW( 6,  6, 1, _low,	"RTS Output"	);
		FIELD_SHOW( 5,  5, 1, _low,	"TxD Output"	);
		FIELD_SHOW( 4,  4, 1, _low,	"TxC Output"	);
		FIELD_SHOW( 3,  3, 1, _low,	"DCD Input"		);
		FIELD_SHOW( 2,  2, 1, _low,	"CTS Input"		);
		FIELD_SHOW( 1,  1, 1, _low,	"RxD Input"		);
		FIELD_SHOW( 0,  0, 1, _low,	"RxC Input"		);
	}

	return(ret);
}



//*****************************************************************************
static int _gsc_gpiosr(int fd, int supported, u32 value, int width)
{
	int	errs;
	int	ret;
	s32	val;

	if (supported)
	{
		errs	= QUERY_REG_GPIOSR(fd, &val);
		ret		= errs ? 0 : (val ? 1 : 0);
	}
	else
	{
		ret	= 0;
	}

	return(ret);
}



//*****************************************************************************
static int _gsc_iocr(int fd, int supported, u32 value, int width)
{
	static const char*	_edge[]	= { "Level",		"Edge/Latched"	};
	static const char*	_input[]= { "Input",		"Output"		};
	static const char*	_low[]	= { "0, Low",		"1, High"		};
	static const char*	_zero[]	= { "Low/Falling",	"High/Rising"	};

	int	errs;
	int	ret		= 0;
	s32	val;

	if (supported)
	{
		errs	= QUERY_REG_IOCR(fd, &val);
		ret		= errs ? 0 : (val ? 1 : 0);
	}
	else
	{
		FIELD_SHOW(31, 31, 1, NULL,		"Reserved"				);
		FIELD_SHOW(30, 30, 1, _edge,	"Sensitivity"			);
		FIELD_SHOW(29, 29, 1, _zero,	"Upr RxClk/GPIO5 Pol."	);
		FIELD_SHOW(28, 28, 1, _zero,	"Upr RxD/GPIO4 Pol."	);
		FIELD_SHOW(27, 27, 1, _zero,	"DCD/GPIO3 Pol."		);
		FIELD_SHOW(26, 26, 1, _zero,	"Lwr RxClk/GPIO2 Pol."	);
		FIELD_SHOW(25, 25, 1, _zero,	"Lwr RxD/GPIO1 Pol."	);
		FIELD_SHOW(24, 24, 1, _zero,	"CTS/GPIO0 Pol."		);
		FIELD_SHOW(23, 22, 1, NULL,		"Reserved"				);
		FIELD_SHOW(21, 21, 1, _low,		"Upr RxClk/GPIO5 In"	);
		FIELD_SHOW(20, 20, 1, _low,		"Upr RxD/GPIO4 In"		);
		FIELD_SHOW(19, 19, 1, _low,		"DCD/GPIO3 In"			);
		FIELD_SHOW(18, 18, 1, _low,		"Lwr RxClk/GPIO2 In"	);
		FIELD_SHOW(17, 17, 1, _low,		"Lwr RxD/GPIO1 In"		);
		FIELD_SHOW(16, 16, 1, _low,		"CTS/GPIO0 In"			);
		FIELD_SHOW(15, 14, 1, NULL,		"Reserved"				);
		FIELD_SHOW(13, 13, 1, _low,		"Upr RxClk/GPIO5"		);
		FIELD_SHOW(12, 12, 1, _low,		"Upr RxD/GPIO4"			);
		FIELD_SHOW(11, 11, 1, _low,		"DCD/GPIO3"				);
		FIELD_SHOW(10, 10, 1, _low,		"Lwr RxClk/GPIO2"		);
		FIELD_SHOW( 9,  9, 1, _low,		"Lwr RxD/GPIO1"			);
		FIELD_SHOW( 8,  8, 1, _low,		"CTS/GPIO0"				);
		FIELD_SHOW( 7,  6, 1, NULL,		"Reserved"				);
		FIELD_SHOW( 5,  5, 1, _input,	"Upr RxClk/GPIO5"		);
		FIELD_SHOW( 4,  4, 1, _input,	"Upr RxD/GPIO4"			);
		FIELD_SHOW( 3,  3, 1, _input,	"DCD/GPIO3"				);
		FIELD_SHOW( 2,  2, 1, _input,	"Lwr RxClk/GPIO2"		);
		FIELD_SHOW( 1,  1, 1, _input,	"Lwr RxD/GPIO1"			);
		FIELD_SHOW( 0,  0, 1, _input,	"CTS/GPIO0"				);
	}

	return(ret);
}



//*****************************************************************************
static int _gsc_porar(int fd, int supported, u32 value, int width)
{
	int	errs;
	int	ret;
	s32	val;

	if (supported)
	{
		errs	= QUERY_OSC_CHIP(fd, &val);

		if (val == SIO4_OSC_CHIP_CY22393)
			ret	= errs ? 0 : 1;
		else if (val == SIO4_OSC_CHIP_CY22393_2)
			ret	= errs ? 0 : 1;
		else
			ret	= 0;
	}
	else
	{
		ret	= 0;
	}

	return(ret);
}



//*****************************************************************************
static int _gsc_pordr(int fd, int supported, u32 value, int width)
{
	int	errs;
	int	ret;
	s32	val;

	if (supported)
	{
		errs	= QUERY_OSC_CHIP(fd, &val);

		if (val == SIO4_OSC_CHIP_CY22393)
			ret	= errs ? 0 : 1;
		else if (val == SIO4_OSC_CHIP_CY22393_2)
			ret	= errs ? 0 : 1;
		else
			ret	= 0;
	}
	else
	{
		ret	= 0;
	}

	return(ret);
}



//*****************************************************************************
static int _gsc_pocsr(int fd, int supported, u32 value, int width)
{
	int	errs;
	int	ret;
	s32	val;

	if (supported)
	{
		errs	= QUERY_OSC_CHIP(fd, &val);

		if (val == SIO4_OSC_CHIP_CY22393)
			ret	= errs ? 0 : 1;
		else if (val == SIO4_OSC_CHIP_CY22393_2)
			ret	= errs ? 0 : 1;
		else
			ret	= 0;
	}
	else
	{
		ret	= 0;
	}

	return(ret);
}



//*****************************************************************************
static int _gsc_pordr2(int fd, int supported, u32 value, int width)
{
	int	errs;
	int	ret;
	s32	val;

	if (supported)
	{
		errs	= QUERY_OSC_CHIP(fd, &val);
		ret		= errs ? 0 : ((val == SIO4_OSC_CHIP_CY22393_2) ? 1 : 0);
	}
	else
	{
		ret	= 0;
	}

	return(ret);
}



//*****************************************************************************
static int _gsc_pcr(int fd, int supported, u32 value, int width)
{
	int	errs;
	int	ret;
	s32	val;

	if (supported)
	{
		errs	= QUERY_OSC_CHIP(fd, &val);
		ret		= errs ? 0 : ((val == SIO4_OSC_CHIP_IDC2053B_4) ? 1 : 0);
	}
	else
	{
		ret	= 0;
	}

	return(ret);
}



//*****************************************************************************
static int _gsc_pcdr(int fd, int supported, u32 value, int width)
{
	int	errs;
	int	ret;
	s32	val;

	if (supported)
	{
		errs	= QUERY_OSC_CHIP(fd, &val);
		ret		= errs ? 0 : ((val == SIO4_OSC_CHIP_IDC2053B) ? 1 : 0);
	}
	else
	{
		ret	= 0;
	}

	return(ret);
}



//*****************************************************************************
static int _gsc_tcr(int fd, int supported, u32 value, int width)
{
	int	errs;
	int	ret		= 0;
	s32	val;

	if (supported)
	{
		errs	= QUERY_REG_TCR(fd, &val);
		ret		= errs ? 0 : (val ? 1 : 0);
	}
	else
	{
		FIELD_SHOW(31, 16, 0, NULL,	"Gap Bit Count"	);
		val	= FIELD_DECODE(value, 31, 16);
		printf("%ld bits\n", (long) val);

		FIELD_SHOW(15,  0, 0, NULL,	"Tx Bit Count"	);
		val	= FIELD_DECODE(value, 15,  0);
		printf("%ld bits\n", (long) val);
	}

	return(ret);
}



//*****************************************************************************
static int _gsc_rcr(int fd, int supported, u32 value, int width)
{
	int	errs;
	int	ret		= 0;
	s32	val;

	if (supported)
	{
		errs	= QUERY_REG_RCR(fd, &val);
		ret		= errs ? 0 : (val ? 1 : 0);
	}
	else
	{
		FIELD_SHOW(31, 16, 1, NULL,	"Reserved"	);

		FIELD_SHOW(15,  0, 0, NULL,	"Rx Bit Count"	);
		val	= FIELD_DECODE(value, 15,  0);
		printf("%ld bits\n", (long) val);
	}

	return(ret);
}



//*****************************************************************************
static int _gsc_fcr(int fd, int supported, u32 value, int width)
{
	int	errs;
	int	ret		= 0;
	s32	val;

	if (supported)
	{
		errs	= QUERY_REG_FCR(fd, &val);
		ret		= errs ? 0 : (val ? 1 : 0);
	}
	else
	{
		FIELD_SHOW(31, 16, 0, NULL,	"Rx FIFO Count"	);
		val	= FIELD_DECODE(value, 31, 16);
		printf("%ld bytes", (long) val);

		if ((val) && ((val % 1024) == 0))
			printf("  (%ldK)", (long) (val / 1024));

		printf("\n");

		FIELD_SHOW(15,  0, 0, NULL,	"Tx FIFO Count"	);
		val	= FIELD_DECODE(value, 15,  0);
		printf("%ld bytes", (long) val);

		if ((val) && ((val % 1024) == 0))
			printf("  (%ldK)", (long) (val / 1024));

		printf("\n");
	}

	return(ret);
}



//*****************************************************************************
static int _gsc_fsr(int fd, int supported, u32 value, int width)
{
	int	errs;
	int	ret		= 0;
	s32	val;

	if (supported)
	{
		errs	= QUERY_REG_FSR(fd, &val);
		ret		= errs ? 0 : (val ? 1 : 0);
	}
	else
	{
		FIELD_SHOW(31, 16, 0, NULL,	"Rx FIFO Size"	);
		val	= FIELD_DECODE(value, 31, 16);
		printf("%ld bytes", (long) val);

		if ((val) && ((val % 1024) == 0))
			printf("  (%ldK)", (long) (val / 1024));

		printf("\n");

		FIELD_SHOW(15,  0, 0, NULL,	"Tx FIFO Size"	);
		val	= FIELD_DECODE(value, 15,  0);
		printf("%ld bytes", (long) val);

		if ((val) && ((val % 1024) == 0))
			printf("  (%ldK)", (long) (val / 1024));

		printf("\n");
	}

	return(ret);
}



//*****************************************************************************
static void _gsc_ftr_decode(u32 value, int width, int hi, int low, const char* name)
{
	s32	val;

	FIELD_SHOW(hi, low, 0, NULL, name);
	val	= FIELD_DECODE(value, hi, low);

	if (val == SIO4_FW_TYPE_Z16C30)
		printf("Zilog\n");
	else if (val == SIO4_FW_TYPE_SYNC)
		printf("SYNC\n");
	else
		printf("0x%02lX\n", (long) val);
}



//*****************************************************************************
static int _gsc_ftr(int fd, int supported, u32 value, int width)
{
	int	errs;
	int	ret		= 0;
	s32	val;

	if (supported)
	{
		errs	= QUERY_REG_FTR(fd, &val);
		ret		= errs ? 0 : (val ? 1 : 0);
	}
	else
	{
		_gsc_ftr_decode(value, width, 31, 24, "Ch4 Firmware Type");
		_gsc_ftr_decode(value, width, 23, 16, "Ch3 Firmware Type");
		_gsc_ftr_decode(value, width, 15,  8, "Ch2 Firmware Type");
		_gsc_ftr_decode(value, width,  7,  0, "Ch1 Firmware Type");
	}

	return(ret);
}



//*****************************************************************************
static int _gsc_fr(int fd, int supported, u32 value, int width)
{
	static const char*	_osc[]	=
	{
		"Fixed Oscillator (20MHz default)",
		"ICD2053B (1x)",
		"ICD2053B (4x)",
		"CY22393 (1x)",
		"CY22393 (2x)",
		"RESERVED",
		"RESERVED",
		"RESERVED",
		"RESERVED",
		"RESERVED",
		"RESERVED",
		"RESERVED",
		"RESERVED",
		"RESERVED",
		"RESERVED",
		"RESERVED"
	};

	static const char*	_ts[]	=
	{
		"RESERVED",
		"Single External Clock",
		"Single Internal Clock",
		"RESERVED"
	};

	static const char*	_no[]	= { "No",					"Yes"	};
	static const char*	_not[]	= { "No (Not Supported)",	"Yes (Present)"	};

	int	errs;
	int	ret;
	s32	type;
	s32	val;

	if (supported)
	{
		errs	= QUERY_REG_FR(fd, &val);
		ret		= errs ? 0 : (val ? 1 : 0);
	}
	else
	{
		errs	= QUERY_SIO4_TYPE(fd, &type);
		ret		= errs ? 1 : 0;

		if (type <= SIO4_TYPE_SIO4A)
		{
			FIELD_SHOW(31,  5, 1, NULL,	"Reserved"				);
			FIELD_SHOW( 4,  4, 1, _not,	"FIFO Count/Size Regs"	);
			FIELD_SHOW( 3,  0, 1, _osc,	"Oscillator"			);
		}
		else if (value & 0x20)
		{
			// Based on PCIe-SIO4BX2 documentation
			FIELD_SHOW(31, 23, 1, NULL,	"Reserved"				);
			FIELD_SHOW(22, 22, 1, _no,	"CSR Rx Stop On Full"	);
			FIELD_SHOW(21, 21, 1, NULL,	"Reserved"				);
			FIELD_SHOW(20, 20, 1, _no,	"Rx Status Byte"		);
			FIELD_SHOW(19, 18, 1, _ts,	"Timestamp"				);
			FIELD_SHOW(17, 16, 1, NULL,	"FPGA Reprogram"		);
			FIELD_SHOW(15, 14, 1, NULL,	"FIFO Space Config"		);
			FIELD_SHOW(13, 13, 1, _no,	"FIFO Test Bit"			);
			FIELD_SHOW(12, 12, 1, _no,	"Firmware Type Reg"		);
			FIELD_SHOW(11,  8, 1, NULL,	"Feature Level"			);
			FIELD_SHOW( 7,  7, 1, _not,	"DMDMA Single Cyc Dis"	);
			FIELD_SHOW( 6,  6, 1, _not,	"Board Reset"			);
			FIELD_SHOW( 5,  5, 1, _not,	"FIFO Count/Size Regs"	);
			FIELD_SHOW( 4,  4, 1, _no,	"Spec Compliant"		);
			FIELD_SHOW( 3,  0, 1, _osc,	"Oscillator"			);
		}
		else
		{
			FIELD_SHOW(31,  0, 1, NULL,	"Unknown Format"		);
		}
	}

	return(ret);
}



//*****************************************************************************
static int _usc_ccar_decode(int fd, int supported, u32 value, int width)
{
	static const char*	_mode[]	=
	{
		"Normal",
		"Auto Echo",
		"External Local Loopback",
		"Internal Local Loopback"
	};

	static const char*	_idle[]	= { "Idle",	"Active"	};

	int	ret	= 0;

	if (supported)
	{
		ret	= _is_zilog_fw(fd);
	}
	else
	{
		FIELD_SHOW(15, 11, 1, NULL,		"Command"				);
		FIELD_SHOW(10, 10, 1, _idle,	"Channel Reset"			);
		FIELD_SHOW( 9,  8, 1, _mode,	"Mode Control"			);
		FIELD_SHOW( 7,  7, 1, NULL,		"DMA Continue (wo)"		);
		FIELD_SHOW( 6,  6, 1, NULL,		"Byte/Word Access (wo)"	);
		FIELD_SHOW( 5,  1, 1, NULL,		"Address (wo)"			);
		FIELD_SHOW( 0,  0, 1, NULL,		"Upper/Lower Byte (wo)"	);
	}

	return(ret);
}



//*****************************************************************************
static void _usc_cmr_decode_0(int fd, u32 value, int width, int high_byte)
{
	static const char*	_stop_bits[]	=
	{
		"One Stop Bit",
		"Two Stop Bits",
		"One Stop Bit, Shaved",
		"Two Stop Bits, Shaved"
	};

	static const char*	_clock_rate[]	=
	{
		"16x Data Rate",
		"32x Data Rate",
		"64x Data Rate",
		"Reserved"
	};

	// Asynchronous

	if (high_byte)
	{
		FIELD_SHOW(15, 14, 1, _stop_bits,	"Tx Stop Bits"	);
		FIELD_SHOW(13, 12, 1, _clock_rate,	"Tx Clock Rate"	);
	}
	else
	{
		FIELD_SHOW( 7,  6, 1, NULL,			"Reserved"	);
		FIELD_SHOW( 5,  4, 1, _clock_rate,	"Rx Clock Rate"	);
	}
}



//*****************************************************************************
static void _usc_cmr_decode_1(int fd, u32 value, int width, int high_byte)
{
	// External Sync

	if (high_byte)
	{
		FIELD_SHOW(15, 12, 1, NULL,	"Reserved"	);
	}
	else
	{
		FIELD_SHOW( 7,  4, 1, NULL,	"Reserved"	);
	}
}



//*****************************************************************************
static void _usc_cmr_decode_2(int fd, u32 value, int width, int high_byte)
{
	static const char*	_no[]	= { "No",	"Yes"	};

	// Isochronous

	if (high_byte)
	{
		FIELD_SHOW(15, 15, 1, NULL,	"Reserved"		);
		FIELD_SHOW(14, 14, 1, _no,	"Two Stop Bits"	);
		FIELD_SHOW(13, 12, 1, NULL,	"Reserved"		);
	}
	else
	{
		FIELD_SHOW( 7,  4, 1, NULL,	"Reserved"		);
	}
}



//*****************************************************************************
static void _usc_cmr_decode_3(int fd, u32 value, int width, int high_byte)
{
	static const char*	_stop_bits[]	=
	{
		"One Stop Bit",
		"Two Stop Bits",
		"One Stop Bit, Shaved",
		"Two Stop Bits, Shaved"
	};

	static const char*	_no[]	= { "No",	"Yes"	};

	// Asynchronous with Code Violation

	if (high_byte)
	{
		FIELD_SHOW(15, 14, 1, _stop_bits,	"Two Stop Bits"		);
		FIELD_SHOW(13, 13, 1, _no,			"Tx Extended Word"	);
		FIELD_SHOW(12, 12, 1, _no,			"CV Polarity"		);
	}
	else
	{
		FIELD_SHOW( 7,  5, 1, NULL,			"Reserved"			);
		FIELD_SHOW( 4,  4, 1, _no,			"Rx Extended Word"	);
	}
}



//*****************************************************************************
static void _usc_cmr_decode_4(int fd, u32 value, int width, int high_byte)
{
	static const char*	_no[]	= { "No",	"Yes"	};

	// Monosync

	if (high_byte)
	{
		FIELD_SHOW(15, 15, 1, _no,	"Tx CRC On Underrun"		);
		FIELD_SHOW(14, 14, 1, NULL,	"Reserved"					);
		FIELD_SHOW(13, 13, 1, _no,	"Tx Preamble Enable"		);
		FIELD_SHOW(12, 12, 1, _no,	"Tx Short Sync Character"	);
	}
	else
	{
		FIELD_SHOW( 7,  6, 1, NULL,	"Reserved"					);
		FIELD_SHOW( 5,  5, 1, _no,	"Rx Sync Strip"				);
		FIELD_SHOW( 4,  4, 1, _no,	"Rx Short Sync Character"	);
	}
}



//*****************************************************************************
static void _usc_cmr_decode_5(int fd, u32 value, int width, int high_byte)
{
	static const char*	_underrun[]	=
	{
		"SYN1",
		"SYN0/SYN1",
		"CRC/SYN1",
		"CRC/SYN0/SYN1"
	};

	static const char*	_no[]	= { "No",	"Yes"	};

	// Bisync

	if (high_byte)
	{
		FIELD_SHOW(15, 14, 1, _underrun,	"Tx Underrun Condition"		);
		FIELD_SHOW(13, 13, 1, _no,			"Tx Preamble Enable"		);
		FIELD_SHOW(12, 12, 1, _no,			"Tx Short Sync Character"	);
	}
	else
	{
		FIELD_SHOW( 7,  6, 1, NULL,	"Reserved"					);
		FIELD_SHOW( 5,  5, 1, _no,	"Rx Sync Strip"				);
		FIELD_SHOW( 4,  4, 1, _no,	"Rx Short Sync Character"	);
	}
}



//*****************************************************************************
static void _usc_cmr_decode_6(int fd, u32 value, int width, int high_byte)
{
	static const char*	_underrun[]	=
	{
		"Abort",
		"Extended Alert",
		"Flag",
		"CRC/Flag"
	};

	static const char*	_search[]	=
	{
		"Disabled",
		"One Byte, No Control",
		"One Byte, Plus Control",
		"Extended, Plus Control"
	};

	static const char*	_no[]	= { "No",	"Yes"	};

	// HDLC

	if (high_byte)
	{
		FIELD_SHOW(15, 14, 1, _underrun,	"Tx Underrun Condition"		);
		FIELD_SHOW(13, 13, 1, _no,			"Tx Preamble Enable"		);
		FIELD_SHOW(12, 12, 1, _no,			"Shared Zero Flag"			);
	}
	else
	{
		FIELD_SHOW( 7,  7, 1, _no,			"Rx Logical Control Enable"	);
		FIELD_SHOW( 6,  6, 1, _no,			"Rx 16-bit Control"			);
		FIELD_SHOW( 5,  4, 1, _search,		"Rx Search Mode"			);
	}
}



//*****************************************************************************
static void _usc_cmr_decode_7(int fd, u32 value, int width, int high_byte)
{
	static const char*	_underrun[]	=
	{
		"SYN",
		"DLE/SYN",
		"CRC/SYN",
		"CRC/DLE/SYN"
	};

	static const char*	_no[]	= { "No",	"Yes"	};

	// Transparent Bisync

	if (high_byte)
	{
		FIELD_SHOW(15, 14, 1, _underrun,	"Tx Underrun Condition"		);
		FIELD_SHOW(13, 13, 1, _no,			"Tx Preamble Enable"		);
		FIELD_SHOW(12, 12, 1, _no,			"EBCDIC"					);
	}
	else
	{
		FIELD_SHOW( 7,  5, 1, NULL,			"Reserved"					);
		FIELD_SHOW( 4,  4, 1, _no,			"EBCDIC"					);
	}
}



//*****************************************************************************
static void _usc_cmr_decode_8(int fd, u32 value, int width, int high_byte)
{
	static const char*	_rate[]	=
	{
		"16x Data Rate",
		"32x Data Rate",
		"64x Data Rate",
		"Reserved"
	};

	static const char*	_no[]	= { "No",	"Yes"	};

	// NBIP

	if (high_byte)
	{
		FIELD_SHOW(15, 15, 1, _no,		"Tx Address Bit"	);
		FIELD_SHOW(14, 14, 1, _no,		"Tx Parity on Data"	);
		FIELD_SHOW(13, 12, 1, _rate,	"Tx CLock Rate"		);
	}
	else
	{
		FIELD_SHOW( 7,  7, 1, NULL,		"Reserved"			);
		FIELD_SHOW( 6,  6, 1,_no,		"Tx Parity on Data"	);
		FIELD_SHOW( 5,  4, 1, _rate,	"Rx CLock Rate"		);
	}
}



//*****************************************************************************
static void _usc_cmr_decode_9(int fd, u32 value, int width, int high_byte)
{
	static const char*	_no[]	= { "No",	"Yes"	};

	// 802.3 (Ethernet)

	if (high_byte)
	{
		FIELD_SHOW(15, 15, 1, _no,	"Tx CRC on Underrun"	);
		FIELD_SHOW(14, 12, 1, NULL,	"Reserved"				);
	}
	else
	{
		FIELD_SHOW( 7,  5, 1, NULL,	"Reserved"				);
		FIELD_SHOW( 4,  4, 1, _no,	"Rx Address Search"		);
	}
}



//*****************************************************************************
static void _usc_cmr_decode_c(int fd, u32 value, int width, int high_byte)
{
	static const char*	_no[]	= { "No",	"Yes"	};

	// Slaved Monosync

	if (high_byte)
	{
		FIELD_SHOW(15, 15, 1, _no,	"Tx CRC on Underrun"			);
		FIELD_SHOW(14, 14, 1, _no,	"Reserved"						);
		FIELD_SHOW(13, 13, 1, _no,	"Tx Active on Received Sync"	);
		FIELD_SHOW(12, 12, 1, _no,	"Tx Short Sync Character"		);
	}
	else
	{
		FIELD_SHOW( 7,  4, 1, NULL,	"Reserved"						);
	}
}



//*****************************************************************************
static void _usc_cmr_decode_e(int fd, u32 value, int width, int high_byte)
{
	static const char*	_underrun[]	=
	{
		"Abort",
		"Extended Alert",
		"Flag",
		"CRC/Flag"
	};

	static const char*	_no[]	= { "No",	"Yes"	};

	// Slaved Monosync

	if (high_byte)
	{
		FIELD_SHOW(15, 14, 1, _underrun,	"Tx Underrun Condition"	);
		FIELD_SHOW(13, 13, 1, _no,			"Tx Active on Poll"		);
		FIELD_SHOW(12, 12, 1, _no,			"Shared-Zero Flag"		);
	}
	else
	{
		FIELD_SHOW( 7,  4, 1, NULL,			"Reserved"				);
	}
}



//*****************************************************************************
static void _usc_cmr_decode_res(int fd, u32 value, int width, int high_byte)
{
	if (high_byte)
	{
		FIELD_SHOW(15, 11, 1, NULL,	"Reserved"	);
	}
	else
	{
		FIELD_SHOW( 7,  4, 1, NULL,	"Reserved"	);
	}
}



//*****************************************************************************
static int _usc_cmr_decode(int fd, int supported, u32 value, int width)
{
	static const char*	_tx_mode[]	=
	{
		"Asynchronous",
		"Reserved",
		"Isochronous",
		"Reserved",
		"Monosync",
		"Bisync",
		"HDLC/SDLC",
		"Transparent Bisync",
		"Nine Bit",
		"802.3 (Ethernet)",
		"Reserved",
		"Reserved",
		"Slaved Monosync",
		"Reserved",
		"HDLC/SDLC Loop",
		"Reserved"
	};

	static const char*	_rx_mode[]	=
	{
		"Asynchronous",
		"External Sync",
		"Isochronous",
		"Reserved",
		"Monosync",
		"Bisync",
		"HDLC/SDLC",
		"Transparent Bisync",
		"Nine Bit",
		"802.3 (Ethernet)",
		"Reserved",
		"Reserved",
		"Reserved",
		"Reserved",
		"Reserved",
		"Reserved"
	};

	int	i;
	int	ret		= 0;
	u32	val;

	if (supported)
	{
		ret	= _is_zilog_fw(fd);
	}
	else
	{
		for (i = 1; i >= 0; i--)
		{
			val	= i ? ((value >> 8) & 0xF) : (value & 0xF);

			switch (val)
			{
				default:
				case 0x0:	_usc_cmr_decode_0	(fd, value, width, i);	break;	// Asynchronous
				case 0x1:	_usc_cmr_decode_1	(fd, value, width, i);	break;	// External Sync
				case 0x2:	_usc_cmr_decode_2	(fd, value, width, i);	break;	// Isochronous
				case 0x3:	_usc_cmr_decode_3	(fd, value, width, i);	break;	// Asynchronous w/ CV
				case 0x4:	_usc_cmr_decode_4	(fd, value, width, i);	break;	// Monosync
				case 0x5:	_usc_cmr_decode_5	(fd, value, width, i);	break;	// Bisync
				case 0x6:	_usc_cmr_decode_6	(fd, value, width, i);	break;	// HDLC
				case 0x7:	_usc_cmr_decode_7	(fd, value, width, i);	break;	// Transparent Bisync
				case 0x8:	_usc_cmr_decode_8	(fd, value, width, i);	break;	// NBIP
				case 0x9:	_usc_cmr_decode_9	(fd, value, width, i);	break;	// 802.3 (Ethernet)
				case 0xA:	_usc_cmr_decode_res	(fd, value, width, i);	break;	// Reserved
				case 0xB:	_usc_cmr_decode_res	(fd, value, width, i);	break;	// Reserved
				case 0xC:	_usc_cmr_decode_c	(fd, value, width, i);	break;	// Slaved Monosync
				case 0xD:	_usc_cmr_decode_res	(fd, value, width, i);	break;	// Reserved
				case 0xE:	_usc_cmr_decode_e	(fd, value, width, i);	break;	// HDLC Loop
				case 0xF:	_usc_cmr_decode_res	(fd, value, width, i);	break;	// Reserved
			}

			if (i)
				FIELD_SHOW(11,  8, 1, _tx_mode,	"Tx Mode"	);
			else
				FIELD_SHOW( 3,  0, 1, _rx_mode,	"Rx Mode"	);
		}
	}

	return(ret);
}



//*****************************************************************************
static int _usc_ccsr_decode(int fd, int supported, u32 value, int width)
{
	static const char*	_edge[]	=
	{
		"Both Edges",
		"Rising Edge Only",
		"Falling Edge Only",
		"Async/Sync Inhibit"
	};

	static const char*	_length[]	=
	{
		"8 Bits",
		"1 Bits",
		"2 Bits",
		"3 Bits",
		"4 Bits",
		"5 Bits",
		"6 Bits",
		"7 Bits"
	};

	static const char*	_idle[]		= { "Idle",			"Active"		};
	static const char*	_no[]		= { "No",			"Yes"			};
	static const char*	_no_un[]	= { "No/Unlatched",	"Yes/Latched"	};

	int	ret	= 0;

	if (supported)
	{
		ret	= _is_zilog_fw(fd);
	}
	else
	{
		FIELD_SHOW(15, 15, 1, _no,		"RCC FIFO Overflow"				);
		FIELD_SHOW(14, 14, 1, _no,		"RCC FIFO Valid"				);
		FIELD_SHOW(13, 13, 1, _idle,	"RCC FIFO Clear"				);
		FIELD_SHOW(12, 12, 1, _no,		"DPLL in Sync/Quick Sync"		);
		FIELD_SHOW(11, 11, 1, _no_un,	"Clocks Missed Latched"			);
		FIELD_SHOW(10, 10, 1, _no_un,	"Clock Missed Latched"			);
		FIELD_SHOW( 9,  8, 1, _edge,	"DPLL Adjust/Sync Edge"			);
		FIELD_SHOW( 7,  7, 1, _no,		"On Loop"						);
		FIELD_SHOW( 6,  6, 1, _no,		"Sending Loop"					);
		FIELD_SHOW( 5,  5, 1, NULL,		"Reserved"						);
		FIELD_SHOW( 4,  2, 1, _length,	"HDLC Tx Last Character Length"	);
		FIELD_SHOW( 1,  1, 1, _no,		"/TxACK"						);
		FIELD_SHOW( 0,  0, 1, _no,		"/RxACK"						);
	}

	return(ret);
}



//*****************************************************************************
static int _usc_ccr_decode(int fd, int supported, u32 value, int width)
{
	static const char*	_status[]	=
	{
		"No Status Block",
		"One Word Status Block",
		"TRwo Word Status Block",
		"Reserved"
	};

	static const char*	_length[]	=
	{
		"8 Bits",
		"16 Bits",
		"32 Bits",
		"64 Bits"
	};

	static const char*	_pattern[]	=
	{
		"0000...",
		"1111...",
		"1010...",
		"0101..."
	};

	static const char*	_no[]	= { "No",	"Yes"		};

	int	async;
	int	ret		= 0;

	if (supported)
	{
		ret	= _is_zilog_fw(fd);
	}
	else
	{
		async	= _is_tx_mode_async();

		FIELD_SHOW		(15, 14, 1, _status,	"Tx Status Block Transfer"	);
		FIELD_SHOW		(13, 13, 1, _no,		"Wait for Tx DMA Trigger"	);
		FIELD_SHOW		(12, 12, 1, _no,		"Tx Flag Preamble"			);

		if (async)
		{
			FIELD_SHOW	(11,  8, 1, NULL,		"Tx Shaved Bit Length"		);
		}
		else
		{
			FIELD_SHOW	(11, 10, 1, _length,	"Tx Preamble Length"		);
			FIELD_SHOW	( 9,  8, 1, _pattern,	"Tx Preamble Pattern"		);
		}

		FIELD_SHOW		( 7,  6, 1, _status,	"Rx Status Block Transfer"	);
		FIELD_SHOW		( 5,  5, 1, _no,		"Wait for Rx DMA Trigger"	);
		FIELD_SHOW		( 4,  0, 1, NULL,		"Reserved"					);
	}

	return(ret);
}



//*****************************************************************************
static int _usc_prr_decode(int fd, int supported, u32 value, int width)
{
	int	ret		= 0;

	if (supported)
	{
		ret	= _is_zilog_fw(fd);
	}
	else
	{
		// We don't decode this register's content.
	}

	return(ret);
}



//*****************************************************************************
static int _usc_srr_decode(int fd, int supported, u32 value, int width)
{
	int	ret		= 0;

	if (supported)
	{
		ret	= _is_zilog_fw(fd);
	}
	else
	{
		// We don't decode this register's content.
	}

	return(ret);
}



//*****************************************************************************
static int _usc_tmdr_decode(int fd, int supported, u32 value, int width)
{
	int	ret		= 0;

	if (supported)
	{
		ret	= _is_zilog_fw(fd);
	}
	else
	{
		// We don't decode this register's content.
	}

	return(ret);
}



//*****************************************************************************
static int _usc_tmcr_decode(int fd, int supported, u32 value, int width)
{
	int	ret		= 0;

	if (supported)
	{
		ret	= _is_zilog_fw(fd);
	}
	else
	{
		FIELD_SHOW(15,  5, 1, NULL,	"Reserved"				);
		FIELD_SHOW( 4,  0, 1, NULL,	"Test Register Address"	);
	}

	return(ret);
}



//*****************************************************************************
static int _usc_cmcr_decode(int fd, int supported, u32 value, int width)
{
	static const char*	_ctr[]	=
	{
		"Disabled",
		"Disabled",
		"RxC pin",
		"TxC pin"
	};

	static const char*	_brg[]	=
	{
		"CTR0 Output",
		"CTR1 Output",
		"RxC Pin",
		"TxC Pin"
	};

	static const char*	_dpll[]	=
	{
		"BRG0 Output",
		"BRG1 Output",
		"RxC Pin",
		"TxC Pin"
	};

	static const char*	_xfer[]	=
	{
		"Disable",
		"RxC Pin",
		"TxC Pin",
		"DPLL Output",
		"BRG0 Output",
		"BRG1 Output",
		"CTR0 Output",
		"CTR1 Output"
	};

	int	ret	= 0;

	if (supported)
	{
		ret	= _is_zilog_fw(fd);
	}
	else
	{
		FIELD_SHOW(15, 14, 1, _ctr,		"CTR1 Clock Source"	);
		FIELD_SHOW(13, 12, 1, _ctr,		"CTR0 Clock Source"	);
		FIELD_SHOW(11, 10, 1, _brg,		"BRG1 Clock Source"	);
		FIELD_SHOW( 9,  8, 1, _brg,		"BRG0 Clock Source"	);
		FIELD_SHOW( 7,  6, 1, _dpll,	"DPLL Clock Source"	);
		FIELD_SHOW( 5,  3, 1, _xfer,	"Tx CLock Source"	);
		FIELD_SHOW( 2,  0, 1, _xfer,	"Rx CLock Source"	);
	}

	return(ret);
}



//*****************************************************************************
static int _usc_hcr_decode(int fd, int supported, u32 value, int width)
{
	static const char*	_ctr0_rate[]	=
	{
		"32x",
		"16x",
		"8x",
		"4x"
	};

	static const char*	_dpll_rate[]	=
	{
		"32x",
		"16x",
		"8x",
		"4x (For CTR1 Only)"
	};

	static const char*	_dpll_mode[]	=
	{
		"Disable",
		"NRZ",
		"Biphase Mark/Space",
		"Biphase Level"
	};

	static const char*	_tx_ack[]	=
	{
		"3-State Output (*** MUST BE DMA)",
		"Tx Acknowledge Input (DMA)",
		"Output 0 (*** MUST BE DMA)",
		"Output 1 (*** MUST BE DMA)"
	};

	static const char*	_rx_ack[]	=
	{
		"3-State Output (*** MUST BE DMA)",
		"Rx Acknowledge Input (DMA)",
		"Output 0 (*** MUST BE DMA)",
		"Output 1 (*** MUST BE DMA)"
	};

	static const char*	_brg_mode[]		= { "Continuous",	"Single Cycle"	};
	static const char*	_brg_state[]	= { "Disabled",		"Enabled"		};
	static const char*	_ctr1_rate[]	= { "CTR0",			"DPLL"			};
	static const char*	_no[]			= { "No",			"Yes"			};

	int	ret	= 0;

	if (supported)
	{
		ret	= _is_zilog_fw(fd);
	}
	else
	{
		FIELD_SHOW(15, 14, 1, _ctr0_rate,	"CTR0 Clock Rate"			);
		FIELD_SHOW(13, 13, 1, _ctr1_rate,	"CTR1 Rate Source"			);
		FIELD_SHOW(12, 12, 1, _no,			"Accept Code Violations"	);
		FIELD_SHOW(11, 10, 1, _dpll_rate,	"DPLL Clock Rate"			);
		FIELD_SHOW( 9,  8, 1, _dpll_mode,	"DPLL Mode"					);
		FIELD_SHOW( 7,  6, 1, _tx_ack,		"TxACK Pin Control"			);
		FIELD_SHOW( 5,  5, 1, _brg_mode,	"BRG1 Mode"					);
		FIELD_SHOW( 4,  4, 1, _brg_state,	"BRG1 State"				);
		FIELD_SHOW( 3,  2, 1, _rx_ack,		"RxACK Pin Control"			);
		FIELD_SHOW( 1,  1, 1, _brg_mode,	"BRG0 Mode"					);
		FIELD_SHOW( 0,  0, 1, _brg_state,	"BRG0 State"				);
	}

	return(ret);
}



//*****************************************************************************
static int _usc_ivr_decode(int fd, int supported, u32 value, int width)
{
	static const char*	_modified[]	=
	{
		"None",
		"Device Status",
		"I/O Status",
		"Transmit Data",
		"Transmit Status",
		"Receive Data",
		"Receive Status",
		"Not Used"
	};

	int	ret	= 0;

	if (supported)
	{
		ret	= _is_zilog_fw(fd);
	}
	else
	{
		FIELD_SHOW(15, 12, 1, NULL,			"IV"				);
		FIELD_SHOW(11,  9, 1, _modified,	"Modifed Vector"	);
		FIELD_SHOW( 8,  0, 1, NULL,			"IV"				);
	}

	return(ret);
}



//*****************************************************************************
static int _usc_iocr_decode(int fd, int supported, u32 value, int width)
{
	static const char*	_cts[]	=
	{
		"/CTS Input",
		"/CTS Input",
		"Output 0",
		"Output 1"
	};

	static const char*	_dcd[]	=
	{
		"/DCD Input",
		"/DCD//SYNC Input",
		"Output 0",
		"Output 1"
	};

	static const char*	_txreq[]	=
	{
		"3-State Output (*** MUST BE DMA)",
		"Tx Request Output (DMA)",
		"Output 0 (*** MUST BE DMA)",
		"Output 1 (*** MUST BE DMA)"
	};

	static const char*	_rxreq[]	=
	{
		"3-State Output (*** MUST BE DMA)",
		"Rx Request Output (DMA)",
		"Output 0 (*** MUST BE DMA)",
		"Output 1 (*** MUST BE DMA)"
	};

	static const char*	_txd[]	=
	{
		"Tx Data Out (DATA)",
		"3-State Output (*** MUST BE DATA)",
		"Output 0 (*** MUST BE DATA)",
		"Output 1 (*** MUST BE DATA)"
	};

	static const char*	_txc[]	=
	{
		"Input Pin",
		"Tx Clock Output",
		"Tx Byte Count Output",
		"Tx Complete Output",
		"BRG0 Output",
		"BRG1 Output",
		"CTR1 Output",
		"DPLL Tx Output"
	};

	static const char*	_rxc[]	=
	{
		"Input Pin",
		"Rx Clock Output",
		"Rx Byte Count Output",
		"SYNC Output",
		"BRG0 Output",
		"BRG0 Output",
		"CTR1 Output",
		"DPLL Rx Output"
	};

	int	ret	= 0;

	if (supported)
	{
		ret	= _is_zilog_fw(fd);
	}
	else
	{
		FIELD_SHOW(15, 14, 1, _cts,		"CTS Pin Control"	);
		FIELD_SHOW(13, 12, 1, _dcd,		"DCD Pin Control"	);
		FIELD_SHOW(11, 10, 1, _txreq,	"TxREQ Pin Control"	);
		FIELD_SHOW( 9,  8, 1, _rxreq,	"RxREQ Pin Control"	);
		FIELD_SHOW( 7,  6, 1, _txd,		"TxD Pin Control"	);
		FIELD_SHOW( 5,  3, 1, _txc,		"TxC Pin Control"	);
		FIELD_SHOW( 2,  0, 1, _rxc,		"RxC Pin Control"	);
	}

	return(ret);
}



//*****************************************************************************
static int _usc_icr_decode(int fd, int supported, u32 value, int width)
{
	static const char*	_vis[]	=
	{
		"All",
		"All",
		"I/O Status and Above",
		"Transmit Data and Above",
		"Transmit Status and Above",
		"Receive Data and Above",
		"Receive Status Only",
		"None"
	};

	static const char*	_ie[]	=
	{
		"NULL Command",
		"NULL Command",
		"Reset IE",
		"Set IE"
	};

	static const char*	_no[]	= { "No",	"Yes"	};

	int	ret	= 0;

	if (supported)
	{
		ret	= _is_zilog_fw(fd);
	}
	else
	{
		FIELD_SHOW(15, 15, 1, NULL,		"MIE"					);
		FIELD_SHOW(14, 14, 1, NULL,		"DLC"					);
		FIELD_SHOW(13, 13, 1, NULL,		"NV"					);
		FIELD_SHOW(12, 12, 1, NULL,		"VIS"					);
		FIELD_SHOW(11,  9, 1, _vis,		"VIS Level"				);
		FIELD_SHOW( 8,  8, 1, NULL,		"Reserved"				);
		FIELD_SHOW( 7,  6, 1, _ie,		"IE Command (WO)"		);
		FIELD_SHOW( 5,  5, 1, _no,		"Receive Status IE"		);
		FIELD_SHOW( 4,  4, 1, _no,		"Receive Data IE"		);
		FIELD_SHOW( 3,  3, 1, _no,		"Transmit Status IE"	);
		FIELD_SHOW( 2,  2, 1, _no,		"Transmit Data IE"		);
		FIELD_SHOW( 1,  1, 1, _no,		"I/O Status IE"			);
		FIELD_SHOW( 0,  0, 1, _no,		"Device Status IE"		);
	}

	return(ret);
}



//*****************************************************************************
static int _usc_dccr_decode(int fd, int supported, u32 value, int width)
{
	static const char*	_ius[]	=
	{
		"Null Command",
		"Null Command",
		"Reset IUS",
		"Set IUS"
	};

	static const char*	_ip[]	=
	{
		"Null Command",
		"Reset IP and IUS",
		"Reset IP",
		"Set IP"
	};

	static const char*	_no[]	= { "No",	"Yes"	};

	int	ret	= 0;

	if (supported)
	{
		ret	= _is_zilog_fw(fd);
	}
	else
	{
		FIELD_SHOW(15, 14, 1, _ius,		"IUS Command (WO)"		);
		FIELD_SHOW(13, 13, 1, _no,		"Receive Status IUS"	);
		FIELD_SHOW(12, 12, 1, _no,		"Receive Data IUS"		);
		FIELD_SHOW(11, 11, 1, _no,		"Transmit Status IUS"	);
		FIELD_SHOW(10, 10, 1, _no,		"Transmit Data IUS"		);
		FIELD_SHOW( 9,  9, 1, _no,		"I/O Status IUS"		);
		FIELD_SHOW( 8,  8, 1, _no,		"Device Status IUS"		);
		FIELD_SHOW( 7,  6, 1, _ip,		"IP Command (WO)"		);
		FIELD_SHOW( 5,  5, 1, _no,		"Receive Status IP"		);
		FIELD_SHOW( 4,  4, 1, _no,		"Receive Data IP"		);
		FIELD_SHOW( 3,  3, 1, _no,		"Transmit Status IP"	);
		FIELD_SHOW( 2,  2, 1, _no,		"Transmit Data IEIP"	);
		FIELD_SHOW( 1,  1, 1, _no,		"I/O Status IP"			);
		FIELD_SHOW( 0,  0, 1, _no,		"Device Status IP"		);
	}

	return(ret);
}



//*****************************************************************************
static int _usc_misr_decode(int fd, int supported, u32 value, int width)
{
	static const char*	_no[]		= { "No",			"Yes"		};
	static const char*	_unlatch[]	= { "Unlatched",	"Latched"	};

	int	ret	= 0;

	if (supported)
	{
		ret	= _is_zilog_fw(fd);
	}
	else
	{
		FIELD_SHOW(15, 15, 1, _unlatch,	"/RxC Latched/Unlatche"					);
		FIELD_SHOW(14, 14, 1, _no,		"/RxC (RO)"								);
		FIELD_SHOW(13, 13, 1, _unlatch,	"/TxC Latched/Unlatche"					);
		FIELD_SHOW(12, 12, 1, _no,		"/TxC (RO)"								);
		FIELD_SHOW(11, 11, 1, _unlatch,	"/RxREQ Latched/Unlatche"				);
		FIELD_SHOW(10, 10, 1, _no,		"/RxREQ (RO)"							);
		FIELD_SHOW( 9,  9, 1, _unlatch,	"/TxREQ Latched/Unlatche"				);
		FIELD_SHOW( 8,  8, 1, _no,		"/TxREQ (RO)"							);
		FIELD_SHOW( 7,  7, 1, _unlatch,	"/DCD Latched/Unlatche"					);
		FIELD_SHOW( 6,  6, 1, _no,		"/DCD (RO)"								);
		FIELD_SHOW( 5,  5, 1, _unlatch,	"/CTS Latched/Unlatche"					);
		FIELD_SHOW( 4,  4, 1, _no,		"/CTS (RO)"								);
		FIELD_SHOW( 3,  3, 1, _unlatch,	"RCC Overflow Latched/Unlatche"			);
		FIELD_SHOW( 2,  2, 1, _no,		"DPLL SYNC Overflow Latched/Unlatche"	);
		FIELD_SHOW( 1,  1, 1, _unlatch,	"BRG1 ZC Overflow Latched/Unlatche"		);
		FIELD_SHOW( 0,  0, 1, _no,		"BRG0 ZC Overflow Latched/Unlatche"		);
	}

	return(ret);
}



//*****************************************************************************
static int _usc_sicr_decode(int fd, int supported, u32 value, int width)
{
	static const char*	_edge[]	=
	{
		"Disabled",
		"Rising Edge Only",
		"Falling Edge Only",
		"Both Edges"
	};

	static const char*	_no[]	= { "No",	"Yes"	};

	int	ret	= 0;

	if (supported)
	{
		ret	= _is_zilog_fw(fd);
	}
	else
	{
		FIELD_SHOW(15, 14, 1, _edge,	"/RxC Interrupts"	);
		FIELD_SHOW(13, 12, 1, _edge,	"/TxC Interrupts"	);
		FIELD_SHOW(11, 10, 1, _edge,	"/RxREQ Interrupts"	);
		FIELD_SHOW( 9,  8, 1, _edge,	"/TxREQ Interrupts"	);
		FIELD_SHOW( 7,  6, 1, _edge,	"/DCD Interrupts"	);
		FIELD_SHOW( 5,  4, 1, _edge,	"/CTS Interrupts"	);
		FIELD_SHOW( 3,  2, 1, _no,		"RCC Overflow IE"	);
		FIELD_SHOW( 2,  2, 1, _no,		"DPLL SYNC IE"		);
		FIELD_SHOW( 1,  1, 1, _no,		"BRG1 ZC IE"		);
		FIELD_SHOW( 0,  0, 1, _no,		"BRG0 ZC IE"		);
	}

	return(ret);
}



//*****************************************************************************
static int _usc_rdr_decode(int fd, int supported, u32 value, int width)
{
	int	ret	= 0;

	if (supported)
	{
		ret	= _is_zilog_fw(fd);
	}
	else
	{
		// We don't decode this register's content.
	}

	return(ret);
}



//*****************************************************************************
static int _usc_rmr_decode(int fd, int supported, u32 value, int width)
{
	static const char*	_decode[]	=
	{
		"NRZ",
		"NRZB",
		"NRZI-Mark",
		"NRZI-Space",
		"Biphase-Mark",
		"Biphase-Space",
		"Biphase-Level",
		"Differential Biphase-Level"
	};

	static const char*	_crc[]	=
	{
		"CRC-CCITT",
		"CRC-16",
		"CRC-32",
		"Reserved"
	};

	static const char*	_parity[]	=
	{
		"Even",
		"Odd",
		"Space",
		"Mark"
	};

	static const char*	_length[]	=
	{
		"8 Bits",
		"1 Bit",
		"2 Bits",
		"3 Bits",
		"4 Bits",
		"5 Bits",
		"6 Bits",
		"7 Bits"
	};

	static const char*	_enable[]	=
	{
		"Disable Immediately",
		"Disable After Reception",
		"Enable Without Auto-Enables",
		"Enable With Auto-Enables"
	};

	static const char*	_no[]	= { "No",	"Yes"	};

	int	ret	= 0;

	if (supported)
	{
		ret	= _is_zilog_fw(fd);
	}
	else
	{
		FIELD_SHOW(15, 13, 1, _decode,	"Rx Data Decoding"			);
		FIELD_SHOW(12, 11, 1, _crc,		"Rx CRC Polynomial"			);
		FIELD_SHOW(10, 10, 1, NULL,		"Rx CRC Preset Value"		);
		FIELD_SHOW( 9,  9, 1, _no,		"Rx CRC Enable"				);
		FIELD_SHOW( 8,  8, 1, _no,		"Queue Abort"				);
		FIELD_SHOW( 7,  6, 1, _parity,	"Rx Parity Sense"			);
		FIELD_SHOW( 5,  5, 1, _no,		"Rx Parity Enable"			);
		FIELD_SHOW( 4,  2, 1, _length,	"Rx Character LengthSense"	);
		FIELD_SHOW( 1,  0, 1, _enable,	"Rx Enable"					);
	}

	return(ret);
}



//*****************************************************************************
static int _usc_rcsr_decode(int fd, int supported, u32 value, int width)
{
	static const char*	_no[]	= { "No",	"Yes"	};

	int	ret	= 0;

	if (supported)
	{
		ret	= _is_zilog_fw(fd);
	}
	else
	{
		FIELD_SHOW(15, 15, 1, _no,		"Second Byte In Error"		);
		FIELD_SHOW(14, 14, 1, _no,		"First Byte In Error"		);
		FIELD_SHOW(15, 12, 1, NULL,		"Receive Command (WO)"		);
		FIELD_SHOW(11, 11, 1, NULL,		"Residue Code 2"			);
		FIELD_SHOW(10, 10, 1, NULL,		"Residue Code 1"			);
		FIELD_SHOW( 9,  9, 1, NULL,		"Residue Code 0"			);
		FIELD_SHOW( 8,  8, 1, NULL,		"Soft Frame/CV Polarity"	);
		FIELD_SHOW( 7,  7, 1, _no,		"Exited Hunt"				);
		FIELD_SHOW( 6,  6, 1, _no,		"Rx Idle"					);
		FIELD_SHOW( 5,  5, 1, _no,		"Rx Break/Abort"			);
		FIELD_SHOW( 4,  4, 1, _no,		"Rx CV/EOT/EOF"				);
		FIELD_SHOW( 3,  3, 1, _no,		"CRC/Framing Error"			);
		FIELD_SHOW( 2,  2, 1, _no,		"Parity Error/Frame Abort"	);
		FIELD_SHOW( 1,  1, 1, _no,		"Rx Overrun"				);
		FIELD_SHOW( 0,  0, 1, _no,		"Rx Character Available"	);
	}

	return(ret);
}



//*****************************************************************************
static int _usc_ricr_decode(int fd, int supported, u32 value, int width)
{
	static const char*	_arm[]	= { "Unarmed",	"Armed"	};
	static const char*	_no[]	= { "No",		"Yes"	};

	int	ret	= 0;

	if (supported)
	{
		ret	= _is_zilog_fw(fd);
	}
	else
	{
		FIELD_SHOW(15,  8, 1, NULL,		"Rx FIFO Command/Status"		);
		FIELD_SHOW( 7,  7, 1, _arm,		"Exited Hunt IA"				);
		FIELD_SHOW( 6,  6, 1, _arm,		"Rx Idle IA"					);
		FIELD_SHOW( 5,  5, 1, _arm,		"Rx Break/Abort IA"				);
		FIELD_SHOW( 4,  4, 1, _arm,		"Rx CV/EOT/EOF IA"				);
		FIELD_SHOW( 3,  3, 1, _no,		"Status On Words"				);
		FIELD_SHOW( 2,  2, 1, _arm,		"Parity Error/Frame Abort IA"	);
		FIELD_SHOW( 1,  1, 1, _arm,		"Rx Overrun IA"					);
		FIELD_SHOW( 0,  0, 1, _arm,		"TC0R Read Count/TC IA"			);
	}

	return(ret);
}



//*****************************************************************************
static int _usc_rsr_decode(int fd, int supported, u32 value, int width)
{
	int	ret	= 0;

	if (supported)
	{
		ret	= _is_zilog_fw(fd);
	}
	else
	{
		FIELD_SHOW(15,  0, 1, NULL,	"RSYN"	);
	}

	return(ret);
}



//*****************************************************************************
static int _usc_rclr_decode(int fd, int supported, u32 value, int width)
{
	int	ret	= 0;

	if (supported)
	{
		ret	= _is_zilog_fw(fd);
	}
	else
	{
		FIELD_SHOW(15,  0, 1, NULL,	"RCL"	);
	}

	return(ret);
}



//*****************************************************************************
static int _usc_rccr_decode(int fd, int supported, u32 value, int width)
{
	int	ret	= 0;

	if (supported)
	{
		ret	= _is_zilog_fw(fd);
	}
	else
	{
		FIELD_SHOW(15,  0, 1, NULL,	"RCC"	);
	}

	return(ret);
}



//*****************************************************************************
static int _usc_tc0r_decode(int fd, int supported, u32 value, int width)
{
	int	ret	= 0;

	if (supported)
	{
		ret	= _is_zilog_fw(fd);
	}
	else
	{
		FIELD_SHOW(15,  0, 1, NULL,	"TC0"	);
	}

	return(ret);
}



//*****************************************************************************
static int _usc_tdr_decode(int fd, int supported, u32 value, int width)
{
	int	ret	= 0;

	if (supported)
	{
		ret	= _is_zilog_fw(fd);
	}
	else
	{
		// We don't decode this register's content.
	}

	return(ret);
}



//*****************************************************************************
static int _usc_tmr_decode(int fd, int supported, u32 value, int width)
{
	static const char*	_decode[]	=
	{
		"NRZ",
		"NRZB",
		"NRZI-Mark",
		"NRZI-Space",
		"Biphase-Mark",
		"Biphase-Space",
		"Biphase-Level",
		"Differential Biphase-Level"
	};

	static const char*	_crc[]	=
	{
		"CRC-CCITT",
		"CRC-16",
		"CRC-32",
		"Reserved"
	};

	static const char*	_parity[]	=
	{
		"Even",
		"Odd",
		"Space",
		"Mark"
	};

	static const char*	_length[]	=
	{
		"8 Bits",
		"1 Bit",
		"2 Bits",
		"3 Bits",
		"4 Bits",
		"5 Bits",
		"6 Bits",
		"7 Bits"
	};

	static const char*	_enable[]	=
	{
		"Disable Immediately",
		"Disable After Transmission",
		"Enable Without Auto-Enables",
		"Enable With Auto-Enables"
	};

	static const char*	_no[]	= { "No",	"Yes"	};

	int	ret	= 0;

	if (supported)
	{
		ret	= _is_zilog_fw(fd);
	}
	else
	{
		FIELD_SHOW(15, 13, 1, _decode,	"Tx Data Decoding"			);
		FIELD_SHOW(12, 11, 1, _crc,		"Tx CRC Polynomial"			);
		FIELD_SHOW(10, 10, 1, NULL,		"Tx CRC Preset Value"		);
		FIELD_SHOW( 9,  9, 1, _no,		"Tx CRC Enable"				);
		FIELD_SHOW( 8,  8, 1, _no,		"Tx CRC on EOF/EOM"			);
		FIELD_SHOW( 7,  6, 1, _parity,	"Tx Parity Sense"			);
		FIELD_SHOW( 5,  5, 1, _no,		"Tx Parity Enable"			);
		FIELD_SHOW( 4,  2, 1, _length,	"Tx Character LengthSense"	);
		FIELD_SHOW( 1,  0, 1, _enable,	"Tx Enable"					);
	}

	return(ret);
}



//*****************************************************************************
static int _usc_tcsr_decode(int fd, int supported, u32 value, int width)
{
	static const char*	_cmd[]	=
	{
		"Null Command",
		"Reserved",
		"Preset CRC",
		"Reserved",
		"Reserved",
		"Select FIFO Status",
		"Select FIFO Interrupt Level",
		"Select FIFO Request Level",
		"Send Frame/Message",
		"Send Abort",
		"Reserved",
		"Reserved",
		"Reset DLE Inhibit",
		"Set DLE Inhibit",
		"Reset EOF/EOM",
		"Set EOF/EOM"
	};

	static const char*	_idle[]	=
	{
		"YNC/Flag/Normal",
		"Alternating 1 and 0",
		"All Zeros",
		"All Ones",
		"Reserved",
		"Alternating Mark and Space",
		"Space",
		"Mark"
	};

	static const char*	_no[]	= { "No",	"Yes"	};

	int	ret	= 0;

	if (supported)
	{
		ret	= _is_zilog_fw(fd);
	}
	else
	{
		FIELD_SHOW(15, 12, 1, _cmd,		"Transmit Command (WO)"		);
		FIELD_SHOW(11, 11, 1, _no,		"Tx Wait on Underrun"		);
		FIELD_SHOW(10,  8, 1, _idle,	"Tx Idle Line Condition"	);
		FIELD_SHOW( 7,  7, 1, _no,		"Tx Preamble Sent"			);
		FIELD_SHOW( 6,  6, 1, _no,		"Tx Idle Sent"				);
		FIELD_SHOW( 5,  5, 1, _no,		"Tx Abort Sent"				);
		FIELD_SHOW( 4,  4, 1, _no,		"Tx EOF/EOT Sent"			);
		FIELD_SHOW( 3,  3, 1, _no,		"Tx CRC Sent"				);
		FIELD_SHOW( 2,  2, 1, _no,		"All Sent"					);
		FIELD_SHOW( 1,  1, 1, _no,		"Tx Underrun"				);
		FIELD_SHOW( 0,  0, 1, _no,		"Tx Buffer Empty"			);
	}

	return(ret);
}



//*****************************************************************************
static int _usc_ticr_decode(int fd, int supported, u32 value, int width)
{
	static const char*	_arm[]	= { "Unarmed",	"Armed"	};
	static const char*	_no[]	= { "No",		"Yes"	};

	int	ret	= 0;

	if (supported)
	{
		ret	= _is_zilog_fw(fd);
	}
	else
	{
		FIELD_SHOW(15,  8, 1, NULL,	"Tx FIFO Control/Status"	);
		FIELD_SHOW( 7,  7, 1, _arm,	"Tx Preamble Sent IA"		);
		FIELD_SHOW( 6,  6, 1, _arm,	"Tx Idle Sent IA"			);
		FIELD_SHOW( 5,  5, 1, _arm,	"Tx Abort Sent IA"			);
		FIELD_SHOW( 4,  4, 1, _arm,	"Tx EOF/EOT Sent IA"		);
		FIELD_SHOW( 3,  3, 1, _arm,	"Tx CRC Sent IA"			);
		FIELD_SHOW( 2,  2, 1, _no,	"Wait for Send Command"		);
		FIELD_SHOW( 1,  1, 1, _arm,	"Tx Overrun IA"				);
		FIELD_SHOW( 0,  0, 1, _arm,	"TC1R Read Count/TC"		);
	}

	return(ret);
}



//*****************************************************************************
static int _usc_tsr_decode(int fd, int supported, u32 value, int width)
{
	int	ret	= 0;

	if (supported)
	{
		ret	= _is_zilog_fw(fd);
	}
	else
	{
		FIELD_SHOW(15,  0, 1, NULL,	"TSYN"	);
	}

	return(ret);
}



//*****************************************************************************
static int _usc_tclr_decode(int fd, int supported, u32 value, int width)
{
	int	ret	= 0;

	if (supported)
	{
		ret	= _is_zilog_fw(fd);
	}
	else
	{
		FIELD_SHOW(15,  0, 1, NULL,	"TCL"	);
	}

	return(ret);
}



//*****************************************************************************
static int _usc_tccr_decode(int fd, int supported, u32 value, int width)
{
	int	ret	= 0;

	if (supported)
	{
		ret	= _is_zilog_fw(fd);
	}
	else
	{
		FIELD_SHOW(15,  0, 1, NULL,	"TCC"	);
	}

	return(ret);
}



//*****************************************************************************
static int _usc_tc1r_decode(int fd, int supported, u32 value, int width)
{
	int	ret	= 0;

	if (supported)
	{
		ret	= _is_zilog_fw(fd);
	}
	else
	{
		FIELD_SHOW(15,  0, 1, NULL,	"TC1"	);
	}

	return(ret);
}



// variables ******************************************************************

static gsc_reg_def_t	_gsc[]	=
{
	{ _GSC_REG(FRR),	0, _gsc_frr,	"Firmware Revision Register"		},
	{ _GSC_REG(BCR),	0, _gsc_bcr,	"Board Control Register"			},
	{ _GSC_REG(BSR),	1, _gsc_bsr,	"Board Status Register"				},
	{ _GSC_REG(CCR),	1, _gsc_ccr,	"Clock Control Register"			},
	{ _GSC_REG(TSR),	1, _gsc_tsr,	"Time Stamp Register"				},
	{ _GSC_REG(TAR),	0, _gsc_tar,	"Tx Almost Register"				},
	{ _GSC_REG(RAR),	0, _gsc_tar,	"Rx Almost Register"				},
	{ _GSC_REG(FDR),	0, _gsc_fdr,	"FIFO Data Register"				},
	{ _GSC_REG(CSR),	0, _gsc_csr,	"Command/Status Register"			},
	{ _GSC_REG(SBR),	1, _gsc_sbr,	"Sync Byte Register"				},
	{ _GSC_REG(ICR),	0, _gsc_icr,	"Interrupt Control Register"		},
	{ _GSC_REG(ISR),	0, _gsc_isr,	"Interrupt Status Register"			},
	{ _GSC_REG(IELR),	1, _gsc_ielr,	"Interrupt Edge/Level Register"		},
	{ _GSC_REG(IHLR),	1, _gsc_ihlr,	"Interrupt Hi/Low Register"			},
	{ _GSC_REG(PSRCR),	1, _gsc_psrcr,	"Pin Source Register"				},
	{ _GSC_REG(PSTSR),	1, _gsc_pstsr,	"Pin Status Register"				},
	{ _GSC_REG(GPIOSR),	1, _gsc_gpiosr,	"GPIO Source Register"				},
	{ _GSC_REG(IOCR),	1, _gsc_iocr,	"I/O Control Register"				},
	{ _GSC_REG(PORAR),	1, _gsc_porar,	"Programmable Oscillator RAM Address Register"		},
	{ _GSC_REG(PORDR),	1, _gsc_pordr,	"Programmable Oscillator RAM Data Register"			},
	{ _GSC_REG(POCSR),	1, _gsc_pocsr,	"Programmable Oscillator Control/Status Register"	},
	{ _GSC_REG(PORD2R),	1, _gsc_pordr2,	"Programmable Oscillator RAM Data 2 Register"		},
	{ _GSC_REG(PCR),	1, _gsc_pcr,	"Programmable Clock Register"						},
	{ _GSC_REG(PCDR),	1, _gsc_pcdr,	"Programmable Clock/Divider Register"				},
	{ _GSC_REG(TCR),	1, _gsc_tcr,	"Tx Count Register"					},
	{ _GSC_REG(RCR),	1, _gsc_rcr,	"Rx Count Register"					},
	{ _GSC_REG(FCR),	1, _gsc_fcr,	"FIFO Count Register"				},
	{ _GSC_REG(FSR),	1, _gsc_fsr,	"FIFO Size Register"				},
	{ _GSC_REG(FTR),	1, _gsc_ftr,	"Firmware Type Register"			},
	{ _GSC_REG(FR),		1, _gsc_fr,		"Features Register"					},
	{ NULL, 0, 0, 0, 0, NULL, NULL }
};

static gsc_reg_def_t	_usc[33]	=
{
	// These registers are listed in the order they appear
	// in chaper 8 of the smaller of the two blue data books.

	{ _USC_REG(CCAR),	1, _usc_ccar_decode,	"Channel Command/Address Register"		},
	{ _USC_REG(CMR),	1, _usc_cmr_decode,		"Channel Mode Register"					},
	{ _USC_REG(CCSR),	1, _usc_ccsr_decode,	"Channel Command/Status Register"		},
	{ _USC_REG(CCR),	1, _usc_ccr_decode,		"Channel Control Register"				},
	{ _USC_REG(PRR),	1, _usc_prr_decode,		"Primary Reserved Register"				},
	{ _USC_REG(SRR),	1, _usc_srr_decode,		"Secondary Reserved Register"			},
	{ _USC_REG(TMDR),	1, _usc_tmdr_decode,	"Test Mode Data Register"				},
	{ _USC_REG(TMCR),	1, _usc_tmcr_decode,	"Test Mode Control Register"			},
	{ _USC_REG(CMCR),	1, _usc_cmcr_decode,	"Clock Mode Control Register"			},
	{ _USC_REG(HCR),	1, _usc_hcr_decode,		"Hardware Configuration Register"		},
	{ _USC_REG(IVR),	1, _usc_ivr_decode,		"Interrupt Vector Register"				},
	{ _USC_REG(IOCR),	1, _usc_iocr_decode,	"I/O Control Register"					},
	{ _USC_REG(ICR),	1, _usc_icr_decode,		"Interrupt Control Register"			},
	{ _USC_REG(DCCR),	1, _usc_dccr_decode,	"Daisy-Chain Control Register"			},
	{ _USC_REG(MISR),	1, _usc_misr_decode,	"Misc. Interrupt Status Register"		},
	{ _USC_REG(SICR),	1, _usc_sicr_decode,	"Status Interrupt Control Register"		},
	{ _USC_REG(RDR),	1, _usc_rdr_decode,		"Receive Data Register"					},
	{ _USC_REG(RMR),	1, _usc_rmr_decode,		"Receive Mode Register"					},
	{ _USC_REG(RCSR),	1, _usc_rcsr_decode,	"Receive Command/Status Register"		},
	{ _USC_REG(RICR),	1, _usc_ricr_decode,	"Receive Interrupt Control Register"	},
	{ _USC_REG(RSR),	1, _usc_rsr_decode,		"Recieve Sync Register"					},
	{ _USC_REG(RCLR),	1, _usc_rclr_decode,	"Receive Count Limit Register"			},
	{ _USC_REG(RCCR),	1, _usc_rccr_decode,	"Receive Character Count Register"		},
	{ _USC_REG(TC0R),	1, _usc_tc0r_decode,	"Time Constant 0 Register"				},
	{ _USC_REG(TDR),	1, _usc_tdr_decode,		"Transmit Data Register"				},
	{ _USC_REG(TMR),	1, _usc_tmr_decode,		"Transmit Mode Register"				},
	{ _USC_REG(TCSR),	1, _usc_tcsr_decode,	"Transmit Command/Status Register"		},
	{ _USC_REG(TICR),	1, _usc_ticr_decode,	"Transmit Interrupt Control Register"	},
	{ _USC_REG(TSR),	1, _usc_tsr_decode,		"Transmit Sync Register"				},
	{ _USC_REG(TCLR),	1, _usc_tclr_decode,	"Transmit Count Limit Register"			},
	{ _USC_REG(TCCR),	1, _usc_tccr_decode,	"Transmit Character Count Register"		},
	{ _USC_REG(TC1R),	1, _usc_tc1r_decode,	"Time Constant 1 Register"				},
	{ NULL, 0, 0, 0, 0, NULL, NULL }
};



//*****************************************************************************
static int _is_tx_mode_async(void)
{
	int	async	= 0;
	int	i;

	for (i = 0; _usc[i].name; i++)
	{
		if (_usc[i].reg == (int) SIO4_USC_CMR)
		{
			if ((_usc[i].value & 0x0F00) == 0)
				async	= 1;

			break;
		}
	}

	return(async);
}



/******************************************************************************
*
*	Function:	sio4_reg_get_def_index
*
*	Purpose:
*
*		Retrieve the register definition structure based on an index.
*
*	Arguments:
*
*		index	The index of the register to access.
*
*	Returned:
*
*		NULL	The index doesn't correspond to a known register.
*		else	A pointer to the register definition.
*
******************************************************************************/

const gsc_reg_def_t* sio4_reg_get_def_index(int index)
{
	const gsc_reg_def_t*	def;

	for (;;)	// A convenience loop.
	{
		if (index < 0)
		{
			def	= NULL;
			break;
		}

		if (index < ((int) SIZEOF_ARRAY(_gsc) - 1))
		{
			def	= &_gsc[index];
			break;
		}

		index	-= ((int) SIZEOF_ARRAY(_gsc) - 1);

		if (index < ((int) SIZEOF_ARRAY(_usc) - 1))
		{
			def	= &_usc[index];
			break;
		}

		def	= NULL;
		break;
	}

	return(def);
}




/******************************************************************************
*
*	Function:	sio4_reg_get_desc
*
*	Purpose:
*
*		Retrieve the description of the specified register.
*
*	Arguments:
*
*		reg		The register whose description is desired.
*
*	Returned:
*
*		!NULL	The register's name.
*
******************************************************************************/

const char* sio4_reg_get_desc(int reg)
{
	const gsc_reg_def_t*	def;
	const char*				desc;

	def	= _find_reg(reg, _gsc);

	if (def == NULL)
		def	= _find_reg(reg, _usc);

	if (def)
		desc	= def->desc;
	else
		desc	= "UNKNOWN";

	return(desc);
}



/******************************************************************************
*
*	Function:	sio4_reg_list
*
*	Purpose:
*
*		List the GSC registers and their values.
*
*	Arguments:
*
*		fd		The handle to access the device.
*
*		detail	List the register details?
*
*	Returned:
*
*		>= 0	The number of errors encountered here.
*
******************************************************************************/

int sio4_reg_list(int fd, int gsc, int gsc_detail, int usc, int usc_detail)
{
	int	errs	= 0;

	if (gsc)
		errs	+= gsc_reg_list(fd, _gsc, gsc_detail, reg_read);

	if ((gsc) && (usc))
		printf("\n");

	if (usc)
		errs	+= gsc_reg_list(fd, _usc, usc_detail, reg_read);

	return(errs);
}



/******************************************************************************
*
*	Function:	reg_list_pci
*
*	Purpose:
*
*		List the PCI registers.
*
*	Arguments:
*
*		fd		The handle to access the device.
*
*		detail	Produce a detailed listing?
*
*	Returned:
*
*		>= 0	The number of errors encountered here.
*
******************************************************************************/

int reg_list_pci(int fd)
{
	int	errs;

	errs	= gsc_reg_list(fd, _pci, 0, reg_read);
	return(errs);
}



/******************************************************************************
*
*	Function:	reg_list_plx
*
*	Purpose:
*
*		List the PLX registers.
*
*	Arguments:
*
*		fd		The handle to access the device.
*
*		detail	Produce a detailed listing?
*
*	Returned:
*
*		>= 0	The number of errors encountered here.
*
******************************************************************************/

int reg_list_plx(int fd)
{
	int	errs;

	errs	= gsc_reg_list(fd, _plx, 0, reg_read);
	return(errs);
}



//*****************************************************************************
int reg_mod(int fd, u32 reg, u32 value, u32 mask)
{
	int	errs;
	int	ret;

	ret		= sio4_reg_mod(fd, reg, value, mask);
	errs	= (ret < 0) ? 1 : 0;
	return(errs);
}



//*****************************************************************************
int reg_read(int fd, int index, int verbose, u32 reg, u32* value)
{
	int			errs;
	const char*	name;
	int			ret;
	u32			v;

	if (verbose)
		gsc_label_index("Register Read", index);

	ret		= sio4_reg_read(fd, reg, &v);
	errs	= (ret < 0) ? 1 : 0;

	if (verbose)
	{
		name	= sio4_reg_get_name(reg);
		printf(	"%s  (%s: val 0x%08lX)\n",
				errs ? "FAIL <---" : "PASS",
				name ? name : "UNKNOWN",
				(unsigned long) v);
	}

	if (value)
		value[0]	= v;

	return(errs);
}



//*****************************************************************************
int reg_write(int fd, u32 reg, u32 value)
{
	int	errs;
	int	ret;

	ret		= sio4_reg_write(fd, reg, value);
	errs	= (ret < 0) ? 1 : 0;
	return(errs);
}



/******************************************************************************
*
*	Function:	sio4_reg_get_def_id
*
*	Purpose:
*
*		Retrieve the register definition structure given the register id.
*
*	Arguments:
*
*		reg		The id of the register to access.
*
*	Returned:
*
*		NULL	The register id wasn't found.
*		else	A pointer to the register definition.
*
******************************************************************************/

const gsc_reg_def_t* sio4_reg_get_def_id(int reg)
{
	const gsc_reg_def_t*	def;

	def	= _find_reg(reg, _gsc);

	if (def == NULL)
		def	= _find_reg(reg, _usc);

	if (def == NULL)
		def	= _find_reg(reg, _plx);

	if (def == NULL)
		def	= _find_reg(reg, _pci);

	return(def);
}



/******************************************************************************
*
*	Function:	sio4_reg_get_name
*
*	Purpose:
*
*		Retrieve the name of the specified register.
*
*	Arguments:
*
*		reg		The register whose name is desired.
*
*	Returned:
*
*		!NULL	The register's name.
*
******************************************************************************/

const char* sio4_reg_get_name(int reg)
{
	const gsc_reg_def_t*	def;
	const char*				name;

	def	= _find_reg(reg, _gsc);

	if (def == NULL)
		def	= _find_reg(reg, _usc);

	if (def == NULL)
		def	= _find_reg(reg, _pci);

	if (def == NULL)
		def	= _find_reg(reg, _plx);

	if (def)
		name	= def->name;
	else
		name	= "UNKNOWN";

	return(name);
}


