// $URL: http://subversion:8080/svn/gsc/trunk/drivers/LINUX/SIO4%20and%20SIO8/SIO4_Linux_1.x.x_GSC_DN/utils/util_id.c $
// $Rev: 53094 $
// $Date: 2023-06-13 10:21:23 -0500 (Tue, 13 Jun 2023) $

// SIO4: Utilities: source file

#include "main.h"



//*****************************************************************************
static void _id_device_jumpers(s32 qty, s32 sense, s32 value)
{
	u32	bit;
	int	i;

	if (qty > 0)
	{
		gsc_label("User Jumpers");

		if (qty < 0)
		{
			printf("UNKNOWN\n");
		}
		else if (qty == 0)
		{
			printf("(Not supported.)\n");
		}
		else
		{
			printf("0x%01lX         (", (long) value);

			for (i = qty; i > 0; i--)
			{
				bit	= 0x1 << (i - 1);

				if (sense)
					printf("%s", (bit & value) ? "On" : "Off");
				else
					printf("%s", (bit & value) ? "Off" : "On");

				if ((qty > 1) && (i > 1))
					printf(",");
			}

			printf(")\n");
		}
	}
}



//*****************************************************************************
static int _id_device_pci(u32 vid, u32 did, u32 svid, u32 sid)
{
	int	errs	= 0;

	gsc_label("Vendor ID");
	printf("0x%04lX      ", (long) vid);

	if (vid == 0x10B5)
	{
		printf("(PLX)\n");
	}
	else
	{
		errs	= 1;
		printf("(UNKNOWN) FAIL <---\n");
	}


	gsc_label("Device ID");
	printf("0x%04lX      ", (long) did);

	if (did == 0x9056)
	{
		printf("(PCI9056)\n");
	}
	else if (did == 0x9080)
	{
		printf("(PCI9080)\n");
	}
	else
	{
		errs++;
		printf("(UNKNOWN) FAIL <---\n");
	}

	gsc_label("Subsystem Vendor ID");
	printf("0x%04lX      ", (long) svid);

	if (svid == 0x10B5)
	{
		printf("(PLX)\n");
	}
	else
	{
		errs++;
		printf("(UNKNOWN) FAIL <---\n");
	}

	gsc_label("Subsystem ID");
	printf("0x%04lX      ", (long) sid);

	if ((did == 0x9080) && (sid == 0x2401))
	{
		printf("(SIO4/A/B/BX/BXS)\n");
	}
	else if ((did == 0x9056) && (sid == 0x3198))
	{
		printf("(SIO4BXR/BX2, SIO8BXS/BX2)\n");
	}
	else
	{
		errs++;
		printf("(UNKNOWN) FAIL <---\n");
	}

	return(errs);
}



//*****************************************************************************
int sio4_id_fifo_size_total(int fd, int dash, const char** name)
{
	static char	buf[64];
	int			errs	= 0;
	s32			fifo;
	s32			model;
	char*		psz;
	s32			qty;

	errs	+= sio4_feature_test(fd, SIO4_FEATURE_FIFO_SIZE_TOTAL, &fifo, 0);
	errs	+= sio4_feature_test(fd, SIO4_FEATURE_MODEL_BASE, &model, 0);
	errs	+= sio4_feature_test(fd, SIO4_FEATURE_DEVICE_QTY, &qty, 0);

	switch (model)
	{
		default:

			if (dash)
			{
				strcpy(buf, "-");
				psz	= buf + 1;
			}
			else
			{
				psz	= buf;
			}

			if (qty > 1)
				fifo	*= qty;

			if (fifo % 1024)
				sprintf(psz, "%ld", (long) fifo);
			else
				sprintf(psz, "%ldK", (long) fifo / 1024);

			break;

		case SIO4_MODEL_SIO4BX2:
		case SIO4_MODEL_SIO4BX2_SYNC:
		case SIO4_MODEL_SIO8BX2:
		case SIO4_MODEL_SIO8BX2_SYNC:

			break;
	}

	if (name)
		name[0]	= buf;

	return(errs);
}



//*****************************************************************************
int sio4_id_form_factor(int fd, const char** name, int dash)
{
	static char	buf[64];
	int			errs;
	s32			ff;

	errs	= sio4_feature_test(fd, SIO4_FEATURE_FORM_FACTOR, &ff, 0);

	switch (ff)
	{
		default:	sprintf(buf, "UNKNOWN(%ld)", (long) ff);	break;
		case SIO4_FORM_FACTOR_PCI:		strcpy(buf, "PCI");		break;
		case SIO4_FORM_FACTOR_PCI66:	strcpy(buf, "PCI66");	break;
		case SIO4_FORM_FACTOR_PMC:		strcpy(buf, "PMC");		break;
		case SIO4_FORM_FACTOR_PMC66:	strcpy(buf, "PMC66");	break;
		case SIO4_FORM_FACTOR_CPCI:		strcpy(buf, "CPCI");	break;
		case SIO4_FORM_FACTOR_PC104P:	strcpy(buf, "PC104P");	break;
		case SIO4_FORM_FACTOR_XMC:		strcpy(buf, "XMC");		break;
		case SIO4_FORM_FACTOR_UNKNOWN:	strcpy(buf, "");		break;
		case SIO4_FORM_FACTOR_PCIE:		strcpy(buf, "PCIe");	break;
		case SIO4_FORM_FACTOR_PCIE4:	strcpy(buf, "PCIe4");	break;
	}

	if ((dash) && (buf[0]))
		strcat(buf, "-");

	if (name)
		name[0]	= buf;

	return(errs);
}



//*****************************************************************************
int sio4_id_model(int fd, const char** name)
{
	static char	buf[64];
	int			errs;
	s32			model;

	errs	= sio4_feature_test(fd, SIO4_FEATURE_MODEL_BASE, &model, 0);

	switch (model)
	{
		default:	sprintf(buf, "UNKNOWN(%ld)", (long) model);			break;
		case SIO4_MODEL_SIO4:			strcpy(buf, "SIO4");			break;
		case SIO4_MODEL_SIO4A:			strcpy(buf, "SIO4A");			break;
		case SIO4_MODEL_SIO4A_SYNC:		strcpy(buf, "SIO4A-SYNC");		break;
		case SIO4_MODEL_SIO4AR:			strcpy(buf, "SIO4AR");			break;
		case SIO4_MODEL_SIO4AR_SYNC:	strcpy(buf, "SIO4AR-SYNC");		break;
		case SIO4_MODEL_SIO4AHRM:		strcpy(buf, "SIO4AHRM");		break;
		case SIO4_MODEL_SIO4AHRM_SYNC:	strcpy(buf, "SIO4AHRM-SYNC");	break;
		case SIO4_MODEL_SIO4B:			strcpy(buf, "SIO4B");			break;
		case SIO4_MODEL_SIO4B_SYNC:		strcpy(buf, "SIO4B-SYNC");		break;
		case SIO4_MODEL_SIO4BX:			strcpy(buf, "SIO4BX");			break;
		case SIO4_MODEL_SIO4BX_SYNC:	strcpy(buf, "SIO4BX-SYNC");		break;
		case SIO4_MODEL_SIO4BX2:		strcpy(buf, "SIO4BX2");			break;
		case SIO4_MODEL_SIO4BX2_SYNC:	strcpy(buf, "SIO4BX2-SYNC");	break;
		case SIO4_MODEL_SIO4BXR:		strcpy(buf, "SIO4BXR");			break;
		case SIO4_MODEL_SIO4BXR_SYNC:	strcpy(buf, "SIO4BXR-SYNC");	break;
		case SIO4_MODEL_SIO8BXS:		strcpy(buf, "SIO8BXS");			break;
		case SIO4_MODEL_SIO8BXS_SYNC:	strcpy(buf, "SIO8BXS-SYNC");	break;
		case SIO4_MODEL_SIO8BX2:		strcpy(buf, "SIO8BX2");			break;
		case SIO4_MODEL_SIO8BX2_SYNC:	strcpy(buf, "SIO8BX2-SYNC");	break;
	}

	if (name)
		name[0]	= buf;

	return(errs);
}



//*****************************************************************************
int sio4_id_type(int fd, const char** name)
{
	static char	buf[64];
	int			errs;
	s32			type;

	errs	= sio4_feature_test(fd, SIO4_FEATURE_SIO4_TYPE, &type, 0);

	switch (type)
	{
		default:	sprintf(buf, "UNKNOWN(%ld)", (long) type);	break;
		case SIO4_TYPE_SIO4:		strcpy(buf, "SIO4");		break;
		case SIO4_TYPE_SIO4A:		strcpy(buf, "SIO4A");		break;
		case SIO4_TYPE_SIO4AR:		strcpy(buf, "SIO4AR");		break;
		case SIO4_TYPE_SIO4AHRM:	strcpy(buf, "SIO4AHRM");	break;
		case SIO4_TYPE_SIO4B:		strcpy(buf, "SIO4B");		break;
		case SIO4_TYPE_SIO4BX:		strcpy(buf, "SIO4BX");		break;
		case SIO4_TYPE_SIO4BX2:		strcpy(buf, "SIO4BX2");		break;
		case SIO4_TYPE_SIO4BXR:		strcpy(buf, "SIO4BXR");		break;
		case SIO4_TYPE_SIO8BX2:		strcpy(buf, "SIO8BX2");		break;
		case SIO4_TYPE_SIO8BXS:		strcpy(buf, "SIO8BXS");		break;
	}

	if (name)
		name[0]	= buf;

	return(errs);
}



/******************************************************************************
*
*	Function:	sio4_id_device
*
*	Purpose:
*
*		Identify the device.
*
*	Arguments:
*
*		fd		The handle to use to access the device.
*
*		index	The index of the device to access. Use -1 to ignore.
*
*		verbose	Work verbosely?
*
*	Returned:
*
*		>= 0	The number of errors encountered here.
*
******************************************************************************/

int sio4_id_device(int fd, int index, int verbose)
{
	char		buf[64];
	u32			did;
	int			errs	= 0;
	s32			feat_fr;
	u32			fr;			// Features Register
	u32			frr;		// Firmware Revision Register
	const char*	psz		= NULL;
	s32			qty;
	s32			sense;
	u32			sid;
	u32			svid;
	s32			value;
	u32			vid;

	if (verbose)
		gsc_label_index("Device", index);

	errs	+= reg_read(fd, -1, 0, SIO4_PCI_VID, &vid);
	errs	+= reg_read(fd, -1, 0, SIO4_PCI_DID, &did);
	errs	+= reg_read(fd, -1, 0, SIO4_PCI_SID, &sid);
	errs	+= reg_read(fd, -1, 0, SIO4_PCI_SVID, &svid);
	errs	+= reg_read(fd, -1, 0, SIO4_GSC_FRR, &frr);
	errs	+= reg_read(fd, -1, 0, SIO4_GSC_FR, &fr);

	errs	+= feature_test(fd, SIO4_FEATURE_REG_FR, &feat_fr, 1);
	errs	+= feature_test(fd, SIO4_FEATURE_USER_JUMPER_QTY, &qty, 1);
	errs	+= feature_test(fd, SIO4_FEATURE_USER_JUMPER_SENSE, &sense, 1);
	errs	+= feature_test(fd, SIO4_FEATURE_USER_JUMPER_VAL, &value, 1);

	errs	+= sio4_id_form_factor(fd, &psz, 1);
	strcpy(buf, psz);

	errs	+= sio4_id_model(fd, &psz);
	strcat(buf, psz);

	errs	+= sio4_id_fifo_size_total(fd, 1, &psz);
	strcat(buf, psz);

	if (verbose)
	{
		printf("%s\n", errs ? "" : buf);
		gsc_label_level_inc();
	}

	errs	+= _id_device_pci(vid, did, svid, sid);

	if (verbose)
	{
		gsc_label("Firmware Revision Reg.");
		printf("0x%08lX\n", (long) frr);

		gsc_label("Features Register");
		printf("0x%08lX", (long) fr);

		if (feat_fr == 0)
			printf("  (Not supported.)\n");
		else
			printf("\n");
	}

	_id_device_jumpers(qty, sense, value);

	if (verbose)
		gsc_label_level_dec();

	return(errs);
}



//*****************************************************************************
static int _proc_open(int index, int share, int* fd)
{
	int	errs;

	if ((index != -1) || (fd == NULL))
	{
		errs	= 1;
	}
	else
	{
		fd[0]	= open("/proc/sio4", S_IRUSR);

		if (fd[0] < 0)
		{
			fd[0]	= -errno;
			errs	= 1;
		}
		else
		{
			errs	= 0;
		}
	}

	return(errs);
}



//*****************************************************************************
static int _proc_read(int fd, void* dst, size_t size)
{
	int	ret;

	ret	= read(fd, dst, size);
	return(ret);
}



//*****************************************************************************
int id_driver(void)
{
	int	errs;

	errs	= os_id_driver(_proc_open, _proc_read, close);
	return(errs);
}
