// $URL: http://subversion:8080/svn/gsc/trunk/drivers/LINUX/SIO4%20and%20SIO8/SIO4_Linux_1.x.x_GSC_DN/sync/samples/txrate/perform.c $
// $Rev: 53087 $
// $Date: 2023-06-13 10:12:24 -0500 (Tue, 13 Jun 2023) $

// SIO4: SYNC Protocol Library: Sample Application: source file

#include "main.h"




// variables ******************************************************************

u32	poscr;



//*****************************************************************************
static int _calc_program(const args_t* args, dev_data_t* dev, int index, long tx_hz)
{
	sio4_osc_t	arg;
	int			errs		= 0;
	int			ret;

	gsc_label("Calculation");

	if (tx_hz)
	{
		memset(&arg, 0, sizeof(arg));
		arg.freq_want	= tx_hz * 2;
		ret				= osc_cy22393_sio4_program(dev, index, &arg);
		errs			+= ret ? 1 : 0;
		printf("%s  (", ret ? "FAIL <---" : "PASS");
		print_long(tx_hz);

		if (ret)
		{
			printf(")\n");
		}
		else
		{
			printf(" -> ");
			calc_measure(dev, index, 1, tx_hz);
		}
	}
	else if (args->reset)
	{
		sio4_osc_t	osc;

		memset(&osc, 0, sizeof(osc));
		ret		= osc_cy22393_sio4_reset(dev, index, &osc);
		errs	+= ret ? 1 : 0;
		printf("%s  (RESET", ret ? "FAIL <---" : "PASS");

		if (ret)
		{
			printf(")\n");
		}
		else
		{
			printf(" -> ");
			calc_measure(dev, index, 1, tx_hz);
		}
	}
	else if (args->init)
	{
		sio4_osc_t	osc;

		memset(&osc, 0, sizeof(osc));
		ret		= osc_cy22393_sio4_init(dev, index, &osc);
		errs	+= ret ? 1 : 0;
		printf("%s  (INIT", ret ? "FAIL <---" : "PASS");

		if (ret)
		{
			printf(")\n");
		}
		else
		{
			printf(" -> ");
			calc_measure(dev, index, 1, tx_hz);
		}
	}
	else
	{
		printf("ERROR\n");
		errs++;
	}

	return(errs);
}



//*****************************************************************************
static int _device_program(const args_t* args, dev_data_t* dev, int index, long rate)
{
	char	buf[80];
	int		chan	= args->index * 4 + 1 + index;
	int		errs	= 0;
	__s32	got;
	int		ret;

	sprintf(buf, "SIO4 %d", chan);
	gsc_label(buf);
	printf("\n");

	gsc_label_level_inc();

	// Perform a calculations.
	errs	+= _calc_program(args, dev, index, rate);

	// Access the actual device.

	if (args->fd[index] >= 0)
	{
		sprintf(buf, "SIO4 #%d", chan);
		gsc_label(buf);

		if (rate)
		{
			ret		= sio4_osc_program(args->fd[index], rate * 2, &got, 0);
			errs	+= ret ? 1 : 0;
			printf("%s  (%ld", ret ? "FAIL <---" : "PASS", rate);

			if (ret)
				printf(")\n");
			else
				printf(" -> %ld, %ld / 2)\n", (long) got / 2, (long) got);
		}
		else if (args->reset)
		{
			ret		= sio4_osc_reset(args->fd[index], &got, 0);
			errs	+= ret ? 1 : 0;
			printf("%s  (RESET", ret ? "FAIL <---" : "PASS");

			if (ret)
				printf(")\n");
			else
				printf(" -> %ld, %ld / 2)\n", (long) got / 2, (long) got);
		}
		else if (args->init)
		{
			ret		= sio4_osc_init(args->fd[index], &got, 0);
			errs	+= ret ? 1 : 0;
			printf("%s  (INIT", ret ? "FAIL <---" : "PASS");

			if (ret)
				printf(")\n");
			else
				printf(" -> %ld, %ld / 2)\n", (long) got / 2, (long) got);
		}
		else
		{
			printf("INTERNAL ERROR: %s, %d\n", __FILE__, __LINE__);
			exit(EXIT_FAILURE);
		}
	}

	gsc_label_level_dec();
	return(errs);
}



//*****************************************************************************
static int _device_reference(const args_t* args, dev_data_t* dev)
{
	char		buf[80];
	int			chan;
	int			errs	= 0;
	int			i;
	sio4_osc_t	ref;
	int			ret;
	s32			v;

	gsc_label("Reference");
	printf("\n");
	gsc_label_level_inc();

	gsc_label("Calculation");
	dev->osc.cy22393.ref_comp	= 0;
	dev->osc.cy22393.ref_max	= FREQ_REF_MAX;
	dev->osc.cy22393.ref_min	= FREQ_REF_MIN;
	memset(&ref, 0, sizeof(ref));
	ref.freq_ref	= args->fref;
	ret				= osc_cy22393_sio4_reference(dev, 0, &ref);
	errs			+= ret ? 1 : 0;
	printf("%s  (", ret ? "FAIL <---" : "PASS");
	print_long(args->fref);
	printf(" Hz)\n");

	// Now do the real devices.

	for (i = 0; i <= 3; i++)
	{
		chan	= args->index * 4 + 1 + i;
		sprintf(buf, "SIO4 %d", chan);
		gsc_label(buf);
		v		= args->fref;
		ret		= sio4_osc_reference(args->fd[i], &v, 0);
		errs	+= ret ? 1 : 0;
		printf("%s  (", ret ? "FAIL <---" : "PASS");
		print_long(v);
		printf(" Hz)\n");
	}

	gsc_label_level_dec();
	return(errs);
}



//*****************************************************************************
static int _device_startup(const args_t* args, dev_data_t* dev)
{
	gsc_label("Startup");
	memset(dev, 0, sizeof(dev[0]));
	dev->cache.osc_chip		= SIO4_OSC_CHIP_CY22393;
	dev->cache.osc_pd_max	= args->pd_max;
	dev->vaddr.gsc_pocsr_32	= SIO4_GSC_POCSR;
	osc_cy22393_sio4_startup(dev);
	printf("Done  (CY22393)\n");
	return(0);
}



//*****************************************************************************
static int _scan_program(const args_t* args, dev_data_t* dev)
{
	int		errs	= 0;
	long	hz;
	int		status;

	for (hz = args->begin; hz <= args->end; hz++)
	{
		status	= _calc_program(args, dev, 0, hz);
		errs	= status ? 1 : errs;
	}

	return(errs);
}



/******************************************************************************
*
*	Function:	calc_measure
*
*	Purpose:
*
*		Compute the configured frequency for the specified channel.
*
*	Arguments:
*
*		dev		The structure for the device to access.
*
*		chan	The channel number to access.
*
*		verbose	Report some output.
*
*		tx_hz	The frequency requested (for verbose output only).
*
*	Returned:
*
*		>= 0	The computed frequency.
*
******************************************************************************/

float calc_measure(dev_data_t* dev, int chan, int verbose, long tx_hz)
{
	float				acc;
	float				d;
	float				got;
	int					p;
	int					p0;
	osc_cy22393_pll_t*	pll	= &dev->osc.cy22393.osc.pll[0];
	int					q;

	if (dev->osc.cy22393.osc.clock[chan].in_use == 0)
	{
		got	= 0;

		if (verbose)
			printf("DISABLED)\n");
	}
	else if (dev->osc.cy22393.osc.clock[chan].freq_src == CY22393_FREQ_SRC_REF)
	{
		got	= (float) dev->osc.cy22393.osc.freq_ref
			/ (float) dev->osc.cy22393.osc.clock[chan].post_div
			/ (float) (0x1L << dev->osc.cy22393.chan[chan].pd);

		if (verbose == 0)
		{
		}
		else if (tx_hz)
		{
			got	/= 2;
			d	= got - tx_hz;
			acc	= d / tx_hz * 100;

			printf("%.3f", got);
			printf(", acc %+.4f", acc);
			printf(", delta %.3f", d);
			printf(", PD %d", (int) dev->osc.cy22393.osc.clock[chan].post_div);
			printf(", FWPD %d", (int) (0x1L << dev->osc.cy22393.chan[chan].pd));
			printf(", /2)\n");
		}
		else
		{
			got	/= 2;
			printf("%.3f", got);
			printf(", PD %d", (int) dev->osc.cy22393.osc.clock[chan].post_div);
			printf(", FWPD %d", (int) (0x1L << dev->osc.cy22393.chan[chan].pd));
			printf(", /2)\n");
		}
	}
	else
	{
		switch (dev->osc.cy22393.osc.clock[chan].freq_src)
		{
			case CY22393_FREQ_SRC_REF:

				printf("INTERNAL ERROR: %s, %d\n", __FILE__, __LINE__);
				exit(EXIT_FAILURE);
				break;

			case CY22393_FREQ_SRC_PLL1:

				pll	= &dev->osc.cy22393.osc.pll[0];
				break;

			case CY22393_FREQ_SRC_PLL2:

				pll	= &dev->osc.cy22393.osc.pll[1];
				break;

			case CY22393_FREQ_SRC_PLL3:

				pll	= &dev->osc.cy22393.osc.pll[2];
				break;
		}

		p	= pll->p;
		p0	= pll->p0;
		q	= pll->q;

		got	= (float) dev->osc.cy22393.osc.freq_ref
			* (float) ((2 * (p + 3)) + p0)
			/ (float) (q + 2)
			/ (float) dev->osc.cy22393.osc.clock[chan].post_div
			/ (float) (0x1L << dev->osc.cy22393.chan[chan].pd);

		if (verbose)
		{
			if (tx_hz <= 0)
				tx_hz	= 1;

			got	/= 2;
			d	= got - tx_hz;
			acc	= d / tx_hz * 100;

			printf("%.3f", got);
			printf(", acc %+.4f", acc);
			printf(", delta %.3f", d);
			printf(", P %d", p);
			printf(", P0 %d", p0);
			printf(", Q %d", q);
			printf(", PD %d", (int) dev->osc.cy22393.osc.clock[chan].post_div);
			printf(", FWPD %d", (int) (0x1L << dev->osc.cy22393.chan[chan].pd));
			printf(", /2)\n");
		}
	}

	return(got);
}



//*****************************************************************************
int perform_tests(const args_t* args)
{
	dev_data_t	dev;
	int			errs;
	int			skip	= 0;

	for (;;)	// A convenience loop.
	{
		memset(&dev, 0, sizeof(dev_data_t));
		poscr	= 0;
		errs	= firmware_types(args, &skip);

		if ((errs) || (skip))
			break;

		errs	+= osc_types(args, &skip);

		if ((errs) || (skip))
			break;

		errs	+= _device_startup(args, &dev);
		errs	+= _device_reference(args, &dev);

		if (errs)
		{
		}
		else if (args->scan)
		{
			printf("SCAN =============================================\n");
			errs	+= _scan_program(args, &dev);
		}
		else
		{
			errs	+= _device_program(args, &dev, 0, args->bit_rate[0]);
			errs	+= _device_program(args, &dev, 1, args->bit_rate[1]);
			errs	+= _device_program(args, &dev, 2, args->bit_rate[2]);
			errs	+= _device_program(args, &dev, 3, args->bit_rate[3]);
		}

		break;
	}

	return(errs);
}


