// $URL: http://subversion:8080/svn/gsc/trunk/drivers/LINUX/SIO4%20and%20SIO8/SIO4_Linux_1.x.x_GSC_DN/driver/osc_cy22393.c $
// $Rev: 53095 $
// $Date: 2023-06-13 10:42:41 -0500 (Tue, 13 Jun 2023) $

// SIO4: Device Driver: source file

#include "main.h"



// macros *********************************************************************

#define	INDEX_IS_INVALID(i)					(((i) < 0) || ((i) > 3))

#define	LIMIT_RANGE(v,min,max)				(((v)<(min))?(min):((v)>(max))?(max):(v))

#define	P(pt,p0)							((((pt) - (p0)) / 2) - 3)

#define	P_MAX								0x3FF
#define	P_MIN								0

#define	P0_MAX								1
#define	P0_MIN								0

#define	PD_MAX								127
#define	PD_MIN								1

#define	PLL_FREQ_MAX						400000000L
#define	PLL_FREQ_MIN						100000000L

#define	PT_MAX								CY22393_PT((P_MAX),(P0_MAX))
#define	PT_MIN								CY22393_PT((P_MIN),(P0_MIN))

#define	Q(qt)								((qt) - 2)

#define	Q_MAX								0xFF
#define	Q_MIN								0

#define	QT_MAX								CY22393_QT((Q_MAX))
#define	QT_MIN								CY22393_QT((Q_MIN))

#define	FREQ_REF_MAX						30000000L
#define	FREQ_REF_MIN						8000000L



// data types *****************************************************************

typedef struct
{
	long	freq_ref;
	long	freq_want;
	long	freq_got;
	long	freq_pll;
	long	iterations;
	int		pd;
	int		p;
	int		lf;
	int		p0;
	int		q;
} gen_t;

typedef struct
{
	int				found;
	long			delta;
	osc_cy22393_t	osc;
} osc_test_t;



// prototypes *****************************************************************

static void _clock_source_queue(osc_cy22393_t* osc, osc_cy22393_clock_t* clock);



/******************************************************************************
*
*	Function:	_compute_dp_p_p0_q_lf
*
*	Purpose:
*
*		Compute the Post Divider, P, P0 and Q values appropriate for the
*		requested frequency.
*
*	Arguments:
*
*		gen		The structure where our data is recorded.
*
*	Returned:
*
*		None.
*
******************************************************************************/

static void _compute_dp_p_p0_q_lf(gen_t* gen)
{
	long	delta;
	long	delta_min	= 999999999L;
	int		found		= 0;
	long	freq_got;
	long	freq_pll;
	long	iterations	= 0;

	int		p0;

	int		p;
	int		p_max;
	int		p_min;

	int		pd;
	int		pd_max;
	int		pd_min;

	long	pll_want;

	int		pt		= 0;
	int		pt_max;
	int		pt_min;

	int		q;
	int		q_min;
	int		q_max;

	int		qt;
	int		qt_max;
	int		qt_min;

	// Compute the Post Divider range we'll run through.
	pd_max	= PLL_FREQ_MAX / gen->freq_want + 1;	// round up
	pd_min	= PLL_FREQ_MIN / gen->freq_want;		// round down

	pd_max	= LIMIT_RANGE(pd_max, PD_MIN, PD_MAX);
	pd_min	= LIMIT_RANGE(pd_min, PD_MIN, PD_MAX);

	for (pd = pd_min; pd <= pd_max; pd++)
	{
		pll_want	= (long) (gen->freq_want * pd);
		pll_want	= LIMIT_RANGE(pll_want, PLL_FREQ_MIN, PLL_FREQ_MAX);

		// Compute the Qt range.
		qt_max	= (int) ((long long) gen->freq_ref * PT_MAX / (long long) pll_want) + 1;
		qt_min	= (int) ((long long) gen->freq_ref * PT_MIN / (long long) pll_want) - 1;

		qt_max	= LIMIT_RANGE(qt_max, QT_MIN, QT_MAX);
		qt_min	= LIMIT_RANGE(qt_min, QT_MIN, QT_MAX);

		// Compute the Q range.
		q_max	= Q(qt_max);
		q_min	= Q(qt_min);

		q_max	= LIMIT_RANGE(q_max, Q_MIN, Q_MAX);
		q_min	= LIMIT_RANGE(q_min, Q_MIN, Q_MAX);

		for (q = q_min; q <= q_max; q++)
		{
			qt	= CY22393_QT(q);

			// Compute the Pt range.
			pt_max	= (int) ((long long) pll_want * qt / (long long) gen->freq_ref) + 1;	// round up
			pt_min	= (int) ((long long) pll_want * qt / (long long) gen->freq_ref);		// round down

			pt_max	= LIMIT_RANGE(pt_max, PT_MIN, PT_MAX);
			pt_min	= LIMIT_RANGE(pt_min, PT_MIN, PT_MAX);

			// Compute the P range.
			p_max	= P(pt_max, P0_MAX);
			p_min	= P(pt_min, P0_MIN);

			p_max	= LIMIT_RANGE(p_max, P_MIN, P_MAX);
			p_min	= LIMIT_RANGE(p_min, P_MIN, P_MAX);

			for (p = p_min; p <= p_max; p++)
			{
				for (p0 = P0_MIN; p0 <= P0_MAX; p0++)
				{
					iterations++;
					pt			= CY22393_PT(p,p0);
					freq_pll	= (long) ((long long) gen->freq_ref * pt / (long long) qt);

					if ((freq_pll < PLL_FREQ_MIN) || (freq_pll > PLL_FREQ_MAX))
						continue;

					freq_got	= freq_pll / pd;
					delta		= gen->freq_want - freq_got;
					delta		= (delta < 0) ? -delta : delta;

					if ((found == 0) || (delta < delta_min))
					{
						found			= 1;
						delta_min		= delta;
						gen->pd			= pd;
						gen->p			= p;
						gen->p0			= p0;
						gen->q			= q;
						gen->freq_got	= freq_got;
						gen->freq_pll	= freq_pll;

						if (delta == 0)
						{
							p0	= P0_MAX;
							p	= P_MAX;
							q	= Q_MAX;
							pd	= PD_MAX;
						}
					}
				}
			}
		}
	}

	if (pt <= 231)
		gen->lf	= 0;
	else if (pt <= 626)
		gen->lf	= 1;
	else if (pt <= 834)
		gen->lf	= 2;
	else if (pt <= 1043)
		gen->lf	= 3;
	else
		gen->lf	= 4;

	gen->iterations	= iterations;
}



/******************************************************************************
*
*	Function:	_pll_enable_queue
*
*	Purpose:
*
*		Queue program changes needed to configure the given PLL with the
*		respective Enable value.
*
*	Arguments:
*
*		osc		The structure for the oscillator to access.
*
*		clock	The structure for the clock to access.
*
*	Returned:
*
*		None.
*
******************************************************************************/

static void _pll_enable_queue(osc_cy22393_t* osc, osc_cy22393_pll_t* pll)
{
	switch (pll->freq_src)
	{
		default:
		case CY22393_FREQ_SRC_REF:

			break;

		case CY22393_FREQ_SRC_PLL1:

			osc->data._42	&= ~0x40;
			osc->data._42	|= (u8) (pll->enable << 6);
			break;

		case CY22393_FREQ_SRC_PLL2:

			osc->data._13	&= ~0x40;
			osc->data._13	|= (u8) (pll->enable << 6);
			break;

		case CY22393_FREQ_SRC_PLL3:

			osc->data._16	&= ~0x40;
			osc->data._16	|= (u8) (pll->enable << 6);
			break;
	}
}



/******************************************************************************
*
*	Function:	_pll_lf_queue
*
*	Purpose:
*
*		Queue changes need to program the referenced PLL per the given LF
*		value.
*
*	Arguments:
*
*		osc		The structure for the oscillator to access.
*
*		clock	The structure for the clock to access.
*
*	Returned:
*
*		None.
*
******************************************************************************/

static void _pll_lf_queue(osc_cy22393_t* osc, osc_cy22393_pll_t* pll)
{
	switch (pll->freq_src)
	{
		default:
		case CY22393_FREQ_SRC_REF:

			break;

		case CY22393_FREQ_SRC_PLL1:

			osc->data._42	&= ~0x38;
			osc->data._42	|= (u8) (pll->lf << 3);
			break;

		case CY22393_FREQ_SRC_PLL2:

			osc->data._13	&= ~0x38;
			osc->data._13	|= (u8) (pll->lf << 3);
			break;

		case CY22393_FREQ_SRC_PLL3:

			osc->data._16	&= ~0x38;
			osc->data._16	|= (u8) (pll->lf << 3);
			break;
	}
}



/******************************************************************************
*
*	Function:	_pll_p_queue
*
*	Purpose:
*
*		Queue changes need to program the referenced PLL per the given P value.
*
*	Arguments:
*
*		osc		The structure for the oscillator to access.
*
*		clock	The structure for the clock to access.
*
*	Returned:
*
*		None.
*
******************************************************************************/

static void _pll_p_queue(osc_cy22393_t* osc, osc_cy22393_pll_t* pll)
{
	switch (pll->freq_src)
	{
		default:
		case CY22393_FREQ_SRC_REF:

			break;

		case CY22393_FREQ_SRC_PLL1:

			osc->data._41	= pll->p & 0xFF;
			osc->data._42	&= ~0x03;
			osc->data._42	|= (pll->p >> 8) & 0x3;
			break;

		case CY22393_FREQ_SRC_PLL2:

			osc->data._12	= pll->p & 0xFF;
			osc->data._13	&= ~0x03;
			osc->data._13	|= (pll->p >> 8) & 0x3;
			break;

		case CY22393_FREQ_SRC_PLL3:

			osc->data._15	= pll->p & 0xFF;
			osc->data._16	&= ~0x03;
			osc->data._16	|= (pll->p >> 8) & 0x3;
			break;
	}
}



/******************************************************************************
*
*	Function:	_pll_p0_queue
*
*	Purpose:
*
*		Queue changes need to program the referenced PLL per the given P0
*		value.
*
*	Arguments:
*
*		osc		The structure for the oscillator to access.
*
*		pll		The PLL to access.
*
*	Returned:
*
*		None.
*
******************************************************************************/

static void _pll_p0_queue(osc_cy22393_t* osc, osc_cy22393_pll_t* pll)
{
	switch (pll->freq_src)
	{
		default:
		case CY22393_FREQ_SRC_REF:

			break;

		case CY22393_FREQ_SRC_PLL1:

			osc->data._42	&= ~0x04;
			osc->data._42	|= (u8) (pll->p0 << 2);
			break;

		case CY22393_FREQ_SRC_PLL2:

			osc->data._13	&= ~0x04;
			osc->data._13	|= (u8) (pll->p0 << 2);
			break;

		case CY22393_FREQ_SRC_PLL3:

			osc->data._16	&= ~0x04;
			osc->data._16	|= (u8) (pll->p0 << 2);
			break;
	}
}



/******************************************************************************
*
*	Function:	_pll_q_queue
*
*	Purpose:
*
*		Queue changes need to program the referenced PLL per the given Q value.
*
*	Arguments:
*
*		osc		The structure for the oscillator to access.
*
*		pll		The PLL to access.
*
*	Returned:
*
*		None.
*
******************************************************************************/

static void _pll_q_queue(osc_cy22393_t* osc, osc_cy22393_pll_t* pll)
{
	switch (pll->freq_src)
	{
		default:
		case CY22393_FREQ_SRC_REF:

			break;

		case CY22393_FREQ_SRC_PLL1:

			osc->data._40	= pll->q;
			break;

		case CY22393_FREQ_SRC_PLL2:

			osc->data._11	= pll->q;
			break;

		case CY22393_FREQ_SRC_PLL3:

			osc->data._14	= pll->q;
			break;
	}
}



/******************************************************************************
*
*	Function:	_pll_config_queue
*
*	Purpose:
*
*		Queue program changes needed to program the oscillator with the given
*		PLL's configuration.
*
*	Arguments:
*
*		osc		The structure for the oscillator to access.
*
*		pll		The PLL to access.
*
*	Returned:
*
*		None.
*
******************************************************************************/

static void _pll_config_queue(osc_cy22393_t* osc, osc_cy22393_pll_t* pll)
{
	_pll_enable_queue(osc, pll);
	_pll_p_queue(osc, pll);
	_pll_p0_queue(osc, pll);
	_pll_lf_queue(osc, pll);
	_pll_q_queue(osc, pll);
}



/******************************************************************************
*
*	Function:	_pll_free_queue
*
*	Purpose:
*
*		Free the given clock's PLL from its use. If noone else is using the
*		PLL, then disable it.
*
*	Arguments:
*
*		osc		The structure for the oscillator to access.
*
*		clock	The structure for the clock to access.
*
*	Returned:
*
*		None.
*
******************************************************************************/

static void _pll_free_queue(osc_cy22393_t* osc, osc_cy22393_clock_t* clock)
{
	int					i;
	int					qty;
	osc_cy22393_pll_t*	pll		= NULL;

	if (clock->freq_src != CY22393_FREQ_SRC_REF)
	{
		switch (clock->freq_src)
		{
			default:
			case CY22393_FREQ_SRC_REF:

				pll	= NULL;
				break;

			case CY22393_FREQ_SRC_PLL1:

				pll	= &osc->pll[0];
				break;

			case CY22393_FREQ_SRC_PLL2:

				pll	= &osc->pll[1];
				break;

			case CY22393_FREQ_SRC_PLL3:

				pll	= &osc->pll[2];
				break;
		}

		if (pll)
		{
			// See how many channels are using the same PLL.

			for (qty = 0, i = 0; i <= 3; i++)
			{
				if (osc->clock[i].freq_src == clock->freq_src)
				{
					qty++;
				}
			}

			if (qty == 1)
			{
				// This is the only channel using this PLL so disable the PLL.
				pll->enable		= 0;
				pll->freq_got	= 0;
				pll->lf			= 0;
				pll->p			= 0;
				pll->p0			= 0;
				pll->q			= 0;
				_pll_config_queue(osc, pll);
			}
		}

		clock->freq_src	= CY22393_FREQ_SRC_REF;
		_clock_source_queue(osc, clock);
	}
}



/******************************************************************************
*
*	Function:	_pll_locate_free
*
*	Purpose:
*
*		Locate a PLL that is not in use by any output clock.
*
*	Arguments:
*
*		osc		The oscillator data structure.
*
*	Returned:
*
*		>= 0	The index of the PLL to use.
*		< 0		All PLLs are in use.
*
******************************************************************************/

static int _pll_locate_free(osc_cy22393_t* osc)
{
	int	clock;
	int	pll;

	for (pll = 0; pll<= 2; pll++)
	{
		for (clock = 4; clock--;)
		{
			if (osc->clock[clock].freq_src == osc->pll[pll].freq_src)
				break;
		}

		if (clock < 0)
			break;
	}

	if (pll > 2)
		pll	= -1;

	return(pll);
}



/******************************************************************************
*
*	Function:	_clock_source_queue
*
*	Purpose:
*
*		Queue program changes needed to program the Frequency Source for the
*		given output clock.
*
*	Arguments:
*
*		osc		The structure for the oscillator to access.
*
*		clock	The structure for the clock to access.
*
*	Returned:
*
*		None.
*
******************************************************************************/

static void _clock_source_queue(osc_cy22393_t* osc, osc_cy22393_clock_t* clock)
{
	u32	source;

	switch (clock->freq_src)
	{
		default:

			clock->freq_src	= CY22393_FREQ_SRC_REF;

		case CY22393_FREQ_SRC_REF:

			source	= clock->freq_src;
			break;

		case CY22393_FREQ_SRC_PLL1:
		case CY22393_FREQ_SRC_PLL2:
		case CY22393_FREQ_SRC_PLL3:

			source	= clock->freq_src | (clock->id & 0x1);
			break;
	}

	switch (clock->id)
	{
		default:

			break;

		case CY22393_CLOCK_A:

			osc->data._08	&= ~0x80;
			osc->data._08	|= (source & 0x1) << 7;
			osc->data._0E	&= ~0x03;
			osc->data._0E	|= (source & 0x6) >> 1;
			break;

		case CY22393_CLOCK_B:

			osc->data._0A	&= ~0x80;
			osc->data._0A	|= (source & 0x1) << 7;
			osc->data._0E	&= ~0x0C;
			osc->data._0E	|= (source & 0x6) << 1;
			break;

		case CY22393_CLOCK_C:

			osc->data._0C	&= ~0x80;
			osc->data._0C	|= (source & 0x1) << 7;
			osc->data._0E	&= ~0x30;
			osc->data._0E	|= (source & 0x6) << 3;
			break;

		case CY22393_CLOCK_D:

			osc->data._0D	&= ~0x80;
			osc->data._0D	|= (source & 0x1) << 7;
			osc->data._0E	&= ~0xC0;
			osc->data._0E	|= (source & 0x6) << 5;
			break;
	}
}



/******************************************************************************
*
*	Function:	_clock_post_divide_queue
*
*	Purpose:
*
*		Program in the current Post Divider value.
*
*	Arguments:
*
*		osc		The structure for the oscillator to access.
*
*		clock	The structure for the clock to access.
*
*	Returned:
*
*		None.
*
******************************************************************************/

static void _clock_post_divide_queue(
	osc_cy22393_t*			osc,
	osc_cy22393_clock_t*	clock)
{
	switch (clock->id)
	{
		default:

			break;

		case CY22393_CLOCK_A:

			osc->data._08	&= ~0x7F;
			osc->data._08	|= clock->post_div & 0x7F;
			break;

		case CY22393_CLOCK_B:

			osc->data._0A	&= ~0x7F;
			osc->data._0A	|= clock->post_div & 0x7F;
			break;

		case CY22393_CLOCK_C:

			osc->data._0C	&= ~0x7F;
			osc->data._0C	|= clock->post_div & 0x7F;
			break;

		case CY22393_CLOCK_D:

			osc->data._0D	&= ~0x7F;
			osc->data._0D	|= clock->post_div & 0x7F;
			break;
	}
}



/******************************************************************************
*
*	Function:	_clock_disable_queue
*
*	Purpose:
*
*		Disable the given clock's output.
*
*	Arguments:
*
*		osc		The structure for the oscillator to access.
*
*		clock	The structure for the clock to access.
*
*	Returned:
*
*		None.
*
******************************************************************************/

static void _clock_disable_queue(
	osc_cy22393_t*			osc,
	osc_cy22393_clock_t*	clock)
{
	clock->post_div	= 0;
	_clock_post_divide_queue(osc, clock);
}



/******************************************************************************
*
*	Function:	_clock_config_queue
*
*	Purpose:
*
*		Program the oscillator with the given clock's configuration.
*
*	Arguments:
*
*		osc		The structure for the oscillator to access.
*
*		clock	The structure for the clock to access.
*
*	Returned:
*
*		None.
*
******************************************************************************/

static void _clock_config_queue(osc_cy22393_t* osc, osc_cy22393_clock_t* clock)
{
	_clock_source_queue(osc, clock);
	_clock_post_divide_queue(osc, clock);
}



/******************************************************************************
*
*	Function:	_clock_use_existing_pll
*
*	Purpose:
*
*		Try to configure the given clock to use a PLL that is already in use.
*
*	Arguments:
*
*		osc			The oscillator data structure.
*
*		clock		The output clock to access.
*
*		freq_want	The frequency desired for the given output clock.
*
*	Returned:
*
*		None.
*
******************************************************************************/

static void _clock_use_existing_pll(
	osc_test_t*				osc,
	osc_cy22393_clock_t*	clock,
	u32						freq_want)
{
	long					delta;
	long					delta1;
	long					delta2;
	int						i;
	long					got;
	long					got1;
	long					got2;
	long					got_best	= 0;
	long					pd;
	long					pd1;
	long					pd2;
	long					pd_best		= 0;
	osc_cy22393_pll_t*		pll			= NULL;
	osc_cy22393_clock_t*	src			= &osc->osc.clock[0];
	osc_cy22393_pll_t*		use			= NULL;

	osc->found	= 0;

	for (i = 0; i <= 3; i++, src++)
	{
		if (src->in_use == 0)
			continue;

		switch (src->freq_src)
		{
			default:
			case CY22393_FREQ_SRC_REF:

				continue;

			case CY22393_FREQ_SRC_PLL1:

				pll	= &osc->osc.pll[0];
				break;

			case CY22393_FREQ_SRC_PLL2:

				pll	= &osc->osc.pll[1];
				break;

			case CY22393_FREQ_SRC_PLL3:

				pll	= &osc->osc.pll[2];
				break;
		}

		pd1		= (long) pll->freq_got / (long) freq_want;	// rounds down
		pd2		= pd1 + 1;							// round up

		pd1		= LIMIT_RANGE(pd1, PD_MIN, PD_MAX);
		pd2		= LIMIT_RANGE(pd2, PD_MIN, PD_MAX);

		got1	= osc->osc.freq_ref / pd1;
		got2	= osc->osc.freq_ref / pd2;

		delta1	= got1 - (long) freq_want;
		delta2	= got2 - (long) freq_want;

		delta1	= (delta1 < 0) ? -delta1 : delta1;
		delta2	= (delta2 < 0) ? -delta2 : delta2;

		if (delta1 < delta2)
		{
			pd		= pd1;
			got		= got1;
			delta	= delta1;
		}
		else
		{
			pd		= pd2;
			got		= got2;
			delta	= delta2;
		}

		if ((osc->found == 0) || (delta < osc->delta))
		{
			osc->found		= 1;
			osc->delta		= delta;
			use				= pll;
			got_best		= got;
			pd_best			= pd;
		}
	}

	if (use)
	{
		clock->in_use		= 1;
		clock->freq_want	= freq_want;
		clock->freq_got		= got_best;
		clock->freq_src		= use->freq_src;
		clock->post_div		= (u8) pd_best;;
		_clock_config_queue(&osc->osc, clock);
	}
}



/******************************************************************************
*
*	Function:	_clock_use_available_pll
*
*	Purpose:
*
*		Acquire an available PLL and configure things for the desired
*		frequence.
*
*	Arguments:
*
*		osc			The oscillator data structure.
*
*		clock		The output clock to access.
*
*		freq_want	The frequency desired for the given output clock.
*
*	Returned:
*
*		None.
*
******************************************************************************/

static void _clock_use_available_pll(
	osc_test_t*				osc,
	osc_cy22393_clock_t*	clock,
	u32						freq_want)
{
	long				delta;
	gen_t				gen;
	int					i;
	osc_cy22393_pll_t*	pll;

	osc->found	= 0;
	i			= _pll_locate_free(&osc->osc);

	if (i >= 0)
	{
		memset(&gen, 0, sizeof(gen_t));
		gen.freq_ref	= osc->osc.freq_ref;
		gen.freq_want	= freq_want;
		_compute_dp_p_p0_q_lf(&gen);

		if (gen.freq_got)
		{
			pll					= &osc->osc.pll[i];
			pll->enable			= 1;
			pll->freq_got		= gen.freq_pll;
			pll->p				= (u16) gen.p;
			pll->p0				= (u8) gen.p0;
			pll->lf				= (u8) gen.lf;
			pll->q				= (u8) gen.q;

			clock->freq_got		= gen.freq_got;
			clock->freq_src		= pll->freq_src;
			clock->freq_want	= freq_want;
			clock->in_use		= 1;
			clock->post_div		= (u8) gen.pd;

			_clock_config_queue(&osc->osc, clock);
			_pll_config_queue(&osc->osc, pll);

			osc->found	= 1;
			delta		= freq_want - gen.freq_got;
			delta		= (delta < 0) ? -delta : delta;
			osc->delta	= delta;
		}
	}
}



/******************************************************************************
*
*	Function:	_clock_use_existing_clock
*
*	Purpose:
*
*		Examine the existing clock configurations and pick the one that will
*		give the best results.
*
*	Arguments:
*
*		osc			The oscillator data structure.
*
*		clock		The output clock to access.
*
*		freq_want	The frequency desired for the given output clock.
*
*	Returned:
*
*		None.
*
******************************************************************************/

static void _clock_use_existing_clock(
	osc_test_t*				osc,
	osc_cy22393_clock_t*	clock,
	u32						freq_want)
{
	long					delta	= 0;
	int						i;
	osc_cy22393_clock_t*	src		= &osc->osc.clock[0];
	osc_cy22393_clock_t*	use		= NULL;

	osc->found	= 0;

	for (i = 0; i <= 3; i++, src++)
	{
		if (src->in_use == 0)
			continue;

		delta	= (long) src->freq_got - (long) freq_want;
		delta	= (delta < 0) ? - delta : delta;

		if ((osc->found == 0) || (delta < osc->delta))
		{
			osc->found	= 1;
			osc->delta	= delta;
			use			= src;
		}
	}

	if (use)
	{
		clock->in_use		= 1;
		clock->freq_want	= freq_want;
		clock->freq_got		= use->freq_got;
		clock->freq_src		= use->freq_src;
		clock->post_div		= use->post_div;
		_clock_config_queue(&osc->osc, clock);
	}
}



/******************************************************************************
*
*	Function:	_clock_use_freq_ref
*
*	Purpose:
*
*		Configure the given clock to use the reference source. Choose the
*		configuration that will give the best results.
*
*	Arguments:
*
*		osc			The oscillator data structure.
*
*		clock		The output clock to access.
*
*		freq_want	The frequency desired for the given output clock.
*
*	Returned:
*
*		None.
*
******************************************************************************/

static void _clock_use_freq_ref(
	osc_test_t*				osc,
	osc_cy22393_clock_t*	clock,
	u32						freq_want)
{
	long	delta;
	long	delta1;
	long	delta2;
	long	got;
	long	got1;
	long	got2;
	long	pd;
	long	pd1;
	long	pd2;

	pd1		= (long) osc->osc.freq_ref / (long) freq_want;	// rounds down
	pd2		= pd1 + 1;										// round up

	pd1		= LIMIT_RANGE(pd1, PD_MIN, PD_MAX);
	pd2		= LIMIT_RANGE(pd2, PD_MIN, PD_MAX);

	got1	= osc->osc.freq_ref / pd1;
	got2	= osc->osc.freq_ref / pd2;

	delta1	= got1 - (long) freq_want;
	delta2	= got2 - (long) freq_want;

	delta1	= (delta1 < 0) ? -delta1 : delta1;
	delta2	= (delta2 < 0) ? -delta2 : delta2;

	if (delta1 < delta2)
	{
		pd		= pd1;
		got		= got1;
		delta	= delta1;
	}
	else
	{
		pd		= pd2;
		got		= got2;
		delta	= delta2;
	}

	osc->found			= 1;
	osc->delta			= delta;
	clock->in_use		= 1;
	clock->freq_want	= freq_want;
	clock->freq_got		= got;
	clock->freq_src		= CY22393_FREQ_SRC_REF;
	clock->post_div		= (u8) pd;
	_clock_config_queue(&osc->osc, clock);
}



/******************************************************************************
*
*	Function:	_compute_osc_drv_queue
*
*	Purpose:
*
*		Compute Osc_Drv value based on the reference frequency and record the
*		needed data setting.
*
*	Arguments:
*
*		osc		The structure for the oscillator to access.
*
*	Returned:
*
*		None.
*
******************************************************************************/

static void _compute_osc_drv_queue(osc_cy22393_t* osc)
{
	u8	osc_drv;

	if (osc->freq_ref <= 25000000L)
		osc_drv	= 0;
	else if (osc->freq_ref <= 50000000L)
		osc_drv	= 1;
	else if (osc->freq_ref <= 90000000L)
		osc_drv	= 2;
	else
		osc_drv	= 3;

	osc->data._17	&= ~0xFC;
	osc->data._17	|= osc_drv;
}



/******************************************************************************
*
*	Function:	osc_cy22393_info
*
*	Purpose:
*
*		Provide information on the referenced output clock.
*
*	Arguments:
*
*		osc		The structure for the oscillator to access.
*
*		index	The oscillator index to access.
*
*		arg		Data is exchanged with the caller here.
*
*	Returned:
*
*		0		Success.
*		< 0		An appropriate error status.
*
******************************************************************************/

int osc_cy22393_info(osc_cy22393_t* osc, int index, osc_cy22393_arg_t* arg)
{
	osc_cy22393_clock_t*	clock;
	int						ret;

	if (INDEX_IS_INVALID(index))
	{
		ret	= -EINVAL;
	}
	else
	{
		ret				= 0;
		clock			= &osc->clock[index];
		arg->freq_ref	= osc->freq_ref;
		arg->freq_want	= clock->freq_want;
		arg->freq_got	= clock->freq_got;
	}

	return(ret);
}



/******************************************************************************
*
*	Function:	osc_cy22393_program
*
*	Purpose:
*
*		Program the given oscillator to produce the given frequence for the
*		given serial channel.
*
*	Arguments:
*
*		osc		The structure for the oscillator to access.
*
*		index	The oscillator index to access.
*
*		arg		Data is exchanged with the caller here.
*
*	Returned:
*
*		0		Success.
*		< 0		An appropriate error status.
*
******************************************************************************/

int osc_cy22393_program(osc_cy22393_t* osc, int index, osc_cy22393_arg_t* arg)
{
	osc_cy22393_clock_t*	clock;
	u32						freq_orig	= (u32) arg->freq_want;
	u32						freq_want	= (u32) arg->freq_want;
	int						ret;
	osc_test_t				test;
	osc_test_t				use;

	if (freq_want < OSC_FREQ_MIN)
		freq_want	= OSC_FREQ_MIN;
	else if (freq_want > OSC_FREQ_MAX)
		freq_want	= OSC_FREQ_MAX;

	if (INDEX_IS_INVALID(index))
	{
		ret	= -EINVAL;
	}
	else
	{
		ret		= 0;
		clock	= &osc->clock[index];
		_compute_osc_drv_queue(osc);
		_clock_disable_queue(osc, clock);
		_pll_free_queue(osc, clock);
		clock->in_use		= 0;
		clock->freq_want	= 0;
		clock->freq_got		= 0;

		// Use the reference frequency.
		memset(&use, 0, sizeof(use));
		use.osc	= osc[0];
		clock	= &use.osc.clock[index];
		_clock_use_freq_ref(&use, clock, freq_want);

		if (use.delta)
		{
			// Try an existing clock configuration.
			memset(&test, 0, sizeof(test));
			test.osc	= osc[0];
			clock		= &test.osc.clock[index];
			_clock_use_existing_clock(&test, clock, freq_want);

			if ((test.found) && (test.delta < use.delta))
				use	= test;
		}

		if (use.delta)
		{
			// Try an existing PLL with a new clock configuration.
			memset(&test, 0, sizeof(test));
			test.osc	= osc[0];
			clock		= &test.osc.clock[index];
			_clock_use_existing_pll(&test, clock, freq_want);

			if ((test.found) && (test.delta < use.delta))
				use	= test;
		}

		if (use.delta)
		{
			// Try an unused PLL and configure the best we can.
			memset(&test, 0, sizeof(test));
			test.osc	= osc[0];
			clock		= &test.osc.clock[index];
			_clock_use_available_pll(&test, clock, freq_want);

			if ((test.found) && (test.delta < use.delta))
				use	= test;
		}

		// Record the results found.
		osc[0]	= use.osc;

		memset(arg, 0, sizeof(arg[0]));
		arg->freq_ref	= osc->freq_ref;
		arg->freq_want	= freq_orig;	// freq_want;
		arg->freq_got	= clock->freq_got;
	}

	return(ret);
}



/******************************************************************************
*
*	Function:	osc_cy22393_reference
*
*	Purpose:
*
*		Tell the oscillator code the channel's reference frequency. If the
*		specified reference frequency is zero, then we report the current
*		setting.
*
*	Arguments:
*
*		osc		The structure for the oscillator to access.
*
*		index	The oscillator index to access.
*
*		arg		Data is exchanged with the caller here.
*
*	Returned:
*
*		>= 0	Success.
*		< 0		Try another configuration option.
*
******************************************************************************/

int osc_cy22393_reference(
	osc_cy22393_t*		osc,
	int					index,
	osc_cy22393_arg_t*	arg)
{
	osc_cy22393_clock_t*	clock;
	int						ret;

	if (INDEX_IS_INVALID(index))
	{
		ret	= -EINVAL;
	}
	else
	{
		clock	= &osc->clock[index];

		if (arg->freq_ref == -1)
		{
			ret	= 0;
		}
		else if (	(arg->freq_ref < FREQ_REF_MIN) ||
					(arg->freq_ref > FREQ_REF_MAX))
		{
			ret	= -EINVAL;
		}
		else
		{
			osc->freq_ref	= arg->freq_ref;
			ret				= 0;
			_compute_osc_drv_queue(osc);
		}

		if (ret == 0)
		{
			arg->freq_ref	= osc->freq_ref;
			arg->freq_want	= clock->freq_want;
			arg->freq_got	= clock->freq_got;
		}
	}

	return(ret);
}



/******************************************************************************
*
*	Function:	osc_cy22393_reset
*
*	Purpose:
*
*		Reset the referenced output clock.
*
*	Arguments:
*
*		osc		The structure for the oscillator to access.
*
*		index	The oscillator index to access.
*
*		arg		Data is exchanged with the caller here.
*
*	Returned:
*
*		0		Success.
*		< 0		An appropriate error status.
*
******************************************************************************/

int osc_cy22393_reset(osc_cy22393_t* osc, int index, osc_cy22393_arg_t* arg)
{
	osc_cy22393_clock_t*	clock;
	int						ret;

	if (INDEX_IS_INVALID(index))
	{
		ret	= -EINVAL;
	}
	else
	{
		ret	= 0;
		clock	= &osc->clock[index];
		_compute_osc_drv_queue(osc);
		_clock_disable_queue(osc, clock);
		clock->in_use	= 0;
		_pll_free_queue(osc, clock);

		clock->freq_got		= 0;
		clock->freq_src		= CY22393_FREQ_SRC_REF;
		clock->freq_want	= 0;
		clock->post_div		= 0;

		_clock_config_queue(osc, clock);

		memset(arg, 0, sizeof(arg[0]));
		arg->freq_ref	= osc->freq_ref;
		arg->freq_want	= clock->freq_want;
		arg->freq_got	= clock->freq_got;
	}

	return(ret);
}



/******************************************************************************
*
*	Function:	osc_cy22393_startup
*
*	Purpose:
*
*		Perform a one-time initialization of the oscillator.
*
*	Arguments:
*
*		osc		The oscillator structure to access.
*
*		arg		We receive initial data here.
*
*	Returned:
*
*		None.
*
******************************************************************************/

void osc_cy22393_startup(osc_cy22393_t* osc, const osc_cy22393_arg_t* arg)
{
	memset(osc, 0, sizeof(osc[0]));
	osc->freq_ref	= arg->freq_ref;
	_compute_osc_drv_queue(osc);

	osc->clock[0].freq_src	= CY22393_FREQ_SRC_REF;
	osc->clock[0].id		= CY22393_CLOCK_A;
	osc->clock[0].post_div	= 1;

	osc->clock[1].freq_src	= CY22393_FREQ_SRC_REF;
	osc->clock[1].id		= CY22393_CLOCK_B;
	osc->clock[1].post_div	= 1;

	osc->clock[2].freq_src	= CY22393_FREQ_SRC_REF;
	osc->clock[2].id		= CY22393_CLOCK_C;
	osc->clock[2].post_div	= 1;

	osc->clock[3].freq_src	= CY22393_FREQ_SRC_REF;
	osc->clock[3].id		= CY22393_CLOCK_D;
	osc->clock[3].post_div	= 1;

	osc->pll[0].freq_src	= CY22393_FREQ_SRC_PLL1;
	osc->pll[1].freq_src	= CY22393_FREQ_SRC_PLL2;
	osc->pll[2].freq_src	= CY22393_FREQ_SRC_PLL3;
}



