// $URL: http://subversion:8080/svn/gsc/trunk/drivers/LINUX/SIO4%20and%20SIO8/SIO4_Linux_1.x.x_GSC_DN/sync/samples/syncc2c/utils.c $
// $Rev: 53085 $
// $Date: 2023-06-13 10:11:43 -0500 (Tue, 13 Jun 2023) $

// SIO4: SYNC Protocol Library: Sample Application: source file

#include "main.h"



//*****************************************************************************
static int _channel_setup_common(
	int				fd,
	const char*		title,
	int				dce,
	sio4_mp_prot_t	mp,
	int				lb,
	int				bit_rate)
{
	char		buf[64];
	int			clock	= bit_rate * 2;
	int			errs	= 0;
	__s32		got;
	int			status;
	sio4_sync_t	sync;

	for (;;)	// A convenience loop.
	{
		sprintf(buf, "Common Setup (%s)", title);
		gsc_label(buf);

		errs	+= osc_init(fd, NULL, 1);
		errs	+= osc_program(fd, clock, NULL, 1);

		if (errs)
			break;

		memset(&sync, 0, sizeof(sync));
		sync.dce_enable		= dce ? 1 : 0;
		sync.dte_enable		= dce ? 0 : 1;
		sync.lb.enable		= lb;
		sync.lb.internal	= 0;
		status				= sio4_sync_set(fd, &sync);

		if (status)
		{
			errs	= 1;
			printf("FAIL <--- (sio4_sync_set())\n");
			break;
		}

		if (mp != SIO4_MP_PROT_READ)
		{
			status	= sio4_mp_config(fd, mp, &got, 0);

			if (status)
			{
				errs	= 1;
				printf("FAIL <--- (sio4_mp_config())\n");
				break;
			}

			if (mp != got)
			{
				errs	= 1;
				printf(	"FAIL <---  (MP: wanted %d, got %d)\n",
						(int) mp,
						(int) got);
				break;
			}
		}

		printf("PASS");

		printf("  (");
		gsc_label_long_comma(clock);
		printf(" Hz clock -> ");
		gsc_label_long_comma(clock / 2);
		printf(" bits/sec)\n");
		break;
	}

	return(errs);
}



//*****************************************************************************
static int _channel_setup_rx(const args_t* args)
{
	int				errs	= 0;
	sio4_sync_rx_t	rx;
	int				status;

	for (;;)	// A convenience loop.
	{
		gsc_label("Rx Setup");
		errs	+= fifo_reset(args->fd_rx, RX_FIFO, 1);

		memset(&rx, 0, sizeof(rx));
		rx.enable		= 1;
		rx.lsbf			= 0;
		rx.clock.high	= 0;
		rx.env.enable	= 1;
		rx.env.high		= 1;
		rx.data.high	= 1;
		status			= sio4_sync_rx_set(args->fd_rx, &rx);

		if (status)
		{
			errs	= 1;
			printf("FAIL <--- (sio4_sync_rx_set())\n");
			break;
		}

		errs	+= rx_io_mode_config(args->fd_rx, SIO4_IO_MODE_PIO, NULL, 1);
		errs	+= timeout_set_read(args->fd_rx, 0, 1);	// zero timeout with PIO

		// Make sure everything is flushed before we procede.
		os_sleep_ms(10);
		errs	+= fifo_reset(args->fd_rx, RX_FIFO, 1);

		if (errs == 0)
			printf("PASS\n");

		break;
	}

	return(errs);
}



//*****************************************************************************
static int _channel_setup_tx(const args_t* args)
{
	int				errs	= 0;
	int				status;
	sio4_sync_tx_t	tx;

	for (;;)	// A convenience loop.
	{
		gsc_label("Tx Setup");
		errs	+= fifo_reset(args->fd_tx, TX_FIFO, 1);

		memset(&tx, 0, sizeof(tx));
		tx.enable			= 1;
		tx.auto_dis			= 0;
		tx.lsbf				= 0;
		tx.word_size		= 8;
		tx.gap_size			= 0;
		tx.clock.clock		= 1;
		tx.clock.ext		= 0;
		tx.clock.high		= 1;
		tx.clock.idle		= 0;
		tx.env.env			= 1;
		tx.env.high			= 1;
		tx.data.data		= 1;
		tx.data.high		= 1;
		tx.aux_clock.enable	= 0;
		tx.aux_clock.clock	= 0;
		tx.aux_clock.high	= 0;
		tx.spare.enable		= 0;
		tx.spare.high		= 0;
		status				= sio4_sync_tx_set(args->fd_tx, &tx);

		if (status)
		{
			errs	= 1;
			printf("FAIL <--- (sio4_sync_tx_set())\n");
			break;
		}

		errs	+= tx_io_mode_config(args->fd_tx, SIO4_IO_MODE_PIO, NULL, 1);
		errs	+= timeout_set_write(args->fd_tx, 0, 1);	// zero timeout with PIO

		// Make sure everything is flushed before we procede.
		os_sleep_ms(10);
		errs	+= fifo_reset(args->fd_tx, TX_FIFO, 1);

		if (errs == 0)
			printf(	"PASS\n");

		break;
	}

	return(errs);
}



//*****************************************************************************
static int _fifo_size_rx(const args_t* args, int* size)
{
	int	errs;

	for (;;)	// A convenience loop.
	{
		gsc_label("Rx FIFO Size");
		errs	= rx_fifo_size(args->fd_rx, size, 0);

		if ((errs == 0) && (size[0] == 0))
			errs	= rx_fifo_size_compute(args->fd_rx, size, 0);

		if (errs)
		{
			size[0]	= 512;
			printf("FAIL <---  (assuming 512 bytes)\n");
			break;
		}

		if (size[0] == 0)
		{
			size[0]	= 512;
			printf("PASS  (size unknown, assuming 512 Bytes)\n");
		}
		else if (size[0] % 1024)
		{
			printf("PASS  (%d Bytes)\n", size[0]);
		}
		else
		{
			printf("PASS  (%dK Bytes)\n", size[0] / 1024);
		}

		break;
	}

	return(errs);
}



//*****************************************************************************
static int _fifo_size_tx(const args_t* args, int* size)
{
	int	errs;

	for (;;)	// A convenience loop.
	{
		gsc_label("Tx FIFO Size");
		errs	= tx_fifo_size(args->fd_tx, size, 0);

		if ((errs == 0) && (size[0] == 0))
			errs	= tx_fifo_size_compute(args->fd_tx, size, 0);

		if (errs)
		{
			size[0]	= 512;
			printf("FAIL <---  (assuming 512 bytes)\n");
			break;
		}

		if (size[0] == 0)
		{
			size[0]	= 512;
			printf("PASS  (size unknown, assuming 512 Bytes)\n");
		}
		else if (size[0] % 1024)
		{
			printf("PASS  (%d Bytes)\n", size[0]);
		}
		else
		{
			printf("PASS  (%dK Bytes)\n", size[0] / 1024);
		}

		break;
	}

	return(errs);
}



//*****************************************************************************
static int _fw_type_validate(int fd, const char* desc, int* skip)
{
	char	buf[80];
	int		errs	= 0;
	int		i;
	s32		set;

	sprintf(buf, "%sFirmware Type", desc);
	gsc_label(buf);

	set	= SIO4_FW_TYPE_CONFIG_READ;
	i	= ioctl(fd, SIO4_FW_TYPE_CONFIG, &set);

	if (i == -1)
	{
		printf("FAIL <---\n");
		errs++;
	}
	else
	{
		switch (set)
		{
			default:

				errs++;
				printf("FAIL <---  (Unknown firmware type: %ld)\n", (long) set);
				break;

			case SIO4_FW_TYPE_CONFIG_SYNC:

				printf("PASS  (SYNC)\n");
				break;

			case SIO4_FW_TYPE_CONFIG_Z16C30:

				skip[0]	= 1;
				printf("SKIP <---  (Z16C30, wrong firmware type)\n");
				break;
		}
	}

	return(errs);
}



//*****************************************************************************
static int _mp_check(int fd, mp_t* mp)
{
	int	errs	= 0;

	errs	+= feature_test__mp(fd, &mp->mp, 0);
	errs	+= mp_test(fd, SIO4_MP_PROT_RS_232, &mp->_232, 0);
	errs	+= mp_test(fd, SIO4_MP_PROT_RS_422_485, &mp->_422, 0);
	errs	+= mp_test(fd, SIO4_MP_PROT_RS_423, &mp->_423, 0);
	return(errs);
}



//*****************************************************************************
int channels_reset(const args_t* args)
{
	int	errs	= 0;

	errs	+= channel_reset(args->fd_tx, 1);
	errs	+= channel_reset(args->fd_rx, 1);
	return(errs);
}



//*****************************************************************************
void connection_type(const args_t* args)
{
	int	rx_b	= args->index_rx / 4;
	int	rx_c	= args->index_rx % 4;
	int	tx_b	= args->index_tx / 4;
	int	tx_c	= args->index_tx % 4;

	gsc_label("Connection Type");

	if (args->index_tx == args->index_rx)
	{
		printf("External Loopback (no cable)\n");
	}
	else if (tx_b == rx_b)
	{
		printf(	"Channel-to-Channel Loopback Cable"
				" (B# %d, C#s %d+%d)\n",
				tx_b,
				tx_c,
				rx_c);
	}
	else
	{
		printf(	"Channel-to-Channel Passthrough Cable"
				" (B%d C%d, B%d C%d)\n",
				tx_b,
				tx_c,
				rx_b,
				rx_c);
	}
}



//*****************************************************************************
int fifo_sizes_compute(const args_t* args, int* fifo)
{
	int	errs	= 0;
	s32	fifo_rx;
	s32	fifo_tx;

	gsc_label("FIFO Sizes");
	printf("\n");
	gsc_label_level_inc();

	errs	+= _fifo_size_tx(args, &fifo_tx);
	errs	+= _fifo_size_rx(args, &fifo_rx);

	if ((fifo_tx < 512) || (fifo_rx < 512))
		fifo[0]	= 512;	// Presume the FIFO is at least this size.
	else if (fifo_tx < fifo_rx)
		fifo[0]	= fifo_tx;
	else
		fifo[0]	= fifo_rx;

	gsc_label("Results");
	printf("%s  (", errs ? "FAIL <---" : "PASS");

	if ((fifo[0]) && (fifo[0] % 1024))
		printf("%d", fifo[0]);
	else
		printf("%dK", fifo[0] / 1024);

	printf(" Bytes)\n");

	gsc_label_level_dec();
	return(errs);
}



//*****************************************************************************
int fw_type_validate(const args_t* args, int* skip)
{
	int	errs	= 0;

	if (args->index_tx == args->index_rx)
	{
		errs	+= _fw_type_validate(args->fd_tx, "", skip);
	}
	else
	{
		gsc_label("Firmware Types");
		printf("\n");
		gsc_label_level_inc();

		errs	+= _fw_type_validate(args->fd_tx, "Tx ", skip);
		errs	+= _fw_type_validate(args->fd_rx, "Rx ", skip);

		gsc_label_level_dec();
	}

	return(errs);
}



//*****************************************************************************
int rx_setup(const args_t* args, int lb, sio4_mp_prot_t mp)
{
	int	errs	= 0;

	errs	+= _channel_setup_common(args->fd_rx, "Rx", 0, mp, lb, args->baud);
	errs	+= _channel_setup_rx(args);
	return(errs);
}



//*****************************************************************************
int tx_setup(const args_t* args, int lb, sio4_mp_prot_t mp)
{
	int	errs	= 0;

	errs	+= _channel_setup_common(args->fd_tx, "Tx", 1, mp, lb, args->baud);
	errs	+= _channel_setup_tx(args);
	return(errs);
}



//*****************************************************************************
int xcvr_compute(const args_t* args, sio4_mp_prot_t* mp)
{
	int		errs	= 0;
	mp_t	mp_rx;
	mp_t	mp_tx;

	mp[0]	= SIO4_MP_PROT_RS_422_485;

	for (;;)	// A convenience loop.
	{
		gsc_label("Transceiver Setup");

		errs	+= _mp_check(args->fd_tx, &mp_tx);
		errs	+= _mp_check(args->fd_rx, &mp_rx);

		if (errs)
		{
			printf("FAIL <---\n");
			break;
		}

		if ((SIO4_FEATURE_NO == mp_tx.mp) ||
			(SIO4_FEATURE_NO == mp_rx.mp))
		{
			// The user must make sure the devices match.
			mp[0]	= SIO4_MP_PROT_READ;
			printf("SKIPPED  (insufficient support)\n");
			break;
		}

		if ((SIO4_MP_PROT_RS_422_485 == mp_tx._422) &&
			(SIO4_MP_PROT_RS_422_485 == mp_rx._422))
		{
			errs	= 0;
			mp[0]	= SIO4_MP_PROT_RS_422_485;
			printf("PASS  (RS-422/485)\n");
			break;
		}

		if ((SIO4_MP_PROT_RS_423 == mp_tx._423) &&
			(SIO4_MP_PROT_RS_423 == mp_rx._423))
		{
			errs	= 0;
			mp[0]	= SIO4_MP_PROT_RS_423;
			printf("PASS  (RS-423)\n");
			break;
		}

		if ((SIO4_MP_PROT_RS_232 == mp_tx._232) &&
			(SIO4_MP_PROT_RS_232 == mp_rx._232))
		{
			errs	= 0;
			mp[0]	= SIO4_MP_PROT_RS_232;
			printf("PASS  (RS-232)\n");
			break;
		}

		errs++;
		mp[0]	= SIO4_MP_PROT_INVALID;
		printf("FAIL <---  (insufficient support)\n");
		break;
	}

	return(errs);
}


