// $URL: http://subversion:8080/svn/gsc/trunk/drivers/LINUX/SIO4%20and%20SIO8/SIO4_Linux_1.x.x_GSC_DN/samples/sbtest/osc_measure.c $
// $Rev: 53081 $
// $Date: 2023-06-13 10:10:10 -0500 (Tue, 13 Jun 2023) $

// SIO4: Sample Application: source file

#include "main.h"



// macros *********************************************************************

#define	TEST_CHIP(w,o,l)					_value_test("chip",(w),(o)->chip,(l))
#define	TEST_FREQ_GOT(w,o,l)				_value_test("freq_got",(w),(o)->freq_got,(l))
#define	TEST_FREQ_REF(w,o,l)				_value_test("freq_ref",(w),(o)->freq_ref,(l))
#define	TEST_FREQ_WANT(w,o,l)				_value_test("freq_want",(w),(o)->freq_want,(l))



//*****************************************************************************
static int _service_test(int fd)
{
	return(0);
}



//*****************************************************************************
static int _value_test(const char* item, u32 want, u32 got, int line)
{
	int	errs	= 0;

	if (got != want)
	{
		errs	= 1;
		printf(	"FAIL <---  (%d. %s: got %lu, expected %lu)\n",
				line,
				item,
				(long) got,
				(long) want);
	}

	return(errs);
}



//*****************************************************************************
static int _reference_set(int fd, u32 want, int line)
{
	int			errs	= 0;
	sio4_osc_t	osc;
	int			ret;

	memset(&osc, 0, sizeof(osc));
	osc.freq_ref	= want;
	ret				= sio4_ioctl(fd, SIO4_OSC_REFERENCE, &osc);

	if (ret < 0)
	{
		errs++;
		printf(	"FAIL <---  (%d. SIO4_OSC_REFERENCE)\n",
				line);
	}
	else if (osc.freq_ref != want)
	{
		errs++;
		printf(	"FAIL <---  (%d. SIO4_OSC_REFERENCE, expected %ld, got %ld)\n",
				line,
				(long) want,
				(long) osc.freq_ref);
	}

	return(errs);
}



//*****************************************************************************
static void _osc_dump_prog_regs(int fd)
{
	static const int list[]	=
	{
		0x06,
		0x07,

		0x08,
		0x09,
		0x0A,
		0x0B,
		0x0C,
		0x0D,
		0x0E,
		0x0F,
		0x10,
		0x11,
		0x12,
		0x13,
		0x14,
		0x15,
		0x16,
		0x17,

		0x18,
		0x19,
		0x1A,
		0x1B,

		0x40,
		0x41,
		0x42,
		0x43,
		0x44,
		0x45,
		0x46,
		0x47,
		0x48,
		0x49,
		0x4A,
		0x4B,
		0x4C,
		0x4D,
		0x4E,
		0x4F,
		0x50,
		0x51,
		0x52,
		0x53,
		0x54,
		0x55,
		0x56,
		0x57,
		-1
	};

	int	i;
	u32	reg;

	printf("\n");
	sio4_reg_write(fd, SIO4_GSC_POCSR, 0x00);
	sio4_reg_read(fd, SIO4_GSC_POCSR, &reg);
	printf("Measure POCSR 0x%08lX\n", (long) reg);

	sio4_reg_write(fd, SIO4_GSC_POCSR, 0x40);
	sio4_reg_read(fd, SIO4_GSC_POCSR, &reg);
	printf("Control POCSR 0x%08lX\n", (long) reg);

	for (i = 0; list[i] >= 0; i++)
	{
		sio4_reg_write(fd, SIO4_GSC_PORAR, list[i]);
		sio4_reg_read(fd, SIO4_GSC_PORDR, &reg);

		printf("OSC Adrs 0x%02lX, Val 0x%02lX\n", (long) list[i], (long) reg);
	}
}


//*****************************************************************************
static int _osc_measure_reset_test(int fd, s32 ref, const data_t* data)
{
	s32			delta		= 0;
	int			errs		= 0;
	s32			freq_got	= 0;
	s32			freq_ref;
	int			loop;
	s32			max			= 0;
	s32			min			= 0;
	long		mmax;
	long		mmin;
	sio4_osc_t	osc;
	int			ret;

	memset(&osc, 0, sizeof(osc));
	errs	+= sio4_reset_channel(fd, 0);

	switch (data->osc_chip)
	{
		default:

			errs++;
			printf("FAIL <---  (%d. internal error)\n", __LINE__);
			break;

		case SIO4_OSC_CHIP_FIXED:

			min			= 1000000L;
			max			= 20000000L;
			delta		= 4000000L;
			freq_got	= -1;
			break;

		case SIO4_OSC_CHIP_CY22393:
		case SIO4_OSC_CHIP_CY22393_2:

			min			= ref;
			max			= ref;
			delta		= 4000000L;
			freq_got	= 0;
			break;
	}

	for (freq_ref = min; (errs == 0) && (freq_ref <= max); freq_ref += delta)
	{
		errs	+= _reference_set(fd, freq_ref, __LINE__);
		memset(&osc, 0, sizeof(osc));
		ret		= sio4_ioctl(fd, SIO4_OSC_RESET, &osc);

		if (ret < 0)
		{
			errs++;
			printf("FAIL <---  (%d. SIO4_OSC_RESET)\n", __LINE__);
			break;
		}

		// This won't pass every conceivable circumstance, but it will most instances.
		mmax	= ((osc.freq_want / 10) + 1 + ((osc.freq_want + 9999) / 10000)) * 10 + 100;
		mmin	= ((osc.freq_want / 10) - 1 - ((osc.freq_want + 9999) / 10000)) * 10 - 100;

		for (loop = 0; loop <= 20; loop++)
		{
			// Measure the frequency and verify the results.
			memset(&osc, 0, sizeof(osc));
			ret	= sio4_ioctl(fd, SIO4_OSC_MEASURE, &osc);

			if (ret < 0)
			{
				errs++;
				printf("FAIL <---  (%d. SIO4_OSC_MEASURE)\n", __LINE__);
				break;
			}

			if (osc.freq_got < mmin)
			{
//				printf(	"\n(%d. loop %d, SMALL: W %ld, G %ld, Ref %ld, mmin %ld)",
//						__LINE__, loop, (long) osc.freq_want, (long) osc.freq_got, (long) freq_ref, (long) mmin);
				continue;
			}
			else if (osc.freq_got > mmax)
			{
//				printf(	"\n(%d. loop %d, LARGE: W %ld, G %ld, Ref %ld, mmax %ld)",
//						__LINE__, loop, (long) osc.freq_want, (long) osc.freq_got, (long) freq_ref, (long) mmax);
				continue;
			}
			else
			{
				osc.freq_got	= osc.freq_want;
				break;
			}
		}

		errs	+= TEST_FREQ_GOT(freq_got, &osc, __LINE__);
		errs	+= TEST_FREQ_REF(freq_ref, &osc, __LINE__);
		errs	+= TEST_CHIP(data->osc_chip, &osc, __LINE__);
	}

	if ((errs) && (data->detail))
	{
		printf("\n%d. _osc_measure_reset_test:\n", __LINE__);
		printf(	"want %ld, got %ld\n",
				(long) osc.freq_want,
				(long) osc.freq_got);
		_osc_dump_prog_regs(fd);
	}

	return(errs);
}



//*****************************************************************************
static int _osc_measure_init_test(int fd, s32 ref, const data_t* data)
{
	s32			delta		= 0;
	int			errs		= 0;
	s32			freq_got	= 0;
	s32			freq_ref;
	int			loop;
	s32			max			= 0;
	s32			min			= 0;
	long		mmax;
	long		mmin;
	sio4_osc_t	osc;
	int			ret;

	memset(&osc, 0, sizeof(osc));
	errs	+= sio4_reset_channel(fd, 0);

	switch (data->osc_chip)
	{
		default:

			errs++;
			printf("FAIL <---  (%d. internal error)\n", __LINE__);
			break;

		case SIO4_OSC_CHIP_FIXED:

			min			= 1000000L;
			max			= 20000000L;
			delta		= 4000000L;
			freq_got	= -1;
			break;

		case SIO4_OSC_CHIP_CY22393:
		case SIO4_OSC_CHIP_CY22393_2:

			min			= ref;
			max			= ref;
			delta		= 4000000L;
			freq_got	= ref;
			break;
	}

	for (freq_ref = min; (errs == 0) && (freq_ref <= max); freq_ref += delta)
	{
		errs	+= _reference_set(fd, freq_ref, __LINE__);
		memset(&osc, 0, sizeof(osc));
		ret		= sio4_ioctl(fd, SIO4_OSC_INIT, &osc);

		if (ret < 0)
		{
			errs++;
			printf("FAIL <---  (%d. SIO4_OSC_INIT)\n", __LINE__);
			break;
		}

		// This won't pass every conceivable circumstance, but it will most instances.
		mmax	= ((osc.freq_want / 10) + 1 + ((osc.freq_want + 9999) / 10000)) * 10 + 100;
		mmin	= ((osc.freq_want / 10) - 1 - ((osc.freq_want + 9999) / 10000)) * 10 - 100;

		for (loop = 0; loop <= 20; loop++)
		{
			// Measure the frequency and verify the results.
			memset(&osc, 0, sizeof(osc));
			ret	= sio4_ioctl(fd, SIO4_OSC_MEASURE, &osc);

			if (ret < 0)
			{
				errs++;
				printf("FAIL <---  (%d. SIO4_OSC_measure)\n", __LINE__);
				break;
			}

			if (osc.freq_got < mmin)
			{
//				printf(	"\n(%d. loop %d, SMALL: W %ld, G %ld, Ref %ld, mmin, %ld)",
//						__LINE__, loop, (long) osc.freq_want, (long) osc.freq_got, (long) freq_ref, (long) mmin);
				continue;
			}
			else if (osc.freq_got > mmax)
			{
//				printf(	"\n(%d. loop %d, LARGE: W %ld, G %ld, Ref %ld, mmax %ld)",
//						__LINE__, loop, (long) osc.freq_want, (long) osc.freq_got, (long) freq_ref, (long) mmax);
				continue;
			}
			else
			{
				osc.freq_got	= osc.freq_want;
				break;
			}
		}

		errs	+= TEST_FREQ_GOT(freq_got, &osc, __LINE__);
		errs	+= TEST_FREQ_WANT(freq_ref, &osc, __LINE__);
		errs	+= TEST_FREQ_REF(freq_ref, &osc, __LINE__);
		errs	+= TEST_CHIP(data->osc_chip, &osc, __LINE__);
	}

	if ((errs) && (data->detail))
	{
		printf("\n%d. _osc_measure_init_test:\n", __LINE__);
		printf(	"want %ld, got %ld\n",
				(long) osc.freq_want,
				(long) osc.freq_got);
		_osc_dump_prog_regs(fd);
	}

	return(errs);
}



//*****************************************************************************
static int _osc_measure_program_test(int fd, s32 ref, const data_t* data)
{
	s32			delta		= 0;
	int			errs		= 0;
	s32			freq_ref	= 0;
	s32			freq_got	= 0;
	s32			freq_want;
	int			loop;
	s32			max			= 0;
	s32			min			= 0;
	long		mmax;
	long		mmin;
	sio4_osc_t	osc;
	int			ret;

	memset(&osc, 0, sizeof(osc));
	errs	+= sio4_reset_channel(fd, 0);

	switch (data->osc_chip)
	{
		default:

			errs++;
			printf("FAIL <---  (%d. internal error)\n", __LINE__);
			break;

		case SIO4_OSC_CHIP_FIXED:

			min			= 1000000L;
			max			= 20000000L;
			delta		= 4000000L;
			freq_ref	= 20000000L;
			break;

		case SIO4_OSC_CHIP_CY22393:
		case SIO4_OSC_CHIP_CY22393_2:

			min			= 1000000L;
			max			= 20000000L;
			delta		= 1000000L;
			freq_ref	= ref;
			break;
	}

	for (freq_want = min; freq_want <= max; freq_want += delta)
	{
		if (errs)
			break;

		// Adjust and verify the reference frequency.
		errs	+= _reference_set(fd, freq_ref, __LINE__);

		if (errs)
			break;

		// Perform a programming effort and verify the results.
		memset(&osc, 0, sizeof(osc));
		osc.freq_want	= freq_want;
		ret				= sio4_ioctl(fd, SIO4_OSC_PROGRAM, &osc);

		if (ret < 0)
		{
			errs++;
			printf("FAIL <---  (%d. SIO4_OSC_PROGRAM)\n", __LINE__);
			break;
		}

		switch (data->osc_chip)
		{
			default:

				errs++;
				printf("FAIL <---  (%d. internal error)\n", __LINE__);
				break;

			case SIO4_OSC_CHIP_FIXED:

				freq_got	= -1;
				break;

			case SIO4_OSC_CHIP_CY22393:
			case SIO4_OSC_CHIP_CY22393_2:

				freq_got	= freq_want;
				break;
		}

		// This won't pass every conceivable circumstance, but it will most instances.
		mmax	= ((osc.freq_want / 10) + 1 + ((osc.freq_want + 9999) / 10000)) * 10 + 100;
		mmin	= ((osc.freq_want / 10) - 1 - ((osc.freq_want + 9999) / 10000)) * 10 - 100;

		for (loop = 0; loop <= 20; loop++)
		{
			// Measure the frequency and verify the results.
			memset(&osc, 0, sizeof(osc));
			ret	= sio4_ioctl(fd, SIO4_OSC_MEASURE, &osc);

			if (ret < 0)
			{
				errs++;
				printf("FAIL <---  (%d. SIO4_OSC_MEASURE)\n", __LINE__);
				break;
			}

			if (osc.freq_got < mmin)
			{
//				printf(	"\n(%d. loop %d, SMALL: FW %ld, W %ld, G %ld, Ref %ld, mmin, %ld)",
//						__LINE__, loop, (long) freq_want, (long) osc.freq_want, (long) osc.freq_got, (long) freq_ref, (long) mmin);
				continue;
			}
			else if (osc.freq_got > mmax)
			{
//				printf(	"\n(%d. loop %d, LARGE: FW %ld, W %ld, G %ld, Ref %ld, mmax %ld)",
//						__LINE__, loop, (long) freq_want, (long) osc.freq_want, (long) osc.freq_got, (long) freq_ref, (long) mmax);
				continue;
			}
			else
			{
				osc.freq_got	= osc.freq_want;
				break;
			}
		}

		errs	+= TEST_FREQ_GOT(freq_got, &osc, __LINE__);
		errs	+= TEST_FREQ_WANT(freq_want, &osc, __LINE__);
		errs	+= TEST_FREQ_REF(freq_ref, &osc, __LINE__);
		errs	+= TEST_CHIP(data->osc_chip, &osc, __LINE__);
	}

	if ((errs) && (data->detail))
	{
		printf("\n%d. _osc_measure_program_test:\n", __LINE__);
		printf(	"want %ld, got %ld\n",
				(long) osc.freq_want,
				(long) osc.freq_got);
		_osc_dump_prog_regs(fd);
	}

	return(errs);
}



//*****************************************************************************
static int _function_test(int fd, s32 ref, const data_t* data)
{
	int	errs	= 0;

	errs	+= _osc_measure_reset_test(fd, ref, data);
	errs	+= _osc_measure_init_test(fd, ref, data);

	if (data->osc_program)
		errs	+= _osc_measure_program_test(fd, ref, data);

	return(errs);
}



//*****************************************************************************
int osc_measure_tests(int fd, const data_t* data)
{
	int	errs	= 0;
	s32	ref		= SIO4_OSC_FREQ_REF_DEFAULT;

	gsc_label("SIO4_OSC_MEASURE");

	if (data->osc_measure)
		errs	= sio4_osc_reference(fd, &ref, 0);

	if (errs)
	{
	}
	else if (data->osc_measure == 0)
	{
		printf(	"SKIPPED  (Not supported.)\n");
	}
	else
	{
		switch (data->osc_chip)
		{
			default:

				printf(	"SKIPPED  (Unrecognized oscillator option: %ld)\n",
						(long) data->osc_chip);
				break;

			case SIO4_OSC_CHIP_UNKNOWN:

				errs++;
				printf("FAIL <---  (Oscillator chip not recognized.)\n");
				break;

			case SIO4_OSC_CHIP_IDC2053B:
			case SIO4_OSC_CHIP_IDC2053B_4:

				printf("SKIPPED  (Oscillator chip not supported by test.)\n");
				break;

			case SIO4_OSC_CHIP_FIXED:
			case SIO4_OSC_CHIP_CY22393:
			case SIO4_OSC_CHIP_CY22393_2:

				errs	+= _service_test(fd);
				errs	+= _function_test(fd, ref, data);

				if (errs == 0)
					printf("PASS\n");

				break;
		}
	}

	errs	= errs ? 1 : 0;
	return(errs);
}


