// $URL: http://subversion:8080/svn/gsc/trunk/drivers/LINUX/SIO4%20and%20SIO8/SIO4_Linux_1.x.x_GSC_DN/samples/irq/usc_rxc_down.c $
// $Rev: 53079 $
// $Date: 2023-06-13 10:07:58 -0500 (Tue, 13 Jun 2023) $

// SIO4: Sample Application: source file

#include "main.h"



// variables ******************************************************************

static	int	_qty;



/******************************************************************************
*
*	Function:	_irq_reset
*
*	Purpose:
*
*		Clean things up after we're done.
*
*	Arguments:
*
*		fd		The file descriptor for the channel to access.
*
*	Returned:
*
*		>= 0	The number of errors seen.
*
******************************************************************************/

static int _irq_reset(int fd)
{
	int	errs	= 0;
	int	status;

	// Reset the device.
	status	= sio4_channel_reset(fd, 0);
	errs	+= status ? 1 : 0;

	// Turn off asynchronous interrupt notification.
	errs	+= usc_notify_disable(fd);

	// Disable the interrupt.
	errs	+= GSC_INT_ALL_DISABLE(fd);
	errs	+= USC_MASTER_INT_DISABLE(fd);
	errs	+= USC_IOP_INT_DISABLE(fd);
	errs	+= USC_RXC_DN_INT_DISARM(fd);
	return(errs);
}



/******************************************************************************
*
*	Function:	_irq_setup
*
*	Purpose:
*
*		Set things up so generation of the interrupt will be detected.
*
*	Arguments:
*
*		fd		The file descriptor for the channel to access.
*
*	Returned:
*
*		>= 0	The number of errors seen.
*
******************************************************************************/

static int _irq_setup(int fd)
{
	int	errs	= 0;
	s32	prog;
	int	status;

	// Reset the device.
	status	= sio4_channel_reset(fd, 0);
	errs	+= status ? 1 : 0;

	status	= sio4_feature_test(fd, SIO4_FEATURE_OSC_PROGRAM, &prog, 0);
	errs	+= status ? 1 : 0;

	if (prog == SIO4_FEATURE_YES)
		errs	+= osc_init(fd, NULL, 0);

	errs	+= rxc_usc_config(fd, SIO4_RXC_USC_CONFIG_OUT_BRG0, NULL, 0);
	// TxC -> CTR0
	errs	+= reg_mod  (fd, SIO4_USC_CMCR,	0x3000, 0x3000);
	// CRT0 / 32
	errs	+= reg_mod  (fd, SIO4_USC_HCR,	0x0000, 0xC000);
	// CTR0 -> BRG0
	errs	+= reg_mod  (fd, SIO4_USC_CMCR,	0x0000, 0x0300);
	// BRG0 Disable (it is enabled later)
	errs	+= reg_mod  (fd, SIO4_USC_HCR,	0x0000, 0x0001);
	// TC0R = 0xFFFF
	errs	+= reg_write(fd, SIO4_USC_TC0R,	0xFFFF);
	// BRG0 -> RxC
	errs	+= reg_mod  (fd, SIO4_USC_IOCR,	0x0004, 0x0007);

	// Setup asynchronous interrupt notification.
	errs	+= USC_IOP_INT_CLEAR(fd);
	errs	+= USC_RXC_STATUS_CLEAR(fd);
	errs	+= usc_notify_enable(fd, USC_IRQ_RXC_DN, &_qty);

	return(errs);
}



/******************************************************************************
*
*	Function:	_irq_trigger
*
*	Purpose:
*
*		Trigger the interrupt.
*
*	Arguments:
*
*		fd		The file descriptor for the channel to access.
*
*	Returned:
*
*		>= 0	The number of errors seen.
*
******************************************************************************/

static int _irq_trigger(int fd)
{
	int	errs	= 0;

	// Enable the interrupt.
	errs	+= USC_RXC_DN_INT_ARM(fd);
	errs	+= USC_IOP_INT_ENABLE(fd);
	errs	+= USC_MASTER_INT_ENABLE(fd);
	errs	+= GSC_LOCAL_INT_ENABLE(fd);

	// BRG0 Enable
	os_sleep_ms(1);
	errs	+= reg_mod(fd, SIO4_USC_HCR, 0x0001, 0x0001);

	wait_for_irq(&_qty);
	return(errs);
}



/******************************************************************************
*
*	Function:	usc_rxc_down_test
*
*	Purpose:
*
*		Verify the operation of the USC RxC Down interrupt.
*
*	Arguments:
*
*		fd		The file descriptor for the channel to access.
*
*	Returned:
*
*		>= 0	The number of errors seen.
*
******************************************************************************/

int usc_rxc_down_test(int fd)
{
	int	errs	= 0;

	gsc_label("RxC Down");
	errs	+= _irq_setup(fd);
	errs	+= _irq_trigger(fd);
	errs	+= (_qty > 0) ? 0 : 1;
	errs	+= _irq_reset(fd);
	printf("%s\n", errs ? "FAIL <---" : "PASS");
	return(errs ? 1 : 0);
}


