// $URL: http://subversion:8080/svn/gsc/trunk/drivers/LINUX/SIO4%20and%20SIO8/SIO4_Linux_1.x.x_GSC_DN/samples/irq/usc.c $
// $Rev: 53079 $
// $Date: 2023-06-13 10:07:58 -0500 (Tue, 13 Jun 2023) $

// SIO4: Sample Application: source file

#include "main.h"



// variables ******************************************************************

static	int			_fd;
static	usc_irq_t	_irq;
static	int*		_qty;



/******************************************************************************
*
*	Function:	_service_none
*
*	Purpose:
*
*		Service a spurios interrupt.
*
*	Arguments:
*
*		fd		The file descriptor for the channel to access.
*
*		irq		Tell the caller which USC interrupt occurred.
*
*	Returned:
*
*		>= 0	The number of errors seen.
*
******************************************************************************/

static int _service_none(int fd, usc_irq_t* irq)
{
	printf("(none      )");
	irq[0]	= USC_IRQ_NONE;
	return(0);
}



/******************************************************************************
*
*	Function:	_service_usc_iop_irq
*
*	Purpose:
*
*		Service a USC I/O Pin interrupt.
*
*	Arguments:
*
*		fd		The file descriptor for the channel to access.
*
*		irq		Tell the caller which USC interrupt occurred.
*
*	Returned:
*
*		>= 0	The number of errors seen.
*
******************************************************************************/

static int _service_usc_iop_irq(int fd, usc_irq_t* irq)
{
	int	errs	= 0;
	u32	misr;

	errs	+= reg_read(fd, -1, 0, SIO4_USC_MISR, &misr);

	if (errs)
	{
		irq[0]	= USC_IRQ_NONE;
	}
	else if (misr & 0x8000)
	{
		if (misr & 0x4000)
		{
			errs	= USC_RXC_DN_INT_DISARM(fd);
			irq[0]	= USC_IRQ_RXC_DN;
		}
		else
		{
			errs	= USC_RXC_UP_INT_DISARM(fd);
			irq[0]	= USC_IRQ_RXC_UP;
		}
	}
	else if (misr & 0x2000)
	{
		if (misr & 0x1000)
		{
			errs	= USC_TXC_DN_INT_DISARM(fd);
			irq[0]	= USC_IRQ_TXC_DN;
		}
		else
		{
			errs	= USC_TXC_UP_INT_DISARM(fd);
			irq[0]	= USC_IRQ_TXC_UP;
		}
	}
	else if (misr & 0x0800)
	{
		if (misr & 0x0400)
		{
			errs	= USC_RXR_DN_INT_DISARM(fd);
			irq[0]	= USC_IRQ_RXR_DN;
		}
		else
		{
			errs	= USC_RXR_UP_INT_DISARM(fd);
			irq[0]	= USC_IRQ_RXR_UP;
		}
	}
	else if (misr & 0x0200)
	{
		if (misr & 0x0100)
		{
			errs	= USC_TXR_DN_INT_DISARM(fd);
			irq[0]	= USC_IRQ_TXR_DN;
		}
		else
		{
			errs	= USC_TXR_UP_INT_DISARM(fd);
			irq[0]	= USC_IRQ_TXR_UP;
		}
	}
	else if (misr & 0x0080)
	{
		if (misr & 0x0040)
		{
			errs	= USC_DCD_DN_INT_DISARM(fd);
			irq[0]	= USC_IRQ_DCD_DN;
		}
		else
		{
			errs	= USC_DCD_UP_INT_DISARM(fd);
			irq[0]	= USC_IRQ_DCD_UP;
		}
	}
	else if (misr & 0x0020)
	{
		if (misr & 0x0010)
		{
			errs	= USC_CTS_DN_INT_DISARM(fd);
			irq[0]	= USC_IRQ_CTS_DN;
		}
		else
		{
			errs	= USC_CTS_UP_INT_DISARM(fd);
			irq[0]	= USC_IRQ_CTS_UP;
		}
	}
	else
	{
		irq[0]	= USC_IRQ_NONE;
	}

	errs	+= USC_IOP_INT_CLEAR(fd);
	return(errs);
}



/******************************************************************************
*
*	Function:	_service_usc_misc_irq
*
*	Purpose:
*
*		Service a USC Miscellaneous interrupt.
*
*	Arguments:
*
*		fd		The file descriptor for the channel to access.
*
*		irq		Tell the caller which USC interrupt occurred.
*
*	Returned:
*
*		>= 0	The number of errors seen.
*
******************************************************************************/

static int _service_usc_misc_irq(int fd, usc_irq_t* irq)
{
	int	errs	= 0;
	u32	irqs;
	u32	misr;
	u32	sicr;

	errs	+= reg_read(fd, -1, 0, SIO4_USC_SICR, &sicr);
	errs	+= reg_read(fd, -1, 0, SIO4_USC_MISR, &misr);
	irqs	= sicr & misr & 0x000F;

	if (errs)
	{
		irq[0]	= USC_IRQ_NONE;
	}
	else if (irqs & 0x0008)
	{
		errs	+= USC_RCC_UNDER_INT_DISARM(fd);
		irq[0]	= USC_IRQ_RCC_UNDER;
	}
	else if (irqs & 0x0004)
	{
		errs	+= USC_DPLL_DESYNC_INT_DISARM(fd);
		irq[0]	= USC_IRQ_DPLL_DESYNC;
	}
	else if (irqs & 0x0002)
	{
		errs	+= USC_BRG1_ZERO_INT_DISARM(fd);
		irq[0]	= USC_IRQ_BRG1_ZERO;
	}
	else if (irqs & 0x0001)
	{
		errs	+= USC_BRG0_ZERO_INT_DISARM(fd);
		irq[0]	= USC_IRQ_BRG0_ZERO;
	}
	else
	{
		errs	+= USC_MISC_INT_DISABLE(fd);
		irq[0]	= USC_IRQ_NONE;
	}

	errs	+= USC_MISC_INT_CLEAR(fd);
	return(errs);
}



/******************************************************************************
*
*	Function:	_service_usc_rx_data_irq
*
*	Purpose:
*
*		Service a Rx Data interrupt.
*
*	Arguments:
*
*		fd		The file descriptor for the channel to access.
*
*		irq		Tell the caller which USC interrupt occurred.
*
*	Returned:
*
*		>= 0	The number of errors seen.
*
******************************************************************************/

static int _service_usc_rx_data_irq(int fd, usc_irq_t* irq)
{
	int	errs;

	irq[0]	= USC_IRQ_RX_DATA;
	errs	= USC_RX_DATA_INT_CLEAR(fd);
	return(errs);
}



/******************************************************************************
*
*	Function:	_service_usc_rx_status_irq
*
*	Purpose:
*
*		Service a Rx Status interrupt.
*
*	Arguments:
*
*		fd		The file descriptor for the channel to access.
*
*		irq		Tell the caller which USC interrupt occurred.
*
*	Returned:
*
*		>= 0	The number of errors seen.
*
******************************************************************************/

static int _service_usc_rx_status_irq(int fd, usc_irq_t* irq)
{
	int	errs	= 0;
	u32	irqs;
	u32	rcsr;
	u32	ricr;

	errs	+= reg_read(fd, -1, 0, SIO4_USC_RCSR, &rcsr);
	errs	+= reg_read(fd, -1, 0, SIO4_USC_RICR, &ricr);
	irqs	= ricr & rcsr & 0x00F6;

	if (errs)
	{
		irq[0]	= USC_IRQ_NONE;
	}
	else if (irqs & 0x0080)
	{
		errs	+= USC_RX_EXIT_HUNT_INT_DISARM(fd);
		irq[0]	= USC_IRQ_RX_EXITED_HUNT;
	}
	else if (irqs & 0x0040)
	{
		errs	+= USC_RX_IDLE_RCVD_INT_DISARM(fd);
		irq[0]	= USC_IRQ_RX_IDLE_RECEIVED;
	}
	else if (irqs &0x0020)
	{
		errs	+= USC_RX_BRK_ABRT_INT_DISARM(fd);
		irq[0]	= USC_IRQ_RX_BREAK_ABORT;
	}
	else if (irqs & 0x0010)
	{
		errs	+= USC_RX_BOUND_INT_DISARM(fd);
		irq[0]	= USC_IRQ_RX_BOUND;
	}
	else if (irqs & 0x0004)
	{
		errs	+= USC_RX_ABORT_PE_INT_DISARM(fd);
		irq[0]	= USC_IRQ_RX_ABORT_PE;
	}
	else if (irqs & 0x0002)
	{
		errs	+= USC_RX_OVERRUN_INT_DISARM(fd);
		irq[0]	= USC_IRQ_RX_OVERRUN;
	}
	else
	{
		irq[0]	= USC_IRQ_NONE;
	}

	errs	+= USC_RX_STATUS_INT_CLEAR(fd);
	return(errs);
}



/******************************************************************************
*
*	Function:	_service_usc_tx_data_irq
*
*	Purpose:
*
*		Service a USC Tx Data interrupt.
*
*	Arguments:
*
*		fd		The file descriptor for the channel to access.
*
*		irq		Tell the caller which USC interrupt occurred.
*
*	Returned:
*
*		>= 0	The number of errors seen.
*
******************************************************************************/

static int _service_usc_tx_data_irq(int fd, usc_irq_t* irq)
{
	int	errs;

	irq[0]	= USC_IRQ_TX_DATA;
	errs	= USC_TX_DATA_INT_CLEAR(fd);
	return(errs);
}



/******************************************************************************
*
*	Function:	_service_usc_tx_status_irq
*
*	Purpose:
*
*		Service a Tx Status interrupt.
*
*	Arguments:
*
*		fd		The file descriptor for the channel to access.
*
*		irq		Tell the caller which USC interrupt occurred.
*
*	Returned:
*
*		>= 0	The number of errors seen.
*
******************************************************************************/

static int _service_usc_tx_status_irq(int fd, usc_irq_t* irq)
{
	int	errs	= 0;
	u32	irqs;
	u32	tcsr;
	u32	ticr;

	errs	+= reg_read(fd, -1, 0, SIO4_USC_TCSR, &tcsr);
	errs	+= reg_read(fd, -1, 0, SIO4_USC_TICR, &ticr);
	irqs	= ticr & tcsr & 0x00FA;

	if (errs)
	{
		irq[0]	= USC_IRQ_NONE;
	}
	else if (irqs & 0x0080)
	{
		errs	+= USC_TX_PRE_SENT_INT_DISARM(fd);
		irq[0]	= USC_IRQ_TX_PRE_SENT;
	}
	else if (irqs & 0x0040)
	{
		errs	+= USC_TX_IDLE_SENT_INT_DISARM(fd);
		irq[0]	= USC_IRQ_TX_IDLE_SENT;
	}
	else if (irqs & 0x0020)
	{
		errs	+= USC_TX_ABORT_SENT_INT_DISARM(fd);
		irq[0]	= USC_IRQ_TX_ABORT_SENT;
	}
	else if (irqs & 0x0010)
	{
		errs	+= USC_TX_END_SENT_INT_DISARM(fd);
		irq[0]	= USC_IRQ_TX_END_SENT;
	}
	else if (irqs & 0x0008)
	{
		errs	+= USC_TX_CRC_SENT_INT_DISARM(fd);
		irq[0]	= USC_IRQ_TX_CRC_SENT;
	}
	else if (irqs & 0x0002)
	{
		errs	+= USC_TX_UNDER_RUN_INT_DISARM(fd);
		irq[0]	= USC_IRQ_TX_UNDER_RUN;
	}
	else
	{
		irq[0]	= USC_IRQ_NONE;
	}

	errs	+= USC_TX_STATUS_INT_DISABLE(fd);
	return(errs);
}



/******************************************************************************
*
*	Function:	_service_usc_irq
*
*	Purpose:
*
*		Service a USC interrupt.
*
*	Arguments:
*
*		fd		The file descriptor for the channel to access.
*
*		irq		Tell the caller which USC interrupt occurred.
*
*	Returned:
*
*		>= 0	The number of errors seen.
*
******************************************************************************/

static int _service_usc_irq(int fd, usc_irq_t* irq)
{
	static const struct
	{
		int	(*service)(int fd, usc_irq_t* irq);
	} list[]	=
	{
		{ _service_none					},
		{ _service_usc_misc_irq			},
		{ _service_usc_iop_irq			},
		{ _service_usc_tx_data_irq		},
		{ _service_usc_tx_status_irq	},
		{ _service_usc_rx_data_irq		},
		{ _service_usc_rx_status_irq	},
		{ _service_none					}
	};

	int	errs	= 0;
	u32	ivr;
	int	type;

	errs	+= reg_read(fd, -1, 0, SIO4_USC_IVR, &ivr);
	type	= (ivr & 0xE00) >> 9;

	if (errs)
		irq[0]	= USC_IRQ_NONE;
	else
		errs	= (list[type].service)(fd, irq);

	return(errs);
}



/******************************************************************************
*
*	Function:	_handle_sigio
*
*	Purpose:
*
*		Handle a SIGIO signal.
*
*	Arguments:
*
*		signal	The signal generated, whichis SIGIO.
*
*	Returned:
*
*		None.
*
******************************************************************************/

static void _handle_sigio(int signo)
{
	int						errs	= 0;
	SIO4_INTERRUPT_STATUS	int_stat;
	usc_irq_t				irq	= USC_IRQ_NONE;
	int						status;

//	printf("(SIGIO)");

	status	= ioctl(_fd, SIO4_READ_INT_STATUS, &int_stat);

	if ((status == 0) &&
		(int_stat.u8SIO4Status & SIO4_INT_NOTIFY_USC_INTERRUPTS))
	{
		errs	= _service_usc_irq(_fd, &irq);
	}

	if ((errs == 0) && (_qty) && (irq != USC_IRQ_NONE) && (irq == _irq))
		_qty[0]++;
}



/******************************************************************************
*
*	Function:	usc_notify_enable
*
*	Purpose:
*
*		Enable asynchronous notification for USC interrupts.
*
*	Arguments:
*
*		fd		The file descriptor for the channel to access.
*
*	Returned:
*
*		>= 0	The number of errors seen.
*
******************************************************************************/

int usc_notify_enable(int fd, usc_irq_t irq, int* qty)
{
	int		errs;
	int		flags;
	pid_t	pid;
	int		status;

	ioctl(fd, SIO4_INT_NOTIFY, 0);
	_fd		= fd;
	_irq	= irq;
	_qty	= qty;
	signal(SIGIO, _handle_sigio);
	pid		= getpid();
	fcntl(fd, F_SETOWN, pid);
	flags	= fcntl(fd, F_GETFL);
	flags	|= FASYNC;
	fcntl(fd, F_SETFL, flags);
	status	= ioctl(fd, SIO4_INT_NOTIFY, SIO4_INT_NOTIFY_USC_INTERRUPTS);
	errs	= status ? 1 : 0;

	if (qty)
		qty[0]	= 0;

	return(errs);
}



/******************************************************************************
*
*	Function:	usc_notify_disable
*
*	Purpose:
*
*		Disable asynchronous notification for USC interrupts.
*
*	Arguments:
*
*		fd		The file descriptor for the channel to access.
*
*	Returned:
*
*		>= 0	The number of errors seen.
*
******************************************************************************/

int usc_notify_disable(int fd)
{
	int		errs;
	int		flags;
	pid_t	pid;
	int		status;

	ioctl(fd, SIO4_INT_NOTIFY, 0);
	_fd		= 0;
	_irq	= USC_IRQ_NONE;
	_qty	= NULL;
	signal(SIGIO, NULL);
	pid		= getpid();
	fcntl(fd, F_SETOWN, pid);
	flags	= fcntl(fd, F_GETFL);
	flags	&= ~FASYNC;
	fcntl(fd, F_SETFL, flags);
	status	= ioctl(fd, SIO4_INT_NOTIFY, 0);
	errs	= status ? 1 : 0;
	return(errs);
}



//*****************************************************************************
int usc_tests(int fd, int fw_type)
{
	int	errs	= 0;

	gsc_label("USC Interrupts");


	if (fw_type == SIO4_FW_TYPE_CONFIG_Z16C30)
	{
		printf("\n");
		gsc_label_level_inc();

		errs	+= usc_rx_exited_hunt_test(fd);		// Rx Status
		errs	+= usc_rx_idle_received_test(fd);	// Rx Status
		errs	+= usc_rx_break_abort_test(fd);		// Rx Status
		errs	+= usc_rx_bound_test(fd);			// Rx Status
		errs	+= usc_rx_abort_pe_test(fd);		// Rx Status
		errs	+= usc_rx_overrun_test(fd);			// Rx Status

		errs	+= usc_rx_data_test(fd);			// Rx Data

		errs	+= usc_tx_preamble_sent_test(fd);	// Tx Status
		errs	+= usc_tx_idle_sent_test(fd);		// Tx Status
		errs	+= usc_tx_abort_sent_test(fd);		// Tx Status
		errs	+= usc_tx_eof_eom_sent_test(fd);	// Tx Status
		errs	+= usc_tx_crc_sent_test(fd);		// Tx Status
		errs	+= usc_tx_underrun_test(fd);		// Tx Status

		errs	+= usc_tx_data_test(fd);			// Tx Data

		errs	+= usc_rxc_down_test(fd);			// I/O Pin
		errs	+= usc_rxc_up_test(fd);				// I/O Pin
		errs	+= usc_txc_down_test(fd);			// I/O Pin
		errs	+= usc_txc_up_test(fd);				// I/O Pin
		errs	+= usc_rxreq_down_test(fd);			// I/O Pin
		errs	+= usc_rxreq_up_test(fd);			// I/O Pin
		errs	+= usc_txreq_down_test(fd);			// I/O Pin
		errs	+= usc_txreq_up_test(fd);			// I/O Pin
		errs	+= usc_dcd_down_test(fd);			// I/O Pin
		errs	+= usc_dcd_up_test(fd);				// I/O Pin
		errs	+= usc_cts_down_test(fd);			// I/O Pin
		errs	+= usc_cts_up_test(fd);				// I/O Pin

		errs	+= usc_rcc_underrun_test(fd);		// Misc.
		errs	+= usc_dpll_desync_test(fd);		// Misc.
		errs	+= usc_brg1_zero_test(fd);			// Misc.
		errs	+= usc_brg0_zero_test(fd);			// Misc.

		gsc_label_level_dec();
	}
	else
	{
		printf("SKIPPED  (Not supported.)\n");
	}

	return(errs);
}



