// $URL: http://subversion:8080/svn/gsc/trunk/drivers/LINUX/SIO4%20and%20SIO8/SIO4_Linux_1.x.x_GSC_DN/samples/irq/rx_fifo_empty.c $
// $Rev: 53079 $
// $Date: 2023-06-13 10:07:58 -0500 (Tue, 13 Jun 2023) $

// SIO4: Sample Application: source file

#include "main.h"



// variables ******************************************************************

static	int	_qty;



/******************************************************************************
*
*	Function:	_irq_reset
*
*	Purpose:
*
*		Clean things up after we're done.
*
*	Arguments:
*
*		fd		The file descriptor for the channel to access.
*
*	Returned:
*
*		>= 0	The number of errors seen.
*
******************************************************************************/

static int _irq_reset(int fd)
{
	int	errs	= 0;
	int	status;

	// Reset the device.
	status	= sio4_channel_reset(fd, 0);
	errs	+= status ? 1 : 0;

	// Turn off asynchronous interrupt notification.
	errs	+= notify_disable(fd);

	// Disable the interrupt.
	errs	+= GSC_INT_ALL_DISABLE(fd);
	return(errs);
}



/******************************************************************************
*
*	Function:	_irq_setup
*
*	Purpose:
*
*		Set things up so generation of the interrupt will be detected.
*
*	Arguments:
*
*		fd		The file descriptor for the channel to access.
*
*	Returned:
*
*		>= 0	The number of errors seen.
*
******************************************************************************/

static int _irq_setup(int fd)
{
	sio4_async_t	async;
	int				errs	= 0;
	s32				fw;
	s32				index;
	s32				prog;
	int				status;

	errs	+= sio4_fw_type_config(fd, -1, 0, -1, &fw);

	// Reset the device.
	status	= sio4_channel_reset(fd, 0);
	errs	+= status ? 1 : 0;

	if (fw == SIO4_FW_TYPE_CONFIG_Z16C30)
	{
		// This is a Z16C30 based device.
		// Setup the channel for Asynchronous loopback.
		sio4_async_t_init(&async);
		async.cable.legacy.config	= SIO4_CABLE_CONFIG_TXUPR_RXUPR;
		async.loopback.enable		= 1;
		async.loopback.internal		= 1;
		async.rx.enable				= 1;
		async.tx.enable				= 1;
		errs	+= sio4_async_config(fd, &async, 0, 0);
	}
	else
	{
		// This is a SYNC model device.
		// Setup the channel for loopback operation.
		status	= sio4_feature_test(fd, SIO4_FEATURE_OSC_PROGRAM, &prog, 0);
		errs	+= status ? 1 : 0;

		if (prog == SIO4_FEATURE_YES)
			errs	+= osc_program(fd, 20000000, NULL, 0);

		// Select defaults.
		errs	+= reg_mod(fd, SIO4_GSC_PSRCR, 0, 0xFFFFFFFF);

		// Enable the transceivers.
		errs	+= reg_mod(fd, SIO4_GSC_PSRCR, ~0, 0x80000000);

		// Enable the external loop back.
		errs	+= reg_mod(fd, SIO4_GSC_PSRCR, ~0, 0x20000000);

		// Enable the transmitter.
		errs	+= reg_mod(fd, SIO4_GSC_CSR, ~0, 0x02000000);

		// Enable the receiver.
		errs	+= reg_mod(fd, SIO4_GSC_CSR, ~0, 0x01000000);

		// Set 16-bit gaps abd 8-bit bytes.
		errs	+= reg_write(fd, SIO4_GSC_TCR, 0x00100008);
	}

	// Reset the FIFOs.
	errs	+= fifo_reset(fd, TX_AND_RX_FIFO, 0);

	// Setup asynchronous interrupt notification.
	errs	+= notify_enable(fd, SIO4_INT_NOTIFY_RX_FIFO_E, &_qty);

	// Enable the interrupt.
	errs	+= sio4_feature_test(fd, SIO4_FEATURE_INDEX_CHANNEL, &index, 0);
	errs	+= reg_write(fd, SIO4_GSC_ICR, 0x40000 << (4 * index));
	return(errs);
}



/******************************************************************************
*
*	Function:	_irq_trigger
*
*	Purpose:
*
*		Trigger the interrupt.
*
*	Arguments:
*
*		fd		The file descriptor for the channel to access.
*
*	Returned:
*
*		>= 0	The number of errors seen.
*
******************************************************************************/

static int _irq_trigger(int fd)
{
	char	buf[1024];
	u32		csr;
	int		errs;
	int		stat;

	memset(buf, 0, sizeof(buf));

	for (;;)
	{
		errs	= sio4_write(fd, buf, sizeof(buf), NULL);

		if (errs)
			break;

		// Wait for the Tx FIFO to empty.

		for (; errs == 0;)
		{
			stat	= reg_read(fd, -1, 0, SIO4_GSC_CSR, &csr);
			errs	+= stat ? 1 : 0;

			if ((csr & SIO4_GSC_CSR_TFE) == 0)
				break;

			os_sleep_ms(100);
		}

		// Check for the Rx FIFO NOT Empty.
		stat	= reg_read(fd, -1, 0, SIO4_GSC_CSR, &csr);
		errs	+= stat ? 1 : 0;

		if (csr & SIO4_GSC_CSR_RFE)
			break;
	}

	// Disable the receiver.
	errs	+= reg_mod(fd, SIO4_GSC_CSR, ~0, 0x00000000);

	// Reset the Rx FIFO.
	errs	+= fifo_reset(fd, RX_FIFO, 0);

	if (errs == 0)
		wait_for_irq(&_qty);

	return(errs);
}



//*****************************************************************************
int rx_fifo_empty_test(int fd, int irq32)
{
	int	errs	= 0;

	gsc_label("Rx FIFO Empty");


	if (irq32 == SIO4_FEATURE_YES)
	{
		errs	+= _irq_setup(fd);
		errs	+= _irq_trigger(fd);
		errs	+= (_qty > 0) ? 0 : 1;
		errs	+= _irq_reset(fd);
		printf("%s\n", errs ? "FAIL <---" : "PASS");
	}
	else
	{
		printf("SKIPPED  (Not supported.)\n");
	}

	return(errs ? 1 : 0);
}


