// $URL: http://subversion:8080/svn/gsc/trunk/drivers/LINUX/SIO4%20and%20SIO8/SIO4_Linux_1.x.x_GSC_DN/include/sio4_utils.h $
// $Rev: 53093 $
// $Date: 2023-06-13 10:20:11 -0500 (Tue, 13 Jun 2023) $

// SIO4: Utilities: header file

#ifndef	__SIO4_UTILS_H__
#define	__SIO4_UTILS_H__

#include <pthread.h>

#include "sio4.h"
#include "gsc_utils.h"



// macros *********************************************************************

#define	SIZEOF_ARRAY(a)						(sizeof((a))/sizeof((a)[0]))

#define	GSC_CSR_WRITE(fd,v)					reg_write(fd,SIO4_GSC_CSR,v)

#define	USC_CMCR_READ(fd,v)					reg_read(fd,-1,0,SIO4_USC_CMCR,v)
#define	USC_CMCR_WRITE(fd,v)				reg_write(fd,SIO4_USC_CMCR,v)
#define	USC_RCLR_WRITE(fd,v)				reg_write(fd,SIO4_USC_RCLR,v)
#define	USC_TCLR_WRITE(fd,v)				reg_write(fd,SIO4_USC_TCLR,v)



// data types *****************************************************************

typedef struct
{
	// Receiver baud rate configuration.

	// Input parameters.
	unsigned long	rxc_rate;				// The clock rate on RxC.
	long			baudrate;				// The desired baud rate.
	long			baudrate_comp;			// The computed baud rate.
	unsigned int	encoding;				// Signal encoding format.

	// Output parameters.
	unsigned int	brg1_enable		: 1;	// HCR.BRG1E
	unsigned int	brg1_src		: 2;	// CMCR.BRG1Src
	unsigned int	ctr1_src		: 2;	// CMCR.CTR1Src
	unsigned int	ctr1_div_sel	: 1;	// HCR.CTR1DSel
	unsigned int	dpll_div		: 2;	// HCR.DPLLDiv
	unsigned int	dpll_mode		: 2;	// HCR.DPLLMode
	unsigned int	dpll_src		: 2;	// CMCR.DPLLSrc
	unsigned int	rxclk_src		: 3;	// CMCR.RxCLKSrc
	unsigned int	tc1r_value		: 16;	// TC1R
} zilog_rx_baud_config_t;

typedef struct
{
	// Transmitter baud rate configuration.

	// Input parameters.
	unsigned long	txc_rate;				// The clock rate on TxC.
	long			baudrate;				// The desired baud rate.
	long			baudrate_comp;			// The computed baud rate.

	// Output parameters.
	unsigned int	brg0_enable		: 1;	// HCR.BRG0E
	unsigned int	brg0_src		: 2;	// CMCR.BRG0Src
	unsigned int	ctr0_src		: 2;	// CMCR.CTR0Src
	unsigned int	ctr0_div		: 2;	// HCR.CTR0Div
	unsigned int	txclk_src		: 3;	// CMCR.TxCLKSrc
	unsigned int	tc0r_value		: 16;	// TC0R
} zilog_tx_baud_config_t;



// prototypes *****************************************************************

// IOCTL utility services

void	async_block_size(s32 tx, s32 rx, u32* size);

int		cable_config(int fd, s32 set, s32* get, int loud);
int		chan_close(int index, int fd, int verbose);
int		chan_open(int chan, int* fd, int verbosely);
int		channel_command_send(int fd, SIO4_CHAN_CMD cmd, int loud);
int		channel_init(int fd, int loud);
int		channel_reset(int fd, int loud);
int		count_boards(void);
int		cts_cable_config(int fd, s32 set, s32* get, int loud);

int		dcd_cable_config(int fd, s32 set, s32* get, int loud);
int		dcedte_config(int fd, int enable, int dce, int loopback, int loud);
int		driver_info_get(int fd, SIO4_DRIVER_INFO* info, int loud);

int		feature_test(int fd, s32 feature, s32* get, int loud);
int		feature_test_query(int fd, int index, int verbose, s32 set, s32* get);
int		feature_test__bcr_scd(int fd, s32* result, int loud);
int		feature_test__count(int fd, s32* result, int loud);
int		feature_test__fifo_size(int fd, s32* result, int loud);
int		feature_test__mp(int fd, s32* result, int loud);
int		feature_test__osc_program(int fd, s32* result, int loud);
int		feature_test__sync_model(int fd, s32* result, int loud);
int		feature_test__z16c30_model(int fd, s32* result, int loud);
int		fifo_reset(int fd, TX_RX fifo, int loud);

void	gsc_label(const char* label);
int		gsc_label_fail_func(int errs, int loud, const char* func);
int		gsc_label_fail_text(int errs, int loud, const char* text);
int		gsc_label_fail_value(int errs, int loud, long value);
int		gsc_label_indent(int delta);
void	gsc_label_index(const char* label, int index);
void	gsc_label_init(int width);
void	gsc_label_level_dec(void);
void	gsc_label_level_inc(void);
void	gsc_label_long_comma(long long value);
void	gsc_label_suffix(const char* label, const char* suffix);

int		id_driver(void);
void	id_host(void);
int		int_notify(int fd, unsigned char notify, int loud);

int		mp_config(int fd, s32 set, s32* get, int loud);
int		mp_test(int fd, s32 set, s32* get, int loud);

int		os_thread_wait(const os_thread_t* thread, unsigned long timeout_ms);
int		osc_init(int fd, s32* got, int loud);
int		osc_measure(int fd, s32* got, int loud);
int		osc_program(int fd, s32 want, s32* got, int loud);
int		osc_ref_compute(int fd, s32* ref, s32 baud, int loud);
int		osc_setup(int fd, s32 want, s32* got, int loud);

int		reg_list_pci(int fd);
int		reg_list_plx(int fd);
int		reg_mod(int fd, u32 reg, u32 value, u32 mask);
int		reg_read(int fd, int index, int verbose, u32 reg, u32* value);
int		reg_write(int fd, u32 reg, u32 value);

int		rx_cable_config(int fd, s32 set, s32* get, int loud);
int		rx_enable(int fd, int enable);
int		rx_fifo_ae_config(int fd, s32 set, s32* get, int loud);
int		rx_fifo_af_config(int fd, s32 set, s32* get, int loud);
int		rx_fifo_full_cfg_chan(int fd, s32 set, s32* get, int loud);
int		rx_fifo_full_cfg_glb(int fd, s32 set, s32* get, int loud);
int		rx_fifo_size(int fd, s32* size, int loud);
int		rx_fifo_size_compute(int fd, s32* size, int loud);
int		rx_io_mode_config(int fd, s32 set, s32* get, int loud);
int		rxc_usc_config(int fd, s32 set, s32* get, int loud);

int		sio4_board_jumpers_quiet(int fd, s32* get);
int		sio4_feature_test(int fd, s32 set, s32* get, int verbose);
int		sio4_fw_type_config(int fd, int index, int verbose, s32 set, s32* get);	// SIO4_FW_TYPE_CONFIG
int		sio4_id_device(int fd, int index, int verbose);
int		sio4_id_fifo_size_total(int fd, int dash, const char** name);
int		sio4_id_form_factor(int fd, const char** name, int dash);
int		sio4_id_model(int fd, const char** name);
int		sio4_id_type(int fd, const char** name);
int		sio4_init_board_quiet(int fd);
int		sio4_reg_list(int fd, int gsc, int gsc_detail, int usc, int usc_detail);
int		sio4_reset_channel(int fd, int verbose);
int		sio4_rx_fifo_status(int fd, s32* get);
int		sio4_rx_fifo_status_quiet(int fd, s32* get);
int		sio4_tx_fifo_status(int fd, s32* get);
int		sio4_tx_fifo_status_quiet(int fd, s32* get);

int		timeout_set_read(int fd, u32 timeout, int loud);
int		timeout_set_write(int fd, u32 timeout, int loud);
int		tx_cable_clock_config(int fd, s32 set, s32* get, int loud);
int		tx_cable_config(int fd, s32 set, s32* get, int loud);
int		tx_cable_data_config(int fd, s32 set, s32* get, int loud);
int		tx_fifo_ae_config(int fd, s32 set, s32* get, int loud);
int		tx_fifo_af_config(int fd, s32 set, s32* get, int loud);
int		tx_fifo_size(int fd, s32* size, int loud);
int		tx_fifo_size_compute(int fd, s32* size, int loud);
int		tx_io_mode_config(int fd, s32 set, s32* get, int loud);
int		txc_usc_config(int fd, s32 set, s32* get, int loud);

void	zilog_dpll_baud_compute(zilog_rx_baud_config_t* config);
int		zilog_mode_config(int fd, int normal, int intern);
void	zilog_rx_baud_compute(zilog_rx_baud_config_t* config);
void	zilog_tx_baud_compute(zilog_tx_baud_config_t* config);

// Auxiliary utility services

const gsc_reg_def_t*	sio4_reg_get_def_id(int id);
const gsc_reg_def_t*	sio4_reg_get_def_index(int index);
const char*				sio4_reg_get_desc(int reg);
const char*				sio4_reg_get_name(int reg);



#endif
