// $URL: http://subversion:8080/svn/gsc/trunk/drivers/LINUX/SIO4%20and%20SIO8/SIO4_Linux_1.x.x_GSC_DN/driver/usc.c $
// $Rev: 48525 $
// $Date: 2020-11-23 18:29:37 -0600 (Mon, 23 Nov 2020) $

// SIO4: Device Driver: source file

#include "main.h"



/******************************************************************************
*
*	Function:	usc_reg_vaddr_mod
*
*	Purpose:
*
*		Perform a read-modify-write operation on a USC register.
*
*	Arguments:
*
*		dev		The structure for the device to access.
*
*		vaddr	The base virtual address of the register to access.
*
*		value	The value to write to the register.
*
*		mask	The set of bits to modify.
*
*	Returned:
*
*		None.
*
******************************************************************************/

void usc_reg_vaddr_mod(dev_data_t* dev, VADDR_T vaddr, u16 value, u16 mask)
{
	u16	v;

	v		= usc_reg_vaddr_read(dev, vaddr);
	value	&= mask;
	v		&= ~mask;
	v		|= value;
	usc_reg_vaddr_write(dev, vaddr, v);
}



/******************************************************************************
*
*	Function:	usc_reg_vaddr_read
*
*	Purpose:
*
*		Read a value from a USC register given its base virtual address.
*
*	Arguments:
*
*		dev		The structure for the device to access.
*
*		vaddr	The base virtual address of the register to access.
*
*	Returned:
*
*		The value read.
*
******************************************************************************/

u16 usc_reg_vaddr_read(dev_data_t* dev, VADDR_T vaddr)
{
	u16	lb;
	u16	ub;
	u16	value;

	lb		= os_reg_mem_rx_u8(dev, vaddr);
	ub		= os_reg_mem_rx_u8(dev, (VADDR_T) ((long) vaddr + 1));
	value	= (u16) (ub << 8) | lb;
	return(value);
}



/******************************************************************************
*
*	Function:	usc_reg_vaddr_write
*
*	Purpose:
*
*		Write a value to a USC register given its base virtual address.
*
*	Arguments:
*
*		dev		The structure for the device to access.
*
*		vaddr	The base virtual address of the register to access.
*
*		value	The value to write to the register.
*
*	Returned:
*
*		None.
*
******************************************************************************/

void usc_reg_vaddr_write(dev_data_t* dev, VADDR_T vaddr, u16 value)
{
	u8	v;

	v		= value & 0xFF;
	os_reg_mem_tx_u8(dev, vaddr, v);
	v		= value >> 8;
	vaddr	= (VADDR_T) ((long) vaddr + 1);
	os_reg_mem_tx_u8(dev, vaddr, v);
}



/******************************************************************************
*
*	Function:	usc_reset_ioctl
*
*	Purpose:
*
*		Implement the USC Reset IOCTL service.
*
*	Arguments:
*
*		chan	The structure for the channel to initialize.
*
*		arg		The argument passed by the application.
*
*	Returned:
*
*		0		All went well.
*		< 0		An appropriate error status.
*
******************************************************************************/

s32 usc_reset_ioctl(chan_data_t* chan, void* arg)
{
	dev_data_t* dev		= chan->dev;
	s32			fw_type;
	int			ret;
	u16			value;

	ret	= fw_type_config_get(chan, &fw_type);

	if ((ret == 0) && (fw_type == SIO4_FW_TYPE_CONFIG_Z16C30))
	{
		// Fields which can be ignored and left at zero are omitted.
		// Apply and release the Software Reset.
		value	= SIO4_USC_CCAR_RTR_RESET;		// Software Reset
		usc_reg_vaddr_write(dev, chan->vaddr.usc_ccar, value);
		value	= SIO4_USC_CCAR_RTR_RELEASE;	// Reset Release
		usc_reg_vaddr_write(dev, chan->vaddr.usc_ccar, value);

		// Initialize signals to appropriate values.
		value	= SIO4_USC_HCR_TAM_DMAA		// REQUIRED FOR DMA
				| SIO4_USC_HCR_RAM_DMAA;	// REQUIRED FOR DMA
		usc_reg_vaddr_write(dev, chan->vaddr.usc_hcr, value);

		value	= SIO4_USC_IOCR_CTSM_IN		// A safe hardware option.
				| SIO4_USC_IOCR_DCDM_RCD_IN	// A safe hardware option.
				| SIO4_USC_IOCR_TRM_DMAR	// REQUIRED FOR DMA
				| SIO4_USC_IOCR_RRM_DMAR	// REQUIRED FOR DMA
				| SIO4_USC_IOCR_TDM_HIZ		// A safe hardware option.
				| SIO4_USC_IOCR_TCM_IN		// A safe hardware option.
				| SIO4_USC_IOCR_RCM_IN;		// A safe hardware option.
		usc_reg_vaddr_write(dev, chan->vaddr.usc_iocr, value);
	}

	return(ret);
}


