// $URL: http://subversion:8080/svn/gsc/trunk/drivers/LINUX/SIO4%20and%20SIO8/SIO4_Linux_1.x.x_GSC_DN/driver/tx.c $
// $Rev: 48525 $
// $Date: 2020-11-23 18:29:37 -0600 (Mon, 23 Nov 2020) $

// SIO4: Device Driver: source file

#include "main.h"



/******************************************************************************
*
*	Function:	tx_cable_clock_config_ioctl
*
*	Purpose:
*
*		Implement the Tx Cable Clock Config IOCTL service.
*
*	Arguments:
*
*		chan	The structure for the chanel being accessed.
*
*		arg		The argument passed by the user.
*
*	Returned:
*
*		0		All went well.
*		< 0		An appropriate error status.
*
******************************************************************************/

s32 tx_cable_clock_config_ioctl(chan_data_t* chan, void* arg)
{
	static const parm_32_t	list[]	=
	{
		{
		/* parm	*/	SIO4_TX_CABLE_CLOCK_CONFIG_PRG_CLK,
		/* mask	*/	SIO4_GSC_PSRCR_CBL_TCS,
		/* val	*/	0 << 6,
		},
		{
		/* parm	*/	SIO4_TX_CABLE_CLOCK_CONFIG_PRG_CLK_INV,
		/* mask	*/	SIO4_GSC_PSRCR_CBL_TCS,
		/* val	*/	1 << 6,
		},
		{
		/* parm	*/	SIO4_TX_CABLE_CLOCK_CONFIG_DRV_LOW,
		/* mask	*/	SIO4_GSC_PSRCR_CBL_TCS,
		/* val	*/	2 << 6,
		},
		{
		/* parm	*/	SIO4_TX_CABLE_CLOCK_CONFIG_DRV_HI,
		/* mask	*/	SIO4_GSC_PSRCR_CBL_TCS,
		/* val	*/	3 << 6,
		},
		{
		/* parm	*/	SIO4_TX_CABLE_CLOCK_CONFIG_USC_TC,
		/* mask	*/	SIO4_GSC_PSRCR_CBL_TCS,
		/* val	*/	4 << 6,
		},
		{
		/* parm	*/	SIO4_TX_CABLE_CLOCK_CONFIG_USC_RC,
		/* mask	*/	SIO4_GSC_PSRCR_CBL_TCS,
		/* val	*/	5 << 6,
		},
		{
		/* parm	*/	SIO4_TX_CABLE_CLOCK_CONFIG_CBL_RC,
		/* mask	*/	SIO4_GSC_PSRCR_CBL_TCS,
		/* val	*/	6 << 6,
		},
		{
		/* parm	*/	SIO4_TX_CABLE_CLOCK_CONFIG_CBL_RAC,
		/* mask	*/	SIO4_GSC_PSRCR_CBL_TCS,
		/* val	*/	7 << 6,
		},
		{ 0, 0, 0 }
	};

	dev_data_t*	dev		= chan->dev;
	int			index;
	s32			ret;
	s32*		value	= arg;

	if (value[0] == SIO4_TX_CABLE_CLOCK_CONFIG_READ)
		index	= 0;
	else if (dev->cache.gsc_frr_32 & SIO4_GSC_FRR_SIO4)
		index	= parm_32_set(dev, chan->vaddr.gsc_psrcr_32, (u32) value[0], list);
	else
		index	= parm_32_check(value[0], list);

	if (index < 0)
		ret	= -EINVAL;
	else
		ret	= 0;

	if (dev->cache.gsc_frr_32 & SIO4_GSC_FRR_SIO4)
	{
		index	= parm_32_get(dev, chan->vaddr.gsc_psrcr_32, list);

		if (index >= 0)
			value[0]	= list[index].parm;
		else
			value[0]	= SIO4_TX_CABLE_CLOCK_CONFIG_INVALID;
	}
	else
	{
		value[0]	= SIO4_TX_CABLE_CLOCK_CONFIG_UNKNOWN;
	}

	return(ret);
}



/******************************************************************************
*
*	Function:	tx_cable_config_ioctl
*
*	Purpose:
*
*		Implement the Tx Cable Config IOCTL service.
*
*	Arguments:
*
*		chan	The structure for the chanel being accessed.
*
*		arg		The argument passed by the user.
*
*	Returned:
*
*		0		All went well.
*		< 0		An appropriate error status.
*
******************************************************************************/

s32 tx_cable_config_ioctl(chan_data_t* chan, void* arg)
{
	static const parm_2x32_t	list[]	=
	{
		{
		/* parm		*/	SIO4_TX_CABLE_CONFIG_DISABLE,
		/* mask1	*/	_GSC_CCR_TCCE,
		/* value1	*/	0,
		/* shift1	*/	4,
		/* mask2	*/	SIO4_GSC_CSR_TDULE,
		/* value2	*/	0,
		/* shift2	*/	0
		},
		{
		/* parm		*/	SIO4_TX_CABLE_CONFIG_LOWER,
		/* mask1	*/	_GSC_CCR_TCCE,
		/* value1	*/	_GSC_CCR_TCCE_L,
		/* shift1	*/	4,
		/* mask2	*/	SIO4_GSC_CSR_TDULE,
		/* value2	*/	SIO4_GSC_CSR_TDLE,
		/* shift2	*/	0
		},
		{
		/* parm		*/	SIO4_TX_CABLE_CONFIG_UPPER,
		/* mask1	*/	_GSC_CCR_TCCE,
		/* value1	*/	_GSC_CCR_TCCE_U,
		/* shift1	*/	4,
		/* mask2	*/	SIO4_GSC_CSR_TDULE,
		/* value2	*/	SIO4_GSC_CSR_TDUE,
		/* shift2	*/	0
		},
		{
		/* parm		*/	SIO4_TX_CABLE_CONFIG_BOTH,
		/* mask1	*/	_GSC_CCR_TCCE,
		/* value1	*/	_GSC_CCR_TCCE_UL,
		/* shift1	*/	4,
		/* mask2	*/	SIO4_GSC_CSR_TDULE,
		/* value2	*/	SIO4_GSC_CSR_TDULE,
		/* shift2	*/	0
		},
		{ 0, 0, 0, 0, 0, 0, 0 }
	};

	dev_data_t*	dev		= chan->dev;
	int			index;
	s32			ret;
	s32*		value	= arg;

	if (chan->dev->cache.legacy_cable == 0)
	{
		return(0);
	}

	if (value[0] == SIO4_TX_CABLE_CONFIG_READ)
	{
		index	= 0;
	}
	else
	{
		index	= parm_2x32_set(chan,
								dev->vaddr.gsc_ccr_32,
								chan->vaddr.gsc_csr_32,
								value[0],
								list);
	}

	if (index < 0)
		ret	= -EINVAL;
	else
		ret	= 0;

	index		= parm_2x32_get(chan,
								dev->vaddr.gsc_ccr_32,
								chan->vaddr.gsc_csr_32,
								list);

	if (index >= 0)
		value[0]	= list[index].parm;
	else
		value[0]	= SIO4_TX_CABLE_CONFIG_INVALID;

	return(ret);
}



/******************************************************************************
*
*	Function:	tx_cable_data_config_ioctl
*
*	Purpose:
*
*		Implement the Tx Cable Data Config IOCTL service.
*
*	Arguments:
*
*		chan	The structure for the chanel being accessed.
*
*		arg		The argument passed by the user.
*
*	Returned:
*
*		0		All went well.
*		< 0		An appropriate error status.
*
******************************************************************************/

s32 tx_cable_data_config_ioctl(chan_data_t* chan, void* arg)
{
	static const parm_32_t	list[]	=
	{
		{
		/* parm	*/	SIO4_TX_CABLE_DATA_CONFIG_USC_TXD,
		/* mask	*/	SIO4_GSC_PSRCR_CBL_TXD,
		/* val	*/	0 << 19,
		},
		{
		/* parm	*/	SIO4_TX_CABLE_DATA_CONFIG_USC_TXD,
		/* mask	*/	SIO4_GSC_PSRCR_CBL_TXD,
		/* val	*/	1 << 19,
		},
		{
		/* parm	*/	SIO4_TX_CABLE_DATA_CONFIG_DRV_LOW,
		/* mask	*/	SIO4_GSC_PSRCR_CBL_TXD,
		/* val	*/	2 << 19,
		},
		{
		/* parm	*/	SIO4_TX_CABLE_DATA_CONFIG_DRV_HI,
		/* mask	*/	SIO4_GSC_PSRCR_CBL_TXD,
		/* val	*/	3 << 19,
		},
		{ 0, 0, 0 }
	};

	dev_data_t*	dev		= chan->dev;
	int			index;
	s32			ret;
	s32*		value	= arg;

	if (value[0] == SIO4_TX_CABLE_DATA_CONFIG_READ)
		index	= 0;
	else if (chan->cache.reg_psrcr)
		index	= parm_32_set(dev, chan->vaddr.gsc_psrcr_32, value[0], list);
	else
		index	= parm_32_check(value[0], list);

	if (index < 0)
		ret	= -EINVAL;
	else
		ret	= 0;

	if (dev->cache.gsc_frr_32 & SIO4_GSC_FRR_SIO4)
	{
		index	= parm_32_get(dev, chan->vaddr.gsc_psrcr_32, list);

		if (index >= 0)
			value[0]	= list[index].parm;
		else
			value[0]	= SIO4_TX_CABLE_DATA_CONFIG_INVALID;
	}
	else
	{
		value[0]	= SIO4_TX_CABLE_DATA_CONFIG_UNKNOWN;
	}

	return(ret);
}



/******************************************************************************
*
*	Function:	tx_fifo_ae_config_ioctl
*
*	Purpose:
*
*		Implement the Tx FIFO Almost Empty Config IOCTL service.
*
*	Arguments:
*
*		chan	The structure for the chanel being accessed.
*
*		arg		The argument passed by the user.
*
*	Returned:
*
*		0		All went well.
*		< 0		An appropriate error status.
*
******************************************************************************/

s32 tx_fifo_ae_config_ioctl(chan_data_t* chan, void* arg)
{
	s32*	value	= arg;

	value[0]	= fifo_almost_config(	chan,
										value[0],
										chan->vaddr.gsc_tar_32,
										0,
										SIO4_GSC_CSR_TFR);
	return(0);
}



/******************************************************************************
*
*	Function:	tx_fifo_af_config_ioctl
*
*	Purpose:
*
*		Implement the Tx FIFO Almost Full Config IOCTL service.
*
*	Arguments:
*
*		chan	The structure for the chanel being accessed.
*
*		arg		The argument passed by the user.
*
*	Returned:
*
*		0		All went well.
*		< 0		An appropriate error status.
*
******************************************************************************/

s32 tx_fifo_af_config_ioctl(chan_data_t* chan, void* arg)
{
	s32*	value	= arg;

	value[0]	= fifo_almost_config(	chan,
										value[0],
										chan->vaddr.gsc_tar_32,
										16,
										SIO4_GSC_CSR_TFR);
	return(0);
}



/******************************************************************************
*
*	Function:	tx_fifo_count_ioctl
*
*	Purpose:
*
*		Implement the Tx FIFO Count IOCTL service.
*
*	Arguments:
*
*		chan	The structure for the chanel being accessed.
*
*		arg		The argument passed by the user.
*
*	Returned:
*
*		0		All went well.
*		< 0		An appropriate error status.
*
******************************************************************************/

s32 tx_fifo_count_ioctl(chan_data_t* chan, void* arg)
{
	s32*	value	= arg;

	value[0]	= fifo_count(chan, 0);
	return(0);
}



/******************************************************************************
*
*	Function:	tx_fifo_size_ioctl
*
*	Purpose:
*
*		Implement the Tx FIFO Size IOCTL service.
*
*	Arguments:
*
*		chan	The structure for the chanel being accessed.
*
*		arg		The argument passed by the user.
*
*	Returned:
*
*		0		All went well.
*		< 0		An appropriate error status.
*
******************************************************************************/

s32 tx_fifo_size_ioctl(chan_data_t* chan, void* arg)
{
	s32*	value	= arg;

	value[0]	= chan->cache.fifo_size_tx;
	return(0);
}



// ****************************************************************************
s32 tx_fifo_status_ioctl(chan_data_t* chan, void* arg)
{
	u32		csr;
	s32*	value	= arg;

	csr			= os_reg_mem_rx_u32(chan->dev, chan->vaddr.gsc_csr_32);
	value[0]	= SIO4_DECODE(csr, 11, 8);

	if ((value[0] & 0x1) == 0x0)
		value[0]	= SIO4_FIFO_STATUS_EMPTY;
	else if ((value[0] & 0x2) == 0x0)
		value[0]	= SIO4_FIFO_STATUS_ALMOST_EMPTY;
	else if ((value[0] & 0x8) == 0x0)
		value[0]	= SIO4_FIFO_STATUS_FULL;
	else if ((value[0] & 0x4) == 0x0)
		value[0]	= SIO4_FIFO_STATUS_ALMOST_FULL;
	else
		value[0]	= SIO4_FIFO_STATUS_MEDIAN;

	return(0);
}



/******************************************************************************
*
*	Function:	tx_io_abort_ioctl
*
*	Purpose:
*
*		Implement the Tx I/O Abort IOCTL service.
*
*	Arguments:
*
*		chan	The structure for the channel being accessed.
*
*		arg		The argument passed by the user.
*
*	Returned:
*
*		0		All went well.
*		< 0		An appropriate error status.
*
******************************************************************************/

s32 tx_io_abort_ioctl(chan_data_t* chan, void* arg)
{
	s32*	ptr	= arg;

	ptr[0]	= io_abort(&chan->tx);
	return(0);
}



/******************************************************************************
*
*	Function:	tx_io_mode_config_ioctl
*
*	Purpose:
*
*		Implement the Tx I/O Mode Configuration IOCTL service.
*
*	Arguments:
*
*		chan	The structure for the chanel being accessed.
*
*		arg		The argument passed by the user.
*
*	Returned:
*
*		0		All went well.
*		< 0		An appropriate error status.
*
******************************************************************************/

s32 tx_io_mode_config_ioctl(chan_data_t* chan, void* arg)
{
	s32		ret;
	s32*	value	= arg;

	ret	= io_mode_config(&chan->tx, value);
	return(ret);
}



/******************************************************************************
*
*	Function:	txc_usc_config_ioctl
*
*	Purpose:
*
*		Implement the TxC USC Configuration IOCTL service.
*
*	Arguments:
*
*		chan	The structure for the chanel being accessed.
*
*		arg		The argument passed by the user.
*
*	Returned:
*
*		0		All went well.
*		< 0		An appropriate error status.
*
******************************************************************************/

s32 txc_usc_config_ioctl(chan_data_t* chan, void* arg)
{
	static const parm_32_usc_t	list[]	=
	{
		{
		/* parm		*/	SIO4_TXC_USC_CONFIG_IN_LOW,
		/* mask1	*/	SIO4_GSC_PSRCR_USC_TCS,
		/* value1	*/	0x00000002,
		/* mask2	*/	SIO4_USC_IOCR_TCM,
		/* value2	*/	0x0000
		},
		{
		/* parm		*/	SIO4_TXC_USC_CONFIG_IN_HI,
		/* mask1	*/	SIO4_GSC_PSRCR_USC_TCS,
		/* value1	*/	0x00000003,
		/* mask2	*/	SIO4_USC_IOCR_TCM,
		/* value2	*/	0x0000
		},
		{
		/* parm		*/	SIO4_TXC_USC_CONFIG_IN_PRG_CLK,
		/* mask1	*/	SIO4_GSC_PSRCR_USC_TCS,
		/* value1	*/	0x00000000,
		/* mask2	*/	SIO4_USC_IOCR_TCM,
		/* value2	*/	0x0000
		},
		{
		/* parm		*/	SIO4_TXC_USC_CONFIG_IN_PRG_CLK_INV,
		/* mask1	*/	SIO4_GSC_PSRCR_USC_TCS,
		/* value1	*/	0x00000001,
		/* mask2	*/	SIO4_USC_IOCR_TCM,
		/* value2	*/	0x0000
		},
		{
		/* parm		*/	SIO4_TXC_USC_CONFIG_IN_CBL_RC,
		/* mask1	*/	SIO4_GSC_PSRCR_USC_TCS,
		/* value1	*/	0x00000004,
		/* mask2	*/	SIO4_USC_IOCR_TCM,
		/* value2	*/	0x0000
		},
		{
		/* parm		*/	SIO4_TXC_USC_CONFIG_OUT_TCLK,
		/* mask1	*/	SIO4_GSC_PSRCR_USC_TCS,
		/* value1	*/	0x00000007,
		/* mask2	*/	SIO4_USC_IOCR_TCM,
		/* value2	*/	0x0008
		},
		{
		/* parm		*/	SIO4_TXC_USC_CONFIG_OUT_TCC,
		/* mask1	*/	SIO4_GSC_PSRCR_USC_TCS,
		/* value1	*/	0x00000007,
		/* mask2	*/	SIO4_USC_IOCR_TCM,
		/* value2	*/	0x0010
		},
		{
		/* parm		*/	SIO4_TXC_USC_CONFIG_OUT_TCOMP,
		/* mask1	*/	SIO4_GSC_PSRCR_USC_TCS,
		/* value1	*/	0x00000007,
		/* mask2	*/	SIO4_USC_IOCR_TCM,
		/* value2	*/	0x0018
		},
		{
		/* parm		*/	SIO4_TXC_USC_CONFIG_OUT_BRG0,
		/* mask1	*/	SIO4_GSC_PSRCR_USC_TCS,
		/* value1	*/	0x00000007,
		/* mask2	*/	SIO4_USC_IOCR_TCM,
		/* value2	*/	0x0020
		},
		{
		/* parm		*/	SIO4_TXC_USC_CONFIG_OUT_BRG1,
		/* mask1	*/	SIO4_GSC_PSRCR_USC_TCS,
		/* value1	*/	0x00000007,
		/* mask2	*/	SIO4_USC_IOCR_TCM,
		/* value2	*/	0x0028
		},
		{
		/* parm		*/	SIO4_TXC_USC_CONFIG_OUT_CTR1,
		/* mask1	*/	SIO4_GSC_PSRCR_USC_TCS,
		/* value1	*/	0x00000007,
		/* mask2	*/	SIO4_USC_IOCR_TCM,
		/* value2	*/	0x0030
		},
		{
		/* parm		*/	SIO4_TXC_USC_CONFIG_OUT_DPLL,
		/* mask1	*/	SIO4_GSC_PSRCR_USC_TCS,
		/* value1	*/	0x00000007,
		/* mask2	*/	SIO4_USC_IOCR_TCM,
		/* value2	*/	0x0038
		},
		{ 0, 0, 0, 0, 0 }
	};

	dev_data_t*	dev		= chan->dev;
	s32			fw_type;
	int			index;
	s32			ret;
	s32*		value	= arg;

	fw_type_config_get(chan, &fw_type);

	if (fw_type != SIO4_FW_TYPE_CONFIG_Z16C30)
	{
		value[0]	= SIO4_TXC_USC_CONFIG_UNKNOWN;
		return(0);
	}

	if (value[0] == SIO4_TXC_USC_CONFIG_READ)
	{
		index	= 0;
	}
	else if (chan->cache.reg_psrcr)
	{
		index	= parm_32_usc_set(	dev,
									chan->vaddr.gsc_psrcr_32,
									chan->vaddr.usc_iocr,
									value[0],
									list);
	}
	else
	{
		index	= parm_32_usc_check(value[0], list);
	}

	if (index < 0)
		ret	= -EINVAL;
	else
		ret	= 0;

	if (chan->cache.reg_psrcr)
	{
		index		= parm_32_usc_get(	dev,
										chan->vaddr.gsc_psrcr_32,
										chan->vaddr.usc_iocr,
										list);

		if (index >= 0)
			value[0]	= list[index].parm;
		else
			value[0]	= SIO4_TXC_USC_CONFIG_INVALID;
	}
	else
	{
		value[0]	= SIO4_TXC_USC_CONFIG_UNKNOWN;
	}

	return(ret);
}



