// $URL: http://subversion:8080/svn/gsc/trunk/drivers/LINUX/SIO4%20and%20SIO8/SIO4_Linux_1.x.x_GSC_DN/driver/sio4ioctl.c $
// $Rev: 53095 $
// $Date: 2023-06-13 10:42:41 -0500 (Tue, 13 Jun 2023) $

// SIO4: Device Driver: source file

#include "main.h"



// variables ******************************************************************

static	const u8	au8DPLLSources[8]	= {0,2,3,0,0,1,0,0};
static	const u8	au8BRGSources[8]	= {0,2,3,0,0,0,0,1};
static	const u8	au8CTRSources[8]	= {0,2,3,0,0,0,0,0};



/******************************************************************************
*
*	Function:	cable_config_ioctl
*
*	Purpose:
*
*		Implement the Cable Config IOCTL service.
*
*	Arguments:
*
*		chan	The structure for the chanel being accessed.
*
*		arg		The argument passed by the user.
*
*	Returned:
*
*		0		All went well.
*		< 0		An appropriate error status.
*
******************************************************************************/

s32 cable_config_ioctl(chan_data_t* chan, void* arg)
{
	u32			ccr_msk	= 0xF << (chan->index * 4);
	u32			ccr_val;
	u32			csr_msk	= 0x3C;
	u32			csr_val;
	dev_data_t*	dev		= chan->dev;
	s32			ret;
	s32*		value	= arg;

	if (chan->dev->cache.legacy_cable == 0)
	{
		value[0]	= SIO4_CABLE_CONFIG_INVALID;
		return(0);
	}

	if (value[0] == SIO4_CABLE_CONFIG_READ)
	{
		ret	= 0;
	}
	else if (value[0] & ~SIO4_CABLE_CONFIG__MASK_)
	{
		ret	= -EINVAL;
	}
	else
	{
		ret	= 0;
		ccr_val	= value[0] << (chan->index * 4);
		csr_val	= value[0] << 2;
		os_reg_mem_mx_u32(dev, dev->vaddr.gsc_ccr_32, ccr_val, ccr_msk);
		os_reg_mem_mx_u32(dev, chan->vaddr.gsc_csr_32, csr_val, csr_msk);
	}

	if (ret == 0)
	{
		ccr_val	= os_reg_mem_rx_u32(dev, dev->vaddr.gsc_ccr_32);
		ccr_val	= (ccr_val & ccr_msk) >> (chan->index * 4);
		csr_val	= os_reg_mem_rx_u32(dev, chan->vaddr.gsc_csr_32);
		csr_val	= (csr_val & csr_msk) >> 2;

		if (ccr_val == csr_val)
			value[0]	= ccr_val;
		else
			value[0]	= SIO4_CABLE_CONFIG_INVALID;
	}

	return(ret);
}



/******************************************************************************
*
*	Function:	cts_cable_config_ioctl
*
*	Purpose:
*
*		Implement the CTS Cable Configuration IOCTL service.
*
*	Arguments:
*
*		chan	The structure for the chanel being accessed.
*
*		arg		The argument passed by the user.
*
*	Returned:
*
*		0		All went well.
*		< 0		An appropriate error status.
*
******************************************************************************/

s32 cts_cable_config_ioctl(chan_data_t* chan, void* arg)
{
	static const parm_32_usc_t	list[]	=
	{
		{			// The preferred configuration.
		/* parm		*/	SIO4_CTS_CABLE_CONFIG_DISABLE,
		/* mask1	*/	0x00000600,
		/* value1	*/	0x00000000,
		/* mask2	*/	0x8000,
		/* value2	*/	0x8000
		},
		{			// An alternate configuration.
		/* parm		*/	SIO4_CTS_CABLE_CONFIG_DISABLE,
		/* mask1	*/	0x00000600,
		/* value1	*/	0x00000000,
		/* mask2	*/	0x0000,
		/* value2	*/	0x0000
		},
		{			// The preferred configuration.
		/* parm		*/	SIO4_CTS_CABLE_CONFIG_CTS_IN,
		/* mask1	*/	0x00000600,
		/* value1	*/	0x00000200,
		/* mask2	*/	0x8000,
		/* value2	*/	0x0000
		},
		{			// The preferred configuration.
		/* parm		*/	SIO4_CTS_CABLE_CONFIG_DRV_LOW,
		/* mask1	*/	0x00006600,
		/* value1	*/	0x00004600,
		/* mask2	*/	0xC000,
		/* value2	*/	0x8000
		},
		{			// An alternate configuration.
		/* parm		*/	SIO4_CTS_CABLE_CONFIG_DRV_LOW,
		/* mask1	*/	0x00006600,
		/* value1	*/	0x00004600,
		/* mask2	*/	0x0000,
		/* value2	*/	0x0000
		},
		{			// An alternate configuration.
		/* parm		*/	SIO4_CTS_CABLE_CONFIG_DRV_LOW,
		/* mask1	*/	0x00006600,
		/* value1	*/	0x00000600,
		/* mask2	*/	0xC000,
		/* value2	*/	0x8000
		},
		{			// The preferred configuration.
		/* parm		*/	SIO4_CTS_CABLE_CONFIG_DRV_HI,
		/* mask1	*/	0x00006600,
		/* value1	*/	0x00006600,
		/* mask2	*/	0xC000,
		/* value2	*/	0xC000
		},
		{			// An alternate configuration.
		/* parm		*/	SIO4_CTS_CABLE_CONFIG_DRV_HI,
		/* mask1	*/	0x00006600,
		/* value1	*/	0x00006600,
		/* mask2	*/	0x0000,
		/* value2	*/	0x0000
		},
		{			// An alternate configuration.
		/* parm		*/	SIO4_CTS_CABLE_CONFIG_DRV_HI,
		/* mask1	*/	0x00006600,
		/* value1	*/	0x00000600,
		/* mask2	*/	0xC000,
		/* value2	*/	0xC000
		},
		{			// The preferred configuration.
		/* parm		*/	SIO4_CTS_CABLE_CONFIG_DCD_IN,
		/* mask1	*/	0x00000600,
		/* value1	*/	0x00000400,
		/* mask2	*/	0x8000,
		/* value2	*/	0x0000
		},
		{			// The preferred configuration.
		/* parm		*/	SIO4_CTS_CABLE_CONFIG_RTS_OUT,
		/* mask1	*/	0x00006600,
		/* value1	*/	0x00002600,
		/* mask2	*/	0x8000,
		/* value2	*/	0x8000
		},
		{			// An alternate configuration.
		/* parm		*/	SIO4_CTS_CABLE_CONFIG_RTS_OUT,
		/* mask1	*/	0x00006600,
		/* value1	*/	0x00002600,
		/* mask2	*/	0x0000,
		/* value2	*/	0x0000
		},

		{ 0, 0, 0, 0, 0 }
	};

	dev_data_t*	dev		= chan->dev;
	s32			fw_type;
	int			index;
	s32			ret;
	s32*		value	= arg;

	fw_type_config_get(chan, &fw_type);

	if (fw_type != SIO4_FW_TYPE_CONFIG_Z16C30)
	{
		value[0]	= SIO4_CTS_CABLE_CONFIG_UNKNOWN;
		return(0);
	}

	if (value[0] == SIO4_CTS_CABLE_CONFIG_READ)
	{
		index	= 0;
	}
	else if (chan->cache.reg_psrcr)
	{
		index	= parm_32_usc_set(	dev,
									chan->vaddr.gsc_psrcr_32,
									chan->vaddr.usc_iocr,
									value[0],
									list);
	}
	else
	{
		index	= parm_32_usc_check(value[0], list);
	}

	if (index < 0)
		ret	= -EINVAL;
	else
		ret	= 0;

	if (chan->cache.reg_psrcr)
	{
		index	= parm_32_usc_get(	dev,
									chan->vaddr.gsc_psrcr_32,
									chan->vaddr.usc_iocr,
									list);

		if (index >= 0)
			value[0]	= list[index].parm;
		else
			value[0]	= SIO4_CTS_CABLE_CONFIG_INVALID;
	}
	else
	{
		value[0]	= SIO4_CTS_CABLE_CONFIG_UNKNOWN;
	}

	return(ret);
}



/******************************************************************************
*
*	Function:	dcd_cable_config_ioctl
*
*	Purpose:
*
*		Implement the DCD Cable Configuration IOCTL service.
*
*	Arguments:
*
*		chan	The structure for the chanel being accessed.
*
*		arg		The argument passed by the user.
*
*	Returned:
*
*		0		All went well.
*		< 0		An appropriate error status.
*
******************************************************************************/

s32 dcd_cable_config_ioctl(chan_data_t* chan, void* arg)
{
	static const parm_32_usc_t	list[]	=
	{
		{
		/* parm		*/	SIO4_DCD_CABLE_CONFIG_DISABLE,
		/* mask1	*/	SIO4_GSC_PSRCR_CBL_DCD,
		/* value1	*/	0x00000000,
		/* mask2	*/	0x2000,
		/* value2	*/	0x2000
		},
		{
		/* parm		*/	SIO4_DCD_CABLE_CONFIG_DISABLE,
		/* mask1	*/	SIO4_GSC_PSRCR_CBL_DCD,
		/* value1	*/	0x00000000,
		/* mask2	*/	0x0000,
		/* value2	*/	0x0000
		},
		{
		/* parm		*/	SIO4_DCD_CABLE_CONFIG_DCD_IN,
		/* mask1	*/	SIO4_GSC_PSRCR_CBL_DCD,
		/* value1	*/	0x00000800,
		/* mask2	*/	0x3000,
		/* value2	*/	0x0000
		},
		{
		/* parm		*/	SIO4_DCD_CABLE_CONFIG_DCD_IN_SYNC,
		/* mask1	*/	SIO4_GSC_PSRCR_CBL_DCD,
		/* value1	*/	0x00000800,
		/* mask2	*/	0x3000,
		/* value2	*/	0x1000
		},
		{
		/* parm		*/	SIO4_DCD_CABLE_CONFIG_CTS_IN,
		/* mask1	*/	SIO4_GSC_PSRCR_CBL_DCD,
		/* value1	*/	0x00001000,
		/* mask2	*/	0x3000,
		/* value2	*/	0x0000
		},
		{
		/* parm		*/	SIO4_DCD_CABLE_CONFIG_CTS_IN_SYNC,
		/* mask1	*/	SIO4_GSC_PSRCR_CBL_DCD,
		/* value1	*/	0x00001000,
		/* mask2	*/	0x3000,
		/* value2	*/	0x1000
		},
		{			// The preferred configuration.
		/* parm		*/	SIO4_DCD_CABLE_CONFIG_RTS_OUT,
		/* mask1	*/	SIO4_GSC_PSRCR_CBL_DCD |
						SIO4_GSC_PSRCR_CBL_DCD_OUT,
		/* value1	*/	0x00009800,
		/* mask2	*/	0x2000,
		/* value2	*/	0x2000
		},
		{			// An alternate configuration.
		/* parm		*/	SIO4_DCD_CABLE_CONFIG_RTS_OUT,
		/* mask1	*/	SIO4_GSC_PSRCR_CBL_DCD |
						SIO4_GSC_PSRCR_CBL_DCD_OUT,
		/* value1	*/	0x00009800,
		/* mask2	*/	0x0000,
		/* value2	*/	0x0000
		},
		{			// The preferred configuration.
		/* parm		*/	SIO4_DCD_CABLE_CONFIG_DRV_HI,
		/* mask1	*/	SIO4_GSC_PSRCR_CBL_DCD |
						SIO4_GSC_PSRCR_CBL_DCD_OUT,
		/* value1	*/	0x00019800,
		/* mask2	*/	0x3000,
		/* value2	*/	0x3000
		},
		{			// An alternate configuration.
		/* parm		*/	SIO4_DCD_CABLE_CONFIG_DRV_HI,
		/* mask1	*/	SIO4_GSC_PSRCR_CBL_DCD |
						SIO4_GSC_PSRCR_CBL_DCD_OUT,
		/* value1	*/	0x00019800,
		/* mask2	*/	0x0000,
		/* value2	*/	0x0000
		},
		{			// An alternate configuration.
		/* parm		*/	SIO4_DCD_CABLE_CONFIG_DRV_HI,
		/* mask1	*/	SIO4_GSC_PSRCR_CBL_DCD |
						SIO4_GSC_PSRCR_CBL_DCD_OUT,
		/* value1	*/	0x00001800,
		/* mask2	*/	0x3000,
		/* value2	*/	0x3000
		},
		{			// The preferred configuration.
		/* parm		*/	SIO4_DCD_CABLE_CONFIG_DRV_LOW,
		/* mask1	*/	SIO4_GSC_PSRCR_CBL_DCD |
						SIO4_GSC_PSRCR_CBL_DCD_OUT,
		/* value1	*/	0x00011800,
		/* mask2	*/	0x3000,
		/* value2	*/	0x2000
		},
		{			// An alternate configuration.
		/* parm		*/	SIO4_DCD_CABLE_CONFIG_DRV_LOW,
		/* mask1	*/	SIO4_GSC_PSRCR_CBL_DCD |
						SIO4_GSC_PSRCR_CBL_DCD_OUT,
		/* value1	*/	0x00011800,
		/* mask2	*/	0x0000,
		/* value2	*/	0x0000
		},
		{			// An alternate configuration.
		/* parm		*/	SIO4_DCD_CABLE_CONFIG_DRV_LOW,
		/* mask1	*/	SIO4_GSC_PSRCR_CBL_DCD |
						SIO4_GSC_PSRCR_CBL_DCD_OUT,
		/* value1	*/	0x00001800,
		/* mask2	*/	0x3000,
		/* value2	*/	0x2000
		},

		{ 0, 0, 0, 0, 0 }
	};

	dev_data_t*	dev		= chan->dev;
	s32			fw_type;
	int			index;
	s32			ret;
	s32*		value	= arg;

	fw_type_config_get(chan, &fw_type);

	if (fw_type != SIO4_FW_TYPE_CONFIG_Z16C30)
	{
		value[0]	= SIO4_DCD_CABLE_CONFIG_UNKNOWN;
		return(0);
	}

	if (value[0] == SIO4_DCD_CABLE_CONFIG_READ)
	{
		index	= 0;
	}
	else if (chan->cache.reg_psrcr)
	{
		index	= parm_32_usc_set(	dev,
									chan->vaddr.gsc_psrcr_32,
									chan->vaddr.usc_iocr,
									value[0],
									list);
	}
	else
	{
		index	= parm_32_usc_check(value[0], list);
	}

	if (index < 0)
		ret	= -EINVAL;
	else
		ret	= 0;

	if (chan->cache.reg_psrcr)
	{
		index	= parm_32_usc_get(	dev,
									chan->vaddr.gsc_psrcr_32,
									chan->vaddr.usc_iocr,
									list);

		if (index >= 0)
			value[0]	= list[index].parm;
		else
			value[0]	= SIO4_DCD_CABLE_CONFIG_INVALID;
	}
	else
	{
		value[0]	= SIO4_DCD_CABLE_CONFIG_UNKNOWN;
	}

	return(ret);
}



/******************************************************************************
* SIO4InitChannel - function invoked when the application calls ioctl with
*	the option SIO4_INIT_CHANNEL
*
* This function initilizes the channel. Sets the clock control, idle
* line conditions almost values etc.
*
* RETURNS : 0 = OK, <0 = Error
*/
int SIO4InitChannel
(
chan_data_t*	chan,	// pointer to the channel structure
void*			pArg	// channel initializing values
)
{
	dev_data_t*		dev				= chan->dev;
	SIO4_INIT_CHAN*	pstrInitChan	= (SIO4_INIT_CHAN *) pArg;
	s32				fw_type;
	u32*			pu32Addr;
	u8*				pu8Addr;
	u32				u32Val;
	u32				u32RegVal;
	u8				u8ChanNum;

	fw_type_config_get(chan, &fw_type);

	if (fw_type != SIO4_FW_TYPE_CONFIG_Z16C30)
		return(0);

	// Make sure the parameters are valid.
	if ((pstrInitChan					== ((SIO4_INIT_CHAN *) NULL))	||
		(pstrInitChan->u32BaudRate		< 1)							||
		(pstrInitChan->u32BaudRate		> 20000000L)					||
		(pstrInitChan->eMode			< NORMAL)						||
		(pstrInitChan->eMode			> INT_LOCAL_LOOPBACK)			||
		(pstrInitChan->eRxEnable		< DISABLE_IMMED)				||
		(pstrInitChan->eRxEnable		> ENABLE_WITH_AUTO)				||
		(pstrInitChan->eRxDataFormat	< NRZ)							||
		(pstrInitChan->eRxDataFormat	> DIFF_BIPHASE_LEVEL)			||
		(pstrInitChan->eRxDataLength	< BITS8)						||
		(pstrInitChan->eRxDataLength	> BITS7)						||
		(pstrInitChan->eRxParityType	< EVEN_PARITY)					||
		(pstrInitChan->eRxParityType	> MARK_PARITY)					||
		(pstrInitChan->eTxEnable		< DISABLE_IMMED)				||
		(pstrInitChan->eTxEnable		> ENABLE_WITH_AUTO)				||
		(pstrInitChan->eTxDataFormat	< NRZ)							||
		(pstrInitChan->eTxDataFormat	> DIFF_BIPHASE_LEVEL)			||
		(pstrInitChan->eTxDataLength	< BITS8)						||
		(pstrInitChan->eTxDataLength	> BITS7)						||
		(pstrInitChan->eTxParityType	< EVEN_PARITY)					||
		(pstrInitChan->eTxParityType	> MARK_PARITY)					||
		(pstrInitChan->eTxIdleLineCond	< SYNC_FLAG_NORMAL_IDLE)		||
		(pstrInitChan->eTxIdleLineCond	> MARK_IDLE))
	{
		return(-EINVAL);
	}

	pu32Addr = (u32*) chan->dev->vaddr.gsc_regs;
	pu8Addr = (u8*) pu32Addr;

	// initilize the clocks
	u8ChanNum = (u8) chan->index + 1;

	u32RegVal = os_reg_mem_rx_u32(dev, chan->dev->vaddr.gsc_ccr_32);
	u32RegVal &= ~((u32) 0xf << ((u8ChanNum -1) * 4));

	u32Val = 0;
	u32Val |= (pstrInitChan->u8EnableRxCableUpper & 0x1) << EN_RX_UPPER_SHIFT;
	u32Val |= (pstrInitChan->u8EnableRxCableLower & 0x1) << EN_RX_LOWER_SHIFT;
	u32Val |= (pstrInitChan->u8EnableTxCableUpper & 0x1) << EN_TX_UPPER_SHIFT;
	u32Val |= (pstrInitChan->u8EnableTxCableLower & 0x1) << EN_TX_LOWER_SHIFT;

	u32Val = u32Val << ((u8ChanNum - 1) * 4);
	u32Val |= u32RegVal;

	os_reg_mem_tx_u32(dev, chan->dev->vaddr.gsc_ccr_32, u32Val);

	// set Tx Idle line conditions
	u32RegVal = os_reg_mem_rx_u8(dev, (VADDR_T) (pu8Addr + ZILOG_TX_CMD_STAT_HIGH_REG(chan)));

	u32RegVal &= ~IDLE_LINE_COND_MASK;

	u32RegVal &= ~WAIT_ON_UNDERRUN_MASK;

	pstrInitChan->u8TxWaitOnUnderrun &= 0x01;

	u32RegVal |= ((pstrInitChan->u8TxWaitOnUnderrun <<
					TX_WAIT_ON_UNDERRUN_SHIFT) |
				(pstrInitChan->eTxIdleLineCond	<<
				TX_IDLE_LINE_COND_SHIFT));

	os_reg_mem_tx_u8(dev, (VADDR_T) (pu8Addr + ZILOG_TX_CMD_STAT_HIGH_REG(chan)), u32RegVal);

	// set the almost values
	u32Val = (pstrInitChan->u16TxAlmostEmpty << 0) |
			(pstrInitChan->u16TxAlmostFull << 16);

	os_reg_mem_tx_u32(dev, chan->vaddr.gsc_tar_32, u32Val);

	u32Val = (pstrInitChan->u16RxAlmostEmpty << 0) |
			(pstrInitChan->u16RxAlmostFull << 16);

	os_reg_mem_tx_u32(dev, chan->vaddr.gsc_rar_32, u32Val);

	// Reset FIFO here
	SIO4ResetFifo(chan, (void*) TX_AND_RX_FIFO);

	// enable cable clock edges
	if (chan->dev->cache.legacy_cable)
	{
		u32RegVal = os_reg_mem_rx_u32(dev, chan->vaddr.gsc_csr_32);
		u32RegVal &= ~0x3C;
		u32Val = 0;
		u32Val |= ((pstrInitChan->u8EnableRxCableUpper & 0x01) << 4);
		u32Val |= ((pstrInitChan->u8EnableRxCableLower & 0x01) << 5);
		u32Val |= ((pstrInitChan->u8EnableTxCableUpper & 0x01) << 2);
		u32Val |= ((pstrInitChan->u8EnableTxCableLower & 0x01) << 3);

		u32RegVal |= u32Val;
		os_reg_mem_tx_u32(dev, chan->vaddr.gsc_csr_32, u32RegVal);
	}

	// set the baud rate
	u32RegVal	= (((SIO4_OSC_FREQ_REF_DEFAULT / 4)
			/ pstrInitChan->u32BaudRate))
			- 1;

	os_reg_mem_tx_u8(dev, (VADDR_T) (pu8Addr + ZILOG_TIME_CONST0_LOW_REG(chan)), u32RegVal);

	// removing the LSB & shifting the MSB byte into the LSB
	u32RegVal >>= 8;
	os_reg_mem_tx_u8(dev, (VADDR_T) (pu8Addr + ZILOG_TIME_CONST0_HIGH_REG(chan)), u32RegVal);

	// set the transmit mode low register
	u32Val = (pstrInitChan->eTxEnable << ENABLE_SHIFT) |
			(pstrInitChan->eTxDataLength << DATA_LENGTH_SHIFT) |
			((pstrInitChan->u8TxParityEnable & 0x01) << PARITY_ENABLE_SHIFT)|
			(pstrInitChan->eTxParityType << PARITY_TYPE_SHIFT);

	os_reg_mem_tx_u8(dev, (VADDR_T) (pu8Addr + ZILOG_TX_MODE_LOW_REG(chan)), u32Val);

	// set the transmit mode high register
	u32RegVal = os_reg_mem_rx_u8(dev, (VADDR_T) (pu8Addr + ZILOG_TX_MODE_HIGH_REG(chan)));

	u32RegVal &= ~DATA_FORMAT_MASK;
	u32RegVal |= (pstrInitChan->eTxDataFormat << DATA_FORMAT_SHIFT);

	os_reg_mem_tx_u8(dev, (VADDR_T) (pu8Addr + ZILOG_TX_MODE_HIGH_REG(chan)), u32RegVal);

	// set the receive mode low register
	u32Val = (pstrInitChan->eRxEnable << ENABLE_SHIFT) |
			(pstrInitChan->eRxDataLength << DATA_LENGTH_SHIFT) |
			((pstrInitChan->u8RxParityEnable & 0x01) << PARITY_ENABLE_SHIFT)|
			(pstrInitChan->eRxParityType << PARITY_TYPE_SHIFT);


	os_reg_mem_tx_u8(dev, (VADDR_T) (pu8Addr + ZILOG_RX_MODE_LOW_REG(chan)), u32Val);

	// set the transmit mode high register
	u32RegVal = os_reg_mem_rx_u8(dev, (VADDR_T) (pu8Addr + ZILOG_RX_MODE_HIGH_REG(chan)));
	u32RegVal &= ~DATA_FORMAT_MASK;
	u32RegVal |= (pstrInitChan->eRxDataFormat << DATA_FORMAT_SHIFT);

	os_reg_mem_tx_u8(dev, (VADDR_T) (pu8Addr + ZILOG_RX_MODE_HIGH_REG(chan)), u32RegVal);

	// set the mode
	u32RegVal = os_reg_mem_rx_u8(dev, (VADDR_T) (pu8Addr + ZILOG_CMD_ADD_HIGH_REG(chan)));
	u32RegVal &= ~ZILOG_MODE_MASK;
	u32RegVal |= pstrInitChan->eMode;

	os_reg_mem_tx_u8(dev, (VADDR_T) (pu8Addr + ZILOG_CMD_ADD_HIGH_REG(chan)), u32RegVal);

	return(0);
}

/******************************************************************************
* SIO4SendChanCmd - function invoked when the application calls ioctl with
*	the option SIO4_SEND_CHANNEL_COMMAND
*
* This function writes the channel command sent by the user into the
* command address high register bits 7..3.
*
* RETURNS : 0 = OK, <0 = Error
*/
int SIO4SendChanCmd(chan_data_t* chan, void* arg)
{
	dev_data_t*		dev		= chan->dev;
	s32				fw_type;
	u32				u32RegVal;
	SIO4_CHAN_CMD	ChanCmd = (SIO4_CHAN_CMD) arg;
	u8*				pu8Addr;

	fw_type_config_get(chan, &fw_type);

	if (fw_type != SIO4_FW_TYPE_CONFIG_Z16C30)
		return(0);

	// make sure the parameter is good
	if ((ChanCmd < NULL_CMD) || (ChanCmd > RX_PURGE_CMD))
	{
		return(-EINVAL);
	}

	pu8Addr = (u8*) chan->dev->vaddr.gsc_regs;
	u32RegVal = os_reg_mem_rx_u8(dev, (VADDR_T) (pu8Addr + ZILOG_CMD_ADD_HIGH_REG(chan)));

	u32RegVal &= ~CHAN_CMD_MASK;
	u32RegVal |= ChanCmd << CHAN_CMD_SHIFT;

	// write the command into the channel command field
	os_reg_mem_tx_u8(dev, (VADDR_T) (pu8Addr + ZILOG_CMD_ADD_HIGH_REG(chan)), u32RegVal);

	// u32RegVal = os_reg_mem_rx_u8(dev, (VADDR_T) (pu8Addr + ZILOG_CMD_ADD_HIGH_REG(chan)));

	return(0);
}

/******************************************************************************
* SIO4ResetFifo - function invoked when the application calls ioctl with
*	the option SIO4_RESET_FIFO
*
* This function resets the FIFO, either Tx FIFO, Rx FIFO or both depending
* on the argument sent by the user. This is accomplished by setting bit 0
* for Tx FIFO & setting Bit 1 for Rx FIFO in the appropriate channel' s
* Control status register.
*
* RETURNS : 0 = OK, <0 = Error
*/
int SIO4ResetFifo(chan_data_t* chan, void* arg)
{
	dev_data_t*	dev		= chan->dev;
	u32			u32RegVal;
	TX_RX		TxRx	= (TX_RX) arg;

	// make sure the parameter is good
	if ((TxRx < TX_FIFO) || (TxRx > TX_AND_RX_FIFO))
	{
		return(-EINVAL);
	}

	u32RegVal	= os_reg_mem_rx_u32(dev, chan->vaddr.gsc_csr_32);
	u32RegVal	|= TxRx;

	// reset the FIFOs
	os_reg_mem_tx_u32(dev, chan->vaddr.gsc_csr_32, u32RegVal);
	os_time_us_delay(10000);	// worst case for older devices
	return(0);
}

/******************************************************************************
* SIO4SetSyncByte - The function sets the sync byte value.
*
* The sync byte may be used to trigger certain conditions when the
* byte value is received.
*
* RETURNS : 0 = OK, <0 = Error
*/

int SIO4SetSyncByte(chan_data_t* chan, void* arg)
{
	dev_data_t*	dev		= chan->dev;
	s32			fw_type;
	u32			u32Arg	= (u32) (unsigned long) arg;
	u8*			pu8Addr;

	fw_type_config_get(chan, &fw_type);

	if (fw_type == SIO4_FW_TYPE_CONFIG_Z16C30)
	{
		pu8Addr	= (u8*) chan->dev->vaddr.gsc_regs;

		os_reg_mem_tx_u8(dev, (VADDR_T) (pu8Addr + ZILOG_RX_SYNC_LOW_REG(chan)), u32Arg & 0xff);
	}

	return(0);
}

/******************************************************************************
* SIO4SetReadTimeout - The function sets the timeout used for read from serial
*	channel.
* This timeout value is given in clock ticks. Any access to the driver
* read functions will be terminated if the amount of characters has not been
* received within the timeout value.
*
* NOTE: A timeout value of "0" means "don't wait" and -1 means "wait for ever"
*
* RETURNS : 0 = OK, <0 = Error
*/
int SIO4SetReadTimeout(chan_data_t* chan, void* arg)
{
	s32	s32Arg	= (s32) (unsigned long) arg;

	// if the input is any negative number make it -1 to wait forever
	if ((s32Arg < 0) && (s32Arg != -1))
		s32Arg	= -1;

	chan->rx.timeout = s32Arg;
	return(0);
}

/******************************************************************************
* SIO4SetWriteTimeout - The function sets the timeout used for writes to the
*	serial channel.
* This timeout value is given in clock ticks. Any access to the
* driver write functions will be terminated if the amount of characters to be
* written could not be written to the FIFO within the timeout value.
*
* NOTE: A timeout value of "0" means "don't wait" and -1 means "wait for ever"
*
* RETURNS : 0 = OK, <0 = Error
*/

int SIO4SetWriteTimeout(chan_data_t* chan, void* arg)
{
	s32	s32Arg	= (s32) (unsigned long) arg;

	// if the input is any negative number make it -1 to wait forever
	if ((s32Arg < 0) && (s32Arg != -1))
		s32Arg = -1;

	chan->tx.timeout = s32Arg;
	return(0);
}



/******************************************************************************
* SIO4SetTransmitHDLCProtocol - The function sets certain transmit parameters
*	for the HDLC protocol
* The parameters are:
* Shared Zero Flags
* Preamble Enable
* TXUnderrun
*
* RETURNS : 0 = OK, <0 = Error
*/
int SIO4SetTransmitHDLCProtocol(chan_data_t* chan, void* pArg)
{
	dev_data_t*			dev			= chan->dev;
	u8*					pu8Addr;
	u8					u8RegVal;
	XMT_HDLC_PROTOCOL*	pstrTxHDLC	= (XMT_HDLC_PROTOCOL *) pArg;
	s32					fw_type;

	fw_type_config_get(chan, &fw_type);

	if (fw_type != SIO4_FW_TYPE_CONFIG_Z16C30)
		return(0);

	// check parameters
	if ((pstrTxHDLC == ((XMT_HDLC_PROTOCOL *) NULL))	||
		(pstrTxHDLC->eTxUnderrun < ABORT_COND)			||
		(pstrTxHDLC->eTxUnderrun > CRC_FLAG_COND))
	{
		return(-EINVAL);
	}
	else
	{
		pu8Addr = (u8*) chan->dev->vaddr.gsc_regs;
		u8RegVal	= 0;
		u8RegVal |= HDLC_MODE;
		u8RegVal |= (pstrTxHDLC->u8SharedZeroFlags & 0x1) <<
					SHARED_ZERO_FLAGS_SHIFT;
		u8RegVal |= (pstrTxHDLC->u8TxPreambleEnable & 0x1) <<
					TX_PREAMBLE_ENABLE_SHIFT;
		u8RegVal |= pstrTxHDLC->eTxUnderrun << TX_UNDERRUN_SHIFT;

		os_reg_mem_tx_u8(dev, (VADDR_T) (pu8Addr + ZILOG_CHAN_MODE_HIGH_REG(chan)), u8RegVal);

		// u8RegVal = os_reg_mem_rx_u8(dev, (VADDR_T) (pu8Addr + ZILOG_CHAN_MODE_HIGH_REG(chan)));

		return(0);
	}
}

/******************************************************************************
* SIO4SetReceiveHDLCProtocol - The function sets certain receive parameters
*	for the HDLC protocol.
*
* The parameters are:
* 16 Bit Control Enabled
* Logical Control Enabled
* Address Search Mode
*
* RETURNS : 0 = OK, <0 = Error
*/
int SIO4SetReceiveHDLCProtocol(chan_data_t* chan, void* pArg)
{
	dev_data_t*			dev			= chan->dev;
	u8*					pu8Addr;
	RCV_HDLC_PROTOCOL*	pstrRxHDLC = (RCV_HDLC_PROTOCOL *) pArg;
	u8					u8RegVal;
	s32					fw_type;

	fw_type_config_get(chan, &fw_type);

	if (fw_type != SIO4_FW_TYPE_CONFIG_Z16C30)
		return(0);

	// check parameters
	if ((pstrRxHDLC == ((RCV_HDLC_PROTOCOL *) NULL))	||
		(pstrRxHDLC->eAddrSearchMode < DISABLED)		||
		(pstrRxHDLC->eAddrSearchMode > EXT_PLUS_CTRL))
	{
		return(-EINVAL);
	}
	else
	{
		pu8Addr = (u8*) chan->dev->vaddr.gsc_regs;
		u8RegVal	= 0;
		u8RegVal |= HDLC_MODE;
		u8RegVal |= (pstrRxHDLC->u816BitControlEnable & 0x1) <<
					B16_CTRL_ENABLE_SHIFT;
		u8RegVal |= (pstrRxHDLC->u8LogicalControlEnable & 0x1) <<
					LOGICAL_CTRL_ENABLE_SHIFT;
		u8RegVal |= pstrRxHDLC->eAddrSearchMode << ADDR_SEARCH_SHIFT;

		os_reg_mem_tx_u8(dev, (VADDR_T) (pu8Addr + ZILOG_CHAN_MODE_LOW_REG(chan)), u8RegVal);
		return(0);
	}
}

/******************************************************************************
* SIO4SetTransmitHDLCSDLCloopProtocol - The function sets certain receive
*	parameters for the HDLC SDLC Loop protocol.
*
* The parameters are:
* Shared Zero Flags
* Preamble Enable
* TXUnderrun
*
* RETURNS : 0 = OK, <0 = Error
*/

int SIO4SetTransmitHDLCSDLCLoopProtocol(chan_data_t* chan, void* pArg)
{
	dev_data_t*						dev			= chan->dev;
	u8*								pu8Addr;
	XMT_HDLC_SDLC_LOOP_PROTOCOL*	pHDLCLoop	= (XMT_HDLC_SDLC_LOOP_PROTOCOL *) pArg;
	u8								u8RegVal;
	s32								fw_type;

	fw_type_config_get(chan, &fw_type);

	if (fw_type != SIO4_FW_TYPE_CONFIG_Z16C30)
		return(0);

	// check parameters
	if ((pHDLCLoop == ((XMT_HDLC_SDLC_LOOP_PROTOCOL *) NULL))	||
		(pHDLCLoop->eTxUnderrun < ABORT_COND)					||
		(pHDLCLoop->eTxUnderrun > CRC_FLAG_COND))
	{
		return(-EINVAL);
	}
	else
	{
		pu8Addr = (u8*) chan->dev->vaddr.gsc_regs;
		u8RegVal	= 0;
		u8RegVal |= HDLC_LOOP_MODE;
		u8RegVal |= (pHDLCLoop->u8SharedZeroFlags & 0x1) <<
					SHARED_ZERO_FLAGS_SHIFT;
		u8RegVal |= (pHDLCLoop->u8TxActiveOnPoll & 0x1) <<
					TX_ACTIVE_ON_POLL_SHIFT;
		u8RegVal |= pHDLCLoop->eTxUnderrun << TX_UNDERRUN_SHIFT;

		os_reg_mem_tx_u8(dev, (VADDR_T) (pu8Addr + ZILOG_CHAN_MODE_HIGH_REG(chan)), u8RegVal);

		// u8RegVal = os_reg_mem_rx_u8(dev, (VADDR_T) (pu8Addr + ZILOG_CHAN_MODE_HIGH_REG(chan)));

		return(0);

	}
}

/******************************************************************************
* SIO4SetTransmitASYNCProtocol - The function sets certain transmit parameters
*	for the ASYNC protocol.
* The parameters are
* the clock rate and
* the number of stop bits.
*
* RETURNS : 0 = OK, <0 = Error
*/
int SIO4SetTransmitASYNCProtocol(chan_data_t* chan, void* pArg)
{
	dev_data_t*			dev			= chan->dev;
	u8*					pu8Addr;
	XMT_ASYNC_PROTOCOL*	pstrTxASYNC	= (XMT_ASYNC_PROTOCOL *) pArg;
	u8					u8RegVal;
	s32					fw_type;

	fw_type_config_get(chan, &fw_type);

	if (fw_type != SIO4_FW_TYPE_CONFIG_Z16C30)
		return(0);

	// check parameters
	if ((pstrTxASYNC == ((XMT_ASYNC_PROTOCOL *) NULL)) ||
		(pstrTxASYNC->eTxClockRate < RATE_X16) ||
		(pstrTxASYNC->eTxClockRate> LOCK_RATE) ||
		(pstrTxASYNC->eTxStopBits < ONE_STOP_BIT) ||
		(pstrTxASYNC->eTxStopBits > TWO_STOP_BITS_SHAVED))
	{
		return(-EINVAL);
	}
	else
	{
		pu8Addr = (u8*) chan->dev->vaddr.gsc_regs;

		u8RegVal	= 0;
		u8RegVal |= ASYNC_MODE;
		u8RegVal |= pstrTxASYNC->eTxClockRate << CLOCK_RATE_SHIFT;
		u8RegVal |= pstrTxASYNC->eTxStopBits << STOP_BITS_SHIFT;

		os_reg_mem_tx_u8(dev, (VADDR_T) (pu8Addr + ZILOG_CHAN_MODE_HIGH_REG(chan)), u8RegVal);

		// u8RegVal = os_reg_mem_rx_u8(dev, (VADDR_T) (pu8Addr + ZILOG_CHAN_MODE_HIGH_REG(chan)));

		return(0);
	}
}

/******************************************************************************
* SIO4SetReceiveASYNCProtocol - The function sets certain Receive parameters
*	for the ASYNC protocol.
*
* The parameter set is the clock rate.
*
* RETURNS : 0 = OK, <0 = Error
*/
int SIO4SetReceiveASYNCProtocol(chan_data_t* chan, void* pArg)
{
	dev_data_t*			dev			= chan->dev;
	u8*					pu8Addr;
	RCV_ASYNC_PROTOCOL*	pstrRxASYNC = (RCV_ASYNC_PROTOCOL *) pArg;
	u8					u8RegVal;
	s32					fw_type;

	fw_type_config_get(chan, &fw_type);

	if (fw_type != SIO4_FW_TYPE_CONFIG_Z16C30)
		return(0);

	// check parameters
	if ((pstrRxASYNC == ((RCV_ASYNC_PROTOCOL *) NULL))	||
		(pstrRxASYNC->eRxClockRate < RATE_X16)			||
		(pstrRxASYNC->eRxClockRate > LOCK_RATE))
	{
		return(-EINVAL);
	}
	else
	{
		pu8Addr = (u8*) chan->dev->vaddr.gsc_regs;

		u8RegVal	= 0;
		u8RegVal |= ASYNC_MODE;
		u8RegVal |= pstrRxASYNC->eRxClockRate << CLOCK_RATE_SHIFT;

		os_reg_mem_tx_u8(dev, (VADDR_T) (pu8Addr + ZILOG_CHAN_MODE_LOW_REG(chan)), u8RegVal);
		return(0);
	}
}

/******************************************************************************
* SIO4SetTransmitISOCHRProtocol - Sets the transmid parameters for
*	ISO CHR protocol.
*
* The function sets certain Transmit parameters for the ISOCHR protocol.
* Specifically, this function will set ISOCHRONOUS protocol for one or two
* stop bits. A FALSE value will set the protocol for one stop bit.
* A TRUE value will set the protocol for two stop bits.
*
* RETURNS : 0 = OK, <0 = Error
*/
int SIO4SetTransmitISOCHRProtocol(chan_data_t* chan, void* pArg)
{
	dev_data_t*				dev			= chan->dev;
	u8*						pu8Addr;
	u8						u8RegVal;
	XMT_ISOCHR_PROTOCOL*	pstrTxISOCHR = (XMT_ISOCHR_PROTOCOL *) pArg;
	s32						fw_type;

	fw_type_config_get(chan, &fw_type);

	if (fw_type != SIO4_FW_TYPE_CONFIG_Z16C30)
		return(0);

	// check parameters
	if (pstrTxISOCHR == ((XMT_ISOCHR_PROTOCOL *) NULL))
	{
		return(-EINVAL);
	}
	else
	{
		pu8Addr = (u8*) chan->dev->vaddr.gsc_regs;
		u8RegVal = ISOCHR_MODE |
				((pstrTxISOCHR->u8TwoStopBits & 0x1) << TWO_STOP_BITS_SHIFT);

		os_reg_mem_tx_u8(dev, (VADDR_T) (pu8Addr + ZILOG_CHAN_MODE_HIGH_REG(chan)), u8RegVal);
		return(0);
	}
}

/******************************************************************************
* SIO4SetReceiveISOCHRProtocol - The function sets certain Receive parameters
*	for the ISOCHR protocol.
*
* This function sets the corresponding bits of the Channel Mode Low register
* in the USC to setup receive parameters for ISO CHR protocol.
*
* RETURNS : 0 = OK, <0 = Error
*/
int SIO4SetReceiveISOCHRProtocol(chan_data_t* chan, void* pArg)
{
	dev_data_t*	dev		= chan->dev;
	u8*			pu8Addr;
	s32			fw_type;

	fw_type_config_get(chan, &fw_type);

	if (fw_type != SIO4_FW_TYPE_CONFIG_Z16C30)
		return(0);

	pu8Addr = (u8*) chan->dev->vaddr.gsc_regs;

	os_reg_mem_tx_u8(dev, (VADDR_T) (pu8Addr + ZILOG_CHAN_MODE_LOW_REG(chan)), ISOCHR_MODE);
	return(0);
}

/******************************************************************************
* SIO4SetTxClockSource - The function sets the transmit clock source.
*
* This function sets the Transmit clock source by writing into appropriate bits
* of the Clock mode control low register of the USC.
*
* RETURNS : 0 = OK, <0 = Error
*/
int SIO4SetTxClockSource(chan_data_t* chan, void* arg)
{
	dev_data_t*		dev				= chan->dev;
	u8*				pu8Addr;
	CLOCK_SOURCE	TxClockSource	= (CLOCK_SOURCE) arg;
	u8				u8RegVal;
	s32				fw_type;

	fw_type_config_get(chan, &fw_type);

	if (fw_type != SIO4_FW_TYPE_CONFIG_Z16C30)
		return(0);

	// make sure the parameters are OK
	if ((TxClockSource < CLOCK_DISABLED) ||
		(TxClockSource > CTR1_CLOCK))
	{
		return(-EINVAL);
	}
	else
	{
		pu8Addr = (u8*) chan->dev->vaddr.gsc_regs;

		u8RegVal = os_reg_mem_rx_u8(dev, (VADDR_T) (pu8Addr + ZILOG_CLOCK_MODE_CTRL_LOW_REG(chan)));

		u8RegVal &= ~TX_SOURCE_MASK;
		TxClockSource	= (CLOCK_SOURCE) ((u16) TxClockSource << TX_SOURCE_SHIFT);
		u8RegVal |= TxClockSource;

		os_reg_mem_tx_u8(dev, (VADDR_T) (pu8Addr+ ZILOG_CLOCK_MODE_CTRL_LOW_REG(chan)), u8RegVal);
		return(0);
	}
}

/******************************************************************************
* SIO4SetRxClockSource - The function sets the receive clock source.
*
* This function sets the receive clock source by writing into the appropritate
* bits of the clock mode control low register of the USC.
*
* RETURNS : 0 = OK, <0 = Error
*/
int SIO4SetRxClockSource(chan_data_t* chan, void* arg)
{
	dev_data_t*		dev				= chan->dev;
	u8*				pu8Addr;
	CLOCK_SOURCE	RxClockSource	= (CLOCK_SOURCE) arg;
	u8				u8RegVal;
	s32				fw_type;

	fw_type_config_get(chan, &fw_type);

	if (fw_type != SIO4_FW_TYPE_CONFIG_Z16C30)
		return(0);

	// make sure the parameters are OK
	if ((RxClockSource < CLOCK_DISABLED) ||
		(RxClockSource > CTR1_CLOCK))
	{
		return(-EINVAL);
	}
	else
	{
		pu8Addr = (u8*) chan->dev->vaddr.gsc_regs;

		u8RegVal = os_reg_mem_rx_u8(dev, (VADDR_T) (pu8Addr + ZILOG_CLOCK_MODE_CTRL_LOW_REG(chan)));
		u8RegVal &= ~RX_SOURCE_MASK;
		RxClockSource	= (CLOCK_SOURCE) ((unsigned) RxClockSource << RX_SOURCE_SHIFT);
		u8RegVal |= RxClockSource;

		os_reg_mem_tx_u8(dev, (VADDR_T) (pu8Addr + ZILOG_CLOCK_MODE_CTRL_LOW_REG(chan)), u8RegVal);
		return(0);
	}
}

/******************************************************************************
* SIO4SetBRG0Source - The function sets the source for Baud rate Generator 0.
*
* This function sets the source for the first baud rate generator by writing
* into the appropriate bits of the Clock mode control low register of the
* USC. The source is taken from the global array au8BRGSources
*
* RETURNS : 0 = OK, <0 = Error
*/
int SIO4SetBRG0Source(chan_data_t* chan, void* arg)
{
	dev_data_t*		dev				= chan->dev;
	u8*				pu8Addr;
	u8				u8RegVal;
	u8				u8TmpVal;
	CLOCK_SOURCE	BRG0ClockSource	= (CLOCK_SOURCE) arg;
	s32				fw_type;

	fw_type_config_get(chan, &fw_type);

	if (fw_type != SIO4_FW_TYPE_CONFIG_Z16C30)
		return(0);

	// make sure the parameters are OK
	if ((BRG0ClockSource != RXC_PIN_CLOCK) &&
		(BRG0ClockSource != TXC_PIN_CLOCK) &&
		(BRG0ClockSource != CTR0_CLOCK)	&&
		(BRG0ClockSource != CTR1_CLOCK))
	{
		return(-EINVAL);
	}
	else
	{
		pu8Addr = (u8*) chan->dev->vaddr.gsc_regs;

		u8RegVal = os_reg_mem_rx_u8(dev, (VADDR_T) (pu8Addr + ZILOG_CLOCK_MODE_CTRL_HI_REG(chan)));

		u8RegVal &= ~BRG0_SOURCE_MASK;
		u8TmpVal = au8BRGSources [BRG0ClockSource];
		u8TmpVal <<= BRG0_SOURCE_SHIFT;
		u8RegVal |= u8TmpVal;
		os_reg_mem_tx_u8(dev, (VADDR_T) (pu8Addr + ZILOG_CLOCK_MODE_CTRL_HI_REG(chan)), u8RegVal);
		return(0);
	}
}

/******************************************************************************
* SIO4SetBRG1Source - The function sets the source for Baud rate Generator 1.
*
* This function sets the source for the second baud rate generator by writing
* into the appropriate bits of the Clock mode control low register of the
* USC. The source is taken from the global array au8BRGSources
*
* RETURNS : 0 = OK, <0 = Error
*/

int SIO4SetBRG1Source(chan_data_t* chan, void* arg)
{
	dev_data_t*		dev				= chan->dev;
	u8*				pu8Addr;
	CLOCK_SOURCE	BRG1ClockSource	= (CLOCK_SOURCE) arg;
	u8				u8RegVal;
	u8				u8TmpVal;
	s32				fw_type;

	fw_type_config_get(chan, &fw_type);

	if (fw_type != SIO4_FW_TYPE_CONFIG_Z16C30)
		return(0);

	// make sure the parameters are OK
	if ((BRG1ClockSource != RXC_PIN_CLOCK) &&
		(BRG1ClockSource != TXC_PIN_CLOCK) &&
		(BRG1ClockSource != CTR0_CLOCK)	&&
		(BRG1ClockSource != CTR1_CLOCK))
	{
		return(-EINVAL);
	}
	else
	{
		pu8Addr = (u8*) chan->dev->vaddr.gsc_regs;

		u8RegVal = os_reg_mem_rx_u8(dev, (VADDR_T) (pu8Addr + ZILOG_CLOCK_MODE_CTRL_HI_REG(chan)));

		u8RegVal &= ~BRG1_SOURCE_MASK;
		u8TmpVal = au8BRGSources [BRG1ClockSource];
		u8TmpVal <<= BRG1_SOURCE_SHIFT;
		u8RegVal |= u8TmpVal;

		os_reg_mem_tx_u8(dev, (VADDR_T) (pu8Addr + ZILOG_CLOCK_MODE_CTRL_HI_REG(chan)), u8RegVal);
		return(0);
	}
}

/******************************************************************************
* SIO4SetCTR0Source - The function sets the clock source for Counter 0
*
* This function sets the source for the first counter by writing into
* appropriate locations of the Clock mode control high register of the USC.
* The source is taken from the global array au8CTRSources.
*
* RETURNS : 0 = OK, <0 = Error
*/
int SIO4SetCTR0Source(chan_data_t* chan, void* arg)
{
	dev_data_t*		dev				= chan->dev;
	u8*				pu8Addr;
	CLOCK_SOURCE	CTR0ClockSource	= (CLOCK_SOURCE) arg;
	u8				u8RegVal;
	u8				u8TmpVal;
	s32				fw_type;

	fw_type_config_get(chan, &fw_type);

	if (fw_type != SIO4_FW_TYPE_CONFIG_Z16C30)
		return(0);

	// make sure the parameters are OK
	if ((CTR0ClockSource != CLOCK_DISABLED) &&
		(CTR0ClockSource != RXC_PIN_CLOCK)	&&
		(CTR0ClockSource != TXC_PIN_CLOCK))
	{
		return(-EINVAL);
	}
	else
	{
		pu8Addr = (u8*) chan->dev->vaddr.gsc_regs;

		u8RegVal = os_reg_mem_rx_u8(dev, (VADDR_T) (pu8Addr + ZILOG_CLOCK_MODE_CTRL_HI_REG(chan)));

		u8RegVal &= (~CTR0_SOURCE_MASK);
		u8TmpVal = au8CTRSources [CTR0ClockSource];
		u8TmpVal <<= CTR0_SOURCE_SHIFT;
		u8RegVal |= u8TmpVal;

		os_reg_mem_tx_u8(dev, (VADDR_T) (pu8Addr + ZILOG_CLOCK_MODE_CTRL_HI_REG(chan)), u8RegVal);
		return(0);
	}
}

/******************************************************************************
* SIO4SetCTR1Source - The function sets the clock source for Counter 1.
*
* This function sets the source for the second counter by writing into the
* appropriate bits of the Clock mode control high register of the USC.
* The source is taken from the global array au8CTRSources.
*
* RETURNS : 0 = OK, <0 = Error
*/
int SIO4SetCTR1Source(chan_data_t* chan, void* arg)
{
	dev_data_t*		dev				= chan->dev;
	u8*				pu8Addr;
	CLOCK_SOURCE	CTR1ClockSource	= (CLOCK_SOURCE) arg;
	u8				u8RegVal;
	u8				u8TmpVal;
	s32				fw_type;

	fw_type_config_get(chan, &fw_type);

	if (fw_type != SIO4_FW_TYPE_CONFIG_Z16C30)
		return(0);

	// make sure the parameters are OK
	if ((CTR1ClockSource != CLOCK_DISABLED) &&
		(CTR1ClockSource != RXC_PIN_CLOCK)	&&
		(CTR1ClockSource != TXC_PIN_CLOCK))
	{
		return(-EINVAL);
	}
	else
	{
		pu8Addr = (u8*) chan->dev->vaddr.gsc_regs;
		u8RegVal = os_reg_mem_rx_u8(dev, (VADDR_T) (pu8Addr + ZILOG_CLOCK_MODE_CTRL_HI_REG(chan)));

		u8RegVal &= ~CTR1_SOURCE_MASK;
		u8TmpVal = au8CTRSources [CTR1ClockSource];
		u8TmpVal <<= CTR1_SOURCE_SHIFT;
		u8RegVal |= u8TmpVal;

		os_reg_mem_tx_u8(dev, (VADDR_T) (pu8Addr + ZILOG_CLOCK_MODE_CTRL_HI_REG(chan)), u8RegVal);
		return(0);
	}
}

/******************************************************************************
* SIO4setDPLLSource - This function sets the source for the Digital
*	Phase Locked Loop Counter.
*
* This function is used to set the clock source for the Digital Phased
* Lock Loop Counter in the
* Zilog USC. The Z16C30 chip has a register named Clock Mode Control register.
* The bits 6,7 of this 16-bit register are filled up with the source for the
* DPLL counter. The source is taken from the global array au8DPLLSources
*
* RETURNS : 0 = OK, <0 = Error
*/
int SIO4SetDPLLSource(chan_data_t* chan, void* arg)
{
	dev_data_t*		dev				= chan->dev;
	CLOCK_SOURCE	DPLLClockSource	= (CLOCK_SOURCE) arg;
	u8				u8Reg;
	u8				u8TmpVal;
	u8*				pu8Addr;
	s32				fw_type;

	fw_type_config_get(chan, &fw_type);

	if (fw_type != SIO4_FW_TYPE_CONFIG_Z16C30)
		return(0);

	// make sure the parameters are OK
	if ((DPLLClockSource != RXC_PIN_CLOCK) &&
		(DPLLClockSource != TXC_PIN_CLOCK) &&
		(DPLLClockSource != BRG0_CLOCK)	&&
		(DPLLClockSource != BRG1_CLOCK))
	{
		return(-EINVAL);
	}
	else
	{
		// write into clock mode register low in bits 6,7
		pu8Addr = (u8*) chan->dev->vaddr.gsc_regs;

		u8Reg = os_reg_mem_rx_u8(dev, (VADDR_T) (pu8Addr + ZILOG_CLOCK_MODE_CTRL_LOW_REG(chan)));

		u8Reg &= ~DPLL_SOURCE_MASK;
		u8TmpVal = au8DPLLSources [DPLLClockSource];
		u8TmpVal <<= DPLL_SOURCE_SHIFT;
		u8Reg |= u8TmpVal;

		os_reg_mem_tx_u8(dev, (VADDR_T) (pu8Addr + ZILOG_CLOCK_MODE_CTRL_LOW_REG(chan)), u8Reg);
		return(0);
	}
}

/******************************************************************************
* SIO4SetBRG0Mode - Set the mode for first baud rate generator
*
* This function is used to set the mode for the first baud rate generator in
* the Zilog USC. The Z16C30 has a register named Hardware Configuration
* Register. The bit 1 of this 16-bit register signifies whether the operation
* for this baud rate generator is continuous or not. If this bit is 1,
* the operation of the baud rate generator stops once the count reaches zero.
* If this bit is 0, the operation is continuous, the initial value used for
* counting is reloaded and the counting process is continued. The user
* passes the mode, which is set by this function
*
* RETURNS : 0 = OK, <0 = Error
*/
int SIO4SetBRG0Mode(chan_data_t* chan, void* arg)
{
	dev_data_t*	dev		= chan->dev;
	BRG_MODE	BrgMode	= (BRG_MODE) arg;
	u8			u8Reg;
	u8*			pu8Addr;
	s32			fw_type;

	fw_type_config_get(chan, &fw_type);

	if (fw_type != SIO4_FW_TYPE_CONFIG_Z16C30)
		return(0);

	// Make sure the parameters are valid.
	if ((BrgMode != BRG_SINGLE_CYCLE) &&
		(BrgMode != BRG_CONTINUOUS))
	{
		return(-EINVAL);
	}
	else
	{
		pu8Addr = (u8*) chan->dev->vaddr.gsc_regs;

		u8Reg = os_reg_mem_rx_u8(dev, (VADDR_T) (pu8Addr + ZILOG_HWC_LOW_REG(chan)));

		u8Reg = (u8Reg & ~SIO4_USC_HCR_BRG0S) | (BrgMode << 1);

		os_reg_mem_tx_u8(dev, (VADDR_T) (pu8Addr + ZILOG_HWC_LOW_REG(chan)), u8Reg);
		return(0);
	}
}

/******************************************************************************
* SIO4SetBRG1Mode - Set the mode for first baud rate generator
*
* This function is used to set the mode for the first baud rate generator in
* the Zilog USC. The Z16C30 has a register named Hardware Configuration
* Register. The bit 2 of this 16-bit register signifies whether the operation
* for this baud rate generator is continuous or not. If this bit is 1,
* the operation of the baud rate generator stops once the count reaches zero.
* If this bit is 0, the operation is continuous, the initial value used for
* counting is reloaded and the counting process is continued. The user
* passes the mode, which is set by this function
*
* RETURNS : 0 = OK, <0 = Error
*/
int SIO4SetBRG1Mode(chan_data_t* chan, void* arg)
{
	dev_data_t*	dev		= chan->dev;
	BRG_MODE	BrgMode	= (BRG_MODE) arg;
	u8			u8Reg;
	u8*			pu8Addr;
	s32			fw_type;

	fw_type_config_get(chan, &fw_type);

	if (fw_type != SIO4_FW_TYPE_CONFIG_Z16C30)
		return(0);

	// Make sure the parameters are valid.
	if ((BrgMode != BRG_SINGLE_CYCLE) &&
		(BrgMode != BRG_CONTINUOUS))
	{
		return(-EINVAL);
	}
	else
	{
		pu8Addr = (u8*) chan->dev->vaddr.gsc_regs;

		u8Reg = os_reg_mem_rx_u8(dev, (VADDR_T) (pu8Addr + ZILOG_HWC_LOW_REG(chan)));

		u8Reg = (u8Reg & ~SIO4_USC_HCR_BRG1S) | (BrgMode << 5);
		os_reg_mem_tx_u8(dev, (VADDR_T) (pu8Addr + ZILOG_HWC_LOW_REG(chan)), u8Reg);
		return(0);
	}
}

/******************************************************************************
* SIO4EnableBRG0 - enable first baud rate generator
*
* This function is used to enable the first baud rate generator in the Zilog
* USC. The Z16C30 has a register named Hardware Configuration Register. The bit
* 0 of this 16-bit register signifies whether this baud rate generator is
* enabled or not. The user passes the value either to enable or disable the
* first baud rate generator and this function sets that value into the
* appropriate location in the register.
*
* RETURNS: 0 always.
*/
int SIO4EnableBRG0(chan_data_t* chan, void* arg)
{
	dev_data_t*	dev			= chan->dev;
	u8			u8BRGEnable = (u8) (unsigned long) arg;
	u8			u8Reg;
	u8*			pu8Addr;
	s32			fw_type;

	fw_type_config_get(chan, &fw_type);

	if (fw_type != SIO4_FW_TYPE_CONFIG_Z16C30)
		return(0);

	pu8Addr = (u8*) chan->dev->vaddr.gsc_regs;

	u8Reg	= os_reg_mem_rx_u8(dev, (VADDR_T) (pu8Addr + ZILOG_HWC_LOW_REG(chan)));

	u8Reg	= (u8Reg & ~SIO4_USC_HCR_BRG0E) | (u8BRGEnable << 0);

	os_reg_mem_tx_u8(dev, (VADDR_T) (pu8Addr + ZILOG_HWC_LOW_REG(chan)), u8Reg);
	return(0);
}

/******************************************************************************
* SIO4EnableBRG1 - enable second baud rate generator
*
* This function is used to enable the first baud rate generator in the Zilog
* USC. The Z16C30 has a register named Hardware Configuration Register. The
* bit 4 of this 16-bit register signifies whether this baud rate generator
* is not. The user passes the value either to enable or disable the first
* baud rate generator and this function sets that value into the
* appropriate location in the register.
*
* RETURNS: 0 always.
*/
int SIO4EnableBRG1(chan_data_t* chan, void* arg)
{
	dev_data_t*	dev			= chan->dev;
	u8			u8BRGEnable = (u8) (unsigned long) arg;
	u8			u8Reg;
	u8*			pu8Addr;
	s32			fw_type;

	fw_type_config_get(chan, &fw_type);

	if (fw_type != SIO4_FW_TYPE_CONFIG_Z16C30)
		return(0);

	pu8Addr = (u8*) chan->dev->vaddr.gsc_regs;

	u8Reg	= os_reg_mem_rx_u8(dev, (VADDR_T) (pu8Addr + ZILOG_HWC_LOW_REG(chan)));

	u8Reg	= (u8) (u8Reg & ~SIO4_USC_HCR_BRG1E) | (u8) (u8BRGEnable << 4);

	os_reg_mem_tx_u8(dev, (VADDR_T) (pu8Addr + ZILOG_HWC_LOW_REG(chan)), u8Reg);
	return(0);
}

/******************************************************************************
* SIO4SelectDPLLResync - selects what DPLL uses to resync
*
* This function is used to select when the DPLL should be
* resynchronized. The Z16C30 has a register named Channel
* command/status register. The bits 8,9 signify when the DPLL should be
* resynchronized. If both the bits are zero resynchronization occurs at
* both raising and falling edges. If the value here is 1,
* resynchronization occurs only at rising edges and for the value 2,
* it's only at falling edges. If the value is 3 it means
* resynchronization is inhibited. The user passes the value and this
* function sets the valued into the register.
*
* RETURNS : 0 = OK, <0 = Error
*/
int SIO4SelectDPLLResync(chan_data_t* chan, void* arg)
{
	dev_data_t*	dev			= chan->dev;
	DPLL_RESYNC	DpllResync	= (DPLL_RESYNC) arg;
	u8			u8Reg;
	u8*			pu8Addr;
	s32			fw_type;

	fw_type_config_get(chan, &fw_type);

	if (fw_type != SIO4_FW_TYPE_CONFIG_Z16C30)
		return(0);

	// Validate the parameters.
	if ((DpllResync < BOTH_EDGES) || (DpllResync > SYNC_INHIBIT))
	{
		return(-EINVAL);
	}
	else
	{
		pu8Addr = (u8*) chan->dev->vaddr.gsc_regs;
		u8Reg = os_reg_mem_rx_u8(dev, (VADDR_T) (pu8Addr + ZILOG_CMD_STATUS_HIGH_REG(chan)));

		u8Reg = (u8Reg & ~DPLL_RESYNC_MASK) |
				(DpllResync << DPLL_RESYNC_SHIFT);

		os_reg_mem_tx_u8(dev, (VADDR_T) (pu8Addr + ZILOG_CMD_STATUS_HIGH_REG(chan)), u8Reg);
		return(0);
	}
}

/******************************************************************************
* SIO4SetDPLLMode - selects what DPLL uses to resync
*
* This function is used to select the mode in which the DPLL of the
* Zilog should operate. The Z16C30 has a register named Hardware
* Configuration register. The bits 8,9 of this register signify the
* mode in which the DPLL operates. If the value in these bits is 0 the
* DPLL is disabled. If 1 the DPLL operates in any Non return to Zero
* (NRZ) mode. If it is 2, it operates in Biphase-Mark mode. If it is 3,
* it operates in either - biphase level mode. The user passes one of
* these values as argument, and this function puts the value into
* appropriate position in the Hardware Configuration register.
*
* RETURNS : 0 = OK, <0 = Error
*/
int SIO4SetDPLLMode(chan_data_t* chan, void* arg)
{
	dev_data_t*	dev			= chan->dev;
	DPLL_MODE	DpllMode	= (DPLL_MODE) arg;
	u8			u8Reg;
	u8*			pu8Addr;
	s32			fw_type;

	fw_type_config_get(chan, &fw_type);

	if (fw_type != SIO4_FW_TYPE_CONFIG_Z16C30)
		return(0);

	// Validate the parameters.
	if ((DpllMode < DPLL_DISABLED) || (DpllMode > DPLL_BIPHASE_LEVEL))
	{
		return(-EINVAL);
	}
	else
	{
		pu8Addr = (u8*) chan->dev->vaddr.gsc_regs;
		u8Reg = os_reg_mem_rx_u8(dev, (VADDR_T) (pu8Addr + ZILOG_HWC_HIGH_REG(chan)));

		u8Reg = (u8Reg & ~(SIO4_USC_HCR_DPLLM >> 8)) | (DpllMode << 0);

		os_reg_mem_tx_u8(dev, (VADDR_T) (pu8Addr + ZILOG_HWC_HIGH_REG(chan)), u8Reg);
		return(0);
	}
}

/******************************************************************************
* SIO4SetDPLLDivisor - selects what DPLL uses to resync
*
* This function is used to select the divisor by which the DPLL divides
* its reference clock in the Zilog USC. The Z16C30 has a register named
* Hardware Configuration register. The bits 10,11 of this register
* signify the divisor value. If the value in these bits is 0, the
* divisor value is 32. If 1, the divisor is 16 and if 2 the divisor is
* 8. The value 3 is reserverd and cannot be used for DPLL operation.
* The user passes one of these values, which this function sets in the
* bit positions 10,11 of the Hardware Configuration register.
*
* RETURNS : 0 = OK, <0 = Error
*/
int SIO4SetDPLLDivisor(chan_data_t* chan, void* arg)
{
	dev_data_t*		dev			= chan->dev;
	DPLL_DIVISOR	DpllDivisor	= (DPLL_DIVISOR) arg;
	u8				u8Reg;
	u8*				pu8Addr;
	s32				fw_type;

	fw_type_config_get(chan, &fw_type);

	if (fw_type != SIO4_FW_TYPE_CONFIG_Z16C30)
		return(0);

	// Validate the parameters.
	if ((DpllDivisor < DPLL_32X) || (DpllDivisor > DPLL_8X))
	{
		return(-EINVAL);
	}
	else
	{
		pu8Addr = (u8*) chan->dev->vaddr.gsc_regs;
		u8Reg = os_reg_mem_rx_u8(dev, (VADDR_T) (pu8Addr + ZILOG_HWC_HIGH_REG(chan)));

		u8Reg	= (u8Reg & ~(SIO4_USC_HCR_DPLLD >> 8))
			| (DpllDivisor << 2);

		os_reg_mem_tx_u8(dev, (VADDR_T) (pu8Addr + ZILOG_HWC_HIGH_REG(chan)), u8Reg);
		return(0);
	}
}

/******************************************************************************
* SIO4ClearDPLLStatus - selects what DPLL uses to resync
*
* This function is used to clear the DPLL status in the Zilog USC. The
* Z16C30 has a register named Channel Command/Status Register. The bits
* 8,9,10,11,12 of this register signify the status of the DPLL. The
* user can choose to enable or disable one of these bits. The user
* passes the value and this function just puts that value into the bit
* positions 8*to 12 of the Channel command/status register.
*
* RETURNS : 0 = OK, <0 = Error
*/
int SIO4ClearDPLLStatus(chan_data_t* chan, void* arg)
{
	dev_data_t*	dev		= chan->dev;
	u32			u32Arg	= (u32) (unsigned long) arg;
	u8			u8Reg;
	u8*			pu8Addr;
	int			ret;
	s32			fw_type;

	fw_type_config_get(chan, &fw_type);

	if (fw_type != SIO4_FW_TYPE_CONFIG_Z16C30)
		return(0);

	if ((u32Arg & CLEAR_DPLL_ALL_STATUS) != u32Arg)
	{
		ret	= -EINVAL;
	}
	else
	{
		pu8Addr	= (u8*) chan->dev->vaddr.gsc_regs;
		u8Reg	= os_reg_mem_rx_u8(dev, (VADDR_T) (pu8Addr + ZILOG_CMD_STATUS_HIGH_REG(chan)));
		u8Reg	= (u8) (u8Reg & ~CLEAR_DPLL_ALL_STATUS) | (u8) u32Arg;

		os_reg_mem_tx_u8(dev, (VADDR_T) (pu8Addr + ZILOG_CMD_STATUS_HIGH_REG(chan)), u8Reg);
		ret	= 0;
	}

	return(ret);
}

/******************************************************************************
* SIO4SetUSCDMAOptions - sets the options for DMA transfer
*
* This function sets the DMA options on the USC register. The options
* are sent by the user
*
* RETURNS : 0 = OK, <0 = Error
*/
int SIO4SetUSCDMAOptions(chan_data_t* chan, void *pArg)
{
	dev_data_t*			dev				= chan->dev;
	u8					u8Reg;
	u8*					pu8Addr;
	USC_DMA_OPTIONS*	pstrDMAOptions	= (USC_DMA_OPTIONS *) pArg;
	s32					fw_type;

	fw_type_config_get(chan, &fw_type);

	if (fw_type != SIO4_FW_TYPE_CONFIG_Z16C30)
		return(0);

	// Validate the parameters.
	if ((pstrDMAOptions == ((USC_DMA_OPTIONS *) NULL))					||
		(pstrDMAOptions->eTxStatusBlockOptions < NO_STATUS_BLOCK)		||
		(pstrDMAOptions->eTxStatusBlockOptions > TWO_WORD_STATUS_BLOCK)	||
		(pstrDMAOptions->eRxStatusBlockOptions < NO_STATUS_BLOCK)		||
		(pstrDMAOptions->eRxStatusBlockOptions > TWO_WORD_STATUS_BLOCK))
	{
		return(-EINVAL);
	}
	else
	{
		pu8Addr = (u8*) chan->dev->vaddr.gsc_regs;

		u8Reg = os_reg_mem_rx_u8(dev, (VADDR_T) (pu8Addr + ZILOG_CHAN_CTRL_LOW_REG(chan)));
		u8Reg &= ~RX_STATUS_BLOCK_OPTIONS_MASK;
		u8Reg &= ~WAIT_FOR_RX_DMA_TRIGGER_MASK;

		u8Reg |= (pstrDMAOptions->eRxStatusBlockOptions <<
			RX_STATUS_BLOCK_OPTIONS_SHIFT) ;

		u8Reg |= (u8) (pstrDMAOptions->u8RxDMAWaitForTrigger <<
			WAIT_FOR_RX_DMA_TRIGGER_SHIFT);

		os_reg_mem_tx_u8(dev, (VADDR_T) (pu8Addr + ZILOG_CHAN_CTRL_LOW_REG(chan)), u8Reg);

		u8Reg = os_reg_mem_rx_u8(dev, (VADDR_T) (pu8Addr + ZILOG_CHAN_CTRL_HIGH_REG(chan)));

		u8Reg &= ~TX_STATUS_BLOCK_OPTIONS_MASK;
		u8Reg &= ~WAIT_FOR_TX_DMA_TRIGGER_MASK;

		u8Reg |= (pstrDMAOptions->eTxStatusBlockOptions <<
			TX_STATUS_BLOCK_OPTIONS_SHIFT) ;

		u8Reg |= (u8) (pstrDMAOptions->u8TxDMAWaitForTrigger <<
			WAIT_FOR_TX_DMA_TRIGGER_SHIFT);

		os_reg_mem_tx_u8(dev, (VADDR_T) (pu8Addr + ZILOG_CHAN_CTRL_HIGH_REG(chan)), u8Reg);

		// write 0x1 into TxRmode, RxRmode bits of
		// IO Control register of the USC

		u8Reg = os_reg_mem_rx_u8(dev, (VADDR_T) (pu8Addr + ZILOG_IO_CTRL_HIGH_REG(chan)));
		u8Reg |= 0x5;
		os_reg_mem_tx_u8(dev, (VADDR_T) (pu8Addr + ZILOG_IO_CTRL_HIGH_REG(chan)), u8Reg);

		// write 0x1 into TxAmode, RxAmode bits of
		// Hardwar Configuration register of the USC

		u8Reg = os_reg_mem_rx_u8(dev, (VADDR_T) (pu8Addr + ZILOG_HWC_LOW_REG(chan)));
		u8Reg |= 0x44;
		os_reg_mem_tx_u8(dev, (VADDR_T) (pu8Addr + ZILOG_HWC_LOW_REG(chan)), u8Reg);

		return(0);
	}

}	// SIO4SetUSCDMAOptions



//*****************************************************************************
int fw_type_config_ioctl(chan_data_t* chan, void* arg)
{
	dev_data_t*	dev		= chan->dev;
	u32			mask;
	int			ret		= 0;
	int			shift	= chan->index * 8;
	u32			val;
	s32*		value	= arg;

	if (value[0] == SIO4_FW_TYPE_CONFIG_READ)
	{
		if ((dev->cache.reg_ftr == 0) ||
			(dev->cache.fw_type_config == 0))
		{
			value[0]	= dev->cache.fw_type;
		}
		else
		{
			value[0]	= os_reg_mem_rx_u32(dev, dev->vaddr.gsc_ftr_32);
			value[0]	= (value[0] >> shift) & 0xFF;
		}
	}
	else if (	(value[0] == SIO4_FW_TYPE_CONFIG_SYNC) ||
				(value[0] == SIO4_FW_TYPE_CONFIG_Z16C30))
	{
		if (dev->cache.fw_type_config)
		{
			mask	= 0xFF << shift;
			val		= value[0] << shift;
			reg_mem32_mod_serial(dev, dev->vaddr.gsc_ftr_32, val, mask);

			if (value[0] == SIO4_FW_TYPE_CONFIG_Z16C30)
			{
				// Reset and initialize the Zilog chip.
				os_reg_mem_tx_u32(dev, chan->vaddr.gsc_csr_32, SIO4_GSC_CSR_ZIL_RST);
				os_reg_mem_tx_u8(dev, chan->vaddr.usc_bcr, 0);
			}

			// Update the features based on the new selection.
			channel_features_init(chan);
		}

		// Report the current setting.
		value[0]	= -1;
		ret			= fw_type_config_ioctl(chan, arg);
	}
	else
	{
		ret	= -EINVAL;
	}

	return(ret);
}



//*****************************************************************************
int fw_type_config_get(chan_data_t* chan, s32* get)
{
	int	ret;
	s32	set	= SIO4_FW_TYPE_CONFIG_READ;

	ret	= fw_type_config_ioctl(chan, &set);

	if (get)
		get[0]	= set;

	return(ret);
}


