// $URL: http://subversion:8080/svn/gsc/trunk/drivers/LINUX/SIO4%20and%20SIO8/SIO4_Linux_1.x.x_GSC_DN/driver/rx.c $
// $Rev: 48525 $
// $Date: 2020-11-23 18:29:37 -0600 (Mon, 23 Nov 2020) $

// SIO4: Device Driver: source file

#include "main.h"



/******************************************************************************
*
*	Function:	rx_cable_config_ioctl
*
*	Purpose:
*
*		Implement the Rx Cable Config IOCTL service.
*
*	Arguments:
*
*		chan	The structure for the chanel being accessed.
*
*		arg		The argument passed by the user.
*
*	Returned:
*
*		0		All went well.
*		< 0		An appropriate error status.
*
******************************************************************************/

s32 rx_cable_config_ioctl(chan_data_t* chan, void* arg)
{
	static const parm_2x32_t	list[]	=
	{
		{
		/* parm		*/	SIO4_RX_CABLE_CONFIG_DISABLE,
		/* mask1	*/	_GSC_CCR_RCCE,
		/* value1	*/	0,
		/* shift1	*/	4,
		/* mask2	*/	SIO4_GSC_CSR_RDULE,
		/* value2	*/	0,
		/* shift2	*/	0
		},
		{
		/* parm		*/	SIO4_RX_CABLE_CONFIG_LOWER,
		/* mask1	*/	_GSC_CCR_RCCE,
		/* value1	*/	_GSC_CCR_RCCE_L,
		/* shift1	*/	4,
		/* mask2	*/	SIO4_GSC_CSR_RDULE,
		/* value2	*/	SIO4_GSC_CSR_RDLE,
		/* shift2	*/	0
		},
		{
		/* parm		*/	SIO4_RX_CABLE_CONFIG_UPPER,
		/* mask1	*/	_GSC_CCR_RCCE,
		/* value1	*/	_GSC_CCR_RCCE_U,
		/* shift1	*/	4,
		/* mask2	*/	SIO4_GSC_CSR_RDULE,
		/* value2	*/	SIO4_GSC_CSR_RDUE,
		/* shift2	*/	0
		},
		{ 0, 0, 0, 0, 0, 0, 0 }
	};

	dev_data_t*	dev		= chan->dev;
	int			index;
	s32			ret;
	s32*		value	= arg;

	if (chan->dev->cache.legacy_cable == 0)
	{
		return(0);
	}

	if (value[0] == SIO4_RX_CABLE_CONFIG_READ)
	{
		index	= 0;
	}
	else
	{
		index	= parm_2x32_set(chan,
								dev->vaddr.gsc_ccr_32,
								chan->vaddr.gsc_csr_32,
								value[0],
								list);
	}

	if (index < 0)
		ret	= -EINVAL;
	else
		ret	= 0;

	index		= parm_2x32_get(chan,
								dev->vaddr.gsc_ccr_32,
								chan->vaddr.gsc_csr_32,
								list);

	if (index >= 0)
		value[0]	= list[index].parm;
	else
		value[0]	= SIO4_RX_CABLE_CONFIG_INVALID;

	return(ret);
}



/******************************************************************************
*
*	Function:	rx_fifo_ae_config_ioctl
*
*	Purpose:
*
*		Implement the Rx FIFO Almost Empty Config IOCTL service.
*
*	Arguments:
*
*		chan	The structure for the chanel being accessed.
*
*		arg		The argument passed by the user.
*
*	Returned:
*
*		0		All went well.
*		< 0		An appropriate error status.
*
******************************************************************************/

s32 rx_fifo_ae_config_ioctl(chan_data_t* chan, void* arg)
{
	s32*	value	= arg;

	value[0]	= fifo_almost_config(	chan,
										value[0],
										chan->vaddr.gsc_rar_32,
										0,
										SIO4_GSC_CSR_RFR);
	return(0);
}



/******************************************************************************
*
*	Function:	rx_fifo_af_config_ioctl
*
*	Purpose:
*
*		Implement the Rx FIFO Almost Full Config IOCTL service.
*
*	Arguments:
*
*		chan	The structure for the chanel being accessed.
*
*		arg		The argument passed by the user.
*
*	Returned:
*
*		0		All went well.
*		< 0		An appropriate error status.
*
******************************************************************************/

s32 rx_fifo_af_config_ioctl(chan_data_t* chan, void* arg)
{
	s32*	value	= arg;

	value[0]	= fifo_almost_config(	chan,
										value[0],
										chan->vaddr.gsc_rar_32,
										16,
										SIO4_GSC_CSR_RFR);
	return(0);
}



/******************************************************************************
*
*	Function:	rx_fifo_count_ioctl
*
*	Purpose:
*
*		Implement the Rx FIFO Count IOCTL service.
*
*	Arguments:
*
*		chan	The structure for the chanel being accessed.
*
*		arg		The argument passed by the user.
*
*	Returned:
*
*		0		All went well.
*		< 0		An appropriate error status.
*
******************************************************************************/

s32 rx_fifo_count_ioctl(chan_data_t* chan, void* arg)
{
	s32*	value	= arg;

	value[0]	= fifo_count(chan, 16);
	return(0);
}



/******************************************************************************
*
*	Function:	rx_fifo_full_cfg_chan_ioctl
*
*	Purpose:
*
*		Implement the Rx FIFO Full Config Channel IOCTL service.
*
*	Arguments:
*
*		chan	The structure for the chanel being accessed.
*
*		arg		The argument passed by the user.
*
*	Returned:
*
*		0		All went well.
*		< 0		An appropriate error status.
*
******************************************************************************/

s32 rx_fifo_full_cfg_chan_ioctl(chan_data_t* chan, void* arg)
{
	static const parm_32_t	list[]	=
	{
		{
		/* parm	*/	SIO4_RX_FIFO_FULL_CFG_CHAN_HALT,
		/* mask	*/	D19,
		/* val	*/	D19
		},
		{
		/* parm	*/	SIO4_RX_FIFO_FULL_CFG_CHAN_OVER,
		/* mask	*/	D19,
		/* val	*/	0
		},
		{ 0, 0, 0 }
	};

	dev_data_t*	dev		= chan->dev;
	int			index;
	s32			ret;
	s32*		value	= arg;

	if (value[0] == SIO4_RX_FIFO_FULL_CFG_CHAN_READ)
		index	= 0;
	else if (chan->cache.rx_fifo_full_cfg_chan)
		index	= parm_32_set(dev, chan->vaddr.gsc_csr_32, value[0], list);
	else
		index	= parm_32_check(value[0], list);

	if (index < 0)
		ret	= -EINVAL;
	else
		ret	= 0;

	if (chan->cache.rx_fifo_full_cfg_chan)
	{
		index		= parm_32_get(dev, chan->vaddr.gsc_csr_32, list);
		value[0]	= list[index].parm;
	}
	else
	{
		value[0]	= SIO4_RX_FIFO_FULL_CFG_CHAN_OVER;
	}

	return(ret);
}



/******************************************************************************
*
*	Function:	rx_fifo_full_cfg_glb_ioctl
*
*	Purpose:
*
*		Implement the Rx FIFO Full Config Global IOCTL service.
*
*	Arguments:
*
*		chan	The structure for the chanel being accessed.
*
*		arg		The argument passed by the user.
*
*	Returned:
*
*		0		All went well.
*		< 0		An appropriate error status.
*
******************************************************************************/

s32 rx_fifo_full_cfg_glb_ioctl(chan_data_t* chan, void* arg)
{
	static const parm_32_t	list[]	=
	{
		{
		/* parm	*/	SIO4_RX_FIFO_FULL_CFG_GLB_HALT,
		/* mask	*/	SIO4_GSC_BCR_RX_FFC,
		/* val	*/	SIO4_GSC_BCR_RX_FFC
		},
		{
		/* parm	*/	SIO4_RX_FIFO_FULL_CFG_GLB_OVER,
		/* mask	*/	SIO4_GSC_BCR_RX_FFC,
		/* val	*/	0
		},
		{ 0, 0, 0 }
	};

	dev_data_t*	dev		= chan->dev;
	int			index;
	s32			ret;
	s32*		value	= arg;

	if (value[0] == SIO4_RX_FIFO_FULL_CFG_GLB_READ)
		index	= 0;
	else if (dev->cache.rx_fifo_full_cfg_glb)
		index	= parm_32_set(dev, dev->vaddr.gsc_bcr_32, value[0], list);
	else
		index	= parm_32_check(value[0], list);

	if (index < 0)
		ret	= -EINVAL;
	else
		ret	= 0;

	if (dev->cache.rx_fifo_full_cfg_glb)
	{
		index		= parm_32_get(dev, dev->vaddr.gsc_bcr_32, list);
		value[0]	= list[index].parm;
	}
	else
	{
		value[0]	= SIO4_RX_FIFO_FULL_CFG_GLB_OVER;
	}

	return(ret);
}



/******************************************************************************
*
*	Function:	rx_fifo_size_ioctl
*
*	Purpose:
*
*		Implement the Rx FIFO Size IOCTL service.
*
*	Arguments:
*
*		chan	The structure for the chanel being accessed.
*
*		arg		The argument passed by the user.
*
*	Returned:
*
*		0		All went well.
*		< 0		An appropriate error status.
*
******************************************************************************/

s32 rx_fifo_size_ioctl(chan_data_t* chan, void* arg)
{
	s32*	value	= arg;

	value[0]	= chan->cache.fifo_size_rx;
	return(0);
}



/******************************************************************************
*
*	Function:	rx_io_abort_ioctl
*
*	Purpose:
*
*		Implement the Rx I/O Abort IOCTL service.
*
*	Arguments:
*
*		chan	The structure for the channel being accessed.
*
*		arg		The argument passed by the user.
*
*	Returned:
*
*		0		All went well.
*		< 0		An appropriate error status.
*
******************************************************************************/

s32 rx_io_abort_ioctl(chan_data_t* chan, void* arg)
{
	int		aborted;
	s32*	ptr		= arg;

	aborted	= io_abort(&chan->rx);
	ptr[0]	= aborted;
	return(0);
}



/******************************************************************************
*
*	Function:	rx_io_mode_config_ioctl
*
*	Purpose:
*
*		Implement the Rx I/O Mode Configuration IOCTL service.
*
*	Arguments:
*
*		chan	The structure for the channel being accessed.
*
*		arg		The argument passed by the user.
*
*	Returned:
*
*		0		All went well.
*		< 0		An appropriate error status.
*
******************************************************************************/

s32 rx_io_mode_config_ioctl(chan_data_t* chan, void* arg)
{
	s32		ret;
	s32*	value	= arg;

	ret	= io_mode_config(&chan->rx, value);
	return(ret);
}



/******************************************************************************
*
*	Function:	rxc_usc_config_ioctl
*
*	Purpose:
*
*		Implement the RxC USC Configuration IOCTL service.
*
*	Arguments:
*
*		chan	The structure for the chanel being accessed.
*
*		arg		The argument passed by the user.
*
*	Returned:
*
*		0		All went well.
*		< 0		An appropriate error status.
*
******************************************************************************/

s32 rxc_usc_config_ioctl(chan_data_t* chan, void* arg)
{
	static const parm_32_usc_t	list[]	=
	{
		{
		/* parm		*/	SIO4_RXC_USC_CONFIG_IN_LOW,
		/* mask1	*/	SIO4_GSC_PSRCR_USC_RCS,
		/* value1	*/	0x00000010,
		/* mask2	*/	SIO4_USC_IOCR_RCM,
		/* value2	*/	0x0000
		},
		{
		/* parm		*/	SIO4_RXC_USC_CONFIG_IN_HI,
		/* mask1	*/	SIO4_GSC_PSRCR_USC_RCS,
		/* value1	*/	0x00000018,
		/* mask2	*/	SIO4_USC_IOCR_RCM,
		/* value2	*/	0x0000
		},
		{
		/* parm		*/	SIO4_RXC_USC_CONFIG_IN_PRG_CLK,
		/* mask1	*/	SIO4_GSC_PSRCR_USC_RCS,
		/* value1	*/	0x00000000,
		/* mask2	*/	SIO4_USC_IOCR_RCM,
		/* value2	*/	0x0000
		},
		{
		/* parm		*/	SIO4_RXC_USC_CONFIG_IN_PRG_CLK,
		/* mask1	*/	SIO4_GSC_PSRCR_USC_RCS,
		/* value1	*/	0x00000008,
		/* mask2	*/	SIO4_USC_IOCR_RCM,
		/* value2	*/	0x0000
		},
		{
		/* parm		*/	SIO4_RXC_USC_CONFIG_IN_CBL_RC,
		/* mask1	*/	SIO4_GSC_PSRCR_USC_RCS,
		/* value1	*/	0x00000020,
		/* mask2	*/	SIO4_USC_IOCR_RCM,
		/* value2	*/	0x0000
		},
		{
		/* parm		*/	SIO4_RXC_USC_CONFIG_OUT_TCLK,
		/* mask1	*/	SIO4_GSC_PSRCR_USC_RCS,
		/* value1	*/	0x00000038,
		/* mask2	*/	SIO4_USC_IOCR_RCM,
		/* value2	*/	0x0001
		},
		{
		/* parm		*/	SIO4_RXC_USC_CONFIG_OUT_TCC,
		/* mask1	*/	SIO4_GSC_PSRCR_USC_RCS,
		/* value1	*/	0x00000038,
		/* mask2	*/	SIO4_USC_IOCR_RCM,
		/* value2	*/	0x0002
		},
		{
		/* parm		*/	SIO4_RXC_USC_CONFIG_OUT_TCOMP,
		/* mask1	*/	SIO4_GSC_PSRCR_USC_RCS,
		/* value1	*/	0x00000038,
		/* mask2	*/	SIO4_USC_IOCR_RCM,
		/* value2	*/	0x0003
		},
		{
		/* parm		*/	SIO4_RXC_USC_CONFIG_OUT_BRG0,
		/* mask1	*/	SIO4_GSC_PSRCR_USC_RCS,
		/* value1	*/	0x00000038,
		/* mask2	*/	SIO4_USC_IOCR_RCM,
		/* value2	*/	0x0004
		},
		{
		/* parm		*/	SIO4_RXC_USC_CONFIG_OUT_BRG1,
		/* mask1	*/	SIO4_GSC_PSRCR_USC_RCS,
		/* value1	*/	0x00000038,
		/* mask2	*/	SIO4_USC_IOCR_RCM,
		/* value2	*/	0x0005
		},
		{
		/* parm		*/	SIO4_RXC_USC_CONFIG_OUT_CTR1,
		/* mask1	*/	SIO4_GSC_PSRCR_USC_RCS,
		/* value1	*/	0x00000038,
		/* mask2	*/	SIO4_USC_IOCR_RCM,
		/* value2	*/	0x0006
		},
		{
		/* parm		*/	SIO4_RXC_USC_CONFIG_OUT_DPLL,
		/* mask1	*/	SIO4_GSC_PSRCR_USC_RCS,
		/* value1	*/	0x00000038,
		/* mask2	*/	SIO4_USC_IOCR_RCM,
		/* value2	*/	0x0007
		},
		{ 0, 0, 0, 0, 0 }
	};

	dev_data_t*	dev		= chan->dev;
	s32			fw_type;
	int			index;
	s32			ret;
	s32*		value	= arg;

	fw_type_config_get(chan, &fw_type);

	if (fw_type != SIO4_FW_TYPE_CONFIG_Z16C30)
	{
		value[0]	= SIO4_RXC_USC_CONFIG_UNKNOWN;
		return(0);
	}

	if (value[0] == SIO4_RXC_USC_CONFIG_READ)
	{
		index	= 0;
	}
	else if (chan->cache.reg_psrcr)
	{
		index	= parm_32_usc_set(	dev,
									chan->vaddr.gsc_psrcr_32,
									chan->vaddr.usc_iocr,
									value[0],
									list);
	}
	else
	{
		index	= parm_32_usc_check(value[0], list);
	}

	if (index < 0)
		ret	= -EINVAL;
	else
		ret	= 0;

	if (chan->cache.reg_psrcr)
	{
		index		= parm_32_usc_get(	dev,
										chan->vaddr.gsc_psrcr_32,
										chan->vaddr.usc_iocr,
										list);

		if (index >= 0)
			value[0]	= list[index].parm;
		else
			value[0]	= SIO4_RXC_USC_CONFIG_INVALID;
	}
	else
	{
		value[0]	= SIO4_RXC_USC_CONFIG_UNKNOWN;
	}

	return(ret);
}



// ****************************************************************************
s32 rx_fifo_status_ioctl(chan_data_t* chan, void* arg)
{
	u32		csr;
	s32*	value	= arg;

	csr			= os_reg_mem_rx_u32(chan->dev, chan->vaddr.gsc_csr_32);
	value[0]	= SIO4_DECODE(csr, 15, 12);

	if ((value[0] & 0x1) == 0x0)
		value[0]	= SIO4_FIFO_STATUS_EMPTY;
	else if ((value[0] & 0x2) == 0x0)
		value[0]	= SIO4_FIFO_STATUS_ALMOST_EMPTY;
	else if ((value[0] & 0x8) == 0x0)
		value[0]	= SIO4_FIFO_STATUS_FULL;
	else if ((value[0] & 0x4) == 0x0)
		value[0]	= SIO4_FIFO_STATUS_ALMOST_FULL;
	else
		value[0]	= SIO4_FIFO_STATUS_MEDIAN;

	return(0);
}


