// $URL: http://subversion:8080/svn/gsc/trunk/drivers/LINUX/SIO4%20and%20SIO8/SIO4_Linux_1.x.x_GSC_DN/driver/reg.c $
// $Rev: 48525 $
// $Date: 2020-11-23 18:29:37 -0600 (Mon, 23 Nov 2020) $

// SIO4: Device Driver: source file

#include "main.h"



// data types *****************************************************************

typedef struct
{
	chan_data_t*	chan;
	u32				ch_mult;	// GSC registers only.
	u32				field_1;	// GSC registers only.
	u32				field_2;	// GSC registers only.
	u32				offset;
	u32				other;		// Should always be zero.
	u32				reg;
	u32				size;
	u32				type;

	u32				block;
	u32				mask;		// GSC registers only.
	VADDR_T			vaddr;		// Non PCI registers only.

} reg_def_t;



/******************************************************************************
*
*	Function:	_reg_decode
*
*	Purpose:
*
*		Decode a register into its component parts.
*
*	Arguments:
*
*		chan	The structure for the channel being accessed.
*
*		reg		The register id.
*
*		def		The decoded data goes here.
*
*	Returned:
*
*		0		All went well.
*		< 0		An error for the condition that occurred.
*
******************************************************************************/

static int _reg_decode(chan_data_t* chan, u32 reg, reg_def_t* def)
{
	int	ret		= 0;
	int	shift;

	memset(def, 0, sizeof(reg_def_t));

	def->ch_mult	= SIO4_CH_MULT_DECODE(reg);
	def->field_1	= SIO4_FIELD_1_DECODE(reg);
	def->field_2	= SIO4_FIELD_2_DECODE(reg);
	def->offset		= SIO4_OFFSET_DECODE(reg);
	def->other		= SIO4_OTHER_DECODE(reg);
	def->reg		= reg;
	def->size		= SIO4_SIZE_DECODE(reg);
	def->type		= SIO4_TYPE_DECODE(reg);

	switch (def->size)
	{
		default:

			ret			= -EINVAL;
			break;

		case 1:

			def->mask	= 0xFF;
			break;

		case 2:

			def->mask	= 0xFFFF;
			break;

		case 3:

			def->mask	= 0xFFFFFF;
			break;

		case 4:

			def->mask	= 0xFFFFFFFF;
			break;
	}

	switch (def->type)
	{
		default:

			ret	= -EINVAL;
			break;

		case SIO4_REG_GSC:

			switch (def->field_1)
			{
				default:
				case 2:

					ret			= -EINVAL;
					def->mask	= 0;
					break;

				case 0:

					break;

				case 1:

					shift		= chan->index * 4;
					def->mask	&= 0xF << shift;
					break;

				case 3:

					shift		= chan->index * 8;
					def->mask	&= 0xFFUL << shift;
					break;
			}

			switch (def->field_2)
			{
				default:
				case 1:
				case 3:

					ret			= -EINVAL;
					break;

				case 0:

					break;

				case 2:

					shift		= (chan->index * 4)
								+ 16;
					def->mask	|= 0xF << shift;
					break;
			}

			if (def->size != 4)
				ret	= -EINVAL;

			def->block	= 256;
			def->vaddr	= (VADDR_T)
						((long) chan->dev->vaddr.gsc_regs
						+ def->offset
						+ (chan->index * def->ch_mult));
			break;

		case SIO4_REG_PCI:

			if ((def->field_1) || (def->field_2) || (def->ch_mult))
				ret	= -EINVAL;

			def->block	= 256;
			break;

		case SIO4_REG_PLX:

			if ((def->field_1) || (def->field_2) || (def->ch_mult))
				ret	= -EINVAL;

			def->block	= 256;
			def->vaddr	= (VADDR_T)
						((long) chan->dev->vaddr.plx_regs
						+ def->offset);
			break;

		case SIO4_REG_USC:

			if ((def->field_1) || (def->field_2) || (def->ch_mult))
				ret	= -EINVAL;

			if ((def->size != 2) || (def->offset % 2))
				ret	= -EINVAL;

			def->block	= 64;
			def->vaddr	= (VADDR_T)
						((long) chan->vaddr.usc_regs
						+ def->offset);
			break;
	}

	if (def->other)
		ret	= -EINVAL;

	if ((def->offset + def->size) > def->block)
		ret	= -EINVAL;

	def->chan	= chan;
	return(ret);
}



/******************************************************************************
*
*	Function:	_reg_def_mod
*
*	Purpose:
*
*		Perform a read-modify-write operation on a memory based register.
*
*	Arguments:
*
*		def		The register definition.
*
*		value	The value to apply to the modifiable bits.
*
*		mask	The set of bits that may be modified. These are in addition
*				to the mask supplied with the register definition.
*
*	Returned:
*
*		0		All went well.
*		< 0		An appropriate error status.
*
******************************************************************************/

static s32 _reg_def_mod(const reg_def_t* def, u32 value, u32 mask)
{
	s32	ret	= 0;

	mask	&= def->mask;

	switch (def->size)
	{
		default:

			ret	= -EINVAL;
			break;

		case 1:

			if ((mask & 0xFF) == 0xFF)
				os_reg_mem_tx_u8(def->chan->dev, def->vaddr, value);
			else
				os_reg_mem_mx_u8(def->chan->dev, def->vaddr, value, mask);

			break;

		case 2:

			if ((mask & 0xFFFF) == 0xFFFF)
				os_reg_mem_tx_u16(def->chan->dev, def->vaddr, (u16) value);
			else
				os_reg_mem_mx_u16(def->chan->dev, def->vaddr, value, mask);

			break;

		case 4:

			if (mask == 0xFFFFFFFF)
				os_reg_mem_tx_u32(def->chan->dev, def->vaddr, value);
			else
				os_reg_mem_mx_u32(def->chan->dev, def->vaddr, value, mask);

			break;
	}

	return(ret);
}



/******************************************************************************
*
*	Function:	_reg_def_read
*
*	Purpose:
*
*		Perform a read of a memory based register.
*
*	Arguments:
*
*		def		The register definition.
*
*		value	The value read is put here.
*
*	Returned:
*
*		0		All went well.
*		< 0		An appropriate error status.
*
******************************************************************************/

static s32 _reg_def_read(const reg_def_t* def, u32* value)
{
	s32	ret	= 0;

	switch (def->size)
	{
		default:

			ret	= -EINVAL;
			break;

		case 1:

			value[0]	= os_reg_mem_rx_u8(def->chan->dev, def->vaddr);
			break;

		case 2:

			value[0]	= os_reg_mem_rx_u16(def->chan->dev, def->vaddr);
			break;

		case 4:

			value[0]	= os_reg_mem_rx_u32(def->chan->dev, def->vaddr);
			break;
	}

	return(ret);
}



/******************************************************************************
*
*	Function:	_reg_pci_read
*
*	Purpose:
*
*		Read a PCI Configuration register.
*
*	Arguments:
*
*		dev		The structure for the device to access.
*
*		def		The decoded register definition to use.
*
*		value	Put the value read here.
*
*	Returned:
*
*		0		All went well.
*		< 0		An appropriate error status.
*
******************************************************************************/

static s32 _reg_pci_read(dev_data_t* dev, const reg_def_t* def, u32* value)
{
	u8	b;
	u32	d;
	s32	ret	= 0;
	u16	w;

	switch (def->size)
	{
		default:

			ret			= -EINVAL;
			break;

		case 1:

			b			= os_reg_pci_rx_u8(dev, 1, def->offset);
			value[0]	= b;
			break;

		case 2:

			w			= os_reg_pci_rx_u16(dev, 1, def->offset);
			value[0]	= w;
			break;

		case 3:

			d			= os_reg_pci_rx_u32(dev, 1, def->offset);
			value[0]	= d & 0xFFFFFF;
			break;

		case 4:

			d			= os_reg_pci_rx_u32(dev, 1, def->offset);
			value[0]	= d;
			break;
	}

	return(ret);
}



/******************************************************************************
*
*	Function:	reg_mod_ioctl
*
*	Purpose:
*
*		Implement the Register Modify IOCTL service.
*
*	Arguments:
*
*		chan	The structure for the channel to access.
*
*		arg		The argument passed by the application.
*
*	Returned:
*
*		0		All went well.
*		< 0		An appropriate error status.
*
******************************************************************************/

s32 reg_mod_ioctl(chan_data_t* chan, void* arg)
{
	reg_def_t				def;
	dev_data_t*				dev		= chan->dev;
	REGISTER_MOD_PARAMS*	parm	= arg;
	s32						ret;

	ret		= _reg_decode(chan, parm->u32RegisterNumber, &def);

	if (ret)
		;
	else if (def.type == SIO4_REG_GSC)
		ret	= _reg_def_mod(&def, parm->u32Value, parm->u32Mask);
	else if (def.type == SIO4_REG_USC)
		usc_reg_vaddr_mod(dev, def.vaddr, parm->u32Value, parm->u32Mask);
	else
		ret	= -EINVAL;

	return(ret);
}



/******************************************************************************
*
*	Function:	reg_read_ioctl
*
*	Purpose:
*
*		Implement the Register Read IOCTL service.
*
*	Arguments:
*
*		chan	The structure for the channel to access.
*
*		arg		The argument passed by the application.
*
*	Returned:
*
*		0		All went well.
*		< 0		An appropriate error status.
*
******************************************************************************/

s32 reg_read_ioctl(chan_data_t* chan, void* arg)
{
	reg_def_t			def;
	dev_data_t*			dev		= chan->dev;
	REGISTER_PARAMS*	parm	= arg;
	s32					ret;
	u32					value	= 0;

	ret		= _reg_decode(chan, parm->u32RegisterNumber, &def);

	if (ret)
	{
	}
	else if (def.type == SIO4_REG_GSC)
	{
		ret				= _reg_def_read(&def, &value);
		parm->u32Value	= value;
	}
	else if (def.type == SIO4_REG_USC)
	{
		parm->u32Value	= usc_reg_vaddr_read(dev, def.vaddr);
	}
	else if (def.type == SIO4_REG_PLX)
	{
		ret				= _reg_def_read(&def, &value);
		parm->u32Value	= value;
	}
	else if (def.type == SIO4_REG_PCI)
	{
		ret				= _reg_pci_read(chan->dev, &def, &value);
		parm->u32Value	= value;
	}
	else
	{
		ret				= -EINVAL;
	}

	return(ret);
}



/******************************************************************************
*
*	Function:	reg_read_raw_ioctl
*
*	Purpose:
*
*		Implement the Register Read Raw IOCTL service.
*
*	Arguments:
*
*		chan	The structure for the channel to access.
*
*		arg		The argument passed by the application.
*
*	Returned:
*
*		0		All went well.
*		< 0		An appropriate error status.
*
******************************************************************************/

s32 reg_read_raw_ioctl(chan_data_t* chan, void* arg)
{
	dev_data_t*			dev		= chan->dev;
	REGISTER_PARAMS*	parm	= arg;
	s32					ret;
	VADDR_T				vaddr;

	if (parm->u32RegisterNumber <= 63)
	{
		ret				= 0;
		vaddr			= chan->dev->vaddr.gsc_regs;
		vaddr			= (VADDR_T) ((long) vaddr + (parm->u32RegisterNumber * 4));
		parm->u32Value	= os_reg_mem_rx_u32(dev, vaddr);
	}
	else
	{
		ret				= -EINVAL;
	}

	return(ret);
}



/******************************************************************************
*
*	Function:	reg_write_ioctl
*
*	Purpose:
*
*		Implement the Register Write IOCTL service.
*
*	Arguments:
*
*		chan	The structure for the channel to access.
*
*		arg		The argument passed by the application.
*
*	Returned:
*
*		0		All went well.
*		< 0		An appropriate error status.
*
******************************************************************************/

s32 reg_write_ioctl(chan_data_t* chan, void* arg)
{
	reg_def_t			def;
	dev_data_t*			dev	= chan->dev;
	REGISTER_PARAMS*	parm;
	s32					ret;

	parm	= (REGISTER_PARAMS*) arg;
	ret		= _reg_decode(chan, parm->u32RegisterNumber, &def);

	if (ret)
		;
	else if (def.type == SIO4_REG_GSC)
		ret	= _reg_def_mod( &def, parm->u32Value, 0xFFFFFFFF);
	else if (def.type == SIO4_REG_USC)
		usc_reg_vaddr_write(dev, def.vaddr, (u16) parm->u32Value);
	else
		ret	= -EINVAL;

	return(ret);
}



//*****************************************************************************
void reg_mem32_mod_serial(dev_data_t* dev, VADDR_T va, u32 val, u32 mask)
{
	os_reg_mem_mx_u32(dev, va, val, mask);
}


