// $URL: http://subversion:8080/svn/gsc/trunk/drivers/LINUX/SIO4%20and%20SIO8/SIO4_Linux_1.x.x_GSC_DN/driver/osc_cy22393_sio4.c $
// $Rev: 48525 $
// $Date: 2020-11-23 18:29:37 -0600 (Mon, 23 Nov 2020) $

// SIO4: Device Driver: source file

#include "main.h"



/******************************************************************************
*
*	Function:	_compute_pd_osc_program
*
*	Purpose:
*
*		Figure out what post divider and oscillator configuration best produces
*		the requested frquency, given the available oscillator resources and
*		its current usage.
*
*	Arguments:
*
*		osc		The structure for the oscillator to access.
*
*		index	The index of the channel to access.
*
*		arg		The structre we use to exchange data with the caller.
*
*	Returned:
*
*		0		All went well.
*		< 0		An appropriate error status.
*
******************************************************************************/

static int _compute_pd_osc_program(
	osc_cy22393_sio4_t*	osc,
	int					index,
	osc_cy22393_arg_t*	arg)
{
	osc_cy22393_arg_t	data;
	long				delta;
	long				delta_min	= 999999999L;
	long				div;
	long				div_max;
	long				freq_osc;
	long				freq_want	= arg->freq_want;
	int					pd;
	int					ret			= 0;
	osc_cy22393_sio4_t	test;
	osc_cy22393_sio4_t	use			= osc[0];;

	osc->chan[index].freq_want	= (u32) freq_want;

	if (freq_want)
		div_max	= (OSC_FREQ_MAX / ((freq_want + 1) / 2)) + 1;
	else
		div_max	= OSC_FREQ_MAX;

	for (pd = 15; pd >= 0; pd--)
	{
		if (pd > (int) osc->pd_max)
			continue;

		div	= 0x1L << pd;

		if (div > div_max)
			continue;

		freq_osc	= freq_want * div;

		if (freq_osc > OSC_FREQ_MAX)
			continue;

		memset(&data, 0, sizeof(data));
		data.freq_want	= freq_osc;
		test			= osc[0];
		ret				= osc_cy22393_program(&test.osc, index, &data);

		if (ret)
			break;

		delta	= freq_osc - test.osc.clock[index].freq_got;
		delta	= (delta < 0) ? -delta : delta;

		if ((delta < delta_min) ||
			((delta == delta_min) &&
			(test.osc.clock[index].freq_src == CY22393_FREQ_SRC_REF)))
		{
			delta_min					= delta;
			use							= test;
			use.chan[index].freq_got	= (u32) (data.freq_got / div);
			use.chan[index].pd			= (u8) pd;

			if ((delta == 0) && (test.osc.clock[index].freq_src == CY22393_FREQ_SRC_REF))
				break;
		}
	}

	osc[0]	= use;
	return(ret);
}



/******************************************************************************
*
*	Function:	_osc_reg_read
*
*	Purpose:
*
*		Read a value from an oscillator register.
*
*	Arguments:
*
*		dev		The structure for the device to access.
*
*		address	The register address to access.
*
*		value	Record the value read here.
*
*	Returned:
*
*		None.
*
******************************************************************************/

static void _osc_reg_read(dev_data_t* dev, u8 address, u8* value)
{
	os_reg_mem_tx_u32(dev, dev->vaddr.gsc_porar_32, address);
	value[0]	= (u8) (os_reg_mem_rx_u32(dev, dev->vaddr.gsc_pordr_32) & 0xFF);
}



/******************************************************************************
*
*	Function:	_osc_reg_write
*
*	Purpose:
*
*		Write a value to an oscillator register.
*
*	Arguments:
*
*		dev		The structure for the device to access.
*
*		address	The register address to access.
*
*		value	The value to write.
*
*	Returned:
*
*		None.
*
******************************************************************************/

static void _osc_reg_write(dev_data_t* dev, u8 address, u8 value)
{
	os_reg_mem_tx_u32(dev, dev->vaddr.gsc_porar_32, address);
	os_reg_mem_tx_u32(dev, dev->vaddr.gsc_pordr_32, value);

	if (dev->cache.osc_chip == SIO4_OSC_CHIP_CY22393_2)
		os_reg_mem_tx_u32(dev, dev->vaddr.gsc_pord2r_32, value);
}



/******************************************************************************
*
*	Function:	_osc_regs_write
*
*	Purpose:
*
*		Write values to all oscillator registers.
*
*	Arguments:
*
*		dev		The structure for the device to access.
*
*		osc		The structure for the oscillator to access.
*
*	Returned:
*
*		None.
*
******************************************************************************/

static void _osc_regs_write(dev_data_t* dev, osc_cy22393_sio4_t* osc)
{
	_osc_reg_write(dev, 0x06, osc->_06);
	_osc_reg_write(dev, 0x07, osc->_07);

	_osc_reg_write(dev, 0x08, osc->osc.data._08);
	_osc_reg_write(dev, 0x09, osc->osc.data._08);
	_osc_reg_write(dev, 0x0A, osc->osc.data._0A);
	_osc_reg_write(dev, 0x0B, osc->osc.data._0A);
	_osc_reg_write(dev, 0x0C, osc->osc.data._0C);
	_osc_reg_write(dev, 0x0D, osc->osc.data._0D);
	_osc_reg_write(dev, 0x0E, osc->osc.data._0E);
	_osc_reg_write(dev, 0x0F, osc->osc.data._0F);
	_osc_reg_write(dev, 0x10, osc->osc.data._10);
	_osc_reg_write(dev, 0x11, osc->osc.data._11);
	_osc_reg_write(dev, 0x12, osc->osc.data._12);
	_osc_reg_write(dev, 0x13, osc->osc.data._13);
	_osc_reg_write(dev, 0x14, osc->osc.data._14);
	_osc_reg_write(dev, 0x15, osc->osc.data._15);
	_osc_reg_write(dev, 0x16, osc->osc.data._16);
	_osc_reg_write(dev, 0x17, osc->osc.data._17);

	_osc_reg_write(dev, 0x18, osc->_18);
	_osc_reg_write(dev, 0x19, osc->_19);
	_osc_reg_write(dev, 0x1A, osc->_1a);
	_osc_reg_write(dev, 0x1B, osc->_1b);

	_osc_reg_write(dev, 0x40, osc->osc.data._40);
	_osc_reg_write(dev, 0x41, osc->osc.data._41);
	_osc_reg_write(dev, 0x42, osc->osc.data._42);
	_osc_reg_write(dev, 0x43, osc->osc.data._40);
	_osc_reg_write(dev, 0x44, osc->osc.data._41);
	_osc_reg_write(dev, 0x45, osc->osc.data._42);
	_osc_reg_write(dev, 0x46, osc->osc.data._40);
	_osc_reg_write(dev, 0x47, osc->osc.data._41);
	_osc_reg_write(dev, 0x48, osc->osc.data._42);
	_osc_reg_write(dev, 0x49, osc->osc.data._40);
	_osc_reg_write(dev, 0x4A, osc->osc.data._41);
	_osc_reg_write(dev, 0x4B, osc->osc.data._42);
	_osc_reg_write(dev, 0x4C, osc->osc.data._40);
	_osc_reg_write(dev, 0x4D, osc->osc.data._41);
	_osc_reg_write(dev, 0x4E, osc->osc.data._42);
	_osc_reg_write(dev, 0x4F, osc->osc.data._40);
	_osc_reg_write(dev, 0x50, osc->osc.data._41);
	_osc_reg_write(dev, 0x51, osc->osc.data._42);
	_osc_reg_write(dev, 0x52, osc->osc.data._40);
	_osc_reg_write(dev, 0x53, osc->osc.data._41);
	_osc_reg_write(dev, 0x54, osc->osc.data._42);
	_osc_reg_write(dev, 0x55, osc->osc.data._40);
	_osc_reg_write(dev, 0x56, osc->osc.data._41);
	_osc_reg_write(dev, 0x57, osc->osc.data._42);
}



/******************************************************************************
*
*	Function:	_osc_regs_set_constants
*
*	Purpose:
*
*		Fill in the constant data for the given oscillator.
*
*	Arguments:
*
*		osc		The oscillator data structure.
*
*	Returned:
*
*		None.
*
******************************************************************************/

static void _osc_regs_set_constants(osc_cy22393_t* osc)
{
	osc->data._0F	= 0x50;		// Xbuf_OE: 0 (disable)
								// PdnEn: 0 (OE Control)
								// Clk{A,B,D,E}_ACAdj: 1 (nominal)
								// Clk{C,X}_ACAdj: 1 (nominal)
	osc->data._10	= 0x55;		// Clk{A,B}_DCAdj: 1 (nominal)
								// ClkC_DCAdj: 1 (nominal)
								// Clk{D,E}_DCAdj: 1 (nominal)
								// ClkX_DCAdj: 1 (nominal)
	osc->data._13	&= ~0x80;	// Reserved: 0 (required)
	osc->data._16	&= ~0x80;	// Reserved: 0 (required)
	osc->data._17	= 0x00;		// OscCap: 0 (Extern Ref Clock)
	osc->data._42	&= ~0x80;	// DivSel: 0 (convention)
}



/******************************************************************************
*
*	Function:	_wait_for_ready
*
*	Purpose:
*
*		Wait for the firmware to indicate the oscillator is ready.
*
*	Arguments:
*
*		dev		The device structure to access.
*
*	Returned:
*
*		0		All went well.
*		< 0		An appropriate error status.
*
******************************************************************************/

static int _wait_for_ready(dev_data_t* dev)
{
	int	end		= 2 * os_time_tick_rate();
	u32	pocsr;
	int	ret		= -EIO;
	int	wait;

	for (wait = 0; wait < end; wait++)
	{
		pocsr	= os_reg_mem_rx_u32(dev, dev->vaddr.gsc_pocsr_32);

		if (pocsr & SIO4_GSC_POCSR_PRG_DONE)
		{
			ret	= 0;
			break;
		}

		os_time_tick_sleep(1);
	}

	return(ret);
}



/******************************************************************************
*
*	Function:	_osc_program
*
*	Purpose:
*
*		Submit the recorded Programmable Oscillator RAM data to the oscillator
*		and wait for completion.
*
*	Arguments:
*
*		dev		The structure for the device to access.
*
*		osc		The structure for the oscillator to access.
*
*	Returned:
*
*		0		All went well.
*		< 0		An appropriate error status.
*
******************************************************************************/

static int _osc_program(dev_data_t* dev, osc_cy22393_sio4_t* osc)
{
	int	loop;
	u32	pocsr;
	int	ret		= -EIO;

	for (loop = 1;; loop++)
	{
		ret	= _wait_for_ready(dev);

		if (ret < 0)
			break;

		_osc_regs_write(dev, osc);
		pocsr	= SIO4_GSC_POCSR_PROGRAM;
		os_reg_mem_tx_u32(dev, dev->vaddr.gsc_pocsr_32, pocsr);
		os_time_us_delay(10000);	// Wait for Lock Time to pass.
		ret	= _wait_for_ready(dev);

		if (ret < 0)
			break;

		pocsr	= os_reg_mem_rx_u32(dev, dev->vaddr.gsc_pocsr_32);

		if ((pocsr & SIO4_GSC_POCSR_PRG_ERR) == 0)
			break;

		if (loop >= 10)
		{
			ret	= -EIO;
			break;
		}
	}

	return(ret);
}



/******************************************************************************
*
*	Function:	_chan_fw_config_write
*
*	Purpose:
*
*		Write channel's firmware configuration to the device.
*
*	Arguments:
*
*		dev		The structure for the device to access.
*
*		osc		The structure for the oscillator to access.
*
*	Returned:
*
*		None.
*
******************************************************************************/

static void _chan_fw_config_write(dev_data_t* dev, osc_cy22393_sio4_t* osc)
{
	u32	pocsr;

	pocsr	= (((u32) osc->chan[0].pd) << 8)
			| (((u32) osc->chan[1].pd) << 12)
			| (((u32) osc->chan[2].pd) << 16)
			| (((u32) osc->chan[3].pd) << 20)
			| SIO4_GSC_POCSR_PD_SET;
	os_reg_mem_tx_u32(dev, dev->vaddr.gsc_pocsr_32, pocsr);
}



/******************************************************************************
*
*	Function:	_chan_measure_freq
*
*	Purpose:
*
*		Measure the channel's oscillator output frequency.
*
*	Arguments:
*
*		dev		The structure for the device to access.
*
*		index	The index of the channel to access.
*
*		arg		The argument structure we use to exchange with the CY22393
*				support code.
*
*	Returned:
*
*		0		All went well.
*		< 0		An appropriate error status.
*
******************************************************************************/

static int _chan_measure_freq(dev_data_t* dev, int index, sio4_osc_t* arg)
{
	int	end		= os_time_tick_rate();
	int	i;
	u32	pocsr;
	int	ret;

	pocsr	= 0x2 << index;
	os_reg_mem_tx_u32(dev, dev->vaddr.gsc_pocsr_32, pocsr);

	for (i = 0; i < end; i++)
	{
		os_time_tick_sleep(1);
		pocsr	= os_reg_mem_rx_u32(dev, dev->vaddr.gsc_pocsr_32);

		if (pocsr & SIO4_GSC_POCSR_MSR_DONE)
			break;
	}

	if (pocsr & SIO4_GSC_POCSR_MSR_DONE)
		ret		= 0;
	else
		ret		= -ETIMEDOUT;

	arg->freq_got	= (pocsr >> 8) * 10;
	return(ret);
}



/******************************************************************************
*
*	Function:	_chan_reset
*
*	Purpose:
*
*		Reset the channel's oscillator configuration. Attempt to disable
*		everything.
*
*	Arguments:
*
*		dev		The structure for the device to access.
*
*		osc		The structure for the oscillator to access.
*
*		index	The index of the channel to access.
*
*		arg		The programmed results are put here.
*
*	Returned:
*
*		0		All went well.
*		< 0		An appropriate error status.
*
******************************************************************************/

static int _chan_reset(
	dev_data_t*			dev,
	osc_cy22393_sio4_t*	osc,
	int					index,
	sio4_osc_t*			arg)
{
	osc_cy22393_arg_t	data;
	int					loop;
	long				max;
	long				min;
	int					ret		= 0;
	sio4_osc_t			tmp;

	for (loop = 0; loop <= 20; loop++)
	{
		arg->chip		= dev->cache.osc_chip;
		arg->freq_want	= 0;
		ret				= os_sem_lock(&osc->sem);
		arg->freq_ref	= osc->osc.freq_ref;

		if (ret)
		{
			ret	= -ERESTARTSYS;
			break;
		}

		osc->chan[index].pd			= dev->cache.osc_pd_max;
		osc->chan[index].freq_want	= 0;
		osc->chan[index].freq_got	= 0;

		_chan_fw_config_write(dev, osc);
		ret	= osc_cy22393_reset(&osc->osc, index, &data);

		if (ret == 0)
			ret	= _osc_program(dev, osc);

		if (ret == 0)
			ret	= _chan_measure_freq(dev, index, &tmp);
		else
			_chan_measure_freq(dev, index, &tmp);

		arg->freq_got	= tmp.freq_got;
		min				= (((s32) osc->chan[index].freq_want / 10) - 1 - (((s32) osc->chan[index].freq_want + 9999) / 10000)) * 10;
		max				= (((s32) osc->chan[index].freq_want / 10) + 1 + (((s32) osc->chan[index].freq_want + 9999) / 10000)) * 10;
		os_sem_unlock(&osc->sem);

		if (tmp.freq_got < min)
		{
			continue;
		}
		else if (tmp.freq_got > max)
		{
			continue;
		}
		else
		{
			break;
		}
	}

	return(ret);
}



/******************************************************************************
*
*	Function:	_measure_service
*
*	Purpose:
*
*		Perform a frequency measurement test of the serial channel. The
*		hardware takes 10ms to do this. We might take longer.
*
*	Arguments:
*
*		dev		The device structure to access.
*
*		index	The channel index to access.
*
*		arg		We exchange data here. We set everything appropriately
*				before returning.
*
*	Returned:
*
*		0		All went well.
*		< 0		An appropriate error status.
*
******************************************************************************/

static int _measure_service(dev_data_t* dev, int index, sio4_osc_t* arg)
{
	osc_cy22393_arg_t	data;
	osc_cy22393_sio4_t*	osc		= &dev->osc.cy22393;
	int					ret;

	for (;;)	// A convenience loop.
	{
		if ((index < 0) || (index > 3))
		{
			ret	= -EINVAL;
			break;
		}

		ret	= os_sem_lock(&osc->sem);

		if (ret)
		{
			ret	= -ERESTARTSYS;
			break;
		}

		ret				= osc_cy22393_info(&osc->osc, index, &data);
		arg->chip		= dev->cache.osc_chip;
		arg->freq_ref	= osc->osc.freq_ref;
		arg->freq_want	= osc->chan[index].freq_want;
		arg->freq_got	= osc->chan[index].freq_got;

		if (ret == 0)
			ret	= _chan_measure_freq(dev, index, arg);

		os_sem_unlock(&osc->sem);
		break;
	}

	return(ret);
}



/******************************************************************************
*
*	Function:	_program_service
*
*	Purpose:
*
*		Program the referenced channel's clock feature for the requested
*		frequency, or as close as possible.
*
*	Arguments:
*
*		dev		The device structure to access.
*
*		index	The channel index to access.
*
*		arg		We exchange data here. We ignore everything but the
*				requested frequency on input, and set all other fields
*				before returning.
*
*	Returned:
*
*		0		All went well.
*		< 0		An appropriate error status.
*
******************************************************************************/

static int _program_service(dev_data_t* dev, int index, sio4_osc_t* arg)
{
	osc_cy22393_arg_t	data;
	int					loop;
	long				max;
	long				min;
	osc_cy22393_sio4_t*	osc		= &dev->osc.cy22393;
	int					ret		= 0;
	sio4_osc_t			tmp;

	for (loop = 0; loop <= 20; loop++)	// A convenience loop.
	{
		arg->chip		= dev->cache.osc_chip;
		arg->freq_ref	= osc->osc.freq_ref;

		if ((index < 0) ||
			(index > 3) ||
			(arg->freq_want < 0) ||
			(arg->freq_want > _20MHZ))
		{
			ret	= -EINVAL;
			break;
		}

		ret	= os_sem_lock(&osc->sem);

		if (ret)
		{
			ret	= -ERESTARTSYS;
			break;
		}

		data.freq_ref	= osc->osc.freq_ref;
		data.freq_want	= arg->freq_want;
		ret				= _compute_pd_osc_program(osc, index, &data);
		arg->chip		= dev->cache.osc_chip;
		arg->freq_ref	= osc->osc.freq_ref;
		arg->freq_want	= osc->chan[index].freq_want;
		arg->freq_got	= osc->chan[index].freq_got;

		if (ret == 0)
			ret	= _osc_program(dev, osc);

		_chan_fw_config_write(dev, osc);

		if (ret)
		{
			os_sem_unlock(&osc->sem);
			break;
		}

		ret	= _chan_measure_freq(dev, index, &tmp);
		os_sem_unlock(&osc->sem);
		arg->freq_got	= tmp.freq_got;

		if (ret)
			break;

		// Verify the results.
		min	= (((s32) arg->freq_want / 10) - 1 - (((s32) arg->freq_want + 9999) / 10000)) * 10;
		max	= (((s32) arg->freq_want / 10) + 1 + (((s32) arg->freq_want + 9999) / 10000)) * 10;

		if (tmp.freq_got < min)
		{
			continue;
		}
		else if (tmp.freq_got > max)
		{
			continue;
		}
		else
		{
			break;
		}
	}

	return(ret);
}



/******************************************************************************
*
*	Function:	_init_service
*
*	Purpose:
*
*		Initialize the referenced channels clock feature.
*
*	Arguments:
*
*		dev		The device structure to access.
*
*		index	The channel index to access.
*
*		arg		We exchange data here. We set everything except the reference
*				frequency to zero.
*
*	Returned:
*
*		0		All went well.
*		< 0		An appropriate error status.
*
******************************************************************************/

static int _init_service(dev_data_t* dev, int index, sio4_osc_t* arg)
{
	int	ret;

	arg->freq_want	= dev->osc.cy22393.osc.freq_ref;
	ret				= _program_service(dev, index, arg);
	return(ret);
}



/******************************************************************************
*
*	Function:	_reference_service
*
*	Purpose:
*
*		Tell the oscillator code the channel's reference frequency. If the
*		specified reference frequency is zero, then we report the current
*		setting.
*
*	Arguments:
*
*		dev		The device structure to access.
*
*		index	The channel index to access.
*
*		arg		We exchange data here. We set everything except the
*				reference frequency to zero.
*
*	Returned:
*
*		0		All went well.
*		< 0		An appropriate error status.
*
******************************************************************************/

static int _reference_service(dev_data_t* dev, int index, sio4_osc_t* arg)
{
	osc_cy22393_arg_t	data;
	osc_cy22393_sio4_t*	osc	= &dev->osc.cy22393;
	int					ret;

	for (;;)	// A convenience loop.
	{
		arg->chip	= dev->cache.osc_chip;

		if ((index < 0) || (index > 3))
		{
			ret	= -EINVAL;
			break;
		}

		if ((arg->freq_ref != -1) &&	// A query.
			((arg->freq_ref < _8MHZ) ||	// Ext Ref Crystal
			(arg->freq_ref > _30MHZ)))	// Ext Ref Crystal
		{
			ret	= -EINVAL;
			break;
		}

		if ((arg->freq_ref != -1) &&			// A query.
			(osc->ref_comp) &&					// Computed
			((arg->freq_ref < osc->ref_min) ||	// Minimum
			(arg->freq_ref > osc->ref_max)))	// Maximum
		{
			ret	= -EINVAL;
			break;
		}

		ret	= os_sem_lock(&osc->sem);

		if (ret)
		{
			ret	= -ERESTARTSYS;
			break;
		}

		data.freq_ref	= arg->freq_ref;
		ret				= osc_cy22393_reference(&osc->osc, index, &data);
		arg->chip		= dev->cache.osc_chip;
		arg->freq_ref	= osc->osc.freq_ref;
		arg->freq_want	= osc->chan[index].freq_want;
		arg->freq_got	= osc->chan[index].freq_got;

		if (ret == 0)
			ret	= _osc_program(dev, osc);

		os_sem_unlock(&osc->sem);
		break;
	}

	return(ret);
}



/******************************************************************************
*
*	Function:	osc_cy22393_sio4_close
*
*	Purpose:
*
*		Perform any needed work as a device is being closed.
*
*	Arguments:
*
*		dev		The device structure to access.
*
*		index	The index of the channel to access.
*
*	Returned:
*
*		None.
*
******************************************************************************/

void osc_cy22393_sio4_close(dev_data_t* dev, int index)
{
	sio4_osc_t	arg;

	osc_cy22393_sio4_t*	osc	= &dev->osc.cy22393;

	_chan_reset(dev, osc, index, &arg);
}



/******************************************************************************
*
*	Function:	osc_cy22393_sio4_info
*
*	Purpose:
*
*		Provide information on the channel's oscillator configuration.
*
*	Arguments:
*
*		dev		The device structure to access.
*
*		index	The serial channel to access.
*
*		arg		The structure passed from the application
*
*	Returned:
*
*		>= 0	Success.
*		< 0		Try another configuration option.
*
******************************************************************************/

int osc_cy22393_sio4_info(dev_data_t* dev, int index, sio4_osc_t* arg)
{
	osc_cy22393_arg_t	data;
	osc_cy22393_sio4_t*	osc		= &dev->osc.cy22393;
	int					ret;

	for (;;)	// A convenience loop.
	{
		if ((index < 0) || (index > 3))
		{
			ret	= -EINVAL;
			break;
		}


		ret	= os_sem_lock(&osc->sem);

		if (ret)
		{
			ret	= -ERESTARTSYS;
			break;
		}

		ret				= osc_cy22393_info(&osc->osc, index, &data);
		arg->chip		= dev->cache.osc_chip;
		arg->freq_ref	= osc->osc.freq_ref;
		arg->freq_want	= osc->chan[index].freq_want;
		arg->freq_got	= osc->chan[index].freq_got;

		os_sem_unlock(&osc->sem);
		break;
	}

	return(ret);
}



/******************************************************************************
*
*	Function:	osc_cy22393_sio4_init
*
*	Purpose:
*
*		Initialize the referenced channels clock feature.
*
*	Arguments:
*
*		dev		The device structure to access.
*
*		index	The channel index to access.
*
*		arg		We exchange data here. We set everything except the reference
*				frequency to zero.
*
*	Returned:
*
*		0		All went well.
*		< 0		An appropriate error status.
*
******************************************************************************/

int osc_cy22393_sio4_init(dev_data_t* dev, int index, sio4_osc_t* arg)
{
	int	ret;

	ret	= _init_service(dev, index, arg);
	return(ret);
}



/******************************************************************************
*
*	Function:	osc_cy22393_sio4_measure
*
*	Purpose:
*
*		Perform a frequency measurement test of the serial channel. The
*		hardware takes 10ms to do this. We might take longer.
*
*	Arguments:
*
*		dev		The device structure to access.
*
*		index	The channel index to access.
*
*		arg		We exchange data here. We set everything appropriately before
*				returning.
*
*	Returned:
*
*		0		All went well.
*		< 0		An appropriate error status.
*
******************************************************************************/

int osc_cy22393_sio4_measure(dev_data_t* dev, int index, sio4_osc_t* arg)
{
	int	ret;

	ret	= _measure_service(dev, index, arg);
	return(ret);
}



/******************************************************************************
*
*	Function:	osc_cy22393_sio4_open
*
*	Purpose:
*
*		Perform any needed work as a device is being opened.
*
*	Arguments:
*
*		dev		The device structure to access.
*
*		index	The index of the channel to access.
*
*	Returned:
*
*		None.
*
******************************************************************************/

void osc_cy22393_sio4_open(dev_data_t* dev, int index)
{
	sio4_osc_t	arg;

	_init_service(dev, index, &arg);
}



/******************************************************************************
*
*	Function:	osc_cy22393_sio4_program
*
*	Purpose:
*
*		Program the referenced channels clock feature for the requested
*		frequency, or as close as possible.
*
*	Arguments:
*
*		dev		The device structure to access.
*
*		index	The channel index to access.
*
*		arg		We exchange data here. We ignore everything but the requested
*				frequency on input, and set all other fields before returning.
*
*	Returned:
*
*		0		All went well.
*		< 0		An appropriate error status.
*
******************************************************************************/

int osc_cy22393_sio4_program(dev_data_t* dev, int index, sio4_osc_t* arg)
{
	int	ret;

	ret	= _program_service(dev, index, arg);
	return(ret);
}



/******************************************************************************
*
*	Function:	osc_cy22393_sio4_reference
*
*	Purpose:
*
*		Tell the oscillator code the channel's reference frequency. If the
*		specified reference frequency is zero, then we report the current
*		setting.
*
*	Arguments:
*
*		dev		The device structure to access.
*
*		index	The channel index to access.
*
*		arg		We exchange data here. We set everything except the reference
*				frequency to zero.
*
*	Returned:
*
*		0		All went well.
*		< 0		An appropriate error status.
*
******************************************************************************/

int osc_cy22393_sio4_reference(dev_data_t* dev, int index, sio4_osc_t* arg)
{
	int	ret;

	ret	= _reference_service(dev, index, arg);
	return(ret);
}



/******************************************************************************
*
*	Function:	osc_cy22393_sio4_reset
*
*	Purpose:
*
*		Reset the referenced channels clock feature.
*
*	Arguments:
*
*		dev		The device structure to access.
*
*		index	The channel index to access.
*
*		arg		We exchange data here. We set everything except the reference
*				frequency to zero.
*
*	Returned:
*
*		0		All went well.
*		< 0		An appropriate error status.
*
******************************************************************************/

int osc_cy22393_sio4_reset(dev_data_t* dev, int index, sio4_osc_t* arg)
{
	int	ret;

	ret				= _chan_reset(dev, &dev->osc.cy22393, index, arg);
	arg->chip		= dev->cache.osc_chip;
	arg->freq_ref	= dev->osc.cy22393.osc.freq_ref;
	arg->freq_want	= dev->osc.cy22393.chan[index].freq_want;
	arg->freq_got	= dev->osc.cy22393.chan[index].freq_got;
	return(ret);
}



/******************************************************************************
*
*	Function:	osc_cy22393_sio4_startup
*
*	Purpose:
*
*		Perform a one-time initialization.
*
*	Arguments:
*
*		dev		The device structure to access.
*
*	Returned:
*
*		None.
*
******************************************************************************/

void osc_cy22393_sio4_startup(dev_data_t* dev)
{
	sio4_osc_t			arg;
	long				fudge;
	osc_cy22393_sio4_t*	osc		= &dev->osc.cy22393;
	osc_cy22393_arg_t	tmp;

	memset(osc, 0, sizeof(osc[0]));
	os_sem_create(&osc->sem);
	osc->pd_max	= dev->cache.osc_pd_max;
	_osc_reg_read(dev, 0x06, &osc->_06);
	_osc_reg_read(dev, 0x07, &osc->_07);
	_osc_reg_read(dev, 0x18, &osc->_18);
	_osc_reg_read(dev, 0x19, &osc->_19);
	_osc_reg_read(dev, 0x1A, &osc->_1a);
	_osc_reg_read(dev, 0x1B, &osc->_1b);

	memset(&arg, 0, sizeof(arg));
	tmp.freq_ref	= SIO4_OSC_FREQ_REF_DEFAULT;
	osc_cy22393_startup(&osc->osc, &tmp);
	_osc_regs_set_constants(&osc->osc);
	_osc_program(dev, osc);

	// Measure the reference clock.
	memset(&arg, 0, sizeof(arg));
	arg.freq_ref	= SIO4_OSC_FREQ_REF_DEFAULT;
	_reference_service(dev, 0, &arg);
	_init_service(dev, 0, &arg);
	_measure_service(dev, 0, &arg);

	fudge			= arg.freq_got / 10000;
	osc->ref_meas	= arg.freq_got;
	osc->ref_max	= arg.freq_got + fudge;
	osc->ref_min	= arg.freq_got - fudge;
	arg.freq_got	= arg.freq_got + fudge;

	arg.freq_got	/= 10000;
	arg.freq_got	*= 10000;
	osc->ref_comp	= arg.freq_got;
	arg.freq_ref	= arg.freq_got;
	_reference_service(dev, 0, &arg);

	// Disable the channel clocks.
	_chan_reset(dev, osc, 0, &arg);
	_chan_reset(dev, osc, 1, &arg);
	_chan_reset(dev, osc, 2, &arg);
	_chan_reset(dev, osc, 3, &arg);
	_osc_program(dev, osc);
}



/******************************************************************************
*
*	Function:	osc_cy22393_sio4_test
*
*	Purpose:
*
*		Test the requested frequency to how close we can get.
*
*	Arguments:
*
*		dev		The device structure to access.
*
*		index	The channel index to access.
*
*		arg		We exchange data here. We ignore everything but the requested
*				frequency on input, and set all other fields before returning.
*
*	Returned:
*
*		0		All went well.
*		< 0		An appropriate error status.
*
******************************************************************************/

int osc_cy22393_sio4_test(dev_data_t* dev, int index, sio4_osc_t* arg)
{
	osc_cy22393_arg_t	data;
	osc_cy22393_sio4_t*	osc		= &dev->osc.cy22393;
	int					ret;
	osc_cy22393_sio4_t	temp;

	for (;;)	// A convenience loop.
	{
		if ((index < 0) ||
			(index > 3) ||
			(arg->freq_want < 0) ||
			(arg->freq_want > _20MHZ))
		{
			ret	= -EINVAL;
			break;
		}

		ret	= os_sem_lock(&osc->sem);

		if (ret)
		{
			ret	= -ERESTARTSYS;
			break;
		}

		temp	= osc[0];
		os_sem_unlock(&osc->sem);

		data.freq_ref	= temp.osc.freq_ref;
		data.freq_want	= arg->freq_want;

		ret	= _compute_pd_osc_program(&temp, index, &data);

		if (ret)
			break;

		arg->chip		= dev->cache.osc_chip;
		arg->freq_ref	= temp.osc.freq_ref;
		arg->freq_got	= temp.chan[index].freq_want;
		break;
	}

	return(ret);
}



