// $URL: http://subversion:8080/svn/gsc/trunk/drivers/LINUX/SIO4%20and%20SIO8/SIO4_Linux_1.x.x_GSC_DN/driver/main.c $
// $Rev: 53095 $
// $Date: 2023-06-13 10:42:41 -0500 (Tue, 13 Jun 2023) $

// SIO4: Device Driver: source file

#include "main.h"



// variables ******************************************************************

gsc_global_t	gsc_global;



// ****************************************************************************
static VADDR_T _map_bar(dev_data_t* dev, u8 index)
{
	u32		adrs;
	u16		offset	= 0x10 + (4 * index);
	u32		reg;
	u32		size;
	VADDR_T	vaddr;

	reg		= os_reg_pci_rx_u32(dev, 1, offset);
	adrs	= (u32) os_bar_pci_address(dev, index);
	size	= (u32) os_bar_pci_size(dev, index);

	if (reg & 0x1)
		vaddr	= 0;
	else
		vaddr	= (VADDR_T) ioremap(adrs, size);

#if DEV_BAR_SHOW
	printk(	"BAR%d"
			": Offset 0x%02lX"
			", Reg 0x%08lX"
			", Map %s"
			", Adrs 0x%08lX"
			", Size %4ld"
			", vaddr 0x%lX"
			"\n",
			(int) index,
			(long) offset,
			(long) reg,
			size ? (reg & 0x1 ? "I/O" : "mem") : "N/A",
			(long) adrs,
			(long) size,
			(long) vaddr);
#endif

	return(vaddr);
}



// ****************************************************************************
static void _map_bar0(dev_data_t* dev)
{
	dev->vaddr.plx_regs	= _map_bar(dev, 0);
}



// ****************************************************************************
static void _map_bar2(dev_data_t* dev)
{
	dev->vaddr.gsc_regs	= _map_bar(dev, 2);
}



/******************************************************************************
*
*	Function:	_dev_add
*
*	Purpose:
*
*		Add an entry to the device list.
*
*	Arguments:
*
*		None.
*
*	Returned:
*
*		NULL	Not able to add device.
*		!NULL	The device was added and this is its structure.
*
******************************************************************************/

static dev_data_t* _dev_add(void)
{
	dev_data_t*	dev;

	if (gsc_global.dev_qty >= (int) SIZEOF_ARRAY(gsc_global.dev_list))
	{
		printk("<1>%s: too many device found.\n", SIO4_NAME);
		dev	= NULL;
	}
	else
	{
		dev	= (dev_data_t*) kmalloc(sizeof(dev_data_t), GFP_KERNEL);

		if (dev == NULL)
		{
			printk("<1>%s: memory allocation failed.\n", SIO4_NAME);
		}
		else
		{
			gsc_global.dev_list[gsc_global.dev_qty]	= dev;
			dev->board_index	= gsc_global.dev_qty;
			gsc_global.dev_qty++;
		}
	}

	return(dev);
}



/******************************************************************************
*
*	Function:	dev_lock_filp
*
*	Purpose:
*
*		Obtain exclusive access to the channel reference by the given file
*		structure.
*
*	Arguments:
*
*		filp	The file structure we received from the kernel.
*
*		chan	We record the channel structure pointer here.
*
*	Returned:
*
*		0		All went well.
*		< 0		An appropriate error status.
*
******************************************************************************/

int dev_lock_filp(struct file* filp, chan_data_t** chan)
{
	int	ret;

	for (;;)	// A convenience loop.
	{
		if (filp == NULL)
		{
			// This should never happen.
			ret	= -ENODEV;
			break;
		}

		chan[0]	= (chan_data_t*) filp->private_data;

		if (chan[0] == NULL)
		{
			// The referenced device doesn't exist.
			ret	= -ENODEV;
			break;
		}

		ret	= os_sem_lock(&chan[0]->sem);

		if (ret)
		{
			// We got a signal rather than the semaphore.
			ret	= -ERESTARTSYS;
			break;
		}

		if (chan[0]->in_use == 0)
		{
			// The device isn't open.
			ret	= -ENODEV;
			os_sem_unlock(&chan[0]->sem);
		}

		break;
	}

	return(ret);
}



/******************************************************************************
*
*	Function:	dev_lock_inode
*
*	Purpose:
*
*		Obtain exclusive access to the channel reference by the given
*		inode structure.
*
*	Arguments:
*
*		inode	The inode structure we received from the kernel.
*
*		opening	Is the device being opened?
*
*		chan	We record the channel structure pointer here.
*
*	Returned:
*
*		0		All went well.
*		< 0		An appropriate error status.
*
******************************************************************************/

int dev_lock_inode(struct inode* inode, int opening, chan_data_t** chan)
{
	int			board;
	int			channel;
	dev_data_t*	dev;
	int			index;
	int			minor;
	int			ret;

	for (;;)	// A convenience loop.
	{
		if (inode == NULL)
		{
			// This should never happen.
			ret	= -ENODEV;
			break;
		}

		minor	= MINOR(inode->i_rdev);
		index	= minor - 1;
		board	= index / 4;

		if (board >= (int) SIZEOF_ARRAY(gsc_global.dev_list))
		{
			// The referenced device doesn't exist.
			ret	= -ENODEV;
			break;
		}

		dev	= gsc_global.dev_list[board];

		if (dev == NULL)
		{
			// The referenced device doesn't exist.
			ret	= -ENODEV;
			break;
		}

		channel	= index % 4;
		chan[0]	= &dev->channel[channel];
		ret	= os_sem_lock(&chan[0]->sem);

		if (ret)
		{
			// We got a signal rather than the semaphore.
			ret	= -ERESTARTSYS;
			break;
		}

		if (opening)
		{
			// The device should be closed.

			if (chan[0]->in_use)
			{
				// The device is already open.
				ret	= -EBUSY;
				os_sem_unlock(&chan[0]->sem);
			}
		}
		else
		{
			// The device should be open.

			if (chan[0]->in_use == 0)
			{
				// The device isn't open.
				ret	= -ENODEV;
				os_sem_unlock(&chan[0]->sem);
			}
		}

		break;
	}

	return(ret);
}



/******************************************************************************
* cleanup_module - function invoked on rmmod command
*
* This function unregisters the device, unmaps the remapped addresses &
* frees allocated memory for the device.
*
* RETURNS: nothing
*/
void cleanup_module(void)
{
	dev_data_t*	dev;
	int			i;

	os_proc_stop();
	unregister_chrdev((unsigned int) gsc_global.major_number, SIO4_NAME);

	for (i = 0; i < (int) SIZEOF_ARRAY(gsc_global.dev_list); i++)
	{
		dev	= gsc_global.dev_list[i];

		if (dev == NULL)
			continue;

		os_irq_destroy(dev);
		os_sem_destroy(&dev->irq.sem);
		os_spinlock_destroy(&dev->spinlock);
		iounmap((void*) dev->vaddr.plx_regs);
		iounmap((void*) dev->vaddr.gsc_regs);
		kfree(dev);

		gsc_global.dev_list[i]	= NULL;
	}

	if (gsc_global.driver_loaded)
	{
		printk(	"SIO4 driver version %s successfully removed.\n",
				GSC_DRIVER_VERSION);
	}
}



/******************************************************************************
*
*	Function:	_id_device
*
*	Purpose:
*
*		Perform a device identification check to see if this is an SIO4 model
*		device.
*
*	Arguments:
*
*		dev		The device to be accessed.
*
*	Returned:
*
*		> 0		The device is an SIO4 of some type.
*		0		The device is not an SIO4.
*
******************************************************************************/

static int _id_device(dev_data_t* dev)
{
	static const struct
	{
		u16	vid;
		u16	did;
		u16	svid;
		u16	sdid;
	} list[]	=
	{
		// Vid		Did		Svid	Sdid
		{ 0x10B5,	0x9080,	0x10B5,	0x2401 },
		{ 0x10B5,	0x9056,	0x10B5,	0x3198 },
		{ 0, 0, 0, 0 }
	};

	int			found	= 0;
	int			i;
#if DEV_PCI_ID_SHOW
	int			index	= -1;
#endif
	os_pci_t*	pci		= &dev->pci;
	u16			sdid;
	u16			svid;

	svid	= os_reg_pci_rx_u16(dev, 1, PCI_SUBSYSTEM_VENDOR_ID);
	sdid	= os_reg_pci_rx_u16(dev, 1, PCI_SUBSYSTEM_ID);

	for (i = 0; list[i].vid; i++)
	{
		if ((list[i].vid	== pci->pd->vendor)	&&
			(list[i].did	== pci->pd->device)	&&
			(list[i].svid	== svid)		&&
			(list[i].sdid	== sdid))
		{
			found	= 1;
#if DEV_PCI_ID_SHOW
			index	= i;
#endif
			break;
		}
	}

#if DEV_PCI_ID_SHOW
	printk(	"ID: %04lX %04lX %04lX %04lX",
			(long) pci->pd->vendor,
			(long) pci->pd->device,
			(long) svid,
			(long) sdid);

	if (found)
		printk(" <--- DID 0x%04lX, index %d", (long) list[i].did, index);

	printk("\n");
#endif

	return(found);
}



//*****************************************************************************
// RETURNS: 0 = Ok, < 0 = error
s32 init_module(void)
{
	dev_data_t*		dev;
	int				found;
	int				i;
	struct pci_dev*	pci		= NULL;
	int				ret		= 0;
	dev_data_t*		tmp		= NULL;

	memset(&gsc_global, 0, sizeof(gsc_global));
	gsc_global.major_number	= -1;

	if (BITS_PER_LONG == 32)
		gsc_global.ioctl_32bit	= GSC_IOCTL_32BIT_NATIVE;
	else
		gsc_global.ioctl_32bit	= GSC_IOCTL_32BIT_NONE;

	PCI_DEVICE_LOOP(pci)
	{
		if (tmp == NULL)
		{
			tmp	= (dev_data_t*) kmalloc(sizeof(dev_data_t), GFP_KERNEL);

			if (tmp == NULL)
				break;
		}

		memset(tmp, 0, sizeof(dev_data_t));
		tmp->pci.pd	= pci;
		found		= _id_device(tmp);

		if (!found)
			continue;

		ret	= PCI_ENABLE_DEVICE(pci);

		if (ret)
			continue;

		pci_set_master(pci);

		dev	= _dev_add();

		if (dev == NULL)
			continue;

		i	= dev->board_index;
		memset(dev, 0, sizeof(dev[0]));
		dev->board_index	= i;
		dev->pci.pd	= pci;
#ifdef __SMP__
		spin_lock_init(&dev->irq.spinlock);
#endif

		os_sem_create(&dev->irq.sem);
		os_irq_create(dev);

		// Map in access to the PLX and GSC registers.
		_map_bar0(dev);
		_map_bar2(dev);

		if ((dev->vaddr.plx_regs == 0) || (dev->vaddr.gsc_regs == 0))
		{
			ret	= -EPERM;
			break;
		}

		ret	= dev_device_create(dev);
	}

	if (tmp)
	{
		kfree(tmp);
		//tmp	= NULL;
	}

	// check if any SIO4 device is present
	if ((ret == 0) && (gsc_global.dev_qty == 0))
		ret	= -ENODEV;

	if (ret == 0)
	{
		// Register the device.
		gsc_global.fops.open	= os_open;
		gsc_global.fops.release	= os_close;
		gsc_global.fops.read	= os_read;
		gsc_global.fops.write	= os_write;
		IOCTL_SET_BKL(&gsc_global.fops, os_ioctl_bkl);
		IOCTL_SET_COMPAT(&gsc_global.fops, os_ioctl_compat);
		IOCTL_SET_UNLOCKED(&gsc_global.fops, os_ioctl_unlocked);
		gsc_global.fops.fasync	= gsc_fasync;
		SET_MODULE_OWNER(&gsc_global.fops);
		gsc_global.major_number	= register_chrdev (0, SIO4_NAME, &gsc_global.fops);

		if (gsc_global.major_number < 0)
			ret	= -EIO;
	}

	if (ret == 0)
		ret	= os_proc_start();

	if (ret)
	{
		cleanup_module();
	}
	else
	{
		gsc_global.driver_loaded	= 1;
		printk(	"SIO4 driver version %s successfully inserted.\n",
				GSC_DRIVER_VERSION);
	}

	return(ret);
}


