// $URL: http://subversion:8080/svn/gsc/trunk/drivers/LINUX/SIO4%20and%20SIO8/SIO4_Linux_1.x.x_GSC_DN/driver/irq.c $
// $Rev: 53095 $
// $Date: 2023-06-13 10:42:41 -0500 (Tue, 13 Jun 2023) $

// SIO4: Device Driver: source file

#include "main.h"



// macros *********************************************************************

#define	INTCSR_PCI_INT_ENABLE				0x00000100
#define	INTCSR_LOCAL_IRQ_ENABLE				0x00000800
#define	INTCSR_LOCAL_IRQ_ACTIVE				0x00008000
#define	INTCSR_DMA_0_IRQ_ACTIVE				0x00200000
#define	INTCSR_DMA_0_IRQ_ENABLE				0x00040000
#define	INTCSR_DMA_1_IRQ_ACTIVE				0x00400000
#define	INTCSR_DMA_1_IRQ_ENABLE				0x00080000



/******************************************************************************
*
*	Function:	gsc_irq_access_lock
*
*	Purpose:
*
*		Apply a locking mechanism to prevent simultaneous access to the
*		device's IRQ substructure.
*
*	Arguments:
*
*		dev		The data for the device of interest.
*
*	Returned:
*
*		None.
*
******************************************************************************/

void gsc_irq_access_lock(dev_data_t* dev)
{
	os_spinlock_lock(&dev->spinlock);
}



/******************************************************************************
*
*	Function:	gsc_irq_access_unlock
*
*	Purpose:
*
*		Remove the locking mechanism that prevented simultaneous access to the
*		device's IRQ substructure.
*
*	Arguments:
*
*		dev		The data for the device of interest.
*
*	Returned:
*
*		None.
*
******************************************************************************/

void gsc_irq_access_unlock(dev_data_t* dev)
{
	os_spinlock_unlock(&dev->spinlock);
}



/******************************************************************************
*
*	Function:	int_notify_ioctl
*
*	Purpose:
*
*		Implement the Interrupt Notify IOCTL service.
*
*	Arguments:
*
*		dev		The structure for the device to initialize.
*
*		arg		The argument passed by the application.
*
*	Returned:
*
*		0		All went well.
*		< 0		An appropriate error status.
*
******************************************************************************/

s32 int_notify_ioctl(chan_data_t* chan, void* arg)
{
	dev_data_t*	dev		= chan->dev;
	int			ret		= 0;
	u32			u		= (u32) (unsigned long) arg;

	for (;;)	// A convenience loop.
	{
		gsc_irq_access_lock(dev);

		if (chan->irq.driver & u)
		{
			// The interrupt is in use by the driver.
			ret	= -EBUSY;
			gsc_irq_access_unlock(dev);
			break;
		}

		// The user can have them.
		// Record which interrupts are of interest.
		chan->irq.notify	= 0xFF & u;

		// Clear the current status.
		chan->irq.status	= 0;

		// Adjust the Interrupt Control Register.
		u	= (u << 12) | u;
		u	= u << chan->cache.gsc_irq_shift;

		os_reg_mem_mx_u32(
			NULL,
			dev->vaddr.gsc_icr_32,
			u,
			chan->cache.gsc_irq_mask & dev->cache.irq_mask);

		gsc_irq_access_unlock(dev);
		break;
	}

	return(ret);
}



/******************************************************************************
*
*	Function:	int_status_ioctl
*
*	Purpose:
*
*		Implement the Interrupt Status IOCTL service.
*
*	Arguments:
*
*		dev		The structure for the device to initialize.
*
*		arg		The argument passed by the application.
*
*	Returned:
*
*		0		All went well.
*		< 0		An appropriate error status.
*
******************************************************************************/

s32 int_status_ioctl(chan_data_t* chan, void* arg)
{
	SIO4_INTERRUPT_STATUS*	sts		= arg;

	gsc_irq_access_lock(chan->dev);
	sts->u8SIO4Status	= chan->irq.status;
	chan->irq.status	= 0;
	gsc_irq_access_unlock(chan->dev);
	return(0);
}



/******************************************************************************
*
*	Function:	dev_irq_close
*
*	Purpose:
*
*		Perform needed processing associated with closing a channel.
*
*	Arguments:
*
*		dev		The data for the device whose channel is being closed.
*
*	Returned:
*
*		None.
*
******************************************************************************/

void dev_irq_close(dev_data_t* dev)
{
	int	ret;

	ret	= os_sem_lock(&dev->irq.sem);

	if (ret)
	{
	}
	else
	{
		if (dev->irq.users == 1)
			os_irq_close(dev);

		dev->irq.users--;
		os_sem_unlock(&dev->irq.sem);
	}
}



/******************************************************************************
*
*	Function:	gsc_irq_isr_common
*
*	Purpose:
*
*		Service interrupts.
*
*	Arguments:
*
*		dev_id	The private data we've associated with the device.
*
*		flags	Not used by this driver.
*
*	Returned:
*
*		0		The interrupt was NOT ours.
*		1		The interrupt was ours.
*
******************************************************************************/

int gsc_irq_isr_common(void* dev_id, u32 flags)
{
	chan_data_t*	chan;
	dev_data_t*		dev		= (dev_data_t*) dev_id;
	dev_dma_t*		dma;
	int				handled	= 1;
	int				i;
	u32				icr;
	u32				intcsr;
	u32				bits;
	u32				ch_bits;
	u32				isr;
	u32				sts;

	gsc_irq_access_lock(dev);

	for (;;)	// A convenience loop.
	{
		intcsr	= os_reg_mem_rx_u32(NULL, dev->vaddr.plx_intcsr_32);

		if ((intcsr & INTCSR_PCI_INT_ENABLE) == 0)
		{
			// This is not our interrupt.
			handled	= 0;
			break;
		}

		if ((intcsr & INTCSR_DMA_0_IRQ_ENABLE) &&
			(intcsr & INTCSR_DMA_0_IRQ_ACTIVE))
		{
			// A DMA 0 Done interrupt.
			dma	= &dev->dma.channel[0];
			os_reg_mem_tx_u8(NULL, dma->vaddr.csr_8, DMA_CSR_CLEAR);	// Clear
			intcsr	&= ~dma->int_enable;
			os_reg_mem_tx_u32(NULL, dev->vaddr.plx_intcsr_32, intcsr);	// Disable
			EVENT_RESUME_IRQ(dma->queue, dma->condition);				//Resume
			break;
		}

		if ((intcsr & INTCSR_DMA_1_IRQ_ENABLE) &&
			(intcsr & INTCSR_DMA_1_IRQ_ACTIVE))
		{
			// A DMA 1 Done interrupt.
			dma	= &dev->dma.channel[1];
			os_reg_mem_tx_u8(NULL, dma->vaddr.csr_8, DMA_CSR_CLEAR);	// Clear
			intcsr	&= ~dma->int_enable;
			os_reg_mem_tx_u32(NULL, dev->vaddr.plx_intcsr_32, intcsr);	// Disable
			EVENT_RESUME_IRQ(dma->queue, dma->condition);				//Resume
			break;
		}

		if ((intcsr & INTCSR_LOCAL_IRQ_ENABLE) &&
			(intcsr & INTCSR_LOCAL_IRQ_ACTIVE))
		{
			// A firmware interrupt is active.
			icr	= os_reg_mem_rx_u32(NULL, dev->vaddr.gsc_icr_32);
			isr	= os_reg_mem_rx_u32(NULL, dev->vaddr.gsc_isr_32);

			for (i = 0; i < 4; i++)
			{
				chan	= &dev->channel[i];
				bits	= icr & isr & chan->cache.gsc_irq_mask;

				if (bits == 0)
					continue;

				// Disable (and service) the interrupt.
				os_reg_mem_mx_u32(
					NULL,
					dev->vaddr.gsc_icr_32,
					0,
					bits & dev->cache.irq_mask);

				// Add to the Interrupt Status record.
				sts	= bits >> chan->cache.gsc_irq_shift;
				ch_bits	= sts & 0xF000F;
				sts	= (sts | (sts >> 12)) & 0xFF;
				chan->irq.status	|= (u8) sts;

				if (ch_bits & SIO4_IRQ_RFAF)
				{
					// Resume the blocked Rx task.
					EVENT_RESUME_IRQ(
							&chan->rx.queue,
							chan->rx.condition);
				}

				if (sts & SIO4_IRQ_TFAE)
				{
					// Resume the blocked Tx task.
					EVENT_RESUME_IRQ(
							&chan->tx.queue,
							chan->tx.condition);
				}

				if ((chan->irq.fasync) &&
					(sts & chan->irq.notify))
				{
					KILL_FASYNC(chan->irq.fasync,
								SIGIO,
								POLL_IN);
				}
			}

			break;
		}

		handled	= 0;
		break;
	}

	gsc_irq_access_unlock(dev);
	return(handled);
}



/******************************************************************************
*
*	Function:	dev_irq_open
*
*	Purpose:
*
*		Perform needed processing associated with opening a channel.
*
*	Arguments:
*
*		dev		The data for the device whose channel is being opened.
*
*	Returned:
*
*		0		All went well.
*		< 0		An appropriate error status.
*
******************************************************************************/

int dev_irq_open(dev_data_t* dev)
{
	int	ret;

	ret	= os_sem_lock(&dev->irq.sem);

	if (ret)
	{
	}
	else
	{
		if (dev->irq.users == 0)
			ret	= os_irq_open(dev);

		dev->irq.users++;
		os_sem_unlock(&dev->irq.sem);
	}

	return(ret);
}



/******************************************************************************
*
*	Function:	irq_intcsr_mod
*
*	Purpose:
*
*		Perform safe modifications to the PLX INTCSR register.
*
*	Arguments:
*
*		dev		The structure for the device to initialize.
*
*		arg		The argument passed by the application.
*
*	Returned:
*
*		0		All went well.
*		< 0		An appropriate error status.
*
******************************************************************************/

int irq_intcsr_mod(dev_data_t* dev, u32 value, u32 mask)
{
	os_reg_mem_mx_u32(
		dev,
		dev->vaddr.plx_intcsr_32,
		value,
		mask);
	return(0);
}


