// $URL: http://subversion:8080/svn/gsc/trunk/drivers/LINUX/SIO4%20and%20SIO8/SIO4_Linux_1.x.x_GSC_DN/driver/ioctl.c $
// $Rev: 48537 $
// $Date: 2020-12-09 13:46:18 -0600 (Wed, 09 Dec 2020) $

// SIO4: Device Driver: source file

#include "main.h"



/******************************************************************************
*
*	Function:	_ioctl_service
*
*	Purpose:
*
*		Do common IOCTL service processing and call the given service function
*		as appropriet.
*
*	Arguments:
*
*		chan	The channel data structure.
*
*		cmd		The IOCTL code given. We decode this so we know what to do.
*
*		arg		The argument passed with the IOCTL call.
*
*		func	The function implementing the IOCTL specific functionality. We
*				call this as appropriate.
*
*	Returned:
*
*		0		All went well.
*		< 0		An appropriate error status.
*
******************************************************************************/

static s32 _ioctl_service(
	chan_data_t*	chan,
	int				cmd,
	void*			arg,
	s32				(*func)(chan_data_t* chan, void* arg))
{
	char			buf[512];
	unsigned long	dir		= _IOC_DIR(cmd);
	void*			f_arg	= arg;
	s32				ret;
	unsigned long	size	= _IOC_SIZE(cmd);
	unsigned long	type	= _IOC_TYPE(cmd);
	unsigned long	ul;

	for (;;)	// A convenience loop.
	{
		memset(buf, 0, sizeof(buf));

		if (type != 'S')
		{
			// This isn't our IOCTL code.
			ret	= -EINVAL;
			break;
		}

		if (size > sizeof(buf))
		{
			// Our transfer buffer is too small.
			ret	= -EINVAL;
			break;
		}

		if (dir & _IOC_WRITE)
		{
			f_arg	= buf;
			ul		= MEM_ACCESS_OK(VERIFY_READ, arg, size);

			if (ul == 0)
			{
				// We can't read the user's memory.
				ret	= -EFAULT;
				break;
			}

			ul	= copy_from_user(buf, arg, size);

			if (ul)
			{
				// We can't read the user's memory.
				ret	= -EFAULT;
				break;
			}
		}

		if (dir & _IOC_READ)
		{
			f_arg	= buf;
			ul		= MEM_ACCESS_OK(VERIFY_WRITE, arg, size);

			if (ul == 0)
			{
				// Can't write to user's memory.
				ret	= -EFAULT;
				break;
			}
		}

		ret	= (func)(chan, f_arg);

		if (ret)
		{
			// The request failed in some way.
			break;
		}

		if (dir & _IOC_READ)
		{
			ul	= copy_to_user(arg, buf, size);

			if (ul)
			{
				// We can't write to user's memory.
				ret	= -EFAULT;
				break;
			}
		}

		// All went well.
		ret	= 0;
		break;
	}

	return(ret);
}



// ****************************************************************************
static s32 _common_ioctl(
	struct inode*	inode,
	struct file*	filp,
	unsigned int	cmd,
	void*			arg)
{
	#define	SERVICE(f)	_ioctl_service(chan, cmd, arg, (f))

	chan_data_t*	chan;
	int				ret;

	ret	= dev_lock_inode(inode, 0, &chan);

	if (ret)
		return(ret);

	switch ((int) cmd)
	{
		case SIO4_BOARD_JUMPERS:

			ret	= SERVICE((void*) board_jumpers_ioctl);
			break;

		case SIO4_CABLE_CONFIG:

			ret	= SERVICE(cable_config_ioctl);
			break;

		case SIO4_CTS_CABLE_CONFIG:

			ret	= SERVICE(cts_cable_config_ioctl);
			break;

		case SIO4_DCD_CABLE_CONFIG:

			ret	= SERVICE(dcd_cable_config_ioctl);
			break;

		case SIO4_FEATURE_TEST:

			ret	= SERVICE(feature_test_ioctl);
			break;

		case SIO4_FW_TYPE_CONFIG:

			ret	= SERVICE(fw_type_config_ioctl);
			break;

		case SIO4_INIT_BOARD:

			ret	= SERVICE(board_init_ioctl);
			break;

		case SIO4_INT_NOTIFY:

			ret	= SERVICE(int_notify_ioctl);
			break;

		case SIO4_MOD_REGISTER:

			ret	= SERVICE(reg_mod_ioctl);
			break;

		case SIO4_MP_CONFIG:

			ret	= SERVICE(mp_config_ioctl);
			break;

		case SIO4_MP_INFO:

			ret	= SERVICE(mp_info_ioctl);
			break;

		case SIO4_MP_INIT:

			ret	= SERVICE(mp_init_ioctl);
			break;

		case SIO4_MP_RESET:

			ret	= SERVICE(mp_reset_ioctl);
			break;

		case SIO4_MP_TEST:

			ret	= SERVICE(mp_test_ioctl);
			break;

		case SIO4_OSC_INFO:

			ret	= SERVICE(osc_info_ioctl);
			break;

		case SIO4_OSC_INIT:

			ret	= SERVICE(osc_init_ioctl);
			break;

		case SIO4_OSC_MEASURE:

			ret	= SERVICE(osc_measure_ioctl);
			break;

		case SIO4_OSC_PROGRAM:

			ret	= SERVICE(osc_program_ioctl);
			break;

		case SIO4_OSC_REFERENCE:

			ret	= SERVICE(osc_reference_ioctl);
			break;

		case SIO4_OSC_RESET:

			ret	= SERVICE(osc_reset_ioctl);
			break;

		case SIO4_OSC_TEST:

			ret	= SERVICE(osc_test_ioctl);
			break;

		case SIO4_READ_INT_STATUS:

			ret	= SERVICE(int_status_ioctl);
			break;

		case SIO4_READ_REGISTER:

			ret	= SERVICE(reg_read_ioctl);
			break;

		case SIO4_READ_REGISTER_RAW:

			ret	= SERVICE(reg_read_raw_ioctl);
			break;

		case SIO4_RESET_DEVICE:

			ret	= SERVICE(device_reset_ioctl);
			break;

		case SIO4_RESET_CHANNEL:

			ret	= SERVICE(channel_reset_ioctl);
			break;

		case SIO4_RESET_USC:

			ret	= SERVICE(usc_reset_ioctl);
			break;

		case SIO4_RESET_ZILOG_CHIP:

			ret	= SERVICE(zilog_reset_ioctl);
			break;

		case SIO4_RX_CABLE_CONFIG:

			ret	= SERVICE(rx_cable_config_ioctl);
			break;

		case SIO4_RX_FIFO_AE_CONFIG:

			ret	= SERVICE(rx_fifo_ae_config_ioctl);
			break;

		case SIO4_RX_FIFO_AF_CONFIG:

			ret	= SERVICE(rx_fifo_af_config_ioctl);
			break;

		case SIO4_RX_FIFO_COUNT:

			ret	= SERVICE(rx_fifo_count_ioctl);
			break;

		case SIO4_RX_FIFO_FULL_CFG_CHAN:

			ret	= SERVICE(rx_fifo_full_cfg_chan_ioctl);
			break;

		case SIO4_RX_FIFO_FULL_CFG_GLB:

			ret	= SERVICE(rx_fifo_full_cfg_glb_ioctl);
			break;

		case SIO4_RX_FIFO_SIZE:

			ret	= SERVICE(rx_fifo_size_ioctl);
			break;

		case SIO4_RX_FIFO_STATUS:

			ret	= SERVICE(rx_fifo_status_ioctl);
			break;

		case SIO4_RX_IO_ABORT:

			ret	= SERVICE(rx_io_abort_ioctl);
			break;

		case SIO4_RX_IO_MODE_CONFIG:

			ret	= SERVICE(rx_io_mode_config_ioctl);
			break;

		case SIO4_RXC_USC_CONFIG:

			ret	= SERVICE(rxc_usc_config_ioctl);
			break;

		case SIO4_TX_CABLE_CLOCK_CONFIG:

			ret	= SERVICE(tx_cable_clock_config_ioctl);
			break;

		case SIO4_TX_CABLE_CONFIG:

			ret	= SERVICE(tx_cable_config_ioctl);
			break;

		case SIO4_TX_CABLE_DATA_CONFIG:

			ret	= SERVICE(tx_cable_data_config_ioctl);
			break;

		case SIO4_TX_FIFO_AE_CONFIG:

			ret	= SERVICE(tx_fifo_ae_config_ioctl);
			break;

		case SIO4_TX_FIFO_AF_CONFIG:

			ret	= SERVICE(tx_fifo_af_config_ioctl);
			break;

		case SIO4_TX_FIFO_COUNT:

			ret	= SERVICE(tx_fifo_count_ioctl);
			break;

		case SIO4_TX_FIFO_SIZE:

			ret	= SERVICE(tx_fifo_size_ioctl);
			break;

		case SIO4_TX_FIFO_STATUS:

			ret	= SERVICE(tx_fifo_status_ioctl);
			break;

		case SIO4_TX_IO_ABORT:

			ret	= SERVICE(tx_io_abort_ioctl);
			break;

		case SIO4_TX_IO_MODE_CONFIG:

			ret	= SERVICE(tx_io_mode_config_ioctl);
			break;

		case SIO4_TXC_USC_CONFIG:

			ret	= SERVICE(txc_usc_config_ioctl);
			break;

		case SIO4_WRITE_REGISTER:

			ret	= SERVICE(reg_write_ioctl);
			break;

		case SIO4_NO_FUNCTION:

			ret = 0;
			break;

		case SIO4_INIT_CHANNEL:

			{
				SIO4_INIT_CHAN	st;

				if (copy_from_user(&st, arg, sizeof(st)))
				{
					ret	= -EFAULT;
					break;
				}

				ret = SIO4InitChannel(chan, &st);
				break;
			}

		case SIO4_SEND_CHANNEL_COMMAND:

			ret = SIO4SendChanCmd(chan, arg);
			break;

		case SIO4_RESET_FIFO:

			ret = SIO4ResetFifo(chan, arg);
			break;

		case SIO4_SET_SYNC_BYTE:

			ret = SIO4SetSyncByte(chan, arg);
			break;

		case SIO4_SET_READ_TIMEOUT:

			ret = SIO4SetReadTimeout(chan, arg);
			break;

		case SIO4_SET_WRITE_TIMEOUT:

			ret = SIO4SetWriteTimeout(chan, arg);
			break;

		case SIO4_SET_XMT_ASYNC_PROT:

			{
				XMT_ASYNC_PROTOCOL	st;

				if (copy_from_user(&st, arg, sizeof(st)))
				{
					ret	= -EFAULT;
					break;
				}

				ret = SIO4SetTransmitASYNCProtocol(chan, &st);
				break;
			}

		case SIO4_SET_RCV_ASYNC_PROT:

			{
				RCV_ASYNC_PROTOCOL	st;

				if (copy_from_user(&st, arg, sizeof(st)))
				{
					ret	= -EFAULT;
					break;
				}

				ret = SIO4SetReceiveASYNCProtocol(chan, &st);
				break;
			}

		case SIO4_SET_XMT_ISOCHR_PROT:

			{
				XMT_ISOCHR_PROTOCOL	st;

				if (copy_from_user(&st, arg, sizeof(st)))
				{
					ret	= -EFAULT;
					break;
				}

				ret = SIO4SetTransmitISOCHRProtocol(chan, &st);
				break;
			}

		case SIO4_SET_RCV_ISOCHR_PROT:

			ret = SIO4SetReceiveISOCHRProtocol(chan, arg);
			break;

		case SIO4_SET_XMT_HDLC_PROT:

			{
				XMT_HDLC_PROTOCOL	st;

				if (copy_from_user(&st, arg, sizeof(st)))
				{
					ret	= -EFAULT;
					break;
				}

				ret = SIO4SetTransmitHDLCProtocol(chan, &st);
				break;
			}

		case SIO4_SET_RCV_HDLC_PROT:

			{
				RCV_HDLC_PROTOCOL	st;

				if (copy_from_user(&st, arg, sizeof(st)))
				{
					ret	= -EFAULT;
					break;
				}

				ret = SIO4SetReceiveHDLCProtocol(chan, &st);
				break;
			}

		case SIO4_SET_XMT_HDLC_SDLC_LOOP_PROT:

			{
				XMT_HDLC_SDLC_LOOP_PROTOCOL	st;

				if (copy_from_user(&st, arg, sizeof(st)))
				{
					ret	= -EFAULT;
					break;
				}

				ret = SIO4SetTransmitHDLCSDLCLoopProtocol(chan, &st);
				break;
			}

		case SIO4_SET_TX_CLOCK_SOURCE:

			ret = SIO4SetTxClockSource(chan, arg);
			break;

		case SIO4_SET_RX_CLOCK_SOURCE:

			ret = SIO4SetRxClockSource(chan, arg);
			break;

		case SIO4_SET_BRG0_SOURCE:

			ret = SIO4SetBRG0Source(chan, arg);
			break;

		case SIO4_SET_BRG1_SOURCE:

			ret = SIO4SetBRG1Source(chan, arg);
			break;

		case SIO4_SET_CTR0_SOURCE:

			ret = SIO4SetCTR0Source(chan, arg);
			break;

		case SIO4_SET_CTR1_SOURCE:

			ret = SIO4SetCTR1Source(chan, arg);
			break;

		case SIO4_SET_DPLL_SOURCE:

			ret = SIO4SetDPLLSource(chan, arg);
			break;

		case SIO4_SET_BRG0_MODE:

			ret = SIO4SetBRG0Mode(chan, arg);
			break;

		case SIO4_SET_BRG1_MODE:

			ret = SIO4SetBRG1Mode(chan, arg);
			break;

		case SIO4_ENABLE_BRG0:

			ret = SIO4EnableBRG0(chan, arg);
			break;

		case SIO4_ENABLE_BRG1:

			ret = SIO4EnableBRG1(chan, arg);
			break;

		case SIO4_SELECT_DPLL_RESYNC:

			ret = SIO4SelectDPLLResync(chan, arg);
			break;

		case SIO4_SET_DPLL_MODE:

			ret = SIO4SetDPLLMode(chan, arg);
			break;

		case SIO4_SET_DPLL_DIVISOR:

			ret = SIO4SetDPLLDivisor(chan, arg);
			break;

		case SIO4_CLEAR_DPLL_STATUS:

			ret = SIO4ClearDPLLStatus(chan, arg);
			break;

		case SIO4_SET_USC_DMA_OPTIONS:

			{
				USC_DMA_OPTIONS	st;

				if (copy_from_user(&st, arg, sizeof(st)))
				{
					ret	= -EFAULT;
					break;
				}

				ret = SIO4SetUSCDMAOptions(chan ,&st);
				break;
			}

		case SIO4_GET_DRIVER_INFO:

			{
				SIO4_DRIVER_INFO	info;

				memset(&info, 0, sizeof(info));
				info.u8Built[0]	= 0;	// no longer available
				strcpy((char*) info.u8VersionNumber, GSC_DRIVER_VERSION);

				if (copy_to_user(arg, &info, sizeof(SIO4_DRIVER_INFO)))
				{
					ret	= -EFAULT;
				}
				else
				{
					ret = 0;
				}

				break;
			}

		default:

			ret	= -EINVAL;
			break;
	}

	os_sem_unlock(&chan->sem);
	return(ret);
}



/******************************************************************************
*
*	Function:	os_ioctl
*
*	Purpose:
*
*		Implement the ioctl() procedure.
*
*	Arguments:
*
*		inode	This is the device inode structure.
*
*		fp		This is the file structure.
*
*		cmd		The desired IOCTL service.
*
*		arg		The optional service specific argument.
*
*	Returned:
*
*		0		All went well.
*		< 0		There was a problem.
*
******************************************************************************/

#if GSC_HAVE_IOCTL_BKL
int os_ioctl_bkl(
	struct inode*	inode,
	struct file*	fp,
	unsigned int	cmd,
	unsigned long	arg)
{
	int	ret;

	ret	= _common_ioctl(inode, fp, cmd, (void*) arg);
	return(ret);
}
#endif



/******************************************************************************
*
*	Function:	os_ioctl_compat
*
*	Purpose:
*
*		Implement the compat_ioctl() method.
*
*	Arguments:
*
*		fp		A pointer to the file structure.
*
*		cmd		The service being requested.
*
*		arg		The service specific argument.
*
*	Returned:
*
*		0		All went well.
*		< 0		There was an error.
*
******************************************************************************/

#if HAVE_COMPAT_IOCTL
long os_ioctl_compat(struct file* fp, unsigned int cmd, unsigned long arg)
{
	int	ret;

	// No additional locking needed as we use a per device lock.
	ret	= _common_ioctl(fp->f_dentry->d_inode, fp, cmd, (void*) arg);
	return(ret);
}
#endif



/******************************************************************************
*
*	Function:	os_ioctl_unlocked
*
*	Purpose:
*
*		Implement the unlocked_ioctl() method.
*
*	Arguments:
*
*		fp		A pointer to the file structure.
*
*		cmd		The service being requested.
*
*		arg		The service specific argument.
*
*	Returned:
*
*		0		All went well.
*		< 0		There was an error.
*
******************************************************************************/

#if HAVE_UNLOCKED_IOCTL
long os_ioctl_unlocked(struct file* fp, unsigned int cmd, unsigned long arg)
{
	int	ret;

	// As of 8/13/2008 all data types are sized identically for 32-bit
	// and 64-bit environments.
	// No additional locking needed as we use a per device lock.
	ret	= _common_ioctl(fp->f_dentry->d_inode, fp, cmd, (void*) arg);
	return(ret);
}
#endif



