// $URL: http://subversion:8080/svn/gsc/trunk/drivers/LINUX/SIO4%20and%20SIO8/SIO4_Linux_1.x.x_GSC_DN/driver/io_pio.c $
// $Rev: 53095 $
// $Date: 2023-06-13 10:42:41 -0500 (Tue, 13 Jun 2023) $

// SIO4: Device Driver: source file

#include "main.h"



// macros *********************************************************************

#if defined(DEV_SUPPORTS_READ)

#ifndef	GSC_READ_PIO_PER_1MS
#define	GSC_READ_PIO_PER_1MS				250
#endif

#if (GSC_READ_PIO_PER_1MS > 1000)
#undef	GSC_READ_PIO_PER_1MS
#define	GSC_READ_PIO_PER_1MS				1000
#endif

#endif

#if defined(DEV_SUPPORTS_WRITE)

#ifndef	GSC_WRITE_PIO_PER_1MS
#define	GSC_WRITE_PIO_PER_1MS				250
#endif

#if (GSC_WRITE_PIO_PER_1MS > 1000)
#undef	GSC_WRITE_PIO_PER_1MS
#define	GSC_WRITE_PIO_PER_1MS				1000
#endif

#endif



/******************************************************************************
*
*	Function:	_rx_avail_csr_fsr_rar
*
*	Purpose:
*
*		See how much space is available using the Control/Status Register, the
*		FIFO Size Register and the Rx Almost Register.
*
*	Arguments:
*
*		chan	The channel data structure.
*
*	Returned:
*
*		>= 0	This is the number of bytes available.
*
******************************************************************************/

static ssize_t _rx_avail_csr_fsr_rar(chan_data_t* chan)
{
	u32			ae;
	u32			af;
	u32			csr;
	dev_data_t*	dev	= chan->dev;
	ssize_t		qty;
	u32			rar;

	for (;;)	// A convenience loop.
	{
		csr	= os_reg_mem_rx_u32(dev, chan->vaddr.gsc_csr_32);

		if ((csr & SIO4_GSC_CSR_RFF) == 0)
		{
			// The Rx FIFO is Full.
			qty	= chan->cache.fifo_size_rx;
			break;
		}

		if ((csr & SIO4_GSC_CSR_RFAF) == 0)
		{
			// The Rx FIFO is Almost Full.
			rar	= os_reg_mem_rx_u32(dev, chan->vaddr.gsc_rar_32);
			af	= rar >> 16;
			qty	= chan->cache.fifo_size_rx - af;
			break;
		}

		if (csr & SIO4_GSC_CSR_RFAE)
		{
			// The Rx FIFO is more than Almost Empty.
			rar	= os_reg_mem_rx_u32(dev, chan->vaddr.gsc_rar_32);
			ae	= rar & 0xFFFF;
			qty	= ae;
			break;
		}

		if (csr & SIO4_GSC_CSR_RFE)
		{
			// The Rx FIFO is above Empty.
			qty	= 1;
			break;
		}

		// The Rx FIFO is empty.
		qty	= 0;
		break;
	}

	return(qty);
}



/******************************************************************************
*
*	Function:	_rx_avail_csr_rar
*
*	Purpose:
*
*		See how much space is available using the Control/Status Register and
*		the Rx Almost Register.
*
*	Arguments:
*
*		chan	The channel data structure.
*
*	Returned:
*
*		>= 0	This is the number of bytes available.
*
******************************************************************************/

static ssize_t _rx_avail_csr_rar(chan_data_t* chan)
{
	u32			ae;
	u32			csr;
	dev_data_t*	dev	= chan->dev;
	ssize_t		qty;
	u32			rar;

	for (;;)	// A convenience loop.
	{
		csr	= os_reg_mem_rx_u32(dev, chan->vaddr.gsc_csr_32);

		if (csr & SIO4_GSC_CSR_RFAE)
		{
			// The Rx FIFO is more than Almost Empty.
			rar	= os_reg_mem_rx_u32(dev, chan->vaddr.gsc_rar_32);
			ae	= rar & 0xFFFF;
			qty	= ae;
			break;
		}

		if (csr & SIO4_GSC_CSR_RFE)
		{
			// The Rx FIFO is more than Empty.
			qty	= 1;
			break;
		}

		// The Rx FIFO is empty.
		qty	= 0;
		break;
	}

	return(qty);
}



/******************************************************************************
*
*	Function:	_rx_avail_fcr
*
*	Purpose:
*
*		See how much data is available using the FIFO Count Register.
*
*	Arguments:
*
*		chan	The channel data structure.
*
*	Returned:
*
*		>= 0	This is the number of bytes available.
*
******************************************************************************/

static ssize_t _rx_avail_fcr(chan_data_t* chan)
{
	dev_data_t*	dev	= chan->dev;
	u32			fcr;
	ssize_t		qty;

	fcr	= os_reg_mem_rx_u32(dev, chan->vaddr.gsc_fcr_32);
	qty	= fcr >> 16;
	return(qty);
}



/******************************************************************************
*
*	Function:	_tx_avail_csr_fsr_tar
*
*	Purpose:
*
*		See how much space is available using the Control/Status Register, the
*		FIFO Size Register and the Tx Almost Register.
*
*	Arguments:
*
*		chan	The channel data structure.
*
*	Returned:
*
*		>= 0	This is the number of spaces available.
*
******************************************************************************/

static ssize_t _tx_avail_csr_fsr_tar(chan_data_t* chan)
{
	u32			ae;
	u32			af;
	u32			csr;
	dev_data_t*	dev	= chan->dev;
	ssize_t		qty;
	u32			tar;

	for (;;)	// A convenience loop.
	{
		csr	= os_reg_mem_rx_u32(dev, chan->vaddr.gsc_csr_32);

		if ((csr & SIO4_GSC_CSR_TFE) == 0)
		{
			// The Tx FIFO is Empty.
			qty	= chan->cache.fifo_size_tx;
			break;
		}

		if ((csr & SIO4_GSC_CSR_TFAE) == 0)
		{
			// The Tx FIFO is Almost Empty.
			tar	= os_reg_mem_rx_u32(dev, chan->vaddr.gsc_tar_32);
			ae	= tar & 0xFFFF;
			qty	= chan->cache.fifo_size_tx - ae;
			break;
		}

		if (csr & SIO4_GSC_CSR_TFAF)
		{
			// The Tx FIFO is less than Almost Full.
			tar	= os_reg_mem_rx_u32(dev, chan->vaddr.gsc_tar_32);
			af	= tar >> 16;
			qty	= af;
			break;
		}

		if (csr & SIO4_GSC_CSR_TFF)
		{
			// The Tx FIFO is less than Full.
			qty	= 1;
			break;
		}

		// The Tx FIFO is Full.
		qty	= 0;
		break;
	}

	return(qty);
}



/******************************************************************************
*
*	Function:	_tx_avail_csr_tar
*
*	Purpose:
*
*		See how much space is available using the Control/Status Register and
*		the Tx Almost Register.
*
*	Arguments:
*
*		chan	The channel data structure.
*
*	Returned:
*
*		>= 0	This is the number of bytes to write.
*
******************************************************************************/

static ssize_t _tx_avail_csr_tar(chan_data_t* chan)
{
	u32			af;
	u32			csr;
	dev_data_t*	dev	= chan->dev;
	ssize_t		qty;
	u32			tar;

	for (;;)	// A convenience loop.
	{
		csr	= os_reg_mem_rx_u32(dev, chan->vaddr.gsc_csr_32);

		if (csr & SIO4_GSC_CSR_TFAF)
		{
			// The Tx FIFO is less than Almost Full.
			tar	= os_reg_mem_rx_u32(dev, chan->vaddr.gsc_tar_32);
			af	= tar >> 16;
			qty	= af;
			break;
		}

		if (csr & SIO4_GSC_CSR_TFF)
		{
			// The Tx FIFO is less than Full.
			qty	= 1;
			break;
		}

		// The Tx FIFO is Full.
		qty	= 0;
		break;
	}

	return(qty);
}



/******************************************************************************
*
*	Function:	_tx_avail_fcr_fsr
*
*	Purpose:
*
*		See how much space is available using the FIFO Count Register and the
*		FIFO Size Register.
*
*	Arguments:
*
*		chan	The channel data structure.
*
*	Returned:
*
*		>= 0	This is the number of bytes to write.
*
******************************************************************************/

static ssize_t _tx_avail_fcr_fsr(chan_data_t* chan)
{
	u32			count;
	dev_data_t*	dev	= chan->dev;
	u32			fcr;
	ssize_t		qty;

	fcr	= os_reg_mem_rx_u32(dev, chan->vaddr.gsc_fcr_32);
	count	= fcr & 0xFFFF;
	qty	= chan->cache.fifo_size_tx - count;
	return(qty);
}



/******************************************************************************
*
*	Function:	pio_read_available
*
*	Purpose:
*
*		Determine the number of bytes that can be read for a PIO mode read
*		operation.
*
*	Arguments:
*
*		dev		The device data structure.
*
*		chan	The channel data structure.
*
*		bytes	The number of bytes desired.
*
*	Returned:
*
*		>= 0	This is the number of bytes to read.
*
******************************************************************************/

ssize_t pio_read_available(dev_data_t* dev, chan_data_t* chan, ssize_t bytes)
{
	ssize_t	qty;

	if (dev->cache.reg_fcr)
		qty	= _rx_avail_fcr(chan);
	else if (dev->cache.reg_fsr)
		qty	= _rx_avail_csr_fsr_rar(chan);
	else
		qty	= _rx_avail_csr_rar(chan);

	if (qty > bytes)
		qty	= bytes;

	return(qty);
}



/******************************************************************************
*
*	Function:	pio_read_work
*
*	Purpose:
*
*		Implement the data reading portion of the PIO read operation.
*
*	Arguments:
*
*		dev		The device data structure.
*
*		chan	The channel data structure.
*
*		buff	The samples read go here.
*
*		count	The number of bytes to read.
*
*		st_end	End at this point in time (in system timer ticks). This is
*				zero if we're to end as soon at no data is available.
*
*	Returned:
*
*		>= 0	This is the number of samples read.
*
******************************************************************************/

ssize_t pio_read_work(
	dev_data_t*		dev,
	chan_data_t*	chan,
	char*			buff,
	ssize_t			count,
	os_time_tick_t	st_end)
{
	VADDR_T	fdr			= chan->vaddr.gsc_fdr_8;
	int		interval	= 0;
	ssize_t	qty			= count;

	for (; count; )
	{
		count--;
		buff[0]	= os_reg_mem_rx_u8(dev, fdr);
		buff++;

		// Rather then check for a timeout on every transfer we'll just do it
		// periodically. If a read takes about 2.5us, then the below count
		// means we check about every 1ms.

		if (interval < GSC_READ_PIO_PER_1MS)
		{
			interval++;
			continue;
		}

		interval	= 0;

		if (st_end)
		{
			// st_end: 0 = never timeout

			if (os_time_tick_timedout(st_end))
			{
				// We've timed out.
				break;
			}
		}
	}

	qty	-= count;
	return(qty);
}



/******************************************************************************
*
*	Function:	pio_write_available
*
*	Purpose:
*
*		Determine the number of bytes that can be written for a PIO mode write
*		operation.
*
*	Arguments:
*
*		dev		The device data structure.
*
*		samples	The number of samples desired.
*
*	Returned:
*
*		>= 0	This is the number of bytes to write.
*
******************************************************************************/

ssize_t pio_write_available(dev_data_t* dev, chan_data_t* chan, ssize_t bytes)
{
	ssize_t	qty;

	if (dev->cache.reg_fsr == 0)
		qty	= _tx_avail_csr_tar(chan);
	else if (dev->cache.reg_fcr)
		qty	= _tx_avail_fcr_fsr(chan);
	else
		qty	= _tx_avail_csr_fsr_tar(chan);

	if (qty > bytes)
		qty	= bytes;

	return(qty);
}



/******************************************************************************
*
*	Function:	pio_write_work
*
*	Purpose:
*
*		Implement the data writing portion of the PIO write operation.
*
*	Arguments:
*
*		dev		The device data structure.
*
*		chan	The channel data structure.
*
*		buff	The samples read go here.
*
*		count	The number of bytes to write.
*
*		st_end	End at this point in time (in system timer ticks). This is
*				zero if we're to end as soon at no data is available.
*
*	Returned:
*
*		>= 0	This is the number of samples written.
*
******************************************************************************/

ssize_t pio_write_work(
	dev_data_t*		dev,
	chan_data_t*	chan,
	const char*		buff,
	ssize_t			count,
	os_time_tick_t	st_end)
{
	VADDR_T	fdr			= chan->vaddr.gsc_fdr_8;
	int		interval	= 0;
	ssize_t	qty			= count;

	for (; count; )
	{
		count--;
		os_reg_mem_tx_u8(dev, fdr, buff[0]);
		buff++;

		// Rather then check for a timeout on every transfer we'll just do it
		// periodically. If a write takes about 2.5us, then the below count
		// means we check about every 1ms.

		if (interval < GSC_WRITE_PIO_PER_1MS)
		{
			interval++;
			continue;
		}

		interval	= 0;

		if (st_end)
		{
			// st_end: 0 = never timeout
			// timeout_s: 0 = stop when no more space is available

			if (os_time_tick_timedout(st_end))
			{
				// We've timed out.
				break;
			}
		}
	}

	qty	-= count;
	return(qty);
}



