// $URL: http://subversion:8080/svn/gsc/trunk/drivers/LINUX/SIO4%20and%20SIO8/SIO4_Linux_1.x.x_GSC_DN/driver/io_dmdma.c $
// $Rev: 53095 $
// $Date: 2023-06-13 10:42:41 -0500 (Tue, 13 Jun 2023) $

// SIO4: Device Driver: source file

#include "main.h"



/******************************************************************************
*
*	Function:	dmdma_read_available
*
*	Purpose:
*
*		Determine the number of bytes that can be read for a DMDMA mode read
*		operation.
*
*	Arguments:
*
*		dev		The device data structure.
*
*		chan	The channel data structure.
*
*		bytes	The number of bytes desired.
*
*	Returned:
*
*		>= 0	This is the number of bytes to read.
*
******************************************************************************/

ssize_t dmdma_read_available(dev_data_t* dev, chan_data_t* chan, ssize_t bytes)
{
	if (bytes > DMA_PIO_THRESHOLD)
		bytes	&= ~0x3L;	// Force it to a 32-bit boundary.

	return(bytes);
}



/******************************************************************************
*
*	Function:	dmdma_read_work
*
*	Purpose:
*
*		Implement the data reading portion of the DMDMA read operation.
*
*	Arguments:
*
*		dev	The device data structure.
*
*		chan	The channel data structure.
*
*		buff	The samples read go here.
*
*		count	The number of bytes to read.
*
*		st_end	End at this point in time (in system timer ticks). This is
*				zero if we're to end as soon at no data is available.
*
*	Returned:
*
*		>= 0	This is the number of samples read.
*
******************************************************************************/

ssize_t dmdma_read_work(
	dev_data_t*		dev,
	chan_data_t*	chan,
	char*			buff,
	ssize_t			count,
	os_time_tick_t	st_end)
{
	ssize_t	qty;

	if (count <= DMA_PIO_THRESHOLD)
	{
		qty	= pio_read_work(dev, chan, buff, count, st_end);
	}
	else
	{
		qty	= dma_perform(	dev,
					chan,
					0,	// read
					st_end,
					1,	// DMDMA
					buff,
					count);
	}

	return(qty);
}



/******************************************************************************
*
*	Function:	dmdma_write_available
*
*	Purpose:
*
*		Determine the number of bytes that can be written for a DMDMA mode
*		write operation.
*
*	Arguments:
*
*		dev		The device data structure.
*
*		samples	The number of samples desired.
*
*	Returned:
*
*		>= 0	This is the number of bytes to write.
*
******************************************************************************/

ssize_t dmdma_write_available(dev_data_t* dev, chan_data_t* chan, ssize_t bytes)
{
	if (bytes > DMA_PIO_THRESHOLD)
		bytes	&= ~0x3L;	// Force it to a 32-bit boundary.

	return(bytes);
}



/******************************************************************************
*
*	Function:	dmdma_write_work
*
*	Purpose:
*
*		Implement the data writing portion of the PIO write operation.
*
*	Arguments:
*
*		dev		The device data structure.
*
*		chan	The channel data structure.
*
*		buff	The samples read go here.
*
*		count	The number of bytes to write.
*
*		st_end	End at this point in time (in system timer ticks). This is
*				zero if we're to end as soon at no data is available.
*
*	Returned:
*
*		>= 0	This is the number of samples written.
*
******************************************************************************/

ssize_t dmdma_write_work(
	dev_data_t*		dev,
	chan_data_t*	chan,
	const char*		buff,
	ssize_t			count,
	os_time_tick_t	st_end)
{
	ssize_t	qty;

	if (count <= DMA_PIO_THRESHOLD)
	{
		qty	= pio_write_work(dev, chan, buff, count, st_end);
	}
	else
	{
		qty	= dma_perform(	dev,
					chan,
					1,	// write
					st_end,
					1,	// DMDMA
					buff,
					count);
	}

	return(qty);
}



