// $URL: http://subversion:8080/svn/gsc/trunk/drivers/LINUX/SIO4%20and%20SIO8/SIO4_Linux_1.x.x_GSC_DN/driver/io.c $
// $Rev: 53095 $
// $Date: 2023-06-13 10:42:41 -0500 (Tue, 13 Jun 2023) $

// SIO4: Device Driver: source file

#include "main.h"



/******************************************************************************
*
*	Function:	dev_io_close
*
*	Purpose:
*
*		Perform cleanup as a device is being closed.
*
*	Arguments.
*
*		io		The structure to cleanup.
*
*	Returned:
*
*		None.
*
******************************************************************************/

void dev_io_close(dev_io_t* io)
{
	if (io->mem.ptr)
		os_mem_dma_free(&io->mem);
}



/******************************************************************************
*
*	Function:	dev_io_open
*
*	Purpose:
*
*		Perform initialization work for a device that is being opened.
*
*	Arguments.
*
*		chan	The data for the channel being accessed.
*
*		io		The structure to initialize.
*
*	Returned:
*
*		0		All went well.
*		< 0		An appropriate error code.
*
******************************************************************************/

int dev_io_open(chan_data_t* chan, dev_io_t* io)
{
	size_t	bytes	= _128K;
	int		ret;

	WAIT_QUEUE_HEAD_INIT(&io->queue);
	io->mode	= SIO4_IO_MODE_DEFAULT;
	io->timeout	= SIO4_TIMEOUT_DEFAULT;

	os_mem_dma_alloc(chan, &bytes, &io->mem);
	ret	= io->mem.ptr ? 0 : -ENOMEM;
	return(ret);
}



/******************************************************************************
*
*	Function:	_read_work
*
*	Purpose:
*
*		Implement the mode independent working portion of the read() procedure.
*		If a timeout is permitted and called for, we use Rx FIFO status to wake
*		the task when the corresponding status has changed. If that interrupt
*		is not available to use, then we merely wait a short period as needed
*		instead.
*
*	Arguments:
*
*		dev			The device data structure.
*
*		chan		The channel data structure.
*
*		buff		The bytes read are placed here.
*
*		count		The number of bytes requested. This is positive.
*
*		st_end		End at this point in time (in system timer ticks). This is
*					zero if we're to end as soon at no data is available.
*
*		fn_avail	This is the mode specific function that determines the
*					number of bytes that may be read.
*
*		fn_work		This is the function implementing the mode specific reading
*					functionality.
*
*		irq_bit		The SIO4_IRQ_XXX bit to wait for when we run out of data.
*
*		edge		Use an edge triggerred intettupt?
*
*		high		Use an active high triggerred interrupt?
*
*	Returned:
*
*		>= 0		This is the number of bytes read.
*		< 0			An appropriate error status.
*
******************************************************************************/

static ssize_t _read_work(
	dev_data_t*		dev,
	chan_data_t*	chan,
	char*			buff,
	ssize_t			count,
	os_time_tick_t	st_end,
	ssize_t			(*fn_avail)(dev_data_t*		dev,
								chan_data_t*	chan,
								ssize_t			count),
	ssize_t			(*fn_work)(	dev_data_t*		dev,
								chan_data_t*	chan,
								char*			buff,
								ssize_t			count,
								os_time_tick_t	st_end),
	u32				irq_bit,
	int				edge,
	int				high)
{
	ssize_t			total	= 0;	// Total bytes given to the user.
	ssize_t			got		= 0;	// Bytes we've collected per block.
	char*			dest	= chan->rx.mem.ptr;
	ssize_t			qty;
	ssize_t			want	= 0;	// Bytes we want to read this time.
	unsigned long	ul;

	for (;;)
	{
		if (chan->rx.abort)
		{
			chan->rx.abort	= 0;
			break;
		}

		if (want <= 0)
		{
			// Transfer one block at a time.

			if (got > 0)
			{
				// Copy the data to user space.

				ul	= copy_to_user(buff, chan->rx.mem.ptr, (unsigned long) (long) got);

				if (ul)
				{
					// We couldn't get user data.
					got	= 0;
					total	= -EFAULT;
					break;
				}

				buff	+= got;
				total	+= got;
			}

			// Get a full or partial block?

			if (count > (ssize_t) chan->rx.mem.bytes)
				want	= (ssize_t) chan->rx.mem.bytes;
			else
				want	= count;

			got		= 0;
			dest	= chan->rx.mem.ptr;
		}

		qty	= (fn_avail)(dev, chan, want);

		if (qty)
		{
			qty	= (fn_work)(dev, chan, dest, qty, st_end);

			if (qty < 0)
			{
				// There was a problem.
				total	= qty;
				got	= 0;
				break;
			}

			count	-= qty;
			want	-= qty;
			dest	+= qty;
			got	+= qty;

			if (count == 0)
			{
				// The request has been fulfilled.
				break;
			}
			else if ((st_end) && (os_time_tick_timedout(st_end)))
			{
				// We got some data then timed out.
				break;
			}
			else if (st_end)
			{
				// We've got more time to try for more.
				continue;
			}
			else if (qty)
			{
				// We can't block bit we did get some data.
				continue;
			}
			else
			{
				// We can't block and we got no data.
				break;
			}
		}
		else if (st_end == 0)
		{
			// No data is available and we can't block.
			break;
		}
		else if (os_time_tick_timedout(st_end))
		{
			// No data is available and we can't wait any
			// longer for more.
			break;
		}

		// Wait a while for data to become available.
		os_time_tick_sleep(1);
		continue;
	}

	if (got > 0)
	{
		// Copy the data to user space.

		ul	= copy_to_user(buff, chan->rx.mem.ptr, (unsigned long) (long) got);

		if (ul)
			total	= -EFAULT;
		else
			total	+= got;
	}

	return(total);
}



/******************************************************************************
*
*	Function:	os_read
*
*	Purpose:
*
*		Perform a PIO based read.
*
*	Arguments.
*
*		filp	This is the file structure.
*
*		buff	The data requested goes here.
*
*		count	The number of bytes requested.
*
*		offp	The file position the user is requesting.
*
*	Returned:
*
*		>= 0	The number of bytes read.
*		< 0		An appropriate error code.
*
******************************************************************************/

ssize_t os_read(struct file* filp, char* buff, size_t count, loff_t* offp)
{
	chan_data_t*	chan;
	dev_data_t*		dev;
	os_time_tick_t	st_end;
	ssize_t			ret;
	int				sem		= 1;
	unsigned long	ul;

	for (;;)	// We'll use a loop for convenience.
	{
		// Gain the neccessary access.

		ret	= dev_lock_filp(filp, &chan);

		if (ret)
			break;

		sem	= os_sem_lock(&chan->rx.sem);
		os_sem_unlock(&chan->sem);

		if (sem)
		{
			// We got a signal rather than the READ semaphore.
			ret	= -ERESTARTSYS;
			break;
		}

		// Validate some of the parameters.

		if (count == 0)
		{
			// No data transfer called for.
			ret	= 0;
			break;
		}

		if (buff == NULL)
		{
			// A buffer wasn't provided.
			ret	= -EINVAL;
			break;
		}

		ul	= (unsigned long) MEM_ACCESS_OK(VERIFY_WRITE, buff, (unsigned long) count);

		if (ul == 0)
		{
			// We can't write to the user's memory.
			ret	= -EFAULT;
			break;
		}

		// Prepare for the operation.

		if ((filp->f_flags & O_NONBLOCK) || (chan->rx.timeout <= 0))
			st_end	= 0;
		else
			st_end	= os_time_tick_get() + chan->rx.timeout * os_time_tick_rate();

		// Perform the operation.

		dev	= chan->dev;

		if (chan->rx.mode == SIO4_IO_MODE_DMDMA)
		{
			if (dev->cache.dmdma_scd)
			{
				os_reg_mem_mx_u32(	dev,
									dev->vaddr.gsc_bcr_32,
									0xFFFFFFFF,
									SIO4_GSC_BCR_DMA0_SCD |
									SIO4_GSC_BCR_DMA1_SCD);
			}

			ret	= _read_work(	dev,
						chan,
						buff,
						count,
						st_end,
						dmdma_read_available,
						dmdma_read_work,
						SIO4_IRQ_RFAF,
						1,
						1);
		}
		else if (chan->rx.mode == SIO4_IO_MODE_DMA)
		{
			ret	= _read_work(	dev,
						chan,
						buff,
						count,
						st_end,
						dma_read_available,
						dma_read_work,
						SIO4_IRQ_RFAF,
						1,
						1);
		}
		else
		{
			ret	= _read_work(	dev,
						chan,
						buff,
						count,
						st_end,
						pio_read_available,
						pio_read_work,
						SIO4_IRQ_RFAF,
						1,
						1);
		}

		break;
	}

	if (sem == 0)
		os_sem_unlock(&chan->rx.sem);

	return(ret);
}



/******************************************************************************
*
*	Function:	_write_work
*
*	Purpose:
*
*		Implement the mode independent working portion of the write()
*		procedure. If a timeout is permitted and called for, we use
*		Tx FIFO status to wake the task when the corresponding status
*		has changed. If that interrupt is not available to use, then
*		we merely wait a short period as needed instead.
*
*	Arguments:
*
*		dev			The device data structure.
*
*		chan		The channel data structure.
*
*		buff		The bytes to write come from here.
*
*		count		The number of bytes requested. This is positive.
*
*		st_end		End at this point in time (in system timer ticks). This is
*					zero if we're to end as soon at no data is available.
*
*		fn_avail	This is the mode specific function that determines the
*					number of bytes that may be written.
*
*		fn_work		This is the function implementing the mode specific write
*					functionality.
*
*		irq_bit		The SIO4_IRQ_XXX bit to wait for when we run out of space.
*
*		edge		Use an edge triggerred intettupt?
*
*		high		Use an active high triggerred interrupt?
*
*	Returned:
*
*		>= 0		This is the number of bytes read.
*		< 0			An appropriate error status.
*
******************************************************************************/

static ssize_t _write_work(
	dev_data_t*		dev,
	chan_data_t*	chan,
	const char*		buff,
	ssize_t			count,
	os_time_tick_t	st_end,
	ssize_t			(*fn_avail)(dev_data_t*		dev,
								chan_data_t*	chan,
								ssize_t			count),
	ssize_t			(*fn_work)(	dev_data_t*		dev,
								chan_data_t*	chan,
								const char*		buff,
								ssize_t			count,
								os_time_tick_t	st_end),
	u8				irq_bit,
	int				edge,
	int				high)
{
	ssize_t			total	= 0;	// Total bytes written to the device.
	char*			dest	= chan->tx.mem.ptr;
	ssize_t			qty;
	ssize_t			want	= 0;	// Bytes we want to write this time.
	unsigned long	ul;

	for (;;)
	{
		if (chan->tx.abort)
		{
			chan->tx.abort	= 0;
			break;
		}

		if (want <= 0)
		{
			// Transfer one block at a time.

			if (count > (ssize_t) chan->tx.mem.bytes)
				want	= chan->tx.mem.bytes;
			else
				want	= count;

			dest	= chan->tx.mem.ptr;
			ul	= copy_from_user(dest, buff, (unsigned long) want);
			buff	+= want;

			if (ul)
			{
				// We couldn't get user data.
				total	= -EFAULT;
				break;
			}
		}

		qty	= (fn_avail)(dev, chan, want);

		if (qty)
		{
			qty	= (fn_work)(dev, chan, dest, qty, st_end);

			if (qty < 0)
			{
				// There was a problem.
				total	= qty;
				break;
			}

			count	-= qty;
			want	-= qty;
			dest	+= qty;
			total	+= qty;

			if (count == 0)
			{
				// The request has been fulfilled.
				break;
			}
			else if ((st_end) && (os_time_tick_timedout(st_end)))
			{
				// Some data was written then we timed out.
				break;
			}
			else if (st_end)
			{
				// We've got more time to try for more.
				continue;
			}
			else if (qty)
			{
				// We can't block but we did send some data.
				continue;
			}
			else
			{
				// We can't block and we sent no data.
				break;
			}
		}
		else if (st_end == 0)
		{
			// No space is available and we can't block.
			break;
		}
		else if (os_time_tick_timedout(st_end))
		{
			// No space is available and we can't wait any
			// longer for more.

			break;
		}

		// Wait a while for space to become available.
		os_time_tick_sleep(1);
		continue;
	}

	return(total);
}



/******************************************************************************
*
*	Function:	os_write
*
*	Purpose:
*
*		Perform a PIO based write.
*
*	Arguments.
*
*		filp	This is the file structure.
*
*		buff	The data requested goes here.
*
*		count	The number of bytes requested.
*
*		offp	The file position the user is requesting.
*
*	Returned:
*
*		>= 0	The number of bytes read.
*		< 0		An appropriate error code.
*
******************************************************************************/

ssize_t os_write(struct file* filp, const char* buff, size_t count, loff_t* offp)
{
	chan_data_t*	chan;
	dev_data_t*		dev;
	os_time_tick_t	st_end;
	ssize_t			ret;
	int				sem		= 1;
	unsigned long	ul;

	for (;;)	// We'll use a loop for convenience.
	{
		// Gain the neccessary access.

		ret	= dev_lock_filp(filp, &chan);

		if (ret)
			break;

		sem	= os_sem_lock(&chan->tx.sem);
		os_sem_unlock(&chan->sem);

		if (sem)
		{
			// We got a signal rather than the WRITE semaphore.
			ret	= -ERESTARTSYS;
			break;
		}

		// Validate some of the parameters.

		if (count == 0)
		{
			// No data transfer called for.
			ret	= 0;
			break;
		}

		if (buff == NULL)
		{
			// A buffer wasn't provided.
			ret	= -EINVAL;
			break;
		}

		ul	= (unsigned long) MEM_ACCESS_OK(VERIFY_READ, buff, count);

		if (ul == 0)
		{
			// We can't read from the user's memory.
			ret	= -EFAULT;
			break;
		}

		// Prepare for the operation.

		if ((filp->f_flags & O_NONBLOCK) || (chan->tx.timeout <= 0))
			st_end	= 0;
		else
			st_end	= os_time_tick_get() + chan->tx.timeout * os_time_tick_rate();

		// Perform the operation.

		dev	= chan->dev;

		if (chan->tx.mode == SIO4_IO_MODE_DMDMA)
		{
			if (dev->cache.dmdma_scd)
			{
				os_reg_mem_mx_u32(	dev,
									dev->vaddr.gsc_bcr_32,
									0xFFFFFFFF,
									SIO4_GSC_BCR_DMA0_SCD |
									SIO4_GSC_BCR_DMA1_SCD);
			}

			ret	= _write_work(	dev,
						chan,
						buff,
						count,
						st_end,
						dmdma_write_available,
						dmdma_write_work,
						SIO4_IRQ_TFAE,
						1,
						1);
		}
		else if (chan->tx.mode == SIO4_IO_MODE_DMA)
		{
			ret	= _write_work(	dev,
						chan,
						buff,
						count,
						st_end,
						dma_write_available,
						dma_write_work,
						SIO4_IRQ_TFAE,
						1,
						1);
		}
		else
		{
			ret	= _write_work(	dev,
						chan,
						buff,
						count,
						st_end,
						pio_write_available,
						pio_write_work,
						SIO4_IRQ_TFAE,
						1,
						1);
		}

		break;
	}

	if (sem == 0)
		os_sem_unlock(&chan->tx.sem);

	return(ret);
}



/******************************************************************************
*
*	Function:	io_mode_config
*
*	Purpose:
*
*		Process an I/O Mode Configuration request.
*
*	Arguments:
*
*		io		The I/O structure to access. We don't lock this as it is OK to
*				change it while I/O is in progress.
*
*		value	The value to use and report is here.
*
*	Returned:
*
*		0		All went well.
*		< 0		An appropriate error status.
*
******************************************************************************/

int io_mode_config(dev_io_t* io, s32* value)
{
	int	ret;

	switch (value[0])
	{
		default:

			ret	= -EINVAL;
			break;

		case SIO4_IO_MODE_READ:

			value[0]	= io->mode;
			ret		= 0;
			break;

		case SIO4_IO_MODE_PIO:
		case SIO4_IO_MODE_DMA:
		case SIO4_IO_MODE_DMDMA:

			io->mode	= value[0];
			ret		= 0;
			break;
	}

	return(ret);
}



/******************************************************************************
*
*	Function:	io_abort
*
*	Purpose:
*
*		Abort the I/O on the given structure.
*
*	Arguments:
*
*		io		The strcture for the I/O to abort.
*
*	Returned:
*
*		None.
*
******************************************************************************/

int io_abort(dev_io_t* io)
{
	int				aborted	= 0;
	os_time_tick_t	st_end	= os_time_tick_get() + 10 * os_time_tick_rate();

	io->abort	= 1;

	for (;;)
	{
		if (io->abort == 0)
		{
			aborted	= 1;
			break;
		}

		if (os_time_tick_timedout(st_end))
			break;

		os_time_tick_sleep(1);
	}

	io->abort	= 0;
	return(aborted);
}



/******************************************************************************
*
*	Function:	io_dev_data_t_init
*
*	Purpose:
*
*		Perform a one-time initialization of the device's referenced I/O
*		structure.
*
*	Arguments:
*
*		dev		The device of interest.
*
*		index	The index of the channel to access.
*
*		tx		Access the Tx I/O structure? If not, the Rx is used.
*
*	Returned:
*
*		None.
*
******************************************************************************/

void io_dev_data_t_init(dev_data_t* dev, int index, int tx)
{
	chan_data_t*	chan	= &dev->channel[index];
	dev_io_t*		io		= tx ? &chan->tx : &chan->rx;

	os_sem_create(&io->sem);
	io->abort		= 0;
	io->dmdma_dir	= ((index & 2) ? 1 : 0)
					| ((index & 1) ? 2 : 0)
					| (tx ? 4 : 0);
	io->mode		= SIO4_IO_MODE_DEFAULT;
	io->timeout		= SIO4_TIMEOUT_DEFAULT;
	WAIT_QUEUE_HEAD_INIT(&io->queue);
}



