// $URL: http://subversion:8080/svn/gsc/trunk/drivers/LINUX/SIO4%20and%20SIO8/SIO4_Linux_1.x.x_GSC_DN/driver/fifo.c $
// $Rev: 53095 $
// $Date: 2023-06-13 10:42:41 -0500 (Tue, 13 Jun 2023) $

// SIO4: Device Driver: source file

#include "main.h"



/******************************************************************************
*
*	Function:	fifo_almost_config
*
*	Purpose:
*
*		Configure the referenced Almost Level for the referenced FIFO.
*
*	Arguments:
*
*		chan	The structure for the channel being accessed.
*
*		value	The value to apply to the desired status level.
*
*		vaddr	The virtual address of the register containing the almost
*				level to access.
*
*		shift	The amount to shift the value given when accessing the given
*				almost register.
*
*		reset	The mask of the Control/Status Register bit to set in order to
*				reset the FIFO.
*
*	Returned:
*
*		The appropriate Almost value.
*
******************************************************************************/

s32 fifo_almost_config(
	chan_data_t*	chan,
	s32				value,
	VADDR_T			vaddr,
	int				shift,
	u32				reset)
{
	dev_data_t*	dev		= chan->dev;
	u32			mask	= 0xFFFF << shift;
	u32			val;

	if (value != -1)
	{
		// Apply the value provided.
		val	= (u32) value << shift;
		os_reg_mem_mx_u32(dev, vaddr, val, mask);
		os_reg_mem_mx_u32(dev, chan->vaddr.gsc_csr_32, reset, reset);
		os_time_us_delay(10000);	// worst case for older devices
	}

	// Retrieve the current value.
	val		= os_reg_mem_rx_u32(dev, vaddr);
	val		= val & mask;
	value	= (s32) (val >> shift);
	return(value);
}



/******************************************************************************
*
*	Function:	fifo_count
*
*	Purpose:
*
*		Tell the user the FIFO Count.
*
*	Arguments:
*
*		dev		The structure for the device being accessed.
*
*		shift	Shift the FIFO Count Register this amount to have the desired
*				level in the lower bits.
*
*	Returned:
*
*		-1		The size is unknown.
*		> 0		The FIFO size in bytes.
*
******************************************************************************/


s32 fifo_count(chan_data_t* chan, int shift)
{
	dev_data_t*	dev		= chan->dev;
	u32			fcr;
	s32			ret;

	if (dev->cache.reg_fcr)
	{
		fcr	= os_reg_mem_rx_u32(dev, chan->vaddr.gsc_fcr_32);
		ret	= (fcr >> shift) & 0xFFFF;
	}
	else
	{
		ret	= SIO4_FIFO_COUNT_UNKNOWN;
	}

	return(ret);
}


