// $URL: http://subversion:8080/svn/gsc/trunk/drivers/LINUX/SIO4%20and%20SIO8/SIO4_Linux_1.x.x_GSC_DN/driver/device.c $
// $Rev: 53095 $
// $Date: 2023-06-13 10:42:41 -0500 (Tue, 13 Jun 2023) $

// SIO4: Device Driver: source file

#include "main.h"



// macros *********************************************************************

#undef	GSC_VADDR
#define	GSC_VADDR(d,o)						(VADDR_T) (((u8*) (d)->vaddr.gsc_regs) + (o))

#undef	PLX_VADDR
#define	PLX_VADDR(d,o)						(VADDR_T) (((u8*) (d)->vaddr.plx_regs) + (o))

#define	FW_VADDR(d,r,i,m)					GSC_VADDR((d),GSC_REG_OFFSET((r))+((i)*(m)))
#define	INTCSR_DMA_0_IRQ_ENABLE				0x00040000

#define	RS232								((s32) SIO4_MP_PROT_RS_232)
#define	RS422_485							((s32) SIO4_MP_PROT_RS_422_485)
#define	RS423								((s32) SIO4_MP_PROT_RS_423)
#define	RS530A								((s32) SIO4_MP_PROT_RS_530_M1)
#define	RS530B								((s32) SIO4_MP_PROT_RS_530_M2)
#define	RS530x								(RS530A | RS530B)
#define	V35A								((s32) SIO4_MP_PROT_V35_M1)



// ****************************************************************************
static void _dev_dma_t_init(dev_data_t* dev, int index)
{
	dev_dma_t*	dma	= &dev->dma.channel[index];

	dma->int_enable		= INTCSR_DMA_0_IRQ_ENABLE << index;
	dma->dir_mask		= 0x07 << (index * 4);
	dma->dir_shift		= (unsigned int) index * 4;
	dma->vaddr.mode_32	= (VADDR_T) ((long) dev->vaddr.plx_regs + 0x80 + ((long) 0x14 * index));
	dma->vaddr.padr_32	= (VADDR_T) ((long) dev->vaddr.plx_regs + 0x84 + ((long) 0x14 * index));
	dma->vaddr.ladr_32	= (VADDR_T) ((long) dev->vaddr.plx_regs + 0x88 + ((long) 0x14 * index));
	dma->vaddr.siz_32	= (VADDR_T) ((long) dev->vaddr.plx_regs + 0x8C + ((long) 0x14 * index));
	dma->vaddr.dpr_32	= (VADDR_T) ((long) dev->vaddr.plx_regs + 0x90 + ((long) 0x14 * index));
	dma->vaddr.csr_8	= (VADDR_T) ((long) dev->vaddr.plx_regs + 0xA8 + index);
}



// ****************************************************************************
static void _board_reset_compute(dev_data_t* dev)
{
	switch (dev->cache.sio4_type)
	{
		default:

			printf(	"%s: INTERNAL ERROR, line %d, file %s\n",
					DEV_NAME,
					__LINE__,
					__FILE__);

			FALLTHROUGH;	/* fall through */

		case SIO4_TYPE_SIO4:
		case SIO4_TYPE_SIO4A:
		case SIO4_TYPE_SIO4AR:
		case SIO4_TYPE_SIO4AHRM:

			dev->cache.board_reset	= 0;
			break;

		case SIO4_TYPE_SIO4B:
		case SIO4_TYPE_SIO4BX:
		case SIO4_TYPE_SIO4BX2:
		case SIO4_TYPE_SIO4BXR:
		case SIO4_TYPE_SIO8BX2:
		case SIO4_TYPE_SIO8BXS:

			if (dev->cache.reg_fr == 0)
				dev->cache.board_reset	= 0;			// FR required to find out
			else if ((dev->cache.gsc_fr_32 & D4) == 0)
				dev->cache.board_reset	= 0;			// FR Format is unknown
			else if (dev->cache.gsc_fr_32 & D6)
				dev->cache.board_reset	= 1;			// FR D6 is the indicator
			else
				dev->cache.board_reset	= 0;			// FR D6 is the indicator

			break;
	}

	if (dev->cache.board_reset)
	{
		os_reg_mem_tx_u32(	dev,
							dev->vaddr.gsc_bcr_32,
							SIO4_GSC_BCR_BRD_RST);
		os_time_us_delay(2);
	}
}



// ****************************************************************************
static void _bus_speed_compute(dev_data_t* dev)
{
	switch (dev->cache.sio4_type)
	{
		default:

			printf(	"%s: INTERNAL ERROR, line %d, file %s\n",
					DEV_NAME,
					__LINE__,
					__FILE__);

			FALLTHROUGH;	/* fall through */

		case SIO4_TYPE_SIO4:

			dev->cache.bus_speed	= 33;
			break;

		case SIO4_TYPE_SIO4A:
		case SIO4_TYPE_SIO4AR:
		case SIO4_TYPE_SIO4AHRM:
		case SIO4_TYPE_SIO4B:
		case SIO4_TYPE_SIO4BX:
		case SIO4_TYPE_SIO4BX2:
		case SIO4_TYPE_SIO4BXR:
		case SIO4_TYPE_SIO8BX2:
		case SIO4_TYPE_SIO8BXS:

			if ((dev->cache.gsc_frr_32 & 0x40000000) == 0)
				dev->cache.bus_speed	= 33;
			else if (dev->cache.gsc_frr_32 & 0x20000000)
				dev->cache.bus_speed	= 66;
			else
				dev->cache.bus_speed	= 33;

			break;
	}
}



// ****************************************************************************
static void _bus_width_compute(dev_data_t* dev)
{
	switch (dev->cache.sio4_type)
	{
		default:

			printf(	"%s: INTERNAL ERROR, line %d, file %s\n",
					DEV_NAME,
					__LINE__,
					__FILE__);

			FALLTHROUGH;	/* fall through */

		case SIO4_TYPE_SIO4:
		case SIO4_TYPE_SIO4A:
		case SIO4_TYPE_SIO4AR:
		case SIO4_TYPE_SIO4AHRM:
		case SIO4_TYPE_SIO4B:

			dev->cache.bus_width	= 32;
			break;

		case SIO4_TYPE_SIO4BX:
		case SIO4_TYPE_SIO4BX2:
		case SIO4_TYPE_SIO4BXR:
		case SIO4_TYPE_SIO8BX2:
		case SIO4_TYPE_SIO8BXS:

			if ((dev->cache.gsc_frr_32 & 0x40000000) == 0)
				dev->cache.bus_width	= 32;
			else if (dev->cache.gsc_frr_32 & 0x10000000)
				dev->cache.bus_width	= 64;
			else
				dev->cache.bus_width	= 32;

			break;
	}
}



// ****************************************************************************
static void _dmdma_scd_compute(dev_data_t* dev)
{
	switch (dev->cache.sio4_type)
	{
		default:

			printf(	"%s: INTERNAL ERROR, line %d, file %s\n",
					DEV_NAME,
					__LINE__,
					__FILE__);

			FALLTHROUGH;	/* fall through */

		case SIO4_TYPE_SIO4:
		case SIO4_TYPE_SIO4A:
		case SIO4_TYPE_SIO4AR:
		case SIO4_TYPE_SIO4AHRM:

			dev->cache.dmdma_scd	= 0;
			break;

		case SIO4_TYPE_SIO4B:
		case SIO4_TYPE_SIO4BX:
		case SIO4_TYPE_SIO4BX2:
		case SIO4_TYPE_SIO4BXR:
		case SIO4_TYPE_SIO8BX2:
		case SIO4_TYPE_SIO8BXS:

			if (dev->cache.reg_fr == 0)
				dev->cache.dmdma_scd	= 0;		// FR required to find out
			else if ((dev->cache.gsc_fr_32 & D4) == 0)
				dev->cache.dmdma_scd	= 0;		// FR Format is unknown
			else if (dev->cache.gsc_fr_32 & D7)
				dev->cache.dmdma_scd	= 1;		// FR D7 is the indicator
			else
				dev->cache.dmdma_scd	= 0;		// FR D7 is the indicator

			break;
	}
}



// ****************************************************************************
static void _fifo_size_compute_side(chan_data_t* chan, int shift, s32* size)
{
	u32		csr;
	long	l;

	size[0]	= 0;

	if (chan->cache.gsc_fsr_32)
	{
		size[0]	= (chan->cache.gsc_fsr_32 >> shift) & 0xFFFF;
	}
	else if (shift == 0)
	{
		// Compute the size of the Tx FIFO.
		os_reg_mem_tx_u32(chan->dev, chan->vaddr.gsc_csr_32, 0x3);	// Reset the FIFOs
		os_time_us_delay(10000);	// worst case for older devices

		for (l = 0; l < 0xFFFF;)
		{
			l++;
			os_reg_mem_tx_u8(chan->dev, chan->vaddr.gsc_fdr_8, 0);
			csr	= os_reg_mem_rx_u32(chan->dev, chan->vaddr.gsc_csr_32);

			if ((csr & D11) == 0)
			{
				size[0]	= (s32) l;
				break;
			}
		}

		os_reg_mem_tx_u32(chan->dev, chan->vaddr.gsc_csr_32, 0x3);	// Reset the FIFOs
		os_time_us_delay(10000);	// worst case for older devices
	}
	else
	{
		// Presume the Rx FIFO is the same size as the Tx FIFO.
		size[0]	= chan->cache.fifo_size_tx;
	}

	chan->dev->cache.fifo_size_total	+= size[0];
}



// ****************************************************************************
static void _fifo_size_compute(chan_data_t* chan, int index)
{
	_fifo_size_compute_side(chan,  0, &chan->cache.fifo_size_tx);
	_fifo_size_compute_side(chan, 16, &chan->cache.fifo_size_rx);
}



// ****************************************************************************
static void _form_factor_compute(dev_data_t* dev)
{
	u32	ff;
	u32	frr;

	frr	= dev->cache.gsc_frr_32;

	if ((frr & 0xFFFF0000) == 0xFFFF0000)
	{
		// An original SIO4.
		// The form factor cannot be determined for basic SIO4 model devices.
		dev->cache.form_factor	= SIO4_FORM_FACTOR_UNKNOWN;
	}
	else
	{
		// Everything else is presumed to be an SIO4A or later.
		// The form factor can be determined for SIO4A and all subsequent devices.
		ff	= GSC_FIELD_DECODE(frr, 27, 24);

		switch (ff)
		{
			default:	dev->cache.form_factor	= SIO4_FORM_FACTOR_UNKNOWN;	break;
			case 1:		dev->cache.form_factor	= SIO4_FORM_FACTOR_PCI;		break;
			case 2:		dev->cache.form_factor	= SIO4_FORM_FACTOR_PMC;		break;
			case 3:		dev->cache.form_factor	= SIO4_FORM_FACTOR_CPCI;	break;
			case 4:		dev->cache.form_factor	= SIO4_FORM_FACTOR_PC104P;	break;
			case 5:		dev->cache.form_factor	= SIO4_FORM_FACTOR_PCIE;	break;
			case 6:		dev->cache.form_factor	= SIO4_FORM_FACTOR_XMC;		break;
		}
	}
}



// ****************************************************************************
static void _fw_type_compute(dev_data_t* dev)
{
	if ((dev->cache.gsc_frr_32 & 0xFFFF0000) == 0xFFFF0000)
	{
		// This is an original SIO4 model device.
		// Z16C30 firmware is type 0x00 or 0x80.
		// Z16C30 will be assumed for all variations.
		dev->cache.fw_type		= SIO4_FW_TYPE_Z16C30;
		dev->cache.model_sync	= 0;
		dev->cache.model_z16c30	= 1;
	}

	// For all other devices ...

	else if ((dev->cache.gsc_frr_32 & 0xFF00) == 0x0100)
	{
		dev->cache.fw_type		= SIO4_FW_TYPE_Z16C30;
		dev->cache.model_sync	= 0;
		dev->cache.model_z16c30	= 1;
	}
	else if ((dev->cache.gsc_frr_32 & 0xFF00) == 0x0400)
	{
		strcat(dev->model_buf, "-SYNC");
		dev->cache.fw_type		= SIO4_FW_TYPE_SYNC;
		dev->board_type			= GSC_DEV_TYPE_SIO4_SYNC;
		dev->cache.model_sync	= 1;
		dev->cache.model_z16c30	= 0;
	}
	else
	{
		// Assume Z16C30 for all other variations.
		dev->cache.fw_type		= SIO4_FW_TYPE_Z16C30;
		dev->cache.model_sync	= 0;
		dev->cache.model_z16c30	= 1;
	}
}



// ****************************************************************************
static void _index_subdevice_compute(dev_data_t* dev)
{
	u32	bsr;

	switch (dev->cache.sio4_type)
	{
		default:

			printf(	"%s: INTERNAL ERROR, line %d, file %s\n",
					DEV_NAME,
					__LINE__,
					__FILE__);

			FALLTHROUGH;	/* fall through */

		case SIO4_TYPE_SIO4:
		case SIO4_TYPE_SIO4A:
		case SIO4_TYPE_SIO4AR:
		case SIO4_TYPE_SIO4AHRM:
		case SIO4_TYPE_SIO4B:
		case SIO4_TYPE_SIO4BX:
		case SIO4_TYPE_SIO4BX2:
		case SIO4_TYPE_SIO4BXR:

			dev->cache.index_subdevice	= 0;
			break;

		case SIO4_TYPE_SIO8BX2:
		case SIO4_TYPE_SIO8BXS:

			bsr	= os_reg_mem_rx_u32(dev, dev->vaddr.gsc_bsr_32);
			dev->cache.index_subdevice	= (bsr & 0x1) ? 1 : 0;
			break;
	}
}



// ****************************************************************************
static void _irq_compute(dev_data_t* dev)
{
	if (dev->cache.sio4_type == SIO4_TYPE_SIO4)
	{
		dev->cache.irq_32		= 0;
		dev->cache.irq_mask		= 0xFFFF;
		dev->cache.reg_ielr		= 0;
		dev->cache.reg_ihlr		= 0;
		dev->cache.reg_ielr_32	= 0;
	}
	else
	{
		dev->cache.irq_32		= 1;
		dev->cache.irq_mask		= (u32) 0xFFFFFFFF;
		dev->cache.reg_ielr		= 1;
		dev->cache.reg_ihlr		= 1;
		dev->cache.reg_ielr_32	= os_reg_mem_rx_u32(dev, dev->vaddr.gsc_ielr_32);
	}
}



// ****************************************************************************
static void _led_compute(dev_data_t* dev)
{
	u32	rev;

	switch (dev->cache.sio4_type)
	{
		default:

			printf(	"%s: INTERNAL ERROR, line %d, file %s\n",
					DEV_NAME,
					__LINE__,
					__FILE__);

			FALLTHROUGH;	/* fall through */

		case SIO4_TYPE_SIO4:
		case SIO4_TYPE_SIO4A:
		case SIO4_TYPE_SIO4AR:
		case SIO4_TYPE_SIO4AHRM:
		case SIO4_TYPE_SIO4B:
		case SIO4_TYPE_SIO4BX:
		case SIO4_TYPE_SIO8BXS:

			dev->cache.led_channel	= 0;
			dev->cache.led_main		= 0;
			break;

		case SIO4_TYPE_SIO4BX2:

			// These are dual-color LEDs: off, red, green, orange
			dev->cache.led_channel	= 2;		// CSR D20-D23
			dev->cache.led_main		= 2;		// BCR D24-D25
			break;

		case SIO4_TYPE_SIO4BXR:

			rev	= GSC_FIELD_DECODE(dev->cache.gsc_frr_32, 19, 16);

			if ((dev->cache.form_factor == SIO4_FORM_FACTOR_PMC66) &&
				(rev >= 4))
			{
				dev->cache.led_channel	= 2;	// CSR D20-D23
				dev->cache.led_main		= 3;	// BCR D24-D26
			}
			else if (dev->cache.form_factor == SIO4_FORM_FACTOR_PCI66)
			{
				dev->cache.led_channel	= 2;	// CSR D20-D23
				dev->cache.led_main		= 3;	// BCR D24-D26
			}
			else
			{
				dev->cache.led_channel	= 0;
				dev->cache.led_main		= 0;
			}

			break;

		case SIO4_TYPE_SIO8BX2:

			// These are dual-color LEDs: off, red, green, orange
			dev->cache.led_channel	= 2;		// CSR D20-D23
			dev->cache.led_main		= 2;		// BCR D24-D25
			break;
	}
}



// ****************************************************************************
static void _legacy_cable_compute(dev_data_t* dev)
{
	switch (dev->cache.sio4_type)
	{
		default:

			printf(	"%s: INTERNAL ERROR, line %d, file %s\n",
					DEV_NAME,
					__LINE__,
					__FILE__);

			FALLTHROUGH;	/* fall through */

		case SIO4_TYPE_SIO4:
		case SIO4_TYPE_SIO4A:
		case SIO4_TYPE_SIO4AR:
		case SIO4_TYPE_SIO4AHRM:

			dev->cache.legacy_cable	= 1;
			break;

		case SIO4_TYPE_SIO4B:
		case SIO4_TYPE_SIO4BX:
		case SIO4_TYPE_SIO4BXR:
		case SIO4_TYPE_SIO4BX2:
		case SIO4_TYPE_SIO8BX2:
		case SIO4_TYPE_SIO8BXS:

			if (dev->cache.reg_fr == 0)
				dev->cache.legacy_cable	= 1;	// Feature supported.
			else if ((dev->cache.gsc_fr_32 & 0x10) == 0)
				dev->cache.legacy_cable	= 1;	// Feature supported.
			else if ((dev->cache.gsc_fr_32 & 0xFF00) < 0x0900)
				dev->cache.legacy_cable	= 1;	// Feature supported.
			else
				dev->cache.legacy_cable	= 0;	// Feature not supported.

			break;
	}
}



// ****************************************************************************
static void _model_compute(dev_data_t* dev)
{
	static const struct
	{
		s32	type;
		s32	fw;
		s32	model;
	} list[]	=
	{	// type					fw						model
		{ SIO4_TYPE_SIO4,		SIO4_FW_TYPE_Z16C30,	SIO4_MODEL_SIO4				},

		{ SIO4_TYPE_SIO4A,		SIO4_FW_TYPE_Z16C30,	SIO4_MODEL_SIO4A			},
		{ SIO4_TYPE_SIO4A,		SIO4_FW_TYPE_SYNC,		SIO4_MODEL_SIO4A_SYNC		},
		{ SIO4_TYPE_SIO4AR,		SIO4_FW_TYPE_Z16C30,	SIO4_MODEL_SIO4AR			},
		{ SIO4_TYPE_SIO4AR,		SIO4_FW_TYPE_SYNC,		SIO4_MODEL_SIO4AR_SYNC		},
		{ SIO4_TYPE_SIO4AHRM,	SIO4_FW_TYPE_Z16C30,	SIO4_MODEL_SIO4AHRM			},
		{ SIO4_TYPE_SIO4AHRM,	SIO4_FW_TYPE_SYNC,		SIO4_MODEL_SIO4AHRM_SYNC	},

		{ SIO4_TYPE_SIO4B,		SIO4_FW_TYPE_Z16C30,	SIO4_MODEL_SIO4B			},
		{ SIO4_TYPE_SIO4B,		SIO4_FW_TYPE_SYNC,		SIO4_MODEL_SIO4B_SYNC		},
		{ SIO4_TYPE_SIO4BX,		SIO4_FW_TYPE_Z16C30,	SIO4_MODEL_SIO4BX			},
		{ SIO4_TYPE_SIO4BX,		SIO4_FW_TYPE_SYNC,		SIO4_MODEL_SIO4BX_SYNC		},
		{ SIO4_TYPE_SIO4BX2,	SIO4_FW_TYPE_Z16C30,	SIO4_MODEL_SIO4BX2			},
		{ SIO4_TYPE_SIO4BX2,	SIO4_FW_TYPE_SYNC,		SIO4_MODEL_SIO4BX2_SYNC		},
		{ SIO4_TYPE_SIO4BXR,	SIO4_FW_TYPE_Z16C30,	SIO4_MODEL_SIO4BXR			},
		{ SIO4_TYPE_SIO4BXR,	SIO4_FW_TYPE_SYNC,		SIO4_MODEL_SIO4BXR_SYNC		},

		{ SIO4_TYPE_SIO8BX2,	SIO4_FW_TYPE_Z16C30,	SIO4_MODEL_SIO8BX2			},
		{ SIO4_TYPE_SIO8BX2,	SIO4_FW_TYPE_SYNC,		SIO4_MODEL_SIO8BX2_SYNC		},
		{ SIO4_TYPE_SIO8BXS,	SIO4_FW_TYPE_Z16C30,	SIO4_MODEL_SIO8BXS			},
		{ SIO4_TYPE_SIO8BXS,	SIO4_FW_TYPE_SYNC,		SIO4_MODEL_SIO8BXS_SYNC		},
		{ -1,					-1,						SIO4_MODEL_SIO4				}
	};

	int	i;

	for (i = 0;; i++)
	{
		if (list[i].type == -1)
		{
			printf(	"%s: INTERNAL ERROR, line %d, file %s\n",
					DEV_NAME,
					__LINE__,
					__FILE__);
		}
		else if (dev->cache.sio4_type != list[i].type)
		{
			continue;
		}
		else if (dev->cache.fw_type != list[i].fw)
		{
			continue;
		}

		dev->cache.model	= (sio4_model_t) list[i].model;
		break;
	}
}



// ****************************************************************************
static void _mp_compute(chan_data_t* chan)
{
	static const struct
	{
		s32	sio4_type;
		s32	fw_type;
		s32	form_factor;
		s32	chip;
		s32	prog;
		s32	bitmap;
	} data[]	=
	{
		// sio4_type			fw_type					form_factor					chip				prog	bitmap
		{ SIO4_TYPE_SIO4,		SIO4_FW_TYPE_Z16C30,	-1,							SIO4_MP_CHIP_FIXED,	0,		RS422_485 | RS232							},	// per user manuals and specs, 1/10/2014

		{ SIO4_TYPE_SIO4A,		SIO4_FW_TYPE_Z16C30,	-1,							SIO4_MP_CHIP_FIXED,	0,		RS422_485									},	// per PCI um Rev A, 1/10/2014
		{ SIO4_TYPE_SIO4A,		SIO4_FW_TYPE_SYNC,		-1,							SIO4_MP_CHIP_FIXED,	0,		RS422_485									},

		{ SIO4_TYPE_SIO4AR,		SIO4_FW_TYPE_Z16C30,	-1,							SIO4_MP_CHIP_FIXED,	0,		RS422_485									},	// per PMC um Rev B, 1/10/2014
		{ SIO4_TYPE_SIO4AR,		SIO4_FW_TYPE_SYNC,		-1,							SIO4_MP_CHIP_FIXED,	0,		RS422_485									},	// per PMC um Rev D, 1/10/2014

		{ SIO4_TYPE_SIO4AHRM,	SIO4_FW_TYPE_Z16C30,	-1,							SIO4_MP_CHIP_FIXED,	0,		RS422_485									},
		{ SIO4_TYPE_SIO4AHRM,	SIO4_FW_TYPE_SYNC,		-1,							SIO4_MP_CHIP_FIXED,	0,		RS422_485									},

		{ SIO4_TYPE_SIO4B,		SIO4_FW_TYPE_Z16C30,	SIO4_FORM_FACTOR_PC104P,	SIO4_MP_CHIP_FIXED,	0,		RS422_485									},	// per spec Rev 2, 1/10/2014
		{ SIO4_TYPE_SIO4B,		SIO4_FW_TYPE_Z16C30,	SIO4_FORM_FACTOR_PCI,		SIO4_MP_CHIP_FIXED,	0,		RS422_485 | RS232							},	// per um Rev E.1. 1/10/2014
		{ SIO4_TYPE_SIO4B,		SIO4_FW_TYPE_Z16C30,	SIO4_FORM_FACTOR_PCI66,		SIO4_MP_CHIP_FIXED,	0,		RS422_485									},	// per spec Rev 0, 1/10/2014
		{ SIO4_TYPE_SIO4B,		SIO4_FW_TYPE_Z16C30,	-1,							SIO4_MP_CHIP_FIXED,	0,		RS422_485									},

		{ SIO4_TYPE_SIO4B,		SIO4_FW_TYPE_SYNC,		SIO4_FORM_FACTOR_PC104P,	SIO4_MP_CHIP_FIXED,	0,		RS422_485									},	// per spec Rev 2, 1/10/2014
		{ SIO4_TYPE_SIO4B,		SIO4_FW_TYPE_SYNC,		SIO4_FORM_FACTOR_PCI,		SIO4_MP_CHIP_FIXED,	0,		RS422_485									},	// per spec Rev 4, 1/10/2014
		{ SIO4_TYPE_SIO4B,		SIO4_FW_TYPE_SYNC,		SIO4_FORM_FACTOR_PCI66,		SIO4_MP_CHIP_FIXED,	0,		RS422_485									},	// per spec Rev 0, 1/10/2014
		{ SIO4_TYPE_SIO4B,		SIO4_FW_TYPE_SYNC,		-1,							SIO4_MP_CHIP_FIXED,	0,		RS422_485									},

		{ SIO4_TYPE_SIO4BX,		SIO4_FW_TYPE_Z16C30,	SIO4_FORM_FACTOR_PC104P,	SIO4_MP_CHIP_SP508,	1,		RS422_485 | RS232 | RS423 | V35A			},	// verified: PC104P-SIO4BX-256K, 9/24/2013
		{ SIO4_TYPE_SIO4BX,		SIO4_FW_TYPE_Z16C30,	SIO4_FORM_FACTOR_PCIE,		SIO4_MP_CHIP_SP508,	1,		RS422_485 | RS232							},	// per spec Rev 0, 1/10/2014
		{ SIO4_TYPE_SIO4BX,		SIO4_FW_TYPE_Z16C30,	SIO4_FORM_FACTOR_PMC,		SIO4_MP_CHIP_SP508,	1,		RS422_485 | RS232 | RS423 | RS530x | V35A	},	// verified: PMC-SIO4BX-256K, 9/23/2013
		{ SIO4_TYPE_SIO4BX,		SIO4_FW_TYPE_Z16C30,	SIO4_FORM_FACTOR_XMC,		SIO4_MP_CHIP_SP508,	1,		RS422_485 | RS232 | RS423 | V35A			},	// per spec Rev 0, 1/10/2014
		{ SIO4_TYPE_SIO4BX,		SIO4_FW_TYPE_Z16C30,	-1,							SIO4_MP_CHIP_SP508,	1,		RS422_485 | RS232							},

		{ SIO4_TYPE_SIO4BX,		SIO4_FW_TYPE_SYNC,		SIO4_FORM_FACTOR_PC104P,	SIO4_MP_CHIP_SP508,	1,		RS422_485 | RS232 | RS423 | V35A			},	// verified: PC104P-SIO4BX-SYNC-256K, 9/24/2013
		{ SIO4_TYPE_SIO4BX,		SIO4_FW_TYPE_SYNC,		SIO4_FORM_FACTOR_PCIE,		SIO4_MP_CHIP_SP508,	1,		RS422_485 | RS232							},
		{ SIO4_TYPE_SIO4BX,		SIO4_FW_TYPE_SYNC,		SIO4_FORM_FACTOR_PMC,		SIO4_MP_CHIP_SP508,	1,		RS422_485 | RS232 | RS423 | V35A			},	// verified: PMC-SIO4BX-SYNC-256K, 9/25/2013
		{ SIO4_TYPE_SIO4BX,		SIO4_FW_TYPE_SYNC,		SIO4_FORM_FACTOR_XMC,		SIO4_MP_CHIP_SP508,	1,		RS422_485 | RS232 | RS423					},	// per spec Rev 0, 1/10/2014
		{ SIO4_TYPE_SIO4BX,		SIO4_FW_TYPE_SYNC,		-1,							SIO4_MP_CHIP_SP508,	1,		RS422_485 | RS232 | RS423					},

		{ SIO4_TYPE_SIO4BX2,	SIO4_FW_TYPE_Z16C30,	SIO4_FORM_FACTOR_PCIE,		SIO4_MP_CHIP_SP508,	1,		RS422_485 | RS232							},	// per spec (Rev 1), 1/10/2014
		{ SIO4_TYPE_SIO4BX2,	SIO4_FW_TYPE_Z16C30,	-1,							SIO4_MP_CHIP_SP508,	1,		RS422_485 | RS232							},

		{ SIO4_TYPE_SIO4BX2,	SIO4_FW_TYPE_SYNC,		SIO4_FORM_FACTOR_PCIE,		SIO4_MP_CHIP_SP508,	1,		RS422_485 | RS232							},	// per spec (Rev 1), 1/10/2014
		{ SIO4_TYPE_SIO4BX2,	SIO4_FW_TYPE_SYNC,		-1,							SIO4_MP_CHIP_SP508,	1,		RS422_485 | RS232							},

		{ SIO4_TYPE_SIO4BXR,	SIO4_FW_TYPE_Z16C30,	SIO4_FORM_FACTOR_PMC66,		SIO4_MP_CHIP_SP508,	1,		RS422_485 | RS232 | RS423 | RS530x | V35A	},	// verified: PMC66-SIO4BXR-256K, 9/23/2013
		{ SIO4_TYPE_SIO4BXR,	SIO4_FW_TYPE_Z16C30,	-1,							SIO4_MP_CHIP_SP508,	1,		RS422_485 | RS232 | RS423					},

		{ SIO4_TYPE_SIO4BXR,	SIO4_FW_TYPE_SYNC,		SIO4_FORM_FACTOR_PMC66,		SIO4_MP_CHIP_SP508,	1,		RS422_485 | RS232 | RS423 | RS530x | V35A	},	// verified: PMC66-SIO4BXR-SYNC-256K, 9/23/2013
		{ SIO4_TYPE_SIO4BXR,	SIO4_FW_TYPE_SYNC,		-1,							SIO4_MP_CHIP_SP508,	1,		RS422_485 | RS232 | RS423					},

		{ SIO4_TYPE_SIO8BX2,	SIO4_FW_TYPE_Z16C30,	SIO4_FORM_FACTOR_PCIE4,		SIO4_MP_CHIP_SP508,	1,		RS422_485 | RS232							},	// verified: PCIe4-SIO8BX2, 9/20/2013
		{ SIO4_TYPE_SIO8BX2,	SIO4_FW_TYPE_Z16C30,	-1,							SIO4_MP_CHIP_SP508,	1,		RS422_485 | RS232							},

		{ SIO4_TYPE_SIO8BX2,	SIO4_FW_TYPE_SYNC,		SIO4_FORM_FACTOR_PCIE4,		SIO4_MP_CHIP_SP508,	1,		RS422_485 | RS232							},	// verified: PCIe4-SIO8BX2-SYNC, 9/21/2013
		{ SIO4_TYPE_SIO8BX2,	SIO4_FW_TYPE_SYNC,		-1,							SIO4_MP_CHIP_SP508,	1,		RS422_485 | RS232							},

		{ SIO4_TYPE_SIO8BXS,	SIO4_FW_TYPE_Z16C30,	SIO4_FORM_FACTOR_PCI,		SIO4_MP_CHIP_SP508,	1,		RS422_485 | RS232 | RS423 | V35A | RS530x	},	// verified: PCI-SIO8BXS-128K, 9/21/2013
		{ SIO4_TYPE_SIO8BXS,	SIO4_FW_TYPE_Z16C30,	-1,							SIO4_MP_CHIP_SP508,	1,		RS422_485 | RS232 | RS423					},

		{ SIO4_TYPE_SIO8BXS,	SIO4_FW_TYPE_SYNC,		SIO4_FORM_FACTOR_PCI,		SIO4_MP_CHIP_SP508,	1,		RS422_485 | RS232 | RS423 | V35A			},	// verified: PCI-SIO8BXS-SYNC-512K, 9/21/2013
		{ SIO4_TYPE_SIO8BXS,	SIO4_FW_TYPE_SYNC,		-1,							SIO4_MP_CHIP_SP508,	1,		RS422_485 | RS232 | RS423					},

		{ -1,					-1,						-1,							SIO4_MP_CHIP_FIXED,	0,		RS422_485 | RS232	},
	};

	dev_data_t*	dev	= chan->dev;
	s32			fw_type;
	int			i;

	fw_type_config_get(chan, &fw_type);

	for (i = 0;; i++)
	{
		if (data[i].sio4_type == -1)
			break;

		if (data[i].sio4_type != dev->cache.sio4_type)
			continue;

		if (data[i].fw_type == -1)
			break;

		if (data[i].fw_type != fw_type)
			continue;

		if (data[i].form_factor == -1)
			break;

		if (data[i].form_factor == dev->cache.form_factor)
			break;
	}

	dev->cache.mp_chip		= (sio4_mp_chip_t) data[i].chip;
	chan->cache.mp_program	= data[i].prog;
	chan->cache.mp_bitmap	= data[i].bitmap;

	if (chan->cache.reg_psrcr_bits & 0xF000000)
		chan->cache.mp		= 1;
	else
		chan->cache.mp		= 0;
}



// ****************************************************************************
static void _osc_compute(dev_data_t* dev)
{
	u32	type;

	switch (dev->cache.sio4_type)
	{
		default:

			printf(	"%s: INTERNAL ERROR, line %d, file %s\n",
					DEV_NAME,
					__LINE__,
					__FILE__);

			FALLTHROUGH;	/* fall through */

		case SIO4_TYPE_SIO4:

			type	= 0;
			break;

		case SIO4_TYPE_SIO4A:
		case SIO4_TYPE_SIO4AR:
		case SIO4_TYPE_SIO4AHRM:
		case SIO4_TYPE_SIO4B:
		case SIO4_TYPE_SIO4BX:
		case SIO4_TYPE_SIO4BX2:
		case SIO4_TYPE_SIO4BXR:
		case SIO4_TYPE_SIO8BX2:
		case SIO4_TYPE_SIO8BXS:

			type	= dev->cache.gsc_fr_32 & 0xF;
			break;
	}

	switch (type)
	{
		default:

			printf(	"%s: INTERNAL ERROR: type %ld, line %d, file %s\n",
					DEV_NAME,
					(long) type,
					__LINE__,
					__FILE__);

			FALLTHROUGH;	/* fall through */

		case 0:

			// A fixed oscillator.
			dev->cache.osc_chip		= SIO4_OSC_CHIP_FIXED;
			dev->cache.osc_measure	= 0;
			dev->cache.osc_per_chan	= 0;
			dev->cache.osc_program	= 0;

			dev->cache.reg_pcr		= 0;

			dev->cache.reg_pocsr	= 0;
			dev->cache.reg_porar	= 0;
			dev->cache.reg_pordr	= 0;
			dev->cache.reg_pord2r	= 0;
			break;

		case 1:

			// 1 x IDC2053B
			dev->cache.osc_chip		= SIO4_OSC_CHIP_IDC2053B;
			dev->cache.osc_measure	= 0;
			dev->cache.osc_per_chan	= 0;
			dev->cache.osc_program	= 0;

			dev->cache.reg_pcr		= 1;

			dev->cache.reg_pocsr	= 0;
			dev->cache.reg_porar	= 0;
			dev->cache.reg_pordr	= 0;
			dev->cache.reg_pord2r	= 0;
			break;

		case 2:

			// 4 x IDC2053B
			dev->cache.osc_chip		= SIO4_OSC_CHIP_IDC2053B_4;
			dev->cache.osc_measure	= 0;
			dev->cache.osc_per_chan	= 1;
			dev->cache.osc_program	= 0;

			dev->cache.reg_pcr		= 1;

			dev->cache.reg_pocsr	= 0;
			dev->cache.reg_porar	= 0;
			dev->cache.reg_pordr	= 0;
			dev->cache.reg_pord2r	= 0;
			break;

		case 3:

			// CY22393
			dev->cache.osc_chip		= SIO4_OSC_CHIP_CY22393;
			dev->cache.osc_measure	= 1;
			dev->cache.osc_per_chan	= 1;
			dev->cache.osc_program	= 1;

			dev->cache.reg_pcr		= 0;

			dev->cache.reg_pocsr	= 1;
			dev->cache.reg_porar	= 1;
			dev->cache.reg_pordr	= 1;
			dev->cache.reg_pord2r	= 0;
			break;

		case 4:

			// 2 x CY22393
			dev->cache.osc_chip		= SIO4_OSC_CHIP_CY22393_2;
			dev->cache.osc_measure	= 1;
			dev->cache.osc_per_chan	= 1;
			dev->cache.osc_program	= 1;

			dev->cache.reg_pcr		= 0;

			dev->cache.reg_pocsr	= 1;
			dev->cache.reg_porar	= 1;
			dev->cache.reg_pordr	= 1;
			dev->cache.reg_pord2r	= 1;
			break;
	}
}



// ****************************************************************************
static void _post_divider_compute(dev_data_t* dev)
{
	switch (dev->cache.sio4_type)
	{
		default:

			printf(	"%s: INTERNAL ERROR, line %d, file %s\n",
					DEV_NAME,
					__LINE__,
					__FILE__);

			FALLTHROUGH;	/* fall through */

		case SIO4_TYPE_SIO4:

			dev->cache.osc_pd_max	= 0;
			break;

		case SIO4_TYPE_SIO4A:
		case SIO4_TYPE_SIO4AR:
		case SIO4_TYPE_SIO4AHRM:

			dev->cache.osc_pd_max	= 8;
			break;

		case SIO4_TYPE_SIO4B:

			if (dev->cache.gsc_fr_32 == 0)
				dev->cache.osc_pd_max	= 11;	// Hz = F / (2**PD)	// max 2048
			else if ((dev->cache.gsc_fr_32 & 0x10) == 0)
				dev->cache.osc_pd_max	= 11;	// Hz = F / (2**PD)	// max 2048
			else if ((dev->cache.gsc_fr_32 & 0xFF00) > 0x0800)
				dev->cache.osc_pd_max	= 15;	// Hz = F / (2**PD)	// max 32768
			else
				dev->cache.osc_pd_max	= 11;	// Hz = F / (2**PD)	// max 2048

			break;

		case SIO4_TYPE_SIO4BX:
		case SIO4_TYPE_SIO4BX2:
		case SIO4_TYPE_SIO4BXR:
		case SIO4_TYPE_SIO8BX2:
		case SIO4_TYPE_SIO8BXS:

			dev->cache.osc_pd_max	= 15;	// Hz = F / (2**PD)	// max 32768
			break;
	}
}



// ****************************************************************************
static void _reg_bsr_compute(dev_data_t* dev)
{
	if (dev->cache.sio4_type == SIO4_TYPE_SIO4)
		dev->cache.reg_bsr	= 0;
	else
		dev->cache.reg_bsr	= 1;
}



// ****************************************************************************
static void _reg_ccr_compute(chan_data_t* chan)
{
	static const struct
	{
		s32	sio4_type;
		s32	fw_type;
		s32	reg_ccr;
	} data[]	=
	{
		// sio4_type			fw_type					reg_ccr
		{ SIO4_TYPE_SIO4A,		SIO4_FW_TYPE_SYNC,		0	},
		{ SIO4_TYPE_SIO4AR,		SIO4_FW_TYPE_SYNC,		0	},
		{ SIO4_TYPE_SIO4AHRM,	SIO4_FW_TYPE_SYNC,		0	},
		{ SIO4_TYPE_SIO4B,		SIO4_FW_TYPE_SYNC,		0	},
		{ SIO4_TYPE_SIO4BX,		SIO4_FW_TYPE_SYNC,		0	},
		{ SIO4_TYPE_SIO4BX2,	SIO4_FW_TYPE_SYNC,		0	},
		{ SIO4_TYPE_SIO4BXR,	SIO4_FW_TYPE_SYNC,		0	},
		{ SIO4_TYPE_SIO8BX2,	SIO4_FW_TYPE_SYNC,		0	},
		{ SIO4_TYPE_SIO8BXS,	SIO4_FW_TYPE_SYNC,		0	},

		{ SIO4_TYPE_SIO4,		SIO4_FW_TYPE_Z16C30,	1	},
		{ SIO4_TYPE_SIO4A,		SIO4_FW_TYPE_Z16C30,	1	},
		{ SIO4_TYPE_SIO4AR,		SIO4_FW_TYPE_Z16C30,	1	},
		{ SIO4_TYPE_SIO4AHRM,	SIO4_FW_TYPE_Z16C30,	1	},

		{ SIO4_TYPE_SIO4B,		SIO4_FW_TYPE_Z16C30,	-1	},
		{ SIO4_TYPE_SIO4BX,		SIO4_FW_TYPE_Z16C30,	-1	},
		{ SIO4_TYPE_SIO4BX2,	SIO4_FW_TYPE_Z16C30,	-1	},
		{ SIO4_TYPE_SIO4BXR,	SIO4_FW_TYPE_Z16C30,	-1	},
		{ SIO4_TYPE_SIO8BX2,	SIO4_FW_TYPE_Z16C30,	-1	},
		{ SIO4_TYPE_SIO8BXS,	SIO4_FW_TYPE_Z16C30,	-1	},

		{ -1,					-1,						0	}
	};

	dev_data_t*	dev		= chan->dev;
	s32			fw_type;
	int			i;

	fw_type_config_get(chan, &fw_type);

	for (i = 0;; i++)
	{
		if (data[i].sio4_type == -1)
		{
			printf(	"%s: INTERNAL ERROR, line %d, file %s\n",
					DEV_NAME,
					__LINE__,
					__FILE__);
			break;
		}

		if (dev->cache.sio4_type != data[i].sio4_type)
			continue;

		if (fw_type == data[i].fw_type)
			break;
	}

	if (data[i].reg_ccr >= 0)
	{
		chan->cache.reg_ccr	= data[i].reg_ccr;
	}
	else
	{
		if ((dev->cache.gsc_fr_32 & 0xFF00) < 0x0900)
			chan->cache.reg_ccr	= 1;	// This is a legacy feature.
		else
			chan->cache.reg_ccr	= 0;	// Feature not supported.
	}
}



// ****************************************************************************
static void _reg_fr_compute(dev_data_t* dev)
{
	switch (dev->cache.sio4_type)
	{
		default:

			printf(	"%s: INTERNAL ERROR, line %d, file %s\n",
					DEV_NAME,
					__LINE__,
					__FILE__);

			FALLTHROUGH;	/* fall through */

		case SIO4_TYPE_SIO4:

			dev->cache.reg_fr	= 0;
			break;

		case SIO4_TYPE_SIO4A:
		case SIO4_TYPE_SIO4AR:
		case SIO4_TYPE_SIO4AHRM:
		case SIO4_TYPE_SIO4B:
		case SIO4_TYPE_SIO4BX:
		case SIO4_TYPE_SIO4BX2:
		case SIO4_TYPE_SIO4BXR:
		case SIO4_TYPE_SIO8BX2:
		case SIO4_TYPE_SIO8BXS:

			if ((dev->cache.gsc_frr_32 & 0x40000000) == 0)
				dev->cache.reg_fr	= 0;	// Can't check FR bit.
			else if (dev->cache.gsc_frr_32 & 0x80000000)
				dev->cache.reg_fr	= 1;	// FR is present.
			else
				dev->cache.reg_fr	= 0;	// FR is absent.

			if (dev->cache.gsc_fr_32 == 0xFFFFFFFF)
				dev->cache.reg_fr	= 0;

			break;
	}

	if (dev->cache.reg_fr == 0)
		dev->cache.gsc_fr_32	= 0;
}



// ****************************************************************************
static void _reg_fsr_fcr_compute(dev_data_t* dev)
{
	switch (dev->cache.sio4_type)
	{
		default:

			printf(	"%s: INTERNAL ERROR, line %d, file %s\n",
					DEV_NAME,
					__LINE__,
					__FILE__);

			FALLTHROUGH;	/* fall through */

		case SIO4_TYPE_SIO4:
		case SIO4_TYPE_SIO4AR:

			dev->cache.reg_fsr	= 0;	// Feature not supported.
			break;

		case SIO4_TYPE_SIO4A:
		case SIO4_TYPE_SIO4AHRM:

			if (dev->cache.gsc_fr_32 == 0)
				dev->cache.reg_fsr	= 0;	// Can't check the feature bit.
			else if (dev->cache.gsc_fr_32 & D4)
				dev->cache.reg_fsr	= 1;	// Feature supported.
			else
				dev->cache.reg_fsr	= 0;	// Feature not supported.

			break;

		case SIO4_TYPE_SIO4B:
		case SIO4_TYPE_SIO4BX:
		case SIO4_TYPE_SIO4BX2:
		case SIO4_TYPE_SIO4BXR:
		case SIO4_TYPE_SIO8BX2:
		case SIO4_TYPE_SIO8BXS:

			if (dev->cache.gsc_fr_32 == 0)
				dev->cache.reg_fsr	= 0;	// Can't check the feature bit.
			else if ((dev->cache.gsc_fr_32 & D4) == 0)
				dev->cache.reg_fsr	= 0;	// Can't check the feature bit.
			else if (dev->cache.gsc_fr_32 & D5)
				dev->cache.reg_fsr	= 1;	// Feature supported.
			else
				dev->cache.reg_fsr	= 0;	// Feature not supported.

			break;
	}

	dev->cache.reg_fcr	= dev->cache.reg_fsr;
}



// ****************************************************************************
static void _reg_ftr_compute(dev_data_t* dev)
{
	if (dev->cache.reg_fr == 0)
		dev->cache.reg_ftr	= 0;		// Feature not supported.
	else if ((dev->cache.gsc_fr_32 & D4) == 0)
		dev->cache.reg_ftr	= 0;		// Can't check the feature bit.
	else if (dev->cache.gsc_fr_32 & D12)
		dev->cache.reg_ftr	= 1;		// Feature supported.
	else
		dev->cache.reg_ftr	= 0;		// Feature not supported.
}



// ****************************************************************************
static void _reg_gpiosr_compute(chan_data_t* chan)
{
	dev_data_t*	dev		= chan->dev;
	s32			fw_type;
	u32			reg;
	VADDR_T		va;

	switch (dev->cache.sio4_type)
	{
		default:

			printf(	"%s: INTERNAL ERROR, line %d, file %s\n",
					DEV_NAME,
					__LINE__,
					__FILE__);

			FALLTHROUGH;	/* fall through */

		case SIO4_TYPE_SIO4:
		case SIO4_TYPE_SIO4B:
		case SIO4_TYPE_SIO4BX:
		case SIO4_TYPE_SIO4BX2:
		case SIO4_TYPE_SIO4BXR:
		case SIO4_TYPE_SIO8BX2:
		case SIO4_TYPE_SIO8BXS:

			chan->cache.reg_gpiosr	= 0;
			break;

		case SIO4_TYPE_SIO4A:
		case SIO4_TYPE_SIO4AR:
		case SIO4_TYPE_SIO4AHRM:

			fw_type_config_get(chan, &fw_type);

			if (fw_type == SIO4_FW_TYPE_CONFIG_SYNC)
			{
				chan->cache.reg_gpiosr	= 0;
			}
			else
			{
				chan->cache.reg_gpiosr	= 1;
				va	= FW_VADDR(dev, SIO4_GSC_GPIOSR, 0, 0);
				reg	= os_reg_mem_rx_u32(chan->dev, va);

				if (reg == 0xFFFFFFFF)
					chan->cache.reg_gpiosr	= 0;
			}

			break;
	}
}



// ****************************************************************************
static void _reg_iocr_compute(chan_data_t* chan)
{
	dev_data_t*	dev	= chan->dev;
	s32			fw_type;

	switch (dev->cache.sio4_type)
	{
		default:

			printf(	"%s: INTERNAL ERROR, line %d, file %s\n",
					DEV_NAME,
					__LINE__,
					__FILE__);

			FALLTHROUGH;	/* fall through */

		case SIO4_TYPE_SIO4:
		case SIO4_TYPE_SIO4B:
		case SIO4_TYPE_SIO4BX:
		case SIO4_TYPE_SIO4BX2:
		case SIO4_TYPE_SIO4BXR:
		case SIO4_TYPE_SIO8BX2:
		case SIO4_TYPE_SIO8BXS:

			dev->cache.reg_iocr	= 0;
			break;

		case SIO4_TYPE_SIO4A:
		case SIO4_TYPE_SIO4AR:
		case SIO4_TYPE_SIO4AHRM:

			fw_type_config_get(chan, &fw_type);

			if (fw_type == SIO4_FW_TYPE_CONFIG_SYNC)
				dev->cache.reg_iocr	= 1;
			else
				dev->cache.reg_iocr	= 0;

			break;
	}
}



// ****************************************************************************
static void _reg_psrcr_compute(chan_data_t* chan)
{
	static const struct
	{
		s32	sio4_type;
		s32	fw_type;
		s32	form_factor;
		s32	reg_psrcr;
		u32	reg_psrcr_bits;
	} data[]	=
	{
		// sio4_type			fw_type					form_factor					reg_psrcr	reg_psrcr_bits
		{ SIO4_TYPE_SIO4,		SIO4_FW_TYPE_Z16C30,	-1,							0,			0x00000000	},

		{ SIO4_TYPE_SIO4A,		SIO4_FW_TYPE_Z16C30,	-1,							1,			0x000007FF	},
		{ SIO4_TYPE_SIO4A,		SIO4_FW_TYPE_SYNC,		-1,							0,			0x00000000	},

		{ SIO4_TYPE_SIO4AR,		SIO4_FW_TYPE_Z16C30,	-1,							1,			0x000007FF	},
		{ SIO4_TYPE_SIO4AR,		SIO4_FW_TYPE_SYNC,		-1,							0,			0x00000000	},

		{ SIO4_TYPE_SIO4AHRM,	SIO4_FW_TYPE_Z16C30,	-1,							1,			0x000007FF	},
		{ SIO4_TYPE_SIO4AHRM,	SIO4_FW_TYPE_SYNC,		-1,							0,			0x00000000	},

		{ SIO4_TYPE_SIO4B,		SIO4_FW_TYPE_Z16C30,	-1,							1,			0xBF99FFFF	},
		{ SIO4_TYPE_SIO4B,		SIO4_FW_TYPE_SYNC,		SIO4_FORM_FACTOR_PC104P,	1,			0xBF8BFFFF	},
		{ SIO4_TYPE_SIO4B,		SIO4_FW_TYPE_SYNC,		-1,							1,			0x00000000	},

		{ SIO4_TYPE_SIO4BX,		SIO4_FW_TYPE_Z16C30,	-1,							1,			0xFF9FFFFF	},
		{ SIO4_TYPE_SIO4BX,		SIO4_FW_TYPE_SYNC,		-1,							1,			0xBF99FFFF	},

		{ SIO4_TYPE_SIO4BXR,	SIO4_FW_TYPE_Z16C30,	-1,							1,			0xFF9FFFFF	},
		{ SIO4_TYPE_SIO4BXR,	SIO4_FW_TYPE_SYNC,		-1,							1,			0xBF99FFFF	},

		{ SIO4_TYPE_SIO4BX2,	SIO4_FW_TYPE_Z16C30,	-1,							1,			0xFFBFFFFF	},
		{ SIO4_TYPE_SIO4BX2,	SIO4_FW_TYPE_SYNC,		-1,							1,			0xBF99FFFF	},

		{ SIO4_TYPE_SIO8BXS,	SIO4_FW_TYPE_Z16C30,	-1,							1,			0xFFFFFFFF	},
		{ SIO4_TYPE_SIO8BXS,	SIO4_FW_TYPE_SYNC,		-1,							1,			0xBF99FFFF	},

		{ SIO4_TYPE_SIO8BX2,	SIO4_FW_TYPE_Z16C30,	-1,							1,			0xFFB9FFFF	},
		{ SIO4_TYPE_SIO8BX2,	SIO4_FW_TYPE_SYNC,		-1,							1,			0xBF99FFFF	},
		{ -1,					-1,						-1,							0,			0x00000000	}
	};

	dev_data_t*	dev		= chan->dev;
	s32			fw_type;
	int			i;

	fw_type_config_get(chan, &fw_type);

	for (i = 0;; i++)
	{
		if (data[i].sio4_type == -1)
		{
			printf(	"%s: INTERNAL ERROR, line %d, file %s\n",
					DEV_NAME,
					__LINE__,
					__FILE__);
			break;
		}

		if (data[i].sio4_type != dev->cache.sio4_type)
			continue;

		if (data[i].fw_type != fw_type)
			continue;

		if ((data[i].form_factor == -1) ||
			(data[i].form_factor == dev->cache.form_factor))
		{
			break;
		}
	}

	chan->cache.reg_psrcr		= data[i].reg_psrcr;
	chan->cache.reg_psrcr_bits	= data[i].reg_psrcr_bits;
}



// ****************************************************************************
static void _reg_pstsr_compute(chan_data_t* chan)
{
	static const struct
	{
		s32	sio4_type;
		s32	fw_type;
		s32	form_factor;
		s32	reg_pstsr;
		u32	reg_pstsr_bits;
	} data[]	=
	{
		// sio4_type			fw_type					form_factor					reg_pstsr	reg_pstsr_bits
		{ SIO4_TYPE_SIO4,		SIO4_FW_TYPE_Z16C30,	-1,							0,			0x00000000	},

		{ SIO4_TYPE_SIO4A,		SIO4_FW_TYPE_SYNC,		-1,							0,			0x00000000	},
		{ SIO4_TYPE_SIO4A,		SIO4_FW_TYPE_Z16C30,	-1,							0,			0x00000000	},

		{ SIO4_TYPE_SIO4AR,		SIO4_FW_TYPE_SYNC,		-1,							0,			0x00000000	},
		{ SIO4_TYPE_SIO4AR,		SIO4_FW_TYPE_Z16C30,	-1,							0,			0x00000000	},

		{ SIO4_TYPE_SIO4AHRM,	SIO4_FW_TYPE_SYNC,		-1,							0,			0x00000000	},
		{ SIO4_TYPE_SIO4AHRM,	SIO4_FW_TYPE_Z16C30,	-1,							0,			0x00000000	},

		{ SIO4_TYPE_SIO4B,		SIO4_FW_TYPE_SYNC,		SIO4_FORM_FACTOR_PC104P,	1,			0x000003FF	},
		{ SIO4_TYPE_SIO4B,		SIO4_FW_TYPE_SYNC,		SIO4_FORM_FACTOR_PCI,		1,			0x00000077	},
		{ SIO4_TYPE_SIO4B,		SIO4_FW_TYPE_Z16C30,	SIO4_FORM_FACTOR_PC104P,	1,			0x000003FF	},
		{ SIO4_TYPE_SIO4B,		SIO4_FW_TYPE_Z16C30,	SIO4_FORM_FACTOR_PCI,		1,			0x000003FF	},

		{ SIO4_TYPE_SIO4BX,		SIO4_FW_TYPE_SYNC,		SIO4_FORM_FACTOR_PC104P,	1,			0x000003FF	},
		{ SIO4_TYPE_SIO4BX,		SIO4_FW_TYPE_SYNC,		SIO4_FORM_FACTOR_PMC,		1,			0x000003FF	},
		{ SIO4_TYPE_SIO4BX,		SIO4_FW_TYPE_Z16C30,	SIO4_FORM_FACTOR_PC104P,	1,			0x000003FF	},
		{ SIO4_TYPE_SIO4BX,		SIO4_FW_TYPE_Z16C30,	SIO4_FORM_FACTOR_PMC,		1,			0x000003FF	},

		{ SIO4_TYPE_SIO4BXR,	SIO4_FW_TYPE_SYNC,		SIO4_FORM_FACTOR_PCI66,		1,			0x000003FF	},
		{ SIO4_TYPE_SIO4BXR,	SIO4_FW_TYPE_SYNC,		SIO4_FORM_FACTOR_PMC66,		1,			0x000003FF	},
		{ SIO4_TYPE_SIO4BXR,	SIO4_FW_TYPE_SYNC,		SIO4_FORM_FACTOR_XMC,		1,			0x000003FF	},
		{ SIO4_TYPE_SIO4BXR,	SIO4_FW_TYPE_Z16C30,	SIO4_FORM_FACTOR_PCI66,		1,			0x000003FF	},
		{ SIO4_TYPE_SIO4BXR,	SIO4_FW_TYPE_Z16C30,	SIO4_FORM_FACTOR_PMC66,		1,			0x000003FF	},
		{ SIO4_TYPE_SIO4BXR,	SIO4_FW_TYPE_Z16C30,	SIO4_FORM_FACTOR_XMC,		1,			0x000003FF	},

		{ SIO4_TYPE_SIO4BX2,	SIO4_FW_TYPE_SYNC,		SIO4_FORM_FACTOR_PCIE,		1,			0x000003FF	},
		{ SIO4_TYPE_SIO4BX2,	SIO4_FW_TYPE_Z16C30,	SIO4_FORM_FACTOR_PCIE,		1,			0x000003FF	},

		{ SIO4_TYPE_SIO8BXS,	SIO4_FW_TYPE_SYNC,		SIO4_FORM_FACTOR_PCI,		1,			0x000003FF	},
		{ SIO4_TYPE_SIO8BXS,	SIO4_FW_TYPE_Z16C30,	SIO4_FORM_FACTOR_PCI,		1,			0x00000FFF	},

		{ SIO4_TYPE_SIO8BX2,	SIO4_FW_TYPE_SYNC,		SIO4_FORM_FACTOR_PCIE4,		1,			0x000003FF	},
		{ SIO4_TYPE_SIO8BX2,	SIO4_FW_TYPE_Z16C30,	SIO4_FORM_FACTOR_PCIE4,		1,			0x000003FF	},

		{ -1,					-1,						-1,							1,			0x000003FF	}
	};

	dev_data_t*	dev		= chan->dev;
	s32			fw_type;
	int			i;

	fw_type_config_get(chan, &fw_type);

	for (i = 0;; i++)
	{
		if (data[i].sio4_type == -1)
		{
			printf(	"%s: INTERNAL ERROR, line %d, file %s\n",
					DEV_NAME,
					__LINE__,
					__FILE__);
			break;
		}

		if (data[i].sio4_type != dev->cache.sio4_type)
			continue;

		if (data[i].fw_type != fw_type)
			continue;

		if ((data[i].form_factor < 0) ||
			(data[i].form_factor == dev->cache.form_factor))
		{
			break;
		}
	}

	chan->cache.reg_pstsr		= data[i].reg_pstsr;
	chan->cache.reg_pstsr_bits	= data[i].reg_pstsr_bits;
}



// ****************************************************************************
static void _reg_rcr_compute(chan_data_t* chan)
{
	s32	fw_type;

	fw_type_config_get(chan, &fw_type);

	if (fw_type == SIO4_FW_TYPE_CONFIG_SYNC)
		chan->cache.reg_rcr	= 1;
	else
		chan->cache.reg_rcr	= 0;
}



// ****************************************************************************
static void _reg_sbr_compute(chan_data_t* chan)
{
	s32	fw_type;

	fw_type_config_get(chan, &fw_type);

	if (fw_type == SIO4_FW_TYPE_CONFIG_Z16C30)
		chan->cache.reg_sbr	= 1;
	else
		chan->cache.reg_sbr	= 0;
}



// ****************************************************************************
static void _reg_tcr_compute(chan_data_t* chan)
{
	s32	fw_type;

	fw_type_config_get(chan, &fw_type);

	if (fw_type == SIO4_FW_TYPE_CONFIG_SYNC)
		chan->cache.reg_tcr	= 1;
	else
		chan->cache.reg_tcr	= 0;
}



// ****************************************************************************
static void _rx_fifo_full_cfg_compute(chan_data_t* chan)
{
	dev_data_t*	dev		= chan->dev;
	u16			did;
	s32			fw_type;

	// The GLOBAL setting.

	switch (dev->cache.sio4_type)
	{
		default:

			printf(	"%s: INTERNAL ERROR, line %d, file %s\n",
					DEV_NAME,
					__LINE__,
					__FILE__);

			FALLTHROUGH;	/* fall through */

		case SIO4_TYPE_SIO4:
		case SIO4_TYPE_SIO4A:
		case SIO4_TYPE_SIO4AR:
		case SIO4_TYPE_SIO4AHRM:

			dev->cache.rx_fifo_full_cfg_glb	= 0;
			break;

		case SIO4_TYPE_SIO4B:		// BCR D8: Rx FIFO Stop On Full?
		case SIO4_TYPE_SIO4BX:		// BCR D8: Rx FIFO Stop On Full?
		case SIO4_TYPE_SIO4BX2:		// BCR D8: Rx FIFO Stop On Full?
		case SIO4_TYPE_SIO4BXR:		// BCR D8: Rx FIFO Stop On Full?
		case SIO4_TYPE_SIO8BX2:		// BCR D8: Rx FIFO Stop On Full?
		case SIO4_TYPE_SIO8BXS:		// BCR D8: Rx FIFO Stop On Full?

			dev->cache.rx_fifo_full_cfg_glb	= 1;
			break;
	}

	// The CHANNEL SPECIFIC setting.

	did	= os_reg_pci_rx_u16(dev, 1, 0x02);
	fw_type_config_get(chan, &fw_type);

	if (did != 0x9056)
		chan->cache.rx_fifo_full_cfg_chan	= 0;			// 9056 specific
	else if (fw_type != SIO4_FW_TYPE_CONFIG_Z16C30)
		chan->cache.rx_fifo_full_cfg_chan	= 0;			// Z16C30 specific
	else if ((dev->cache.gsc_frr_32 & 0x40000000) == 0)
		chan->cache.rx_fifo_full_cfg_chan	= 0;			// unknown format
	else if (dev->cache.gsc_frr_32 & D22)
		chan->cache.rx_fifo_full_cfg_chan	= 1;
	else
		chan->cache.rx_fifo_full_cfg_chan	= 0;			// Type is indicator.
}



// ****************************************************************************
static void _fifo_space_cfg_compute(dev_data_t* dev)
{
	switch (dev->cache.sio4_type)
	{
		default:

			printf(	"%s: INTERNAL ERROR, line %d, file %s\n",
					DEV_NAME,
					__LINE__,
					__FILE__);

			FALLTHROUGH;	/* fall through */

		case SIO4_TYPE_SIO4:
		case SIO4_TYPE_SIO4A:
		case SIO4_TYPE_SIO4AR:
		case SIO4_TYPE_SIO4AHRM:
		case SIO4_TYPE_SIO4B:
		case SIO4_TYPE_SIO4BX:
		case SIO4_TYPE_SIO4BX2:
		case SIO4_TYPE_SIO8BX2:
		case SIO4_TYPE_SIO8BXS:

			dev->cache.fifo_space_cfg	= 0;
			break;

		case SIO4_TYPE_SIO4BXR:

			dev->cache.fifo_space_cfg	= 1;
			break;
	}
}



// ****************************************************************************
// This is debug code to be used and removed at a later date.
#if 0
static void _usc_dump_reg(dev_data_t* dev, const char* name, int reg)
{
	int		offset;
	VADDR_T	va;
	u32		val;

	offset	= SIO4_OFFSET_DECODE(reg);
	va		= (VADDR_T) ((long) dev->channel[0].vaddr.usc_regs + offset);
	val		= usc_reg_vaddr_read(dev, va);
	printf("%4d. %s: reg %4s, val 0x%04X\n", __LINE__, __FUNCTION__, name, (int) val);
}
#endif



//*****************************************************************************
// This is debug code to be used and removed at a later date.
#if 0
static void _usc_dump_regs(dev_data_t* dev, int line)
{
	printf("%4d. %s: line %d ===========================\n", __LINE__, __FUNCTION__, line);
	_usc_dump_reg(dev, "CCAR", SIO4_USC_CCAR);
	_usc_dump_reg(dev, "CMR", SIO4_USC_CMR);
	_usc_dump_reg(dev, "CCSR", SIO4_USC_CCSR);
	_usc_dump_reg(dev, "CCR", SIO4_USC_CCR);
	_usc_dump_reg(dev, "PRR", SIO4_USC_PRR);
	_usc_dump_reg(dev, "SRR", SIO4_USC_SRR);
	_usc_dump_reg(dev, "TMDR", SIO4_USC_TMDR);
	_usc_dump_reg(dev, "TMCR", SIO4_USC_TMCR);
	_usc_dump_reg(dev, "CMCR", SIO4_USC_CMCR);
	_usc_dump_reg(dev, "HCR", SIO4_USC_HCR);
	_usc_dump_reg(dev, "IVR", SIO4_USC_IVR);
	_usc_dump_reg(dev, "IOCR", SIO4_USC_IOCR);
	_usc_dump_reg(dev, "ICR", SIO4_USC_ICR);
	_usc_dump_reg(dev, "DCCR", SIO4_USC_DCCR);
	_usc_dump_reg(dev, "MISR", SIO4_USC_MISR);
	_usc_dump_reg(dev, "SICR", SIO4_USC_SICR);
	_usc_dump_reg(dev, "RDR", SIO4_USC_RDR);
	_usc_dump_reg(dev, "RMR", SIO4_USC_RMR);
	_usc_dump_reg(dev, "RCSR", SIO4_USC_RCSR);
	_usc_dump_reg(dev, "RICR", SIO4_USC_RICR);
	_usc_dump_reg(dev, "RSR", SIO4_USC_RSR);
	_usc_dump_reg(dev, "RCLR", SIO4_USC_RCLR);
	_usc_dump_reg(dev, "RCCR", SIO4_USC_RCCR);
	_usc_dump_reg(dev, "TC0R", SIO4_USC_TC0R);
	_usc_dump_reg(dev, "TDR", SIO4_USC_TDR);
	_usc_dump_reg(dev, "TMR", SIO4_USC_TMR);
	_usc_dump_reg(dev, "TCSR", SIO4_USC_TCSR);
	_usc_dump_reg(dev, "TICR", SIO4_USC_TICR);
	_usc_dump_reg(dev, "TSR", SIO4_USC_TSR);
	_usc_dump_reg(dev, "TCLR", SIO4_USC_TCLR);
	_usc_dump_reg(dev, "TCCR", SIO4_USC_TCCR);
	_usc_dump_reg(dev, "TC1R", SIO4_USC_TC1R);
}
#endif



// ****************************************************************************
static void _fw_type_config_compute(dev_data_t* dev)
{
	u32	tmp;

	if (dev->cache.reg_ftr == 0)
	{
		dev->cache.fw_type_config	= 0;
	}
	else
	{
		dev->cache.fw_type_config	= 1;	// Assume for now.

		// First, can we select SYNC?
		os_reg_mem_tx_u32(dev, dev->vaddr.gsc_ftr_32, 0x04040404);
		tmp	= os_reg_mem_rx_u32(dev, dev->vaddr.gsc_ftr_32);

		if (tmp != 0x04040404)
			dev->cache.fw_type_config	= 0;

		// Second, can we select Z16C30?
		os_reg_mem_tx_u32(dev, dev->vaddr.gsc_ftr_32, 0x01010101);
		tmp	= os_reg_mem_rx_u32(dev, dev->vaddr.gsc_ftr_32);

		if (tmp != 0x01010101)
		{
			dev->cache.fw_type_config	= 0;
		}
		else
		{
			// Reset of the Zilog chips while we're here.
			// This is needed here if this is a SYNC device.

			// Prepare the PLX chip for the operation.
			os_reg_mem_tx_u32(dev, dev->vaddr.plx_intcsr_32, 0);

			// Reset the first Zilog chip.
			os_reg_mem_tx_u32(dev, dev->channel[0].vaddr.gsc_csr_32, SIO4_GSC_CSR_ZIL_RST);
			os_reg_mem_tx_u8(dev, dev->channel[0].vaddr.usc_bcr, 0);

			// Reset the second Zilog chip.
			os_reg_mem_tx_u32(dev, dev->channel[2].vaddr.gsc_csr_32, SIO4_GSC_CSR_ZIL_RST);
			os_reg_mem_tx_u8(dev, dev->channel[2].vaddr.usc_bcr, 0);

			// This is debug code to be used and removed at a later date.
			// _usc_dump_regs(dev, __LINE__);
		}

		// Restore the default firmware type.

		if ((dev->cache.fw_type_config) && (dev->cache.fw_type == SIO4_FW_TYPE_SYNC))
			os_reg_mem_tx_u32(dev, dev->vaddr.gsc_ftr_32, 0x04040404);
	}
}



// ****************************************************************************
static void _rx_fifo_overrun_compute(chan_data_t* chan)
{
	static const struct
	{
		s32	sio4_type;
		s32	fw_type;
		s32	overrun;
	} data[]	=
	{
		// sio4_type			fw_type					overrun
		{ SIO4_TYPE_SIO4,		SIO4_FW_TYPE_Z16C30,	0	},

		{ SIO4_TYPE_SIO4A,		SIO4_FW_TYPE_SYNC,		0	},
		{ SIO4_TYPE_SIO4A,		SIO4_FW_TYPE_Z16C30,	0	},

		{ SIO4_TYPE_SIO4AR,		SIO4_FW_TYPE_SYNC,		0	},
		{ SIO4_TYPE_SIO4AR,		SIO4_FW_TYPE_Z16C30,	0	},

		{ SIO4_TYPE_SIO4AHRM,	SIO4_FW_TYPE_SYNC,		0	},
		{ SIO4_TYPE_SIO4AHRM,	SIO4_FW_TYPE_Z16C30,	0	},

		{ SIO4_TYPE_SIO4B,		SIO4_FW_TYPE_SYNC,		1	},
		{ SIO4_TYPE_SIO4B,		SIO4_FW_TYPE_Z16C30,	0	},

		{ SIO4_TYPE_SIO4BX,		SIO4_FW_TYPE_SYNC,		1	},
		{ SIO4_TYPE_SIO4BX,		SIO4_FW_TYPE_Z16C30,	0	},

		{ SIO4_TYPE_SIO4BXR,	SIO4_FW_TYPE_SYNC,		1	},
		{ SIO4_TYPE_SIO4BXR,	SIO4_FW_TYPE_Z16C30,	1	},

		{ SIO4_TYPE_SIO4BX2,	SIO4_FW_TYPE_SYNC,		1	},
		{ SIO4_TYPE_SIO4BX2,	SIO4_FW_TYPE_Z16C30,	1	},

		{ SIO4_TYPE_SIO8BXS,	SIO4_FW_TYPE_SYNC,		1	},
		{ SIO4_TYPE_SIO8BXS,	SIO4_FW_TYPE_Z16C30,	0	},

		{ SIO4_TYPE_SIO8BX2,	SIO4_FW_TYPE_SYNC,		1	},
		{ SIO4_TYPE_SIO8BX2,	SIO4_FW_TYPE_Z16C30,	1	},

		{ -1,					-1,						0	}
	};

	dev_data_t*	dev		= chan->dev;
	s32			fw_type;
	int			i;

	fw_type_config_get(chan, &fw_type);

	for (i = 0;; i++)
	{
		if (data[i].sio4_type == -1)
		{
			printk(	"%s: INTERNAL ERROR, line %d, file %s\n",
					DEV_NAME,
					__LINE__,
					__FILE__);
			break;
		}

		if (data[i].sio4_type != dev->cache.sio4_type)
			continue;

		if (data[i].fw_type == fw_type)
			break;
	}

	chan->cache.rx_fifo_overrun	= data[i].overrun;
}



// ****************************************************************************
static void _rx_fifo_underrun_compute(dev_data_t* dev)
{
	switch (dev->cache.sio4_type)
	{
		default:

			printf(	"%s: INTERNAL ERROR, line %d, file %s\n",
					DEV_NAME,
					__LINE__,
					__FILE__);

			FALLTHROUGH;	/* fall through */

		case SIO4_TYPE_SIO4:
		case SIO4_TYPE_SIO4A:
		case SIO4_TYPE_SIO4AR:
		case SIO4_TYPE_SIO4AHRM:
		case SIO4_TYPE_SIO4B:
		case SIO4_TYPE_SIO4BX:
		case SIO4_TYPE_SIO4BX2:
		case SIO4_TYPE_SIO8BXS:

			dev->cache.rx_fifo_underrun	= 0;
			break;

		case SIO4_TYPE_SIO4BXR:			// CSR D18: Rx FIFO Underrun
		case SIO4_TYPE_SIO8BX2:			// CSR D18: Rx FIFO Underrun

			dev->cache.rx_fifo_underrun	= 1;
			break;
	}
}



// ****************************************************************************
static void _rx_status_word_compute(chan_data_t* chan)
{
	static const struct
	{
		s32	sio4_type;
		s32	fw_type;
		s32	status;
	} data[]	=
	{
		// sio4_type			fw_type					status
		{ SIO4_TYPE_SIO4,		SIO4_FW_TYPE_Z16C30,	0	},

		{ SIO4_TYPE_SIO4A,		SIO4_FW_TYPE_SYNC,		0	},
		{ SIO4_TYPE_SIO4A,		SIO4_FW_TYPE_Z16C30,	0	},

		{ SIO4_TYPE_SIO4AR,		SIO4_FW_TYPE_SYNC,		0	},
		{ SIO4_TYPE_SIO4AR,		SIO4_FW_TYPE_Z16C30,	0	},

		{ SIO4_TYPE_SIO4AHRM,	SIO4_FW_TYPE_SYNC,		0	},
		{ SIO4_TYPE_SIO4AHRM,	SIO4_FW_TYPE_Z16C30,	0	},

		{ SIO4_TYPE_SIO4B,		SIO4_FW_TYPE_SYNC,		0	},
		{ SIO4_TYPE_SIO4B,		SIO4_FW_TYPE_Z16C30,	-1	},

		{ SIO4_TYPE_SIO4BX,		SIO4_FW_TYPE_SYNC,		0	},
		{ SIO4_TYPE_SIO4BX,		SIO4_FW_TYPE_Z16C30,	-1	},

		{ SIO4_TYPE_SIO4BXR,	SIO4_FW_TYPE_SYNC,		0	},
		{ SIO4_TYPE_SIO4BXR,	SIO4_FW_TYPE_Z16C30,	-1	},

		{ SIO4_TYPE_SIO4BX2,	SIO4_FW_TYPE_SYNC,		0	},
		{ SIO4_TYPE_SIO4BX2,	SIO4_FW_TYPE_Z16C30,	-1	},

		{ SIO4_TYPE_SIO8BXS,	SIO4_FW_TYPE_SYNC,		0	},
		{ SIO4_TYPE_SIO8BXS,	SIO4_FW_TYPE_Z16C30,	-1	},

		{ SIO4_TYPE_SIO8BX2,	SIO4_FW_TYPE_SYNC,		0	},
		{ SIO4_TYPE_SIO8BX2,	SIO4_FW_TYPE_Z16C30,	-1	},

		{ -1,					-1,						0	}
	};

	dev_data_t*	dev		= chan->dev;
	s32			fw_type;
	int			i;

	fw_type_config_get(chan, &fw_type);

	for (i = 0;; i++)
	{
		if (data[i].sio4_type == -1)
		{
			printf(	"%s: INTERNAL ERROR, line %d, file %s\n",
					DEV_NAME,
					__LINE__,
					__FILE__);
			break;
		}

		if (data[i].sio4_type != dev->cache.sio4_type)
			continue;

		if (data[i].fw_type == fw_type)
			break;
	}

	if (data[i].status >= 0)
	{
		chan->cache.rx_status_word	= data[i].status;
	}
	else
	{
		if (dev->cache.reg_fr == 0)
			chan->cache.rx_status_word	= 0;
		else if ((dev->cache.gsc_fr_32 & D4) == 0)
			chan->cache.rx_status_word	= 0;
		else if (dev->cache.gsc_fr_32 & D20)
			chan->cache.rx_status_word	= 1;
		else
			chan->cache.rx_status_word	= 0;
	}
}



// ****************************************************************************
static void _sio4_type_compute(dev_data_t* dev)
{
	typedef struct
	{
		s32			form_factor;
		u32			didr;	// Device ID Register
		s32			field;
		sio4_type_t	sio4_type;
		const char*	model;
		s32			ch_qty;
		s32			dev_qty;
		s32			new_ff;
	} type_data_t;

	static const type_data_t	list_1[]	=
	{
		// form_factor				didr	field	sio4_type			model		ch_qty	dev_qty	new_ff

		{ SIO4_FORM_FACTOR_PCI,		0x9080,	0,		SIO4_TYPE_SIO4A,	"SIO4A",	4,		1,		-1						},
		{ SIO4_FORM_FACTOR_PMC,		0x9080,	0,		SIO4_TYPE_SIO4AR,	"SIO4AR",	4,		1,		-1						},
		{ SIO4_FORM_FACTOR_CPCI,	0x9080,	0,		SIO4_TYPE_SIO4AHRM,	"SIO4AHRM",	4,		1,		-1						},
		{ -1,						~0,		-1,		SIO4_TYPE_SIO4,		"SIO4",		4,		1,		-1						},
	};

	static const type_data_t	list_2[]	=
	{
		// form_factor				didr	field	sio4_type			model		ch_qty	dev_qty	new_ff

		{ SIO4_FORM_FACTOR_PCI,		0x9056,	2,		SIO4_TYPE_SIO4BXR,	"SIO4BXR",	4,		1,		SIO4_FORM_FACTOR_PCI66	},

		{ SIO4_FORM_FACTOR_PCI,		0x9080,	0,		SIO4_TYPE_SIO4B,	"SIO4B",	4,		1,		-1						},
		{ SIO4_FORM_FACTOR_PCI,		0x9080,	1,		SIO4_TYPE_SIO4BX,	"SIO4BX",	4,		1,		-1						},
		{ SIO4_FORM_FACTOR_PCI,		0x9080,	2,		SIO4_TYPE_SIO8BXS,	"SIO8BXS",	8,		2,		-1						},
		{ SIO4_FORM_FACTOR_PCI,		0x9080,	3,		SIO4_TYPE_SIO4B,	"SIO4B",	4,		1,		SIO4_FORM_FACTOR_PCI66	},

		{ SIO4_FORM_FACTOR_PMC,		0x9080,	0,		SIO4_TYPE_SIO4AR,	"SIO4AR",	4,		1,		-1						},
		{ SIO4_FORM_FACTOR_PMC,		0x9080,	1,		SIO4_TYPE_SIO4BX,	"SIO4BX",	4,		1,		-1						},
		{ SIO4_FORM_FACTOR_PMC,		0x9056,	2,		SIO4_TYPE_SIO4BXR,	"SIO4BXR",	4,		1,		SIO4_FORM_FACTOR_PMC66	},

		{ SIO4_FORM_FACTOR_CPCI,	0x9080,	0,		SIO4_TYPE_SIO4A,	"SIO4A",	4,		1,		-1						},

		{ SIO4_FORM_FACTOR_PC104P,	0x9080,	0,		SIO4_TYPE_SIO4B,	"SIO4B",	4,		1,		-1						},
		{ SIO4_FORM_FACTOR_PC104P,	0x9080,	1,		SIO4_TYPE_SIO4BX,	"SIO4BX",	4,		1,		-1						},

		{ SIO4_FORM_FACTOR_PCIE,	0x9056,	0,		SIO4_TYPE_SIO8BX2,	"SIO8BX2",	8,		2,		SIO4_FORM_FACTOR_PCIE4	},
		{ SIO4_FORM_FACTOR_PCIE,	0x9056,	1,		SIO4_TYPE_SIO4BX2,	"SIO4BX2",	4,		1,		-1						},

		{ SIO4_FORM_FACTOR_XMC,		0x9056,	2,		SIO4_TYPE_SIO4BX,	"SIO4BX",	4,		1,		-1						},

		{ SIO4_FORM_FACTOR_UNKNOWN,	0x9080,	-1,		SIO4_TYPE_SIO4,		"SIO4",		4,		1,		-1						},
		{ -1,						~0,		-1,		SIO4_TYPE_SIO4,		"SIO4",		4,		1,		-1						},
	};

	u16					didr;
	s32					field;
	int					i;
	const type_data_t*	list	= NULL;

	if ((dev->cache.gsc_frr_32 & 0xFFFF0000) == 0xFFFF0000)
	{
		// An original SIO4.
		strcpy(dev->model_buf, "SIO4");
		dev->cache.sio4_type	= SIO4_TYPE_SIO4;
		dev->cache.channel_qty	= 4;
		dev->cache.device_qty	= 1;
	}
	else if ((dev->cache.gsc_frr_32 & 0x40000000) == 0)
	{
		// The FRR format is unknown. Presume it is an SIO4A.
		strcpy(dev->model_buf, "SIO4A");
		dev->cache.sio4_type	= SIO4_TYPE_SIO4A;
		dev->cache.channel_qty	= 4;
		dev->cache.device_qty	= 1;
	}
	else if ((dev->cache.gsc_frr_32 & 0x80000000) == 0)
	{
		// The Features Register is unsupported. Presume it is an SIO4A.
		strcpy(dev->model_buf, "SIO4A");
		dev->cache.sio4_type	= SIO4_TYPE_SIO4A;
		dev->cache.channel_qty	= 4;
		dev->cache.device_qty	= 1;
	}
	else if (	((dev->cache.gsc_fr_32 & 0xF) == 1) ||
				((dev->cache.gsc_fr_32 & 0xF) == 2))
	{
		// This field is supported on all FR implementations.
		// Only the SIO4A used these oscillator configurations.
		list	= list_1;
	}
	else
	{
		// This is an SIO4B or later device.
		list	= list_2;
	}

	if (list)
	{
		didr	= os_reg_pci_rx_u16(dev, 1, 0x02);
		field	= GSC_FIELD_DECODE(dev->cache.gsc_frr_32, 23, 20);

		for (i = 0;; i++)
		{
			if (list[i].form_factor == -1)
				break;

			if (list[i].didr != (u32) didr)
				continue;

			if (list[i].form_factor != dev->cache.form_factor)
				continue;

			if ((list[i].field == -1) || (list[i].field == field))
				break;
		}

		strcpy(dev->model_buf, list[i].model);
		dev->cache.sio4_type	= list[i].sio4_type;
		dev->cache.channel_qty	= list[i].ch_qty;
		dev->cache.device_qty	= list[i].dev_qty;

		if (list[i].new_ff >= 0)
			dev->cache.form_factor	= (sio4_form_factor_t) list[i].new_ff;
	}

	dev->model	= dev->model_buf;
}



// ****************************************************************************
static void _time_stamp_compute(dev_data_t* dev)
{
	switch (dev->cache.sio4_type)
	{
		default:

			printf(	"%s: INTERNAL ERROR, line %d, file %s\n",
					DEV_NAME,
					__LINE__,
					__FILE__);

			FALLTHROUGH;	/* fall through */

		case SIO4_TYPE_SIO4:
		case SIO4_TYPE_SIO4A:
		case SIO4_TYPE_SIO4AR:
		case SIO4_TYPE_SIO4AHRM:

			dev->cache.reg_tsr		= 0;
			dev->cache.time_stamp	= 0;
			break;

		case SIO4_TYPE_SIO4B:
		case SIO4_TYPE_SIO4BX:
		case SIO4_TYPE_SIO4BX2:
		case SIO4_TYPE_SIO4BXR:
		case SIO4_TYPE_SIO8BX2:
		case SIO4_TYPE_SIO8BXS:

			if (dev->cache.reg_fr == 0)
			{
				dev->cache.reg_tsr		= 0;
				dev->cache.time_stamp	= 0;
			}
			else if ((dev->cache.gsc_fr_32 & D4) == 0)
			{
				dev->cache.reg_tsr		= 0;
				dev->cache.time_stamp	= 0;
			}
			else if (dev->cache.gsc_fr_32 & (D18 | D19))
			{
				dev->cache.reg_tsr		= 1;
				dev->cache.time_stamp	= 1;
			}
			else
			{
				dev->cache.reg_tsr		= 0;
				dev->cache.time_stamp	= 0;
			}

			break;
	}
}



// ****************************************************************************
static void _tx_fifo_empty_cfg_compute(chan_data_t* chan)
{
	s32	fw_type;

	fw_type_config_get(chan, &fw_type);

	if (fw_type == SIO4_FW_TYPE_CONFIG_SYNC)
		chan->cache.tx_fifo_empty_cfg	= 1;
	else
		chan->cache.tx_fifo_empty_cfg	= 0;
}



// ****************************************************************************
static void _tx_fifo_overrun_compute(dev_data_t* dev)
{
	switch (dev->cache.sio4_type)
	{
		default:

			printf(	"%s: INTERNAL ERROR, line %d, file %s\n",
					DEV_NAME,
					__LINE__,
					__FILE__);

			FALLTHROUGH;	/* fall through */

		case SIO4_TYPE_SIO4:
		case SIO4_TYPE_SIO4A:
		case SIO4_TYPE_SIO4AR:
		case SIO4_TYPE_SIO4AHRM:
		case SIO4_TYPE_SIO4B:
		case SIO4_TYPE_SIO4BX:
		case SIO4_TYPE_SIO8BXS:

			dev->cache.tx_fifo_overrun		= 0;
			break;

		case SIO4_TYPE_SIO4BX2:				// CSR D17: Tx FIFO Overrun
		case SIO4_TYPE_SIO4BXR:				// CSR D17: Tx FIFO Overrun
		case SIO4_TYPE_SIO8BX2:				// CSR D17: Tx FIFO Overrun

			dev->cache.tx_fifo_overrun		= 1;
			break;
	}
}



// ****************************************************************************
static void _user_jumpers_compute(dev_data_t* dev)
{
	u32			bsr;
	const char*	sub;

	switch (dev->cache.sio4_type)
	{
		default:

			printf(	"%s: INTERNAL ERROR, line %d, file %s\n",
					DEV_NAME,
					__LINE__,
					__FILE__);

			FALLTHROUGH;	/* fall through */

		case SIO4_TYPE_SIO4:				// verified on PCI, PMC

			sub	= "";
			dev->cache.user_jumper_qty		= 0;
			dev->cache.user_jumper_sense	= 0;
			dev->cache.user_jumper_val		= 0;
			break;

		case SIO4_TYPE_SIO4A:				// BSR D0, D1 - verified on PCI, PMC
		case SIO4_TYPE_SIO4AR:
		case SIO4_TYPE_SIO4AHRM:

			sub	= "";
			bsr	= os_reg_mem_rx_u32(dev, dev->vaddr.gsc_bsr_32);
			bsr	&= 0x3;
			dev->cache.user_jumper_qty		= 2;
			dev->cache.user_jumper_sense	= 0;
			dev->cache.user_jumper_val		= (s32) bsr;
			break;

		case SIO4_TYPE_SIO4B:				// BSR D0, D1 - verified on PCI

			sub	= "";
			bsr	= os_reg_mem_rx_u32(dev, dev->vaddr.gsc_bsr_32);
			bsr	&= 0x3;
			dev->cache.user_jumper_qty		= 2;
			dev->cache.user_jumper_val		= (s32) bsr;
			dev->cache.user_jumper_sense	= 1;
			break;

		case SIO4_TYPE_SIO4BX:				// BSR D0, D1 - verified on PMC, PC/104+

			sub	= "";
			bsr	= os_reg_mem_rx_u32(dev, dev->vaddr.gsc_bsr_32);
			bsr	&= 0x3;
			dev->cache.user_jumper_qty		= 2;
			dev->cache.user_jumper_sense	= 1;
			dev->cache.user_jumper_val		= (s32) bsr;
			break;

		case SIO4_TYPE_SIO4BX2:			// BSR D0, D1, D2, D3

			sub	= "";
			bsr	= os_reg_mem_rx_u32(dev, dev->vaddr.gsc_bsr_32);
			bsr	&= 0xF;
			dev->cache.user_jumper_qty		= 4;
			dev->cache.user_jumper_sense	= 1;
			dev->cache.user_jumper_val		= (s32) bsr;
			break;

		case SIO4_TYPE_SIO4BXR:			// BSR D0, D1, D2, D3 - verified on PMC

			sub	= "";
			bsr	= os_reg_mem_rx_u32(dev, dev->vaddr.gsc_bsr_32);
			bsr	&= 0xF;
			dev->cache.user_jumper_qty		= 4;
			dev->cache.user_jumper_sense	= 1;
			dev->cache.user_jumper_val		= (s32) bsr;
			break;

		case SIO4_TYPE_SIO8BX2:			// BSR D0, D1, D2, D3 - verified on PCIe4-SIO8BX2

			bsr	= os_reg_mem_rx_u32(dev, dev->vaddr.gsc_bsr_32);
			bsr	&= 0xF;
			sub	= dev->cache.index_subdevice ? ".1" : ".0";
			dev->cache.user_jumper_qty		= 4;
			dev->cache.user_jumper_sense	= 1;
			dev->cache.user_jumper_val		=(s32) bsr;
			break;

		case SIO4_TYPE_SIO8BXS:			// BSR D1, D3, D4, D5 - verified on PCI

			bsr	= os_reg_mem_rx_u32(dev, dev->vaddr.gsc_bsr_32);
			bsr	= ((bsr >> 1) & 0x1) | ((bsr >> 2) & 0xE);
			sub	= dev->cache.index_subdevice ? ".1" : ".0";
			dev->cache.user_jumper_qty		= 4;
			dev->cache.user_jumper_sense	= 1;
			dev->cache.user_jumper_val		= (s32) bsr;
			break;
	}

	sprintf(dev->proc_id_str, "0x%lX", (long) dev->cache.user_jumper_val);
	strcat(dev->proc_id_str, sub);
}



// ****************************************************************************
static void _channel_data_init_pre(dev_data_t* dev, int index)
{
	chan_data_t*	chan	= &dev->channel[index];

	os_sem_create(&chan->sem);
	chan->dev			= dev;
	chan->in_use		= 0;
	chan->index			= index;
	chan->dev_index		= dev->board_index * 4 + index + 1;
	chan->irq.driver	= 0;
	chan->irq.notify	= 0;
	chan->irq.status	= 0;
	chan->irq.fasync	= NULL;
}



// ****************************************************************************
static void _channel_data_init_post(dev_data_t* dev, int index)
{
	chan_data_t*	chan	= &dev->channel[index];
	u32				mask;

	chan->vaddr.gsc_tar_32		= (VADDR_T) ((long) dev->vaddr.gsc_regs + 0x10 + ((long) 0x10 * index));
	chan->vaddr.gsc_rar_32		= (VADDR_T) ((long) dev->vaddr.gsc_regs + 0x14 + ((long) 0x10 * index));
	chan->vaddr.gsc_fdr_8		= (VADDR_T) ((long) dev->vaddr.gsc_regs + 0x18 + ((long) 0x10 * index));
	chan->vaddr.gsc_csr_32		= (VADDR_T) ((long) dev->vaddr.gsc_regs + 0x1C + ((long) 0x10 * index));
	chan->vaddr.gsc_sbr_32		= (VADDR_T) ((long) dev->vaddr.gsc_regs + 0x50 + ((long) 0x04 * index));
	chan->vaddr.gsc_psrcr_32	= (VADDR_T) ((long) dev->vaddr.gsc_regs + 0x80 + ((long) 0x04 * index));
	chan->vaddr.gsc_fcr_32		= (VADDR_T) ((long) dev->vaddr.gsc_regs + 0xD0 + ((long) 0x04 * index));
	chan->vaddr.gsc_fsr_32		= (VADDR_T) ((long) dev->vaddr.gsc_regs + 0xE0 + ((long) 0x04 * index));

	chan->vaddr.usc_regs		= (VADDR_T) ((long) dev->vaddr.gsc_regs + 0x100 + ((long) 0x100 * index));
	chan->vaddr.usc_bcr			= (VADDR_T) ((long) chan->vaddr.usc_regs);
	chan->vaddr.usc_ccar		= (VADDR_T) ((long) chan->vaddr.usc_regs);
	chan->vaddr.usc_hcr			= (VADDR_T) ((long) chan->vaddr.usc_regs + 18);
	chan->vaddr.usc_iocr		= (VADDR_T) ((long) chan->vaddr.usc_regs + 22);

	chan->cache.bsr_rx_fifo		= 0x00000200UL << (2 * index);
	chan->cache.bsr_tx_fifo		= 0x00000100UL << (2 * index);
	chan->cache.gsc_ccr_mask	= 0x0000000FUL << (4 * index);
	chan->cache.gsc_fdr_offset	= (u32) (0x18 + (0x10 * index));

	if (dev->cache.irq_32)
		mask	= 0x000F000F;
	else
		mask	= 0x0000000F;

	if (dev->cache.reg_fsr)
		chan->cache.gsc_fsr_32	= os_reg_mem_rx_u32(dev, chan->vaddr.gsc_fsr_32);
	else
		chan->cache.gsc_fsr_32	= 0;

	chan->cache.gsc_ielr_defalt	= mask << (4 * index);
	chan->cache.gsc_ihlr_defalt	= mask << (4 * index);
	chan->cache.gsc_irq_mask	= mask << (4 * index);
	chan->cache.gsc_irq_shift	= 4 * index;

	io_dev_data_t_init(dev, index, 0);
	io_dev_data_t_init(dev, index, 1);

	_fifo_size_compute(chan, index);
}



//*****************************************************************************
int channel_features_init(chan_data_t* chan)
{
	_reg_ccr_compute(chan);
	_reg_gpiosr_compute(chan);
	_reg_iocr_compute(chan);
	_reg_psrcr_compute(chan);
	_reg_pstsr_compute(chan);
	_reg_rcr_compute(chan);
	_reg_sbr_compute(chan);
	_reg_tcr_compute(chan);

	_mp_compute(chan);				// After PSRCR
	_rx_fifo_full_cfg_compute(chan);
	_rx_fifo_overrun_compute(chan);
	_rx_status_word_compute(chan);
	_tx_fifo_empty_cfg_compute(chan);
	return(0);
}



/******************************************************************************
*
*	Function:	dev_device_create
*
*	Purpose:
*
*		Perform a one-time initialization of the device's main data structure.
*
*	Arguments:
*
*		dev		The structure for the device.
*
*	Returned:
*
*		0		All went well.
*		< 0		An appropriate error status.
*
******************************************************************************/

int dev_device_create(dev_data_t* dev)
{
	int	ret;
	int	tmp;

	// Control ISR access to the device and data structure.
	ret	= os_spinlock_create(&dev->spinlock);

	// vaddr initialization
	dev->vaddr.plx_marbr_32		= PLX_VADDR(dev, 0x08);
	dev->vaddr.plx_intcsr_32	= PLX_VADDR(dev, 0x68);
	dev->vaddr.plx_dmaarb_32	= PLX_VADDR(dev, 0xAC);
	dev->vaddr.plx_dmathr_32	= PLX_VADDR(dev, 0xB0);

	dev->vaddr.gsc_frr_32		= GSC_VADDR(dev, 0x00);
	dev->vaddr.gsc_bcr_32		= GSC_VADDR(dev, 0x04);
	dev->vaddr.gsc_bsr_32		= GSC_VADDR(dev, 0x08);
	dev->vaddr.gsc_ccr_32		= GSC_VADDR(dev, 0x0C);
	dev->vaddr.gsc_icr_32		= GSC_VADDR(dev, 0x60);
	dev->vaddr.gsc_isr_32		= GSC_VADDR(dev, 0x64);
	dev->vaddr.gsc_ielr_32		= GSC_VADDR(dev, 0x68);
	dev->vaddr.gsc_ihlr_32		= GSC_VADDR(dev, 0x6C);
	dev->vaddr.gsc_ftr_32		= GSC_VADDR(dev, 0xF8);
	dev->vaddr.gsc_fr_32		= GSC_VADDR(dev, 0xFC);

	// These are for SIO4s with the CY22393 programmmable oscillator.
	dev->vaddr.gsc_porar_32		= GSC_VADDR(dev, 0xA0);
	dev->vaddr.gsc_pordr_32		= GSC_VADDR(dev, 0xA4);
	dev->vaddr.gsc_pocsr_32		= GSC_VADDR(dev, 0xA8);
	dev->vaddr.gsc_pord2r_32	= GSC_VADDR(dev, 0xAC);

	os_sem_create(&dev->dma.sem);
	_dev_dma_t_init(dev, 0);
	_dev_dma_t_init(dev, 1);

	// device level cache initialization
	dev->cache.gsc_fr_32		= os_reg_mem_rx_u32(dev, dev->vaddr.gsc_fr_32);
	dev->cache.gsc_frr_32		= os_reg_mem_rx_u32(dev, dev->vaddr.gsc_frr_32);

	_form_factor_compute(dev);		// First
	_sio4_type_compute(dev);		// Second
	_fw_type_compute(dev);			// Third
	_model_compute(dev);			// Forth
	_reg_fr_compute(dev);			// Fifth
	_osc_compute(dev);				// Sixth
	_bus_speed_compute(dev);		// Seventh

	_board_reset_compute(dev);		// After FR - performs a device reset if supported.
	_bus_width_compute(dev);
	_dmdma_scd_compute(dev);		// After FR
	_fifo_space_cfg_compute(dev);
	_index_subdevice_compute(dev);	// After BSR
	_irq_compute(dev);
	_legacy_cable_compute(dev);		// After FR
	_post_divider_compute(dev);		// After FR
	_time_stamp_compute(dev);
	_user_jumpers_compute(dev);		// After BSR and _index_subdevice_compute
	_led_compute(dev);
	_reg_ftr_compute(dev);			// After FR
	_reg_bsr_compute(dev);
	_reg_fsr_fcr_compute(dev);		// After FR
	_rx_fifo_underrun_compute(dev);
	_tx_fifo_overrun_compute(dev);

	// channel level initialization
	_channel_data_init_pre(dev, 0);
	_channel_data_init_pre(dev, 1);
	_channel_data_init_pre(dev, 2);
	_channel_data_init_pre(dev, 3);

	channel_features_init(&dev->channel[0]);
	channel_features_init(&dev->channel[1]);
	channel_features_init(&dev->channel[2]);
	channel_features_init(&dev->channel[3]);

	_channel_data_init_post(dev, 0);
	_channel_data_init_post(dev, 1);
	_channel_data_init_post(dev, 2);
	_channel_data_init_post(dev, 3);

	_fw_type_config_compute(dev);	// After FTR and post init.

	channel_features_init(&dev->channel[0]);
	channel_features_init(&dev->channel[1]);
	channel_features_init(&dev->channel[2]);
	channel_features_init(&dev->channel[3]);

	// more device level initialization
	tmp	= os_irq_create(dev);
	ret	= ret ? ret : tmp;

	osc_startup(dev);
	mp_startup(dev);
	device_reset(dev);
	return(ret);
}



/******************************************************************************
*
*	Function:	device_reset_ioctl
*
*	Purpose:
*
*		Implement the Reset Device IOCTL service.
*
*	Arguments:
*
*		dev		The structure for the device to initialize.
*
*		arg		The argument passed by the application.
*
*	Returned:
*
*		0		All went well.
*		< 0		An appropriate error status.
*
******************************************************************************/

s32 device_reset_ioctl(chan_data_t*chan, void* arg)
{
	device_reset(chan->dev);
	return(0);
}



/******************************************************************************
*
*	Function:	device_reset
*
*	Purpose:
*
*		Reset the entire device.
*
*	Arguments:
*
*		dev		The structure for the device to initialize.
*
*	Returned:
*
*		None.
*
******************************************************************************/

void device_reset(dev_data_t *dev)
{
	sio4_osc_t	osc;

	// Prepare the PLX chip for the operation.
	os_reg_mem_tx_u32(dev, dev->vaddr.plx_intcsr_32, 0);

	// Reset the device if it is supported.

	if (dev->cache.board_reset)
	{
		os_reg_mem_tx_u32(	dev,
							dev->vaddr.gsc_bcr_32,
							SIO4_GSC_BCR_BRD_RST);
	}

	if (dev->cache.model_z16c30)
	{
		// Reset the Zilog chips.
		zilog_reset_ioctl(&dev->channel[0], NULL);
		zilog_reset_ioctl(&dev->channel[2], NULL);
	}

	// Reset the channels.
	channel_reset_ioctl(&dev->channel[0], NULL);
	channel_reset_ioctl(&dev->channel[1], NULL);
	channel_reset_ioctl(&dev->channel[2], NULL);
	channel_reset_ioctl(&dev->channel[3], NULL);

	// Reset the programmable oscillator.
	osc_reset_ioctl(&dev->channel[0], &osc);
	osc_reset_ioctl(&dev->channel[1], &osc);
	osc_reset_ioctl(&dev->channel[2], &osc);
	osc_reset_ioctl(&dev->channel[3], &osc);

	// Reset the channel independent resources.
	os_reg_mem_tx_u32(dev, dev->vaddr.gsc_bcr_32, 0);

	// Prepare the PLX chip for use.
	os_reg_mem_tx_u32(dev, dev->vaddr.plx_intcsr_32, 0xD0900);
}



