// $URL: http://subversion:8080/svn/gsc/trunk/drivers/LINUX/SIO4%20and%20SIO8/SIO4_Linux_1.x.x_GSC_DN/async/samples/asyncc2c/utils.c $
// $Rev: 53082 $
// $Date: 2023-06-13 10:10:44 -0500 (Tue, 13 Jun 2023) $

// SIO4: Asynchronous Protocol Library: Sample Application: source file

#include "main.h"



//*****************************************************************************
static int _channel_setup(
	int				fd,
	const char*		title,
	int				dce,
	sio4_mp_prot_t	mp,
	int				lb,
	u32				cable,
	s32*			bit_rate)
{
	#define	CHAR_SIZE	SIO4_USC_CHAR_LENGTH_8
	#define	PARITY		0	// 0 = NO, !0 = YES
	#define	_STOP_BITS	SIO4_USC_CMR_AT_STOPBITS_1

	sio4_async_t	async;
	char			buf[64];
	int				errs	= 0;

	sprintf(buf, "%s Configuration", title);
	gsc_label(buf);

	memset(&async, 0, sizeof(async));

	async.cable.dcedte.enable	= 1;
	async.cable.dcedte.dce		= dce;
	async.cable.legacy.config	= cable;
	async.cable.cts				= SIO4_CTS_CABLE_CONFIG_CTS_IN;
	async.cable.dcd				= SIO4_DCD_CABLE_CONFIG_DCD_IN;
	async.cable.xceiver			= mp;

	async.loopback.enable		= lb;
	async.loopback.internal		= 1;

	async.osc.ref_comp			= 0;	// This will be updated.
	async.osc.ref_default		= SIO4_OSC_FREQ_REF_DEFAULT;
	async.osc.ref_got			= 0;	// This will be updated.

	async.rx.ae					= 16;
	async.rx.af					= 16;
	async.rx.baud_comp			= 0;	// This will be updated.
	async.rx.baud_want			= bit_rate[0];
	async.rx.char_size			= CHAR_SIZE;
	async.rx.enable				= 1;
	async.rx.encoding			= SIO4_USC_ENCODING_NRZ;
	async.rx.fifo_full_chan		= SIO4_RX_FIFO_FULL_CFG_CHAN_OVER;
	async.rx.fifo_full_glb		= SIO4_RX_FIFO_FULL_CFG_GLB_OVER;
	async.rx.io_mode			= SIO4_IO_MODE_PIO;
	async.rx.io_mode_force		= 0;
	async.rx.parity_enable		= PARITY;
	async.rx.parity_type		= SIO4_USC_PARTYPE_EVEN;
	async.rx.timeout			= (async.rx.io_mode == SIO4_IO_MODE_PIO) ? 0 : 5;
	async.rx.usc_rxc_config		= SIO4_RXC_USC_CONFIG_IN_PRG_CLK;
	async.rx.usc_rxc_rate		= SIO4_OSC_FREQ_REF_DEFAULT;

	async.tx.ae					= 16;
	async.tx.af					= 16;
	async.tx.baud_comp			= 0;	// This will be updated.
	async.tx.baud_want			= bit_rate[0];
	async.tx.cable_clock		= SIO4_TX_CABLE_CLOCK_CONFIG_PRG_CLK;
	async.tx.cable_data			= SIO4_TX_CABLE_DATA_CONFIG_USC_TXD;
	async.tx.char_size			= CHAR_SIZE;
	async.tx.enable				= 1;
	async.tx.encoding			= SIO4_USC_ENCODING_NRZ;
	async.tx.idle_output		= SIO4_USC_TCSR_TXIDLE_DEFAULT;
	async.tx.io_mode			= SIO4_IO_MODE_PIO;
	async.tx.io_mode_force		= 0;
	async.tx.parity_enable		= PARITY;
	async.tx.parity_type		= SIO4_USC_PARTYPE_EVEN;
	async.tx.shave_bits			= 0;
	async.tx.stop_bits			= _STOP_BITS;
	async.tx.timeout			= (async.tx.io_mode == SIO4_IO_MODE_PIO) ? 0 : 5;
	async.tx.usc_txc_config		= SIO4_TXC_USC_CONFIG_IN_PRG_CLK;
	async.tx.usc_txc_rate		= SIO4_OSC_FREQ_REF_DEFAULT;

	errs		= sio4_async_config(fd, &async, errs, 0);
	bit_rate[0]	= async.tx.baud_comp;

	if (errs == 0)
		printf("PASS  (%ld baud)\n", (long) bit_rate[0]);

	return(errs);
}



//*****************************************************************************
static int _fifo_size_rx(const args_t* args, int* size)
{
	int	errs;

	for (;;)	// A convenience loop.
	{
		gsc_label("Rx FIFO Size");
		errs	= rx_fifo_size(args->fd_rx, size, 0);

		if ((errs == 0) && (size[0] == 0))
			errs	= rx_fifo_size_compute(args->fd_rx, size, 0);

		if (errs)
		{
			size[0]	= 512;
			printf("FAIL <---  (assuming 512 bytes)\n");
			break;
		}

		if (size[0] == 0)
		{
			size[0]	= 512;
			printf("PASS  (size unknown, assuming 512 Bytes)\n");
		}
		else if (size[0] % 1024)
		{
			printf("PASS  (%d Bytes)\n", size[0]);
		}
		else
		{
			printf("PASS  (%dK Bytes)\n", size[0] / 1024);
		}

		break;
	}

	return(errs);
}



//*****************************************************************************
static int _fifo_size_tx(const args_t* args, int* size)
{
	int	errs;

	for (;;)	// A convenience loop.
	{
		gsc_label("Tx FIFO Size");
		errs	= tx_fifo_size(args->fd_tx, size, 0);

		if ((errs == 0) && (size[0] == 0))
			errs	= tx_fifo_size_compute(args->fd_tx, size, 0);

		if (errs)
		{
			size[0]	= 512;
			printf("FAIL <---  (assuming 512 bytes)\n");
			break;
		}

		if (size[0] == 0)
		{
			size[0]	= 512;
			printf("PASS  (size unknown, assuming 512 Bytes)\n");
		}
		else if (size[0] % 1024)
		{
			printf("PASS  (%d Bytes)\n", size[0]);
		}
		else
		{
			printf("PASS  (%dK Bytes)\n", size[0] / 1024);
		}

		break;
	}

	return(errs);
}



//*****************************************************************************
static int _fw_type_validate(int fd, const char* desc, int* skip)
{
	char	buf[80];
	int		errs	= 0;
	int		i;
	s32		set;

	sprintf(buf, "%sFirmware Type", desc);
	gsc_label(buf);

	set	= SIO4_FW_TYPE_CONFIG_READ;
	i	= ioctl(fd, SIO4_FW_TYPE_CONFIG, &set);

	if (i == -1)
	{
		printf("FAIL <---\n");
		errs++;
	}
	else
	{
		switch (set)
		{
			default:

				errs++;
				printf("FAIL <---  (Unknown firmware type: %ld)\n", (long) set);
				break;

			case SIO4_FW_TYPE_CONFIG_SYNC:

				skip[0]	= 1;
				printf("SKIP <---  (SYNC, wrong firmware type)\n");
				break;

			case SIO4_FW_TYPE_CONFIG_Z16C30:

				printf("PASS  (Z16C30)\n");
				break;
		}
	}

	return(errs);
}



//*****************************************************************************
static int _mp_check(int fd, mp_t* mp)
{
	int	errs	= 0;

	errs	+= feature_test__mp(fd, &mp->mp, 0);
	errs	+= mp_test(fd, SIO4_MP_PROT_RS_232, &mp->_232, 0);
	errs	+= mp_test(fd, SIO4_MP_PROT_RS_422_485, &mp->_422, 0);
	errs	+= mp_test(fd, SIO4_MP_PROT_RS_423, &mp->_423, 0);
	return(errs);
}



//*****************************************************************************
static int _channel_reset(int index, int fd, const char* desc)
{
	char	buf[80];
	int		errs;
	int		ret;

	sprintf(buf, "Reset %sUSC", desc);
	gsc_label(buf);

	ret		= sio4_usc_reset(fd, 0);
	errs	= (ret < 0) ? 1 : 0;
	printf("%s\n", errs ? "FAIL <---" : "PASS");

	return(errs);
}


//*****************************************************************************
int baud_rates_validate(const args_t* args, int baud_tx, int baud_rx)
{
	int	delta	= baud_tx / 100;	// max difference
	int	errs	= 0;

	gsc_label("Baud Rates");

	if (baud_tx <= 0)
	{
		errs	= 1;
		printf("FAIL <---  (Invalid Tx rate: %d)\n", baud_tx);
	}
	else if ((args->index_tx != args->index_rx) && (baud_rx <= 0))
	{
		errs	= 1;
		printf("FAIL <---  (Invalid Rx rate: %d)\n", baud_rx);
	}
	else if ((args->index_tx != args->index_rx)	&&
				(	(baud_rx < (baud_tx - delta)) ||
					(baud_rx > (baud_tx + delta))))
	{
		errs	= 1;
		printf(	"FAIL <---  (baud rate disparity: %d -> %d)\n",
				baud_tx,
				baud_rx);
	}
	else
	{
		printf(	"PASS  (%d -> %d)\n", baud_tx, baud_rx);
	}

	return(errs);
}



//*****************************************************************************
int channels_reset(const args_t* args)
{
	int	errs	= 0;

	if (args->index_tx == args->index_rx)
	{
		errs	+= _channel_reset(args->index_tx, args->fd_tx, "");
	}
	else
	{
		gsc_label("Reset Channels");
		printf("\n");
		gsc_label_level_inc();

		errs	+= _channel_reset(args->index_tx, args->fd_tx, "Tx ");
		errs	+= _channel_reset(args->index_rx, args->fd_rx, "Rx ");

		gsc_label_level_dec();
	}

	return(errs);
}



//*****************************************************************************
void connection_type(const args_t* args)
{
	int	rx_b	= args->index_rx / 4;
	int	rx_c	= args->index_rx % 4;
	int	tx_b	= args->index_tx / 4;
	int	tx_c	= args->index_tx % 4;

	gsc_label("Connection Type");

	if (args->index_tx == args->index_rx)
	{
		printf("Loopback (no cable)\n");
	}
	else if (tx_b == rx_b)
	{
		printf(	"Channel-to-Channel Loopback Cable"
				" (B# %d, C#s %d+%d)\n",
				tx_b,
				tx_c,
				rx_c);
	}
	else
	{
		printf(	"Channel-to-Channel Passthrough Cable"
				" (B%d C%d, B%d C%d)\n",
				tx_b,
				tx_c,
				rx_b,
				rx_c);
	}
}



//*****************************************************************************
int fifo_sizes_compute(const args_t* args, int* fifo)
{
	int	errs	= 0;
	s32	fifo_rx;
	s32	fifo_tx;

	gsc_label("FIFO Sizes");
	printf("\n");
	gsc_label_level_inc();

	errs	+= _fifo_size_tx(args, &fifo_tx);
	errs	+= _fifo_size_rx(args, &fifo_rx);

	if ((fifo_tx < 512) || (fifo_rx < 512))
		fifo[0]	= 512;	// Presume the FIFO is at least this size.
	else if (fifo_tx < fifo_rx)
		fifo[0]	= fifo_tx;
	else
		fifo[0]	= fifo_rx;

	gsc_label("Results");
	printf("%s  (", errs ? "FAIL <---" : "PASS");

	if ((fifo[0]) && (fifo[0] % 1024))
		printf("%d", fifo[0]);
	else
		printf("%dK", fifo[0] / 1024);

	printf(" Bytes)\n");

	gsc_label_level_dec();
	return(errs);
}



//*****************************************************************************
int fw_type_validate(const args_t* args, int* skip)
{
	int	errs	= 0;

	if (args->index_tx == args->index_rx)
	{
		errs	+= _fw_type_validate(args->fd_tx, "", skip);
	}
	else
	{
		gsc_label("Firmware Types");
		printf("\n");
		gsc_label_level_inc();

		errs	+= _fw_type_validate(args->fd_tx, "Tx ", skip);
		errs	+= _fw_type_validate(args->fd_rx, "Rx ", skip);

		gsc_label_level_dec();
	}

	return(errs);
}



//*****************************************************************************
int rx_setup(const args_t* args, int lb, sio4_mp_prot_t mp, int* baud)
{
	int	errs;

	errs	= _channel_setup(
				args->fd_rx,
				"Rx",
				0,
				mp,
				lb,


				SIO4_CABLE_CONFIG_TXLWR_RXUPR,
				baud);
	return(errs);
}



//*****************************************************************************
int tx_setup(const args_t* args, int lb, sio4_mp_prot_t mp, int* baud)
{
	int	errs;

	errs	= _channel_setup(
				args->fd_tx,
				"Rx",
				1,
				mp,
				lb,


				SIO4_CABLE_CONFIG_TXUPR_RXLWR,
				baud);
	return(errs);
}



//*****************************************************************************
int xcvr_compute(const args_t* args, sio4_mp_prot_t* mp)
{
	int		errs	= 0;
	mp_t	mp_rx;
	mp_t	mp_tx;

	mp[0]	= SIO4_MP_PROT_RS_422_485;

	for (;;)	// A convenience loop.
	{
		gsc_label("Transceiver Setup");

		errs	+= _mp_check(args->fd_tx, &mp_tx);
		errs	+= _mp_check(args->fd_rx, &mp_rx);

		if (errs)
		{
			printf("FAIL <---\n");
			break;
		}

		if ((SIO4_FEATURE_NO == mp_tx.mp) ||
			(SIO4_FEATURE_NO == mp_rx.mp))
		{
			// The user must make sure the devices match.
			mp[0]	= SIO4_MP_PROT_READ;
			printf("SKIPPED  (insufficient support)\n");
			break;
		}

		if ((SIO4_MP_PROT_RS_422_485 == mp_tx._422) &&
			(SIO4_MP_PROT_RS_422_485 == mp_rx._422))
		{
			errs	= 0;
			mp[0]	= SIO4_MP_PROT_RS_422_485;
			printf("PASS  (RS-422/485)\n");
			break;
		}

		if ((SIO4_MP_PROT_RS_423 == mp_tx._423) &&
			(SIO4_MP_PROT_RS_423 == mp_rx._423))
		{
			errs	= 0;
			mp[0]	= SIO4_MP_PROT_RS_423;
			printf("PASS  (RS-423)\n");
			break;
		}

		if ((SIO4_MP_PROT_RS_232 == mp_tx._232) &&
			(SIO4_MP_PROT_RS_232 == mp_rx._232))
		{
			errs	= 0;
			mp[0]	= SIO4_MP_PROT_RS_232;
			printf("PASS  (RS-232)\n");
			break;
		}

		errs++;
		mp[0]	= SIO4_MP_PROT_INVALID;
		printf("FAIL <---  (insufficient support)\n");
		break;
	}

	return(errs);
}


