// $URL: http://subversion:8080/svn/gsc/trunk/drivers/LINUX/SIO4%20and%20SIO8/SIO4_Linux_1.x.x_GSC_DN/async/samples/asyncc2c/perform.c $
// $Rev: 53082 $
// $Date: 2023-06-13 10:10:44 -0500 (Tue, 13 Jun 2023) $

// SIO4: Asynchronous Protocol Library: Sample Application: source file

#include "main.h"



// variables ******************************************************************

static	u8	_rx_buf[256];				// Must be 256!!!
static	u8	_tx_buf[sizeof(_rx_buf)];
static	int	_pipeline_limit	= 512;		// modified during init.
static	int	_pipeline_qty	= 0;



//*****************************************************************************
static long _tx_data(const args_t* args)
{
	int		errs;
	long	left;
	size_t	sent;
	u8*		src;
	long	tx_qty	= 0;

	for (;;)
	{
		if (_pipeline_qty > (_pipeline_limit - sizeof(_tx_buf)))
			break;

		// Write out a single block of data.
		src		= _tx_buf;
		left	= sizeof(_tx_buf);

		for (;;)
		{
			if (left == 0)
				break;

			errs	= sio4_write(args->fd_tx, src, left, &sent);

			if (errs)
			{
				printf("FAIL <---  (sio4_write())\n");
				tx_qty	= -1;
				break;
			}

			if (sent == 0)
			{
				os_sleep_ms(1);
				continue;
			}

			tx_qty	+= sent;
			src		+= sent;
			left	-= sent;

			if (left <= 0)
			{
				_pipeline_qty	+= sizeof(_tx_buf);
				break;
			}
		}

		if (tx_qty < 0)
			break;
	}

	return(tx_qty);
}



//*****************************************************************************
static long _rx_data(const args_t* args)
{
	u8*		dst;
	int		errs;
	size_t	got;
	int		i;
	long	left;
	int		sleeps	= 0;
	long	rx_qty	= 0;

	for (;;)
	{
		if (_pipeline_qty < sizeof(_rx_buf))
			break;

		// Read in a single block of data.
		dst		= _rx_buf;
		left	= sizeof(_rx_buf);

		for (;;)
		{
			if (left == 0)
				break;

			errs	= sio4_read(args->fd_rx, dst, left, &got);

			if (errs)
			{
				printf(	"FAIL <---  (sio4_read())\n");
				rx_qty	= -1;
				break;
			}

			if (got == 0)
			{
				if (sleeps >= 1000)
				{
					printf("FAIL <---  (no data received)\n");
					rx_qty	= -1;
					break;
				}

				os_sleep_ms(1);
				sleeps++;
				continue;
			}

			sleeps	= 0;
			rx_qty	+= got;
			dst		+= got;
			left	-= got;

			if (left <= 0)
			{
				_pipeline_qty	-= sizeof(_rx_buf);

				if (memcmp(_rx_buf, _tx_buf, sizeof(_rx_buf)))
				{
					printf("FAIL <---  (data mismatch)\n");
					rx_qty	= -1;

					for (i = 0; i < sizeof(_rx_buf); i++)
					{
						printf(	"%5d. Tx 0x%02lX, Rx 0x%02lX",
								i,
								(long) _tx_buf[i],
								(long) _rx_buf[i]);

						if (_rx_buf[i] != _tx_buf[i])
							printf(" <--- mismatch");

						printf("\n");
					}
				}

				break;
			}
		}

		break;
	}

	return(rx_qty);
}



//*****************************************************************************
static int _data_transfer_test(const args_t* args, int fifo)
{
	time_t	end;
	int		errs	= 0;
	int		i;
	time_t	now;
	long	rx_qty;
	float	rx_total	= 0;
	long	tx_qty;
	float	tx_total	= 0;

	for (;;)	// A convenience loop.
	{
		gsc_label("Tx -> Rx");

		// Make sure the FIFOs are empty.
		errs	+= fifo_reset(args->fd_tx, TX_FIFO, 0);
		errs	+= channel_command_send(args->fd_tx, TX_FIFO_PURGE_CMD, 0);
		os_sleep_ms(10);
		errs	+= channel_command_send(args->fd_rx, RX_FIFO_PURGE_CMD, 0);
		errs	+= fifo_reset(args->fd_rx, RX_FIFO, 0);

		// Clear the USC's Tx Data interrupt caused by the Tx Purge.
		errs	+= reg_mod(args->fd_tx, SIO4_USC_DCCR, 0x44, 0xC4);

		// Initialization.

		for (i = 0; i < sizeof(_tx_buf); i++)
			_tx_buf[i]	= i;

		_pipeline_qty	= 0;
		_pipeline_limit	= fifo;
		end	= time(NULL) + args->seconds;
		now	= time(NULL);

		for (;;)
		{
			if (now <= end)
			{
				tx_qty	= _tx_data(args);

				if (tx_qty < 0)
				{
					errs++;
					break;
				}

				tx_total	+= tx_qty;
			}

			rx_qty		= _rx_data(args);

			if (rx_qty < 0)
			{
				errs++;
				break;
			}

			rx_total	+= rx_qty;
			now			= time(NULL);

			if ((now > end) && (_pipeline_qty == 0))
				break;
		}

		if (errs == 0)
			printf("PASS");

		printf(	"  (Tx Total %.0f bytes, Rx Total %.0f bytes)\n",
				tx_total,
				rx_total);

		break;
	}

	return(errs);
}



//*****************************************************************************
int perform_tests(const args_t* args, int* skip)
{
	s32				baud_rx	= 0;
	s32				baud_tx	= 0;
	int				errs;
	int				fifo;
	int				lb		= (args->index_tx == args->index_rx) ? 1 : 0; // loopback
	sio4_mp_prot_t	mp;

	for (;;)	// A convenience loop.
	{
		errs	= fw_type_validate(args, skip);

		if (skip[0])
		{
			gsc_label("Testing");
			printf("SKIP <---  (Tested aborted due to firmware type.)\n");
			break;
		}

		connection_type(args);
		errs	+= xcvr_compute(args, &mp);
		errs	+= fifo_sizes_compute(args, &fifo);
		errs	+= channels_reset(args);
		baud_tx	= args->baud;
		baud_rx	= args->baud;

		if (errs == 0)
		{
			errs	= tx_setup(args, lb, mp, &baud_tx);
			baud_rx	= baud_tx;
		}

		if (lb == 0)
			errs	= rx_setup(args, lb, mp, &baud_rx);

		errs	+= baud_rates_validate(args, baud_tx, baud_rx);

		if (errs)
		{
			gsc_label("Testing");
			printf("FAIL <--  (Tested aborted due to errors.)\n");
			break;
		}

		gsc_label("Data Transfer");
		printf("(%d Seconds)\n", (int) args->seconds);
		gsc_label_level_inc();

		errs	= _data_transfer_test(args, fifo);

		gsc_label_level_dec();
		break;
	}

	return(errs);
}


