// $URL: http://subversion:8080/svn/gsc/trunk/drivers/LINUX/SIO4%20and%20SIO8/SIO4_Linux_1.x.x_GSC_DN/async/lib/sio4_async.c $
// $Rev: 53088 $
// $Date: 2023-06-13 10:14:37 -0500 (Tue, 13 Jun 2023) $

// SIO4: Asynchronous Protocol Library: source file

#include "main.h"



/******************************************************************************
*
*	Function:	_async_zilog_config
*
*	Purpose:
*
*		Configure the given channel for Asynchronous operation per the given
*		parameters.
*
*	Arguments:
*
*		fd		The handle to the channel of interest.
*
*		async	The desired asynchronous configuration.
*
*		errs	The number of errors already seen.
*
*	Returned:
*
*		>= 0	The number of errors encounterred.
*
******************************************************************************/

static int _async_zilog_config(int fd, sio4_async_t* async, int errs)
{
	static const struct
	{
		int	clks_per_bit;
		int	code;
	} list[]	=
	{
		{ 64, SIO4_USC_CMR_A_CLKBIT_64 },
		{ 32, SIO4_USC_CMR_A_CLKBIT_32 },
		{ 16, SIO4_USC_CMR_A_CLKBIT_16 },
		{ 0, 0 }
	};

	long					best_d			= 0;
	long					delta;
	long					enable;
	int						i;
	unsigned long			mask;
	int						previous		= errs;
	zilog_rx_baud_config_t	rx_baud;
	zilog_rx_baud_config_t	rx_best;
	int						rx_clks_per_bit	= 0;
	zilog_tx_baud_config_t	tx_baud;
	zilog_tx_baud_config_t	tx_best;
	int						tx_clks_per_bit	= 0;
	unsigned long			value;

	memset(&rx_best, 0, sizeof(rx_best));
	memset(&tx_best, 0, sizeof(tx_best));

	//	Try to use the highest number of clocks per bit.

	for (i = 0; list[i].clks_per_bit; i++)
	{
		rx_clks_per_bit		= list[i].code;
		rx_baud.rxc_rate	= async->rx.usc_rxc_rate
							/ list[i].clks_per_bit;
		rx_baud.baudrate	= async->rx.baud_want;
		zilog_rx_baud_compute(&rx_baud);
		delta	= rx_baud.baudrate_comp - async->rx.baud_want;
		delta	= (delta < 0) ? -delta : +delta;

		if ((i == 0) || (delta < best_d))
		{
			rx_best	= rx_baud;
			best_d	= delta;
		}

		if (delta == 0)
			break;
	}

	rx_baud				= rx_best;
	async->rx.baud_comp	= rx_baud.baudrate_comp;

	for (i = 0; list[i].clks_per_bit; i++)
	{
		tx_clks_per_bit		= list[i].code;
		tx_baud.txc_rate	= async->tx.usc_txc_rate
							/ list[i].clks_per_bit;
		tx_baud.baudrate	= async->tx.baud_want;
		zilog_tx_baud_compute(&tx_baud);
		delta	= tx_baud.baudrate_comp - async->tx.baud_want;
		delta	= (delta < 0) ? -delta : +delta;

		if ((i == 0) || (delta < best_d))
		{
			tx_best	= tx_baud;
			best_d	= delta;
		}

		if (delta == 0)
			break;
	}

	tx_baud				= tx_best;
	async->tx.baud_comp	= tx_baud.baudrate_comp;

	/*
	*	Configure the Zilog specific registers. The configuration
	*	we'll use is: Asynchronous, no parity, 8 data bits, 1 stop
	*	bit, NRZ encoding. The baud rate is given as a function
	*	argument.
	*
	*	BCR					DON'T TOUCH!!!
	*	DCCR				DON'T TOUCH!!!
	*	ICR					DON'T TOUCH!!!
	*	IVR					DON'T TOUCH!!!
	*	MISR				DON'T TOUCH!!!
	*	RICR				DON'T TOUCH!!!
	*	SICR				DON'T TOUCH!!!
	*	TICR				DON'T TOUCH!!!
	*
	*	CCSR				Ignore at this time.
	*	CDR (RDR & TDR)		Ignore at this time.
	*	RCCR				Ignore at this time.
	*	RCSR				Ignore at this time.
	*	RSR					Ignore at this time.
	*	TMCR				Ignore at this time.
	*	TMDR				Ignore at this time.
	*	TCCR				Ignore at this time.
	*	TSR					Ignore at this time.
	*/

	/*
	*	CMR.TxMode			= Asynchronous
	*	CMR.TxStopBits		= async->tx.stop_bits
	*	CMR.TxClksPerBit	= tx_clks_per_bit
	*	CMR.RxMode			= Asynchronous
	*	CMR.RxClksPerBit	= rx_clks_per_bit
	*	CMR.all others		= ignore
	*/

	value	= 0;
	value	|= SIO4_USC_CMR_TXMODE_ENCODE(SIO4_USC_CMR_MODE_ASYNC);
	value	|= SIO4_USC_CMR_AT_STOPBITS_ENCODE(async->tx.stop_bits);
	value	|= SIO4_USC_CMR_AT_CLKBIT_ENCODE(tx_clks_per_bit);
	value	|= SIO4_USC_CMR_RXMODE_ENCODE(SIO4_USC_CMR_MODE_ASYNC);
	value	|= SIO4_USC_CMR_AR_CLKBIT_ENCODE(rx_clks_per_bit);
	errs	+= reg_write(fd, SIO4_USC_CMR, value);

	/*
	*	IOCR.CTSMode		= drive low, REQUIRED, CTS device error
	*	IOCR.DCDMode		= carrier detect
	*	IOCR.TxRMode		= DMA request, REQUIRED
	*	IOCR.RxRMode		= DMA request, REQUIRED
	*	IOCR.TxDMode		= transmit data, REQUIRED
	*	IOCR.TxCMode		= input, REQUIRED
	*	IOCR.RxCMode		= input, REQUIRED
	*/

	value	= 0;
	value	|= SIO4_USC_IOCR_CTSMODE_ENCODE(SIO4_USC_IOCR_CTSMODE_INPUT);
	value	|= SIO4_USC_IOCR_DCDMODE_ENCODE(SIO4_USC_IOCR_DCDMODE_DCD);
	value	|= SIO4_USC_IOCR_TXRMODE_ENCODE(SIO4_USC_IOCR_REQMODE_DMA);
	value	|= SIO4_USC_IOCR_RXRMODE_ENCODE(SIO4_USC_IOCR_REQMODE_DMA);
	value	|= SIO4_USC_IOCR_TXDMODE_ENCODE(SIO4_USC_IOCR_TXDMODE_OUTPUT);
	value	|= SIO4_USC_IOCR_TXCMODE_ENCODE(SIO4_USC_IOCR_TXCMODE_INPUT);
	value	|= SIO4_USC_IOCR_RXCMODE_ENCODE(SIO4_USC_IOCR_RXCMODE_INPUT);
	errs	+= reg_write(fd, SIO4_USC_IOCR, value);

	/*
	*	TC0R				= tx_baud.tc0r_value
	*/

	errs	+= reg_write(fd, SIO4_USC_TC0R, tx_baud.tc0r_value);

	/*
	*	TC1R				= rx_baud.tc1r_value
	*/

	errs	+= reg_write(fd, SIO4_USC_TC1R, rx_baud.tc1r_value);

	/*
	*	TCLR				= 0 (disable)
	*/

	errs	+= reg_write(fd, SIO4_USC_TCLR, 0);

	/*
	*	RCLR				= 0 (disable)
	*/

	errs	+= reg_write(fd, SIO4_USC_RCLR, 0);

	/*
	*	CMCR.CTR1Src		= rx_baud.ctr1_src
	*	CMCR.CTR0Src		= tx_baud.ctr0_src
	*	CMCR.BRG1Src		= rx_baud.brg1_src
	*	CMCR.BRG0Src		= tx_baud.brg0_src
	*	CMCR.DPLLSrc		= rx_baud.dpll_src
	*	CMCR.TxCLKSrc		= tx_baud.txclk_src
	*	CMCR.RxCLKSrc		= rx_baud.rxclk_src
	*/

	value	= 0;
	value	|= SIO4_USC_CMCR_CTR1SRC_ENCODE(rx_baud.ctr1_src);
	value	|= SIO4_USC_CMCR_CTR0SRC_ENCODE(tx_baud.ctr0_src);
	value	|= SIO4_USC_CMCR_BRG1SRC_ENCODE(rx_baud.brg1_src);
	value	|= SIO4_USC_CMCR_BRG0SRC_ENCODE(tx_baud.brg0_src);
	value	|= SIO4_USC_CMCR_DPLLSRC_ENCODE(rx_baud.dpll_src);
	value	|= SIO4_USC_CMCR_TXCLKSRC_ENCODE(tx_baud.txclk_src);
	value	|= SIO4_USC_CMCR_RXCLKSRC_ENCODE(rx_baud.rxclk_src);
	errs	+= reg_write(fd, SIO4_USC_CMCR, value);

	/*
	*	HCR.CTR0Div			= tx_baud.ctr0_div
	*	HCR.CTR1DSel		= rx_baud.ctr1_div_sel
	*	HCR.CVOK			= ignore
	*	HCR.DPLLDiv			= rx_baud.dpll_div
	*	HCR.DPLLMode		= disable, not used
	*	HCR.TxAMode			= Tx DMA Acknowledge, REQUIRED
	*	HCR.BRG1S			= continuous, not used
	*	HCR.BRG1E			= rx_baud.brg1_enable
	*	HCR.RxAMode			= Rx DMA Acknowledge, REQUIRED
	*	HCR.BRG0S			= continuous
	*	HCR.BRG0E			= tx_baud.brg0_enable
	*/

	value	= 0;
	value	|= SIO4_USC_HCR_CTR0DIV_ENCODE(tx_baud.ctr0_div);
	value	|= SIO4_USC_HCR_CTR1DSEL_ENCODE(rx_baud.ctr1_div_sel);
	value	|= SIO4_USC_HCR_DPLLDIV_ENCODE(rx_baud.dpll_div);
	value	|= SIO4_USC_HCR_DPLLMODE_ENCODE(SIO4_USC_HCR_DPLLMODE_DISABLE);
	value	|= SIO4_USC_HCR_TXAMODE_ENCODE(SIO4_USC_HCR_AMODE_DMA_ACK);
	value	|= SIO4_USC_HCR_BRG1S_ENCODE(SIO4_USC_HCR_BRGS_CONTINUOUS);
	value	|= SIO4_USC_HCR_BRG1E_ENCODE(rx_baud.brg1_enable);
	value	|= SIO4_USC_HCR_RXAMODE_ENCODE(SIO4_USC_HCR_AMODE_DMA_ACK);
	value	|= SIO4_USC_HCR_BRG0E_ENCODE(tx_baud.brg0_enable);
	errs	+= reg_write(fd, SIO4_USC_HCR, value);

	/*
	*	CCR.TxCtrlBlk		= disable
	*	CCR.Wait4TxTrig		= ignore
	*	CCR.FlagPreamble	= ignore
	*	CCR.TxShaveL		= async->tx.shave_bits
	*	CCR.TxPreL			= ignore
	*	CCR.TxPrePat		= ignore
	*	CCR.RxStatBlk		= disable
	*	CCR.Wait4RxTrig		= ignore
	*/

	value	= 0;
	value	|= SIO4_USC_CCR_TXCTRLBLK_ENCODE(SIO4_USC_CCR_BLOCK_SIZE_NONE);
	value	|= SIO4_USC_CCR_TXSHAVEL_ENCODE(async->tx.shave_bits);
	value	|= SIO4_USC_CCR_RXSTATBLK_ENCODE(SIO4_USC_CCR_BLOCK_SIZE_NONE);
	errs	+= reg_write(fd, SIO4_USC_CCR, value);

	/*
	*	TCSR.TCmd			= 0 (none, ignore)
	*	TCSR.UnderWait		= 0 (ignore)
	*	TCSR.TxIdle			= async->tx.idle_output
	*	TCSR.PreSent		= 0 (ignore)
	*	TCSR.IdleSent		= 0 (ignore)
	*	TCSR.AbortSent		= 0 (ignore)
	*	TCSR.EOF/EOM Sent	= 0 (ignore)
	*	TCSR.CRCSent		= 0 (ignore)
	*	TCSR.AllSent		= 0 (ignore)
	*	TCSR.TxUnder		= 0 (ignore)
	*	TCSR.TxEmpty		= 0 (ignore)
	*/

	value	= 0;
	value	|= SIO4_USC_TCSR_TXIDLE_ENCODE(async->tx.idle_output);
	errs	+= reg_write(fd, SIO4_USC_TCSR, value);

	/*
	*	CCAR.RTcmd			= no command
	*	CCAR.RTReset		= release
	*	CCAR.RTMode			= no change
	*	CCAR.ChanLoad		= ignore
	*	CCAR.BW				= ignore
	*	CCAR.RegAddr		= ignore
	*	CCAR.UL				= ignore
	*/

	value	= 0;
	value	|= SIO4_USC_CCAR_CMD_ENCODE(SIO4_USC_CCAR_CMD_NONE);
	value	|= SIO4_USC_CCAR_RTRESET_ENCODE(SIO4_USC_CCAR_RTRESET_NO);
	mask	= ~SIO4_USC_CCAR_RTMODE_ENCODE(~0UL);
	errs	+= reg_mod(fd, SIO4_USC_CCAR, value, mask);

	/*
	*	TMR.TxEncode		= async->tx.encoding
	*	TMR.TxCRCType		= ignore
	*	TMR.TxCRCStart		= ignore
	*	TMR.TxCRCEnab		= ignore
	*	TMR.TxCRCatEnd		= ignore
	*	TMR.TxParType		= async->tx.parity_type
	*	TMR.TxParEnab		= async->tx.parity_enable
	*	TMR.TxLength		= async->tx.char_size
	*	TMR.TxEnab			= async->tx.enable
	*/

	value	= 0;
	value	|= SIO4_USC_TMR_TXENCODE_ENCODE(async->tx.encoding);
	value	|= SIO4_USC_TMR_TXPARTYPE_ENCODE(async->tx.parity_type);
	value	|= SIO4_USC_TMR_TXPARENAB_ENCODE(async->tx.parity_enable);
	value	|= SIO4_USC_TMR_TXLENGTH_ENCODE(async->tx.char_size);
	enable	= async->tx.enable
			? SIO4_USC_ENAB_ON_NOW
			: SIO4_USC_ENAB_OFF_NOW;
	value	|= SIO4_USC_TMR_TXENAB_ENCODE(enable);
	errs	+= reg_write(fd, SIO4_USC_TMR, value);

	/*
	*	RMR.RxDecode		= async->rx.encoding
	*	RMR.RxCRCType		= ignore
	*	RMR.RxCRCStart		= ignore
	*	RMR.RxCRCEnab		= ignore
	*	RMR.QAbort			= ignore
	*	RMR.RxParType		= async->rx.parity_type
	*	RMR.RxParEnab		= async->rx.parity_enable
	*	RMR.RxLength		= async->rx.char_size
	*	RMR.RxEnab			= async->rx.enable
	*/

	value	= 0;
	value	|= SIO4_USC_RMR_RXDECODE_ENCODE(async->rx.encoding);
	value	|= SIO4_USC_RMR_RXPARTYPE_ENCODE(async->rx.parity_type);
	value	|= SIO4_USC_RMR_RXPARENAB_ENCODE(async->rx.parity_enable);
	value	|= SIO4_USC_RMR_RXLENGTH_ENCODE(async->rx.char_size);
	enable	= async->rx.enable
			? SIO4_USC_ENAB_ON_NOW
			: SIO4_USC_ENAB_OFF_NOW;
	value	|= SIO4_USC_RMR_RXENAB_ENCODE(enable);
	errs	+= reg_write(fd, SIO4_USC_RMR, value);

	errs	-= previous;
	return(errs);
}



/******************************************************************************
*
*	Function:	sio4_async_config
*
*	Purpose:
*
*		Configure the given channel for Asynchronous operation per the given
*		parameters.
*
*	Arguments:
*
*		fd		The handle to the channel of interest.
*
*		async	The desired Async configuration.
*
*		errs	The number of errors already seen.
*
*		loud	Work verbosely?
*
*	Returned:
*
*		>= 0	The number of errors encounterred.
*
******************************************************************************/

int sio4_async_config(int fd, sio4_async_t* async, int errs, int loud)
{
	int		cable;
	int		enable;
	int		mode;
	__s32	scd;
	long	want;
	int		xceiver;

	// Initialize the channel.
	errs	+= channel_init(fd, loud);

	// Apply the I/O settings.
	errs	+= feature_test__bcr_scd(fd, &scd, loud);

	if ((async->rx.io_mode_force) ||
		(async->rx.io_mode != SIO4_IO_MODE_DMDMA))
	{
		mode	= async->rx.io_mode;
	}
	else
	{
		mode	= SIO4_IO_MODE_PIO;
	}

	errs	+= rx_io_mode_config(fd, mode, NULL, loud);

	if ((async->tx.io_mode_force) ||
		(async->tx.io_mode != SIO4_IO_MODE_DMDMA))
	{
		mode	= async->tx.io_mode;
	}
	else
	{
		mode	= SIO4_IO_MODE_PIO;
	}

	errs	+= tx_io_mode_config(fd, mode, NULL, loud);
	errs	+= timeout_set_read(fd, async->rx.timeout, loud);
	errs	+= timeout_set_write(fd, async->tx.timeout, loud);

	// Initialize the FIFOs.
	errs	+= rx_fifo_full_cfg_chan(fd, async->rx.fifo_full_chan, NULL, loud);
	errs	+= rx_fifo_full_cfg_glb(fd, async->rx.fifo_full_glb, NULL, loud);
	errs	+= rx_fifo_ae_config(fd, async->rx.ae, NULL, loud);
	errs	+= rx_fifo_af_config(fd, async->rx.af, NULL, loud);
	errs	+= tx_fifo_ae_config(fd, async->tx.ae, NULL, loud);
	errs	+= tx_fifo_af_config(fd, async->tx.af, NULL, loud);

	// Disable the cable for now.
	errs	+= cable_config(fd, SIO4_CABLE_CONFIG_DISABLE, NULL, loud);

	// Initialize the oscillator.
	want	= async->tx.enable ? async->tx.baud_want : async->rx.baud_want;
	async->osc.ref_comp	= async->osc.ref_default;
	errs	+= osc_ref_compute(	fd, &async->osc.ref_comp, want, loud);
	errs	+= osc_setup(fd, async->osc.ref_comp, &async->osc.ref_got, loud);

	if (async->rx.usc_rxc_config == SIO4_RXC_USC_CONFIG_IN_PRG_CLK)
		async->rx.usc_rxc_rate	= async->osc.ref_comp;

	if (async->tx.usc_txc_config == SIO4_TXC_USC_CONFIG_IN_PRG_CLK)
		async->tx.usc_txc_rate	= async->osc.ref_comp;

	// Configure the Zilog.
	errs	+= _async_zilog_config(fd, async, errs);
	errs	+= rxc_usc_config(fd, async->rx.usc_rxc_config, NULL, loud);
	errs	+= txc_usc_config(fd, async->tx.usc_txc_config, NULL, loud);
	errs	+= zilog_mode_config(	fd,
									!async->loopback.enable,
									async->loopback.internal);

	// Configure the cable interface. THIS MUST COME AFTER UART SETUP!
	errs	+= rx_enable(fd, 0);
	errs	+= cts_cable_config(fd, async->cable.cts, NULL, loud);
	errs	+= dcd_cable_config(fd, async->cable.dcd, NULL, loud);
	errs	+= tx_cable_data_config(fd, async->tx.cable_data, NULL, loud);
	errs	+= tx_cable_clock_config(fd, async->tx.cable_clock, NULL, loud);
	enable	= async->loopback.enable ? 0 : async->cable.dcedte.enable;
	errs	+= dcedte_config(	fd,
								enable,
								async->cable.dcedte.dce,
								async->loopback.enable,
								loud);
	cable	= async->loopback.enable
			? SIO4_CABLE_CONFIG_DISABLE
			: async->cable.legacy.config;
	errs	+= cable_config(fd, cable, NULL, loud);
	xceiver	= async->loopback.enable
			? SIO4_MP_PROT_DISABLE
			: async->cable.xceiver;
	errs	+= mp_config(fd, xceiver, NULL, loud);
	errs	+= rx_enable(fd, 1);

	// Make sure the FIFOs are empty.
	errs	+= fifo_reset(fd, TX_FIFO, loud);
	errs	+= channel_command_send(fd, TX_FIFO_PURGE_CMD, loud);
	os_sleep_ms(10);
	errs	+= channel_command_send(fd, RX_FIFO_PURGE_CMD, loud);
	errs	+= fifo_reset(fd, RX_FIFO, loud);

	// Clear the USC's Tx Data interrupt caused by the Tx Purge.
	errs	+= reg_mod(fd, SIO4_USC_DCCR, 0x44, 0xC4);
	return(errs);
}



/******************************************************************************
*
*	Function:	sio4_async_t_init
*
*	Purpose:
*
*		Initialize the given structure with default settings.
*
*	Arguments:
*
*		async	The settings are recorded here.
*
*	Returned:
*
*		None.
*
******************************************************************************/

void sio4_async_t_init(sio4_async_t* async)
{
	memset(async, 0, sizeof(sio4_async_t));

	async->cable.dcedte.enable	= 0;
	async->cable.dcedte.dce		= 0;
	async->cable.legacy.config	= SIO4_CABLE_CONFIG_TXUPR_RXLWR;
	async->cable.cts			= SIO4_CTS_CABLE_CONFIG_DISABLE;
	async->cable.dcd			= SIO4_DCD_CABLE_CONFIG_DISABLE;
	async->cable.xceiver		= SIO4_MP_PROT_RS_232;

	async->loopback.enable		= 0;
	async->loopback.internal	= 1;

	async->osc.ref_comp			= 0;
	async->osc.ref_default		= SIO4_OSC_FREQ_REF_DEFAULT;
	async->osc.ref_got			= 0;

	async->rx.ae				= 16;
	async->rx.af				= 16;
	async->rx.baud_comp			= 0;
	async->rx.baud_want			= 115200L;
	async->rx.char_size			= SIO4_USC_CHAR_LENGTH_8;
	async->rx.enable			= 1;
	async->rx.encoding			= SIO4_USC_ENCODING_NRZ;
	async->rx.fifo_full_chan	= SIO4_RX_FIFO_FULL_CFG_CHAN_OVER;
	async->rx.fifo_full_glb		= SIO4_RX_FIFO_FULL_CFG_GLB_OVER;
	async->rx.io_mode			= SIO4_IO_MODE_DEFAULT;
	async->rx.io_mode_force		= 0;
	async->rx.parity_enable		= 0;
	async->rx.parity_type		= SIO4_USC_PARTYPE_EVEN;
	async->rx.timeout			= SIO4_TIMEOUT_DEFAULT;
	async->rx.usc_rxc_config	= SIO4_RXC_USC_CONFIG_IN_PRG_CLK;
	async->rx.usc_rxc_rate		= SIO4_OSC_FREQ_REF_DEFAULT;

	async->tx.ae				= 16;
	async->tx.af				= 16;
	async->tx.baud_comp			= 0;
	async->tx.baud_want			= 115200L;
	async->tx.cable_clock		= SIO4_TX_CABLE_CLOCK_CONFIG_USC_TC;
	async->tx.cable_data		= SIO4_TX_CABLE_DATA_CONFIG_USC_TXD;
	async->tx.char_size			= SIO4_USC_CHAR_LENGTH_8;
	async->tx.enable			= 1;
	async->tx.encoding			= SIO4_USC_ENCODING_NRZ;
	async->tx.idle_output		= SIO4_USC_TCSR_TXIDLE_DEFAULT;
	async->tx.io_mode			= SIO4_IO_MODE_DEFAULT;
	async->tx.io_mode_force		= 0;
	async->tx.parity_enable		= 0;
	async->tx.parity_type		= SIO4_USC_PARTYPE_EVEN;
	async->tx.shave_bits		= 0;
	async->tx.stop_bits			= SIO4_USC_CMR_AT_STOPBITS_1;
	async->tx.timeout			= SIO4_TIMEOUT_DEFAULT;
	async->tx.usc_txc_config	= SIO4_TXC_USC_CONFIG_IN_PRG_CLK;
	async->tx.usc_txc_rate		= SIO4_OSC_FREQ_REF_DEFAULT;
}



