/*/*****************************************************/
/* SIO4 HDLC/SDLC Sample Code
/*
/* This sample code is written to test multi-channel 
/* HDLC communications on the SIO4 - of interest to most
/* customers would be the ChanneltoChannelLoopbackTest() function
/* Since HDLC is a derivative of SDLC, we will not go in
/* to great lengths to differentiate between these as this
/* information is readily available on the Internet
/*
/* Please see the documentation for the Zilog Z16C30
/* serial controller available from:
/* http://www.generalstandards.com/faq.php#serialproducts
/* for specific values of Zilog register reads/writes
/* 
/* For testing, a loopback cable was used to connect
/* Channels 1 and 2 and channels 3 and 4
/* This can be done by using a "standard" cable from GSC
/* and connecting the channels' DB25 connectors with pin-to-pin
/* type "gender changers"
/*
/* If desired, these channels can be changed here:
/* ChanneltoChannelLoopbackTest(1,2,CABLE_UPPER);
/* where the 1st param = the TX channel and the
/* 2nd param = the Rx Channel.
/*
/* The following clock jumper settings were used for this test:
/* Channel 1 (J3) - Transmit channel:
/* Pins 5 to 6: routes onboard oscillator to /RxC pin
/* Pins 13 to 14: routes /TxC pin to cable clock
/*
/* This clock jumper setting will cause channel 1 to fail the internal loopback test  
/*	
/* Channel 2 (J8) - Receive channel:
/* Pins 1 to 2: routes onboard oscillator to cable clock (not used in this test)
/* Pins 3 to 4: routes onboard oscillator to /TxC pin (not used in this test)
/* Pins 9 to 10: received cable clock to /RxC pin
/*
/* This code was written for use with the General
/* Standards Corporation PLXSW-SIO4 device driver
/* but may be adapted for use with other drivers
/*****************************************************/

#include <windows.h>
#include <stdio.h>
#include <stdlib.h>
#include <conio.h>
#include "SIO4intface.h"


/*****************************************************/
/* Forward declarations of local functions           */
/*****************************************************/
void ResetBoard(void);
void WriteLocal(void);
void ReadLocal(void);
void WriteUSC(void);
void ReadUSC(void);
void WriteConfig(void);
void ReadConfig(void);
void ShowAPIError(U32 APIReturnCode);
U16 CalculateFIFOSize(U16 iChannel);

// test functions
void SingleChannelInternalLoopbackTest(U32 ulChannel);
void ChanneltoChannelExternalLoopbackTest(U32 ulTxChannel, U32 ulRxChannel);

/*****************************************************/
/* Global Vars                                       */
/*****************************************************/
// read and write data buffers:
U32	*pReadBuffer  = (U32 *) NULL;
U32	*pWriteBuffer = (U32 *) NULL;

HGLOBAL         hWriteGlobal;
HGLOBAL         hReadGlobal;

U32	ulBufferSize = 0x100;

// used to calculate FIFO sizes for each channel - default is 0x100
U32	FIFOSize[5] = {0x0,0x100,0x100,0x100, 0x100};

U32 ulNumBds, ulBdNum, ulAuxBdNum, ulErr = 0, ulLoop;
char mmchar, kbchar;
char cBoardInfo[400];

U16	iCurrentChannel = 1;

#define API_RETURN_CODE_STARTS              0x201   // 513 Starting return code

// API Return Code Values
typedef enum _RETURN_CODE
{
    ApiFailed = API_RETURN_CODE_STARTS,
    ApiAccessDenied,
    ApiDmaChannelUnavailable,
    ApiDmaChannelInvalid,
    ApiDmaChannelTypeError,
    ApiDmaInProgress,
    ApiDmaDone,
    ApiDmaPaused,
    ApiDmaNotPaused,
    ApiDmaCommandInvalid,
    ApiDmaManReady,
    ApiDmaManNotReady,
    ApiDmaInvalidChannelPriority,			// 525
    ApiDmaManCorrupted,
    ApiDmaInvalidElementIndex,
    ApiDmaNoMoreElements,
    ApiDmaSglInvalid,
    ApiDmaSglQueueFull,
    ApiNullParam,
    ApiInvalidBusIndex,
    ApiUnsupportedFunction,
    ApiInvalidPciSpace,
    ApiInvalidIopSpace,						// 535
    ApiInvalidSize,
    ApiInvalidAddress,
    ApiInvalidAccessType,
    ApiInvalidIndex,
    ApiMuNotReady,
    ApiMuFifoEmpty,
    ApiMuFifoFull,
    ApiInvalidRegister,
    ApiDoorbellClearFailed,
    ApiInvalidUserPin,						// 545
    ApiInvalidUserState,
    ApiEepromNotPresent,
    ApiEepromTypeNotSupported,
    ApiEepromBlank,
    ApiConfigAccessFailed,
    ApiInvalidDeviceInfo,
    ApiNoActiveDriver,
    ApiInsufficientResources,
    ApiObjectAlreadyAllocated,
    ApiAlreadyInitialized,					// 555
    ApiNotInitialized,
    ApiBadConfigRegEndianMode,
    ApiInvalidPowerState,
    ApiPowerDown,
    ApiFlybyNotSupported,
    ApiNotSupportThisChannel,
    ApiNoAction,
    ApiHSNotSupported,
    ApiVPDNotSupported,
    ApiVpdNotEnabled,						// 565
    ApiNoMoreCap,
    ApiInvalidOffset,
    ApiBadPinDirection,
    ApiPciTimeout,
    ApiDmaChannelClosed,
    ApiDmaChannelError,
    ApiInvalidHandle,
    ApiBufferNotReady,
    ApiInvalidData,
    ApiDoNothing,							// 575
    ApiDmaSglBuildFailed,
    ApiPMNotSupported,
    ApiInvalidDriverVersion,
	ApiInvalidBoardNumber,
	ApiInvalidDMANumWords,
	ApiDMABufferTooSmall,
	ApiInvalidDMABufferAddr,
	ApiInvalidFWRev,
	ApiInvalidChannel,
    ApiLastError               // Do not add API errors below this line
} RETURN_CODE, *PRETURN_CODE;

// interrupt notification defines:
// SIO4A/SIO4B/SIO4AR/SIO4BR will have more possible IRQ sources
#define		SYNC_DETECTED			0x1
#define		TX_FIFO_ALMOST_EMPTY	0x2
#define		RX_FIFO_ALMOST_FULL		0x4
#define		USC_INTERRUPT			0x8

//==============================================================================
//
//==============================================================================
void main(int argc, char * argv[])
{
	U32 ulMaxFIFOSize = 0x10;

	// locate sio4 cards
	ulNumBds = SIO4_FindBoards(&cBoardInfo[0], &ulErr);

	printf("\nSio4 Boards Found: %d", ulNumBds);

  if(ulErr)
  {
   ShowAPIError(ulErr);
   getch();         // Wait for a Key Press
   exit(0);
  }

  if(ulNumBds < 1)
  {
      printf("  ERROR: No Boards found in the system\n");
      getch();         // Wait for a Key Press
  }

	// set to 1 since test bed only has 1 sio4 card
	ulBdNum = 1;

  if((ulErr = SIO4_Open(ulBdNum)) != 0)
	{
     ShowAPIError(ulErr);
     getch();         // Wait for a Key Press
     exit(0);
	}

	// calculate FIFO size
	for(ulLoop = 1; ulLoop < 5; ulLoop++)
	{
		if(CalculateFIFOSize((U16) ulLoop )){
			printf("\n Error calculating FIFO size for channel %d", ulLoop);
			getch();
		}
	}
	

	//-----------------------------------------------------------
	// Get two buffers for data: one read, one write 
	// GlobalAlloc expects memory size in bytes      
	// determine the largest FIFO size on the card to use to allocate
	// memory with - normally this is chan 1, but just in case
    //-----------------------------------------------------------
	for ( ulLoop = 1, ulMaxFIFOSize = FIFOSize[1] ; ulLoop < 5 ; ulLoop++ )
		 if ( FIFOSize[ulLoop] > ulMaxFIFOSize )
			ulMaxFIFOSize = FIFOSize[ulLoop] ; // end if, for
	
	hWriteGlobal = GlobalAlloc(GMEM_FIXED, ulMaxFIFOSize*4);
	if (! hWriteGlobal)
		printf("\n\t\tFailed to get write buffer\n");

	hReadGlobal  = GlobalAlloc(GMEM_FIXED, ulMaxFIFOSize*4);
	if (! hReadGlobal)
		printf("\n\t\tFailed to get read buffer\n");

	if ((hWriteGlobal) && (hReadGlobal))
	{
		pWriteBuffer = (U32 *) GlobalLock(hWriteGlobal);
		pReadBuffer  = (U32 *) GlobalLock(hReadGlobal);
	}

	// unable to get either ReadBuffer or WriteBuffer - end and exit
	if ((! pWriteBuffer) || (! pReadBuffer))
	{
		printf("\n\t\tCannot Allocate Buffer Memory. Hit Any Key to exit...");
		getch();
		if (hWriteGlobal){GlobalUnlock(hWriteGlobal);}
		if (pWriteBuffer){GlobalFree(hWriteGlobal);}
		if (hReadGlobal){GlobalUnlock(hReadGlobal);}
		if (pReadBuffer){GlobalFree(hReadGlobal);}
		exit(0);
	}

	do
    {
	
	// put menu here
	system("cls");
	printf("\nSIO4 Test Menu");
	printf("\n\n1. Reset SIO4 Card");
	printf("\n2. Write Local Register");
	printf("\n3. Read Local Register");
	printf("\n4. Write Serial Controller Register");
	printf("\n5. Read Serial Controller Register");
	printf("\n6. Single Channel Internal Loopback (requires factory clock jumper settings)");
	printf("\n7. Channel to Channel SDLC/HDLC External Loopback *");
	printf("\nH. Help");
	printf("\nX. Exit");
	printf("\n\n* Requires cable and non-factory jumper settings - see source code for details");

	printf("\n\nPlease make a menu selection: ");

	  mmchar = toupper(getch());
      switch(mmchar)
      {
        case '1': //
           ResetBoard();
          break;
        case '2': //
           WriteLocal();
          break;
        case '3': //
           ReadLocal();
          break;
        case '4': //
           WriteUSC();
          break;
        case '5': //
           ReadUSC();
          break;
        case '6': //
			{
				printf("\nPlease select a channel (1-4): ");
				scanf("%x", &iCurrentChannel);

				while((iCurrentChannel < 0) || (iCurrentChannel > 4))
				{
					printf("\nInvalid channel selection...");
					printf("\nPlease select a channel (1-4): ");
					scanf("%x", &iCurrentChannel);
				}

				SingleChannelInternalLoopbackTest(iCurrentChannel);
				printf("\nPress a key to return to menu...\n");
				getch();
			}
        break;
        case '7': //
			{
				ChanneltoChannelExternalLoopbackTest(1, 2);
				printf("\nPress a key to return to menu...\n");
				getch();
			}
        break;
		case 'H':
			{
				system("cls");
				printf("SIO4 HDLC/SDLC Sample Code\n\n");
				printf("This sample code is written to test multi-channel \n");
				printf("HDLC communications on the SIO4 - of interest to most\n");
				printf("customers would be the ChanneltoChannelLoopbackTest() function\n");
				printf("Since HDLC is a derivative of SDLC, we will not go in\n");
				printf("to great lengths to differentiate between these as this\n");
				printf("information is readily available on the Internet\n\n");
				printf("Please see the documentation for the Zilog Z16C30\n");
				printf("serial controller available from:\n");
				printf("http://www.generalstandards.com/faq.php#serialproducts\n");
				printf("for specific values of Zilog register reads/writes\n\n");
				printf("more...\n");
				getch();
				system("cls");
				printf("For testing, a loopback cable was used to connect\n");
				printf("Channels 1 and 2 and channels 3 and 4\n");
				printf("This can be done by using a \"standard\" cable from GSC\n");
				printf("and connecting the channels' DB25 connectors with pin-to-pin\n");
				printf("type \"gender changers\"\n\n");
				printf("If desired, these channels can be changed here:\n");
				printf("ChanneltoChannelLoopbackTest(1,2,CABLE_UPPER);\n");
				printf("where the 1st param = the TX channel and the\n");
				printf("2nd param = the Rx Channel. \n\n");
				printf("more...\n");
				getch();
				system("cls");
				printf("The following clock jumper settings were used for this test:\n\n");
				printf("Channel 1 (J3) - Transmit channel:\n");
				printf("Pins 5 to 6: routes onboard oscillator to /RxC pin\n");
				printf("Pins 13 to 14: routes /TxC pin to cable clock\n");
				printf("This clock jumper setting causes ch 1 to fail the internal loopback test\n\n");  
				
				printf("Channel 2 (J8) - Receive channel:\n");
				printf("Pins 1 to 2: routes onboard oscillator to cable clock (not used in this test)\n");
				printf("Pins 3 to 4: routes onboard oscillator to /TxC pin (not used in this test)\n");
				printf("Pins 9 to 10: received cable clock to /RxC pin\n\n");
				printf("This code was written for use with the General\n");
				printf("Standards Corporation PLXSW-SIO4 device driver\n");
				printf("but may be adapted for use with other drivers\n\n");			
				printf("Press a key to return to main menu...");
				getch();
				system("cls");
			}
        case 'X': //
          break;
        default:
          printf("\n  Invalid Selection: Press any key to return...\n");
          getch();
          break;
      }
    }while(mmchar!='X');

	SIO4_Close(ulBdNum);
    if(ulAuxBdNum)
	  SIO4_Close(ulAuxBdNum);
	
	// release memory
	if (hWriteGlobal){GlobalUnlock(hWriteGlobal);}
	if (pWriteBuffer){GlobalFree(hWriteGlobal);}
	if (hReadGlobal){GlobalUnlock(hReadGlobal);}
	if (pReadBuffer){GlobalFree(hReadGlobal);}
  
} /* end main */

//------------------------------------------------------------------------------
void ResetBoard(void)
//------------------------------------------------------------------------------
{
  cprintf("Resetting All Channels");

  // Init the Board
  ulErr = SIO4_Board_Reset(ulBdNum);
  if(ulErr)
  {
   ShowAPIError(ulErr);
   getch();         // Wait for a Key Press
   return;
  }
  Sleep(1000);
}

//------------------------------------------------------------------------------
void WriteLocal(void)
//------------------------------------------------------------------------------
{
  U32 ulRegister;
  U32 ulValue;

  cprintf(" 1. Write Local");
  cprintf(" Enter Register 99 to return to menu");
  do
  {
    cprintf("Enter Register(Hex) to Write [0 - FC] ");
	scanf("%x",&ulRegister);
	if(ulRegister == 0x99)
		continue;
	if(ulRegister > 0xFC){
      cprintf("Invalid Register                 ");
	  continue;
	}
    cprintf("Enter 32Bit Value(Hex) to Write ");
	scanf("%x",&ulValue);
	SIO4_WriteLocalRegister(ulBdNum, ulRegister, ulValue);

    cprintf("                                           ");
  }while(ulRegister != 0x99);
}

//------------------------------------------------------------------------------
void ReadLocal(void)
//------------------------------------------------------------------------------
{
  U32 ulRegister;

  cprintf(" 2. Read Local");
  cprintf(" Enter Register 99 to return to menu");
  do
  {
    cprintf("Enter Register(Hex) to Read [0 - FC] ");
	scanf("%x",&ulRegister);
	if(ulRegister == 0x99)
		continue;
	if(ulRegister > 0xFC){
      cprintf("Invalid Register                       ");
	  continue;
	}
    cprintf("Register %02X Value is %08X",ulRegister,SIO4_ReadLocalRegister(ulBdNum, ulRegister, &ulErr));
    cprintf("                                         ");

  }while(ulRegister != 0x99);
}

//------------------------------------------------------------------------------
void WriteUSC(void)
//------------------------------------------------------------------------------
{
  U32 ulChannel;
  U32 ulRegister;
  U32 ulValue;

  cprintf(" 3. Write USC");
  cprintf(" Enter Channel 0 to return to menu");
  do
  {
    cprintf("Enter Channel to Write [1-4] ");
	scanf("%ld",&ulChannel);
	if(ulChannel == 0)
		continue;
	if(ulChannel > 4){
      cprintf("Invalid Channel                          ");
	  continue;
	}
    cprintf("Enter Register(Hex) to Write [0 - 3E] ");
	scanf("%lx",&ulRegister);
	if(ulRegister > 0x3E){
      cprintf("Invalid Register                         ");
	  continue;
	}
    cprintf("Enter 16Bit Value(Hex) to Write ");
	scanf("%lx",&ulValue);
	SIO4_WriteUSCRegister(ulBdNum, ulChannel, ulRegister, ulValue);
    cprintf("                                            ");
  }while(ulChannel != 0);
}

//------------------------------------------------------------------------------
void ReadUSC(void)
//------------------------------------------------------------------------------
{
  U32 ulChannel;
  U32 ulRegister;

  cprintf(" 4. Read USC");
  cprintf(" Enter Channel 0 to return to menu");
  do
  {
    cprintf("Enter Channel to Read [1-4] ");
	scanf("%ld",&ulChannel);
	if(ulChannel == 0)
		continue;
	if(ulChannel > 4){
      cprintf("Invalid Channel                  ");
	  continue;
	}
    cprintf("Enter Register(Hex) to Read ");
	scanf("%lx",&ulRegister);
	if(ulRegister > 0x3E){
      cprintf("Invalid Register                     ");
	  continue;
	}
    cprintf("Register %02lX Value is %04lX",ulRegister,SIO4_ReadUSCRegister(ulBdNum, ulChannel, ulRegister, &ulErr));
    cprintf("                                            ");
  }while(ulChannel != 0);
}


//////////////////////////////////////////////////////////////////////
//
//	Begin Test Functions:
//
//////////////////////////////////////////////////////////////////////

void SingleChannelInternalLoopbackTest(U32 ulChannel)
{

	U32	ulLoop;
	HANDLE myHandle;
	DWORD EventStatus;

	// Fill WriteBuffer and ReadBuffer 
    // Puts a count pattern in ReadBuffer
    // Puts a pattern in WriteBuffer of As, 5s, 0s, Fs, Count
	for (ulLoop = 0; ulLoop < FIFOSize[ulChannel]; ulLoop++)
	{
		pReadBuffer[ulLoop] = ulLoop;
		switch(ulLoop % 5)
		{
         case 0:{pWriteBuffer[ulLoop] = 0xAA;}break;
         case 1:{pWriteBuffer[ulLoop] = 0x55;}break;
         case 2:{pWriteBuffer[ulLoop] = 0x00;}break;
         case 3:{pWriteBuffer[ulLoop] = 0xFF;}break;
         case 4:{pWriteBuffer[ulLoop] = (int)ulLoop/5;}break;
	  } // end switch(remainder)
	} // end for 

//	for (ulLoop = 0; ulLoop < (ulBufferSize / 8); ulLoop++)
//		printf("\npReadBuffer[%X] = %X", ulLoop, pReadBuffer[ulLoop]);

	SIO4_Board_Reset(ulBdNum);

	///////////////////////////////////////////////////////
	// Begin setup of SIO4 Registers
	///////////////////////////////////////////////////////

	// local offset 0x0: fw rev reg 
	// this is RO, so we will just display it
	cprintf("\n\tFirmware Revision:\t\t%08lX", SIO4_ReadLocalRegister(ulBdNum, FW_REV, &ulErr));

	// local offset 0x4: bd control reg 
	// used to assign demand mode DMA channel requests
	// since we are not using any DMA in these tests, we will just set it to 0x0
	SIO4_WriteLocalRegister(ulBdNum, BCR, 0x0);
		
	// local offset 0x8: reserved register on all early SIO4 models
	// the PCI-SIO4A and PMC-SIO4AR use this as a RO status register
	SIO4_WriteLocalRegister(ulBdNum, BSR, 0x0);
	
	// local offset 0xC: clock control reg
	// this register is used to enable the clock transceivers for all
	// four channels - this allows separate direction control of clock and data
	// for internal loopback, we will leave this register set to 0x0
	SIO4_WriteLocalRegister(ulBdNum, CLK_CONTROL, 0x0);
	
	// local offset 0x10/0x20/0x30/0x40: channel X Tx FIFO almost flags reg
	// this register is used to set the trigger point for the transmit FIFO almost
	// full and empty flags which can be read from the chan X control status
	// bits D0..15 = almost empty value, bits 16..31 = almost full value
	// for this example we will set each to 16
	SIO4_WriteLocalRegister(ulBdNum, ((ulChannel<<4) | TX_ALMOST), 0x00100010);

	// local offset 0x14/0x24/0x34/0x44: channel X Rx FIFO almost flags reg
	// this register is used to set the trigger point for the receive FIFO almost
	// full and empty flags which can be read from the chan X control status
	// bits D15..0 = almost empty value, bits 31..16 = almost full value
	// for this example we will set each to 16
	SIO4_WriteLocalRegister(ulBdNum, ((ulChannel<<4) | RX_ALMOST), 0x00100010);

	// local offset 0x18/0x28/0x38/0x48: channel X FIFO 
	// a write to this location will store data in the transmit FIFO
	// a read from this location will fetch data from the receive FIFO
	// at this point in the test, we will not write/read to/from the FIFO
	//SIO4_WriteLocalRegister(ulBdNum, ((ulChannel<<4) | FIFO), 0x0);

	// local offset 0x1C/0x2C/0x3C/0x4C: channel X control/status reg
	// this multi-purpose register performs the following control functions:
	// 1. reset transmit and receive FIFOs
	// 2. reset the Zilog chip (not always recommended as both channel in the 
	//    chip are cleared)
	// 3. enable data transceiver directions and upper or lower cable portions
	// this multi-purpose register performs the following status functions:
	// 1. status of the transmit FIFO can be read from bits D8..15
	// 2. status of the receive FIFO can be read from bits D16..31
	// since this test is internal loopback, we will just set to 0x3 to reset FIFOs
	SIO4_WriteLocalRegister(ulBdNum, ((ulChannel<<4) | CONTROL_STATUS), 0x0003);
	Sleep(10);

	// local offset 0x50/0x54/0x58/0x5C: channel X sync detect byte
	// can be used to trigger an interrupt when a specified byte is loaded
	// in to the receive FIFO - bits D0..8 are used to set the sync byte
	// not used for this test so we leave it set to 0x0
	SIO4_WriteLocalRegister(ulBdNum, SYNC_CHARACTER + ulChannel*4, 0x0);
	
	// local offset 0x60: interrupt control register
	// used to enable individual interrupt sources with a b'1' enabling the 
	// source and a b'0' disabling it - the source must be enabled before 
	// an interrupt will occur
	// not used for this test so we leave it set to 0x0
	SIO4_WriteLocalRegister(ulBdNum, ICR, 0x0);

	// local offset 0x64: interrupt status/clear register
	// used to show the status individual interrupt sources or to clear a 
	// pending interrupt by writing a b'1' to the respective bit
	// not used for this test so we leave it set to 0x0
	SIO4_WriteLocalRegister(ulBdNum, ISR, 0x0);

	// local offset 0x68: interrupt edge/level register
	// used to set individual interrupt sources as edge or level triggered
	// not used for this test so we leave it at default
	// SIO4_WriteLocalRegister(ulBdNum, INTR_EDGE_LEVEL, 0x0);

	// local offset 0x68: interrupt high/low register
	// used to set individual interrupt sources as active high or active low
	// not used for this test so we leave it at default
	// SIO4_WriteLocalRegister(ulBdNum, INTR_HIGH_LOW, 0x0);
	
	///////////////////////////////////////////////////////
	// Begin setup of USC Registers
	//
	// NOTE:
	// since there are so many registers located in the Zilog Z16C30 serial 
	// controller chips and so many functions are controlled with each reg,
	// we will only describe the functions we are configuring/changing
	// see the Z16C30 User Manual and Product Specification Databook for more 
	// info and definitions of each reg/bit
	//
	///////////////////////////////////////////////////////

	// local offset (channel base + 0x00/0x01): chan command/address reg
	// for internal loopback, we set mode to internal loopback
	SIO4_WriteUSCRegister(ulBdNum, ulChannel, CHANNEL_COMMAND_ADDRESS, 0x0300);
	
	// local offset (channel base + 0x02/0x03): chan mode reg
	// for internal loopback, we set protocol to async/16x sampling/1 stop bit
	SIO4_WriteUSCRegister(ulBdNum, ulChannel, CHANNEL_MODE, 0x0000);

	// local offset (channel base + 0x04/0x05): chan command/status reg
	// not used for this test - leave set at 0x0 (some RO bits)
	SIO4_WriteUSCRegister(ulBdNum, ulChannel, CHANNEL_COMMAND_STATUS, 0x0000);

	// local offset (channel base + 0x06/0x07): chan control reg
	// not used for this test - leave set at 0x0 (some RO bits)
	SIO4_WriteUSCRegister(ulBdNum, ulChannel, CHANNEL_CONTROL, 0x0000);

	// local offset (channel base + 0x0C/0x0D): test mode data reg
	// not used for this test - leave set at 0x0
	SIO4_WriteUSCRegister(ulBdNum, ulChannel, TEST_MODE_DATA, 0x0000);

	// local offset (channel base + 0x0E/0x0F): test mode control reg
	// not used for this test - leave set at 0x0
	SIO4_WriteUSCRegister(ulBdNum, ulChannel, TEST_MODE_CONTROL, 0x0000);
	
	// local offset (channel base + 0x10/0x11): clock mode control reg
	// this register defines the clock sources used for the serial channel
	// set receive clock source as baud rate generator 0 output
	// set transmit clock source as baud rate generator 0 output
	// set baud rate generator 0 source as Zilog /TxC pin
	SIO4_WriteUSCRegister(ulBdNum, ulChannel, CLOCK_MODE_CONTROL, 0x0324);

	// local offset (channel base + 0x12/0x13): hardware config reg
	// this register sets BRG functions as well as DMA config
	// enable baud rate generator 0 
	// set /RxACK pin control to Rx Acknowledge input
	// set /TxACK pin control to Tx Acknowledge input
	SIO4_WriteUSCRegister(ulBdNum, ulChannel, HARDWARE_CONFIGURATION, 0x0045);

	// local offset (channel base + 0x14/0x15): interrupt vector reg
	// not used for this test - leave set at 0x0
	SIO4_WriteUSCRegister(ulBdNum, ulChannel, INTERRUPT_VECTOR, 0x0000);

	// local offset (channel base + 0x16/0x17): IO control reg
	// this register sets various pin functions of the serial channel
	// set /RxC pin as an input
	// set /TxC pin as an input
	// set /TxD pin as transmit data output
	// set /RxREQ pin control to Rx Request output
	// set /TxREQ pin control to Tx Request output
	SIO4_WriteUSCRegister(ulBdNum, ulChannel, IO_CONTROL, 0x0500);

	// local offset (channel base + 0x18/0x19): interrupt control reg
	// not used for this test - leave set at 0x0
	SIO4_WriteUSCRegister(ulBdNum, ulChannel, INTERRUPT_CONTROL, 0x0);

	// local offset (channel base + 0x1A/0x1B): daisy chain ctrl reg
	// not used for this test - leave set at 0x0
	SIO4_WriteUSCRegister(ulBdNum, ulChannel, DAISY_CHAIN_CONTROL, 0x0);

	// local offset (channel base + 0x1C/0x1D): misc int ctrl reg
	// not used for this test - leave set at 0x0
	SIO4_WriteUSCRegister(ulBdNum, ulChannel, MISC_INTERRUPT_STATUS, 0x0);

	// local offset (channel base + 0x1E/0x1F): stat int ctrl reg
	// not used for this test - leave set at 0x0
	SIO4_WriteUSCRegister(ulBdNum, ulChannel, STATUS_INTERRUPT_CONTROL, 0x0);

	// local offset (channel base + 0x20): receive data reg
	// can be used to read data directly from Z16C30 internal FIFOs
	// not used for this test - leave set at 0x0
	// SIO4_ReadUSCRegister(ulBdNum, ulChannel, RECEIVE_DATA, &ulErr);

	// local offset (channel base + 0x22/0x23): receive mode reg
	// sets up the mode of the serial channel receiver
	// set Rx enable to enable without auto enables
	// Rx char length 8 bits
	// Rx parity even
	// Rx data encoding NRZ
	SIO4_WriteUSCRegister(ulBdNum, ulChannel, RECEIVE_MODE, 0x0002);

	// local offset (channel base + 0x24/0x25): receive cmd/stat reg
	// not used for this test - leave set at 0x0
	//SIO4_WriteUSCRegister(ulBdNum, ulChannel, RECEIVE_COMMAND_STATUS, 0x0);

	// local offset (channel base + 0x26/0x27): receive int ctrl reg
	// not used for this test - leave set at 0x0
	SIO4_WriteUSCRegister(ulBdNum, ulChannel, RECEIVE_INTERRUPT_CONTROL, 0x0);

	// local offset (channel base + 0x28/0x29): receive int ctrl reg
	// not used for this test - leave set at 0x0
	SIO4_WriteUSCRegister(ulBdNum, ulChannel, RECEIVE_SYNC, 0x0);

	// local offset (channel base + 0x2A/0x2B): receive cnt limit reg
	// not used for this test - leave set at 0x0
	SIO4_WriteUSCRegister(ulBdNum, ulChannel, RECEIVE_COUNT_LIMIT, 0x0);

	// local offset (channel base + 0x2C/0x2D): receive char cnt reg
	// not used for this test - leave set at 0x0 - RO
	// SIO4_WriteUSCRegister(ulBdNum, ulChannel, RECEIVE_CHARACTER_COUNT, 0x0);
	
	// local offset (channel base + 0x2E/0x2F): time constant 0 reg
	// this reg is used to set the time constant for baud rate generator 0
	// output clock = inputclock/(TC0_value + 1)
	// so we use a value of 19 to divide onboard 20MHz/20 = 1MHz
	// set clock speed to 1MHz - with 16x sampling this gives us an
	// actual baud rate of about 62.5 Kbaud
	SIO4_WriteUSCRegister(ulBdNum, ulChannel, TIME_CONSTANT_0, 0x0013);
	
	// local offset (channel base + 0x30): transmit data reg
	// can be used to write data directly from Z16C30 internal FIFOs
	// not used for this test - leave set at 0x0
	// SIO4_WriteUSCRegister(ulBdNum, ulChannel, TRANSMIT_DATA, 0x0);

	// local offset (channel base + 0x32/0x33): transmit mode reg
	// sets up the mode of the serial channel transmitter
	// set Tx enable to enable without auto enables
	// Tx char length 8 bits
	// Tx parity disabled
	// Tx parity even
	// Tx data encoding NRZ
	SIO4_WriteUSCRegister(ulBdNum, ulChannel, TRANSMIT_MODE, 0x0002);

	// local offset (channel base + 0x34/0x35): transmit cmd/stat reg
	// not used for this test - leave set at 0x0
	// SIO4_WriteLocalRegister(ulBdNum, TRANSMIT_COMMAND_STATUS, 0x0);
	
	// local offset (channel base + 0x36/0x37): transmit int ctrl reg
	// not used for this test - leave set at 0x0
	SIO4_WriteUSCRegister(ulBdNum, ulChannel, TRANSMIT_INTERRUPT_CONTROL, 0x0);

	// local offset (channel base + 0x38/0x39): transmit sync reg
	// not used for this test - leave set at 0x0
	SIO4_WriteUSCRegister(ulBdNum, ulChannel, TRANSMIT_SYNC, 0x0);
	
	// local offset (channel base + 0x3A/0x3B): transmit cnt limit reg
	// not used for this test - leave set at 0x0
	SIO4_WriteUSCRegister(ulBdNum, ulChannel, TRANSMIT_COUNT_LIMIT, 0x0);
	
	// local offset (channel base + 0x3C/0x3D): transmit char cnt reg
	// not used for this test - leave set at 0x0
	SIO4_WriteUSCRegister(ulBdNum, ulChannel, TRANSMIT_CHARACTER_COUNT, 0x0);
	
	// local offset (channel base + 0x3E/0x3F): TC1
	// not used for this test - leave set at 0x0
	SIO4_WriteUSCRegister(ulBdNum, ulChannel, TIME_CONSTANT_1, 0x0);
	
    if(ulErr = SIO4_Open_DMA_Channel(ulBdNum, 0)){ // Use DMA Channel 0
		   ShowAPIError(ulErr);
		   return;
	}


	////////////////////////////////////////////////////////////////////////
	//	setup complete, ready to test data xfer
	////////////////////////////////////////////////////////////////////////
	// load data from WriteBuffer in to FIFO
    if(ulErr = SIO4_DMA_To_FIFO(ulBdNum,ulChannel,0,FIFOSize[ulChannel],pWriteBuffer)){
		   ShowAPIError(ulErr);
	       SIO4_Close_DMA_Channel(ulBdNum, 0);
		   return;
	}

	/* We could have used PIO mode and loaded the FIFO ourselves */
//	for (ulLoop = 0; ulLoop < ulBufferSize; ulLoop++)
//		SIO4_WriteLocalRegister(ulBdNum, ((ulChannel<<4) | FIFO), *(pWriteBuffer + ulLoop) );

    SIO4_Attach_Interrupt(ulBdNum,&myHandle,Local,(RX_FIFO_ALMOST_FULL << ((ulChannel-1) * 4)));
    // We have data loaded to the FIFO, now enable the USC to
	// transfer the data and do the loopback.
	// We have to do it this way to see the interrupt on the older 
	// versions of the board because they are level sensitive,
	// not edge triggered.
	//SIO4_WriteUSCRegister(ulBdNum, ulChannel, IO_CONTROL, 0x0500);

	printf("\nICR -> %X", 
		SIO4_ReadLocalRegister(ulBdNum, ICR, &ulErr) );

   EventStatus = WaitForSingleObject(myHandle,2 * 1000); // Wait for the interrupt

   switch(EventStatus)
   {
	case WAIT_OBJECT_0:
		{
			cprintf("\n\tInterrupt was requested    ...");
			// disable the irq so it can happen again next time
			// made no difference - API still saved last IRQ src
			// SIO4_DisableInterrupt(ulBdNum, (RX_FIFO_ALMOST_FULL << ((ulChannel-1) * 4)) );
		}
		break;
	default:
		cprintf("\n\tInterrupt was NOT requested...");
		break;
   }

    /* We would use the following if using PIO mode
	// poll on the Tx FIFO to go empty	
	// !kbhit() gives an out if something goes wrong
	while(((ulValue = SIO4_ReadLocalRegister(ulBdNum, CONTROL_STATUS, &ulErr)) & 0x100) && !kbhit() )
		cprintf("\n\tchannel %d status --> %X", ulChannel, ulValue);
	Sleep(10);
    */

	Sleep(10);

	cprintf("\n\n\tByte\tRead\tExpected");
	cprintf("\n\tNumber:\tValue:\tValue:");

    if(ulErr = SIO4_DMA_From_FIFO(ulBdNum,ulChannel,0,FIFOSize[ulChannel],pReadBuffer)){
		   ShowAPIError(ulErr);
	       SIO4_Close_DMA_Channel(ulBdNum, 0);
		   return;
	}
	
	/* We would use this to fetch the data in PIO mode
	// fetch data from FIFO and load in to ReadBuffer
	for (ulLoop = 0; ulLoop < ulBufferSize; ulLoop++)
		*(pReadBuffer + ulLoop) = SIO4_ReadLocalRegister(ulBdNum, ((ulChannel<<4) | FIFO), &ulErr)&0xFF;
	*/

	printf("\nStatus -> %X", 
		SIO4_ReadLocalRegister(ulBdNum, ((ulChannel<<4) | CONTROL_STATUS), &ulErr) );


	// Only display the first 16 words
	for (ulLoop = 0; ulLoop < ((ulBufferSize < 0x10)?ulBufferSize:0x10); ulLoop++)
		cprintf("\n\t%d\t%02X\t%02X", ulLoop, (*(pReadBuffer + ulLoop)&0xFF), *(pWriteBuffer + ulLoop));

//	Do not close handle as per PLX docs	
//	CloseHandle(myHandle);
	SIO4_Close_DMA_Channel(ulBdNum, 0);


} // end SingleChannelInternalLoopbackTest()

//==============================================================================
void ShowAPIError(U32 APIReturnCode)
{
  printf("\n  An Error has Occured in the API for the Device Driver.");
  printf("\n  Below is all the information that was available about ");
  printf("\n  the encountered Error.                             \n ");

  switch (APIReturnCode)
  {
    case ApiConfigAccessFailed:
      printf("\n  Return Code  %d  = ApiConfigAccessFailed\n", APIReturnCode);
      break;
    case ApiDmaChannelInvalid:
      printf("\n  Return Code  %d  = ApiDmaChannelInvalid\n", APIReturnCode);
      break;
    case ApiDmaChannelUnavailable:
      printf("\n  Return Code  %d  = ApiDmaChannelUnavailable\n", APIReturnCode);
      break;
    case ApiDmaCommandInvalid:
      printf("\n  Return Code  %d  = ApiDmaCommandInvalid\n", APIReturnCode);
      break;
    case ApiDmaDone:
      printf("\n  Return Code  %d  = ApiDmaDone\n", APIReturnCode);
      break;
    case ApiDmaInProgress:
      printf("\n  Return Code  %d  = ApiDmaInProgress\n", APIReturnCode);
      break;
    case ApiDmaInvalidChannelPriority:
      printf("\n  Return Code  %d  = ApiDmaInvalidChannelPriority\n", APIReturnCode);
      break;
    case ApiDmaNotPaused:
      printf("\n  Return Code  %d  = ApiDmaNotPaused\n", APIReturnCode);
      break;
    case ApiDmaPaused:
      printf("\n  Return Code  %d  = ApiDmaPaused\n", APIReturnCode);
      break;
    case ApiEepromTypeNotSupported:
      printf("\n  Return Code  %d  = ApiEepromTypeNotSupported\n", APIReturnCode);
      break;
    case ApiFailed:
      printf("\n  Return Code  %d  = ApiFailed\n", APIReturnCode);
      break;
    case ApiInsufficientResources:
      printf("\n  Return Code  %d  = ApiInsufficientResources\n", APIReturnCode);
      break;
    case ApiInvalidAccessType:
      printf("\n  Return Code  %d  = ApiInvalidAccessType\n", APIReturnCode);
      break;
    case ApiInvalidAddress:
      printf("\n  Return Code  %d  = ApiInvalidAddress\n", APIReturnCode);
      break;
    case ApiInvalidDeviceInfo:
      printf("\n  Return Code  %d  = ApiInvalidDeviceInfo\n", APIReturnCode);
      break;
    case ApiInvalidHandle:
      printf("\n  Return Code  %d  = ApiInvalidHandle\n", APIReturnCode);
      break;
    case ApiInvalidPowerState:
      printf("\n  Return Code  %d  = ApiInvalidPowerState\n", APIReturnCode);
      break;
    case ApiInvalidRegister:
      printf("\n  Return Code  %d  = ApiInvalidRegister\n", APIReturnCode);
      break;
    case ApiInvalidSize:
      printf("\n  Return Code  %d  = ApiInvalidSize\n", APIReturnCode);
      break;
    case ApiInvalidUserPin:
      printf("\n  Return Code  %d  = ApiInvalidUserPin\n", APIReturnCode);
      break;
    case ApiNoActiveDriver:
      printf("\n  Return Code  %d  = ApiNoActiveDriver\n", APIReturnCode);
      break;
    case ApiNullParam:
      printf("\n  Return Code  %d  = ApiNullParam\n", APIReturnCode);
      break;
    case ApiPciTimeout:
      printf("\n  Return Code  %d  = ApiPciTimeout\n", APIReturnCode);
      break;
    case ApiPowerDown:
      printf("\n  Return Code  %d  = ApiPowerDown\n", APIReturnCode);
      break;
	case ApiInvalidBoardNumber:
      printf("\n  Return Code  %d  = ApiInvalidBoardNumber\n", APIReturnCode);
      break;
	case ApiInvalidDMANumWords:
      printf("\n  Return Code  %d  = ApiInvalidDMANumWords\n", APIReturnCode);
      break;
	case ApiDMABufferTooSmall:
      printf("\n  Return Code  %d  = ApiDMABufferTooSmall\n", APIReturnCode);
      break;
	case ApiInvalidDMABufferAddr:
      printf("\n  Return Code  %d  = ApiInvalidDMABufferAddr\n", APIReturnCode);
      break;
	case ApiInvalidChannel:
      printf("\n  Return Code  %d  = ApiInvalidChannel\n", APIReturnCode);
      break;
    case ApiUnsupportedFunction:
      printf("\n  Return Code  %d  = ApiUnsupportedFunction\n", APIReturnCode);
      break;
    default:
      printf("\n  Return Code  %d  Is not Listed as a valid return code. \n", APIReturnCode);
      break;
  }
}

//---------------------------------------------------------------------------
void ChanneltoChannelExternalLoopbackTest(U32 ulTxChannel, U32 ulRxChannel)
//---------------------------------------------------------------------------
{

	U32		ulLoop;
	HANDLE	myHandle;
	DWORD	EventStatus;
	U32		ulWriteValue;


	// Fill WriteBuffer and ReadBuffer 
    // Puts a count pattern in ReadBuffer
    // Puts a pattern in WriteBuffer of As, 5s, 0s, Fs, Count
	for (ulLoop = 0; ulLoop < FIFOSize[ulTxChannel]; ulLoop++)
	{
		pReadBuffer[ulLoop] = ulLoop;
		switch(ulLoop % 5)
		{
         case 0:{pWriteBuffer[ulLoop] = 0xAA;}break;
         case 1:{pWriteBuffer[ulLoop] = 0x55;}break;
         case 2:{pWriteBuffer[ulLoop] = 0x00;}break;
         case 3:{pWriteBuffer[ulLoop] = 0xFF;}break;
         case 4:{pWriteBuffer[ulLoop] = (int)ulLoop/5;}break;
	  } // end switch(remainder)
	} // end for 


	SIO4_Board_Reset(ulBdNum);
	
	// local offset 0x4: bd control reg 
	// used to assign demand mode DMA channel requests
	// since we are not using any DMA in these tests, we will just set it to 0x0
	SIO4_WriteLocalRegister(ulBdNum, BCR, 0x0);

	///////////////////////////////////////////////////////
	// Begin setup of SIO4 Registers - Tx Channel
	///////////////////////////////////////////////////////

	// local offset 0xC: clock control reg
	// this register is used to enable the clock transceivers for all
	// four channels - this allows separate direction control of clock and data
	// enable drive upper clock
	// enable receive lower clock
	ulWriteValue = (0x9 << ( (ulTxChannel - 1) * 4));
	SIO4_WriteLocalRegister(ulBdNum, CLK_CONTROL, ulWriteValue);
	
	// local offset 0x10/0x20/0x30/0x40: channel X Tx FIFO almost flags reg
	// this register is used to set the trigger point for the transmit FIFO almost
	// full and empty flags which can be read from the chan X control status
	// bits D0..15 = almost empty value, bits 16..31 = almost full value
	// for this example we will set each to 16
	SIO4_WriteLocalRegister(ulBdNum, ((ulTxChannel << 4) | TX_ALMOST), 0x00100010);

	// local offset 0x14/0x24/0x34/0x44: channel X Rx FIFO almost flags reg
	// this register is used to set the trigger point for the receive FIFO almost
	// full and empty flags which can be read from the chan X control status
	// bits D15..0 = almost empty value, bits 31..16 = almost full value
	// for this example we will set each to 16
	SIO4_WriteLocalRegister(ulBdNum, ((ulTxChannel << 4) | RX_ALMOST), 0x00100010);

	// local offset 0x1C/0x2C/0x3C/0x4C: channel X control/status reg
	// this multi-purpose register performs the following control functions:
	// 1. reset transmit and receive FIFOs
	// 2. reset the Zilog chip (not always recommended as both channel in the 
	//    chip are cleared)
	// 3. enable data transceiver directions and upper or lower cable portions
	// this multi-purpose register performs the following status functions:
	// 1. status of the transmit FIFO can be read from bits D8..15
	// 2. status of the receive FIFO can be read from bits D16..31
	//
	// reset Tx and Rx FIFOs (these bits clear automatically)
	// Enable transmitters on upper cable portion
	// Enable receivers on lower cable portion
	SIO4_WriteLocalRegister(ulBdNum, ((ulTxChannel << 4) | CONTROL_STATUS), 0x0027);
	Sleep(1);

	///////////////////////////////////////////////////////
	// Begin setup of Serial Controller Registers
	//
	// NOTE:
	// since there are so many registers located in the Zilog Z16C30 serial 
	// controller chips and so many functions are controlled with each reg,
	// we will only describe the functions we are configuring/changing
	// see the Z16C30 User Manual and Product Specification Databook for more 
	// info and definitions of each reg/bit
	//
	///////////////////////////////////////////////////////

	// local offset (channel base + 0x00/0x01): chan command/address reg
	// 0x0 sets channel to standard operation
	SIO4_WriteUSCRegister(ulBdNum, ulTxChannel, CHANNEL_COMMAND_ADDRESS, 0x0000);
	
	// local offset (channel base + 0x02/0x03): chan mode reg
	// set receive mode to HDLC
	// set transmit mode to HDLC
	SIO4_WriteUSCRegister(ulBdNum, ulTxChannel, CHANNEL_MODE, 0x0606);

	// local offset (channel base + 0x10/0x11): clock mode control reg
	// this register defines the clock sources used for the serial channel
	//
	// set receive clock source as disabled (this is the tx channel - 
	// /RxC pin will be used to bring 20 MHz clock in) 
	// set transmit clock source as baud rate generator 0 output
	// set baud rate generator 0 source as Zilog /RxC pin
	SIO4_WriteUSCRegister(ulBdNum, ulTxChannel, CLOCK_MODE_CONTROL, 0x0220);

	// local offset (channel base + 0x12/0x13): hardware config reg
	// this register sets BRG functions as well as DMA config
	// enable baud rate generator 0 
	// set /RxACK pin control to Rx Acknowledge input
	// set /TxACK pin control to Tx Acknowledge input
	SIO4_WriteUSCRegister(ulBdNum, ulTxChannel, HARDWARE_CONFIGURATION, 0x0045);

	// local offset (channel base + 0x16/0x17): IO control reg
	// this register sets various pin functions of the serial channel
	// set /RxC pin as an input to supply BRG0
	// set /TxC pin as BRG0 output
	// set /TxD pin as transmit data output
	// set /RxREQ pin control to Rx Request output
	// set /TxREQ pin control to Tx Request output
	SIO4_WriteUSCRegister(ulBdNum, ulTxChannel, IO_CONTROL, 0x0520);

	// local offset (channel base + 0x22/0x23): receive mode reg
	// sets up the mode of the serial channel receiver
	// disable Rx since this is the Tx channel
	// Rx char length 8 bits
	// Rx parity even
	// Rx data encoding NRZ
	SIO4_WriteUSCRegister(ulBdNum, ulTxChannel, RECEIVE_MODE, 0x0000);

	// local offset (channel base + 0x2E/0x2F): time constant 0 reg
	// this reg is used to set the time constant for baud rate generator 0
	// output clock = inputclock/(TC0_value + 1)
	// standard onboard oscillator is 20MHz
	// 20 MHz/(7 + 1) = 2.5 MHz
	SIO4_WriteUSCRegister(ulBdNum, ulTxChannel, TIME_CONSTANT_0, 0x0007);
	
	// local offset (channel base + 0x32/0x33): transmit mode reg
	// sets up the mode of the serial channel transmitter
	// set Tx enable to enable without auto enables
	// Tx char length 8 bits
	// Tx parity disabled
	// Tx parity even
	// Tx data encoding NRZ
	SIO4_WriteUSCRegister(ulBdNum, ulTxChannel, TRANSMIT_MODE, 0x0002);

	///////////////////////////////////////////////////////
	// Begin setup of SIO4 Registers - Rx Channel
	///////////////////////////////////////////////////////

	// local offset 0xC: clock control reg
	// this register is used to enable the clock transceivers for all
	// four channels - this allows separate direction control of clock and data
	// enable drive lower clock
	// enable receive upper clock
	// do a read/modify/write
	ulWriteValue = SIO4_ReadLocalRegister(ulBdNum, CLK_CONTROL, &ulErr);
	ulWriteValue |= (0x6 << ( (ulRxChannel - 1) * 4));
	SIO4_WriteLocalRegister(ulBdNum, CLK_CONTROL, ulWriteValue);

	// local offset 0x10/0x20/0x30/0x40: channel X Tx FIFO almost flags reg
	// this register is used to set the trigger point for the transmit FIFO almost
	// full and empty flags which can be read from the chan X control status
	// bits D0..15 = almost empty value, bits 16..31 = almost full value
	// for this example we will set each to 16
	SIO4_WriteLocalRegister(ulBdNum, ((ulRxChannel << 4) | TX_ALMOST), 0x00100010);

	// local offset 0x14/0x24/0x34/0x44: channel X Rx FIFO almost flags reg
	// this register is used to set the trigger point for the receive FIFO almost
	// full and empty flags which can be read from the chan X control status
	// bits D15..0 = almost empty value, bits 31..16 = almost full value
	// for this example we will set each to 16
	SIO4_WriteLocalRegister(ulBdNum, ((ulRxChannel << 4) | RX_ALMOST), 0x00100010);


	// local offset 0x1C/0x2C/0x3C/0x4C: channel X control/status reg
	// this multi-purpose register performs the following control functions:
	// 1. reset transmit and receive FIFOs
	// 2. reset the Zilog chip (not always recommended as both channel in the 
	//    chip are cleared)
	// 3. enable data transceiver directions and upper or lower cable portions
	// this multi-purpose register performs the following status functions:
	// 1. status of the transmit FIFO can be read from bits D8..15
	// 2. status of the receive FIFO can be read from bits D16..31
	//
	// reset Tx and Rx FIFOs (these bits clear automatically)
	// Enable transmitters on lower cable portion
	// Enable receivers on upper cable portion
	SIO4_WriteLocalRegister(ulBdNum, ((ulRxChannel << 4) | CONTROL_STATUS), 0x001B);
	Sleep(1);


	// local offset 0x50/0x54/0x58/0x5C: channel X sync detect byte
	// can be used to trigger an interrupt when a specified byte is loaded
	// in to the receive FIFO - bits D0..8 are used to set the sync byte
		// no specific features used in this reg
	SIO4_WriteLocalRegister(ulBdNum, SYNC_CHARACTER + ulRxChannel*4, 0x0);
	
	// local offset 0x60: interrupt control register
	// used to enable individual interrupt sources with a b'1' enabling the 
	// source and a b'0' disabling it - the source must be enabled before 
	// an interrupt will occur
	// no specific features used in this reg
	SIO4_WriteLocalRegister(ulBdNum, ICR, 0x0);

	// local offset 0x64: interrupt status/clear register
	// used to show the status individual interrupt sources or to clear a 
	// pending interrupt by writing a b'1' to the respective bit
	// no specific features used in this reg
	SIO4_WriteLocalRegister(ulBdNum, ISR, 0x0);

	///////////////////////////////////////////////////////
	// Begin setup of USC Registers
	//
	// NOTE:
	// since there are so many registers located in the Zilog Z16C30 serial 
	// controller chips and so many functions are controlled with each reg,
	// we will only describe the functions we are configuring/changing
	// see the Z16C30 User Manual and Product Specification Databook for more 
	// info and definitions of each reg/bit
	//
	///////////////////////////////////////////////////////

	// local offset (channel base + 0x00/0x01): chan command/address reg
	// 0x0 sets card to standard operation
	SIO4_WriteUSCRegister(ulBdNum, ulRxChannel, CHANNEL_COMMAND_ADDRESS, 0x0000);
	
	// local offset (channel base + 0x02/0x03): chan mode reg
	// set receive mode to HDLC
	// set transmit mode to HDLC
	SIO4_WriteUSCRegister(ulBdNum, ulRxChannel, CHANNEL_MODE, 0x0606);
	
	// local offset (channel base + 0x10/0x11): clock mode control reg
	// this register defines the clock sources used for the serial channel
	// set transmit clock source as Zilog /TxC pin (not really used)
	// set receive clock source as Zilog /RxC pin - this will come from cable
	SIO4_WriteUSCRegister(ulBdNum, ulRxChannel, CLOCK_MODE_CONTROL, 0x0011);

	// local offset (channel base + 0x12/0x13): hardware config reg
	// this register sets BRG functions as well as DMA config
	// Disable baud rate generator 0 
	// set /RxACK pin control to Rx Acknowledge input
	// set /TxACK pin control to Tx Acknowledge input
	SIO4_WriteUSCRegister(ulBdNum, ulRxChannel, HARDWARE_CONFIGURATION, 0x0044);

	// local offset (channel base + 0x16/0x17): IO control reg
	// this register sets various pin functions of the serial channel
	// set /RxC pin as an input 
	// set /TxC pin as an input 
	// set /TxD pin as transmit data output
	// set /RxREQ pin control to Rx Request output
	// set /TxREQ pin control to Tx Request output
	SIO4_WriteUSCRegister(ulBdNum, ulRxChannel, IO_CONTROL, 0x0500);

	// local offset (channel base + 0x22/0x23): receive mode reg
	// sets up the mode of the serial channel receiver
	// enable Rx without autoenables
	// Rx char length 8 bits
	// Rx parity even
	// Rx data encoding NRZ
	SIO4_WriteUSCRegister(ulBdNum, ulRxChannel, RECEIVE_MODE, 0x0002);

	// local offset (channel base + 0x32/0x33): transmit mode reg
	// sets up the mode of the serial channel transmitter
	// disable Tx since this is Rx channel
	// Tx char length 8 bits
	// Tx parity disabled
	// Tx parity even
	// Tx data encoding NRZ
	SIO4_WriteUSCRegister(ulBdNum, ulRxChannel, TRANSMIT_MODE, 0x0000);


	//-----------------------------------------------------------------------
    // Begin Data Transfers
	//-----------------------------------------------------------------------
	if(ulErr = SIO4_Open_DMA_Channel(ulBdNum, 0)){ // Use DMA Channel 0
		   ShowAPIError(ulErr);
		   return;
	}

	////////////////////////////////////////////////////////////////////////
	//	setup complete, ready to test data xfer
	////////////////////////////////////////////////////////////////////////
	// load data from WriteBuffer in to FIFO
    if(ulErr = SIO4_DMA_To_FIFO(ulBdNum,ulTxChannel,0,FIFOSize[ulTxChannel],pWriteBuffer)){
//    if(ulErr = SIO4_DMA_To_FIFO(ulBdNum,ulTxChannel,0,FIFOSize[ulTxChannel],pWriteBuffer)){
		   ShowAPIError(ulErr);
	       SIO4_Close_DMA_Channel(ulBdNum, 0);
		   return;
	}

	/* We could have used PIO mode and loaded the FIFO ourselves */
//	for (ulLoop = 0; ulLoop < ulBufferSize; ulLoop++)
//		SIO4_WriteLocalRegister(ulBdNum, ((ulChannel<<4) | FIFO), *(pWriteBuffer + ulLoop) );


	// IRQ sources:
	// SYNC_DETECTED
	// TX_FIFO_ALMOST_EMPTY
	// RX_FIFO_ALMOST_FULL
	// USC_INTERRUPT
	
	SIO4_Attach_Interrupt(ulBdNum,&myHandle,Local,(RX_FIFO_ALMOST_FULL << ((ulRxChannel - 1) * 4)));

	//printf("ICR: %08X", SIO4_ReadLocalRegister(ulBdNum, ICR, &ulErr) );

    // We have data loaded to the FIFO, now enable the USC to
	// transfer the data and do the loopback.
	// We have to do it this way to see the interrupt on the older 
	// versions of the board because they are level sensitive,
	// not edge triggered.
//	SIO4_WriteUSCRegister(ulBdNum, ulTxChannel, IO_CONTROL, 0x0520);


  EventStatus = WaitForSingleObject(myHandle,3000); // Wait up to 3 seconds for the interrupt

   switch(EventStatus)
   {
	case WAIT_OBJECT_0:
		cprintf("\n\tInterrupt was requested    ...");
		break;
	default:
		cprintf("\n\tInterrupt was NOT requested...");
		break;
   }

    /* We would use the following if using PIO mode
	// poll on the Tx FIFO to go empty	
	// !kbhit() gives an out if something goes wrong
	while(((ulValue = SIO4_ReadLocalRegister(ulBdNum, CONTROL_STATUS, &ulErr)) & 0x100) && !kbhit() )
		cprintf("\n\tchannel %d status --> %X", ulChannel, ulValue);
	Sleep(10);
    */

	Sleep(10);

	cprintf("\n\n\tByte\tRead\tExpected");
	cprintf("\n\tNumber:\tValue:\tValue:");

    if(ulErr = SIO4_DMA_From_FIFO(ulBdNum,ulRxChannel,0,FIFOSize[ulTxChannel],pReadBuffer)){
		   ShowAPIError(ulErr);
	       SIO4_Close_DMA_Channel(ulBdNum, 0);
		   return;
	}
	
	/* We would use this to fetch the data in PIO mode
	// fetch data from FIFO and load in to ReadBuffer
	for (ulLoop = 0; ulLoop < ulBufferSize; ulLoop++)
		*(pReadBuffer + ulLoop) = SIO4_ReadLocalRegister(ulBdNum, ((ulChannel<<4) | FIFO), &ulErr) & 0xFF;
	*/
	
	// Only display the first 16 words
	for (ulLoop = 0; ulLoop < ((ulBufferSize < 0x10)?ulBufferSize:0x10); ulLoop++)
		cprintf("\n\t%d\t%02X\t%02X", ulLoop, (*(pReadBuffer + ulLoop)&0xFF), *(pWriteBuffer + ulLoop));


	SIO4_Close_DMA_Channel(ulBdNum, 0);


} // end ChanneltoChannelExternalLoopbackTest()

U16 CalculateFIFOSize(U16 iChannel)
{
	U32	ulSize = 0;
	// Init the Board
	SIO4_Board_Reset(ulBdNum);
	SIO4_WriteLocalRegister(ulBdNum, ((iChannel<<4) | CONTROL_STATUS), 0x0003);

	// continue to write as long as CONTROL_STATUS bit D11 is high
	// indicating that the FIFO is not full
	// currently, 32k or 0x8000 is the largest FIFO available on the
	// SIO4 cards, so break out if we reach 0x8000+
	do{
		SIO4_WriteLocalRegister(ulBdNum, ((iChannel<<4) | FIFO), 0x00);
		ulSize++;
		if(ulSize > 0x8001)
			break;
	}while(	SIO4_ReadLocalRegister(ulBdNum, ((iChannel<<4) | CONTROL_STATUS), &ulErr) & 0x800);
	
	if(ulSize > 0x8001)
		return(1);
	else
	{		
//		printf("\nFIFOSize[%d] = %x", iChannel, ulSize);
		FIFOSize[iChannel] = ulSize;
		printf("\nFIFOSize[%d] = %x", iChannel, FIFOSize[iChannel]);

		return(0);
	}

}


