/****************************************************************************
 *  File    : SIO4intface.H                                                 *
 ****************************************************************************
 *  Content :                                                               *
 *                                                                          *
 *  This file contains the 'C' functions that shall provide the             *
 *  interface to the Windows Driver for the PCI-SIO4                        *
 *                                                                          *
 *  The SOI4 contains 4 high speed serial channels                          *
 *                                                                          *
 *  The following functions are provided by the driver:                     *
 *                                                                          *
 *	SIO4_FindBoards                   : finds all PLX based SIO4 devices    *
 *	SIO4_Open                         : open a SIO4 device                  *
 *	SIO4_Close                        : close a SIO4 device                 *
 *	SIO4_Reset                        : reset the board                     *
 *	SIO4_ReadLocalRegister            : read a SIO4 local register          *
 *	SIO4_WriteLocalRegister           : write a SIO4 local register         *
 *	SIO4_ReadUSCRegister              : read a SIO4 USC register            *
 *	SIO4_WriteUSCRegister             : write a SIO4 USC register           *
 *	SIO4_ReadConfigRegister           : read PCI configuration register     *
 *	SIO4_WriteConfigRegister          : write PCI configuration register    *
 *  SIO4_EnableInterrupt			  : enables the specified interrupt     *
 *  SIO4_DisableInterrupt			  : disables the specified interrupt    *
 *	SIO4_DMA_Mode		              : sets demand mode for DMA transfers  *
 *  SIO4_Attach_Interrupt			  : attaches user handle to interrupt   *
 *                                                                          *
 ****************************************************************************/
#ifdef __cplusplus
extern "C" {
#endif


#ifndef __SIO4_H
#define __SIO4_H

// var types
typedef unsigned char  U8;
typedef unsigned short U16;
typedef unsigned long  U32;
typedef double         U64;


// Interrupt Type Definitions
typedef enum _INTR_TYPE
{
    Local,
    DmaCh0,
    DmaCh1,
} INTR_TYPE;


// These are the options for the ulRegister parameter sent to the
// SIO4_ReadLocalRegister and SIO4_WriteLocalRegister routines.
// Board Register Addresses
#define		FW_REV				0x000  // 32 Bit Access
#define     BCR					0x004
#define     BSR					0x008
#define     CLK_CONTROL			0x00C
#define     CH1_TX_ALMOST		0x010
#define     CH1_RX_ALMOST		0x014
#define     CH1_FIFO		    0x018
#define     CH1_CONTROL_STATUS	0x01C
#define     CH2_TX_ALMOST		0x020
#define     CH2_RX_ALMOST		0x024
#define     CH2_FIFO		    0x028
#define     CH2_CONTROL_STATUS	0x02C
#define     CH3_TX_ALMOST		0x030
#define     CH3_RX_ALMOST		0x034
#define     CH3_FIFO		    0x038
#define     CH3_CONTROL_STATUS	0x03C
#define     CH4_TX_ALMOST		0x040
#define     CH4_RX_ALMOST		0x044
#define     CH4_FIFO		    0x048
#define     CH4_CONTROL_STATUS	0x04C
#define     CH1_SYNC_DETECTED	0x050
#define     CH2_SYNC_DETECTED	0x054
#define     CH3_SYNC_DETECTED	0x058
#define     CH4_SYNC_DETECTED	0x05C
#define     ICR					0x060
#define     ISR					0x064
#define     INTR_EDGE_LEVEL		0x068
#define     INTR_HIGH_LOW	    0x06C
#define     CH1_PIN_SRC			0x080
#define     CH2_PIN_SRC			0x084
#define     CH3_PIN_SRC			0x088
#define     CH4_PIN_SRC			0x08C
#define     CH1_PROG_CLK	    0x0A0
#define     CH2_PROG_CLK	    0x0A4
#define     CH3_PROG_CLK	    0x0A8
#define     CH4_PROG_CLK	    0x0AC
#define     FEATURES		    0x0FC

// These make life easy for programming across channels
#define		TX_ALMOST			0x00
#define		RX_ALMOST			0x04
#define		FIFO				0x08
#define		CONTROL_STATUS		0x0C
#define		SYNC_CHARACTER		0x50

// These are the options for the ulRegister parameter sent to the
// SIO4_ReadUSCRegister and SIO4_WriteUSCRegister routines.
// USC Registers
#define		CHANNEL_COMMAND_ADDRESS		0x00
#define		CHANNEL_MODE				0x02
#define		CHANNEL_COMMAND_STATUS		0x04
#define		CHANNEL_CONTROL				0x06
#define		TEST_MODE_DATA				0x0C
#define		TEST_MODE_CONTROL			0x0E
#define		CLOCK_MODE_CONTROL			0x10
#define		HARDWARE_CONFIGURATION		0x12
#define		INTERRUPT_VECTOR			0x14
#define		IO_CONTROL					0x16
#define		INTERRUPT_CONTROL			0x18
#define		DAISY_CHAIN_CONTROL			0x1A
#define		MISC_INTERRUPT_STATUS		0x1C
#define		STATUS_INTERRUPT_CONTROL	0x1E
#define		RECEIVE_DATA				0x20
#define		RECEIVE_MODE				0x22
#define		RECEIVE_COMMAND_STATUS		0x24
#define		RECEIVE_INTERRUPT_CONTROL	0x26
#define		RECEIVE_SYNC				0x28
#define		RECEIVE_COUNT_LIMIT			0x2A
#define		RECEIVE_CHARACTER_COUNT		0x2C
#define		TIME_CONSTANT_0				0x2E
#define		TRANSMIT_DATA				0x30
#define		TRANSMIT_MODE				0x32
#define		TRANSMIT_COMMAND_STATUS		0x34
#define		TRANSMIT_INTERRUPT_CONTROL	0x36
#define		TRANSMIT_SYNC				0x38
#define		TRANSMIT_COUNT_LIMIT		0x3A
#define		TRANSMIT_CHARACTER_COUNT	0x3C
#define		TIME_CONSTANT_1				0x3E


/*** FUNCTION PROTOTYPES ***/

/****************************************************************************
 *  Function       : SIO4_FindBoards                                        *
 *                                                                          *
 *  Description    : Finds All SIO4 PLX based boards in the system          *
 *                                                                          *
 *  Return Value   : Number of Boards found                                 *
 *                                                                          *
 *  Implementation :                                                        *
 *                                                                          *
 ************************** PARAMETERS **************************************
 *  pDeviceInfo [OUT] : Bd #: Bus: Slot: Type: Ser# for PLX boards found.   *
 *  ulError     [OUT] : 0 on success, non-zero error code on failure.       *
 ****************************************************************************/
U32 SIO4_FindBoards(char *pDeviceInfo, U32 *ulError);

/****************************************************************************
 *  Function       : SIO4_Open                                              *
 *                                                                          *
 *  Description    : Opens an interface to a SIO4 device IN THE DRIVER.     *
 *                                                                          *
 *  Return Value   : Returns 0 on success, non-zero error code on failure.  *
 *                                                                          *
 *  Implementation :                                                        *
 *                                                                          *
 ************************** PARAMETERS **************************************
 *  ulDeviceNumber [IN] : Board number to open starting with one (1).       *
 ****************************************************************************/
U32 SIO4_Open(U32 ulDeviceNumber);


/****************************************************************************
 *  Function       : SIO4_Close                                              *
 *                                                                          *
 *  Description    : Closes an interface to a SIO4 device.                   *
 *                                                                          *
 *  Return Value   : Returns 0 on success, non-zero error code on failure.  *
 *                                                                          *
 *  Implementation :                                                        *
 *                                                                          *
 ************************** PARAMETERS **************************************
 *  ulDeviceNumber [IN] : Board number to close.                            *
 ****************************************************************************/
U32 SIO4_Close(U32 ulDeviceNumber);


/****************************************************************************
 *  Function       : SIO4_Board_Reset                                       *
 *                                                                          *
 *  Description    : Resets the SIO4 Board (All Channels).                  *
 *                                                                          *
 *  Return Value   : Returns 0 on success, non-zero error code on failure.  *
 *                                                                          *
 *  Implementation :                                                        *
 *                                                                          *
 ************************** PARAMETERS **************************************
 *  ulDeviceNumber [IN] : Board number to reset.                            *
 ****************************************************************************/
U32 SIO4_Board_Reset(U32 ulDeviceNumber);


/****************************************************************************
 *  Function       : SIO4_ReadLocalRegister                                 *
 *                                                                          *
 *  Description    : Reads the register requested and returns its value.    *
 *                                                                          *
 *  Return Value   : Returns register value read.                           *
 *                                                                          *
 *  Implementation :                                                        *
 *                                                                          *
 ************************** PARAMETERS **************************************
 *  ulDeviceNumber [IN] : Board number.                                     *
 *  ulRegister     [IN] : Indicates which SIO4 local register to read.      *
 *  pulError       [IN] : Pointer to location to return error code          *
 ****************************************************************************/
U32 SIO4_ReadLocalRegister(U32     ulDeviceNumber,
						   U32     ulRegister,
						   U32    *pulError);


/****************************************************************************
 *  Function       : SIO4_WriteLocalRegister                                *
 *                                                                          *
 *  Description    : Write the value to the register requested.             *
 *                                                                          *
 *  Return Value   : Returns 0 on success, non-zero error code on failure.  *
 *                                                                          *
 *  Implementation :                                                        *
 *                                                                          *
 ************************** PARAMETERS **************************************
 *  ulDeviceNumber [IN] : Board number.                                     *
 *  ulRegister     [IN] : Indicates which SIO4 local register to write.     *
 *  ulValue        [IN] : Value to write to the register.                   *
 ****************************************************************************/
U32 SIO4_WriteLocalRegister(U32    ulDeviceNumber,
							U32    ulRegister,
							U32    ulValue);


/****************************************************************************
 *  Function       : SIO4_ReadUSCRegister                                   *
 *                                                                          *
 *  Description    : Reads the register requested and returns its value.    *
 *                                                                          *
 *  Return Value   : Returns register value read.                           *
 *                                                                          *
 *  Implementation :                                                        *
 *                                                                          *
 ************************** PARAMETERS **************************************
 *  ulDeviceNumber [IN] : Board number.                                     *
 *  ulChannel      [IN] : Indicates which SIO4 Channel to read.             *
 *  ulRegister     [IN] : Indicates which SIO4 USC register to read.        *
 *  pulError       [IN] : Pointer to location to return register value read.*
 ****************************************************************************/
U32 SIO4_ReadUSCRegister(U32     ulDeviceNumber,
						 U32     ulChannel,
						 U32     ulRegister,
						 U32    *pulError);


/****************************************************************************
 *  Function       : SIO4_WriteUSCRegister                                  *
 *                                                                          *
 *  Description    : Write the value to the register requested.             *
 *                                                                          *
 *  Return Value   : Returns 0 on success, non-zero error code on failure.  *
 *                                                                          *
 *  Implementation :                                                        *
 *                                                                          *
 ************************** PARAMETERS **************************************
 *  ulDeviceNumber [IN] : Board number.                                     *
 *  ulChannel      [IN] : Indicates which SIO4 Channel to write.            *
 *  uRegister      [IN] : Indicates which SIO4 USC register to write.       *
 *  ulValue        [IN] : Value to write to the register.                   *
 ****************************************************************************/
U32 SIO4_WriteUSCRegister(U32    ulDeviceNumber,
                          U32    ulChannel,
                          U32    ulRegister,
                          U32    ulValue);


/****************************************************************************
 *  Function       : SIO4_ReadConfigRegister                                *
 *                                                                          *
 *  Description    : Reads the PCI configuration register                   *
 *                                                                          *
 *  Return Value   : Returns register value read.                           *
 *                                                                          *
 *  Implementation :                                                        *
 *                                                                          *
 ************************** PARAMETERS **************************************
 *  ulDeviceNumber [IN] : Board number.                                     *
 *  ulRegister     [IN] : Indicates which SIO4 PCI Config register to read. *
 *  pulError       [IN] : Pointer to location to return error code.         *
 ****************************************************************************/
U32 SIO4_ReadConfigRegister(U32     ulDeviceNumber,
						    U32     ulRegister,
						    U32    *pulError);


/****************************************************************************
 *  Function       : SIO4_WriteConfigRegister                               *
 *                                                                          *
 *  Description    : Write the value to the PCi Configuration register      *
 *                                                                          *
 *  Return Value   : Returns 0 on success, non-zero error code on failure.  *
 *                                                                          *
 *  Implementation :                                                        *
 *                                                                          *
 ************************** PARAMETERS **************************************
 *  ulDeviceNumber [IN] : Board number.                                     *
 *  ulRegister     [IN] : Indicates which SIO4 PCI Config register to write *
 *  ulValue        [IN] : Value to write to the register.                   *
 ****************************************************************************/
U32 SIO4_WriteLocalRegister(U32    ulDeviceNumber,
							U32    ulRegister,
							U32    ulValue);


/****************************************************************************
 *  Function       : SIO4_EnableInterrupt                                   *
 *                                                                          *
 *  Description    : Enables the Specified Interrupt(s) on the Device.      *
 *                                                                          *
 *  Return Value   : Returns 0 on success, non-zero error code on failure.  *
 *                                                                          *
 *  Implementation :                                                        *
 *                                                                          *
 ************************** PARAMETERS **************************************
 *  ulDeviceNumber     [IN] : Board number.                                 *
 *  ulValue            [IN] : Interrupt to Enable                           *
 ****************************************************************************/
U32 SIO4_EnableInterrupt(U32    ulDeviceNumber,
						 U32    ulValue);


/****************************************************************************
 *  Function       : SIO4_DisableInterrupt                                   *
 *                                                                          *
 *  Description    : Disables the Specified Interrupt(s) on the Device.     *
 *                                                                          *
 *  Return Value   : Returns 0 on success, non-zero error code on failure.  *
 *                                                                          *
 *  Implementation :                                                        *
 *                                                                          *
 ************************** PARAMETERS **************************************
 *  ulDeviceNumber     [IN] : Board number.                                 *
 *  ulValue            [IN] : Interrupt to Disable                          *
 ****************************************************************************/
U32 SIO4_DisableInterrupt(U32    ulDeviceNumber,
						  U32    ulValue);


/****************************************************************************
 *  Function       : SIO4_Attach_Interrupt                                  *
 *                                                                          *
 *  Description    : Attaches User Handle to Interrupt                .     *
 *                                                                          *
 *  Return Value   : Returns 0 on success, non-zero error code on failure.  *
 *                                                                          *
 *  Implementation :                                                        *
 *                                                                          *
 ************************** PARAMETERS **************************************
 *  ulDeviceNumber     [IN] : Board number.                                 *
 *  ulValue            [IN] : Interrupt to Disable                          *
 ****************************************************************************/
U32 SIO4_Attach_Interrupt(U32		ulDeviceNumber,
						  HANDLE	*userHandle,
						  INTR_TYPE ulInterrupt,
						  U32		ulLocalIntr);

/****************************************************************************
 *  Function       : SIO4_Dma_Mode                                          *
 *                                                                          *
 *  Description    : Sets DMA Mode, Demand or Non Demand              .     *
 *                                                                          *
 *  Return Value   : Returns 0 on success, non-zero error code on failure.  *
 *                                                                          *
 *  Implementation :                                                        *
 *                                                                          *
 ************************** PARAMETERS **************************************
 *  ulDeviceNumber     [IN] : Board number.                                 *
 *  ulMode             [IN] : Demand Mode Enable                            *
 ****************************************************************************/
U32 SIO4_Dma_Mode(U32		ulDeviceNumber,
				  U32		ulMode);

/****************************************************************************
 *  Function       : SIO4_Open_DMA_Channel                                  *
 *                                                                          *
 *  Description    : Opens the specified DMA Channel for transfers    .     *
 *                                                                          *
 *  Return Value   : Returns 0 on success, non-zero error code on failure.  *
 *                                                                          *
 *  Implementation :                                                        *
 *                                                                          *
 ************************** PARAMETERS **************************************
 *  ulDeviceNumber     [IN] : Board number.                                 *
 *  ulDMAChannel       [IN] : DMA Channel to open                           *
 ****************************************************************************/
U32 SIO4_Open_DMA_Channel(U32 ulDeviceNumber,
						  U32 ulDMAChannel);


/****************************************************************************
 *  Function       : SIO4_DMA_From_FIFO                                     *
 *                                                                          *
 *  Description    : Transfers data from FIFO to user specified memory.     *
 *                                                                          *
 *  Return Value   : Returns 0 on success, non-zero error code on failure.  *
 *                                                                          *
 *  Implementation :                                                        *
 *                                                                          *
 ************************** PARAMETERS **************************************
 *  ulDeviceNumber     [IN] : Board number.                                 *
 *  ulSerChannel       [IN] : Serial Channel number to transfer from        *
 *  ulDMAChannel       [IN] : DMA Channel to use (Must have been opened)    *
 *  ulWords            [IN] : Number of words to transfer                   *
 *  pulData            [IN] : Pointer to user memory for data transfer      *
 ****************************************************************************/
U32 SIO4_DMA_From_FIFO(U32 ulDeviceNumber,
					   U32 ulSerChannel,
					   U32 ulDMAChannel,
					   U32 ulWords,
					   U32* ulData);


/****************************************************************************
 *  Function       : SIO4_DMA_To_FIFO                                       *
 *                                                                          *
 *  Description    : Transfers data from user specified memory to FIFO.     *
 *                                                                          *
 *  Return Value   : Returns 0 on success, non-zero error code on failure.  *
 *                                                                          *
 *  Implementation :                                                        *
 *                                                                          *
 ************************** PARAMETERS **************************************
 *  ulDeviceNumber     [IN] : Board number.                                 *
 *  ulSerChannel       [IN] : Serial Channel number to transfer to          *
 *  ulDMAChannel       [IN] : DMA Channel to use (Must have been opened)    *
 *  ulWords            [IN] : Number of words to transfer                   *
 *  pulData            [IN] : Pointer to user memory for data transfer      *
 ****************************************************************************/
U32 SIO4_DMA_To_FIFO(U32 ulDeviceNumber,
					   U32 ulSerChannel,
					   U32 ulDMAChannel,
					   U32 ulWords,
					   U32* ulData);


/****************************************************************************
 *  Function       : SIO4_Close_DMA_Channel                                  *
 *                                                                          *
 *  Description    : Closes the specified DMA Channel                 .     *
 *                                                                          *
 *  Return Value   : Returns 0 on success, non-zero error code on failure.  *
 *                                                                          *
 *  Implementation :                                                        *
 *                                                                          *
 ************************** PARAMETERS **************************************
 *  ulDeviceNumber     [IN] : Board number.                                 *
 *  ulDMAChannel       [IN] : DMA Channel to close                          *
 ****************************************************************************/
U32 SIO4_Close_DMA_Channel(U32 ulDeviceNumber,
						   U32 ulDMAChannel);


#endif  // #ifndef __SIO4_H

#ifdef __cplusplus
}
#endif
