// $URL: http://subversion:8080/svn/gsc/trunk/drivers/LINUX/PEX8111/PEX8111_Linux_1.x.x.x_GSC_DN/driver/reg.c $
// $Rev: 47359 $
// $Date: 2020-06-02 12:12:00 -0500 (Tue, 02 Jun 2020) $

// PEX8111: Device Driver: source file

#include "main.h"



// data types *****************************************************************

typedef struct
{
	unsigned long	reg;
	unsigned long	type;	// Alt, PCI, PLX, GSC
	unsigned long	size;	// 1, 2, 3 or 4 bytes
	unsigned long	offset;	// Allign per size
	VADDR_T			vaddr;	// Virtual Address
	const os_bar_t*	bar;
} dev_reg_t;



// variables *******************************************************************

static const os_bar_t	_alt_region	=
{
	/* index		*/	0,	// not needed for ALT registers
	/* offset		*/	0,	// not needed for ALT registers
	/* reg			*/	0,	// not needed for ALT registers
	/* flags		*/	0,	// not needed for ALT registers
	/* io_mapped	*/	0,	// not needed for ALT registers
	/* phys_adrs	*/	0,	// not needed for ALT registers
	/* size			*/	0xA000,
	/* requested	*/	0,	// not needed for ALT registers
	/* vaddr		*/	0	// not needed for ALT registers
} ;



//*****************************************************************************
static void _reg_decode(
	const os_bar_t*	bar,
	unsigned long	reg,
	dev_reg_t*		rt)
{
	rt->reg		= reg;
	rt->bar		= bar;
	rt->type	= GSC_REG_TYPE(reg);
	rt->offset	= GSC_REG_OFFSET(reg);
	rt->size	= GSC_REG_SIZE(reg);
	rt->vaddr	= (VADDR_T) rt->offset;
	rt->vaddr	= (VADDR_T) ((unsigned long) rt->vaddr + (unsigned long) bar->vaddr);
}



//*****************************************************************************
static int _reg_validate(dev_reg_t* rt)
{
	unsigned long	ul;
	int				ret	= -EINVAL;

	for (;;)	// We'll use a loop for convenience.
	{
		// Are there extranious bits in the register id?
		ul	= GSC_REG_ENCODE(rt->type, rt->size, rt->offset);

		if (ul != rt->reg)
			break;

		// Does the register extend past the end of the region?
		ul	= rt->offset + rt->size - 1;

		if (ul >= rt->bar->size)
			break;

		// Is the register's size valid?

		if (strchr("\001\002\004", (int) rt->size) == NULL)
			break;

		// Is the register properly aligned?

		if ((rt->size == 2) && (rt->offset & 0x1))
			break;
		else if ((rt->size == 4) && (rt->offset & 0x3))
			break;

		//	We don't test the "type" since that is validated
		//	before the register is decoded.

		ret	= 0;
		break;
	}

	return(ret);
}



//*****************************************************************************
int dev_reg_mod_alt(dev_data_t* dev, gsc_reg_t* arg)
{
	int			ret;
	dev_reg_t	rt;

	_reg_decode(&_alt_region, arg->reg, &rt);
	ret	= _reg_validate(&rt);

	if ((ret == 0) && (rt.type != GSC_REG_TYPE_PCI))
		ret	= -EINVAL;

	if (ret == 0)
	{
		os_reg_pci_tx_u32(dev, 1, GSC_REG_OFFSET(GSC_PCI_8111_MCRIR), rt.offset);
		os_reg_pci_mx_u32(dev, 1, GSC_REG_OFFSET(GSC_PCI_8111_MCRDR), arg->value, arg->mask);
	}

	return(ret);
}



//*****************************************************************************
int dev_reg_read_alt(dev_data_t* dev, gsc_reg_t* arg)
{
	int			ret;
	dev_reg_t	rt;
	u32			val;

	_reg_decode(&_alt_region, arg->reg, &rt);
	ret	= _reg_validate(&rt);

	if ((ret == 0) && (rt.type != GSC_REG_TYPE_ALT))
		ret	= -EINVAL;

	if (ret == 0)
	{
		os_reg_pci_tx_u32(dev, 1, GSC_REG_OFFSET(GSC_PCI_8111_MCRIR), rt.offset);
		val	= os_reg_pci_rx_u32(dev, 1, GSC_REG_OFFSET(GSC_PCI_8111_MCRDR));

		switch (rt.size)
		{
			default:
			case 1:

				arg->value	= (val >> (8 * (rt.offset % 4))) & 0xFF;
				break;

			case 2:

				arg->value	= (val >> (16 * (rt.offset % 2))) & 0xFFFF;
				break;

			case 4:

				arg->value	= val;
				break;
		}
	}

	return(ret);
}



//*****************************************************************************
int dev_reg_write_alt(dev_data_t* dev, gsc_reg_t* arg)
{
	int			ret;
	dev_reg_t	rt;

	_reg_decode(&_alt_region, arg->reg, &rt);
	ret	= _reg_validate(&rt);

	if ((ret == 0) && (rt.type != GSC_REG_TYPE_PCI))
		ret	= -EINVAL;

	if (ret == 0)
	{
		os_reg_pci_tx_u32(dev, 1, GSC_REG_OFFSET(GSC_PCI_8111_MCRIR), rt.offset);
		os_reg_pci_tx_u32(dev, 1, GSC_REG_OFFSET(GSC_PCI_8111_MCRDR), arg->value);
	}

	return(ret);
}



//*****************************************************************************
int dev_reg_mod_pci(dev_data_t* dev, gsc_reg_t* arg)
{
	int			ret;
	dev_reg_t	rt;

	_reg_decode(&_alt_region, arg->reg, &rt);
	ret	= _reg_validate(&rt);

	if ((ret == 0) && (rt.type != GSC_REG_TYPE_PCI))
		ret	= -EINVAL;

	if (ret == 0)
	{
		switch (rt.size)
		{
			default:
			case 1:	os_reg_pci_mx_u8 (dev, 1, rt.offset, arg->value, arg->mask);	break;
			case 2:	os_reg_pci_mx_u16(dev, 1, rt.offset, arg->value, arg->mask);	break;
			case 4:	os_reg_pci_mx_u32(dev, 1, rt.offset, arg->value, arg->mask);	break;
		}
	}

	return(ret);
}


