// $URL: http://subversion:8080/svn/gsc/trunk/drivers/LINUX/PEX8111/PEX8111_Linux_1.x.x.x_GSC_DN/driver/ioctl.c $
// $Rev: 51709 $
// $Date: 2022-10-10 13:56:06 -0500 (Mon, 10 Oct 2022) $

// PEX8111: Device Driver: source file

#include "main.h"



//*****************************************************************************
static s32 _payload_size_cap(dev_data_t* dev)
{
	s32	bytes;
	u32	cap;
	u32	reg;

	reg	= gsc_reg_read(dev, GSC_PCI_8111_DCR);
	cap	= GSC_FIELD_DECODE(reg, 2, 0);

	switch (cap)
	{
		default:	bytes	= -1;	break;
		case 0:		bytes	= 128;	break;
		case 1:		bytes	= 256;	break;
		case 2:		bytes	= 512;	break;
		case 3:		bytes	= 1024;	break;
		case 4:		bytes	= 2048;	break;
		case 5:		bytes	= 4096;	break;
	}

	return(bytes);
}



//*****************************************************************************
static int _query(dev_data_t* dev, s32* arg)
{
	switch (arg[0])
	{
		default:								arg[0]	= PEX8111_IOCTL_QUERY_ERROR;	break;
		case PEX8111_QUERY_COUNT:				arg[0]	= PEX8111_IOCTL_QUERY_LAST;		break;
		case PEX8111_QUERY_DEVICE_TYPE:			arg[0]	= dev->board_type;				break;
		case PEX8111_QUERY_PAYLOAD_SIZE_CAP:	arg[0]	= _payload_size_cap(dev);		break;
	}

	return(0);
}



//*****************************************************************************
static int _comp_flow_ctl_crdt(dev_data_t* dev, s32* arg)
{
	gsc_reg_t	reg;
	int			ret;

	switch (arg[0])
	{
		default:

			ret	= -EINVAL;
			break;

		case -1:

			reg.reg	= GSC_PLX_8111_TCC0R;
			ret		= dev_reg_read_alt(dev, &reg);
			arg[0]	= (s32) GSC_FIELD_DECODE(reg.value, 22, 22);
			break;

		case PEX8111_COMP_FLOW_CTL_CRDT_NO:
		case PEX8111_COMP_FLOW_CTL_CRDT_YES:

			reg.reg		= GSC_PLX_8111_TCC0R;
			reg.value	= GSC_FIELD_ENCODE(arg[0], 22, 22);
			reg.mask	= GSC_FIELD_ENCODE(1, 22, 22);
			ret			= dev_reg_mod_alt(dev, &reg);
			break;
	}

	return(ret);
}



//*****************************************************************************
static int _blind_prefetch(dev_data_t* dev, s32* arg)
{
	gsc_reg_t	reg;
	int			ret;

	switch (arg[0])
	{
		default:

			ret	= -EINVAL;
			break;

		case -1:

			reg.reg	= GSC_PCI_8111_DSCR;
			ret		= gsc_reg_read_ioctl(dev, &reg);
			arg[0]	= (s32) GSC_FIELD_DECODE(reg.value, 0, 0);
			break;

		case PEX8111_BLIND_PREFETCH_DISABLE:
		case PEX8111_BLIND_PREFETCH_ENABLE:

			reg.reg		= GSC_PCI_8111_DSCR;
			reg.value	= arg[0];
			reg.mask	= 0x1;
			ret			= dev_reg_mod_pci(dev, &reg);
			break;
	}

	return(ret);
}



//*****************************************************************************
static int _prefetch_size(dev_data_t* dev, s32* arg)
{
	gsc_reg_t	reg;
	int			ret;
	s32			val;

	switch (arg[0])
	{
		default:

			ret	= -EINVAL;
			break;

		case -1:

			reg.reg	= GSC_PLX_8111_PCR;
			ret		= dev_reg_read_alt(dev, &reg);
			arg[0]	= (s32) GSC_FIELD_DECODE(reg.value, 29, 27);
			break;

		case PEX8111_PREFETCH_SIZE_4096:

			val	= 1;
			ret	= _comp_flow_ctl_crdt(dev, &val);

			if (ret)
				break;

			FALLTHROUGH;	/* fall through */

		case PEX8111_PREFETCH_SIZE_CL:
		case PEX8111_PREFETCH_SIZE_64:
		case PEX8111_PREFETCH_SIZE_128:
		case PEX8111_PREFETCH_SIZE_256:
		case PEX8111_PREFETCH_SIZE_512:
		case PEX8111_PREFETCH_SIZE_1024:
		case PEX8111_PREFETCH_SIZE_2048:

			reg.reg		= GSC_PLX_8111_PCR;
			reg.value	= GSC_FIELD_ENCODE(arg[0], 29, 27);
			reg.mask	= GSC_FIELD_ENCODE(0x7, 29, 27);
			ret			= dev_reg_mod_alt(dev, &reg);
			break;
	}

	return(ret);
}



//*****************************************************************************
static int _cache_line_size(dev_data_t* dev, s32* arg)
{
	int	ret;
	u8	val;

	if (arg[0] == -1)
	{
		ret		= 0;
		val		= os_reg_pci_rx_u8(dev, 1, 0xC);
		arg[0]	= val;
	}
	else if ((arg[0] & 0xFF) == arg[0])
	{
		ret	= 0;
		val	= arg[0];
		os_reg_pci_tx_u8(dev, 1, 0xC, val);
	}
	else
	{
		ret	= -EINVAL;
	}

	return(ret);
}



//*****************************************************************************
static int _payload_size_max(dev_data_t* dev, s32* arg)
{
	gsc_reg_t	reg;
	int			ret;
	u16			val;

	switch (arg[0])
	{
		default:

			ret	= -EINVAL;
			break;

		case -1:

			ret		= 0;
			val		= os_reg_pci_rx_u16(dev, 1, 0x68);
			arg[0]	= GSC_FIELD_DECODE(val, 7, 5);
			break;

		case PEX8111_PAYLOAD_SIZE_MAX_128:
		case PEX8111_PAYLOAD_SIZE_MAX_256:
		case PEX8111_PAYLOAD_SIZE_MAX_512:
		case PEX8111_PAYLOAD_SIZE_MAX_1024:
		case PEX8111_PAYLOAD_SIZE_MAX_2048:
		case PEX8111_PAYLOAD_SIZE_MAX_4096:

			reg.reg		= GSC_PCI_8111_PEDCR;
			reg.value	= GSC_FIELD_ENCODE(arg[0], 7, 5);
			reg.mask	= GSC_FIELD_ENCODE(7, 7, 5);
			ret			= dev_reg_mod_pci(dev, &reg);
			break;
	}

	return(ret);
}



//*****************************************************************************
static int _max_read_req_size(dev_data_t* dev, s32* arg)
{
	gsc_reg_t	reg;
	int			ret;
	u16			val;

	switch (arg[0])
	{
		default:

			ret	= -EINVAL;
			break;

		case -1:

			ret		= 0;
			val		= os_reg_pci_rx_u16(dev, 1, 0x68);
			arg[0]	= GSC_FIELD_DECODE(val, 14, 12);
			break;

		case PEX8111_MAX_READ_REQ_SIZE_128:
		case PEX8111_MAX_READ_REQ_SIZE_256:
		case PEX8111_MAX_READ_REQ_SIZE_512:
		case PEX8111_MAX_READ_REQ_SIZE_1024:
		case PEX8111_MAX_READ_REQ_SIZE_2048:
		case PEX8111_MAX_READ_REQ_SIZE_4096:

			reg.reg		= GSC_PCI_8111_PEDCR;
			reg.value	= GSC_FIELD_ENCODE(arg[0], 14, 12);
			reg.mask	= GSC_FIELD_ENCODE(7, 14, 12);
			ret			= dev_reg_mod_pci(dev, &reg);
			break;
	}

	return(ret);
}



// variables *******************************************************************

const gsc_ioctl_t	dev_ioctl_list[]	=
{
	{ PEX8111_IOCTL_REG_READ,			(void*) gsc_reg_read_ioctl	},
	{ PEX8111_IOCTL_REG_WRITE,			(void*) gsc_reg_write_ioctl	},
	{ PEX8111_IOCTL_REG_MOD,			(void*) gsc_reg_mod_ioctl	},
	{ PEX8111_IOCTL_QUERY,				(void*) _query				},
	{ PEX8111_IOCTL_BLIND_PREFETCH,		(void*) _blind_prefetch		},
	{ PEX8111_IOCTL_PREFETCH_SIZE,		(void*) _prefetch_size		},
	{ PEX8111_IOCTL_COMP_FLOW_CTL_CRDT,	(void*) _comp_flow_ctl_crdt	},
	{ PEX8111_IOCTL_CACHE_LINE_SIZE,	(void*) _cache_line_size	},
	{ PEX8111_IOCTL_PAYLOAD_SIZE_MAX,	(void*) _payload_size_max	},
	{ PEX8111_IOCTL_MAX_READ_REQ_SIZE,	(void*) _max_read_req_size	},

	{ -1, NULL }
};


