/***
***  write.c
***
***  General description of this file:
***     Device driver source code for General Standards gsc16aio168
***     family of Delta-Sigma A/D board. This file is part of the Linux
***     driver source distribution for this board.
***
***     This file is not necessary to compile application programs, therefore
***     should NOT be included in binary only driver distributions.
***
***  Copyrights (c):
***     General Standards Corporation (GSC), Feb 2004
***
***  Author:
***     Evan Hillan (evan@generalstandards.com)
***
***  Support:
***     Primary support for this driver is provided by GSC. 
***
***  Platform (tested on, may work with others):
***     Linux, kernel version 2.4.18+, Red Hat distribution, Intel hardware.
***
***/
#include <linux/module.h>
#include <linux/fs.h>
#include <linux/vmalloc.h>
#include <linux/pci.h>
#include <linux/mm.h>
#include <linux/proc_fs.h>
#include <linux/sched.h>
#include <linux/interrupt.h>
#include <linux/timer.h>
#include <linux/ioctl.h>

#include <asm/io.h>
#include <asm/system.h>
#include <asm/uaccess.h>
#include <asm/atomic.h>

#include "sysdep.h"
#include "gsc16ao4mf_ioctl.h"
#include "internals.h"

#include "plx_regs.h"

/************************************************************************/
/* write operation:                        */
/************************************************************************/
/************************************************************************/
/* write operation: either polled or uses PLX DMA on CH0                 */
/************************************************************************/
int device_write(struct file *fp, const char *buf, size_t size, loff_t * lt)
{
    struct device_board *device = (struct device_board *)fp->private_data;
    int nsamples = size / sizeof(long);
    int bor_reg;
    unsigned long regval;

    /* verify parameters */
    if (nsamples <= 0) {
#ifdef DEBUG
        printk(KERN_INFO GSC_NAME "(%d): device_write - zero sample return\n", device->minor);
#endif
        return (0);
    }

    if (!access_ok(VERIFY_READ, (unsigned long)buf, nsamples * sizeof(long))) {
#ifdef DEBUG
        printk(KERN_INFO GSC_NAME "(%d): device_write - access not OK\n", device->minor);
#endif
        return (-EFAULT);
    }

    // if in circular mode, make sure the buffer is ready.  If busy for non-blocking,
    // return immediately.  For blocking wait for the buffer to come around.

    bor_reg = readlocal(device,BUFFER_OPS_REG);
    if ((fp->f_flags & O_NONBLOCK) && (bor_reg&BOR_CIRCULAR_BUFFER))
    {
        device->error = DEVICE_SUCCESS;
        return (-EAGAIN);
    }
    // if circular, wait for buffer ready
    if (bor_reg&BOR_CIRCULAR_BUFFER)
    {
        atomic_set(&device->irq_event_pending[EVENT_LOAD_READY],TRUE);

        // enable the int pass-through
        regval = readlocal(device,BOARD_CTRL_REG);
        regval &= (~(BCR_IRQ_MASK));
        regval |= (BCR_IRQ_LOAD_READY);
        writelocal(device,regval,BOARD_CTRL_REG);

        // start wait for buffer
        regval = readlocal(device,BUFFER_OPS_REG);
        regval |= (BOR_LOAD_REQUEST);
        writelocal(device,regval,BUFFER_OPS_REG);

        device->timeout=FALSE;
        device->watchdog_timer.expires=jiffies+device->timeout_seconds*HZ;
        add_timer(&device->watchdog_timer);
#ifdef DEBUG
        printk(KERN_INFO GSC_NAME "(%d): device_write about to wait for load ready\n", device->minor);
#endif
        wait_event_interruptible(*device->event_queue_waiting[EVENT_LOAD_READY],(!atomic_read(&device->irq_event_pending[EVENT_LOAD_READY])));
        if (signal_pending(current)) {
            del_timer_sync(&device->watchdog_timer);
#ifdef DEBUG
            printk(KERN_INFO GSC_NAME "(%d): device_write load ready signal quit.\n", device->minor);
#endif
            DisableIrqPlx(device);
            return (-ERESTARTSYS); 
        }
        if (device->timeout)
        {
            printk(KERN_ERR GSC_NAME "(%d): timeout during wait for load ready*************************\n", device->minor);
            atomic_set(&device->irq_event_pending[EVENT_LOAD_READY],FALSE);
            device->error = DEVICE_IOCTL_TIMEOUT;
            return (-EIO);
        }
        else
            del_timer_sync(&device->watchdog_timer);

#ifdef DEBUG
        //printk("device_write after wait for load ready\n");
#endif
    }
    else // wait for space in the buffer, as in the low-quarter state.
    {
        atomic_set(&device->irq_event_pending[EVENT_OUT_BUFFER_LOW_QUARTER],TRUE);

        // enable the local int 
        regval = readlocal(device,BOARD_CTRL_REG);
        regval &= (~(BCR_IRQ_MASK));
        regval |= (BCR_IRQ_OUT_BUFFER_LOW_QUARTER);
        writelocal(device,regval,BOARD_CTRL_REG);
        regval = readlocal(device,BUFFER_OPS_REG);

        if (!(regval&BOR_BUFFER_LOW_QUARTER))
        {

            device->timeout=FALSE;
            device->watchdog_timer.expires=jiffies+device->timeout_seconds*HZ;
            add_timer(&device->watchdog_timer);
#ifdef DEBUG
            printk(KERN_ERR GSC_NAME "(%d): device_write about to wait for low quarter\n", device->minor);
#endif
            wait_event_interruptible(*device->event_queue_waiting[EVENT_OUT_BUFFER_LOW_QUARTER],
                (!atomic_read(&device->irq_event_pending[EVENT_OUT_BUFFER_LOW_QUARTER])));
            if (signal_pending(current)) {
                del_timer_sync(&device->watchdog_timer);
#ifdef DEBUG
                printk(KERN_INFO GSC_NAME "(%d): device_write low quarter signal quit.\n", device->minor);
#endif
                DisableIrqPlx(device);
                return (-ERESTARTSYS); 
            }
            if (device->timeout)
            {
                printk(KERN_ERR GSC_NAME "(%d): timeout during wait for low quarter\n", device->minor);
                atomic_set(&device->irq_event_pending[EVENT_OUT_BUFFER_LOW_QUARTER],FALSE);
                device->error = DEVICE_IOCTL_TIMEOUT;
                return (-EIO);
            }
            else
                del_timer_sync(&device->watchdog_timer);
        }
        else
        {
#ifdef DEBUG
            printk(KERN_ERR GSC_NAME "(%d): device_write already at low quarter\n", device->minor);
#endif
            atomic_set(&device->irq_event_pending[EVENT_OUT_BUFFER_LOW_QUARTER],FALSE);

            // enable the local int 
            regval = readlocal(device,BOARD_CTRL_REG);
            regval &= (~(BCR_IRQ_MASK));
            writelocal(device,regval,BOARD_CTRL_REG);
        }
    }

    // now ready to transfer data
    if (device->dmaMode == DMA_ENABLE) { // use DMA for the write.
        unsigned long regval;
        //  int time_remain;
#ifdef DEBUG
        printk(KERN_INFO GSC_NAME "(%d): device_write_dma attempt - %p %p %d\n", device->minor,device->dma_data[TX].intermediateBuffer,buf,size);
#endif
        if (copy_from_user(device->dma_data[TX].intermediateBuffer,buf,size))
        {
            device->error = DEVICE_RESOURCE_ALLOCATION_ERROR;
            return (-EIO);
        }
        // grab the DMA semaphore to ensure that no other channels step on 
        // the transfer

        if (down_interruptible(&device->pHardware->dmaSem)){
            printk(KERN_INFO GSC_NAME "(%d): device_write_dma - semaphore error...******************************\n", device->minor);
            return -ERESTARTSYS;
        }
#ifdef DEBUG
        printk(KERN_INFO GSC_NAME "(%d): device_write_dma have semaphore\n", device->minor);
#endif

        writel(NON_DEMAND_DMA_MODE, DMAMode0(device->pHardware));
        writel(device->dma_data[TX].intermediatePhysicalAddr, DMAPCIAddr0(device->pHardware));
        writel(0x18, DMALocalAddr0(device->pHardware));
        writel(size, DMAByteCnt0(device->pHardware));

        writel(0, DMADescrPtr0(device->pHardware));
        writel(0, DMAArbitr(device->pHardware));
        writel(0, DMAThreshold(device->pHardware));

        regval = readl(DMACmdStatus(device->pHardware));
        regval &= STOP_DMA_CMD_0_MASK;
        writel(regval, DMACmdStatus(device->pHardware));
        // OK, here we go! 
        regval = readl(DMACmdStatus(device->pHardware));
        regval |= CH0_DMA_ENABLE_MASK | CH0_DMA_CLEAR_IRQ_MASK;
        writel(regval, DMACmdStatus(device->pHardware));
        regval |= CH0_DMA_START_MASK;
        atomic_set(&device->irq_event_pending[EVENT_DMA_PENDING],TRUE);
        writel(regval, DMACmdStatus(device->pHardware));

#ifdef DEBUG
        //printk(KERN_INFO GSC_NAME "(%d): device_write_dma - about to sleep...\n", device->minor);
#endif
        device->timeout=FALSE;
        device->watchdog_timer.expires=jiffies+device->timeout_seconds*HZ;
        add_timer(&device->watchdog_timer);

        wait_event_interruptible(*device->event_queue_waiting[EVENT_DMA_PENDING],(!atomic_read(&device->irq_event_pending[EVENT_DMA_PENDING])));
        up (&device->pHardware->dmaSem);
       if (signal_pending(current)) {
            del_timer_sync(&device->watchdog_timer);
#ifdef DEBUG
            printk(KERN_INFO GSC_NAME "(%d): device_write DMA signal quit.\n", device->minor);
#endif
            DisableIrqPlx(device);
            return (-ERESTARTSYS); 
        }
        if (device->timeout)
        {
            printk(KERN_ERR GSC_NAME "(%d): device_write_dma - timeout...***************************************\n", device->minor);
            //printk(" (%d): In progress: %d PXL int/ctrl %.8X BC %.8X DMA %.8X\n", device->minor,device->dmainprogress, readl(IntCntrlStat(device)), readlocal(device,BOARD_CTRL_REG),readl(DMACmdStatus(device)));
            atomic_set(&device->irq_event_pending[EVENT_DMA_PENDING],FALSE);
            return (-EAGAIN);
        }
        else
            del_timer_sync(&device->watchdog_timer);
#ifdef DEBUG
        //  if (time_remain< 1000)
        //      printk(" %d ", time_remain);
#endif
        
    }
    else // pio
    {
        int i;
        unsigned long *dataPtr;
        unsigned long data;
 
        dataPtr=(unsigned long *) buf;
        atomic_set(&device->irq_event_pending[EVENT_DMA_PENDING],FALSE);
        //printk(KERN_INFO GSC_NAME "(%d): device_write PIO...\n",device->minor);
        for(i=0;i<nsamples;i++)
        {
            // this can probably be refined...
            if (copy_from_user(&data,dataPtr+i,sizeof(unsigned long)))
            {
                device->error = DEVICE_RESOURCE_ALLOCATION_ERROR;
                return (-EIO);
            }
            writelocal(device,data, OUTPUT_BUF_REG);
        }
    }
    return 0;
}

