/***
*** gsc16ao_ioctl.c
***
***  General description of this file:
***     Device driver source code for General Standards 16AO family of 
***     16-bit analog output boards. This file is part of the Linux
***     driver source distribution for this board.
***     
***  Copyrights (c):
***     General Standards Corporation (GSC),  2002-2004
***
***  Author:
***     Evan Hillman (evan@generalstandards.com)
***
***  Support:
***     Primary support for this driver is provided by GSC. 
***
***  Platform (tested on, may work with others):
***     Linux, kernel version 2.4.x, 2.6.x, Red Hat distribution, Intel hardware.
***/

#ifndef __KERNEL__
#define __KERNEL__
#endif

#include <linux/fs.h>
#include <linux/vmalloc.h>
#include <linux/pci.h>
#include <linux/mm.h>
#include <linux/sched.h>
#include <linux/interrupt.h>
#include <linux/timer.h>
#include <linux/ioctl.h>

#include <asm/io.h>
#include <asm/system.h>
#include <asm/uaccess.h>
#include "sysdep.h"
#include "plx_regs.h"
#include "gsc16ao4mf_ioctl.h"
#include "internals.h"

/* ioctl file operation: this does the bulk of the work */
int
device_ioctl(struct inode *inode, struct file *fp, unsigned int num, unsigned long arg)
{
    struct device_board *device = (struct device_board *)fp->private_data;
    struct register_params regs;
    unsigned long regval,ulval;

    /*
    main ioctl function dispatch */
    /* 'break' at the end of branches which need to wait for the */
    /* channels ready condition, 'return' from others */
    switch (num) {
    case IOCTL_NO_COMMAND:
        return (0);

    case IOCTL_INIT_BOARD:
#ifdef DEBUG
        printk(KERN_INFO GSC_NAME "(%d): IOCTL_INIT_BOARD initializing device\n", device->minor);
#endif
        atomic_set(&device->irq_event_pending[EVENT_INIT_COMPLETE],TRUE);
        regval = readlocal(device,BOARD_CTRL_REG);
        regval &= (~(BCR_IRQ_MASK));
        regval |= (BCR_IRQ_INIT | (BCR_INITIALIZE));

        device->timeout=FALSE;
        device->watchdog_timer.expires=jiffies+device->timeout_seconds*HZ;
        add_timer(&device->watchdog_timer);
        EnableIrqPlxLocal(device);
#ifdef DEBUG
        printk(KERN_INFO GSC_NAME "(%d): init about to wait\n", device->minor);
#endif
        writelocal(device,regval,BOARD_CTRL_REG);
        wait_event_interruptible(*device->event_queue_waiting[EVENT_INIT_COMPLETE],(!atomic_read(&device->irq_event_pending[EVENT_INIT_COMPLETE])));
        if (signal_pending(current)) {
#ifdef DEBUG
            printk(KERN_INFO GSC_NAME "(%d): init signal termination\n", device->minor);
#endif
            del_timer_sync(&device->watchdog_timer);
            DisableIrqPlx(device);
            return (-ERESTARTSYS); 
        }
        if (device->timeout)
        {
            //printk("%ld %ld\n",jiffies, device->watchdog_timer.expires);
            printk(KERN_ERR GSC_NAME "(%d): timeout during device initialize\n", device->minor);
            atomic_set(&device->irq_event_pending[EVENT_INIT_COMPLETE],FALSE);
            device->error = DEVICE_IOCTL_TIMEOUT;
            return (-EIO);
        }
        else
            del_timer_sync(&device->watchdog_timer);

#ifdef DEBUG
        printk(KERN_INFO GSC_NAME "(%d): plx int ctrl %.8X board ctrl %.8X DMA ctrl status%.8X\n",device->minor, readl(IntCntrlStat(device->pHardware)), readlocal(device,BOARD_CTRL_REG),readl(DMACmdStatus(device->pHardware)));
        printk(KERN_INFO GSC_NAME "(%d): ioctl initialize done\n", device->minor);
#endif
        return (0);

    case IOCTL_READ_REGISTER:
        copy_from_user_ret(&regs, (void *)arg, sizeof(regs), (-EFAULT));
        if ((regs.eRegister < BOARD_CTRL_REG) || (regs.eRegister > LAST_LOCAL_REGISTER)) {
            device->error = DEVICE_INVALID_PARAMETER;
            return (-EIO);
        }
        regs.ulValue = readl(device->pHardware->local_addr + regs.eRegister);
        copy_to_user_ret((void *)arg, &regs, sizeof(regs), (-EFAULT));
        return (0);

    case IOCTL_WRITE_REGISTER:
        copy_from_user_ret(&regs, (void *)arg, sizeof(regs), (-EFAULT));
        if ((regs.eRegister < BOARD_CTRL_REG) || (regs.eRegister > LAST_LOCAL_REGISTER)) {
            device->error = DEVICE_INVALID_PARAMETER;
            return (-EIO);
        }
        writel(regs.ulValue, device->pHardware->local_addr + regs.eRegister);
        return (0);

    case IOCTL_GET_DEVICE_ERROR:
        put_user_ret(device->error, (unsigned long *)arg, (-EFAULT));
        return (0);
        break;

    case IOCTL_SET_DMA_MODE:
        get_user_ret(ulval, (unsigned long *)arg, (-EFAULT));
        if ((ulval != DMA_DISABLE && ulval != DMA_ENABLE)) {
            device->error = DEVICE_INVALID_PARAMETER;
            return (-EIO);
        }
        device->dmaMode=ulval;
        return (0); 

    case IOCTL_AUTOCALIBRATE:
        //printk(KERN_INFO GSC_NAME "(%d) -------------------------\n", device->minor);
        //printk(KERN_INFO GSC_NAME "(%d): calibrating device\n", device->minor);
#ifdef DEBUG
        printk(KERN_INFO GSC_NAME "(%d): IOCTL_AUTOCALIBRATE\n", device->minor);
#endif
        regval = readlocal(device,BOARD_CTRL_REG);
        //printk(KERN_INFO GSC_NAME "(%d): board control %.8X\n", device->minor,readlocal(device,BOARD_CTRL_REG));
        regval &= (~(BCR_IRQ_MASK));
        regval |= BCR_IRQ_AUTOCAL_COMPLETE;
        atomic_set(&device->irq_event_pending[EVENT_AUTOCAL_COMPLETE],TRUE);
        writelocal(device,regval,BOARD_CTRL_REG);

        regval = readlocal(device,BOARD_CTRL_REG);
        regval &=~BCR_CALIBRATION_MODE_MASK;
        regval |=BCR_INIT_CALIBRATION;

        device->timeout=FALSE;
        device->watchdog_timer.expires=jiffies+device->timeout_seconds*HZ;
        add_timer(&device->watchdog_timer);

        EnableIrqPlxLocal(device);
        //printk(KERN_INFO GSC_NAME "(%d): board control %.8X\n", device->minor,readlocal(device,BOARD_CTRL_REG));
        //printk(KERN_INFO GSC_NAME "(%d): auto cal about to wait\n", device->minor);
        writelocal(device,regval,BOARD_CTRL_REG);
#ifdef DEBUG
        printk(KERN_INFO GSC_NAME "(%d): IOCTL_AUTOCALIBRATE about to sleep...\n", device->minor);
#endif
        wait_event_interruptible(*device->event_queue_waiting[EVENT_AUTOCAL_COMPLETE],(!atomic_read(&device->irq_event_pending[EVENT_AUTOCAL_COMPLETE])));
        if (signal_pending(current)) {
#ifdef DEBUG
            printk(KERN_INFO GSC_NAME "(%d): autocalibrate signal termination\n", device->minor);
#endif
            del_timer_sync(&device->watchdog_timer);
            DisableIrqPlx(device);
            return (-ERESTARTSYS); 
        }

        if (device->timeout) {
            printk(KERN_ERR GSC_NAME "(%d): timeout when calibrating device\n", device->minor);
            atomic_set(&device->irq_event_pending[EVENT_AUTOCAL_COMPLETE],FALSE);
            device->error = DEVICE_IOCTL_TIMEOUT;
            return (-EIO);
        }
        else
            del_timer_sync(&device->watchdog_timer);

#ifdef DEBUG
        printk(KERN_INFO GSC_NAME "(%d): ioctl autocal done\n", device->minor);
#endif
        return (0);

    case IOCTL_PROGRAM_RATE_GEN:
        get_user_ret(ulval, (unsigned long *)arg, (-EFAULT));
        if ((ulval < 0) || (ulval > MAX_RATE_GEN)) {
            device->error = DEVICE_INVALID_PARAMETER;
            return (-EIO);
        }
        writelocal(device,ulval,RATE_CTRL_REG);
        break;

    case IOCTL_SET_OUT_BUFFER_SIZE:
        get_user_ret(ulval, (unsigned long *)arg, (-EFAULT));
        if (ulval <0 || ulval > OUT_BUFFER_SIZE_131072) {
            device->error = DEVICE_INVALID_PARAMETER;
            return (-EIO);
        }
        regval = readlocal(device, BUFFER_OPS_REG);
        regval &= (~BOR_BUFFER_SIZE_MASK);
        regval |= ulval;
        writelocal(device,regval, BUFFER_OPS_REG);
        return (0);

    case IOCTL_GET_BUFFER_STATUS:
        put_user_ret(readlocal(device, BUFFER_OPS_REG)&BUFFER_STATUS_MASK, (unsigned long *)arg, (-EFAULT));
        return (0);

    case IOCTL_ENABLE_CLK:
        regval = readlocal(device, BUFFER_OPS_REG);
        regval |= BOR_ENABLE_CLOCK;
        writelocal(device,regval, BUFFER_OPS_REG);
        return (0);
        break;

    case IOCTL_DISABLE_CLK:
        regval = readlocal(device, BUFFER_OPS_REG);
        regval &= ~(BOR_ENABLE_CLOCK);
        writelocal(device,regval, BUFFER_OPS_REG);
        return (0);
        break;

    case IOCTL_SELECT_DATA_FORMAT:
        get_user_ret(ulval, (unsigned long *)arg, (-EFAULT));
        if ((ulval != TWOS_COMP) && (ulval != OFFSET_BINARY)) {
            device->error = DEVICE_INVALID_PARAMETER;
            return (-EIO);
        }
        regval = readlocal(device, BOARD_CTRL_REG);
        if (ulval==OFFSET_BINARY)
        {
            regval |= BCR_OFFSET_BINARY;
        }
        else
        {
            regval &= ~(BCR_OFFSET_BINARY);
        }
        writelocal(device,regval, BOARD_CTRL_REG);
        return (0);

    case IOCTL_SELECT_SAMPLING_MODE:
        get_user_ret(ulval, (unsigned long *)arg, (-EFAULT));
        if ((ulval != CONT_MODE) && (ulval != BURST_MODE)) {
            device->error = DEVICE_INVALID_PARAMETER;
            return (-EIO);
        }
        regval = readlocal(device, BOARD_CTRL_REG);
        if (ulval==BURST_MODE)
        {
            regval |= BCR_BURST_ENABLED;
        }
        else
        {
            regval &= ~(BCR_BURST_ENABLED);
        }
        writelocal(device,regval, BOARD_CTRL_REG);
        return (0);

    case IOCTL_GET_BURSTING_STATUS:
        put_user_ret(readlocal(device, BOARD_CTRL_REG)&BCR_BURST_READY, (unsigned long *)arg, (-EFAULT));
        return (0);

    case IOCTL_BURST_TRIGGER:
        regval = readlocal(device, BOARD_CTRL_REG);
        regval |= BCR_BURST_TRIGGER;
        writelocal(device,regval, BOARD_CTRL_REG);
        return (0);

    case IOCTL_ENABLE_REMOTE_GND_SENSE:
        regval = readlocal(device, BOARD_CTRL_REG);
        regval |= BCR_REMOTE_GROUND_SENSE;
        writelocal(device,regval, BOARD_CTRL_REG);
        return (0);

    case IOCTL_DISABLE_REMOTE_GND_SENSE:
        regval = readlocal(device, BOARD_CTRL_REG);
        regval &= ~BCR_REMOTE_GROUND_SENSE;
        writelocal(device,regval, BOARD_CTRL_REG);
        return (0);

    case IOCTL_SELECT_OUT_CLKING_MODE:
        get_user_ret(ulval, (unsigned long *)arg, (-EFAULT));
        if ((ulval != SEQUENTIAL) && (ulval != SIMULTANEOUS)) {
            device->error = DEVICE_INVALID_PARAMETER;
            return (-EIO);
        }
        regval = readlocal(device, BOARD_CTRL_REG);
        if (ulval==SIMULTANEOUS)
        {
            regval |= BCR_SIMULTANEOUS_OUTPUTS;
        }
        else
        {
            regval &= ~(BCR_SIMULTANEOUS_OUTPUTS);
        }
        writelocal(device,regval, BOARD_CTRL_REG);
        return (0);

    case IOCTL_SELECT_CLK_SOURCE:
        get_user_ret(ulval, (unsigned long *)arg, (-EFAULT));
        if ((ulval != EXTERNAL) && (ulval != INTERNAL)) {
            device->error = DEVICE_INVALID_PARAMETER;
            return (-EIO);
        }
        regval = readlocal(device, BUFFER_OPS_REG);
        if (ulval==EXTERNAL)
        {
            regval |= BOR_EXTERNAL_CLOCK;
        }
        else
        {
            regval &= ~(BOR_EXTERNAL_CLOCK);
        }
        writelocal(device,regval, BUFFER_OPS_REG);
        return (0);

    case IOCTL_GET_CLK_STATUS:
        put_user_ret(readlocal(device, BUFFER_OPS_REG)&BOR_CLOCK_READY, (unsigned long *)arg, (-EFAULT));
        return (0);         

    case IOCTL_SINGLE_OUTPUT_CLK_EVT:
        regval = readlocal(device, BUFFER_OPS_REG);
        regval |= BOR_SW_CLOCK;
        writelocal(device,regval, BUFFER_OPS_REG);
        return (0);         

    case IOCTL_SELECT_BUF_CONFIG:
        get_user_ret(ulval, (unsigned long *)arg, (-EFAULT));
        if ((ulval != OPEN_BUF) && (ulval != CIRCULAR_BUF)) {
            device->error = DEVICE_INVALID_PARAMETER;
            return (-EIO);
        }
        regval = readlocal(device, BUFFER_OPS_REG);
        if (ulval==CIRCULAR_BUF)
        {
            regval |= CIRCULAR_BUF;
        }
        else
        {
            regval &= ~(CIRCULAR_BUF);
        }
        writelocal(device,regval, BUFFER_OPS_REG);
        return (0);         

    case IOCTL_LOAD_ACCESS_REQ:
        regval = readlocal(device, BUFFER_OPS_REG);
        if ((regval & BOR_CIRCULAR_BUFFER) == 0) {
            device->error = DEVICE_INVALID_PARAMETER;
            return (-EIO);
        }
        regval |= BOR_LOAD_REQUEST | BOR_LOAD_READY;
        // set up to wait for the IRQ.

        atomic_set(&device->irq_event_pending[EVENT_LOAD_READY],TRUE);

        device->timeout=FALSE;
        device->watchdog_timer.expires=jiffies+device->timeout_seconds*HZ;
        add_timer(&device->watchdog_timer);

        wait_event_interruptible(*device->event_queue_waiting[EVENT_LOAD_READY],(!atomic_read(&device->irq_event_pending[EVENT_LOAD_READY])));
       if (signal_pending(current)) {
            del_timer_sync(&device->watchdog_timer);
            DisableIrqPlx(device);
            return (-ERESTARTSYS); 
        }
        writelocal(device,regval, BUFFER_OPS_REG);
        if (device->timeout)
        {
            atomic_set(&device->irq_event_pending[EVENT_LOAD_READY],FALSE);
            device->error = DEVICE_IOCTL_TIMEOUT;
            device->timeout=FALSE;
            return (-EIO);
        }
        else
            del_timer_sync(&device->watchdog_timer);

        if (device->timeout) {
            printk(KERN_ERR GSC_NAME "(%d): Access request timeout\n", device->minor);
            atomic_set(&device->irq_event_pending[EVENT_LOAD_READY],FALSE);
            device->error = DEVICE_IOCTL_TIMEOUT;
            return (-EIO);
        }
        return (0);         

    case IOCTL_GET_CIR_BUF_STATUS:
        put_user_ret(readlocal(device, BUFFER_OPS_REG)&BOR_LOAD_READY, (unsigned long *)arg, (-EFAULT));
        return (0);         

    case IOCTL_CLEAR_BUFFER:
        regval = readlocal(device, BUFFER_OPS_REG);
        regval |= BOR_CLEAR_BUFFER;
        writelocal(device,regval, BUFFER_OPS_REG);
        return (0);

    case IOCTL_GET_DEVICE_TYPE:
        put_user_ret(device->board_type, (unsigned long *)arg, (-EFAULT));
        return (0);

    case IOCTL_SET_TIMEOUT:
        get_user_ret(ulval, (unsigned long *)arg, (-EFAULT));
        device->timeout_seconds = ulval;
        return (0);

    case IOCTL_SET_ADJUSTABLE_CLOCK:     
        get_user_ret(ulval, (unsigned long *)arg, (-EFAULT));
#ifdef DEBUG
        printk(KERN_INFO GSC_NAME "(%d): IOCTL_SET_ADJUSTABLE_CLOCK passed %lx Max: %x\n", device->minor,ulval,ACR_CLOCK_RATE_MASK);
#endif

        if (ulval < 0 || ulval > ACR_CLOCK_RATE_MASK) {
            device->error = DEVICE_INVALID_PARAMETER;
            return (-EIO);
        }
        regval = readlocal(device, ADJUSTABLE_CLOCK);
        regval &= (~ACR_CLOCK_RATE_MASK);
        regval |= ulval;
        writelocal(device,regval, ADJUSTABLE_CLOCK);
        return (0);         

    case IOCTL_ENABLE_ADJUSTABLE_CLOCK:  
        get_user_ret(ulval, (unsigned long *)arg, (-EFAULT));
        regval = readlocal(device, ADJUSTABLE_CLOCK);
        if (ulval)
        {
            regval |= ACR_ADJUSTABLE_CLOCK_SELECT;
        }
        else
        {
            regval &= ~(ACR_ADJUSTABLE_CLOCK_SELECT);
        }
        writelocal(device,regval, ADJUSTABLE_CLOCK);
        return (0); 

    case IOCTL_SELECT_INTERNAL_CLOCK_SRC:
        get_user_ret(ulval, (unsigned long *)arg, (-EFAULT));
        
        if (ulval < ICS_DIVIDED || ulval > ICS_FIXED_8) {
            device->error = DEVICE_INVALID_PARAMETER;
            return (-EIO);
        }
        writelocal(device,ulval, INTERNAL_CLOCK_SOURCE);
        return (0);         

    default:
        printk(KERN_INFO GSC_NAME "(%d): ERROR invalid IOCTL\n", device->minor);
        return (-EINVAL);
    }
    return (0);         
}
