/***
*** gsc16ao.h
***
***  General description of this file:
***     Device driver source code for General Standards 16AO family of
***     16-bit analog output boards. This file is part of the Linux
***     driver source distribution for this board.
***
***  Copyrights (c):
***     General Standards Corporation (GSC), 2003
***
***  Author:
***     Evan Hillman (evan@generalstandards.com)
***
***  Support:
***     Primary support for this driver is provided by GSC.
***
***  Platform (tested on, may work with others):
***     Linux, kernel version 2.4.x, Red Hat distribution, Intel hardware.
***/

#ifndef PCI16AO4MF_H
#define PCI16AO4MF_H

#define DRIVER_VERSION "2.0.0"

#ifndef UTS_RELEASE
#include <linux/utsrelease.h>
#endif


#define PCI_NUM_BARS 6

#define MAX_BOARDS 10 /* maximum number of boards that driver supports. Arbitrarily chosen,
change if you can actully support more. */

/* module name NOTE:  This name must be the same as the driver binary.  The install script
uses the name of the binary name determine if the driver is installed properly, and the
driver uses this name to announce itself.  The name of the binary is set in Makefile.
*/
#define GSC_NAME "gsc_16ao4mf"

#define GSC_SUBVENDOR PCI_VENDOR_ID_PLX

struct board_entry {
    int subsystem_vendor;
    int subsystem_device;
    char name[40];
    int index;
};

#if !defined(U64)
typedef union _U64
{
    struct
    {
        u32  LowPart;
        u32  HighPart;
    }u;

    u64 QuadPart;
} U64;
#endif

// PCI BAR Space information 
typedef struct _PCI_BAR_INFO
{
    u32     *pVa;                        // BAR Kernel Virtual Address
    U64      Physical;                   // BAR Physical Address
    u32      Size;                       // BAR size
    int  IsIoMapped;                 // Memory or I/O mapped?
} PCI_BAR_INFO;

enum {
    TX
};

#define EVENT_INIT_COMPLETE             0
#define EVENT_AUTOCAL_COMPLETE          1
#define EVENT_OUT_BUFFER_EMPTY          2
#define EVENT_OUT_BUFFER_LOW_QUARTER    3  
#define EVENT_OUT_BUFFER_HIGH_QUARTER   4 
#define EVENT_BURST_TRIGGER_READY       5
#define EVENT_LOAD_READY                6
#define EVENT_END_LOAD_READY            7      
#define EVENT_DMA_PENDING               8
#define EVENT_LAST_IRQ_EVENT EVENT_DMA_PENDING+1

struct dma_data_struct {
    u32 dmaState;
    u32 *intermediateBuffer;        // DMA buffer 
    u32 intermediatePhysicalAddr;   // physical address of above 
    u32 userOffset;                 // current offset into user buffer.
    u32 userRemaining;              // space remaining in the user buffer.
    char *userBuffer;               // pointer to the current user buffer.
    u32 intermediateSamples;        // unread samples in the DMA buffer 
    u32 intermediateStart;          // first unread sample in the DMA buffer 
    u32 dmasamples;                 // unread samples in the DMA buffer 
    u32 dmastart;                   // first unread sample in the DMA buffer 
};

// hardware descriptor structure

struct device_hardware {
    struct pci_dev *pdev;           /* the kernel PCI device descriptor */
    u32 *runtime_addr;    /* remapped runtime register address space */
    u32 *local_addr;      /* remapped local register address space */
    int irqlevel;                   /* interrupt level */
    atomic_t usageCount;
    struct semaphore hwSem;
    struct semaphore dmaSem;
    struct device_board *channelData[MAX_CHANNELS];
	int error;
    PCI_BAR_INFO           PciBar[PCI_NUM_BARS];    // PCI BARs information
};

// channel descriptor structure

struct device_board {
    int board_type; /* an index for which board type for drivers that support multiple types. */
    struct device_hardware hw;
    struct device_hardware *pHardware;
	long timeout;                   /* timeout flag */
	int timeout_seconds;            /* time to wait before timeout (in seconds) */
	int minor;                      /* device minor number */
	atomic_t irq_event_pending[EVENT_LAST_IRQ_EVENT+1]; // array of flags for what irq is active
	int error;                      /* error code */
	int signalno;                   /* signal index if user requested notification */
	int signalev;                   /* signal the user if this event occurs */
#if LINUX_VERSION_CODE <= KERNEL_VERSION(2,4,0)
	struct wait_queue *ioctlwq;     /* queue of tasks sleeping in ioctl */
	struct wait_queue *writewq;      /* queue of tasks sleeping in write */
	struct wait_queue *dmawq;       /* queue of tasks sleeping on DMA transfer completion */
	struct wait_queue *event_queue_waiting[EVENT_LAST_IRQ_EVENT+1];
#else
	wait_queue_head_t ioctlwq;      /* queue of tasks sleeping in ioctl */
	wait_queue_head_t writewq;          /* queue of tasks sleeping in write */
	wait_queue_head_t dmawq;        /* queue of tasks sleeping on DMA transfer completion */
	wait_queue_head_t *event_queue_waiting[EVENT_LAST_IRQ_EVENT+1];
#endif

	int board_index;
	int dmaMode;
	struct dma_data_struct dma_data[2];
	int busy;                       /* device opened */

    int channelIndex;
#ifdef DEBUG
	int nread;
	int last;
#endif

	int debug_state;
	struct semaphore sem;
	struct file *fp;                // file pointer for user data.

	int fillBuffer;
	struct timer_list watchdog_timer;
    struct device_board *next;             /* next board in linked list */
};

/* non-static function prototypes */
int device_ioctl(struct inode *inode, struct file *fp, unsigned int num, unsigned long arg);

/* timing and delay stuff */
#define INIT_TIMEOUT_MS         2000        /* initialization timeout */
#define AUTOCAL_TIMEOUT_MS      5000        /* auto calibration timeout */
#define CH_READY_TIMEOUT_MS     500         /* timeout after channel setup operations */
#define MSECS_TO_SLEEP(MS)      ((MS * HZ) / 1000)

/* additional, internally used interrupt event codes */
#define NO_EVENT                (-1)

/* DMA buffer size (16 kBytes on Intel) */
#define DMA_ORDER               4
#define DMA_SIZE                (PAGE_SIZE << DMA_ORDER)
#define DMA_SAMPLES             (DMA_SIZE / sizeof(long))

/* DMA command codes */

#define START_DMA_CMD_0         0x0B
#define START_DMA_CMD_1         0xB00
#define STOP_DMA_CMD_0_MASK     0xFF01
#define STOP_DMA_CMD_1_MASK     0x00FF
#define NON_DEMAND_DMA_MODE     0x00020D43
#define DMA_CMD_STAT_INT_CLEAR  0x08
#define PCI_INT_ENABLE          0x00050900

/* Macros to make register access easier */
/* Mapping Local Configuration Registers */
#define PciLocRange0(card)      ((card)->runtime_addr + PCI_TO_LOC_ADDR_0_RNG)
#define PciLocRemap0(card)      ((card)->runtime_addr + LOC_BASE_ADDR_REMAP_0)
#define ModeArb(card)           ((card)->runtime_addr + MODE_ARBITRATION)
#define EndianDescr(card)       ((card)->runtime_addr + BIG_LITTLE_ENDIAN_DESC)
#define PciLERomRange(card)     ((card)->runtime_addr + PCI_TO_LOC_ROM_RNG)
#define PciLERomRemap(card)     ((card)->runtime_addr + LOC_BASE_ADDR_REMAP_EXP_ROM)
#define PciLBRegDescr0(card)    ((card)->runtime_addr + BUS_REG_DESC_0_FOR_PCI_LOC)
#define LocPciRange(card)       ((card)->runtime_addr + DIR_MASTER_TO_PCI_RNG)
#define LocPciMemBase(card)     ((card)->runtime_addr + LOC_ADDR_FOR_DIR_MASTER_MEM)
#define LocPciIOBase(card)      ((card)->runtime_addr + LOC_ADDR_FOR_DIR_MASTER_IO)
#define LocPciRemap(card)       ((card)->runtime_addr + PCI_ADDR_REMAP_DIR_MASTER)
#define LocPciConfig(card)      ((card)->runtime_addr + PCI_CFG_ADDR_DIR_MASTER_IO)
#define PciLocRange1(card)      ((card)->runtime_addr + PCI_TO_LOC_ADDR_1_RNG)
#define PciLocRemap1(card)      ((card)->runtime_addr + LOC_BASE_ADDR_REMAP_1)
#define PciLBRegDescr1(card)    ((card)->runtime_addr + BUS_REG_DESC_1_FOR_PCI_LOC)

/* Mapping Runtime Registers */
#define Mailbox0(card)          ((card)->runtime_addr + MAILBOX_REGISTER_0)
#define Mailbox1(card)          ((card)->runtime_addr + MAILBOX_REGISTER_1)
#define Mailbox2(card)          ((card)->runtime_addr + MAILBOX_REGISTER_2)
#define Mailbox3(card)          ((card)->runtime_addr + MAILBOX_REGISTER_3)
#define Mailbox4(card)          ((card)->runtime_addr + MAILBOX_REGISTER_4)
#define Mailbox5(card)          ((card)->runtime_addr + MAILBOX_REGISTER_5)
#define Mailbox6(card)          ((card)->runtime_addr + MAILBOX_REGISTER_6)
#define Mailbox7(card)          ((card)->runtime_addr + MAILBOX_REGISTER_7)
#define PciLocDoorBell(card)    ((card)->runtime_addr + PCI_TO_LOC_DOORBELL)
#define LocPciDoorBell(card)    ((card)->runtime_addr + LOC_TO_PCI_DOORBELL)

/* Mapping interrupt control/status reg  */
#define IntCntrlStat(card)      ((card)->runtime_addr + INT_CTRL_STATUS)
#define RunTimeCntrl(card)      ((card)->runtime_addr + PROM_CTRL_CMD_CODES_CTRL)
#define DevVenIDc(card)         ((card)->runtime_addr + DEVICE_ID_VENDOR_ID)
#define RevID(card)             ((card)->runtime_addr + REVISION_ID)
#define MailboxReg0(card)       ((card)->runtime_addr + MAILBOX_REG_0)
#define MailboxReg1(card)       ((card)->runtime_addr + MAILBOX_REG_1)

/* Mapping DMA registers */
#define DMAMode0(card)          ((card)->runtime_addr + DMA_CH_0_MODE)
#define DMAPCIAddr0(card)       ((card)->runtime_addr + DMA_CH_0_PCI_ADDR)
#define DMALocalAddr0(card)     ((card)->runtime_addr + DMA_CH_0_LOCAL_ADDR)
#define DMAByteCnt0(card)       ((card)->runtime_addr + DMA_CH_0_TRANS_BYTE_CNT)
#define DMADescrPtr0(card)      ((card)->runtime_addr + DMA_CH_0_DESC_PTR)
#define DMAMode1(card)          ((card)->runtime_addr + DMA_CH_1_MODE)
#define DMAPCIAddr1(card)       ((card)->runtime_addr + DMA_CH_1_PCI_ADDR)
#define DMALocalAddr1(card)     ((card)->runtime_addr + DMA_CH_1_LOCAL_ADDR)
#define DMAByteCnt1(card)       ((card)->runtime_addr + DMA_CH_1_TRANS_BYTE_CNT)
#define DMADescrPtr1(card)      ((card)->runtime_addr + DMA_CH_1_DESC_PTR)
#define DMACmdStatus(card)      ((card)->runtime_addr + DMA_CMD_STATUS)
#define DMAArbitr(card)         ((card)->runtime_addr + DMA_MODE_ARB_REG)
#define DMAThreshold(card)      ((card)->runtime_addr + DMA_THRESHOLD_REG)

/* Mapping FIFO registers */
#define OutPostQIntStat(card)   ((card)->runtime_addr + OUT_POST_Q_INT_STATUS)
#define OutPostQIntMask(card)   ((card)->runtime_addr + OUT_POST_Q_INT_MASK)
#define MsgUnitCfg(card)        ((card)->runtime_addr + MSG_UNIT_CONFIG)
#define QBaseAddr(card)         ((card)->runtime_addr + Q_BASE_ADDR)
#define InFreeHeadPtr(card)     ((card)->runtime_addr + IN_FREE_HEAD_PTR)
#define InFreeTailPtr(card)     ((card)->runtime_addr + IN_FREE_TAIL_PTR)
#define InPostHeadPtr(card)     ((card)->runtime_addr + IN_POST_HEAD_PTR)
#define InPostTailPtr(card)     ((card)->runtime_addr + IN_POST_TAIL_PTR)
#define OutFreeHeadPtr(card)    ((card)->runtime_addr + OUT_FREE_HEAD_PTR)
#define OutFreeTailPtr(card)    ((card)->runtime_addr + OUT_FREE_TAIL_PTR)
#define OutPostHeadPtr(card)    ((card)->runtime_addr + OUT_POST_HEAD_PTR)
#define OutPostTailPtr(card)    ((card)->runtime_addr + OUT_POST_TAIL_PTR)
#define QStatusCtrl(card)       ((card)->runtime_addr + Q_STATUS_CTRL_REG)

/*
#define InpDataBufReg(card)     ((card)->local_addr + INPUT_DATA_BUFFER_REG)
*/
// Interrupt Control/Status register masks

#ifndef IRQ_LOCAL_LSERR_ABORT
#define IRQ_LOCAL_LSERR_ABORT       (1 << 0)
#endif
#ifndef IRQ_LOCAL_LSERR_OVERFLOW
#define IRQ_LOCAL_LSERR_OVERFLOW    (1 << 1)
#endif
#ifndef IRQ_GENERATE_SERR
#define IRQ_GENERATE_SERR           (1 << 2)
#endif
#ifndef IRQ_MAILBOX_ENABLE
#define IRQ_MAILBOX_ENABLE          (1 << 3)
#endif
#ifndef IRQ_RESERVED1
#define IRQ_RESERVED1               (1 << 4)
#endif
#ifndef IRQ_RESERVED2
#define IRQ_RESERVED2               (1 << 5)
#endif
#ifndef IRQ_RESERVED3
#define IRQ_RESERVED3               (1 << 6)
#endif
#ifndef IRQ_RESERVED4
#define IRQ_RESERVED4               (1 << 7)
#endif
#ifndef IRQ_PCI_ENABLE
#define IRQ_PCI_ENABLE              (1 << 8)
#endif
#ifndef IRQ_PCI_DOORBELL_ENABLE
#define IRQ_PCI_DOORBELL_ENABLE     (1 << 9)
#endif
#ifndef IRQ_ABORT_ENABLE
#define IRQ_ABORT_ENABLE            (1 << 10)
#endif
#ifndef IRQ_LOCAL_PCI_ENABLE
#define IRQ_LOCAL_PCI_ENABLE        (1 << 11)
#endif
#ifndef IRQ_RETRY_ENABLE
#define IRQ_RETRY_ENABLE            (1 << 12)
#endif
#ifndef IRQ_PCI_DOORBELL_ACTIVE
#define IRQ_PCI_DOORBELL_ACTIVE     (1 << 13)
#endif
#ifndef IRQ_ABORT_ACTIVE
#define IRQ_ABORT_ACTIVE            (1 << 14)
#endif
#ifndef IRQ_LOCAL_ACTIVE
#define IRQ_LOCAL_ACTIVE            (1 << 15)
#endif
#ifndef IRQ_LOCAL_ENABLE
#define IRQ_LOCAL_ENABLE            (1 << 16)
#endif
#ifndef IRQ_LOCAL_DOORBELL_ENABLE
#define IRQ_LOCAL_DOORBELL_ENABLE   (1 << 17)
#endif
#ifndef IRQ_DMA_0_ENABLE
#define IRQ_DMA_0_ENABLE            (1 << 18)
#endif
#ifndef IRQ_DMA_1_ENABLE
#define IRQ_DMA_1_ENABLE            (1 << 19)
#endif
#ifndef IRQ_LOCAL_DOORBELL_ACTIVE
#define IRQ_LOCAL_DOORBELL_ACTIVE   (1 << 20)
#endif
#ifndef IRQ_DMA_0_ACTIVE
#define IRQ_DMA_0_ACTIVE            (1 << 21)
#endif
#ifndef IRQ_DMA_1_ACTIVE
#define IRQ_DMA_1_ACTIVE            (1 << 22)
#endif
#ifndef IRQ_BIST_ACTIVE
#define IRQ_BIST_ACTIVE             (1 << 23)
#endif
#ifndef IRQ_MASTER_ABORT
#define IRQ_MASTER_ABORT            (1 << 24)
#endif
#ifndef IRQ_DMA_0_ABORT
#define IRQ_DMA_0_ABORT             (1 << 25)
#endif
#ifndef IRQ_DMA_1_ABORT
#define IRQ_DMA_1_ABORT             (1 << 26)
#endif
#ifndef IRQ_TARGET_ABORT
#define IRQ_TARGET_ABORT            (1 << 27)
#endif
#ifndef IRQ_MAILBOX_0
#define IRQ_MAILBOX_0               (1 << 28)
#endif
#ifndef IRQ_MAILBOX_1
#define IRQ_MAILBOX_1               (1 << 29)
#endif
#ifndef IRQ_MAILBOX_2
#define IRQ_MAILBOX_2               (1 << 30)
#endif
#ifndef IRQ_MAILBOX_3
#define IRQ_MAILBOX_3               (1 << 31)
#endif

// global data.  Only created during init.
extern s8 built[128];
extern int num_boards;
extern int proc_enabled;
extern struct board_entry boards_supported[];
extern const char *event_name[];

#ifdef DEBUG
extern u32 int_other_count[MAX_BOARDS];
extern u32 int_count[MAX_BOARDS];
extern u32 dma_count[MAX_BOARDS];
extern u32 channel_irq[MAX_BOARDS];
extern u32 channel_expected[MAX_BOARDS];
extern int board_type[MAX_BOARDS];
extern void * context[MAX_BOARDS];
#endif

//functions
void writelocal(struct device_board *,unsigned value, unsigned address);
unsigned readlocal(struct device_board *,unsigned address);
int read_proc(char* page, char** start, off_t offset, int count, int* eof, void* data);
int proc_get_info(char* page, char** start, off_t offset, int count, int dummy);
int init_module(void);
void cleanup_module(void);
int device_open(struct inode *inode, struct file *fp);
int device_close(struct inode *inode, struct file *fp);
int device_write(struct file *fp, const char *buf, size_t size, loff_t * lt);
int device_read(struct file *fp, char *buf, size_t size, loff_t * lt);
void timeout_handler(unsigned long ptr);

void DisableIrqPlx(struct device_board *device);
void EnableIrqPlxLocal(struct device_board *device);
void EnableIrqPlxDMA(struct device_board *device);
void DisableIrqLocalAll(struct device_board *device);
void EnableIrqLocal(struct device_board *device, u32 irq);
void DisableIrqLocal(struct device_board *device, u32 irq);
extern const int channelOffset[];
void regdump(struct device_board *device,char * label);

#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,0)
irqreturn_t device_interrupt(int irq, void *dev_id, struct pt_regs *regs);
#else
void device_interrupt(int irq, void *dev_id, struct pt_regs *regs);
#endif

#endif
