/***
*** init.c  
***
***  General description of this file:
***     Device driver source code for General Standards 24DSI32 
***     family Delta-Sigma A/D boards. This file is part of the Linux
***     driver source distribution for this board.
***     
***     This file is not necessary to compile application programs, therefore 
***     should NOT be included in binary only driver distributions.
***
***  Copyrights (c):
***     General Standards Corporation (GSC), Feb 2004
***
***  Author:
***     Evan Hillman (evan@generalstandards.com)
***
***  Support:
***     Primary support for this driver is provided by GSC. 
***
***  Platform (tested on, may work with others):
***     Linux, kernel version 2.4.x, Red Hat distribution, Intel hardware.
***/

//////////////////////////////////////////////////////////////////////////
// set the following flag to trace debug messages in this module.
#ifdef DEBUG
#define TRACE_INIT TRUE
#endif
#include <linux/module.h>
#include <linux/fs.h>
#include <linux/vmalloc.h>
#include <linux/pci.h>
#include <linux/mm.h>
#include <linux/proc_fs.h>
#include <linux/sched.h>
#include <linux/interrupt.h>
#include <linux/timer.h>
#include <linux/ioctl.h>
#include <linux/version.h>

#include <asm/io.h>
#include <asm/system.h>
#include <asm/uaccess.h>
#include <asm/atomic.h>

#include "sysdep.h"
#include "gsc16ao4mf_ioctl.h"
#include "internals.h"

#include "plx_regs.h"

/* local module data */
static struct file_operations device_fops = {
    open:    device_open,
    release: device_close,
    read:    device_read,
    write:   device_write,
    ioctl:   device_ioctl
};

static int device_major = 0;                  /* device major number (dynamically assigned) */
static struct device_board *boards = 0;       /* linked list of all boards found */

__s8    built[128];
int     num_boards;
int     proc_enabled;

const int channelOffset[]=
{
    0x00/4,
    0x40/4,
    0x80/4,
    0xC0/4
};

/************************************************************************/
/* module initialization: detect card(s) and set up descriptor(s)        */
/************************************************************************/
int init_module(void)
{
//    unsigned short reg;
    struct device_board *device, *devicenext;
    struct pci_dev *pdev=NULL;
    struct proc_dir_entry*  proc;
    int index;
    int BoardIndex;
    int found;
    int checkIndex;
	int channel;
	int i;
    struct device_hardware *pHw=NULL;

    sprintf(built, "%s, %s ", __DATE__, __TIME__);
    printk(KERN_INFO GSC_NAME ": driver ( version: %s) built %s loading on kernel %s\n",  DRIVER_VERSION, built, UTS_RELEASE);
    printk(KERN_INFO GSC_NAME ": Copyright (C) 2001-2005 General Standards Corp. \n");

    BoardIndex=0;
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,5,74)
    PCI_DEVICE_LOOP(pdev) {
#else
    while((pdev = pci_find_device(PCI_VENDOR_ID_PLX, PCI_ANY_ID, pdev))) {
#endif
#ifdef TRACE_INIT
        printk(KERN_INFO GSC_NAME ": Checking...vendor: %X device: %X subvendor: %X subsystem: %X\n",pdev->vendor, pdev->device, pdev->subsystem_vendor, pdev->subsystem_device);
#endif
        if ((pdev->vendor == PCI_VENDOR_ID_PLX) && (pdev->device == PCI_DEVICE_ID_PLX_9080)) {

            /* determine if this is one of the boards supported. */

            found=0;
            index=0;
            checkIndex=0;
            while(boards_supported[checkIndex].subsystem_device != 0){
                if ((boards_supported[checkIndex].subsystem_device == pdev->subsystem_device) &&
                    (boards_supported[checkIndex].subsystem_vendor == pdev->subsystem_vendor)){
                        found=1;
                        //printk(KERN_INFO GSC_NAME ":     found board %s type %d\n",boards_supported[i].name,i);
                    }
                    if (found) break;
                    checkIndex++;
            }

            if (found){
                for(channel=0;channel<MAX_CHANNELS;channel++)
                {

                printk(KERN_INFO GSC_NAME ":     installing board %s type %d, channel %d\n",boards_supported[checkIndex].name,checkIndex,channel);
                //pci_read_config_word(pdev, 0x2E, &reg);
                //printk(KERN_INFO GSC_NAME ":     config reg=0x%x\n", reg);
                //printk(KERN_INFO GSC_NAME ":     attaching board #%d\n", index + 1);
                device = (struct device_board *)kmalloc(sizeof(struct device_board), GFP_KERNEL);
#ifdef TRACE_INIT
                printk(KERN_INFO GSC_NAME "....................................channel %d device %p\n",channel,device);
#endif
                memset(device,0,sizeof(device));
				device->board_type = checkIndex;
                device->channelIndex=channel;

                if (channel==0) // init the physical hardware
                {
#ifdef TRACE_INIT
                    printk(KERN_INFO GSC_NAME ".....................channel 0 init\n");
#endif
                    i = pci_enable_device(pdev);
                    pHw=&device->hw;
                    atomic_set(&pHw->usageCount,0);
                    {
                        int ResourceCount = 0;

                        for (index = 0; index < PCI_NUM_BARS; ++index)
                        {
                            // Verify the address is valid
                            if (pci_resource_start(pdev, index) == 0)
                            {
                                continue;
                            }

                            printk(KERN_INFO GSC_NAME "   Resource %02d\n", ResourceCount);

                            // Increment resource count
                            ResourceCount++;
                            // Get PCI physical address
                            pHw->PciBar[index].Physical.QuadPart = pci_resource_start(pdev, index);

                            // Determine resource type
                            if (pci_resource_flags(pdev, index) & IORESOURCE_IO)
                            {
                                //printk(KERN_INFO GSC_NAME "     Type     : I/O Port\n");

                                // Make sure flags are cleared properly
                                pHw->PciBar[index].Physical.QuadPart &= ~(0x3);
                                pHw->PciBar[index].IsIoMapped         = TRUE;
                            }
                            else
                            {
                                //printk(KERN_INFO GSC_NAME "     Type     : Memory Space\n");

                                // Make sure flags are cleared properly
                                pHw->PciBar[index].Physical.QuadPart &= ~(0xf);
                                pHw->PciBar[index].IsIoMapped         = FALSE;
                            }

                            //printk(KERN_INFO GSC_NAME "     Address  : %08x\n",device->PciBar[i].Physical.u.LowPart);

                            // Get the size
                            pHw->PciBar[index].Size = 4096; /*
                                                                    Plx_pci_resource_len(
                                                                    device,
                                                                    i
                                                                    );*/
                        }
                    }
#if LINUX_VERSION_CODE <= KERNEL_VERSION(2,4,0)
                    pHw->runtime_addr = (u32 *)ioremap(pdev->base_address[0], 4096);

                    if (pHw->PciBar[2].IsIoMapped)
                    {
                        //device->local_addr = pdev->resource[2];
                        printk(KERN_ERR GSC_NAME ": I/O mapped local registers\n");
                        pHw->local_addr = base_address[2];
                    }
                    else
                    {
                        pHw->local_addr = (u32 *)ioremap(pdev->resource[2].start, 4096);
                    }
#else
                    if (pHw->PciBar[2].IsIoMapped)
                    {
                        //pHw->local_addr = (u32 *)pdev->resource[2].start;
                    }
                    else
                    {
                        pHw->local_addr = (u32 *)ioremap(pdev->resource[2].start, 4096);
                    }
                    pHw->runtime_addr = (u32 *)ioremap(pdev->resource[0].start, pci_resource_len(pdev, 0));
#endif
                    pHw->pdev = pdev;
                    pci_set_master(pdev);
                   pHw->irqlevel = pdev->irq;
                    sema_init(&pHw->hwSem,1);
                    sema_init(&pHw->dmaSem,1);
#ifdef DEBUG
                    printk(KERN_ERR GSC_NAME ": done with channel 0\n");
#endif
                } // if (channel==0) 
#ifdef DEBUG
                printk(KERN_ERR GSC_NAME ": setup of  channel %d hardware %p \n",channel,pHw);
#endif
                pHw->channelData[channel]=device;
				pHw->error=FALSE;
                device->next = boards;
            for (i=0;i<EVENT_LAST_IRQ_EVENT;i++)
            {
                atomic_set(&device->irq_event_pending[i],FALSE);
            }
#ifdef DEBUG

            //for (i=0;i<EVENT_LAST_IRQ_EVENT;i++)
            //{
            //    printk("Event %d value %d\n",i,atomic_read(&device->irq_event_pending[i]));
            //}
#endif
#if LINUX_VERSION_CODE <= KERNEL_VERSION(2,4,0)
					device->ioctlwq = NULL;
					device->readwq = NULL;
					device->writewq = NULL;
					device->dmawq = NULL;
#else
					init_waitqueue_head(&device->ioctlwq);
					init_waitqueue_head(&device->writewq);
					init_waitqueue_head(&device->dmawq);
#endif
					device->fillBuffer=FALSE;
					device->busy = 0;
					device->minor = BoardIndex;
#ifdef DEBUG
					//printk(KERN_INFO GSC_NAME ":     base_address[0]=0x%lX, base_address[2]=0x%lX\n",pdev->resource[0].start, pdev->resource[2].start);
					//              printk(KERN_INFO GSC_NAME ":     local_addr=%p, runtime_addr=%p irq=%d\n", device->local_addr, device->runtime_addr, device->irqlevel);
#endif
					boards = device;
					device->board_index=BoardIndex;
                    device->pHardware=pHw;
                    //regdump(device," init ");
                    DisableIrqPlx(device);
                    DisableIrqLocalAll(device);
#ifdef TRACE_INIT
               //     printk(KERN_INFO GSC_NAME " BCR  %.8X %.8X %.8X %.8X  \n",readlocal(device,BOARD_CTRL_REG),
               //         readlocal(device,BOARD_CTRL_REG+0x40/4),
               //         readlocal(device,BOARD_CTRL_REG+0x80/4),
               //         readlocal(device,BOARD_CTRL_REG+0xc0/4));
                    board_type[BoardIndex]=device->board_type;
#endif
                BoardIndex++;
                num_boards++;
                //writelocal(device,0,BOARD_CTRL_REG);
}
            } // if (found)
		} // (pdev->vendor ...
	} // PCI_DEVICE_LOOP(pdev)

    if (BoardIndex == 0) {
        printk(KERN_ERR GSC_NAME ": no board found\n");
        return (-ENODEV);
    }

    device_major = register_chrdev(0, GSC_NAME, &device_fops);
    if (device_major < 0) {
        /* OK, could not register -- undo the work we have done above */
        printk(KERN_ERR GSC_NAME ": could not register device number\n");
        for (device = boards; device; device = devicenext) {
            devicenext = device->next;
            if (0 != device->hw.local_addr)
            {
                iounmap(device->hw.local_addr);
                iounmap(device->hw.runtime_addr);
                pci_disable_device(device->pHardware->pdev);
            }
            kfree(device);
        }
        boards = NULL;
        return (-ENODEV);
    }
    
    //   Add /proc file system support.

    if (num_boards)
    {
        remove_proc_entry(GSC_NAME, NULL);  /* Just in case. */
        proc    = create_proc_entry(GSC_NAME, S_IRUGO, NULL);

        if (proc)
        {
            proc_enabled    = 1;
            proc->read_proc = read_proc;
            proc->get_info  = (void*)proc_get_info;
        }
        else
        {
            printk( "<1>%s: create_proc_entry() failure.\n",
                GSC_NAME);
            cleanup_module();
        }
    }

#ifdef TRACE_INIT
    printk(KERN_INFO GSC_NAME ": major=%d\n", device_major);
    {   
        int j;
        for (j=0;j<MAX_BOARDS;j++){
            int_other_count[j]=0;
            int_count[j]=0;
            dma_count[j]=0;
            channel_irq[j]=0;
        }
    }
#endif
#ifdef DEBUG
    printk(KERN_INFO GSC_NAME " End of init\n");
#endif

    return 0;
}

/************************************************************************/
/* cleanup when unloading module                                        */
/************************************************************************/
void cleanup_module(void)
{
	struct device_board *device, *devicenext;
    unregister_chrdev(device_major, GSC_NAME);
        
    for (device = boards; device; device = devicenext) {
        devicenext = device->next;
        if ((device->pHardware->pdev) && (0 != device->hw.local_addr))
		{
			pci_disable_device(device->pHardware->pdev);
			device->pHardware->pdev=0;
		}
		        
        if (0 != device->hw.local_addr)
        {
            iounmap(device->hw.local_addr);
            iounmap(device->hw.runtime_addr);
			device->hw.local_addr=0;
			device->hw.runtime_addr=0;
        }
        kfree(device);
    }
    if (proc_enabled)   {
        remove_proc_entry(GSC_NAME, NULL);
        proc_enabled    = 0;
    }

    boards = NULL;
#ifdef DEBUG
    printk(KERN_INFO GSC_NAME ": unloaded\n");
#endif
}

/************************************************************************/
/* open device                                                          */
/************************************************************************/
int device_open(struct inode *inode, struct file *fp)
{
    struct device_board *device;
    u32 regval;
    int i;
    //printk(KERN_INFO GSC_NAME " app device_open()\n");

    for (device = boards; device; device = device->next) {
#ifdef DEBUG
        printk(KERN_INFO GSC_NAME "(%d): checking device %p\n", device->minor, device);
#endif
        if (MINOR(inode->i_rdev) == device->minor) {
            if (device->busy) {
                printk(KERN_ERR GSC_NAME "(%d): board already opened\n", device->minor);
                //up (&device->pHardware->hwSem);
                return (-EBUSY);
            }
            
            atomic_inc(&device->pHardware->usageCount);
#ifdef DEBUG
            printk(KERN_INFO GSC_NAME "(%d): opening board\n", device->minor);
#endif
            if (atomic_read(&device->pHardware->usageCount)==1) // first open for this board
            {
#ifdef DEBUG
                printk(KERN_INFO GSC_NAME "(%d): First open of a channel on the board\n", device->minor);
#endif
                if (request_irq(device->pHardware->irqlevel, (void *) device_interrupt, SA_SHIRQ, GSC_NAME, device->pHardware) < 0) {
                    printk(KERN_ERR GSC_NAME "(%d): cannot get interrupt %d\n", device->minor, device->pHardware->irqlevel);
                    device->busy = 0;
                    //up (&device->pHardware->hwSem);
                    return (-EBUSY);
                }

                // make sure DMA is in a known state

                regval = readl(DMACmdStatus(device->pHardware));
                regval &= STOP_DMA_CMD_0_MASK;
                regval &= STOP_DMA_CMD_1_MASK;
                writel(regval, DMACmdStatus(device->pHardware));

                // enable bus mastering
                writel(readl(PciLocRemap0(device->pHardware))|0x04,PciLocRemap0(device->pHardware));

                DisableIrqPlx(device);
            } // first open...
            
            if (!(device->dma_data[TX].intermediateBuffer = (u32 *)__get_dma_pages(GFP_KERNEL, DMA_ORDER))) {
                printk(KERN_ERR GSC_NAME "(%d): can not allocate DMA pages\n", device->minor);
                free_irq(device->pHardware->irqlevel, device);
                device->busy = 0;
                //up (&device->pHardware->hwSem);
                return (-EBUSY);
            }
#ifdef TRACE_INIT
            //memset(device->intermediateBuffer,0xa5,DMA_ORDER*PAGE_SIZE);
#endif

            device->dma_data[TX].intermediatePhysicalAddr = virt_to_phys((void *)device->dma_data[TX].intermediateBuffer);
#ifdef TRACE_INIT
            printk(KERN_INFO GSC_NAME "(%d): DMA virt=%p, phys=0x%x size %d\n", device->minor, device->dma_data[TX].intermediateBuffer, device->dma_data[TX].intermediatePhysicalAddr, sizeof(device->dma_data[TX].intermediateBuffer));
#endif
            init_timer(&device->watchdog_timer);
            device->timeout_seconds=20;
            device->watchdog_timer.function=timeout_handler;
            device->watchdog_timer.data = (unsigned long) device;
#ifdef DEBUG
            device->nread=0;
            device->last=-1;
#endif
            device->busy = 1;
            device->dma_data[TX].dmaState = DMA_DISABLE;
            device->dma_data[TX].dmastart = FALSE;
            device->dma_data[TX].dmasamples = 0;
#ifdef TRACE_INIT
            printk(KERN_INFO GSC_NAME "(%d): Reset IRQ pending array\n", device->minor);
#endif
            for (i=0;i<EVENT_LAST_IRQ_EVENT;i++)
            {
                atomic_set(&device->irq_event_pending[i],FALSE);
            }

#ifdef TRACE_INIT
            //for (i=0;i<EVENT_LAST_IRQ_EVENT;i++)
            //{
            //    printk("Event %d value %d\n",i,atomic_read(&device->irq_event_pending[i]));
           //}
#endif
            device->error = DEVICE_SUCCESS;
            device->signalno = (-1);
            device->signalev = NO_EVENT;
            device->dma_data[TX].intermediateSamples=0;
            
#if LINUX_VERSION_CODE <= KERNEL_VERSION(2,4,0)
            device->ioctlwq = NULL;
            device->writewq = NULL;
            device->dmawq = NULL;
#else
            init_waitqueue_head(&device->ioctlwq);
            init_waitqueue_head(&device->writewq);
            init_waitqueue_head(&device->dmawq);
#endif          
            device->event_queue_waiting[EVENT_INIT_COMPLETE]=&device->ioctlwq;
            device->event_queue_waiting[EVENT_AUTOCAL_COMPLETE]=&device->ioctlwq;
            device->event_queue_waiting[EVENT_OUT_BUFFER_EMPTY]=&device->writewq;
            device->event_queue_waiting[EVENT_OUT_BUFFER_LOW_QUARTER]=&device->writewq;
            device->event_queue_waiting[EVENT_OUT_BUFFER_HIGH_QUARTER]=&device->writewq;
            device->event_queue_waiting[EVENT_BURST_TRIGGER_READY]=&device->writewq;
            device->event_queue_waiting[EVENT_LOAD_READY]=&device->writewq;
            device->event_queue_waiting[EVENT_END_LOAD_READY]=&device->writewq;
            device->event_queue_waiting[EVENT_DMA_PENDING]=&device->dmawq;

            device->debug_state=0;

            fp->private_data = device;
            /* reset DMA engines */
#ifdef TRACE_INIT
            //printk(KERN_INFO GSC_NAME "(%d): resetting DMA engines\n", device->minor);
#endif

                // reset local IRQ flag 
#ifdef TRACE_INIT
                //printk(KERN_INFO GSC_NAME "(%d): resetting local IRQ flag\n", device->minor);
#endif

#ifdef TRACE_INIT
            printk(KERN_INFO GSC_NAME "(%d): device_open done\n", device->minor);
#endif
            MOD_INC_USE_COUNT;
            up (&device->pHardware->hwSem);
            return (0);
        }
    }
    printk(KERN_INFO GSC_NAME "(%d): cannot find board\n", -999/*MINOR(inode->i_rdev)*/);
    //up (&device->pHardware->hwSem);
    return (-ENODEV);
}

/************************************************************************/
/* close device                                                         */
/************************************************************************/
int device_close(struct inode *inode, struct file *fp)
{
    struct device_board *device = (struct device_board *)fp->private_data;
    u32 regval;

#ifdef TRACE_INIT
    printk(KERN_INFO GSC_NAME "(%d): closing board - usage count: %d\n", device->minor,atomic_read(&device->pHardware->usageCount));
#endif

    if (down_interruptible(&device->pHardware->hwSem)){
        return -ERESTARTSYS;
    }

    device->busy = 0;
    free_pages((u32)device->dma_data[TX].intermediateBuffer, DMA_ORDER);

	// free up common resources on the last close.
    atomic_dec(&device->pHardware->usageCount);
    if (atomic_read(&device->pHardware->usageCount)==0)
    {
#ifdef TRACE_INIT
        printk(KERN_INFO GSC_NAME "(%d): closing board - last device\n", device->minor);
#endif
        DisableIrqPlx(device);
        DisableIrqLocalAll(device);

		/* reset DMA engines */
        regval = readl(DMACmdStatus(device->pHardware));
        regval &= STOP_DMA_CMD_0_MASK;
        regval &= STOP_DMA_CMD_1_MASK;
        writel(regval, DMACmdStatus(device->pHardware));

		/* free resources */
#ifdef TRACE_INIT
        printk(KERN_INFO GSC_NAME "(%d): closing board - last device\n", device->minor);
#endif
        free_irq(device->pHardware->irqlevel, device);
    }
        MOD_DEC_USE_COUNT;
    up (&device->pHardware->hwSem);
#ifdef TRACE_INIT
        printk(KERN_INFO GSC_NAME "(%d): end closing board\n",device->minor);
#endif
    return (0);
}
