/***
*** write.c
***
***  General description of this file:
***     Device driver source code for General Standards 
***     family of analog I/O boards. This file is part of the Linux
***     driver source distribution for this board.
***
***  Copyrights (c):
***     General Standards Corporation (GSC), 2005-2006
***
***  Author:
***     Evan Hillman (evan@generalstandards.com)
***
***  Support:
***     Primary support for this driver is provided by GSC. 
***
***  Platform (tested on, may work with others):
***     Linux, kernel version 2.4.x, 2.6.x,  Red Hat distribution, Intel hardware.
***/  
// set the following flag to trace interrupt debug messages.
#ifdef DEBUG
//#define TRACE_LOCAL TRUE
#endif /* 
 */
  
#include "internals.h"
#ifdef SUPPORT_WRITE
  
/************************************************************************/ 
/* transferFromUser - be sure not to copy more than there is space in the */ 
/* user buffer.  Returns bytes transferred.                             */ 
/************************************************************************/ 
  ulong TransferFromUser (struct device_board * device) 
{
  
long bytesRequested;
  
long wordsRequested;
  
int pageCounter;
  
ulong copied;
  

wordsRequested = device->dmaData[TX].userRemaining;
  

    // convert to bytes.
    bytesRequested = wordsRequested * sizeof (__u32);
  

if (!access_ok
	(VERIFY_READ, (void *) device->dmaData[TX].userBuffer,
	 bytesRequested))
    
    {
      
errmsg ("(%d): device_write - access not OK\n", device->minor);
      
return (-EFAULT);
    
}
  

msg
    ("(%d): transferFromUser userRemain %lX dmaData[TX].intermediateSamples %lX wordsRequested %lX\n",
     device->minor, device->dmaData[TX].userRemaining,
     device->dmaData[TX].intermediateSamples, wordsRequested);
  
msg
    ("(%d): transferFromUser dmaData[TX].userOffset %lX dmaData[TX].intermediateSamples %lX wordsRequested %lX\n",
     device->minor, device->dmaData[TX].userOffset,
     device->dmaData[TX].intermediateSamples, wordsRequested);
  

pageCounter = 0;
  
while ((bytesRequested > 0) && (pageCounter < SG_PAGES))
    
    {
      
ulong sgByteOffset;
      
ulong sgPageIndex;
      
ulong sgPageOffset;
      
long wordsXfered;
      
long bytesToXfer;
      

pageCounter++;
      

sgByteOffset = device->dmaData[TX].intermediateStart * sizeof (__u32);
      
sgPageIndex = sgByteOffset >> PAGE_SHIFT;	// the page on which the data resides
      sgPageOffset = sgByteOffset - (sgPageIndex * PAGE_SIZE);	// the offset on the page where data begins
      bytesToXfer = min ((long) (PAGE_SIZE - sgPageOffset), bytesRequested);
      

msg
	("sgByteOffset %lX sgPageIndex %lX sgPageOffset %lX bytesToXfer %lX bytesRequested %lX user offset %lX\n",
	 
sgByteOffset, sgPageIndex, sgPageOffset, bytesToXfer,
	 bytesRequested, device->dmaData[TX].userOffset);
      

copied =
	__copy_from_user (&device->dmaData[TX].
			  sgPages[sgPageIndex][sgPageOffset],
			  
device->dmaData[TX].userBuffer +
			  device->dmaData[TX].userOffset * sizeof (ulong),
			  bytesToXfer);
      

if (copied != bytesToXfer)
	
	{
	  
errmsg
	    ("write DMA transferFromUser copy error copied %lx expected %lx\n",
	     copied, bytesToXfer);
	  
return -EFAULT;
	
}
      

bytesRequested -= bytesToXfer;
      
bytesRequested = max (0l, bytesRequested);	// update residual
      msg ("residue bytesRequested %lX\n", bytesRequested);
      

wordsXfered = bytesToXfer / sizeof (__u32);
      
device->dmaData[TX].intermediateStart += wordsXfered;
      
device->dmaData[TX].intermediateSamples += wordsXfered;
      

device->dmaData[TX].userOffset += wordsXfered;
      
device->dmaData[TX].userRemaining -= wordsXfered;
      

msg
	("(%d): transferFromUser during userRemain %lX dmaData[TX].intermediateSamples %lX wordsRequested %lX\n",
	 device->minor, device->dmaData[TX].userRemaining,
	 device->dmaData[TX].intermediateSamples, wordsRequested);
    
}
  

msg
    ("(%d): transferFromUser after userRemain %lX dmaData[TX].intermediateSamples %lX wordsRequested %lX\n",
     device->minor, device->dmaData[TX].userRemaining,
     device->dmaData[TX].intermediateSamples, wordsRequested);
  
return wordsRequested * sizeof (__u32);

}



/************************************************************************/ 
/* write operation                                                      */ 
/************************************************************************/ 
  int
device_write (struct file *fp, const char *buf, size_t size, loff_t * lt) 
{
  
struct device_board *device = (struct device_board *) fp->private_data;
  
int retval;
  

device->dmaData[TX].userRemaining = size / sizeof (__u32);
  
device->dmaData[TX].userOffset = 0;
  

    //msg( "(%d): device_write()\n", device->minor);
    msg ("Write() begin buffer count %.8X requested %X\n",
	 readLocal (device, OUT_BUFFER_SIZE_REG), size);
  

    // verify parameters 
    mark (66666662);
  
if (size <= 0)
    {
      
errmsg ("(%d): device_write - zero sample return\n", device->minor);
      
return (0);
    
}
  

    // if in circular mode, make sure the buffer is ready.  If busy for non-blocking,
    // return immediately.  For blocking wait for the buffer to come around.
    
if ((fp->f_flags & O_NONBLOCK)
	 && (readLocal (device, OUT_BUFF_OPS_REG) & OOR_CIRCULAR_BUFFER))
    
    {
      
errmsg ("device_write - noblock return\n");
      
return (-EAGAIN);
    
}
  

    //msg( "(%d): device_write - OCR - %.8X\n", device->minor,readLocal(device,OUT_BUFF_OPS_REG));
    
    // if circular, wait for buffer ready
    retval = waitForOutputReady (device);
  
if (retval != 0)
    
    {
      
errmsg ("device_write - output not ready return\n");
      
return retval;
    
}
  

    // wait for room in output buffer.
    retval = waitForOutputThreshold (device);
  
if (retval != 0)
    
    {
      
errmsg ("device_write - threshold not ready return\n");
      
return retval;
    
}
  

    // now ready to transfer data
    if (device->dmaData[TX].dmaState == DMA_ENABLE)	// use DMA for the write.
    {
      
device->dmaData[TX].intermediateStart = 0;
      
device->dmaData[TX].intermediateSamples = 0;
      
if (!device->dmaData[TX].useMmap)
	
	{
	  
msg ("TX not using mmap\n");
	  
retval = TransferFromUser (device);
	  
if (retval < 0)
	    
	    {
	      
errmsg ("device_write returning TransferFromUser error\n");
	      
return retval;
	    
}
	
}
      

retval = dmaWrite (device, size);
      
if (retval < 0)
	
	{
	  
errmsg ("device_write - dmaWrite error return\n");
	  
return retval;
	
}
    
}
  
  else				// pio
    {
      
int i;
      
unsigned long *dataPtr;
      
unsigned long data;
      

dataPtr = (unsigned long *) buf;
      
	//msg( "(%d): device_write PIO...\n",device->minor);
	for (i = 0; i < device->dmaData[TX].userRemaining; i++)
	
	{
	  
	    // this can probably be refined...
	    __copy_from_user (&data, dataPtr + i, sizeof (unsigned long));
	  
writeLocal (device, OUTPUT_BUFFER_REG, data);
    
} 
} 
    //msg( "(%d): device_write return - OCR - %.8X\n", device->minor,readLocal(device,OUT_BUFF_OPS_REG));
    device->firstWrite = FALSE;
  
retval = size;
  
msg ("Write() end buffer count %.8X returning %X\n",
	readLocal (device, OUT_BUFFER_SIZE_REG), retval);
  

return retval;

}




#else	/* don't support write */
/************************************************************************/ 
/* write operation: easy -- just return an error                        */ 
/************************************************************************/ 
  
#ifdef RTL
  rtl_ssize_t device_write (struct rtl_file * fp, const char *buf,
			    rtl_size_t size, rtl_off_t * lt) 
#else /* 
 */
  int
device_write (struct file *fp, const char *buf, size_t size, loff_t * lt) 
#endif				/* 
 */
{
  
return (-EPERM);

}


#endif /* 
 */
