/***
*** testapp.c  
***
***  General description of this file:
***     Device test program for the General Standards 24DSI32 
***     family Delta-Sigma A/D boards. This file is part of the Linux
***     driver source distribution for this board.
***     
***  Copyrights (c):
***     General Standards Corporation (GSC), 2004-2005
***
***  Author:
***     Evan Hillman (evan@generalstandards.com)
***
***  Support:
***     Primary support for this driver is provided by GSC. 
***
***/  
  
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <fcntl.h>
#include <string.h>
#include <time.h>
#include <sys/ioctl.h>
#include <asm/types.h>
  
#include "gsc24dsi32pll_ioctl.h"
/* text names to describe each type of board supported.
   The order of these is fixed based on the ENUM type_index
   defined in pci24dsi32_ioctl.h */ 

char *boards_supported[] = { 
"PCI_24DSI32", /* PCI_24DSI32 */ 
    "PCI_24DSI12", /* PCI_24DSI32 */ 
  NULL, 
};



/* number of channels for each board supported. */ 
  
#define MAX_CHANNELS 32
int channels[] = { 
32, /* PCI_24DSI32 */ 
    12, /* PCI_24DSI12 */ 
};



#define BUFFER_SIZE (1024*16)
//#define BUFFER_SIZE (1024*256)
//#define BUFFER_SIZE (128)
#define BUFFER_BYTES (BUFFER_SIZE*sizeof(rtl_uint32_t))

int passes, channel;

time_t startTime, lastDisplay, now;


#define OFFSET_HIGH 0x519
#define OFFSET_LOW	0x3CF
#define GAIN_HIGH 0x0501
#define GAIN_LOW 0x03FF	
#define DSI24_NUM_CHANNELS 4  /*channel groups*/
#define DSI24_CHAN_PER_GROUP 8
  
//////////////////////////////////////////////////////////////////////////
  int
dsi24_driver_autocaldump (int handle)
{
  

DEVICE_REGISTER_PARAMS Reg;
  
unsigned long regval;
  
int i = 0, j;
  

rtl_printf ("------- 24DSI %d AUTOCAL DUMP --------\n\n", handle);
  
rtl_printf ("CH\tOFFSET\t\t\tGAIN\n");
  
rtl_printf ("--------------------------------------------------------\n");
  

Reg.eRegister = AUTOCAL_VALUES_REG;
  


    /*Autocal values are stored offset,gain for each channel, total of 2*num
       channel entries.  Write the index (0-2*numchan)
       of the value you want to retrieve, then read the register */ 
    
for (i = 0, j = 0; i < DSI24_NUM_CHANNELS * DSI24_CHAN_PER_GROUP * 2;
	  i++, j++)
    {
      
	/*for each channel (times two) */ 
	
Reg.ulValue = i;
      
if (ioctl (handle, IOCTL_GSC_WRITE_REGISTER, &Reg) < 0)
	
return FALSE;
      
if (ioctl (handle, IOCTL_GSC_READ_REGISTER, &Reg) < 0)	/*READ OFFSET */
	
return FALSE;
      
regval = Reg.ulValue;
      


rtl_printf ("%d\t0x%08x", j, (unsigned int) regval);
      
if (regval < OFFSET_HIGH && regval > OFFSET_LOW)
	
rtl_printf (" (Pass)");
      
      else
	
rtl_printf (" --FAIL--");
      

i++;
      

Reg.ulValue = i;
      
if (ioctl (handle, IOCTL_GSC_WRITE_REGISTER, &Reg) < 0)
	
return FALSE;
      
if (ioctl (handle, IOCTL_GSC_READ_REGISTER, &Reg) < 0)	/*READ GAIN */
	
return FALSE;
      
regval = Reg.ulValue;
      

rtl_printf ("\t0x%08x", (unsigned int) regval);
      
if (regval < GAIN_HIGH && regval > GAIN_LOW)
	
rtl_printf (" (Pass)\n");
      
      else
	
rtl_printf (" --FAIL--\n");
    
}
  

rtl_printf ("\n");
  

return (TRUE);

}



rtl_uint32_t buffer[BUFFER_SIZE], total, count[MAX_CHANNELS], param;


//////////////////////////////////////////////////////////////////////////
  int
main (int argc, char *argv[]) 
{
  
struct gen_assign_params gen_param;
  
struct device_register_params rw_q;
  
int res;
  
//    rtl_uint32_t buffer[BUFFER_SIZE], total, count[MAX_CHANNELS],param;
  int fd, i, print, nread, bytesRead, wordsRead;
  
int board_type;
  
int lapse;			// measure of run time.
  int buffersRead;
  
int error_code;
  
int last;
  
int samplesChecked = 0;
  
struct gen_rate_params gen_rate;
  
int last_group;
  
struct rtl_timespec t1, t2, start;
  

fd = open ("/dev/gs_ad_0", O_RDWR);
  
if (fd < 0)
    {
      
rtl_printf ("%s: can not open device /dev/gs_ad_0\n", argv[0]);
      
return (1);
    
}
  

res = ioctl (fd, IOCTL_GSC_NO_COMMAND, NULL);
  

    // find out what variety of board this is.
    
res = ioctl (fd, IOCTL_GSC_GET_DEVICE_TYPE, &board_type);
  
if (res < 0)
    {
      
rtl_printf ("%s: ioctl IOCTL_GSC_GET_DEVICE_TYPE failed\n", argv[0]);
      
goto OUT;
    
}
  
rtl_printf ("open OK - board type: %s channels: %d\n",
	       boards_supported[board_type], channels[board_type]);
  

    // read the board control register to show before-reset state.
    
rw_q.eRegister = BOARD_CTRL_REG;
  
res = ioctl (fd, IOCTL_GSC_READ_REGISTER, &rw_q);
  
if (res < 0)
    {
      
rtl_printf ("%s: ioctl IOCTL_GSC_READ_REGISTER failed\n", argv[0]);
      
goto OUT;
    
}
  
rtl_printf ("before reset: BCR is 0x%lx\n", rw_q.ulValue);
  

    // initialize the hardware to a known state.
    
res = ioctl (fd, IOCTL_GSC_INITIALIZE, NULL);
  
if (res < 0)
    {
      
rtl_printf ("%s: ioctl IOCTL_GSC_INIT failed\n", argv[0]);
      
goto OUT;
    
}
  
rtl_printf ("board init OK\n");
  

    // read the BCR again to show the reset worked.
    
rw_q.eRegister = BOARD_CTRL_REG;
  
rw_q.ulValue = 0x2000;
  
res = ioctl (fd, IOCTL_GSC_READ_REGISTER, &rw_q);
  
if (res < 0)
    {
      
rtl_printf ("%s: ioctl IOCTL_GSC_READ_REGISTER failed\n", argv[0]);
      
goto OUT;
    
}
  
rtl_printf ("after reset: BCR is 0x%lx\n", rw_q.ulValue);
  

    // make sure there is no data coming into the buffer.
    
param = STOP_ACQUIRE;
  
res = ioctl (fd, IOCTL_GSC_SET_ACQUIRE_MODE, &param);
  
if (res < 0)
    {
      
rtl_printf ("%s: ioctl IOCTL_GSC_SET_ACQUIRE_MODE failed\n", argv[0]);
      
goto OUT;
    
}
  

    // get the buffers to a known state.  The driver waits
    // for the buffer clear to complete.
    
res = ioctl (fd, IOCTL_GSC_CLEAR_BUFFER, NULL);
  
if (res < 0)
    {
      
rtl_printf ("%s: ioctl IOCTL_GSC_CLEAR_BUFFER failed\n", argv[0]);
      
goto OUT;
    
}
  

    // Verify that the buffer is empty.
    
rw_q.eRegister = BUFFER_SIZE_REG;
  
res = ioctl (fd, IOCTL_GSC_READ_REGISTER, &rw_q);
  
if (res < 0)
    {
      
rtl_printf ("%s: ioctl IOCTL_GSC_READ_REGISTER failed\n", argv[0]);
      
goto OUT;
    
}
  
rtl_printf ("after clear: Buffer size is 0x%lx\n", rw_q.ulValue);
  

    // set timeout.  Autocal can take up to 20 seconds.
    
param = 30;		//seconds
  res = ioctl (fd, IOCTL_GSC_SET_TIMEOUT, &param);
  
if (res < 0)
    {
      
rtl_printf ("%s: ioctl IOCTL_GSC_SET_TIMEOUT failed\n", argv[0]);
      
goto OUT;
    
}
  

    // set the buffer overflow threshold.
    
param = INPUT_BUFFER_SIZE - 0x100;	// max allowed samples in buffer.
  res = ioctl (fd, IOCTL_GSC_SET_OVERFLOW_LEVEL, &param);
  
if (res < 0)
    {
      
rtl_printf ("%s: ioctl IOCTL_GSC_SET_OVERFLOW_LEVEL failed\n",
		   argv[0]);
      
goto OUT;
    
}
  

    // Enable buffer overflow checking.
    
param = TRUE;
  
res = ioctl (fd, IOCTL_GSC_SET_OVERFLOW_CHECK, &param);
  
if (res < 0)
    {
      
rtl_printf ("%s: ioctl IOCTL_GSC_SET_OVERFLOW_CHECK failed\n",
		   argv[0]);
      
goto OUT;
    
}
  

    // select data input mode.
    
param = MODE_DIFFERENTIAL;
  
    //  param = MODE_VREF_TEST;
    res = ioctl (fd, IOCTL_GSC_SET_INPUT_MODE, &param);
  
if (res < 0)
    {
      
rtl_printf ("%s: ioctl IOCTL_GSC_SET_INPUT_MODE failed\n", argv[0]);
      
goto OUT;
    
}
  

    // select data input range.
    
param = RANGE_10V;
  
res = ioctl (fd, IOCTL_GSC_SET_INPUT_RANGE, &param);
  
if (res < 0)
    {
      
rtl_printf ("%s: ioctl IOCTL_GSC_SET_INPUT_RANGE failed\n", argv[0]);
      
goto OUT;
    
}
  

    // set the buffer trigger level.
    
//    param = 64;
//    param = 0x30000;
    param = 0x3000;
  
res = ioctl (fd, IOCTL_GSC_SET_BUFFER_THRESHOLD, &param);
  
if (res < 0)
    {
      
rtl_printf ("%s: ioctl IOCTL_GSC_SET_ACQUIRE_MODE failed\n", argv[0]);
      
goto OUT;
    
}
  

    // tell the driver to fill the user buffer before returning.
    
param = TRUE;
  
    //param = FALSE;
    res = ioctl (fd, IOCTL_GSC_FILL_BUFFER, &param);
  
if (res < 0)
    {
      
rtl_printf ("%s: ioctl IOCTL_GSC_FILL_BUFFER failed\n", argv[0]);
      
goto OUT;
    
}
  

    // set software sync mode.
    
res = ioctl (fd, IOCTL_GSC_SET_SW_SYNC, NULL);
  
if (res < 0)
    {
      
rtl_printf ("%s: ioctl IOCTL_GSC_SET_SW_SYNC failed\n", argv[0]);
      
goto OUT;
    
}
  


    //********************************************
    // set up the data sampling rate parameters
    //********************************************
    
    // Set the rate generator rate.  This program only uses generator "A."
    
gen_rate.eGenerator = GEN_A;
  
if (board_type == PCI_24DSI32)
    
    {
      
gen_rate.ulNrate = 100;
    
}
  
  else				// 24DSI12
    {
      
gen_rate.ulNrate = 1000;
    
}
  
res = ioctl (fd, IOCTL_GSC_SET_GEN_RATE, &gen_rate);
  
if (res < 0)
    {
      
rtl_printf ("%s: ioctl IOCTL_GSC_SET_GEN_RATE failed\n", argv[0]);
      
goto OUT;
    
}
  

    // assign groups of channels to generators. Assign all groups to generator "A."
    
if (board_type == PCI_24DSI32)
    
    {
      
last_group = GRP_3;
    
}
  
  else				// 24DSI12
    {
      
last_group = GRP_1;
    
}
  
for (i = GRP_0; i <= last_group; i++)
    
    {
      
gen_param.eGroup = i;
      
gen_param.eGenAssign = ASN_GEN_A;
      
res = ioctl (fd, IOCTL_GSC_ASSIGN_GEN_TO_GROUP, &gen_param);
      
if (res < 0)
	{
	  
printf ("%s: ioctl IOCTL_GSC_ASSIGN_GEN_TO_GROUP failed\n",
		   argv[0]);
	  
goto OUT;
	
}
    
}
  

    // Select method for transfer for data from hardware
    // to the intermediate buffer
    
param = DMA_ENABLE;	// Regular DMA
  //param = DMA_DEMAND_MODE; //Demand Mode DMA
  //param = DMA_DISABLE; //PIO
  
ioctl (fd, IOCTL_GSC_SET_DMA_STATE, &param);
  
if (res < 0)
    {
      
rtl_printf ("%s: ioctl IOCTL_GSC_SET_DMA_STATE failed\n", argv[0]);
      
goto OUT;
    
}
  

    // Set the hardware to synchronize scans.
    
param = TRUE;
  
res = ioctl (fd, IOCTL_GSC_SYNCRONIZE_SCAN, &param);
  
if (res < 0)
    {
      
rtl_printf ("%s: ioctl IOCTL_GSC_SYNCRONIZE_SCAN failed\n", argv[0]);
      
goto OUT;
    
}
  

    // set the rate divisor for each group.
    
for (i = GRP_0; i <= last_group; i++)
    {
      
RATE_DIVISOR_PARAMS rd;
      
rd.ulGroup = i;
      
rd.ulDivisor = i + 10;
      
ioctl (fd, IOCTL_GSC_SET_RATE_DIVISOR, &rd);
      
if (res < 0)
	{
	  
rtl_printf ("%s: ioctl IOCTL_GSC_SET_RATE_DIVISOR failed\n",
		       argv[0]);
	  
goto OUT;
	
}
    
}
  

rtl_printf ("after set rate divisor\n");
  

    // autocal the analog section.  Driver waits till complete
    // before returning.
    
    // while (!kbhit())
    //   for (i=0;i<2;i++)
  {
    
      //time(&startTime);
      rtl_clock_gettime (RTL_CLOCK_REALTIME, &t1);
    
      //        rtl_printf("autocalibration...\n");
      res = ioctl (fd, IOCTL_GSC_AUTO_CAL, NULL);
    
      //        rtl_printf("autocal return: %x\n",res);
      if (res < 0)
      {
	
rtl_printf ("%s: ioctl IOCTL_GSC_AUTOCAL failed\n", argv[0]);
      
}
    
      //        rtl_printf("auto calibration OK\n");
      //dsi24_driver_autocaldump(fd);
      
      //time(&now);
      rtl_clock_gettime (RTL_CLOCK_REALTIME, &t2);
    
rtl_timespec_sub (&t2, &t1);
    
//        if (now-startTime < autocalMin) autocalMin=now-startTime;
//        if (now-startTime > autocalMax) autocalMax=now-startTime;
      
rtl_printf ("Autocal took %d.%dus.\n", (int) (t2.tv_sec),
		   (int) (t2.tv_nsec / 1000));
    
if (res < 0)
      {
	
goto OUT;
      
}
  
}
  

    // start acquiring data 
    param = START_ACQUIRE;
  
ioctl (fd, IOCTL_GSC_SET_ACQUIRE_MODE, &param);
  
if (res < 0)
    {
      
rtl_printf ("%s: ioctl IOCTL_GSC_SET_ACQUIRE_MODE failed\n", argv[0]);
      
goto OUT;
    
}
  

for (i = 0; i < MAX_CHANNELS; i++)
    {
      
count[i] = 0;
    
}
  

total = 0L;
  
print = -1;
  
nread = passes;
  

    //startTime=time(NULL);
    rtl_clock_gettime (RTL_CLOCK_REALTIME, &start);
  
rtl_printf ("\n");
  
lapse = 0;
  
last = -1;
  
buffersRead = 0;
  

    /*******************************************/ 
    rtl_printf ("Starting main loop...\n");
  

lapse = 0;
  
while (lapse < 60)
    {
      

lapse++;
      

	// make sure the buffer contents are invalid, for testing.
	
memset (buffer, 0xaaaaaaaa, sizeof (buffer));
      

	// size of read is in bytes.
	bytesRead = read (fd, buffer, BUFFER_BYTES);
      
	/*
	   if (buffersRead==0) // check if the first sample is channel 0
	   {
	   channel = ((buffer[i] & IDR_CHANNEL_TAG_MASK) >> IDR_CHANNEL_TAG_SHIFT); 
	   if (channel != 0)
	   {
	   rtl_printf("Error on first sample - channel tag is %d not 0\n",channel);
	   }
	   }
	 */ 
	buffersRead++;
      
//      rtl_printf("read: %X\n",numread);
	if (bytesRead < 0)
	{
	  
rtl_printf ("\nread error -> after read...res = %d\n", bytesRead);
	  
goto OUT;
	
}
      

wordsRead = bytesRead / sizeof (long);
      
total += wordsRead;
      
passes++;
      

for (i = 0; i < wordsRead; i++)
	{
	  
channel =
	    ((buffer[i] & IDR_CHANNEL_TAG_MASK) >> IDR_CHANNEL_TAG_SHIFT);
	  
if (channel == 0)
	    {
	      
unsigned long d;
	      
d = (buffer[i] & IDR_DATA_MASK);
	      
printf ("C0 is %x\n", d);
	      
usleep (1000 * 1000);
	    
}
	  

	    //rtl_printf("[%d]%.8X \n",i,buffer[i]);
	    if (channel < 0 || channel > channels[board_type])
	    
	    {
	      
rtl_printf ("Channel out of range: %d\n", channel);
	      
goto OUT;
	    
}
	  
if (channel != last + 1)
	    
	    {
	      
if (last == channels[board_type] - 1 && channel == 0)
		
		{
		  
		    //  rtl_printf("!");
		}
	      
	      else
		
		{
		  
		    // verify channel sequence, if appropriate for the type of board.
		    // not all board models guarantee channels will be in sequence.
		    /*
		       int q;
		       rw_q.eRegister = BUFFER_SIZE_REG;
		       res = ioctl(fd, IOCTL_GSC_READ_REGISTER, &rw_q);
		       rtl_printf("\nChannel sequence error last: %d Channel: %d Samples Checked: %d (0x%X) Buffer index: %d\n", last, channel,samplesChecked,samplesChecked,wordsRead-1);
		       for(q=0;q<=i;q++)
		       rtl_printf("[%d]%.8X ",q,buffer[q]);
		       rtl_printf("Buffer size register: %d 0x%.8X ",rw_q.ulValue,rw_q.ulValue);
		       goto OUT;
		     */ 
		}
	    
}
	  
samplesChecked++;
	  
last = channel;
	  
count[channel]++;
	
}
      
#if 0
	now = time (NULL);
      
if (now != lastDisplay)
	
	{
	  
lastDisplay = now;
	  
print = (total >> 20);
	  
	  {
	    
int j;
	    
lapse = time (NULL) - startTime;
	    
if (lapse != 0)
	      
	      {
		
rtl_printf
		  ("seconds: %.8d Minutes: %d Words read: %.8x samples/sec: %.8ld - ",
		   lapse, lapse / 60, wordsRead, total / lapse);
		
j = 0;
	      
}
	  
}
	  
rtl_printf ("      \r");
	  
fflush (stdout);
	
}
      
#endif /* 
 */
    }
  

rtl_printf ("\n Total samples: %ld Samples per channel:\n", total);
  

for (i = 0; i < MAX_CHANNELS; i++)
    
    {
      
rtl_printf ("%.8d ", count[i]);
      
if ((i > 0) & ((i + 1) % 8 == 0))
	
	{
	  
rtl_printf ("\n");
	
}
    
}

OUT:
    // stop the clock.
    
param = STOP_ACQUIRE;
  
ioctl (fd, IOCTL_GSC_SET_ACQUIRE_MODE, &param);
  

    // read the last hardware/driver error.
    
res = ioctl (fd, IOCTL_GSC_GET_DEVICE_ERROR, &error_code);
  
if (res < 0)
    {
      
rtl_printf ("%s: ioctl IOCTL_GSC_GET_DEVICE_TYPE failed\n", argv[0]);
      
close (fd);
      
return (0);
    
}
  
rtl_printf ("\n Last error code: %d\n", error_code);
  

close (fd);
  

return 0;

}


