/***
*** read.c
***
***  General description of this file:
***     Device driver source code for General Standards 
***     family of analog I/O boards. This file is part of the Linux
***     driver source distribution for this board.
***
***  Copyrights (c):
***     General Standards Corporation (GSC), 2005-2006
***
***  Author:
***     Evan Hillman (evan@generalstandards.com)
***
***  Support:
***     Primary support for this driver is provided by GSC. 
***
***  Platform (tested on, may work with others):
***     Linux, kernel version 2.4.x, 2.6.x,  Red Hat distribution, Intel hardware.
***/  
  
//////////////////////////////////////////////////////////////////////////
// set the following flag to trace interrupt debug messages.
#ifdef DEBUG
#define TRACE_LOCAL TRUE
#endif /* 
 */
  
#include "internals.h"
  
#ifdef SUPPORT_READ

static long FillintermediateBuffer (struct device_board *device,
				     int noblock);


#ifndef SUPPORT_DEMAND_MODE
#define DMA_DEMAND_MODE 100 /* not used in this driver, here for a placeholder */
#endif /* 
 */
  
/************************************************************************/ 
/* TransferToUser - be sure not to copy more than there is space in the */ 
/* user buffer.  Returns bytes transferred.                             */ 
/************************************************************************/ 
  long
TransferToUser (struct device_board *device) 
{
  
long bytesRequested;
  
long wordsRequested;
  
long pageCounter;
  

    // for PIO the data is already in the user buffer.
    
if (device->dmaData[RX].dmaState == DMA_DISABLE)	// PIO
    {
      
wordsRequested = device->dmaData[RX].intermediateSamples;
      
device->dmaData[RX].intermediateSamples = 0;
      
return wordsRequested * sizeof (__u32);
    
}
  

wordsRequested = device->dmaData[RX].userRemaining;
  

    // make sure there is data
    wordsRequested =
    min (device->dmaData[RX].intermediateSamples, wordsRequested);
  

    // convert to bytes.
    bytesRequested = wordsRequested * sizeof (__u32);
  

msg
    ("(%d): TransferToUser userRemain %lX dmaData[RX].intermediateSamples %lX wordsRequested %lX\n",
     
device->minor, device->dmaData[RX].userRemaining,
     device->dmaData[RX].intermediateSamples, wordsRequested);
  
msg
    ("(%d): TransferToUser dmaData[RX].userOffset %lX dmaData[RX].intermediateSamples %lX wordsRequested %lX\n",
     
device->minor, device->dmaData[RX].userOffset,
     device->dmaData[RX].intermediateSamples, wordsRequested);
  
msg ("(%d): TransferToUser dmaData[RX].intermediateStart %lX \n",
	device->minor, device->dmaData[RX].intermediateStart);
  

pageCounter = 0;
  
while ((bytesRequested > 0) && (pageCounter < SG_PAGES))
    
    {
      
long sgByteOffset;
      
long sgPageIndex;
      
long sgPageOffset;
      
long wordsXfered;
      
long bytesToXfer;
      

pageCounter++;
      

sgByteOffset = device->dmaData[RX].intermediateStart * sizeof (__u32);
      
sgPageIndex = sgByteOffset >> PAGE_SHIFT;	// the page on which the data resides
      sgPageOffset = sgByteOffset - (sgPageIndex * PAGE_SIZE);	// the offset on the page where data begins
      bytesToXfer = min ((long) (PAGE_SIZE - sgPageOffset), bytesRequested);
      

//    msg("sgByteOffset 0x%lX sgPageIndex 0x%lX sgPageOffset 0x%lX bytesToXfer 0x%lX bytesRequested 0x%lX\n", 
//        sgByteOffset, sgPageIndex, sgPageOffset, bytesToXfer, bytesRequested);
	//{
	//    int q;
	//    for (q=0;q<wordsRequested;q++)
	//    {
	//        msg("sg buffer - data base %p index: %d ptr: %p data: %.8X\n",
	//            device->dmaData[RX].sgPages[sgPageIndex], q, 
	//            &device->dmaData[RX].sgPages[sgPageIndex][sgPageOffset+q],
	//            device->dmaData[RX].sgPages[sgPageIndex][sgPageOffset+q]);
	//    }
	//}
//    msg("^^sg buffer - sg pages 0x%lX page index 0x%lX data base %p current ptr: %p phys: %p\n",
//        SG_PAGES,
//        sgPageIndex,
//        device->dmaData[RX].sgPages[sgPageIndex],
//        (void*)&device->dmaData[RX].sgPages[sgPageIndex][sgPageOffset],
//        (void*)&device->dmaData[RX].sgPagesPhysical[sgPageIndex][sgPageOffset]);
#ifdef CHECK_DATA
	
#  warning "CHECK_DATA enabled"
	
      {
	
long q;
	
__u32 * ptr =
	  (__u32 *) & device->dmaData[RX].sgPages[sgPageIndex][sgPageOffset];
	
for (q = 0; q < bytesToXfer / 4; q++)
	  
	  {
	    
	      //printk("%lX sys ptr %p ptr+q %.8X\n", q,(ptr+q), *(ptr+q));
	      if (*(ptr + q) == 0xAAAAAAAA)
	      
		//if (*(ptr+q) > 0xaffff || *(ptr+q) < 0x7000)
	      {
		
printk ("%lX sys ptr %p ptr+q %.8X\n", q, (ptr + q),
			 *(ptr + q));
		
mark (0xabcdef00);
		
mark ((__u32) (ptr + q));
		
mark ((__u32) * (ptr + q));
		
		  //mark((__u32)&device->dmaData[RX].sgPagesPhysical[sgPageIndex][sgPageOffset]);
		  
errmsg
		  ("Transfer to user data integrity problem. SG pages %ld SG page table:\n",
		   SG_PAGES);
		
		{
		  
long i;
		  
for (i = 0; i < SG_PAGES; i++)
		    
		    {
		      
msg (" %lX sys: %p phys: %p\n", i,
			    device->dmaData[RX].sgPages[i],
			    device->dmaData[RX].sgPagesPhysical[i]);
		    
}
		
}
		
return -EIO;
	      
}
	  
}
      
}
      

#endif /* 
 */
	if (__copy_to_user
	    (device->dmaData[RX].userBuffer +
	     device->dmaData[RX].userOffset * sizeof (__u32),
	     
&device->dmaData[RX].sgPages[sgPageIndex][sgPageOffset],
	     
bytesToXfer))
	
	{
	  
errmsg ("(%d): TransferToUser copy to user fault\n",
		   device->minor);
	  
return -EFAULT;
	
}
      

#ifdef CHECK_DATA
	memset (&device->dmaData[RX].sgPages[sgPageIndex][sgPageOffset], 0xAA,
		bytesToXfer);
      
#endif /* 
 */
	
bytesRequested -= bytesToXfer;
      
bytesRequested = max (0l, bytesRequested);	// update residual
      msg ("residue bytesRequested %lX\n", bytesRequested);
      

wordsXfered = bytesToXfer / sizeof (__u32);
      
device->dmaData[RX].intermediateStart += wordsXfered;
      
device->dmaData[RX].intermediateSamples -= wordsXfered;
      

device->dmaData[RX].userOffset += wordsXfered;
      
device->dmaData[RX].userRemaining -= wordsXfered;
      

msg
	("(%d): TransferToUser during userRemain %lX dmaData[RX].intermediateSamples %lX wordsRequested %lX\n",
	 device->minor, device->dmaData[RX].userRemaining,
	 device->dmaData[RX].intermediateSamples, wordsRequested);
    
}
  

msg
    ("(%d): TransferToUser after userRemain %lX dmaData[RX].intermediateSamples %lX wordsRequested %lX\n",
     device->minor, device->dmaData[RX].userRemaining,
     device->dmaData[RX].intermediateSamples, wordsRequested);
  
return wordsRequested * sizeof (__u32);

}



//////////////////////////////////////////////////////////////////////////
  long
ProcessRead (struct device_board *device) 
{
  
long bytesCopied;
  
long retval;
  
long transferred;
  

bytesCopied = 0;
  

    // start by copying what is already in the intermediate buffer.
    
msg ("(%d): ProcessRead - residual samples: %ld\n", device->minor,
	  device->dmaData[RX].intermediateSamples);
  
msg ("(%d): ProcessRead - read DMA: %d \n", device->minor,
	device->dmaData[RX].dmaState);
  

    // give either what the user requests or what is left in the buffer, whichever is less.
    if (device->dmaData[RX].useMmap)
    
    {
      
msg ("read mmap\n");
      
return (FillintermediateBuffer
	       (device, (device->fp->f_flags & O_NONBLOCK)));
    
}				// if (!device->dmaData.useMmap)
  else
    
    {
      
msg ("read non-mmap\n");
      
if (device->dmaData[RX].intermediateSamples > 0)
	
	{
	  
msg ("(%d): ProcessRead - local data copy...%lX\n", device->minor,
		device->dmaData[RX].intermediateSamples);
	  
transferred = TransferToUser (device);
	  
if (transferred > 0)
	    
	    {
	      
bytesCopied += transferred;
	    
}
	  
	  else
	    
	    {
	      
errmsg ("ProcessRead TransferToUser error\n");
	      
return transferred;
	    
}
	  
msg ("(%d): ProcessRead - local data copy...%lX\n", device->minor,
		device->dmaData[RX].intermediateSamples);
	  

	    // if the user buffer is now full, return.
	    if (device->dmaData[RX].userRemaining == 0)	// done
	    {
	      
msg ("returning user buffer full...\n");
	      
return (bytesCopied);
	    
}
	  

	    // if the user does not ask for full buffers, return
	    if (!device->fillBuffer)
	    
	    {
	      
msg ("returning don't fill buffer...\n");
	      
return (bytesCopied);
	    
}
	

}			// if (device->dmaData[RX].intermediateSamples > 0)
      
	// loop till done if user wants a full buffer.
	do
	
	{
	  
msg
	    ("(%d): ProcessRead - looping...remaining: %ld fillbuffer %d\n",
	     device->minor, device->dmaData[RX].userRemaining,
	     device->fillBuffer);
	  
retval =
	    FillintermediateBuffer (device,
				    (device->fp->f_flags & O_NONBLOCK));
	  
if (retval < 0)
	    
	    {			// returned error
	      errmsg ("returning FillintermediateBuffer error...\n");
	      
return retval;
	    
}
	  
if (retval == 0)
	    
	    {			// got no data
	      errmsg ("returning got no data error...\n");
	      
return (-EAGAIN);
	    
}
	  
transferred = TransferToUser (device);
	  
if (transferred > 0)
	    
	    {
	      
bytesCopied += transferred;
	    
}
	  
	  else
	    
	    {
	      
errmsg ("ProcessRead TransferToUser error\n");
	      
return transferred;
	    
}
	
}
      
while ((device->dmaData[RX].userRemaining > 0) && device->fillBuffer);
      

msg ("(%d): >>>>>>ProcessRead - returning: %ld bytes %ld samples \n",
	     device->minor, bytesCopied, bytesCopied / sizeof (long));
      
return (bytesCopied);
    
}

}



/************************************************************************/ 
/* read operation: either polled or uses PLX DMA on CH0                 */ 
/************************************************************************/ 
#ifdef RTL
  rtl_ssize_t device_read (struct rtl_file * fp, char *buf, rtl_size_t size,
			   rtl_off_t * lt) 
#else /* 
 */
  int
device_read (struct file *fp, char *buf, size_t size, loff_t * lt) 
#endif				/* 
 */
{
  
struct device_board *device;
  
u32 wordsRequested;
  

#ifdef RTL
    device = (struct device_board *) fp->f_priv;
  
#else /* 
 */
    device = (struct device_board *) fp->private_data;
  
#endif /* 
 */
    
    // fail read if buffer has overflowed.
#ifdef SUPPORT_BUFFER_OVERFLOW
    if (device->detectOverflow && overflowDetected (device))
    
    {
      
errmsg ("Input buffer overflow detected.\n");
      
regdump (device, "input overflow ");
      
return (-EFAULT);
    
}
  

//  if(device->detectUnderflow&&(readLocal(device,BUFFER_CONTROL_REG)&BUCR_BUFFER_UNDERFLOW))
//  {
//    errmsg("Input buffer underflow detected %.8X\n",readLocal(device,BOARD_CTRL_REG));
//    regdump(device,"input underflow ");
//    return (-EFAULT);
//  }
#endif /* 
 */
    device->fp = fp;
  
wordsRequested = size / sizeof (__u32);
  
msg ("(%d): device_read() data requested: %d %x -------------------%d\n",
	device->minor, wordsRequested, wordsRequested, device->nread);
  
#ifdef TRACE_LOCAL
    device->nread++;
  
#endif /* 
 */
    // verify parameters
    if (wordsRequested <= 0)
    
    {
      
msg ("(%d): device_read - zero sample return\n", device->minor);
      
return (0);
    
}
  

if (!access_ok (VERIFY_WRITE, buf, size))
    
    {
      
msg ("(%d): device_read - access not OK\n", device->minor);
      
return (-EFAULT);
    
}
  

device->dmaData[RX].userOffset = 0;	// start at the beginning of the user buffer.
  device->dmaData[RX].userRemaining = wordsRequested;
  
device->dmaData[RX].userBuffer = buf;
  
return ProcessRead (device);

}



/************************************************************************/ 
/*  FillintermediateBuffer                                              */ 
/*                                                                      */ 
/* The method first attempts to fill the DMA buffer to a "reasonable"   */ 
/* level                                                                */ 
/************************************************************************/ 
  static long
FillintermediateBuffer (struct device_board *device, int noblock) 
{
  
long retval;
  
char *buf;
  
long i;
  

retval = -ERESTARTSYS;
  

buf =
    device->dmaData[RX].userBuffer +
    device->dmaData[RX].userOffset * sizeof (__u32);
  

device->dmaData[RX].intermediateStart = 0;
  
device->dmaData[RX].intermediateSamples = 0;
  

if (down_interruptible (&device->hwSem))
    
    {
      
return -ERESTARTSYS;
    
}
  

    //msg("(%d): FilldmaData[RX].intermediateBuffer() wordsRequested %ld SG_BUFFER_SAMPLES %ld\n", device->minor,wordsRequested,SG_BUFFER_SAMPLES);
    
switch (device->dmaData[RX].dmaState)
    
    {
      
	// for PIO and regular DMA, wait for sufficient samples
    case DMA_DISABLE:		// PIO
    case DMA_ENABLE:
      

	// enable the interrupt in case the interrupt happens during this window of decision.
	
up (&device->hwSem);
      
retval = waitForInputThreshold (device, noblock);
      
if (retval != 0)
	
	{
	  
errmsg ("(%d): waitForInputThreshold - returned failed state\n",
		   device->minor);
	  
return retval;
	
}
      
if (down_interruptible (&device->hwSem))
	
	{
	  
return -ERESTARTSYS;
	
}
      
break;
    

case DMA_DEMAND_MODE:
      
	// for demand mode, wait for the DMA done interrupt.
	break;
    

default:			// invalid state
      errmsg
	("(%d): FilldmaData[RX].intermediateBuffer - invalid state failure\n",
	 device->minor);
      
up (&device->hwSem);
      
return (-EIO);
      
break;
    
}				// switch
  
    // move the data from the hardware buffer to the intermediate buffer by the user's chosen method.
    
    //if (device->dmaData[RX].dmaState != DMA_DEMAND_MODE)
    //{
    //    wordsRequested=min(wordsRequested,(long) readLocal(device,INPUT_BUFFER_SIZE_REG));
    //}
    
switch (device->dmaData[RX].dmaState)
    
    {
    
case DMA_DISABLE:		// PIO
      //msg("(%d): PIO starting addr %p to transfer %ld\n", device->minor,device->dmaData[RX].intermediateBuffer,wordsRequested);
#ifdef TRACE_LOCAL
      //memset(device->dmaData[RX].intermediateBuffer,0x55aa55aa, DMA_ORDER*PAGE_SIZE);
#endif /* 
 */
      {
	
long numToXfer;
	
__u32 * ptr;
	

numToXfer =
	  min ((long) device->dmaData[RX].userRemaining,
	       (long) readLocal (device, INPUT_BUFFER_SIZE_REG));
	
ptr = (__u32 *) device->dmaData[RX].userBuffer;
	
msg
	  ("(%d): PIO to transfer %ld, device->dmaData[RX].intermediateSamples %lX\n",
	   device->minor, numToXfer, device->dmaData[RX].intermediateSamples);
	
for (i = 0; i < numToXfer; i++)
	  
	  {
	    
ptr[i] = readLocal (device, INPUT_DATA_BUFF_REG);
	    
msg ("(%d): PIO index %ld data %.8X\n", device->minor, i,
		  ptr[i]);
	  
}
	
device->dmaData[RX].intermediateSamples += numToXfer;
	
device->dmaData[RX].userRemaining -= numToXfer;
      
}
      
retval = i;
      
break;
    

case DMA_DEMAND_MODE:
    
case DMA_ENABLE:
      
msg ("(%d): DMA transfer \n", device->minor);
      
#ifdef TRACE_LOCAL
	//memset(device->dmaData[RX].intermediateBuffer,0x55aa55aa, DMA_ORDER*PAGE_SIZE);
#endif /* 
 */
	if ((retval = dmaRead (device)) < 0)
	
	{
	  
errmsg ("(%d): FillintermediateBuffer - dmaRead failure\n",
		   device->minor);
	  
up (&device->hwSem);
	  
return (-EAGAIN);
	
};
      
break;
    

default:			// invalid state
      errmsg ("(%d): FillintermediateBuffer - invalid DMA state\n",
	      device->minor);
      
up (&device->hwSem);
      
return (-EIO);
      
break;
    
}
  

up (&device->hwSem);
  
msg ("FillintermediateBuffer returning %ld\n", retval);
  
return retval;


}				// FillintermediateBuffer


//////////////////////////////////////////////////////////////////////////
#else	/* don't support read */
/************************************************************************/ 
/* write operation: easy -- just return an error                        */ 
/************************************************************************/ 
  int
device_read (struct file *fp, char *buf, size_t size, loff_t * lt) 
{
  
return (-EPERM);

}


#endif /* 
 */
  
