/***
***  main.c
***
***  General description of this file:
***     Device driver source code for General Standards 
***     family of analog I/O boards. This file is part of the Linux
***     driver source distribution for this board.
***
***  Copyrights (c):
***     General Standards Corporation (GSC), 2005-2006
***
***  Author:
***     Evan Hillman (evan@generalstandards.com)
***
***  Support:
***     Primary support for this driver is provided by GSC. 
***
***  Platform (tested on, may work with others):
***     Linux, kernel version 2.4.x, 2.6.x,  Red Hat distribution, Intel hardware.
***/  

//////////////////////////////////////////////////////////////////////////
// set the following flag to trace interrupt debug messages.
#ifdef DEBUG
#define TRACE_LOCAL TRUE
#endif /* 
*/

#include "internals.h"

static const char *regname[] = 
{ 
    "       BOARD_CTRL_REG", 
        "   RATE_CONTROL_A_REG",

        "   RATE_CONTROL_B_REG", 
        "      RATE_ASSIGN_REG",

        "    RATE_DIVISORS_REG", 
        "           RESERVED_1",

        "     PLL_REF_FREQ_REG", 
        " RANGE/FILTER CONTROL",

        "   BUFFER_CONTROL_REG", 
        "     BOARD_CONFIG_REG",

        "      BUFFER_SIZE_REG", 
        "   AUTOCAL_VALUES_REG",

        "INPUT_DATA_BUFFER_REG", 
};



/************************************************************************/ 
/* regdump - debug only                                                 */ 
/************************************************************************/ 
void
regdump (struct device_board *device, char *label) 
{

    int q;


    for (q = 0; q <= LAST_REG; q++)

    {

        errmsg ("(%d - %s): register %.2d (addr %.2X) %s value: %.8x\n",
            device->minor, label, q, q * 4, regname[q], readLocal (device,
            q));

    }


    errmsg ("(%d - %s)                 IntCntrlStat: %.8X \n", device->minor,
        label, readPLX (device, INT_CTRL_STATUS));

    errmsg ("(%d - %s)                    DMAArbitr: %.8X \n", device->minor,
        label, readPLX (device, DMA_MODE_ARBITRATION));

    errmsg ("(%d - %s)                 DMAThreshold: %.8X \n", device->minor,
        label, readPLX (device, DMA_THRESHOLD_REG));


    errmsg ("(%d - %s)                    DMA mode0: %.8X \n", device->minor,
        label, readPLX (device, DMA_CH_0_MODE));

    errmsg ("(%d - %s)                  DMAByteCnt0: %.8X \n", device->minor,
        label, readPLX (device, DMA_CH_0_TRANS_BYTE_CNT));

    errmsg ("(%d - %s)                 DMADescrPtr0: %.8X \n", device->minor,
        label, readPLX (device, DMA_CH_0_DESC_PTR));

    errmsg ("(%d - %s)                DMALocalAddr0: %.8X \n", device->minor,
        label, readPLX (device, DMA_CH_0_LOCAL_ADDR));

    errmsg ("(%d - %s)            DMA_CH_0_PCI_ADDR: %.8X \n", device->minor,
        label, readPLX (device, DMA_CH_0_PCI_ADDR));

    errmsg ("(%d - %s)                 DMA status 0: %.8X \n\n", device->minor,
        label, readPLXb (device, DMA_CMD_STATUS_0));


    errmsg ("(%d - %s)                    DMA mode1: %.8X \n", device->minor,
        label, readPLX (device, DMA_CH_1_MODE));

    errmsg ("(%d - %s)                  DMAByteCnt1: %.8X \n", device->minor,
        label, readPLX (device, DMA_CH_1_TRANS_BYTE_CNT));

    errmsg ("(%d - %s)                 DMADescrPtr1: %.8X \n", device->minor,
        label, readPLX (device, DMA_CH_1_DESC_PTR));

    errmsg ("(%d - %s)                DMALocalAddr1: %.8X \n", device->minor,
        label, readPLX (device, DMA_CH_1_LOCAL_ADDR));

    errmsg ("(%d - %s)            DMA_CH_1_PCI_ADDR: %.8X \n", device->minor,
        label, readPLX (device, DMA_CH_1_PCI_ADDR));

    errmsg ("(%d - %s)                 DMA status 1: %.8X \n\n", device->minor,
        label, readPLXb (device, DMA_CMD_STATUS_1));


    errmsg ("Intermediate buffer data:\n");

    for (q = 0; q <= 12; q++)

    {

#ifdef SUPPORT_READ
        printk ("%.8X ", device->dmaData[RX].sgPages[0][q]);

#endif /* 
        */
#ifdef SUPPORT_WRITE
        printk ("%.8X ", device->dmaData[TX].sgPages[0][q]);

#endif /* 
        */

    }

    errmsg ("\n----------------------------------\n");

}


//#endif

/************************************************************************/ 
/* timeout_handler                                                      */ 
/************************************************************************/ 
void
timeout_handler (unsigned long ptr) 
{
    struct device_board *device = (struct device_board *) ptr;

    int i;


    device->hardError = TRUE;

#ifdef SUPPORT_TIMER
    stopPolling (device);
#endif

    errmsg ("(%d): ERROR - device timeout \n", device->minor);

    regdump (device, "timeout ");


    errmsg ("\n plx int %.8X board control %.8X \n",

        readPLX (device, INT_CTRL_STATUS), readLocal (device,
        BOARD_CTRL_REG));

    errmsg ("DMA Mode %.8X\n", readPLX (device, DMA_CH_0_MODE));


    // wake up whatever was waiting.

    for (i = 0; i < EVENT_ARRAY_SIZE; i++)

    {

        errmsg ("(%d): ERROR - event %d status %d\n", device->minor, i,
            atomic_read (&device->irq_event_pending[i]));

        if (atomic_read (&device->irq_event_pending[i]))

        {

            errmsg ("(%d): ERROR - event %d timeout\n", device->minor, i);

            clearEventFlag (device, i);

#ifdef RTL
            rtl_sem_post (device->sem_list[i]);

#else /* 
            */
            wake_up_interruptible (&device->wq[i]);

#endif /* 
            */
        }

    }


    errmsg ("(%d): ...............End timeout handler................\n",
        device->minor);

}

#ifdef SUPPORT_TIMER
/************************************************************************/ 
/* Timer handler                                                    */ 
/************************************************************************/ 
void
hw_timer_handler (unsigned long ptr) 
{

    struct device_board *device = (struct device_board *) ptr;


    int currentEvent;


    //msg ("Timer()\n");

    currentEvent = EVENT_INIT_COMPLETE;

    if (atomic_read (&device->irq_event_pending[currentEvent])
        && (initializeDone (device)))

    {

        msg ("Init check\n");

        msg ("(%d): irq init active\n", device->minor);

        //mark(0x15141415);
        clearEventFlag (device, currentEvent);

        wake_up_interruptible (&device->wq[currentEvent]);

    }				// atomic read event

    currentEvent = EVENT_AUTOCAL_COMPLETE;

    if (atomic_read (&device->irq_event_pending[currentEvent])
        && (autocalDone (device)))

    {

        msg ("Autocal check\n");

        msg ("(%d): irq autocal active\n", device->minor);

        //mark(0x15141415);
        clearEventFlag (device, currentEvent);

        wake_up_interruptible (&device->wq[currentEvent]);

    }				// atomic read event

    currentEvent = EVENT_IN_BUFFER_LO_HI;
    if (atomic_read (&device->irq_event_pending[currentEvent])
        && (inputBufferLoHi (device)))

    {

        msg ("EVENT_IN_BUFFER_LO_HI check\n");

        msg ("(%d): irq EVENT_IN_BUFFER_LO_HI active\n", device->minor);

        //mark(0x15141415);
        clearEventFlag (device, currentEvent);

        device->eventDelta[currentEvent]=2/*jiffies - device->eventStart[currentEvent]*/;  
        wake_up_interruptible (&device->wq[currentEvent]);

    }				// atomic read event
#ifdef SUPPORT_WRITE
    currentEvent = EVENT_OUT_BUFFER_HI_LO;

    if (atomic_read (&device->irq_event_pending[currentEvent])
        && (inputBufferHiLo (device)))

    {

        msg ("EVENT_OUT_BUFFER_HI_LO check\n");

        msg ("(%d): irq EVENT_OUT_BUFFER_HI_LO active\n", device->minor);

        //mark(0x15141415);
        clearEventFlag (device, currentEvent);

        wake_up_interruptible (&device->wq[currentEvent]);

    }				// atomic read event
#endif /* 
                    */
    //msg(" (%d): end IRQ: %d PLX int/ctrl %.8X BC %.8X DMA status %.8X DMA mode %.8X\n", device->minor,device->dma_pending, readPLX(INT_STATUS_REG), readLocal(device,BOARD_CTRL_REG),readPLX(device,DMACmdStatus(device)),readPLX(device,DMAMode0(device)));

    if (device->timerEnabled)

    {

        device->hw_timer.expires = jiffies + TIMER_DELTA;

        add_timer (&device->hw_timer);

    }

    else

    {

        msg ("(%d): timer disable\n", device->minor);

        device->timerRunning = FALSE;

    }


    //msg("(%d): End timer handler\n", device->minor);
    //msg("timer end BCR %.8X\n",readLocal(device,BOARD_CTRL_REG));
    return;

}



#endif /* 
*/

/************************************************************************/ 
/* Interrupt handler                                                    */ 
/************************************************************************/ 
#ifdef RTL
extern struct device_board *boards;

unsigned int
device_interrupt (unsigned int irq, struct rtl_frame *regs) 
#elif LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,0)
irqreturn_t device_interrupt (int irq, void *dev_id, struct pt_regs * regs) 
#else /* 
*/
void
device_interrupt (int irq, void *dev_id, struct pt_regs *regs) 
#endif				/* 
*/
{

    struct device_board *device;

    u32 intStatReg;

    unsigned char dmaStatReg;

    int handled;

    int retval;

#ifdef SUPPORT_LOCAL_IRQ
    __u32 bcr_reg;

    __u32 local_irq;

#endif /* 
    */

#ifdef RTL
    device = boards;

    retval = 0;

#else /* 
    */
    device = (struct device_board *) dev_id;

    retval = IRQ_HANDLED;

#endif /* 
    */


#ifdef RTL
    //    rtl_stop_interrupts(); rtl_cprintf("took irq %d, lock!\n",irq); while(1);

    while (device)
    {

        if (device->pHardware->irqlevel == irq)

            break;

        device = device->next;

    }

    if (!device)
    {

        rtl_printf ("Error, no device found for irq %d\n", irq);

        rtl_hard_enable_irq (irq);

        return retval;

    }

#endif /* 
    */
    handled = FALSE;


    //msg(KERN_INFO GSC_NAME " Interrupt()\n");
    //mark(0x11100111);

    // determine if the interrupt is enabled.

    intStatReg = readPLX (device, INT_CTRL_STATUS);


    if (((intStatReg & IRQ_PCI_ENABLE) == 0) || 
        ((((intStatReg & IRQ_DMA_0_ACTIVE) == 0)
        && (intStatReg & IRQ_LOCAL_ACTIVE) == 0)
        && 
        (intStatReg & IRQ_DMA_1_ACTIVE) == 0))

    {				// not ours
        //msg(KERN_INFO GSC_NAME "Interrupt not ours... %.8X\n",intStatReg);
        //mark(0x12121212);
#ifdef TRACE_LOCAL
        int_other_count[device->board_index]++;

#endif /* 
        */
        goto OUT;

    }


    // check for DMA 0 (RX) complete interrupt
    dmaStatReg = readPLXb (device, DMA_CMD_STATUS_0);

    msg ("irq Dma 0 CmdStat: %.8X\n", dmaStatReg);

    msg ("IRQ_DMA_0_ACTIVE: %.8X\n", (intStatReg & IRQ_DMA_0_ACTIVE));

    //msg("Dma CmdStat: %.8X\n",dmaStatReg);
    if ((dmaStatReg & DMA_CHAN_DONE) && (dmaStatReg & DMA_CHAN_ENABLE)
        && ((intStatReg & IRQ_DMA_0_ACTIVE)))

    {

        //msg("(%d): IRQ Dma 0 in progress...\n",device->minor);
        // clear it.
        andPLXb (device, DMA_CMD_STATUS_0, ~DMA_CHAN_ENABLE);

        orPLXb (device, DMA_CMD_STATUS_0, DMA_CHAN_CLR_INT);

        andPLXLock (device, INT_CTRL_STATUS,
            ~(IRQ_DMA_0_ENABLE | IRQ_DMA_0_ACTIVE));

        //msg("Dma CmdStat after clear: %.8X\n",readPLX(device,DMA_CMD_STATUS));
        if (atomic_read (&device->irq_event_pending[EVENT_RX_DMA_PENDING]))

        {


            msg ("(%d): IRQ Dma 0 done...\n", device->minor);

            handled = TRUE;

            // ack interrupt
            // wake up process blocked in 'read()'
            clearEventFlag (device, EVENT_RX_DMA_PENDING);

#ifdef DEBUG
            dma_count[device->board_index]++;

#endif /* 
            */

#ifdef RTL
            rtl_sem_post (device->sem_list[EVENT_RX_DMA_PENDING]);

#else /* 
            */
            wake_up_interruptible (&device->wq[EVENT_RX_DMA_PENDING]);

#endif /* 
            */
            mark (0xbeef0001);


            //intStatReg = readPLX(device,INT_CTRL_STATUS);
            //dmaStatReg = readPLX(device,DMA_CMD_STATUS);
            //goto out;
        }

        else

        {

            mark (0xbeef1001);

            errmsg ("Unexpected DMA 0 irq cleared\n");

            DisableIrqPlx (device);

            //goto out;
        }

    }				// handling DMA channel 0
#ifdef USE_SECOND_DMA
    // check for DMA 1 (TX) complete interrupt
    intStatReg = readPLX (device, INT_CTRL_STATUS);

    dmaStatReg = readPLXb (device, DMA_CMD_STATUS_1);

    msg ("irq Dma 1 CmdStat: %.8X\n", dmaStatReg);

    //msg("DMA1_DONE: %.8X\n",(dmaStatReg & DMA1_DONE));
    msg ("IRQ_DMA_1_ACTIVE: %.8X\n", (intStatReg & IRQ_DMA_1_ACTIVE));


    if ((dmaStatReg & DMA_CHAN_DONE) && (dmaStatReg & DMA_CHAN_ENABLE)
        && ((intStatReg & IRQ_DMA_1_ACTIVE)))

    {

        mark (0xbeef0002);

        //msg("(%d): IRQ Dma 1 in progress...\n",device->minor);
        // clear it.
        andPLXLock (device, INT_CTRL_STATUS,
            ~(IRQ_DMA_1_ENABLE | IRQ_DMA_1_ACTIVE));

        orPLXb (device, DMA_CMD_STATUS_1, DMA_CHAN_CLR_INT);

        andPLXb (device, DMA_CMD_STATUS_1, ~DMA_CHAN_ENABLE);

        //msg("Dma 1 CmdStat after clear: %.8X\n",readPLX(device,DMA_CMD_STATUS));
        if (atomic_read (&device->irq_event_pending[EVENT_TX_DMA_PENDING]))

        {

            //msg("Release dma 1 lock\n");

            msg ("(%d): IRQ Dma 1 done...\n", device->minor);

            handled = TRUE;

            // acq interrupt
            // wake up process blocked in 'read()'
#ifdef DEBUG
            dma_count[device->board_index]++;

#endif /* 
            */
            clearEventFlag (device, EVENT_TX_DMA_PENDING);

#ifdef RTL
            rtl_sem_post (device->sem_list[EVENT_TX_DMA_PENDING]);

#else /* 
            */
            wake_up_interruptible (&device->wq[EVENT_TX_DMA_PENDING]);

#endif /* 
            */
            mark (0xbeef0003);

            //intStatReg = readPLX(device,INT_CTRL_STATUS);
            //dmaStatReg = readPLX(device,DMA_CMD_STATUS);
            //goto out;
        }

        else

        {

            mark (0xbeef2002);

            errmsg ("Unexpected DMA 1 irq cleared\n");

            DisableIrqPlx (device);

            //goto OUT;
        }

    }				// handling DMA channel 1
#endif /* 
                    */
#ifdef SUPPORT_LOCAL_IRQ
    // check for local interrupt 
    bcr_reg = readLocal (device, BOARD_CTRL_REG);

    msg ("(%d): Interrupt!, %.8X\n", device->minor, bcr_reg);


    DisableIrqPlx (device);

    DisableIrqLocalAll (device);


    mark (0x13131313);

    if (bcr_reg & BCR_INT_REQ_FLAG_MASK)
    {

        handled = FALSE;


        // see which kind of event is this.  The hardware is assumed to only have one
        // interrupt active at a time.
#ifdef TRACE_INTS
        int_count[device->board_index]++;

#endif /* 
        */
        local_irq = ((bcr_reg & BCR_INT_EVENT_MASK) >> BCR_INT_EVENT_SHIFT);


        msg ("(%d): local IRQ # %d\n", device->minor, local_irq);

        if (readEventFlag (device, local_irq))
        {

            msg ("(%d): local IRQ # %d expected...\n", device->minor,
                local_irq);

            mark (0x14141414);

#ifdef RTL
            rtl_sem_post (device->sem_list[local_irq]);

#else /* 
            */
            clearEventFlag (device, local_irq);

            wake_up_interruptible (&device->wq[local_irq]);

#endif /* 
            */
            handled = TRUE;

        }

        else

        {

            mark (0x15151515);

            errmsg ("(%d): disabled irq on device %p\n", device->minor,
                device);

        }

    }

#endif /* 
    */
    if (!handled)

    {

        mark (0x16161616);

        errmsg ("(%d): >>>>>>>>>>>>>Unhandled interrupt - disabling...\n",
            device->minor);

        device->hardError = TRUE;

        //DisableIrqLocalAll(device);
        //regdump(device,"IRQ not handled");
        DisableIrqPlx (device);


    }

    mark (0x22222222);

    msg ("(%d): End IRQ handler\n", device->minor);

OUT:
#ifdef RTL 
    rtl_hard_enable_irq (irq);

#endif /* 
    */
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,0)
    return retval;

#else /* 
    */
    return;

#endif /* 
    */
}


