/***
*** local_hw.c
***
***  General description of this file:
***     Device driver source code for General Standards 
***     family of analog I/O boards. This file is part of the Linux
***     driver source distribution for this board.
***
***  Copyrights (c):
***     General Standards Corporation (GSC), 2005-2006
***
***  Author:
***     Evan Hillman (evan@generalstandards.com)
***
***  Support:
***     Primary support for this driver is provided by GSC. 
***
***  Platform (tested on, may work with others):
***     Linux, kernel version 2.4.x, 2.6.x,  Red Hat distribution, Intel hardware.
***/

//////////////////////////////////////////////////////////////////////////
// set the following flag to trace interrupt debug messages.
#ifdef DEBUG
#define TRACE_LOCAL TRUE
#endif

#include "internals.h"

/************************************************************************/
/*                                                                      */
/************************************************************************/
int
setupEvent (struct device_board *device, int currentEvent)
{
  DisableIrqLocalAll (device);
#ifdef SUPPORT_TIMER
  msg ("SetupEvent event %d delta %ld\n", currentEvent, device->eventDelta[currentEvent]);
#endif
  device->timerEnabled = TRUE;
  switch (currentEvent)
    {
    case EVENT_AUTOCAL_COMPLETE:
      setEventFlag (device, currentEvent);
      msg ("BCR after clear local ints %.8X\n",
	   readLocal (device, BOARD_CTRL_REG));
      EnableIrqPlxLocal (device);
      andLocal (device, BOARD_CTRL_REG, ~(BCR_INT_EVENT_MASK));
      orLocalLock (device, BOARD_CTRL_REG,
		   currentEvent << BCR_INT_EVENT_SHIFT);

      msg ("autocal before BCR %.8X\n", readLocal (device, BOARD_CTRL_REG));

      orLocalLock (device, BOARD_CTRL_REG, BCR_AUTOCAL_START);
      msg ("autocal before BCR %.8X\n", readLocal (device, BOARD_CTRL_REG));
      break;

    case EVENT_IN_BUFFER_HI_LO:
      break;

    case EVENT_IN_BUFFER_LO_HI:
      setEventFlag (device, currentEvent);
      mark (0x44444444);
      EnableIrqPlxLocal (device);
      andLocal (device, BOARD_CTRL_REG, ~(BCR_INT_EVENT_MASK));
      orLocal (device, BOARD_CTRL_REG, currentEvent << BCR_INT_EVENT_SHIFT);
      break;

    case EVENT_INIT_COMPLETE:
      setEventFlag (device, currentEvent);
      EnableIrqPlxLocal (device);
      andLocal (device, BOARD_CTRL_REG, ~(BCR_INT_EVENT_MASK));
      orLocal (device, BOARD_CTRL_REG, currentEvent << BCR_INT_EVENT_SHIFT);
      orLocal (device, BOARD_CTRL_REG, BCR_INITIALIZE_MASK);
      break;

    case EVENT_CHANNELS_READY:
      setEventFlag (device, currentEvent);
      EnableIrqPlxLocal (device);
      andOrLocal (device, BOARD_CTRL_REG, ~(BCR_INT_EVENT_MASK),
		  (EVENT_CHANNELS_READY << BCR_INT_EVENT_SHIFT));
      break;

    default:
      errmsg ("Error - Invalid setup event state %d\n", currentEvent);
      device->hardError = TRUE;
      
return -1;
      break;
    };

#ifdef SUPPORT_TIMER
  device->eventStart[currentEvent]=jiffies;
  if ((device->eventDelta[currentEvent] > 3)/*&& (device->eventDelta[currentEvent] < 100)*/ )
  {
      unsigned long ticks = (device->eventDelta[currentEvent]-2);
      msg("timer delay of %ld jiffies current event %d\n",ticks, currentEvent);
      SET_CURRENT_STATE(TASK_UNINTERRUPTIBLE);
      if (schedule_timeout(ticks) > 0) // interrupted by signal
          return -1;
  }
  startPolling (device);
  //device->eventDelta[currentEvent] = 0;
#endif
  return 0;
}

/************************************************************************/
/*                                                                      */
/************************************************************************/
int
cleanupEvent (struct device_board *device, int currentEvent)
{
    msg ("CleanupEvent()\n");
#ifdef SUPPORT_TIMER
  //device->timerEnabled = FALSE;
    stopPolling(device);
    msg ("cleanupEvent event %d delta %ld\n", currentEvent, device->eventDelta[currentEvent]);
#endif

  switch (currentEvent)
    {
    case EVENT_AUTOCAL_COMPLETE:
      clearEventFlag (device, currentEvent);
#ifdef SUPPORT_LOCAL_IRQ
      andLocal (device, BOARD_CTRL_REG, ~(BCR_INT_REQ_FLAG_MASK));
      andLocal (device, BOARD_CTRL_REG,
		~(currentEvent << BCR_INT_EVENT_SHIFT));
#endif
      break;

    case EVENT_IN_BUFFER_HI_LO:
      clearEventFlag (device, currentEvent);
#ifdef SUPPORT_LOCAL_IRQ
      andLocal (device, BOARD_CTRL_REG, ~(BCR_INT_REQ_FLAG_MASK));
      andLocal (device, BOARD_CTRL_REG,
		~(currentEvent << BCR_INT_EVENT_SHIFT));
#endif
      break;

    case EVENT_IN_BUFFER_LO_HI:
      clearEventFlag (device, currentEvent);
#ifdef SUPPORT_LOCAL_IRQ
      andLocal (device, BOARD_CTRL_REG, ~(BCR_INT_REQ_FLAG_MASK));
      andLocal (device, BOARD_CTRL_REG,
		~(currentEvent << BCR_INT_EVENT_SHIFT));
#endif
      break;

    case EVENT_INIT_COMPLETE:
      clearEventFlag (device, currentEvent);
#ifdef SUPPORT_LOCAL_IRQ
      andLocal (device, BOARD_CTRL_REG, ~(BCR_INT_REQ_FLAG_MASK));
      andLocal (device, BOARD_CTRL_REG,
		~(currentEvent << BCR_INT_EVENT_SHIFT));
#endif
      break;

    case EVENT_CHANNELS_READY:
      clearEventFlag (device, currentEvent);
#ifdef SUPPORT_LOCAL_IRQ
      andLocal (device, BOARD_CTRL_REG, ~(BCR_INT_REQ_FLAG_MASK));
      andLocal (device, BOARD_CTRL_REG,
		~(currentEvent << BCR_INT_EVENT_SHIFT));
#endif
      break;

    default:
      errmsg ("Error - Invalid clear event state %d\n", currentEvent);
      device->hardError = TRUE;
      
return -1;
      break;
    };
  return 0;
}

#ifdef SUPPORT_READ
/************************************************************************/
/*                                                                      */
/************************************************************************/
int
getInBufferStatus (struct device_board *device)
{
  return readLocal (device, BOARD_CTRL_REG) & BCR_BUFF_THRES_FLAG_MASK;
}

/************************************************************************/
/*                                                                      */
/************************************************************************/
void
clearInputBufferOverflow (struct device_board *device)
{
  andLocalLock (device, BUFFER_CONTROL_REG, ~BUCR_BUFFER_OVERFLOW);
}
#endif

#ifdef SUPPORT_WRITE
/************************************************************************/
/*                                                                      */
/************************************************************************/
int
getOutBufferStatus (struct device_board *device)
{
  return readLocal (device,
		    OUT_BUFFER_SIZE_REG) > (readLocal (device,
						       OUT_BUFF_THRESHOLD_REG)
					    & OBTR_MASK);
}
#endif

/************************************************************************/
/*                                                                      */
/************************************************************************/
void
DisableIrqLocalAll (struct device_board *device)
{
  andLocal (device, BOARD_CTRL_REG,
	    ~(BCR_INT_EVENT_MASK | BCR_INT_REQ_FLAG_MASK));
}

/************************************************************************/
/*                                                                      */
/************************************************************************/
// Enable the "local" PLX interrupt
void
EnableIrqPlxLocal (struct device_board *device)
{
#ifdef SUPPORT_LOCAL_IRQ
  orPLXLock (device, INT_CTRL_STATUS, IRQ_PCI_ENABLE | IRQ_LOCAL_PCI_ENABLE);
#endif
}

/************************************************************************/
/*                                                                      */
/************************************************************************/
int
overflowDetected (struct device_board *device)
{
  return readLocal (device, BUFFER_CONTROL_REG) & BUCR_BUFFER_OVERFLOW;
}

/************************************************************************/
/*                                                                      */
/************************************************************************/
int
initializeDone (struct device_board *device)
{
  return !(readLocal (device, BOARD_CTRL_REG) & BCR_INITIALIZE_MASK);
}

/************************************************************************/
/*                                                                      */
/************************************************************************/
int
autocalDone (struct device_board *device)
{
  return (readLocal (device, BOARD_CTRL_REG) & BCR_AUTOCAL_PASS_MASK);
}

/************************************************************************/
/*                                                                      */
/************************************************************************/
int
inputBufferLoHi (struct device_board *device)
{
  return (readLocal (device, INPUT_BUFFER_SIZE_REG) >
	  (readLocal (device, BUFFER_CONTROL_REG) & BUCR_THRESHOLD_MASK));
}

#ifdef SUPPORT_WRITE
/************************************************************************/
/*                                                                      */
/************************************************************************/
int
outputBufferHiLo (struct device_board *device)
{
  return (readLocal (device, OUT_BUFFER_SIZE_REG) <=
	  (readLocal (device, OUT_BUFF_THRESHOLD_REG) & OBTR_MASK));
}
#endif
