/***
*** internals.h 
***
***
***  General description of this file:
***     Device driver source code for General Standards 
***     family of analog I/O boards. This file is part of the Linux
***     driver source distribution for this board.
***
***  Copyrights (c):
***     General Standards Corporation (GSC), 2005-2006
***
***  Author:
***     Evan Hillman (evan@generalstandards.com)
***
***  Support:
***     Primary support for this driver is provided by GSC. 
***
***  Platform (tested on, may work with others):
***     Linux, kernel version 2.4.x, 2.6.x,  Red Hat distribution, Intel hardware.
***/  
  
#ifndef INTERNALS_H
#define INTERNALS_H
  
//////////////////////////////////////////////////////////////////////////
// below here are things that are not likely to change:
  
// general driver defines that allow use of a generic code base.
// common include files
  
#define __NO_VERSION__
#include <linux/version.h>
#include <linux/module.h>
#include <linux/fs.h>
#include <linux/vmalloc.h>
#include <linux/pci.h>
#include <linux/mm.h>
#include <linux/proc_fs.h>
#include <linux/sched.h>
#include <linux/interrupt.h>
#include <linux/timer.h>
#include <linux/ioctl.h>
#include <linux/delay.h>
#include <linux/spinlock.h>
  
#ifndef UTS_RELEASE
#include <linux/utsrelease.h>
#endif /* 
 */
  
#include <asm/io.h>
#include <asm/system.h>
#include <asm/uaccess.h>
  
#include "sysdep.h"
#include "id.h"
#include "plx_regs.h"
  
#ifdef RTL
#include <rtl_posixio.h>
#include <rtl_semaphore.h>
#include <rtl_fcntl.h>
#include <rtl_unistd.h>
#include <rtl_stdio.h>
//#include <rtl_stlib.h>
#endif /* 
 */
  
#define PCI_NUM_BARS 6
  
#define GSC_SUBVENDOR PCI_VENDOR_ID_PLX

struct board_entry 
{
  
__u32 subsystem_vendor;
  
__u32 subsystem_device;
  
char name[40];
   
int index;
 
};


#undef msg             /* undef it, just in case */
#ifdef TRACE_LOCAL
#  ifdef __KERNEL__
/* This one if debugging is on, and kernel space */ 
#    define msg(fmt, args...) if (device) printk("%.6ld  ",(jiffies-device->startJiffies)*1000/(HZ)), printk( KERN_INFO GSC_NAME " " fmt, ## args)
#  else
/* This one for user space */ 
#    define msg(fmt, args...) fprintf(stderr, fmt, ## args)
#  endif
#else /* 
 */
#  define msg(fmt, args...) /* not debugging: nothing */
#endif /* 
 */
#    define errmsg(fmt, args...) if (device) printk("%ld  ",(jiffies-device->startJiffies)*1000/(HZ)), printk( KERN_ERR GSC_NAME " " fmt, ## args)
  
#if !defined(U64)
typedef union _U64 
{
  
struct 
  {
    
__u32 LowPart;
    
__u32 HighPart;
  
} 
u;
  

u64 QuadPart;
 
} U64;

#endif /* 
 */
  
// PCI BAR Space information
typedef struct _PCI_BAR_INFO 
{
  
__u32 * *pVa;		// BAR Kernel Virtual Address
  U64 Physical;			// BAR Physical Address
  __u32 Size;			// BAR size
  int IsIoMapped;		// Memory or I/O mapped?
} 
PCI_BAR_INFO;


// interrupt event condition codes
#define GROUP_0_EVENT                       0x00
#define GROUP_1_EVENT                       0x10
#define GROUP_2_EVENT                       0x20
  
//
// structure defining the elements of a scatter-gather link.
//
struct sgElement 
{
  
char *pciAddr;
   
__u32 localAddr;		// PLX data register
  __u32 size;
   
union
  {
    
struct sgElement *nextPtr;
     
__u32 nextU32;
   
} next;
 
};

//
// structure for one DMA channel.
//
struct dma_data_struct 
{
  
int dmaState;		// DMA mode.
   
ulong userOffset;		// current offset into user buffer.
  long userRemaining;		// space remaining in the user buffer.
  char *userBuffer;		// pointer to the current user buffer.
  long intermediateSamples;	// unread samples in the DMA buffer
  ulong intermediateStart;	// first unread sample in the DMA buffer
  long dmasamples;		// unread samples in the DMA buffer
  ulong dmastart;		// first unread sample in the DMA buffer
   
    //scatter-gather stuff
  struct sgElement *sgList;
   
void *sgListPhysical;
   
char *sgPages[SG_PAGES];
   
void *sgPagesPhysical[SG_PAGES];
   
int sgElementSize[SG_PAGES];
  

    // mmap stuff
  int vmas;
   
size_t dmaSize;
   
int useMmap;
   
    //end mmap
#ifdef RTL
  struct rtl_timespec t;
   
#else				/* 
 */
  struct timer_list watchdog_timer;
   
#endif				/* 
 */
};


// hardware descriptor structure
  
struct device_hardware
{
  
struct pci_dev *pdev;	/* the kernel PCI device descriptor */
   
__u32 * runtime_addr;	/* remapped runtime register address space */
   
__u32 * local_addr;		/* remapped local register address space */
   
int irqlevel;		/* interrupt level */
   
atomic_t usageCount;
   
struct semaphore hwSem;
   
struct semaphore dmaSem;
   
struct device_board *channelData[DEVICES_PER_BOARD];
   
int error;
   
PCI_BAR_INFO PciBar[PCI_NUM_BARS];	// PCI BARs information
};


//
// card descriptor structure
//
struct device_board 
{
  
struct device_hardware hw;
   
struct device_hardware *pHardware;
   
int localDeviceIndex;	// for hardware with more than one device per board.
  int timeout_seconds;		// time to wait before timeout (in seconds)
  int fillBuffer;		// Flag to determine if the user buffer should be filled before returning.
  int hardError;
  

struct dma_data_struct dmaData[2];
   
spinlock_t regLock;
   
int firstWrite;
  

#ifdef RTL
  struct rtl_file *fp;		// file pointer for user data.
#else				/* 
 */
  struct file *fp;		// file pointer for user data.
#endif				/* 
 */
   
int detectOverflow;
   
int detectUnderflow;
   
int busy;			// device opened
  //int irqlevel;                   // interrupt level
  int minor;			// device minor number
  volatile int timerEnabled;	// for orderly shutdown
  volatile int timerRunning;	// for orderly shutdown
  atomic_t irq_event_pending[EVENT_ARRAY_SIZE + 1];	// array of flags for what irq is active
  time_t eventStart[EVENT_ARRAY_SIZE + 1];
#ifdef SUPPORT_TIMER
  time_t eventDelta[EVENT_ARRAY_SIZE + 1];
#endif  

#ifdef RTL
    rtl_sem_t * sem_list[EVENT_ARRAY_SIZE + 1];
   
#elif LINUX_VERSION_CODE <= KERNEL_VERSION(2,4,0)
  struct wait_queue *wq[EVENT_ARRAY_SIZE + 1];	// queue of tasks sleeping in ioctl
#else				/* 
 */
    wait_queue_head_t wq[EVENT_ARRAY_SIZE + 1];	// queue of tasks sleeping in ioctl
#endif				/* 
 */
  int board_type;		// an index for which board type for drivers that support multiple types.
  int board_index;
  

struct semaphore hwSem;
   
#ifdef SUPPORT_TIMER
#ifndef RTL
  struct timer_list hw_timer;
   
#endif				/* 
 */
#endif				/* 
 */
    PCI_BAR_INFO PciBar[PCI_NUM_BARS];	// PCI BARs information
  struct device_board *next;	// next board in linked list
  time_t startJiffies;
  

#ifdef TRACE_LOCAL
  int nread;
   
int last;
   
#endif				/* 
 */
 
};				// card structure

// non-static function prototypes
  
#ifdef RTL
unsigned int device_interrupt (unsigned int irq, struct rtl_frame *regs);

#elif LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,0)
  irqreturn_t device_interrupt (int irq, void *dev_id, struct pt_regs *regs);

#else /* 
 */
void device_interrupt (int irq, void *dev_id, struct pt_regs *regs);

#endif /* 
 */
  
#ifdef RTL
#define waitEvent(device, event) rtl_sem_wait(device->sem_list[event])
#else /* 
 */
#define waitEvent(device, event) (wait_event_interruptible(device->wq[event],(!atomic_read(&device->irq_event_pending[event]))) != 0) /* return 0 on success instead of failure */
#endif /* 
 */
  
#define setEventFlag(device, event)   atomic_set(&device->irq_event_pending[event],TRUE)
#define clearEventFlag(device, event) atomic_set(&device->irq_event_pending[event],FALSE)
#define readEventFlag(device, event)  atomic_read(&device->irq_event_pending[event])

int setupEvent (struct device_board *device, int event);

int cleanupEvent (struct device_board *device, int event);


int setupWatchdog (struct device_board *device, int channel, int seconds);

int cleanupWatchdog (struct device_board *device, int channel);


int init_module (void);

void cleanup_module (void);


int read_proc (char *page, char **start, off_t offset, int count, int *eof,
		 void *data);

int proc_get_info (char *page, char **start, off_t offset, int count,
		    int dummy);


#ifdef RTL
int device_open (struct rtl_file *fp);

int device_close (struct rtl_file *fp);

rtl_ssize_t device_write (struct rtl_file *fp, const char *buf,
			   rtl_size_t size, rtl_off_t * lt);

rtl_ssize_t device_read (struct rtl_file *fp, char *buf, rtl_size_t size,
			  rtl_off_t * lt);

int device_ioctl (struct rtl_file *fp, unsigned int ioctl,
		   unsigned long arg);

#else /* 
 */
int device_open (struct inode *inode, struct file *fp);

int device_close (struct inode *inode, struct file *fp);

int device_write (struct file *fp, const char *buf, size_t size,
		   loff_t * lt);

int device_read (struct file *fp, char *buf, size_t size, loff_t * lt);

int device_ioctl (struct inode *inode, struct file *fp, unsigned int num,
		   unsigned long arg);

#endif /* 
 */

void timeout_handler (unsigned long ptr);

void hw_timer_handler (unsigned long ptr);

int doAutocal (struct device_board *device);

int doInit (struct device_board *device);


void DisableIrqPlx (struct device_board *device);

void EnableIrqPlxLocal (struct device_board *device);

void EnableIrqPlxDMA0 (struct device_board *device);

void EnableIrqPlxDMA1 (struct device_board *device);

void DisableIrqLocalAll (struct device_board *device);


void EnableIrqLocal (struct device_board *device, int group, __u32 irq);

void DisableIrqLocal (struct device_board *device, int group, __u32 irq);

long dmaRead (struct device_board *device);

void buildSgList (struct device_board *device, long size, int direction);

int createSglChain (struct device_board *device);

int waitForInputThreshold (struct device_board *device, int noblock);

int waitForOutputThreshold (struct device_board *device);

int waitForOutputReady (struct device_board *device);

long dmaWrite (struct device_board *device, long size);

int getInBufferStatus (struct device_board *device);

int getOutBufferStatus (struct device_board *device);

void clearInputBufferOverflow (struct device_board *device);

int overflowDetected (struct device_board *device);

int inputBufferLoHi (struct device_board *device);

int inputBufferHiLo (struct device_board *device);


void regdump (struct device_board *device, char *label);


int device_mmap (struct file *filp, struct vm_area_struct *vma);

int initializeDone (struct device_board *device);

int autocalDone (struct device_board *device);


// global data.  Only created during init.
extern __s8 built[32];

extern int num_boards;

extern int proc_enabled;

extern struct board_entry boards_supported[];

extern const char *event_name[];

extern const int channelOffset[];


#ifdef SUPPORT_TIMER
void startPolling (struct device_board *device);
void stopPolling (struct device_board *device);
#endif /* 
 */

#ifdef DEBUG
//debug variables
extern __u32 int_other_count[MAX_BOARDS];

extern __u32 int_count[MAX_BOARDS];

extern __u32 dma_count[MAX_BOARDS];

extern __u32 channel_irq[MAX_BOARDS];

extern __u32 channel_expected[MAX_BOARDS];

extern int board_type[MAX_BOARDS];

extern void *context[MAX_BOARDS];


// debug functions
void regdump (struct device_board *device, char *label);

#endif /* 
 */
  
//////////////////////////////////////////////////////////////////////////
// macro used to write to the mailbox register to trace execution with
// a bus analyzer.
#undef mark            /* undef it, just in case */
#ifdef DEBUG
#define mark(x) writePLX(device, MAILBOX_REGISTER_0, x)
#else /* 
 */
#define mark(x)
#endif /* 
 */
  
#include "inline.h"
  
#endif /* 
 */
