/***
*** inline.h 
***
***
***  General description of this file:
***     Device driver source code for General Standards 
***     family of analog I/O boards. This file is part of the Linux
***     driver source distribution for this board.
***
***  Copyrights (c):
***     General Standards Corporation (GSC), 2005-2006
***
***  Author:
***     Evan Hillman (evan@generalstandards.com)
***
***  Support:
***     Primary support for this driver is provided by GSC. 
***
***  Platform (tested on, may work with others):
***     Linux, kernel version 2.4.x, 2.6.x,  Red Hat distribution, Intel hardware.
***/  
  
#ifndef INLINE_H
#define INLINE_H
  

/************************************************************************/ 
/* readlocal                                                            */ 
/*                                                                      */ 
/* Read from the board local registers, whether the local registers     */ 
/* are in I/O space or memory space.                                    */ 
/*                                                                      */ 
/*                                                                      */ 
/************************************************************************/ 
  static __inline__ unsigned
readLocal (struct device_board *device, unsigned address) 
{
  
#ifdef SUPPORT_IO_SPACE
    if (device->PciBar[2].IsIoMapped)	/* i/o space */
    
    {
      
	//      printk(KERN_INFO GSC_NAME "readlocal I/O space addr: %.8X \n",address);
	return inl ((unsigned) device->local_addr + address * sizeof (__u32));
    
}
  
  else
    
#endif // SUPPORT_IO_SPACE
    {
      
return readl (device->pHardware->local_addr + address +
		     (channelOffset[device->localDeviceIndex]));
    
}

};



/************************************************************************/ 
/* writelocal                                                           */ 
/*                                                                      */ 
/* Write to the board local registers, whether the local registers      */ 
/* are in I/O space or memory space.                                    */ 
/*                                                                      */ 
/*                                                                      */ 
/************************************************************************/ 
void static __inline__
writeLocal (struct device_board *device, unsigned address, __u32 value) 
{
  
#ifdef SUPPORT_IO_SPACE
    if (device->PciBar[2].IsIoMapped)	/* i/o space */
    
    {
      
	//printk(KERN_INFO GSC_NAME "writelocal I/O space addr: %.8X value: %.8X.\n",address,value);
	outl (value,
	      (unsigned) device->local_addr + address * sizeof (__u32));
    
}
  
  else
    
#endif // SUPPORT_IO_SPACE
    {
      
writel (value,
	       device->pHardware->local_addr + address +
	       (channelOffset[device->localDeviceIndex]));
    
}

};



/************************************************************************/ 
/* andLocal                                                           */ 
/*                                                                      */ 
/* Write to the board local registers, whether the local registers      */ 
/* are in I/O space or memory space.                                    */ 
/*                                                                      */ 
/*                                                                      */ 
/************************************************************************/ 
void static __inline__
andLocal (struct device_board *device, unsigned address, __u32 value) 
{
  
u32 regval;
  

#ifdef SUPPORT_IO_SPACE
    if (device->PciBar[2].IsIoMapped)	/* i/o space */
    
    {
      
regval =
	inl ((unsigned) device->local_addr + address * sizeof (__u32));
      
regval &= value;
      
	//printk(KERN_INFO GSC_NAME "writelocal I/O space addr: %.8X value: %.8X.\n",address,value);
	outl (regval,
	      (unsigned) device->local_addr + address * sizeof (__u32));
    
}
  
  else
    
#endif // SUPPORT_IO_SPACE
    {
      
regval =
	readl (device->pHardware->local_addr + address +
	       (channelOffset[device->localDeviceIndex]));
      
regval &= value;
      
writel (regval,
	       device->pHardware->local_addr + address +
	       (channelOffset[device->localDeviceIndex]));
    
}

};



/************************************************************************/ 
void static __inline__
orLocal (struct device_board *device, unsigned address, __u32 value) 
{
  
__u32 regval;
  

#ifdef SUPPORT_IO_SPACE
    if (device->PciBar[2].IsIoMapped)	/* i/o space */
    
    {
      
regval =
	inl ((unsigned) device->local_addr + address * sizeof (__u32));
      
regval |= value;
      
	//printk(KERN_INFO GSC_NAME "writelocal I/O space addr: %.8X value: %.8X.\n",address,value);
	outl (regval,
	      (unsigned) device->local_addr + address * sizeof (__u32));
    
}
  
  else
    
#endif // SUPPORT_IO_SPACE
    {
      
regval =
	readl (device->pHardware->local_addr + address +
	       (channelOffset[device->localDeviceIndex]));
      
regval |= value;
      
writel (regval,
	       device->pHardware->local_addr + address +
	       (channelOffset[device->localDeviceIndex]));
    
}

};


/************************************************************************/ 
void static __inline__
andOrLocal (struct device_board *device, unsigned address, __u32 andValue,
	    __u32 orValue) 
{
  
u32 regval;
  

#ifdef SUPPORT_IO_SPACE
    if (device->PciBar[2].IsIoMapped)	/* i/o space */
    
    {
      
regval =
	inl ((unsigned) device->local_addr + address * sizeof (__u32));
      
regval &= andValue;
      
regval |= orValue;
      
	//printk(KERN_INFO GSC_NAME "writelocal I/O space addr: %.8X value: %.8X.\n",address,value);
	outl (regval,
	      (unsigned) device->local_addr + address * sizeof (__u32));
    
}
  
  else
    
#endif // SUPPORT_IO_SPACE
    {
      
regval =
	readl (device->pHardware->local_addr + address +
	       (channelOffset[device->localDeviceIndex]));
      
regval &= andValue;
      
regval |= orValue;
      
writel (regval,
	       device->pHardware->local_addr + address +
	       (channelOffset[device->localDeviceIndex]));
    
}

};



/************************************************************************/ 
/* writelocal                                                           */ 
/*                                                                      */ 
/* Write to the board local registers, whether the local registers      */ 
/* are in I/O space or memory space.                                    */ 
/*                                                                      */ 
/*                                                                      */ 
/************************************************************************/ 
void static __inline__
writeLocalLock (struct device_board *device, unsigned address, __u32 value) 
{
  
unsigned long flags;
  

spin_lock_irqsave (&device->regLock, flags);
  
#ifdef SUPPORT_IO_SPACE
    if (device->PciBar[2].IsIoMapped)	/* i/o space */
    
    {
      
	//printk(KERN_INFO GSC_NAME "writelocal I/O space addr: %.8X value: %.8X.\n",address,value);
	outl (value,
	      (unsigned) device->local_addr + address * sizeof (__u32));
    
}
  
  else
    
#endif // SUPPORT_IO_SPACE
    {
      
writel (value,
	       device->pHardware->local_addr + address +
	       (channelOffset[device->localDeviceIndex]));
    
}
  
spin_unlock_irqrestore (&device->regLock, flags);

};



/************************************************************************/ 
/* andLocal                                                           */ 
/*                                                                      */ 
/* Write to the board local registers, whether the local registers      */ 
/* are in I/O space or memory space.                                    */ 
/*                                                                      */ 
/*                                                                      */ 
/************************************************************************/ 
void static __inline__
andLocalLock (struct device_board *device, unsigned address, __u32 value) 
{
  
unsigned long flags;
  
unsigned long regval;
  

spin_lock_irqsave (&device->regLock, flags);
  
#ifdef SUPPORT_IO_SPACE
    if (device->PciBar[2].IsIoMapped)	/* i/o space */
    
    {
      
regval =
	inl ((unsigned) device->local_addr + address * sizeof (__u32));
      
regval &= value;
      
	//printk(KERN_INFO GSC_NAME "writelocal I/O space addr: %.8X value: %.8X.\n",address,value);
	outl (regval,
	      (unsigned) device->local_addr + address * sizeof (__u32));
    
}
  
  else
    
#endif // SUPPORT_IO_SPACE
    {
      
regval =
	readl (device->pHardware->local_addr + address +
	       (channelOffset[device->localDeviceIndex]));
      
regval &= value;
      
writel (regval,
	       device->pHardware->local_addr + address +
	       (channelOffset[device->localDeviceIndex]));
    
}
  
spin_unlock_irqrestore (&device->regLock, flags);

};



/************************************************************************/ 
void static __inline__
orLocalLock (struct device_board *device, unsigned address, __u32 value) 
{
  
unsigned long flags;
  
unsigned long regval;
  

spin_lock_irqsave (&device->regLock, flags);
  
#ifdef SUPPORT_IO_SPACE
    if (device->PciBar[2].IsIoMapped)	/* i/o space */
    
    {
      
regval =
	inl ((unsigned) device->local_addr + address * sizeof (__u32));
      
regval |= value;
      
	//printk(KERN_INFO GSC_NAME "writelocal I/O space addr: %.8X value: %.8X.\n",address,value);
	outl (regval,
	      (unsigned) device->local_addr + address * sizeof (__u32));
    
}
  
  else
    
#endif // SUPPORT_IO_SPACE
    {
      
regval =
	readl (device->pHardware->local_addr + address +
	       (channelOffset[device->localDeviceIndex]));
      
regval |= value;
      
writel (regval,
	       device->pHardware->local_addr + address +
	       (channelOffset[device->localDeviceIndex]));
    
}
  
spin_unlock_irqrestore (&device->regLock, flags);

};


/************************************************************************/ 
/* writePLX                                                             */ 
/*                                                                      */ 
/* Write to the board local registers, whether the local registers      */ 
/* are in I/O space or memory space.                                    */ 
/*                                                                      */ 
/*                                                                      */ 
/************************************************************************/ 
  static __inline__ void
writePLXLock (struct device_board *device, unsigned address,
	      unsigned long value) 
{
  
unsigned long flags;
  

    //msg("writePLX base addr: %p offset: %x reg: %x value: %.8lx\n", device->pHardware->runtime_addr,address,address*sizeof(__u32),value);
    
spin_lock_irqsave (&device->regLock, flags);
  
writel (value, device->pHardware->runtime_addr + address);
  
spin_unlock_irqrestore (&device->regLock, flags);

};



/************************************************************************/ 
/* andPLX                                                             */ 
/*                                                                      */ 
/* Write to the board local registers, whether the local registers      */ 
/* are in I/O space or memory space.                                    */ 
/*                                                                      */ 
/*                                                                      */ 
/************************************************************************/ 
void static __inline__
andPLXLock (struct device_board *device, unsigned address,
	    unsigned long value) 
{
  
unsigned long flags;
  
unsigned long regval;
  

    //msg("andPLX base addr: %p offset: %x reg: %x value: %.8lX\n", device->runtime_addr,address,address*sizeof(__u32),value);
    spin_lock_irqsave (&device->regLock, flags);
  
regval = readl (device->pHardware->runtime_addr + address);
  
regval &= value;
  
writel (regval, device->pHardware->runtime_addr + address);
  
spin_unlock_irqrestore (&device->regLock, flags);

};



/************************************************************************/ 
/* orPLX                                                                */ 
/*                                                                      */ 
/* Write to the board local registers, whether the local registers      */ 
/* are in I/O space or memory space.                                    */ 
/*                                                                      */ 
/*                                                                      */ 
/************************************************************************/ 
void static __inline__
orPLXLock (struct device_board *device, unsigned address,
	   unsigned long value) 
{
  
unsigned long flags;
  
unsigned long regval;
  

    //msg("orPLX base addr: %p offset: %x reg: %x value: %.8lX\n", device->runtime_addr,address,address*sizeof(__u32),value);
    spin_lock_irqsave (&device->regLock, flags);
  
regval = readl (device->pHardware->runtime_addr + address);
  
regval |= value;
  
writel (regval, device->pHardware->runtime_addr + address);
  
spin_unlock_irqrestore (&device->regLock, flags);

};


#endif /* 
 */
  
/************************************************************************/ 
/* writePLX                                                             */ 
/*                                                                      */ 
/* Write to the board local registers, whether the local registers      */ 
/* are in I/O space or memory space.                                    */ 
/*                                                                      */ 
/*                                                                      */ 
/************************************************************************/ 
  static __inline__ void
writePLX (struct device_board *device, unsigned address,
	  unsigned long value) 
{
  
writel (value, device->pHardware->runtime_addr + address);

};



/************************************************************************/ 
/* andPLX                                                             */ 
/*                                                                      */ 
/* Write to the board local registers, whether the local registers      */ 
/* are in I/O space or memory space.                                    */ 
/*                                                                      */ 
/*                                                                      */ 
/************************************************************************/ 
void static __inline__
andPLX (struct device_board *device, unsigned address, unsigned long value) 
{
  
u32 regval;
  

regval = readl (device->pHardware->runtime_addr + address);
  
regval &= value;
  
writel (regval, device->pHardware->runtime_addr + address);

};



/************************************************************************/ 
/* orPLX                                                                */ 
/*                                                                      */ 
/* Write to the board local registers, whether the local registers      */ 
/* are in I/O space or memory space.                                    */ 
/*                                                                      */ 
/*                                                                      */ 
/************************************************************************/ 
void static __inline__
orPLX (struct device_board *device, unsigned address, unsigned long value) 
{
  
u32 regval;
  

regval = readl (device->pHardware->runtime_addr + address);
  
regval |= value;
  
writel (regval, device->pHardware->runtime_addr + address);

};



/************************************************************************/ 
/* readPLX                                                             */ 
/*                                                                      */ 
/* Write to the board local registers, whether the local registers      */ 
/* are in I/O space or memory space.                                    */ 
/*                                                                      */ 
/*                                                                      */ 
/************************************************************************/ 
__u32 static __inline__
readPLX (struct device_board *device, unsigned address) 
{
  
return readl (device->pHardware->runtime_addr + address);

};



/************************************************************************/ 
/* writePLXb                                                             */ 
/*                                                                      */ 
/* Write to the board local registers, whether the local registers      */ 
/* are in I/O space or memory space.                                    */ 
/*                                                                      */ 
/*                                                                      */ 
/************************************************************************/ 
  static __inline__ void
writePLXb (struct device_board *device, unsigned address,
	   unsigned char value) 
{
  
    //    unsigned long flags;
    
    //msg("writePLX base addr: %p offset: %x reg: %x value: %.8lx\n", device->runtime_addr,address,address*sizeof(__u32),value);
    
    //    spin_lock_irqsave(&device->regLock,flags);
    writeb (value,
	    (unsigned char *) device->pHardware->runtime_addr + address);
  
    //    spin_unlock_irqrestore(&device->regLock,flags);
};



/************************************************************************/ 
/* readPLXb                                                             */ 
/*                                                                      */ 
/* Write to the board local registers, whether the local registers      */ 
/* are in I/O space or memory space.                                    */ 
/*                                                                      */ 
/*                                                                      */ 
/************************************************************************/ 
unsigned char static __inline__
readPLXb (struct device_board *device, unsigned address) 
{
  
return readb ((unsigned char *) device->pHardware->runtime_addr + address);

};



/************************************************************************/ 
/* andPLXb                                                             */ 
/*                                                                      */ 
/* Write to the board local registers, whether the local registers      */ 
/* are in I/O space or memory space.                                    */ 
/*                                                                      */ 
/*                                                                      */ 
/************************************************************************/ 
void static __inline__
andPLXb (struct device_board *device, unsigned address, unsigned char value) 
{
  
unsigned long regval;
  

regval =
    readb ((unsigned char *) device->pHardware->runtime_addr + address);
  
regval &= value;
  
writeb (regval,
	   (unsigned char *) device->pHardware->runtime_addr + address);

};



/************************************************************************/ 
/* orPLXb                                                                */ 
/*                                                                      */ 
/* Write to the board local registers, whether the local registers      */ 
/* are in I/O space or memory space.                                    */ 
/*                                                                      */ 
/*                                                                      */ 
/************************************************************************/ 
void static __inline__
orPLXb (struct device_board *device, unsigned address, unsigned char value) 
{
  
unsigned long regval;
  

    //msg("orPLX base addr: %p offset: %x reg: %x value: %.8lX\n", device->runtime_addr,address,address*sizeof(__u32),value);
    regval =
    readb ((unsigned char *) device->pHardware->runtime_addr + address);
  
regval |= value;
  
writeb (regval,
	   (unsigned char *) device->pHardware->runtime_addr + address);

};



