/***
*** init.c  
***
***  General description of this file:
***     Device driver source code for General Standards 
***     family of analog I/O boards. This file is part of the Linux
***     driver source distribution for this board.
***
***  Copyrights (c):
***     General Standards Corporation (GSC), 2005-2006
***
***  Author:
***     Evan Hillman (evan@generalstandards.com)
***
***  Support:
***     Primary support for this driver is provided by GSC. 
***
***  Platform (tested on, may work with others):
***     Linux, kernel version 2.4.x, 2.6.x,  Red Hat distribution, Intel hardware.
***/  
  
//////////////////////////////////////////////////////////////////////////
// set the following flag to trace debug messages in this module.
#ifdef DEBUG
#define TRACE_LOCAL TRUE
#endif /* 
 */
  
#ifdef RTL
#define RETURN_BUSY RTL_EBUSY
#else /* 
 */
#define RETURN_BUSY EBUSY
#endif /* 
 */
  
#include "internals.h"
  
MODULE_DESCRIPTION (GSC_NAME);

MODULE_AUTHOR ("Evan Hillman (evan@generalstandards.com)");

MODULE_LICENSE ("GPL");


// Driver Version
const char device_version[] = "$Date: " __DATE__ " " __TIME__ " $";


// board models supported
struct board_entry boards_supported[] = 
  { 
{GSC_SUBVENDOR, GSC_DEV_ID, GSC_NAME, 0}, 
{0, 0, "NULL", 0}, /* null terminated list.... */ 
};



// debug info
#ifdef DEBUG
  u32 int_other_count[MAX_BOARDS];

u32 int_count[MAX_BOARDS];

u32 dma_count[MAX_BOARDS];

u32 channel_irq[MAX_BOARDS];

u32 channel_expected[MAX_BOARDS];

int board_type[MAX_BOARDS];

void *context[MAX_BOARDS];

#endif /* 
 */
// local module data
#ifdef RTL
static struct rtl_file_operations device_fops = 
#else /* 
 */
static struct file_operations device_fops = 
#endif /* 
 */
{ 
open: device_open, 
release: device_close, 
read: device_read, 
write: device_write, 
ioctl:device_ioctl,
  
#ifndef RTL
mmap:device_mmap, 
#endif /* 
 */
};



#ifndef RTL
static int device_major = 0;	/* device major number (dynamically assigned) */

#endif /* 
 */
static struct device_board *boards = 0;	/* linked list of all boards found */


__s8 built[32];

int num_boards;

int proc_enabled;


/************************************************************************/ 
/* module initialization: detect card(s) and set up descriptor(s)        */ 
/************************************************************************/ 
#ifdef RTL
  int
main (int argc, char **argv) 
#else /* 
 */
  int
init_module (void) 
#endif				/* 
 */
{
  
#ifdef RTL
  int i;
  
#else /* 
 */
  struct proc_dir_entry *proc;
  
#endif /* 
 */
  
struct device_board *device = NULL;
  
struct device_board *devicenext;
  
struct pci_dev *pdev = NULL;
  

int found;
  
int channel;
  
int index;
  
int boardIndex;
  
int checkIndex;
  
struct device_hardware *pHw = NULL;
  

#ifdef DEBUG
    printk (KERN_INFO GSC_NAME
	    "<><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><>\n");
  
#endif /* 
 */
    
sprintf (built, "%s, %s ", __DATE__, __TIME__);
  
printk (KERN_INFO GSC_NAME
	   ": driver (version: %s) built: %s loading on kernel %s\n",
	   DRIVER_VERSION, built, UTS_RELEASE);
  
printk (KERN_INFO GSC_NAME ": Copyright (C) " COPYRIGHT_YEAR
	   " General Standards Corp. \n");
  
boardIndex = 0;
  

#if LINUX_VERSION_CODE < KERNEL_VERSION(2,5,74)
    PCI_DEVICE_LOOP (pdev) 
  {
    
#else /* 
 */
    while ((pdev = pci_find_device (PCI_VENDOR_ID_PLX, PCI_ANY_ID, pdev)))
    
    {
      
#endif /* 
 */
	msg
	(": Checking...vendor: %X device: %X subvendor: %X subsystem: %X\n",
	 pdev->vendor, pdev->device, pdev->subsystem_vendor,
	 pdev->subsystem_device);
      
if ((pdev->vendor == PCI_VENDOR_ID_PLX) && (pdev->device == PLX_ID))
	
	{
	  

	    /* determine if this is one of the boards supported. */ 
	    
found = 0;
	  
index = 0;
	  
checkIndex = 0;
	  
while (boards_supported[checkIndex].subsystem_device != 0)
	    
	    {
	      
if ((boards_supported[checkIndex].subsystem_device ==
		    pdev->subsystem_device)
		   && 
(boards_supported[checkIndex].subsystem_vendor ==
			pdev->subsystem_vendor))
		
		{
		  
found = 1;
		  
		    //msg(":     found board %s type %d\n",boards_supported[i].name,i);
		}
	      
if (found)
		break;
	      
checkIndex++;
	    
}
	  

if (found)
	    
	    {
	      
for (channel = 0; channel < DEVICES_PER_BOARD; channel++)
		
		{
		  
printk (KERN_INFO GSC_NAME
			   ":     installing board %s type %d, channel %d\n",
			   boards_supported[checkIndex].name, checkIndex,
			   channel);
		  
		    //pci_read_config_word(pdev, 0x2E, &reg);
		    //msg(":     config reg=0x%x\n", reg);
		    //msg(":     attaching board #%d\n", index + 1);
		    device =
		    (struct device_board *)
		    kmalloc (sizeof (struct device_board), GFP_KERNEL);
		  
memset (device, 0, sizeof (device));
		  
device->board_type = checkIndex;
		  
device->localDeviceIndex = channel;
		  

#ifdef DEBUG
		    device->startJiffies = jiffies;
		  
#endif /* 
 */
		    
if (channel == 0)
		    
		    {
		      
pHw = &device->hw;
		      
atomic_set (&pHw->usageCount, 0);
		      
if (pci_enable_device (pdev))
			
			{
			  
errmsg
			    (" ERROR - Unable to get interrupt routing\n");
			  
kfree (device);
			  
return (-ENODEV);
			
}
		      
device->board_type = checkIndex;
		      
		      {
			
int ResourceCount = 0;
			

for (index = 0; index < PCI_NUM_BARS; ++index)
			  
			  {
			    
			      // Verify the address is valid
			      if (pci_resource_start (pdev, index) == 0)
			      
			      {
				
continue;
			      
}
			    

			      //msg("   Resource %02d\n", ResourceCount);
			      
			      // Increment resource count
			      ResourceCount++;
			    
			      // Get PCI physical address
			      pHw->PciBar[index].Physical.QuadPart =
			      pci_resource_start (pdev, index);
			    

			      // Determine resource type
			      if (pci_resource_flags (pdev, index) &
				  IORESOURCE_IO)
			      
			      {
				
				  //msg("     Type     : I/O Port\n");
				  
				  // Make sure flags are cleared properly
				  pHw->PciBar[index].Physical.QuadPart &=
				  ~(0x3);
				
pHw->PciBar[index].IsIoMapped = TRUE;
			      
}
			    
			    else
			      
			      {
				
				  //msg("     Type     : Memory Space\n");
				  
				  // Make sure flags are cleared properly
				  pHw->PciBar[index].Physical.QuadPart &=
				  ~(0xf);
				
pHw->PciBar[index].IsIoMapped = FALSE;
			      
}
			    

			      //msg("     Address  : %08x\n",device->PciBar[i].Physical.u.LowPart);
			      
			      // Get the size
			      pHw->PciBar[index].Size = 4096;
			    
			      /*Plx_pci_resource_len(device, i); */ 
			  }
		      
}
		      

#if LINUX_VERSION_CODE <= KERNEL_VERSION(2,4,0)
			pHw->runtime_addr =
			(u32 *) ioremap (pdev->base_address[0], 4096);
		      
if (pHw->PciBar[2].IsIoMapped)
			
			{
			  
			    //device->local_addr = pdev->resource[2];
			    msg (": I/O mapped local registers\n");
			  
pHw->local_addr = base_address[2];
			
}
		      
		      else
			
			{
			  
pHw->local_addr =
			    (u32 *) ioremap (pdev->resource[2].start, 4096);
			
}
		      
#else /* 
 */
			if (pHw->PciBar[2].IsIoMapped)
			
			{
			  
//#ifdef SUPPORT_IO_SPACE              
//                pHw->local_addr = (u32 *)pdev->resource[2].start;
//#endif
			}
		      
		      else
			
			{
			  
pHw->local_addr =
			    (u32 *) ioremap (pdev->resource[2].start, 4096);
			
}
		      
pHw->runtime_addr =
			(u32 *) ioremap (pdev->resource[0].start,
					 pci_resource_len (pdev, 0));
		      
#endif /* 
 */
			pHw->pdev = pdev;
		      
pci_set_master (pdev);
		      
pHw->irqlevel = pdev->irq;
		      
sema_init (&pHw->hwSem, 1);
		      
sema_init (&pHw->dmaSem, 1);
		    
}		// if (channel ==0)
		  
pHw->channelData[channel] = device;
		  
pHw->error = FALSE;
		  
device->next = boards;
		  
device->fillBuffer = FALSE;
		  
device->busy = 0;
		  
device->minor = boardIndex;
		  
device->next = boards;
		  
#ifdef DEBUG
		    msg (":     base_address[0]=0x%X, base_address[2]=0x%X\n",
			 (unsigned) pdev->resource[0].start,
			 (unsigned) pdev->resource[2].start);
		  
		    //msg(":     local_addr=%p, runtime_addr=%p irq=%d\n", device->local_addr, device->runtime_addr, device->irqlevel);
#endif /* 
 */
		    boards = device;
		  
device->board_index = index;
		  
device->pHardware = pHw;
		  
#ifdef TRACE_LOCAL
		    context[index] = device;
		  
board_type[index] = device->board_type;
		  
		    //readPLX(device,INT_CTRL_STATUS);
		    //readPLX(device,DMA_CMD_STATUS);
#endif /* 
 */
		    boardIndex++;
		  
num_boards++;
		  
#ifdef RTL
		  {
		    
char buf[64];
		    
rtl_sprintf (buf, "/dev/gs_ad_%d", index);
		    
rtl_register_dev (buf, &device_fops, index);
		  
} 
#endif /* 
 */
	    } 
}
	
}
    
} 
msg ("Checking board index\n");
  
if (boardIndex == 0)
    
    {
      
errmsg (": no board found\n");
      
return (-ENODEV);
    
}
  
#ifndef RTL
    device_major = register_chrdev (0, GSC_NAME, &device_fops);
  

if (device_major < 0)
    
    {
      
	/* OK, could not register -- undo the work we have done above */ 
	errmsg (": could not register device number\n");
      
for (device = boards; device; device = devicenext)
	
	{
	  
devicenext = device->next;
	  
iounmap (device->hw.local_addr);
	  
iounmap (device->hw.runtime_addr);
	  
pci_disable_device (device->pHardware->pdev);
	  
kfree (device);
	
}
      
boards = NULL;
      
return (-ENODEV);
    
}
  
#endif /* 
 */
    /*
     *   Add /proc file system support.
     */ 
    
#ifndef RTL
    msg ("Creating proc entry...\n");
  
if (num_boards)
    
    {
      
remove_proc_entry (GSC_NAME, NULL);	/* Just in case. */
      
proc = create_proc_entry (GSC_NAME, S_IRUGO, NULL);
      

if (proc)
	
	{
	  
proc_enabled = 1;
	  
proc->read_proc = read_proc;
	  
proc->get_info = (void *) proc_get_info;
	
}
      
      else
	
	{
	  
errmsg ("<1>%s: create_proc_entry() failure.\n", GSC_NAME);
	  
cleanup_module ();
	
}
    
}
  
#endif /* 
 */
#ifdef TRACE_LOCAL
    msg (": major=%d\n", device_major);
  
  {
    
int j;
    
for (j = 0; j < MAX_BOARDS; j++)
      
      {
	
int_other_count[j] = 0;
	
int_count[j] = 0;
	
dma_count[j] = 0;
	
channel_irq[j] = 0;
      
}
  
}
  
#endif /* 
 */
    msg ("init_module success\n");
  
#ifdef RTL
    rtl_main_wait ();
  
#else /* 
 */
    return 0;

}



/************************************************************************/ 
/* cleanup when unloading module                                        */ 
/************************************************************************/ 
  void
cleanup_module (void) 
{
  
struct device_board *device, *devicenext;
  
unregister_chrdev (device_major, GSC_NAME);
  
#endif /* 
 */
    for (device = boards; device; device = devicenext)
    
    {
      
#ifdef RTL
      char buf[64];
      
rtl_sprintf (buf, "/dev/gs_ad_%d", i);
      
rtl_unregister_dev (buf);
      
#endif /* 
 */
	devicenext = device->next;
      
if ((device->pHardware->pdev) && (0 != device->hw.local_addr))
	
	{
	  
#ifdef RTL
	    rtl_free_irq (device->pHardware->irqlevel);
	  
#else /* 
 */
	    pci_disable_device (device->pHardware->pdev);
	  
#endif /* 
 */
	    device->pHardware->pdev = 0;
	
}
      

#ifdef SUPPORT_TIMER
	stopPolling (device);
#endif /* 
 */
      if (0 != device->hw.local_addr)
	
	{
	  
iounmap (device->hw.local_addr);
	  
iounmap (device->hw.runtime_addr);
	  
device->hw.local_addr = 0;
	  
device->hw.runtime_addr = 0;
	
}
      
kfree (device);
    
}
  
boards = NULL;
  
#ifdef RTL
    return 0;
  
#else /* 
 */
    if (proc_enabled)
    
    {
      
remove_proc_entry (GSC_NAME, NULL);
      
proc_enabled = 0;
    
}
  
#endif /* 
 */
    //msg(": unloaded\n");
}



/************************************************************************/ 
/* open device                                                          */ 
/************************************************************************/ 
#ifdef RTL
  int
device_open (struct rtl_file *fp) 
#else /* 
 */
  int
device_open (struct inode *inode, struct file *fp) 
#endif				/* 
 */
{
  
struct device_board *device;
  
int i;
  
__u32 retval;
  

for (device = boards; device; device = device->next)
    
    {
      
msg ("(%d): checking board %p %d %d\n", device->minor, device,
	    MINOR (inode->i_rdev), device->minor);
      
#ifdef RTL
	if (rtl_inodes[fp->devs_index].priv == device->minor)
	
#else /* 
 */
	if (MINOR (inode->i_rdev) == device->minor)
	
#endif /* 
 */
	{
	  
if (device->busy)
	    
	    {
	      
msg ("(%d): board already opened\n", device->minor);
	      
return (-RETURN_BUSY);
	    
}
	  
msg ("(%d): opening board\n", device->minor);
	  
atomic_inc (&device->pHardware->usageCount);
	  

if (atomic_read (&device->pHardware->usageCount) == 1)	// first open for this board
	    {
	      
#ifdef RTL
		if (rtl_request_irq
		    (device->pHardware->irqlevel,
		     (void *) device_interrupt) < 0)
		
#else /* 
 */
		if (request_irq
		    (device->pHardware->irqlevel, (void *) device_interrupt,
		     SA_SHIRQ, GSC_NAME, device) < 0)
		
#endif /* 
 */
		{
		  
errmsg ("(%d): cannot get interrupt %d\n", device->minor,
			   device->pHardware->irqlevel);
		  
device->busy = 0;
		  
return (-RETURN_BUSY);
		
}
	      
msg ("(%d): before timer init\n", device->minor);
	      
#ifdef SUPPORT_TIMER
		init_timer (&device->hw_timer);
	      
device->hw_timer.function = hw_timer_handler;
	      
device->hw_timer.data = (unsigned long) device;
	      
device->hw_timer.expires = jiffies + TIMER_DELTA;
	      
device->timerRunning = FALSE;
	      
device->timerEnabled = FALSE;
for (i=0;i<EVENT_ARRAY_SIZE ;i++)
{
    device->eventStart[i]=jiffies;
    device->eventDelta[i]=0;
}

#endif /* 
 */
	    }
	  
if ((retval = createSglChain (device)) < 0)
	    
	    {
	      
return retval;
	    
};
	  

device->timeout_seconds = 20;
	  

#ifdef DEBUG
//      device->nread=0;
//      device->last=-1;
	    device->startJiffies = jiffies;
	  
#endif /* 
 */
	    device->busy = 1;
	  
device->firstWrite = TRUE;
	  
memset (device->irq_event_pending, FALSE,
		   sizeof (device->irq_event_pending));
	  

#ifdef RTL
	    rtl_hard_disable_irq (device->pHardware->irqlevel);
	  
for (i = 0; i < EVENT_ARRAY_SIZE; i++)
	    
	    {
	      
rtl_sem_init (device->sem_list[i], 1, 0);
	    
}
	  

#elif LINUX_VERSION_CODE <= KERNEL_VERSION(2,4,0)
	    for (i = 0; i < EVENT_ARRAY_SIZE; i++)
	    
	    {
	      
device->wq[i] = NULL;
	    
}
	  
#else /* 
 */
	    for (i = 0; i < EVENT_ARRAY_SIZE; i++)
	    
	    {
	      
init_waitqueue_head (&device->wq[i]);
	    
}
	  
#endif /* 
 */
	    
sema_init (&device->hwSem, 1);
	  

#ifdef RTL
	    fp->f_priv = (unsigned long) device;
	  
#else /* 
 */
	    fp->private_data = device;
	  
#endif /* 
 */
	    spin_lock_init (&device->regLock);
	  
device->hardError = FALSE;
	  

writePLXb (device, DMA_CMD_STATUS_0, 0);
	  
writePLXb (device, DMA_CMD_STATUS_1, 0);
	  

writePLX (device, DMA_MODE_ARBITRATION, 0);
	  
	    //writePLX(device,DMA_THRESHOLD_REG,0);
	    
msg ("(%d): device_open done\n", device->minor);
	  
MOD_INC_USE_COUNT;
	  
return (0);
	
}
    
}
  
#ifndef RTL
    errmsg ("(%d): ERROR - unable to find board\n", MINOR (inode->i_rdev));
  
#endif /* 
 */
    return (-ENODEV);

}				// device_open


/************************************************************************/ 
/* close device                                                         */ 
/************************************************************************/ 
#ifdef RTL
  int
device_close (struct rtl_file *fp) 
#else /* 
 */
  int
device_close (struct inode *inode, struct file *fp) 
#endif				/* 
 */
{
  
struct device_board *device;
  
#ifdef SUPPORT_TIMER
  unsigned long jStart;
  
#endif /* 
 */
    
#ifdef RTL 
    device = (struct device_board *) fp->f_priv;
  
#else /* 
 */
    device = (struct device_board *) fp->private_data;
  
#endif /* 
 */
    
#ifdef SUPPORT_TIMER
    stopPolling (device);
  
msg ("(%d): closing board\n", device->minor);
  
device->timerEnabled = FALSE;
  
#endif /* 
 */
    msg ("(%d): closing board\n", device->minor);
  

atomic_dec (&device->pHardware->usageCount);
  
if (atomic_read (&device->pHardware->usageCount) == 0)
    
    {
      
DisableIrqPlx (device);
      

	// reset DMA engines
	writePLXb (device, DMA_CMD_STATUS_0, 0);
      
writePLXb (device, DMA_CMD_STATUS_1, 0);
      

	// free resources
	//free_irq(device->pHardware->irqlevel, device);
	//    free_pages((u32)device->dmaData[RX].intermediateBuffer, DMA_ORDER);
	//    free_pages((u32)device->dmaData[TX].intermediateBuffer, DMA_ORDER);
#ifdef SUPPORT_TIMER
	jStart = jiffies;
      
while ((device->timerRunning) && (jiffies < jStart + 10))
	
	{
	  
schedule ();
	
}
      
#endif /* 
 */
    }
  
device->busy = 0;
  
MOD_DEC_USE_COUNT;
  
return (0);

}



//module_init(init_module);
//module_exit(cleanup_module);
  
//EXPORT_SYMBOL(built);
