                                                                                                  /***
*** hardware.c
***
***  General description of this file:
***     Device driver source code for General Standards 
***     family of analog I/O boards. This file is part of the Linux
***     driver source distribution for this board.
***
***  Copyrights (c):
***     General Standards Corporation (GSC), 2005-2006
***
***  Author:
***     Evan Hillman (evan@generalstandards.com)
***
***  Support:
***     Primary support for this driver is provided by GSC. 
***
***  Platform (tested on, may work with others):
***     Linux, kernel version 2.4.x, 2.6.x,  Red Hat distribution, Intel hardware.
***/

//////////////////////////////////////////////////////////////////////////
// set the following flag to trace interrupt debug messages.
#ifdef DEBUG
#define TRACE_LOCAL TRUE
#endif

#include "internals.h"

const int channelOffset[] = {
    0x00 / 4,
        0x40 / 4,
        0x80 / 4,
        0xC0 / 4
};

#ifdef SUPPORT_TIMER
/************************************************************************/
/*                                                                      */
/************************************************************************/
void
startPolling (struct device_board *device) 
{
    msg("Polling started\n");
    device->hw_timer.expires = jiffies + TIMER_DELTA;

    device->timerRunning = TRUE;

    device->timerEnabled = TRUE;

    add_timer (&device->hw_timer);

    msg ("Starting polling\n");

} 

/************************************************************************/
/*                                                                      */
/************************************************************************/
void
stopPolling (struct device_board *device) 
{

    del_timer_sync (&device->hw_timer);

    device->timerRunning = FALSE;

    device->timerEnabled = FALSE;

    msg ("Stopping polling\n");

} 
#endif /* 
*/
/************************************************************************/
/*                                                                      */
/************************************************************************/
int
setupWatchdog (struct device_board *device, int channel, int seconds)
{
#ifdef RTL
    rtl_clock_gettime (RTL_CLOCK_REALTIME, &device->dmaData[channel].t);
    device->dmaData[channel].t.tv_sec += device->timeout_seconds;
#else
    device->dmaData[RX].watchdog_timer.expires = jiffies + seconds * HZ;
    add_timer (&device->dmaData[channel].watchdog_timer);
    mark (0x33333333);
#endif
    return 0;
}

/************************************************************************/
/*                                                                      */
/************************************************************************/
int
cleanupWatchdog (struct device_board *device, int channel)
{
#ifdef RTL
    return 0;
#else
    if (timer_pending (&device->dmaData[channel].watchdog_timer))
    {
        del_timer_sync (&device->dmaData[channel].watchdog_timer);
    }
    if (signal_pending (current) || device->hardError)
    {
        errmsg ("Signal/timeout error waiting for more data\n");
        DisableIrqPlx (device);
        return (-ERESTARTSYS);
    }
    msg ("Cleanup watchdog returning 0\n");
    return 0;
#endif
}

/************************************************************************/
/*                                                                      */
/************************************************************************/
// Disable all interrupts from the PLX
void
DisableIrqPlx (struct device_board *device)
{
    andPLXLock (device, INT_CTRL_STATUS,
        ~(IRQ_PCI_ENABLE | IRQ_DMA_0_ENABLE | IRQ_DMA_1_ENABLE |
        IRQ_LOCAL_PCI_ENABLE));
}

/************************************************************************/
/*                                                                      */
/************************************************************************/
// Enable the PLX DMA channel 0 interrupt.
void
EnableIrqPlxDMA0 (struct device_board *device)
{
    orPLXLock (device, INT_CTRL_STATUS, IRQ_PCI_ENABLE | IRQ_DMA_0_ENABLE);
}

/************************************************************************/
/*                                                                      */
/************************************************************************/
void
EnableIrqPlxDMA1 (struct device_board *device)
{
    orPLXLock (device, INT_CTRL_STATUS, IRQ_PCI_ENABLE | IRQ_DMA_1_ENABLE);
}

#ifdef SUPPORT_READ
//////////////////////////////////////////////////////////////////////////
int
waitForInputThreshold (struct device_board *device, int noblock)
{
    int currentEvent;
    u32 bufferStatus;
    int retval;

    retval = 0;
    msg ("waitForInputThreshold()\n");
    currentEvent = EVENT_IN_BUFFER_LO_HI;
    setupEvent (device, currentEvent);
    mark (0x44444444);

    bufferStatus = getInBufferStatus (device);

    if (bufferStatus == 0)	// need more
    {
        // first of all, check for non-blocking I/O.
        if (noblock)
        {
            msg ("(%d): waitForInputThreshold - return error noblock\n",
                device->minor);
            return (-EAGAIN);
        }

        msg
            ("(%d): FilldmaData[RX].intermediateBuffer DMA/PIO wait for more data... input level %.8X\n",
            device->minor, readLocal (device, INPUT_BUFFER_SIZE_REG));
        //msg("BCR %.8X current event: %d PSR: %.8X\n",readLocal(device,BOARD_CTRL_REG),currentEvent,readLocal(device,PRIMARY_STATUS_REG));
        setupWatchdog (device, RX, device->timeout_seconds);
        retval = waitEvent (device, currentEvent);

        retval |= cleanupWatchdog (device, RX);

    }
    cleanupEvent (device, currentEvent);
    if (retval != 0)
    {
        errmsg ("(%d): waitForInputThreshold - timeout/signal...\n",
            device->minor);
        //msg(" (%d): In progress: PLX int/ctrl %.8X BC %.8X \n", device->minor, readPLX(device, INT_CTRL_STATUS), readLocal(device,BOARD_CTRL_REG));
        return (retval);
    }

    mark (0x55555555);
    // if still no data present, something is wrong.
    msg ("waitForInputThreshold Success\n");
    return retval;		// no error
}
#endif

#ifdef SUPPORT_WRITE
//////////////////////////////////////////////////////////////////////////
int
waitForOutputThreshold (struct device_board *device)
{
    int currentEvent;
    int retval;

    // wait for space in the output buffer.

    currentEvent = EVENT_OUT_BUFFER_HI_LO;
    setupEvent (device, currentEvent);
    msg
        ("Before wait for write out buffer, Buffer size: %.8X threshold: %.8X \n",
        readLocal (device, OUT_BUFFER_SIZE_REG), readLocal (device,
        OUT_BUFF_THRESHOLD_REG));
    if (getOutBufferStatus (device))	//over threshold level
    {
        setupWatchdog (device, TX, device->timeout_seconds);
        msg
            ("device_write about to wait STATUS_OUT_BUFFER_HI_LO, buffer level: %.8X\n",
            readLocal (device, OUT_BUFFER_SIZE_REG));
        retval = waitEvent (device, currentEvent);
        retval |= cleanupWatchdog (device, TX);

    }
    removeEvent (device, currentEvent);
    if (retval != 0)
    {
        errmsg ("(%d): timeout during wait for write buffer low\n",
            device->minor);
        DisableIrqPlx (device);
        return (retval);
    }
    msg ("After wait for write out buffer, Buffer size: %.8X threshold: %.8X\n",
        readLocal (device, OUT_BUFFER_SIZE_REG), readLocal (device,
        OUT_BUFF_THRESHOLD_REG));
    return retval;
}
#endif

#ifdef SUPPORT_WRITE
//////////////////////////////////////////////////////////////////////////
int
waitForOutputReady (struct device_board *device)
{
    int currentEvent;
    __u32 ops_reg;
    int retval;

    currentEvent = EVENT_OUT_BURST_READY;
    setupEvent (device, currentEvent);

    ops_reg = readLocal (device, OUT_BUFF_OPS_REG);
    if (ops_reg & OOR_CIRCULAR_BUFFER)
    {
        msg ("Write wait for circular buffer\n");

        setupWatchdog (device, TX, device->timeout_seconds);
        msg ("device_write about to wait for load ready\n");
        retval = waitEvent (device, currentEvent);

        retval |= cleanupWatchdog (device, TX);
        mark (66666663);

        msg ("device_write after wait for load ready\n");
    }
    removeEvent (device, currentEvent);
    if (retval != 0)
    {
        errmsg ("(%d): timeout during wait for burst\n", device->minor);
        DisableIrqPlx (device);
        return (-EIO);
    }

    return 0;
}
#endif

#ifdef SUPPORT_READ
//////////////////////////////////////////////////////////////////////////
int
doAutocal (struct device_board *device)
{
    int currentEvent;
    int retval;

    retval = 0;
    currentEvent = EVENT_AUTOCAL_COMPLETE;

    setupEvent (device, currentEvent);
    setupWatchdog (device, RX, AUTOCAL_TIMEOUT_SECONDS);
    retval = waitEvent (device, currentEvent);
    retval |= cleanupWatchdog (device, RX);
    cleanupEvent (device, currentEvent);

    if (retval |= 0)
    {
        msg ("(%d): timeout when calibrating board\n", device->minor);
        retval = (-EIO);
        return retval;
    }
    msg ("(%d): ioctl autocal done\n", device->minor);

    return retval;
}
#endif

#ifdef SUPPORT_READ
//////////////////////////////////////////////////////////////////////////
int
doInit (struct device_board *device)
{
    int currentEvent;
    int retval;

    retval = 0;
    currentEvent = EVENT_INIT_COMPLETE;
    msg ("doInit current event %d\n", currentEvent);
    setupEvent (device, currentEvent);
    setupWatchdog (device, RX, INIT_TIMEOUT_SECONDS);
    retval = waitEvent (device, currentEvent);
    cleanupEvent (device, currentEvent);
    retval |= cleanupWatchdog (device, RX);
    if (retval != 0)
    {
        errmsg ("(%d): timeout during board initialize\n", device->minor);
        retval = (-EIO);
    }
    clearInputBufferOverflow (device);
    //msg("%d): %.8X %.8X %.8X\n",device->minor, readl(IntCntrlStat(device)), readLocal(device,BOARD_CTRL_REG),readl(DMACmdStatus(device)));

    msg ("(%d): doInit return %X\n", device->minor, retval);
    return retval;
}
#endif
