#ifndef __GSC_IOCTL_H_INCLUDED__
#define __GSC_IOCTL_H_INCLUDED__

/*! \file gsc24dsi32pll_ioctl.h
    \brief Driver header file.
*
*  pci24dsi32pll_ioctl.h 
*
*  General description of this file:
*     IOCTL code and structure definitions for General Standards 24DSI32 PLL 
*     Delta-Sigma A/D board. This file is part of the Linux
*     driver distribution for this board.
*     
*     This file is necessary to compile application programs, as well as the driver.
*
*  Copyrights (c):
*     General Standards Corporation (GSC), 2002-2007
*
*  Author:
*     Evan Hillman (evan@generalstandards.com)
*
*  Support:
*     Primary support for this driver is provided by GSC. 
*
*  Platform (tested on, may work with others):
*     Linux, kernel version 2.4.x,2.6.x Red Hat distribution, Intel hardware.
*
*/ 
  
/// \file /hillman/doc/header.txt
/*! 
* \author Evan Hillman
*
* \mainpage  The nice index page
* all of this goes on the mainpage.
*
* \section intro_sec Introduction
* all of the intro stuff.
*
* \subsection all about it
* more of the intro stuff.
*
*/ 
/// \page page1 boilerplate
  

/// Size of the scatter-gather buffer, in bytes.
  
/// Sets the size of the scatter-gather DMA buffer.  Setting this equal to the hardware
/// data buffer should be about right.  The hardware manual specifies the buffer
/// size in samples, not bytes, so for convenience the driver expects this value
/// to be in kilo-samples.  This value is used during compile, so the driver must be
/// recompiled before a change to this value has any effect.
  
#define SG_BUFFER_K_SAMPLES  256 /* as in 256 K samples */
  
// the rest don't have to be changed...
#define SG_BUFFER_SAMPLES   ((long)SG_BUFFER_K_SAMPLES*1024) /* convert to samples */
#define SG_BUFFER_BYTES     SG_BUFFER_SAMPLES*sizeof(__u32)
#define SG_BUFFER_K_BYTES   SG_BUFFER_BYTES/1024
#define SG_PAGES            SG_BUFFER_BYTES/PAGE_SIZE
#define PAGE_SAMPLES        PAGE_SIZE/sizeof(__u32)
  
#ifndef TRUE
#define TRUE 1
#endif /* 
 */
#ifndef FALSE
#define FALSE 0
#endif /* 
 */
  
#ifdef RTL
#include <rtl_stdint.h>
//#include <cam_ioctl.h>
#endif /* 
 */
  
#define IOCTL_GSC                           's'
  
/////////////////////////
// IOCTL control codes //
/////////////////////////
  
///////////////////////////////////////////////////////////////////////////////////////////////////////////
///
/// IOCTL_NO_COMMAND 
///
/// Parameter: NONE
///
/// IOCTL_NO_COMMAND is used for development testing.  Not required for 
/// normal use.  Places an image of the registers in the system log
#define IOCTL_GSC_NO_COMMAND                _IO(IOCTL_GSC,   1)
////
  
/////////////////////////////////////////////////////////////////////////////////////////////////////////////
/// IOCTL_READ_REGISTER
/// IOCTL_WRITE_REGISTER
///
/// Parameter: DEVICE_REGISTER_PARAMS *pRegParams;
///
/// This structure is used to store information about a register. The
/// IOCTL_READ_REGISTER and IOCTL_WRITE_REGISTER and other register
/// ioctl commands use this
/// structure to read a particular register and to write a value into a particular
/// register. 'eRegister' stores the index of the register, range being 0-94, and
/// 'ulValue' stores the register value, range being 0x0-0xFFFFFFFF.
////
#define IOCTL_GSC_READ_REGISTER             _IOWR(IOCTL_GSC, 2,struct device_register_params)
#define IOCTL_GSC_WRITE_REGISTER            _IOW(IOCTL_GSC,  3,struct device_register_params)

typedef struct device_register_params 
{
  
__u32 ulRegister;		// range: 0-94, see definitions below
  __u32 ulValue;		// range: 0x0-0xFFFFFFFF, same register values below
} 
DEVICE_REGISTER_PARAMS, *PDEVICE_REGISTER_PARAMS;


/* ==== Register Offsets ==== */ 
/* ---- GSC (local) Registers ---- */ 
#define BOARD_CTRL_REG          0
#define NREF_PLL_CONTROL_REG    1
#define NVCO_PLL_CONTROL_REG    2
#define RATE_ASSIGN_REG         3
#define RATE_DIVISOR_REG        4
#define RESERVED_1              5
#define PLL_REF_FREQ_REG        6
#define RANGE_FILTER_CONTROL    7
#define BUFFER_CONTROL_REG      8
#define BOARD_CONFIG_REG        9
#define BUFFER_SIZE_REG         10
#define AUTOCAL_VALUES_REG      11
#define INPUT_DATA_BUFFER_REG   12
#define LAST_REG                12
  
// aliased names to support a more universal code base.
  
#define INPUT_BUFFER_SIZE_REG   BUFFER_SIZE_REG
#define INPUT_DATA_BUFF_REG     INPUT_DATA_BUFFER_REG
  
//////////////////////////////////////////////////////////////////////////////////////////////////////////////
/* Register definitions.  Normally these are not used from a user application.                              */ 
//////////////////////////////////////////////////////////////////////////////////////////////////////////////
  
/* ---- Board Control Register (BCR). ---- */ 
  
// analog input mode
  
#define BCR_AIM_MASK                 0x0003
#define BCR_AIM_SHIFT                0
#define BCR_AIM_DIFFERENTIAL_MODE    0
#define BCR_AIM_ZERO_TEST            2
#define BCR_AIM_VREF_TEST            3
  
// analog input range
  
#define BCR_RANGE_MASK               0x0003
#define BCR_RANGE_SHIFT              2
#define BCR_RANGE_2_5                0
#define BCR_RANGE_5                  2
#define BCR_RANGE_10                 3
  
// two's complement/offset binary
  
#define BCR_OFFSET_BINARY_MASK       (1<<4)
#define BCR_OFFSET_BINARY_SHIFT      4
  
// initiator/target mode for external clock
  
#define BCR_INITIATOR_MASK           (1<<5)
#define BCR_INITIATOR_SHIFT          5
  
// start an ADC sync operation
  
#define BCR_SW_SYNC_MASK             (1<<6)
  
// start an autocal cycle
  
#define BCR_AUTOCAL_START             (1<<7)
  
// interrupt event masks
  
#define BCR_INT_EVENT_SHIFT          8
#define BCR_INT_EVENT_MASK           (7<<BCR_INT_EVENT_SHIFT)
  
#define BCR_IRQ_AUTOCAL
  
#define BCR_INT_REQ_FLAG_MASK        (1<<11)
#define BCR_AUTOCAL_PASS_MASK        (1<<12)
#define BCR_CHAN_READY_MASK          (1<<13)
  
#define BCR_BUFF_THRES_FLAG_MASK     (1<<14)
  
#define BCR_INITIALIZE_MASK          (1<<15)
  
#define BCR_SYNCHRONIZE_SCAN         (1<<16)
#define BCR_CLEAR_BUFFER_ON_SYNC     (1<<17)
#define BCR_RATE_GEN_EXT_CLK_OUT     (1<<18)
#define BCR_SELECT_IMAGE_FILTER      (1<<19)
  
/* ---- The masks for the GSC Rate Assignments Register. ---- */ 
#define RA_GROUP_MASK               0xF
#define RA_GROUP_0_SHIFT            0
#define RA_GROUP_1_SHIFT            4
#define RA_GROUP_2_SHIFT            8
#define RA_GROUP_3_SHIFT            12

static const int GroupShift[] = 
  { 
RA_GROUP_0_SHIFT, 
RA_GROUP_1_SHIFT, 
RA_GROUP_2_SHIFT,
  
RA_GROUP_3_SHIFT, 
};



// these assignments are for group zero, but apply to all
// channels.
  
#define RA_GROUP_0_INTERNAL   0
#define RA_GROUP_0_EXTERNAL   4
#define RA_GROUP_0_DIRECT     5
#define RA_GROUP_0_DISABLED   6
#define RA_LAST               RA_GROUP_0_DISABLED
  
// these assignments are for groups 1-3.  These groups
// can only be set to either enabled or disabled.
  
#define RA_GROUP_ENABLED    0
#define RA_GROUP_DISABLED   6
  
/* ---- The masks for the Rate Control Register. ---- */ 
#define RC_RATE_CONTROL_MAX        0x01FF
  
/* ---- The masks for the GSC data buffer registers. ---- */ 
  
#define IDR_16_BIT_MASK 0xFFFF
#define IDR_18_BIT_MASK 0x3FFFF
#define IDR_20_BIT_MASK 0xFFFFF
#define IDR_24_BIT_MASK 0xFFFFFF
  
#define IDR_CHANNEL_TAG_MASK (0x1F << 24)
#define IDR_CHANNEL_TAG_SHIFT  24
#define IDR_DATA_MASK (0xFFFFFF)
  
/* ---- The masks for the GSC Rate Divisor Registers. ---- */ 
/* ---- The masks for the GSC Rate Divisor Registers. ---- */ 
#define RD_GROUP_MASK                   0xFF
#define RD_GROUP_0_SHIFT                0
#define RD_GROUP_1_SHIFT                8
#define RD_GROUP_2_SHIFT                16
#define RD_GROUP_3_SHIFT                24
  
#define RD_MIN_DIVISOR_VALUE            0

static const int DivisorShift[] = 
  { 
RD_GROUP_0_SHIFT, 
RD_GROUP_1_SHIFT, 
RD_GROUP_2_SHIFT,
  
RD_GROUP_3_SHIFT 
};



#define RD_MAX_DIVISOR_VALUE            0xFF
  
/* ---- Buffer Control Register masks. ---- */ 
  
#define BUCR_THRESHOLD_MASK         0x0003FFFF
#define BUCR_DISABLE_BUFFER_MASK    (1<<18)
#define BUCR_DISABLE_BUFFER_SHIFT   18
#define BUCR_CLEAR_BUFFER_MASK      (1<<19)
  
#define BUCR_DATA_WIDTH_SHIFT       20
#define BUCR_DATA_WIDTH_MASK        (3<<BUCR_DATA_WIDTH_SHIFT)
#define BUCR_DATA_WIDTH_16          0
#define BUCR_DATA_WIDTH_18          1
#define BUCR_DATA_WIDTH_20          2
#define BUCR_DATA_WIDTH_24          3
  
#define BUCR_BUFFER_OVERFLOW        (1<<24)
#define BUCR_BUFFER_UNDERFLOW       (1<<25)
  
/* ---- NVCO Register masks. ---- */ 
  
#define NVCO_MIN 30
#define NVCO_MAX 1000
  
/* ---- NREF Register masks. ---- */ 
  
#define NREF_MIN 30
#define NREF_MAX 1000
  
/* ---- Legacy Rate Control Register masks. ---- */ 
  
#define LAGACY_RATE_MIN 0
#define LEGACY_RATE_MAX 511
  
////////////////////////////////////////////////////////////////////////////////////////////////////////////
///  IOCTL_SET_INPUT_RANGE
///
///  Parameter: unsigned long *pInputRange;
///  Range: 0-3, see codes below
///  Return: none
#define IOCTL_GSC_SET_INPUT_RANGE           _IOW(IOCTL_GSC,  5,unsigned long)
////
  
/* ---- input range codes ---- */ 
#define RANGE_2p5V                          BCR_RANGE_2_5
#define RANGE_5V                            BCR_RANGE_5
#define RANGE_10V                           BCR_RANGE_10
  
////////////////////////////////////////////////////////////////////////////////////////////////////////////
/// IOCTL_SET_INPUT_MODE
///
/// Parameter: unsigned long *pInputMode;
/// Range: 0-3, see codes below
#define IOCTL_GSC_SET_INPUT_MODE            _IOW(IOCTL_GSC,  6,unsigned long)
////
  
/* ---- input mode codes ---- */ 
#define MODE_DIFFERENTIAL                   BCR_AIM_DIFFERENTIAL_MODE
#define MODE_ZERO_TEST                      BCR_AIM_ZERO_TEST
#define MODE_VREF_TEST                      BCR_AIM_VREF_TEST
  
/////////////////////////////////////////////////////////////////////////////////////////////////////////////
///  IOCTL_SET_SW_SYNC
///
/// Initiate an ADC SYNCH operation.  Also generates the external sync output if the board is in initiator mode.
///
///  Parameter: NONE
#define IOCTL_GSC_SET_SW_SYNC               _IO(IOCTL_GSC,   7)
////
  
/////////////////////////////////////////////////////////////////////////////////////////////////////////////
///  IOCTL_AUTO_CAL
///
/// Initiate an auto-calibration cycle.  Check the hardware manual for what settings should be make before 
/// running an autocalibration cycle.
/// 
///  Parameter: NONE
#define IOCTL_GSC_AUTO_CAL                  _IO(IOCTL_GSC,   8)
////
  
/////////////////////////////////////////////////////////////////////////////////////////////////////////////
///  IOCTL_INITIALIZE
///
/// Initialize the board to a known state.  The initialize operation sets all hardware settings to their 
/// defaults.  The driver waits for an interrupt from the hardware indicating that the initialization cycle 
/// is complete.
///
///  Parameter: NONE
#define IOCTL_GSC_INITIALIZE                _IO(IOCTL_GSC,   9)
////
  
/////////////////////////////////////////////////////////////////////////////////////////////////////////////
/// IOCTL_SET_DATA_FORMAT
///
/// Set the digital data output format.  
///
/// Parameter: unsigned long *pDataFormat;
/// Range: 0-1, see codes below
#define IOCTL_GSC_SET_DATA_FORMAT           _IOW(IOCTL_GSC, 10,unsigned long)
////
  
/* ---- data format codes ---- */ 
#define FORMAT_TWOS_COMPLEMENT              0
#define FORMAT_OFFSET_BINARY                1
  
/////////////////////////////////////////////////////////////////////////////////////////////////////////////
/// IOCTL_SET_INITIATOR_MODE
///
/// Set this board as the initiator for synchronized acquisition. 
///
/// Parameter: unsigned long *pInitTarget;
/// Range: 0-1, see codes below
#define IOCTL_GSC_SET_INITIATOR_MODE        _IOW(IOCTL_GSC, 11,unsigned long)
////
  
/* ---- initiator mode codes ---- */ 
#define TARGET_MODE                         0
#define INITIATOR_MODE                      1
  
/////////////////////////////////////////////////////////////////////////////////////////////////////////////
/// IOCTL_SET_BUFFER_THRESHOLD
///
/// Set the data buffer threshold register.  The threshold value is used to allow the driver to sleep while 
/// waiting for sufficient data for a transfer to the user buffer.  
///
/// Parameter: unsigned long *pulThreshold;
/// Range: 0x0 - 0x3FFFF
#define IOCTL_GSC_SET_BUFFER_THRESHOLD      _IOW(IOCTL_GSC, 12,unsigned long)
////
#define INPUT_BUFFER_SIZE BUCR_THRESHOLD_MASK
  
/////////////////////////////////////////////////////////////////////////////////////////////////////////////
///  IOCTL_CLEAR_BUFFER
///
/// Clear any residual data from the data buffer.  This command does not halt sampling.  For the most 
/// consistent results, use IOCTL_GSC_SET_ACQUIRE_MODE to halt sampling before clearing the buffer.
///
///  Parameter: NONE
#define IOCTL_GSC_CLEAR_BUFFER              _IO(IOCTL_GSC,  13)
////
  
/////////////////////////////////////////////////////////////////////////////////////////////////////////////
/// IOCTL_SET_ACQUIRE_MODE
///
/// Set the hardware to either start or stop acquiring data.  Possible values are:
/// 
/// Parameter: unsigned long *pAcquireMode;
/// Range: 0-1, see codes below
#define IOCTL_GSC_SET_ACQUIRE_MODE          _IOW(IOCTL_GSC, 14,unsigned long)
////
  
/* ---- acquire mode codes ---- */ 
#define START_ACQUIRE 0
#define STOP_ACQUIRE  1
  
/////////////////////////////////////////////////////////////////////////////////////////////////////////////
/// IOCTL_ASSIGN_RATE_GROUP
///
/// Parameter: GEN_ASSIGN_PARAMS *pGenAssign;
///
/// This structure is used to assign a generator to a group consisting of four channels.
///  'eGroup' contains  the channel group, range
/// being 0-3, and 'eGenAssign' specifies the generator, range being 0-5.
#define IOCTL_GSC_ASSIGN_RATE_GROUP         _IOR(IOCTL_GSC, 16,struct gen_assign_params)
////
  typedef struct gen_assign_params
{
  
__u32 eGroup;		/* Range: 0-3, see codes below */
  
__u32 eGenAssign;		/* Range: 0-6, see codes below */

} GEN_ASSIGN_PARAMS, *PGEN_ASSIGN_PARAMS;



/* ---- channel group codes ---- */ 
#define GRP_0                   0
#define GRP_1                   1
#define GRP_2                   2
#define GRP_3                   3
  
/* ---- generator assignment codes ---- */ 
  
#define ASN_RATE_INTERNAL    RA_RATE_INTERNAL
#define ASN_EXTERNAL         RA_EXTERNAL
#define ASN_DIRECT_EXTERNAL  RA_DIRECT_EXTERNAL
#define ASN_DISABLED         RA_DISABLED
#define ASN_LAST             RA_LAST
  
/////////////////////////////////////////////////////////////////////////////////////////////////////////////
/// IOCTL_SET_RATE_DIVISOR
///
/// Parameter: unsigned long *pSetting;
///
/// Set the value that divides the assigned rate generator frequency (Ndiv).
///
////
#define IOCTL_SET_RATE_DIVISOR               _IOW(IOCTL_GSC, 43,unsigned long)
/* ---- rate divisor value limits ---- */ 
#define NDIV_MIN                            0
#define NDIV_MAX                            0xff
  
/////////////////////////////////////////////////////////////////////////////////////////////////////////////
/// IOCTL_SET_TIMEOUT
///
/// Set the wait timeout for reading a data buffer, in seconds.  Default is five seconds.
///
/// Parameter: unsigned long *pulTimeout;       
///
/// Range: 0x0-0xFFFFFFFF, 0x0=No Timeout
#define IOCTL_GSC_SET_TIMEOUT               _IOW(IOCTL_GSC, 23,unsigned long)
////
  
/////////////////////////////////////////////////////////////////////////////////////////////////////////////
/// IOCTL_SET_DMA_STATE
///
/// Enable or disable DMA for read.
/// 
/// Parameter: int *piDMAEnable;
/// 
#define IOCTL_GSC_SET_DMA_STATE             _IOW(IOCTL_GSC, 24,unsigned long)
////
  enum
{ 
DMA_DISABLE, 
DMA_ENABLE, 
DMA_DEMAND_MODE 
};


/////////////////////////////////////////////////////////////////////////////////////////////////////////////
/// IOCTL_FILL_BUFFER
///
/// Parameter: unsigned long *state
/// 
/// Return: none
///
/// This ioctl is used to tell the driver to fill the user buffer before returning.
/// If set, the driver will make one or more read transfers from the hardware to 
/// satisfy the user request.  If set to FALSE, the driver will return one or more samples
/// per the Linux convention.
///
#define IOCTL_GSC_FILL_BUFFER               _IOW(IOCTL_GSC, 26,unsigned long)
////
  
/////////////////////////////////////////////////////////////////////////////////////////////////////////////
/// IOCTL_SYNCRONIZE_SCAN
///
/// Parameter: unsigned long *type
/// 
/// Return: none
///
/// Set to TRUE to set the hardware to synchronous sampling mode, FALSE for
/// normal mode.
///
#define IOCTL_GSC_SYNCRONIZE_SCAN           _IOW(IOCTL_GSC, 27,unsigned long)
////
  
/////////////////////////////////////////////////////////////////////////////////////////////////////////////
/// IOCTL_CLEAR_BUFFER_SYNC
///
/// Parameter: unsigned long *state
/// 
/// Return: none
///
/// This ioctl is used to set the context of the software sync control bit.
/// When TRUE, the software sync control bit becomes "clear buffer."
///
#define IOCTL_GSC_CLEAR_BUFFER_SYNC         _IOW(IOCTL_GSC, 29,unsigned long)
////
  
/////////////////////////////////////////////////////////////////////////////////////////////////////////////
/// IOCTL_SET_OVERFLOW_CHECK
///
/// Parameter: unsigned long *state
/// 
/// Return: none
///
/// This ioctl is used to enable or disable buffer overflow checking.
/// When TRUE, the driver checks for buffer overflow.
///
#define IOCTL_GSC_SET_OVERFLOW_CHECK        _IOW(IOCTL_GSC, 31,unsigned long)
////
  
/////////////////////////////////////////////////////////////////////////////////////////////////////////////
/// IOCTL_SELECT_IMAGE_FILTER
///
/// Parameter: unsigned long *state
/// 
/// Return: none
///
/// This ioctl is used to select low or high frequency image filtering.
/// Possible choices are:
///
///      IMAGE_FILTER_LO_FREQ 
///      IMAGE_FILTER_HI_FREQ 
///
#define IOCTL_GSC_SELECT_IMAGE_FILTER       _IOW(IOCTL_GSC, 33,unsigned long)
////
#define IMAGE_FILTER_LO_FREQ    BCR_SELECT_IMAGE_FILTER
#define IMAGE_FILTER_HI_FREQ    0
  

/// @defgroup ioctl IOCTL Definitions
/// @{
  
/////////////////////////////////////////////////////////////////////////////////////////////////////////////
  
/// Set the width (resolution) of the input data
  
///
/// @Param unsigned long *state
/// 
/// @return non-zero if the parameter is out of range.
///
/// This ioctl is used to select the bit width of the output data.
///
/// Possible choices are:
///
///      - DATA_WIDTH_16
///      - DATA_WIDTH_18
///      - DATA_WIDTH_20
///      - DATA_WIDTH_24
///
/// Default is DATA_WIDTH_24.
///
/// @hideinitializer
#define IOCTL_GSC_SET_DATA_WIDTH       _IOW(IOCTL_GSC, 35,unsigned long) 
  


#define  DATA_WIDTH_16 BUCR_DATA_WIDTH_16
#define  DATA_WIDTH_18 BUCR_DATA_WIDTH_18
#define  DATA_WIDTH_20 BUCR_DATA_WIDTH_20
#define  DATA_WIDTH_24 BUCR_DATA_WIDTH_24
  

/////////////////////////////////////////////////////////////////////////////////////////////////////////////
/// IOCTL_SET_NREF
///
/// Parameter: unsigned long *pSetting;
/// 
/// Range: NREF_MIN to NREF_MAX  (30-1000).
///
/// This ioctl is used to set the value of the PLL Nref register. 
///
#define IOCTL_GSC_SET_NREF                  _IOW(IOCTL_GSC, 17,unsigned long)
////
  
/////////////////////////////////////////////////////////////////////////////////////////////////////////////
/// IOCTL_SET_NVCO
///
/// Parameter: unsigned long *pSetting;
///
/// This ioctl is used to set the value of the PLL Nvco factor.
#define IOCTL_GSC_SET_NVCO                  _IOW(IOCTL_GSC, 18,unsigned long)
////
  
/////////////////////////////////////////////////////////////////////////////////////////////////////////////
/// IOCTL_GET_REF_FREQUENCY
///
/// Parameter: unsigned long *pSetting;
///
/// This ioctl is used to read the reference frequency register.  See the hardware manual 
/// for the usage of this register.
///
#define IOCTL_GSC_GET_REF_FREQUENCY         _IOW(IOCTL_GSC, 19,unsigned long)
////
  
/////////////////////////////////////////////////////////////////
  
///  Set the range filter parameters.
  
///
/// @Param unsigned long *ulValue
/// 
/// @return non-zero if the parameter is out of range.
///
/// This ioctl is used to set the individual channel low image filter
/// and gain.  The entire register is written as a single bit mask.  
/// \NOTE This register is not available on all boards.  Check your
/// hardware manual for details.
///
/// @hideinitializer
#define IOCTL_GSC_SET_RANGE_FILTER      _IOW(IOCTL_GSC, 38,unsigned long)
  
/// @}  // end of ioctl
  
/////////////////////////////////////////////////////////////////
  
#ifndef DOC_SKIP /* don't add to the PDF file. */
  
// aliased IOCTLs for backwards compatibility
  
#define IOCTL_NO_COMMAND               IOCTL_GSC_NO_COMMAND
#define IOCTL_READ_REGISTER            IOCTL_GSC_READ_REGISTER
#define IOCTL_WRITE_REGISTER           IOCTL_GSC_WRITE_REGISTER
#define IOCTL_RESERVED_1               IOCTL_GSC_RESERVED_1
#define IOCTL_SET_INPUT_RANGE          IOCTL_GSC_SET_INPUT_RANGE
#define IOCTL_SET_INPUT_MODE           IOCTL_GSC_SET_INPUT_MODE
#define IOCTL_SET_SW_SYNC              IOCTL_GSC_SET_SW_SYNC
#define IOCTL_AUTO_CAL                 IOCTL_GSC_AUTO_CAL
#define IOCTL_INITIALIZE               IOCTL_GSC_INITIALIZE
#define IOCTL_SET_DATA_FORMAT          IOCTL_GSC_SET_DATA_FORMAT
#define IOCTL_SET_INITIATOR_MODE       IOCTL_GSC_SET_INITIATOR_MODE
#define IOCTL_SET_BUFFER_THRESHOLD     IOCTL_GSC_SET_BUFFER_THRESHOLD
#define IOCTL_CLEAR_BUFFER             IOCTL_GSC_CLEAR_BUFFER
#define IOCTL_SET_ACQUIRE_MODE         IOCTL_GSC_SET_ACQUIRE_MODE
#define IOCTL_GET_DEVICE_ERROR         IOCTL_GSC_GET_DEVICE_ERROR
#define IOCTL_ASSIGN_RATE_GROUP        IOCTL_GSC_ASSIGN_RATE_GROUP
  
#define IOCTL_SET_NREF                 IOCTL_GSC_SET_NREF
#define IOCTL_SET_NVCO                 IOCTL_GSC_SET_NVCO
#define IOCTL_GET_REF_FREQUENCY        IOCTL_GSC_GET_REF_FREQUENCY
  
#define IOCTL_SET_TIMEOUT              IOCTL_GSC_SET_TIMEOUT
#define IOCTL_SET_DMA_STATE            IOCTL_GSC_SET_DMA_STATE
#define IOCTL_GET_DEVICE_TYPE          IOCTL_GSC_GET_DEVICE_TYPE
#define IOCTL_FILL_BUFFER              IOCTL_GSC_FILL_BUFFER
#define IOCTL_SYNCRONIZE_SCAN          IOCTL_GSC_SYNCRONIZE_SCAN
#define IOCTL_CLEAR_BUFFER_SYNC        IOCTL_GSC_CLEAR_BUFFER_SYNC
#define IOCTL_SET_OVERFLOW_CHECK       IOCTL_GSC_SET_OVERFLOW_CHECK
#define IOCTL_SET_RATE_GEN_CLK_OUT     IOCTL_GSC_SET_RATE_GEN_CLK_OUT
#define IOCTL_SELECT_IMAGE_FILTER      IOCTL_GSC_SELECT_IMAGE_FILTER
#define IOCTL_SET_DATA_WIDTH           IOCTL_GSC_SET_DATA_WIDTH
#define IOCTL_READ_AUTOCAL_VALUE       IOCTL_GSC_READ_AUTOCAL_VALUE

#ifdef RTL

// RTLinux defines IOCTLs differently

enum
{
    IOCTL_NO_COMMAND,              
    IOCTL_READ_REGISTER,            
    IOCTL_WRITE_REGISTER,          
    IOCTL_SET_INPUT_RANGE,         
    IOCTL_SET_INPUT_MODE,        
    IOCTL_SET_SW_SYNC,            
    IOCTL_AUTO_CAL,            
    IOCTL_INITIALIZE,              
    IOCTL_SET_DATA_FORMAT,          
    IOCTL_SET_INITIATOR_MODE,      
    IOCTL_SET_BUFFER_THRESHOLD,     
    IOCTL_CLEAR_BUFFER,             
    IOCTL_SET_ACQUIRE_MODE,         
    IOCTL_ASSIGN_RATE_GROUP,        

    IOCTL_SET_NREF,                 
    IOCTL_SET_NVCO,                 
    IOCTL_GET_REF_FREQUENCY,        

    IOCTL_SET_TIMEOUT,              
    IOCTL_SET_DMA_STATE ,           
    IOCTL_GET_DEVICE_TYPE,          
    IOCTL_FILL_BUFFER,              
    IOCTL_SYNCRONIZE_SCAN,          
    IOCTL_CLEAR_BUFFER_SYNC,       
    IOCTL_SET_OVERFLOW_CHECK,       
    IOCTL_SET_RATE_GEN_CLK_OUT,     
    IOCTL_SELECT_IMAGE_FILTER,      
    IOCTL_SET_DATA_WIDTH,          
}

#endif

#endif	/* DOC_SKIP */
  
#endif	/* entire file */
