/***
*** testapp.c  
***
***  General description of this file:
***     Device test program for the General Standards 24DSI32 
***     family Delta-Sigma A/D boards. This file is part of the Linux
***     driver source distribution for this board.
***     
***  Copyrights (c):
***     General Standards Corporation (GSC), 2004-2007
***
***  Author:
***     Evan Hillman (evan@generalstandards.com)
***
***  Support:
***     Primary support for this driver is provided by GSC. 
***
***/  
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <fcntl.h>
#include <string.h>
#include <sys/ioctl.h>
#include <asm/types.h>
  

#include <rtl_stdio.h>
//#include <rtl_stlib.h>
#include <rtl_posixio.h>
#include <rtl_semaphore.h>
#include <rtl_fcntl.h>
#include <rtl_unistd.h>
  
#include "gsc24dsi32pll_ioctl.h"
#include "kbhit.h"

char driver_name[80] = "/dev/gs_ad\0";


#ifdef RTL 
#define PRINT rtl_printf
#else /* 
 */
#define PRINT printf
#endif /* 
 */
  
/* number of channels for each board supported. */ 
  
#define MAX_CHANNELS 32
  
#define BUFFER_SIZE (1024*16)
//#define BUFFER_SIZE (1024*256)
//#define BUFFER_SIZE (128)
#define BUFFER_BYTES (BUFFER_SIZE*sizeof(rtl_uint32_t))

int passes, channel;

time_t startTime, lastDisplay, now;

float channelVoltage[MAX_CHANNELS];

int channelReading[MAX_CHANNELS];


#define OFFSET_HIGH 0x519
#define OFFSET_LOW	0x3CF
#define GAIN_HIGH 0x0501
#define GAIN_LOW 0x03FF	
#define DSI24_NUM_CHANNELS 4  /*channel groups*/
#define DSI24_CHAN_PER_GROUP 8
  
//////////////////////////////////////////////////////////////////////////
// settings for input voltage range.
  
enum				// indexes into the arrays.
{ 
DATA_RANGE_2p5V, 
DATA_RANGE_5V, 
DATA_RANGE_10V, 
};


const int dataRanges[] =	// for the IOCTL calls.
{ 
RANGE_2p5V, 
RANGE_5V, 
RANGE_10V, 
};


//
// the full-scale voltage is double the range setting.  For example,
// 5 volt range is +/- 5 volts, or 10 volts full range.
//
const float fullRangeVolts[] =	// rail to rail
{ 
5,				// RANGE_2p5V
  10,				// RANGE_5V
  20,				// RANGE_10V
};


const float rangeVolts[] =	// range from zero volts
{ 
2.5,				// RANGE_2p5V
  5.0,				// RANGE_5V
  10.0,				// RANGE_10V
};



//////////////////////////////////////////////////////////////////////////
// setting for data resolution (width).
  
enum				// indexes for lookup in the following tables.
{ 
DATA_RESOLUTION_16, 
DATA_RESOLUTION_18, 
DATA_RESOLUTION_20,
  
DATA_RESOLUTION_24, 
};


const int fullRangeBits[] =	// full scale at each resolution.
{ 
0xFFFF,			// DATA_WIDTH_16 
  0x3FFFF,			// DATA_WIDTH_18 
  0xFFFFF,			// DATA_WIDTH_20 
  0xFFFFFF			// DATA_WIDTH_24 
};



// bit-value of zero
const int midValue[] =		// zero or mid-value, depending on format.
{ 
1 << 15,			// 16 bit zero
  1 << 17,			// 18 bit zero 
  1 << 19,			// 20 bit zero
  1 << 23,			// 24 bit zero
};


const int dataWidths[] =	// for the IOCTL calls.
{ 
DATA_WIDTH_16, 
DATA_WIDTH_18, 
DATA_WIDTH_20, 
DATA_WIDTH_24, 
};



//////////////////////////////////////////////////////////////////////////
// settings for data format.
  
enum				// indexes into the arrays.
{ 
DATA_FORMAT_TWOS_COMPLEMENT, 
DATA_FORMAT_OFFSET_BINARY, 
};


int dataFormats[] =		// for the IOCTL calls.
{ 
FORMAT_TWOS_COMPLEMENT, 
FORMAT_OFFSET_BINARY, 
};



//////////////////////////////////////////////////////////////////////////
// settings for input modes.
//
  enum				// indexes into the arrays.
{ 
INPUT_MODE_DIFFERENTIAL, 
INPUT_MODE_ZERO_TEST, 
INPUT_MODE_VREF_TEST, 
};


int dataModes[] = 
{ 
MODE_DIFFERENTIAL, 
MODE_ZERO_TEST, 
MODE_VREF_TEST, 
};


//
////////////////////////////////////////////////////////////////////////////
// these settings are used throughout the program.
//
int format = DATA_FORMAT_TWOS_COMPLEMENT;

int mode = INPUT_MODE_DIFFERENTIAL;

int range = DATA_RANGE_10V;

int width = DATA_RESOLUTION_24;

//
/////////////////////////////////////////////////////////////////////////////////
  double
toVolts (__u32 reading, int format, int mode, int range, int width) 
{
  
float volts;
  

float voltsPerCount = fullRangeVolts[range] / fullRangeBits[width];
  

if (format == FORMAT_OFFSET_BINARY)
    
    {
      
if (reading >= midValue[width])
	
	{
	  
volts = (reading - midValue[width]) * voltsPerCount;
	
}
      
      else
	
	{
	  
volts = (midValue[width] - reading) * -voltsPerCount;
	
}
    
}
  
  else				// format == FORMAT_TWOS_COMPLEMENT
    {
      
if (reading >= midValue[width])
	
	{
	  
volts =
	    -rangeVolts[range] +
	    ((reading - midValue[width]) * voltsPerCount);
	
}
      
      else
	
	{
	  
volts = reading * voltsPerCount;
	
}
    
}
  

// Voltage per count = Voltage range / Number of counts for example a 16bit board on the +/- 10V Range would be
//
// Voltage per count = 20 / 65535
// Voltage per count = .30518mV
//
// Then for offset-binary
//
// if Reading >= 0x8000
//   Voltage = Reading - 0x8000 * .00030518 else
//   -Voltage = 0x8000 - Reading * .00030518
//
// Then for 2's Compliment
//
// if Reading >= 0x8000
//   -Voltage = -10V + ((Reading - 0x8000) * .00030518) else
//   Voltage = Reading * .00030518
    
return volts;

}



rtl_uint32_t buffer[BUFFER_SIZE], total, count[MAX_CHANNELS], param;


//////////////////////////////////////////////////////////////////////////
  int
main (int argc, char *argv[]) 
{
  
struct gen_assign_params gen_param;
  
struct device_register_params rw_q;
  
int res;
  
//    rtl_uint32_t buffer[BUFFER_SIZE], total, count[MAX_CHANNELS],param;
  int fd, i, print, nread, bytesRead, wordsRead;
  
int lapse;			// measure of run time.
  int buffersRead;
  
int error_code;
  
int last;
  
int samplesChecked = 0;
  
int max_group;
  
int last_group;
  
struct rtl_timespec t1, t2, start;
  

if (argc < 2)
    {
      
PRINT ("%s: missing argument\nusage: testapp [0|1|2|...]\n", argv[0]);
      
return (1);
    
}
  

    // open the driver.
    
strcat (driver_name, argv[1]);
  
PRINT ("about to open %s\n", driver_name);
  

fd = open (driver_name, O_RDWR);
  
if (fd < 0)
    {
      
PRINT ("%s: can not open device %s\n", argv[0], driver_name);
      
return (1);
    
}
  

    // This allows use of the keyboard to terminate testing.
    
init_keyboard ();
  

    // only really useful for debug and development testing...
    
    //ioctl(fd, IOCTL_GSC_NO_COMMAND, NULL);
    
    // read the board control register to show before-reset state.
    
rw_q.ulRegister = BOARD_CTRL_REG;
  
res = ioctl (fd, IOCTL_GSC_READ_REGISTER, &rw_q);
  
if (res < 0)
    {
      
PRINT ("%s: ioctl IOCTL_GSC_READ_REGISTER failed\n", argv[0]);
      
goto OUT;
    
}
  
PRINT ("before reset: BCR is 0x%lx\n", rw_q.ulValue);
  

    // initialize the hardware to a known state.
    
res = ioctl (fd, IOCTL_GSC_INITIALIZE, NULL);
  
if (res < 0)
    {
      
PRINT ("%s: ioctl IOCTL_GSC_INIT failed\n", argv[0]);
      
goto OUT;
    
}
  
PRINT ("board init OK\n");
  

    // read the BCR again to show the reset worked.
    
rw_q.ulRegister = BOARD_CTRL_REG;
  
rw_q.ulValue = 0x2000;
  
res = ioctl (fd, IOCTL_GSC_READ_REGISTER, &rw_q);
  
if (res < 0)
    {
      
PRINT ("%s: ioctl IOCTL_GSC_READ_REGISTER failed\n", argv[0]);
      
goto OUT;
    
}
  
PRINT ("after reset: BCR is 0x%lx\n", rw_q.ulValue);
  

    // make sure there is no data coming into the buffer.
    
param = STOP_ACQUIRE;
  
res = ioctl (fd, IOCTL_GSC_SET_ACQUIRE_MODE, &param);
  
if (res < 0)
    {
      
PRINT ("%s: ioctl IOCTL_GSC_SET_ACQUIRE_MODE failed\n", argv[0]);
      
goto OUT;
    
}
  

    // get the buffers to a known state.  The driver waits
    // for the buffer clear to complete.
    
res = ioctl (fd, IOCTL_GSC_CLEAR_BUFFER, NULL);
  
if (res < 0)
    {
      
PRINT ("%s: ioctl IOCTL_GSC_CLEAR_BUFFER failed\n", argv[0]);
      
goto OUT;
    
}
  

    // Verify that the buffer is empty.
    
rw_q.ulRegister = BUFFER_SIZE_REG;
  
res = ioctl (fd, IOCTL_GSC_READ_REGISTER, &rw_q);
  
if (res < 0)
    {
      
PRINT ("%s: ioctl IOCTL_GSC_READ_REGISTER failed\n", argv[0]);
      
goto OUT;
    
}
  
PRINT ("after clear: Buffer size is 0x%lx\n", rw_q.ulValue);
  

    // set timeout.  Timeouts for autocalibration and initialization
    // are hard-coded.  This timeout is only used for filling the 
    // data buffer before returning.
    
param = 30;		//seconds
  res = ioctl (fd, IOCTL_GSC_SET_TIMEOUT, &param);
  
if (res < 0)
    {
      
PRINT ("%s: ioctl IOCTL_GSC_SET_TIMEOUT failed\n", argv[0]);
      
goto OUT;
    
}
  

    // Enable buffer overflow checking.
    
param = TRUE;
  
res = ioctl (fd, IOCTL_GSC_SET_OVERFLOW_CHECK, &param);
  
if (res < 0)
    {
      
PRINT ("%s: ioctl IOCTL_GSC_SET_OVERFLOW_CHECK failed\n", argv[0]);
      
goto OUT;
    
}
  

    // select data input format.
    
param = dataFormats[format];
  
res = ioctl (fd, IOCTL_GSC_SET_DATA_FORMAT, &param);
  
if (res < 0)
    {
      
PRINT ("%s: ioctl IOCTL_GSC_SET_DATA_FORMAT failed\n", argv[0]);
      
goto OUT;
    
}
  

    // select data input mode.
    
param = dataModes[mode];
  
res = ioctl (fd, IOCTL_GSC_SET_INPUT_MODE, &param);
  
if (res < 0)
    {
      
PRINT ("%s: ioctl IOCTL_GSC_SET_INPUT_MODE failed\n", argv[0]);
      
goto OUT;
    
}
  

    // select data input range.
    
param = dataRanges[range];
  
res = ioctl (fd, IOCTL_GSC_SET_INPUT_RANGE, &param);
  
if (res < 0)
    {
      
PRINT ("%s: ioctl IOCTL_GSC_SET_INPUT_RANGE failed\n", argv[0]);
      
goto OUT;
    
}
  

    // set the input data width
    
param = dataWidths[width];
  
res = ioctl (fd, IOCTL_GSC_SET_DATA_WIDTH, &param);
  
if (res < 0)
    {
      
PRINT ("%s: ioctl IOCTL_GSC_SET_DATA_WIDTH failed\n", argv[0]);
      
goto OUT;
    
}
  

    // set the NREF register.  See the hardware manual for calculations.
    
param = 100;
  
res = ioctl (fd, IOCTL_GSC_SET_NREF, &param);
  
if (res < 0)
    {
      
PRINT ("%s: ioctl IOCTL_GSC_SET_NREF failed\n", argv[0]);
      
goto OUT;
    
}
  

    // set the NVCO register.  See the hardware manual for calculations.
    
param = 100;
  
res = ioctl (fd, IOCTL_GSC_SET_NVCO, &param);
  
if (res < 0)
    {
      
PRINT ("%s: ioctl IOCTL_GSC_SET_NVCO failed\n", argv[0]);
      
goto OUT;
    
}
  


    // set the buffer trigger level.
    
//    param = 64;
//    param = 0x30000;
    param = 0x3000;
  
res = ioctl (fd, IOCTL_GSC_SET_BUFFER_THRESHOLD, &param);
  
if (res < 0)
    {
      
PRINT ("%s: ioctl IOCTL_GSC_SET_ACQUIRE_MODE failed\n", argv[0]);
      
goto OUT;
    
}
  

    // tell the driver to fill the user buffer before returning.
    
param = TRUE;
  
    //param = FALSE;
    res = ioctl (fd, IOCTL_GSC_FILL_BUFFER, &param);
  
if (res < 0)
    {
      
PRINT ("%s: ioctl IOCTL_GSC_FILL_BUFFER failed\n", argv[0]);
      
goto OUT;
    
}
  

    // set software sync mode.
    
res = ioctl (fd, IOCTL_GSC_SET_SW_SYNC, NULL);
  
if (res < 0)
    {
      
PRINT ("%s: ioctl IOCTL_GSC_SET_SW_SYNC failed\n", argv[0]);
      
goto OUT;
    
}
  

    // assign groups of channels to generators. Assign all groups to generator "A."
    
last_group = GRP_3;
  
for (i = GRP_0; i <= last_group; i++)
    
    {
      
gen_param.eGroup = i;
      
gen_param.eGenAssign = RA_GROUP_0_INTERNAL;	// also means "enabled" for the non-zero groups.
      res = ioctl (fd, IOCTL_GSC_ASSIGN_RATE_GROUP, &gen_param);
      
if (res < 0)
	{
	  
PRINT ("%s: ioctl IOCTL_GSC_ASSIGN_RATE_GROUP failed\n", argv[0]);
	  
goto OUT;
	
}
    
}
  

    // Select method for transfer for data from hardware
    // to the intermediate buffer
    
param = DMA_ENABLE;	// Regular DMA
  //param = DMA_DEMAND_MODE; //Demand Mode DMA
  //param = DMA_DISABLE; //PIO
  
ioctl (fd, IOCTL_GSC_SET_DMA_STATE, &param);
  
if (res < 0)
    {
      
PRINT ("%s: ioctl IOCTL_GSC_SET_DMA_STATE failed\n", argv[0]);
      
goto OUT;
    
}
  

    // Set the hardware to synchronize scans.
    
    //param = TRUE;
    //res = ioctl(fd, IOCTL_GSC_SYNCRONIZE_SCAN, &param);
    //if (res < 0) {
    //    PRINT("%s: ioctl IOCTL_GSC_SYNCRONIZE_SCAN failed\n", argv[0]);
    //    goto OUT;
    //}
    
    // autocal the analog section.  Driver waits till complete
    // before returning.
    
    // while (!kbhit())
    //   for (i=0;i<2;i++)
  {
    
      //time(&startTime);
      rtl_clock_gettime (RTL_CLOCK_REALTIME, &t1);
    
      //        PRINT("autocalibration...\n");
      res = ioctl (fd, IOCTL_GSC_AUTO_CAL, NULL);
    
      //        PRINT("autocal return: %x\n",res);
      if (res < 0)
      {
	
PRINT ("%s: ioctl IOCTL_GSC_AUTOCAL failed\n", argv[0]);
      
}
    
      //        PRINT("auto calibration OK\n");
      //dsi24_driver_autocaldump(fd);
      
time (&now);
    
if (now - startTime < autocalMin)
      autocalMin = now - startTime;
    
if (now - startTime > autocalMax)
      autocalMax = now - startTime;
    

PRINT ("Autocal took %d.%dus.\n", (int) (t2.tv_sec),
	     (int) (t2.tv_nsec / 1000));
    
if (res < 0)
      {
	
goto OUT;
      
}
  
}
  

    // start acquiring data 
    param = START_ACQUIRE;
  
ioctl (fd, IOCTL_GSC_SET_ACQUIRE_MODE, &param);
  
if (res < 0)
    {
      
PRINT ("%s: ioctl IOCTL_GSC_SET_ACQUIRE_MODE failed\n", argv[0]);
      
goto OUT;
    
}
  

for (i = 0; i < MAX_CHANNELS; i++)
    {
      
count[i] = 0;
    
}
  

total = 0L;
  
print = -1;
  
nread = passes;
  

    //startTime=time(NULL);
    rtl_clock_gettime (RTL_CLOCK_REALTIME, &start);
  
rtl_printf ("\n");
  
lapse = 0;
  
last = -1;
  
buffersRead = 0;
  

    /*******************************************/ 
    PRINT ("Starting main loop...\n");
  
while (!kbhit ())
    {				// run until the user presses a key.
//  while (lapse<60) { // run for a fixed time.
      

	// make sure the buffer contents are invalid, for testing.
	
memset (buffer, 0xaaaaaaaa, sizeof (buffer));
      

	// size of read is in bytes.
	bytesRead = read (fd, buffer, BUFFER_BYTES);
      
	/*
	   // this test is only meaningful if the board is set to 
	   // synchronize channel order.
	   // 
	   if (buffersRead==0) // check if the first sample is channel 0
	   {
	   channel = ((buffer[i] & IDR_CHANNEL_TAG_MASK) >> IDR_CHANNEL_TAG_SHIFT); 
	   if (channel != 0)
	   {
	   PRINT("Error on first sample - channel tag is %d not 0\n",channel);
	   }
	   }
	 */ 
	buffersRead++;
      
//      PRINT("read: %X\n",numread);
	if (bytesRead < 0)
	{
	  
PRINT ("\nread error -> after read...res = %d\n", bytesRead);
	  
goto OUT;
	
}
      

wordsRead = bytesRead / sizeof (long);
      
total += wordsRead;
      
passes++;
      

for (i = 0; i < wordsRead; i++)
	{
	  
if (buffer[i] == 0xAAAAAAAA)
	    
	    {
	      
PRINT ("Data integrity error passes: %d offset: %d %X\n",
		      passes, i, buffer[i]);
	      
goto OUT;
	    
}
	  
channel =
	    ((buffer[i] & IDR_CHANNEL_TAG_MASK) >> IDR_CHANNEL_TAG_SHIFT);
	  

	    //PRINT("[%d]%.8X ",i,buffer[i]);
	    if (channel < 0 || channel > MAX_CHANNELS)
	    
	    {
	      
PRINT ("Channel out of range: %d\n", channel);
	      
goto OUT;
	    
}
	  
if (channel != last + 1)
	    
	    {
	      
if (last == MAX_CHANNELS - 1 && channel == 0)
		
		{
		  
		    //  PRINT("!");
		}
	      
	      else
		
		{
		  
		    // verify channel sequence, if appropriate for the type of board.
		    // not all board models guarantee channels will be in sequence.
		    /*
		       int q;
		       rw_q.ulRegister = BUFFER_SIZE_REG;
		       res = ioctl(fd, IOCTL_GSC_READ_REGISTER, &rw_q);
		       PRINT("\nChannel sequence error last: %d Channel: %d Samples Checked: %d (0x%X) Buffer index: %d\n", last, channel,samplesChecked,samplesChecked,wordsRead-1);
		       for(q=0;q<=i;q++)
		       PRINT("[%d]%.8X ",q,buffer[q]);
		       PRINT("Buffer size register: %d 0x%.8X ",rw_q.ulValue,rw_q.ulValue);
		       goto OUT;
		     */ 
		}
	    
}
	  
samplesChecked++;
	  
last = channel;
	  
count[channel]++;
	  
channelReading[channel] = buffer[i] & fullRangeBits[width];
	  
channelVoltage[channel] =
	    toVolts (channelReading[channel], format, mode, range, width);
	
}
      
	//
	// display data once per second...
	//
	now = time (NULL);
      
if (now != lastDisplay)
	
	{
	  
lastDisplay = now;
	  
	  {
	    
int j;
	    
lapse = time (NULL) - startTime;
	    
if (lapse != 0)
	      
	      {
		
		  //PRINT ("seconds: %.8d Minutes: %d Words read: %.8x samples/sec: %.8ld - ",lapse, lapse/60, wordsRead, total/lapse);
		  for (j = 0; j < MAX_CHANNELS; j++)
		  
		  {
		    
PRINT ("[%.2d]%.8X %.8f\n", j, channelReading[j],
			    channelVoltage[j]);
		  
}
		
PRINT ("\n");
		
j = 0;
	      
}
	  
}
	  
PRINT ("      \r");
	  
fflush (stdout);
	
}
    
}
  

PRINT ("\n Total samples: %ld Samples per channel:\n", total);
  

for (i = 0; i < MAX_CHANNELS; i++)
    
    {
      
PRINT ("%.8d ", count[i]);
      
if ((i > 0) & ((i + 1) % 8 == 0))
	
	{
	  
PRINT ("\n");
	
}
    
}

OUT:
    // stop the clock.
    //ioctl(fd, IOCTL_GSC_NO_COMMAND, NULL);
    
param = STOP_ACQUIRE;
  
ioctl (fd, IOCTL_GSC_SET_ACQUIRE_MODE, &param);
  
close_keyboard ();
  
PRINT ("\n\n");
  
fflush (stdout);
  

    // Close the hardware.
    close (fd);
  

return 0;

}


