/***
*** dma.c
***
***  General description of this file:
***     Device driver source code for General Standards 
***     family of analog I/O boards. This file is part of the Linux
***     driver source distribution for this board.
***
***  Copyrights (c):
***     General Standards Corporation (GSC), 2005-2006
***
***  Author:
***     Evan Hillman (evan@generalstandards.com)
***
***  Support:
***     Primary support for this driver is provided by GSC. 
***
***  Platform (tested on, may work with others):
***     Linux, kernel version 2.4.x, 2.6.x,  Red Hat distribution, Intel hardware.
***/  
  
//////////////////////////////////////////////////////////////////////////
// set the following flag to trace interrupt debug messages.
#ifdef DEBUG
#define TRACE_LOCAL TRUE
#endif /* 
 */
  
#include "internals.h"
  
/************************************************************************/ 
/*  Returns void, assumes no errors possible                            */ 
/************************************************************************/ 
// Create a scatter-gather list usable by the PLX controller
  void
buildSgList (struct device_board *device, long wordsRequested,
	     int direction) 
{
  
long index;
  
int type;
  
long bytesRequested;
  

index = 0;
  
type = 0;
  

msg ("buildSgList requested samples %ld bytes %ld\n", wordsRequested,
	 wordsRequested * sizeof (__u32));
  
bytesRequested = wordsRequested * sizeof (__u32);	// convert to bytes.
  //msg("buildSgList sglist address: %p \n",device->sgList);
  
if (direction == DPR_XFER_FROM_LOCAL)	// read
    {
      
#ifdef SUPPORT_READ
	type = RX;
      
#endif /* 
 */
    }
  
  else
    
    {
      
#ifdef SUPPORT_WRITE
	type = TX;
      
#endif /* 
 */
    }
  

while (bytesRequested > 0)
    
    {
      
device->dmaData[type].sgList[index].size =
	min ((long) device->dmaData[type].sgElementSize[index],
	     bytesRequested);
      
bytesRequested = max (0l, (long) (bytesRequested - device->dmaData[type].sgElementSize[index]));	// update residual
      msg
	("buildSgList building entry: %p %ld addr:  remain: %ld entry size: %d\n",
	 &device->dmaData[type].sgList[index], index, bytesRequested,
	 device->dmaData[type].sgList[index].size);
      
device->dmaData[type].sgList[index].next.nextU32 &= ~(DPR_CONTROL_BITS_MASK);	// reset control bits from prior transfer
      device->dmaData[type].sgList[index].next.nextU32 |= DPR_PCI_DESCRIPTOR;
      
device->dmaData[type].sgList[index].next.nextU32 |= direction;
      

index++;
    
} 
index--;		// point to last used page.
  msg ("Last SG index used %ld\n", index);
  
device->dmaData[type].sgList[index].next.nextU32 |=
    (DPR_INT_ON_DMA_COMPLETE + DPR_END_OF_CHAIN);

} 


//////////////////////////////////////////////////////////////////////////
// create a list of memory pages for each active DMA channel.  Also sets
// up the initial state for a DMA transfer.
  
int
createSglChain (struct device_board *device) 
{
  
long i;
  
long dmaIndex;
  

for (dmaIndex = 0; dmaIndex < NUM_DMA_CHANNELS; dmaIndex++)
    
    {
      
msg ("About to get dma pages for channel %ld\n", dmaIndex);
      
device->dmaData[dmaIndex].dmaSize = SG_BUFFER_BYTES;
      
if (!(device->dmaData[dmaIndex].sgList = (struct sgElement *) /*__get_free_pages*/ 
	     __get_dma_pages (GFP_KERNEL,
			      get_order (SG_PAGES *
					 sizeof (struct sgElement) + 16))))
	{
	  
errmsg ("(%d): can not allocate DMA pages\n", device->minor);
	  
free_irq (device->pHardware->irqlevel, device);
	  
device->busy = 0;
	  
return -ENODEV;
	
}
      
device->dmaData[dmaIndex].sgListPhysical =
	(void *) virt_to_phys ((void *) device->dmaData[dmaIndex].sgList);
      
msg ("sgListPhysical %p\n", device->dmaData[dmaIndex].sgListPhysical);
      
for (i = 0; i < SG_PAGES; i++)
	
	{
	  
if (!
	       (device->dmaData[dmaIndex].sgPages[i] =
		(void *) __get_free_page (GFP_KERNEL)))
	    {
	      
errmsg ("(%d): can not allocate DMA pages\n", device->minor);
	      
free_irq (device->pHardware->irqlevel, device);
	      
device->busy = 0;
	      
return -ENODEV;
	    
}
	  
device->dmaData[dmaIndex].sgPagesPhysical[i] =
	    (void *) virt_to_phys ((void *) device->dmaData[dmaIndex].
				   sgPages[i]);
	  
	    //msg("[%d] device->sgPages[i] %p device->sgPagesPhysical[i] %p\n",i, device->sgPages[i],device->sgPagesPhysical[i]);
	    device->dmaData[dmaIndex].sgList[i].pciAddr =
	    device->dmaData[dmaIndex].sgPagesPhysical[i];
	  
device->dmaData[dmaIndex].sgList[i].next.nextPtr =
	    device->dmaData[dmaIndex].sgListPhysical +
	    ((i + 1) * sizeof (struct sgElement));
	  
device->dmaData[dmaIndex].sgElementSize[i] = PAGE_SIZE;
	  
#ifdef SUPPORT_READ
	    if (dmaIndex == RX)
	    
	    {
	      
		//msg("Read channel set up...\n");
		device->dmaData[dmaIndex].sgList[i].localAddr =
		(INPUT_DATA_BUFF_REG << 2);
	    
}
	  
#endif /* 
 */
#ifdef SUPPORT_WRITE
	    if (dmaIndex == TX)
	    
	    {
	      
		//msg("Write channel set up...\n");
		device->dmaData[dmaIndex].sgList[i].localAddr =
		OUTPUT_BUFFER_REG << 2;
	    
}
	  
#endif /* 
 */
	    //msg("pciAddr %p next %p size %x localAddr %x\n", device->sgList[i].pciAddr, device->sgList[i].next.nextPtr, device->sgList[i].size, device->sgList[i].localAddr);
	
}			//for(i=0;i<SG_PAGES;i++)
      
#ifndef RTL
	init_timer (&device->dmaData[dmaIndex].watchdog_timer);
      
device->dmaData[dmaIndex].watchdog_timer.function = timeout_handler;
      
device->dmaData[dmaIndex].watchdog_timer.data = (ulong) device;
      
#endif /* 
 */
	device->dmaData[dmaIndex].dmastart = FALSE;
      
device->dmaData[dmaIndex].dmasamples = 0;
      
device->dmaData[dmaIndex].dmaState = DMA_DISABLE;
      
device->dmaData[dmaIndex].intermediateSamples = 0;
      
device->dmaData[dmaIndex].useMmap = FALSE;
    
}
  
return 0;

}



#ifdef SUPPORT_READ
/************************************************************************/ 
/* returns > 0 (samples transferred) for success, <= 0 for error        */ 
/************************************************************************/ 
// transfer data from the hardware to the DMA buffer.
  long
dmaRead (struct device_board *device) 
{
  
long wordsRequested;
  
int retval;
  

wordsRequested = readLocal (device, INPUT_BUFFER_SIZE_REG);
  
wordsRequested = min (wordsRequested, SG_BUFFER_SAMPLES);
  

msg ("(%d): dmaRead, requested %ld\n", device->minor, wordsRequested);
  
buildSgList (device, wordsRequested, DPR_XFER_FROM_LOCAL);
  

setEventFlag (device, EVENT_RX_DMA_PENDING);
  
EnableIrqPlxDMA0 (device);
  

if (device->dmaData[RX].dmaState == DMA_ENABLE)	// normal DMA
    {
      
writePLX (device, DMA_CH_0_MODE,
		 NON_DEMAND_DMA_MODE | DMA_MODE_CHAINING);
    
}
  
  else				// demand mode
    {
      
msg ("dmaRead - demand mode\n");
      
writePLX (device, DMA_CH_0_MODE,
		 DEMAND_DMA_MODE_READ | DMA_MODE_CHAINING);
    
}
  

writePLX (device, DMA_CH_0_LOCAL_ADDR,
	      (INPUT_DATA_BUFFER_REG * sizeof (__u32)));
  
writePLX (device, DMA_CH_0_TRANS_BYTE_CNT, wordsRequested * sizeof (u32));
  
writePLX (device, DMA_CH_0_DESC_PTR, 0xA);
  
    //
    writePLXb (device, DMA_CMD_STATUS_0, DMA_CHAN_ENABLE);
  
writePLX (device, DMA_MODE_ARBITRATION, 0);
  
writePLX (device, DMA_THRESHOLD_REG, 0);
  
writePLX (device, DMA_CH_0_DESC_PTR,
	     (__u32) device->dmaData[RX].sgListPhysical | DPR_PCI_DESCRIPTOR);
  
    // OK, here we go!
    
mark (0xeeeeeeee);
  
msg ("(%d): dmaRead - about to sleep...\n", device->minor);
  

setupWatchdog (device, RX, device->timeout_seconds);
  
writePLXb (device, DMA_CMD_STATUS_0, DMA_CHAN_ENABLE | DMA_CHAN_START);
  
retval = waitEvent (device, EVENT_RX_DMA_PENDING);
  
retval |= cleanupWatchdog (device, RX);
  

if (retval != 0)
    
    {
      
errmsg ("(%d): dmaRead - timeout...\n", device->minor);
      
	//              msg(" (%d): In progress: %d PLX int/ctrl %.8X BC %.8X DMA %.8X\n", device->minor,device->dma_pending, readl(IntCntrlStat(device)), readLocal(device,BOARD_CTRL_REG),readl(DMACmdStatus(device)));
	mark (0x11111122);
      
return (-EAGAIN);
    
}
  
device->dmaData[RX].intermediateSamples += wordsRequested;
  
mark (0x11111123);
  
    //{
    //    long i;
    //    for (i=0;i<32;i++)
    //    {
    //        msg("[%d] %.8X\n", i, device->dmaData[RX].sgPages[0][i]);
    //    }
    //}
    msg ("Dma returning %ld intermediate samples %ld\n", wordsRequested,
	 device->dmaData[RX].intermediateSamples);
  
return wordsRequested;

}


#endif /* 
 */
  
#ifdef SUPPORT_WRITE
//////////////////////////////////////////////////////////////////////////
// transfer data from the DMA buffer to the hardware.
  long
dmaWrite (struct device_board *device, long size) 
{
  
long wordsRequested;
  

#ifdef TRACE_LOCAL
  static long total;
  
#endif /* 
 */
    
    //  long time_remain;
    
    //if (down_interruptible(&device->hwSem)){
    //    return -ERESTARTSYS;
    //}
    
msg ("(%d): device_write DMA... size: %ld (0x%lx)\n", device->minor,
	  size, size);
  
mark (66666661);
  

wordsRequested = size / sizeof (__u32);
  
buildSgList (device, wordsRequested, DPR_XFER_TO_LOCAL);
  

setEventFlag (device, EVENT_TX_DMA_PENDING);
  
EnableIrqPlxDMA1 (device);
  
writePLX (device, DMA_CH_1_MODE, NON_DEMAND_DMA_MODE | DMA_MODE_CHAINING);
  
    //BUGBUG writePLX(device, DMA_CH_1_PCI_ADDR, device->dmaData[TX].intermediatePhysicalAddr);
    writePLX (device, DMA_CH_1_LOCAL_ADDR, (OUTPUT_BUFFER_REG * sizeof (__u32)));	// convert to byte-aligned address.
  writePLX (device, DMA_CH_1_TRANS_BYTE_CNT, size);
  
writePLX (device, DMA_CH_1_DESC_PTR,
	     (__u32) device->dmaData[TX].sgListPhysical | DPR_PCI_DESCRIPTOR);
  

writePLXb (device, DMA_CMD_STATUS_1, DMA_CHAN_ENABLE);
  
    //up(&device->hwSem);
    mark (66666666);
  
#ifdef TRACE_LOCAL
    total++;
  
msg ("(%d): dmaWrite - about to sleep...%ld\n", device->minor, total);
  
#endif /* 
 */
    setupWatchdog (device, TX, device->timeout_seconds);
  

if (!device->firstWrite && readLocal (device, OUT_BUFFER_SIZE_REG) < 5)
    
    {
      
static long underruns;
      
errmsg ("Write buffer underrun #%d level: %.8X\n", ++underruns,
	       readLocal (device, OUT_BUFFER_SIZE_REG));
      
if (device->detectUnderflow)
	
	{
	  
regdump (device, "underrun - ");
	  

return (-ERESTARTSYS);
	
}
    
}
  
writePLXb (device, DMA_CMD_STATUS_1, DMA_CHAN_ENABLE | DMA_CHAN_START);
  
msg ("Dma 1 CmdStat before wait: %.8X\n",
	readPLXb (device, DMA_CMD_STATUS_1));
  
retval = waitEvent (device, EVENT_TX_DMA_PENDING);
  

retval |= cleanupWatchdog (device, TX);
  

if (retval != 0)
    
    {
      
errmsg ("(%d): dmaWrite - timeout...\n", device->minor);
      
	//printk(" (%d): In progress: %d PXL int/ctrl %.8X BC %.8X DMA %.8X\n", device->minor,device->dmainprogress, readPLX(device,IntCntrlStat(device)), readLocal(device,BOARD_CTRL_REG),read(DMACmdStatus(device)));
	
return (-EAGAIN);
    
}
  

msg ("(%d): dmaWrite - normal return\n", device->minor);
  
return 0;

}


#endif /* 
 */
  
