/*
SIO4b_sync_sample.c:

=========================================================================
Version:	Date:			Purpose:				Initials:
-------------------------------------------------------------------------
1.0			08-20-2004		Initial code			jdp
1.1			03-24-2005		Added support for		jdp
							-SYNC variation
2.0			06-13/2010		Updated for v1.5.1 API	jdp

=========================================================================

This is a "safe arrival" test for the PCI-SIO4B and PCI-SIO4B-SYNC cards.
This test allows the user to verify that the driver, API and hardware are
installed and working correctly.  

This test performs an "external" loopback test on the SIO4B cards.  The data/clock 
are looped back at the transceivers, so no cable should be connected during the test.
*/


// SIO4B_TestDlg.cpp : implementation file
//

#include "stdafx.h"
#include "SIO4B_Test.h"
#include "SIO4B_TestDlg.h"
#include "GscApi.h"

#ifdef _DEBUG
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif


//------------------------------------------------------------------------
// Global Vars:
//------------------------------------------------------------------------
// Declare a variable to hold the number of boards in the system
int numberOfBoards;	

//	declaration of local functions for SIO4B_TestDlg.cpp
//	these functions do not require access to members of CSIO4B_TestDlg class
int TestLocalReg(int boardNumber, int iChan, int iReg, int iMask, int iLoop);
int GscSio4_RegisterTest(int boardIndex, int channelIndex);
int GscSio4_RegisterTestSYNC(int boardIndex, int channelIndex);

/////////////////////////////////////////////////////////////////////////////
// CAboutDlg dialog used for App About

class CAboutDlg : public CDialog
{
public:
	CAboutDlg();

// Dialog Data
	//{{AFX_DATA(CAboutDlg)
	enum { IDD = IDD_ABOUTBOX };
	//}}AFX_DATA

	// ClassWizard generated virtual function overrides
	//{{AFX_VIRTUAL(CAboutDlg)
	protected:
	virtual void DoDataExchange(CDataExchange* pDX);    // DDX/DDV support
	//}}AFX_VIRTUAL

// Implementation
protected:
	//{{AFX_MSG(CAboutDlg)
	//}}AFX_MSG
	DECLARE_MESSAGE_MAP()
};

CAboutDlg::CAboutDlg() : CDialog(CAboutDlg::IDD)
{
	//{{AFX_DATA_INIT(CAboutDlg)
	//}}AFX_DATA_INIT
}

void CAboutDlg::DoDataExchange(CDataExchange* pDX)
{
	CDialog::DoDataExchange(pDX);
	//{{AFX_DATA_MAP(CAboutDlg)
	//}}AFX_DATA_MAP
}

BEGIN_MESSAGE_MAP(CAboutDlg, CDialog)
	//{{AFX_MSG_MAP(CAboutDlg)
		// No message handlers
	//}}AFX_MSG_MAP
END_MESSAGE_MAP()

/////////////////////////////////////////////////////////////////////////////
// CSIO4B_TestDlg dialog

CSIO4B_TestDlg::CSIO4B_TestDlg(CWnd* pParent /*=NULL*/)
	: CDialog(CSIO4B_TestDlg::IDD, pParent)
{
	//{{AFX_DATA_INIT(CSIO4B_TestDlg)
		// NOTE: the ClassWizard will add member initialization here
	//}}AFX_DATA_INIT
	// Note that LoadIcon does not require a subsequent DestroyIcon in Win32
	m_hIcon = AfxGetApp()->LoadIcon(IDR_MAINFRAME);
}

void CSIO4B_TestDlg::DoDataExchange(CDataExchange* pDX)
{
	CDialog::DoDataExchange(pDX);
	//{{AFX_DATA_MAP(CSIO4B_TestDlg)
		// NOTE: the ClassWizard will add DDX and DDV calls here
	//}}AFX_DATA_MAP
}

BEGIN_MESSAGE_MAP(CSIO4B_TestDlg, CDialog)
	//{{AFX_MSG_MAP(CSIO4B_TestDlg)
	ON_WM_SYSCOMMAND()
	ON_WM_PAINT()
	ON_WM_QUERYDRAGICON()
	ON_BN_CLICKED(IDC_RUN_TESTS, OnRunTests)
	//}}AFX_MSG_MAP
END_MESSAGE_MAP()

/////////////////////////////////////////////////////////////////////////////
// CSIO4B_TestDlg message handlers

BOOL CSIO4B_TestDlg::OnInitDialog()
{
	CDialog::OnInitDialog();
	CString string;
	// ptr to board list box
	CListBox *pBoardBox = (CListBox *)GetDlgItem( IDC_BOARD_LIST);

	SetWindowText("SIO4B/BX/SYNC Safe Arrival Test v2.0");

	// IDM_ABOUTBOX must be in the system command range.
	ASSERT((IDM_ABOUTBOX & 0xFFF0) == IDM_ABOUTBOX);
	ASSERT(IDM_ABOUTBOX < 0xF000);

	CMenu* pSysMenu = GetSystemMenu(FALSE);
	if (pSysMenu != NULL)
	{
		CString strAboutMenu;
		strAboutMenu.LoadString(IDS_ABOUTBOX);
		if (!strAboutMenu.IsEmpty())
		{
			pSysMenu->AppendMenu(MF_SEPARATOR);
			pSysMenu->AppendMenu(MF_STRING, IDM_ABOUTBOX, strAboutMenu);
		}
	}

	// Set the icon for this dialog.  The framework does this automatically
	//  when the application's main window is not a dialog
	SetIcon(m_hIcon, TRUE);			// Set big icon
	SetIcon(m_hIcon, FALSE);		// Set small icon
	

	//----------------------------------------------------------- 
	// Begin GSC Initialization
	//-----------------------------------------------------------


	// Declare a structure to hold board information. This structure
	// will be filled in by the call to GscSio4FindBoards.
	GSC_DEVICES_STRUCT gscDevices[ MAXIMUM_NUMBER_OF_BOARDS];


	// Find all of the Sio4 Boards in the system. This must be the 
	// first call to any of the GSC API or Driver routines.
	//
	// this is not really necessary as the DLL performs this operation
	// but it allows us to get the values for display in the BoardBox
	int status = GscSio4FindBoards( &numberOfBoards, gscDevices); 
	if (status) 
	{
		DisplayErrorMessage( status);
		return TRUE;
	}


	// If no boards found
	if(numberOfBoards < 1) 
	{
		string = "No Boards found in the system\n";
		pBoardBox->AddString(string);
	}

	// Otherwise
	else
	{
		// Display all of the boards that were found in the system
		for (int i=0;i< (int)numberOfBoards; i++)
		{
			string.Format("Board #: %d,  Bus: %d,  Slot: %d,  Vendor: %04X,  Device: %04X,  Name: %s",
				i+1,
				gscDevices[i].busNumber,
				gscDevices[i].slotNumber,
				gscDevices[i].vendorId,
				gscDevices[i].deviceId,
				gscDevices[i].serialNumber);
			pBoardBox->AddString(string);
		}
	}
	
	// To add:
	// 1. right/dbl click on board for more info


	//----------------------------------------------------------- 
	// End GSC Initialization
	//-----------------------------------------------------------
	return TRUE;  // return TRUE  unless you set the focus to a control
}

void CSIO4B_TestDlg::OnSysCommand(UINT nID, LPARAM lParam)
{
	if ((nID & 0xFFF0) == IDM_ABOUTBOX)
	{
		CAboutDlg dlgAbout;
		dlgAbout.DoModal();
	}
	else
	{
		CDialog::OnSysCommand(nID, lParam);
	}
}

//	If you add a minimize button to your dialog, you will need the code below
//  to draw the icon.  For MFC applications using the document/view model,
//  this is automatically done for you by the framework.

void CSIO4B_TestDlg::OnPaint() 
{
	if (IsIconic())
	{
		CPaintDC dc(this); // device context for painting

		SendMessage(WM_ICONERASEBKGND, (WPARAM) dc.GetSafeHdc(), 0);

		// Center icon in client rectangle
		int cxIcon = GetSystemMetrics(SM_CXICON);
		int cyIcon = GetSystemMetrics(SM_CYICON);
		CRect rect;
		GetClientRect(&rect);
		int x = (rect.Width() - cxIcon + 1) / 2;
		int y = (rect.Height() - cyIcon + 1) / 2;

		// Draw the icon
		dc.DrawIcon(x, y, m_hIcon);
	}
	else
	{
		CDialog::OnPaint();
	}
}

// The system calls this to obtain the cursor to display while the user drags
//  the minimized window.
HCURSOR CSIO4B_TestDlg::OnQueryDragIcon()
{
	return (HCURSOR) m_hIcon;
}


//----------------------------------------------------------- 
//	RunTests Button was pushed
//-----------------------------------------------------------
void CSIO4B_TestDlg::OnRunTests() 
{
	int boardIndex = -1;
	int channelIndex = 1;
	int stringIndex = 0;
	int iCounter;
	int iErrors;
	int status;
	int FIFOsizes;
	int libVersion, driverVersion, fpgaVersion;
	int formFactor, boardModel, bdType, lineCounter = 0;
	bool bdTypeSYNC = FALSE;

	CString string;
	CString str;
	iErrors = 0;
	
	// get ptrs to list boxes
	CListBox *pBoardBox = (CListBox *)GetDlgItem( IDC_BOARD_LIST);
	CListBox *pMainListBox = (CListBox *)GetDlgItem( IDC_MAIN_LIST);
	
	// clean mainListBox
	pMainListBox->ResetContent();

	// if only one board, we will just assume to test board(1)
	if(numberOfBoards > 1)
	{
		// check to see which board is highlighted to test
		boardIndex = pBoardBox->GetCurSel();

		// check to see if no board is selected
		if(boardIndex == -1)
		{
			MessageBox("Sorry...No board was selected for test\nPlease highlight the card to test", "Board Not Specified");
			return;
		}
		
		// the SIO4 board number = index from BoardBox + 1; 
		boardIndex++;
	}
	else
	{	// only one board was found, so assume test is on boardIndex 1
		boardIndex = 1;
	}	


	// Initialize the board. This must be done before any channel operations
	// can be performed
 	status = GscSio4Open( boardIndex , GSC_API_VERSION);
	if (status) DisplayErrorMessage(status);

	// begin tests:
	pMainListBox->SetTabStops(15);

	//------------------------------------------------------------------------
	// Display Board Info
	//------------------------------------------------------------------------
	pMainListBox->AddString("Board Information:");
	

	//	print FIFO sizes:
	for(iCounter = 1; iCounter < 2; iCounter++)
	{
		status = GscSio4ChannelFifoSizes(boardIndex, iCounter, &FIFOsizes);
		if(status)  
			DisplayErrorMessage(status);
		else
		{
			str.Format("\tTx FIFO Size:\t\t\t\t\t0x%X bytes", (FIFOsizes  >> 16) & 0xFFFF);
			pMainListBox->AddString(str);
			lineCounter++;
			str.Format("\tRx FIFO Size:\t\t\t\t\t0x%X bytes", (FIFOsizes & 0xFFFF));
			pMainListBox->AddString(str);
			lineCounter++;
		}
	}


	status = GscSio4GetVersions(boardIndex, &libVersion, &driverVersion, &fpgaVersion);
	if(status)  
		DisplayErrorMessage(status);

	str.Format("\tFirmware Revision:\t\t\t\t0x%08X", fpgaVersion);
	pMainListBox->AddString(str);
	lineCounter++;
	pMainListBox->RedrawWindow();  
	
	
	// build part number string here
	str.Format("\tBoard Type:\t\t\t\t\t");

	formFactor = (fpgaVersion >> 24) & 0xF; 
	switch(formFactor)
	{
		case 1:
			str += "PCI-";
		break;
		case 2:
			str += "PMC-";
		break;
		case 3:
			str += "cPCI-";
		break;
		case 4:
			str += "pc104p-";
		break;
		default:
			str += "Unknown-";
	}

	//	Check for SIO4B/BX version
	boardModel = (fpgaVersion >> 20) & 0xF; 
	switch(boardModel)
	{
		case 0:
			str += "SIO4B";
		break;
		case 1:
			str += "SIO4BX";
		break;
		default:
			str += "Unknown";
	}

	//	check for -SYNC
	bdType = (fpgaVersion >> 8) & 0xF;
	if(bdType <= 1)
		str += "";
	else if (bdType >= 3)	// this may change later if any other versions are designed
	{
		str += "-SYNC";
		bdTypeSYNC = TRUE;
	}
	else
		str += "-Unknown";
	
	pMainListBox->AddString(str);
	lineCounter++;
	pMainListBox->RedrawWindow();  
	// end - build part number string here



	//------------------------------------------------------------------------
	// Display Software Version Info
	//------------------------------------------------------------------------
	pMainListBox->AddString("Software Information:");
	lineCounter++;

	str.Format("\tLibrary (API) Revision:\t\t\t0x%08X", libVersion);
	pMainListBox->AddString(str);
	lineCounter++;

	str.Format("\tDevice Driver Revision:\t\t\t0x%08X", driverVersion);
	pMainListBox->AddString(str);
	lineCounter++;

	pMainListBox->RedrawWindow();  

	str.Format("Safe Arrival Tests - Board %d:", boardIndex);
	pMainListBox->AddString(str);
	lineCounter++;
	pMainListBox->RedrawWindow();  


	//------------------------------------------------------------------------
	//	Begin tests:
	//------------------------------------------------------------------------
	if(bdTypeSYNC)
	{	// perform -SYNC tests
		//------------------------------------------------------------------------
		// Register Tests
		//------------------------------------------------------------------------
		str.Format("\tRegister Access Tests:\t\t\tTesting...");
		pMainListBox->AddString(str);
		lineCounter++;
		pMainListBox->RedrawWindow();  

		for(iCounter = 1; iCounter < 5; iCounter++)
			iErrors += GscSio4_RegisterTestSYNC(boardIndex, iCounter);
	
		if(iErrors)
			str.Format("\tRegister Access Tests\t\t\tFAILED - %d Errors", iErrors);
		else
			str.Format("\tRegister Access Tests\t\t\tPassed");

		pMainListBox->DeleteString(lineCounter);
	  	pMainListBox->AddString(str);
		
		//------------------------------------------------------------------------
		//	Perform Loopback Tests
		//------------------------------------------------------------------------
		iErrors = 0;
		str.Format("\tExternal Loopback Tests:\t\tTesting...");
		pMainListBox->AddString(str);
		lineCounter++;
		pMainListBox->RedrawWindow();  

		for(iCounter = 1; iCounter < 5; iCounter++)
			iErrors += GscSio4_IntLoopbackTestSYNC(boardIndex, iCounter);

		if(iErrors)
			str.Format("\tExternal Loopback Tests\t\tFAILED - %d Errors", iErrors);
		else
			str.Format("\tExternal Loopback Tests\t\tPassed");

		// use counter rather than specific values
		pMainListBox->DeleteString(lineCounter);
  		pMainListBox->AddString(str);
		pMainListBox->RedrawWindow();  
	}
	else
	{	// perform standard tests
		//------------------------------------------------------------------------
		// Register Tests
		//------------------------------------------------------------------------
		str.Format("\tRegister Access Tests:\t\t\tTesting...");
		pMainListBox->AddString(str);
		lineCounter++;
		pMainListBox->RedrawWindow();  

		for(iCounter = 1; iCounter < 5; iCounter++)
			iErrors += GscSio4_RegisterTest(boardIndex, iCounter);
	
		if(iErrors)
			str.Format("\tRegister Access Tests\t\t\tFAILED - %d Errors", iErrors);
		else
			str.Format("\tRegister Access Tests\t\t\tPassed");

		pMainListBox->DeleteString(lineCounter);
	  	pMainListBox->AddString(str);
		
		//------------------------------------------------------------------------
		//	Perform Loopback Tests
		//------------------------------------------------------------------------
		iErrors = 0;
		str.Format("\tExternal Loopback Tests:\t\tTesting...");
		pMainListBox->AddString(str);
		lineCounter++;
		pMainListBox->RedrawWindow();  

		for(iCounter = 1; iCounter < 5; iCounter++)
			iErrors += GscSio4_IntLoopbackTest(boardIndex, iCounter);

		if(iErrors)
			str.Format("\tExternal Loopback Tests\t\tFAILED - %d Errors", iErrors);
		else
			str.Format("\tExternal Loopback Tests\t\tPassed");

		// use counter rather than specific values
		pMainListBox->DeleteString(lineCounter);
  		pMainListBox->AddString(str);
		pMainListBox->RedrawWindow();  
	}


	// Shut down the board. This must be the last operation performed. It
	// releases the resources that are used by the GSC API and driver.
	status = GscSio4Close(boardIndex);	
	if (status) DisplayErrorMessage(status);


} // end void CSIO4B_TestDlg::OnRunTests()



//---------------------------------------------------------
// Displays a msg box with error code
//---------------------------------------------------------
void CSIO4B_TestDlg::DisplayErrorMessage(int errorCode)
{
	// Buffer to hold the error message
	char errorMessage[80];

	// Call the API to interpret the message
	GscSio4GetErrorString( errorCode, errorMessage);

	// Display the error
	CString string;
	string.Format("Error Code: %d, %s", errorCode, errorMessage);
	AfxMessageBox( string);
}


//---------------------------------------------------------
// Loopback Test
//---------------------------------------------------------
int CSIO4B_TestDlg::GscSio4_IntLoopbackTest(int boardIndex, int channelIndex)
{
	int iErrors;
	int status = 0x00, index;
	int iBytesXfered;
	GSC_ASYNC_CONFIG asyncConfig;

	#define TEST_BUFFER_SIZE	128			// Size of test transfer
	char TxBuffer[ TEST_BUFFER_SIZE];		// Buffer to hold data to transmit
	char RxBuffer[ TEST_BUFFER_SIZE];		// Buffer to hold received data
	CString str;

	// These variables are used to track the data transfers
	int txId, rxId;

	CListBox *pMainListBox = (CListBox *)GetDlgItem( IDC_MAIN_LIST);

	iErrors = 0;

	// Zero out the Rx buffer
	memset( RxBuffer, 0x00, sizeof(RxBuffer));

	// Fill the Tx buffer with pattern 
	for (index=0; index < TEST_BUFFER_SIZE; index+=4)
	{
		*(TxBuffer + index)		= (char)0xaa;
		*(TxBuffer + (index+1))	= (char)0x55;
		*(TxBuffer + (index+2)) = (char)(index>>10);	// Upper half of index/4
		*(TxBuffer + (index+3)) = (char)(index>>2);		// Lower half of index/4
	}

	// Reset the desired channel. This puts the channel into a known state.
	status = GscSio4ChannelReset( boardIndex, channelIndex);
	if (status) return status;


	// Setup the Channel defaults
	asyncConfig.bitRate =			115200;
	asyncConfig.encoding =			GSC_ENCODING_NRZ;
	asyncConfig.protocol =			GSC_PROTOCOL_RS422_RS485;
	asyncConfig.termination =		GSC_TERMINATION_ENABLED;
	asyncConfig.parity =			GSC_PARITY_NONE;
	asyncConfig.stopBits =			GSC_STOP_BITS_1;
  
	// Setup the Transmitter defaults
	asyncConfig.txStatus =			GSC_ENABLED;
	asyncConfig.txCharacterLength = 8;
	asyncConfig.txClockSource =		GSC_CLOCK_INTERNAL;
	
	// Setup the Receiver defaults
	asyncConfig.rxStatus =			GSC_ENABLED;
	asyncConfig.rxCharacterLength = 8;
	asyncConfig.rxClockSource =		GSC_CLOCK_INTERNAL;

	// Setup the Pin Configuration Variables
	asyncConfig.interfaceMode =		GSC_PIN_DTE;
	asyncConfig.txDataPinMode =		GSC_PIN_AUTO;
	asyncConfig.rxDataPinMode =		GSC_PIN_AUTO;
	asyncConfig.txClockPinMode =	GSC_PIN_GPIO;
	asyncConfig.rxClockPinMode =	GSC_PIN_GPIO;
	asyncConfig.rtsPinMode =		GSC_PIN_GPIO;
	asyncConfig.ctsPinMode =		GSC_PIN_GPIO;
	asyncConfig.loopbackMode =		GSC_LOOP_EXTERNAL;

	//	Set channel config
	status = GscSio4AsyncSetConfig( boardIndex, channelIndex, asyncConfig );


  	// Start processing any recieved data
  	status = GscSio4ChannelReceiveData( boardIndex, channelIndex, RxBuffer, TEST_BUFFER_SIZE, &rxId);
	if (status) return status;

	// Initiate the transfer of data
	status = GscSio4ChannelTransmitData( boardIndex, channelIndex, TxBuffer, TEST_BUFFER_SIZE, &txId);
	if (status) return status;


	// Wait up to 1 seconds for all of the data to arrive. The last parameter, rxId, 
	// informs the routine of the transfer to wait on. This variable was set during
	// the call to GscSio4ReceiveData(...).
	status = GscSio4ChannelWaitForTransfer( boardIndex, channelIndex, 1000, rxId, &iBytesXfered);
	if (status) return status;

	// Verify that the buffers match
	for (index = 0; index < TEST_BUFFER_SIZE; index++)
	{
		if (*(TxBuffer + index) != *(RxBuffer + index)) 
		{
			iErrors++;
			str.Format("Channel %d Data Mismatch at Location 0x%04X\n(Wrote: 0x%02X, Read: 0x%02X)",
				channelIndex,
				index,
				TxBuffer[ index] & 0xff,
				RxBuffer[ index] & 0xff);
			MessageBox(str, "Data Error Detected");
			break;			
		}
	}

	return(iErrors);
}// end int CSIO4B_TestDlg::GscSio4_IntLoopbackTest()

//---------------------------------------------------------
// Register Tests
//---------------------------------------------------------
int GscSio4_RegisterTest(int boardIndex, int channelIndex)
{
	int iErrors, iRdValue, status, iLoopCtr;
	iErrors = 0;
	iLoopCtr = 100;
	CString string;

	// Testing BOARD_CTRL_REG
	iErrors += TestLocalReg(boardIndex, channelIndex, BOARD_CONTROL_REG,0x0077l,iLoopCtr);

	// Testing TX_ALMOST
	iErrors += TestLocalReg(boardIndex, channelIndex, (TX_ALMOST_BASE_REG + ((channelIndex - 1) << 4) ),0x7FFF7FFFl,iLoopCtr);

	// Testing RX_ALMOST
	iErrors += TestLocalReg(boardIndex, channelIndex, (RX_ALMOST_BASE_REG + ((channelIndex - 1) << 4) ),0x7FFF7FFFl,iLoopCtr);

	// Testing SYNC_CHARACTER
	iErrors += TestLocalReg(boardIndex, channelIndex, (SYNC_CHARACTER_BASE_REG + ((channelIndex - 1) * 4) ),0xFFl,iLoopCtr);
   
	// Testing INT_CTRL_REG
	// read int status, invert, pass as mask to Local_Reg_Test and only check 0..15
	status = GscSio4LocalRegisterRead(boardIndex, INTERRUPT_STATUS_REG, &iRdValue);
	if(status)
		return(status);
	iRdValue = ~iRdValue;
	iErrors += TestLocalReg(boardIndex, channelIndex, INTERRUPT_CONTROL_REG, (iRdValue & 0xFFFF), iLoopCtr);
   
	return(iErrors);
} // end CSIO4B_TestDlg::GscSio4_RegisterTest(int boardIndex, int channelIndex)


//---------------------------------------------------------
// Test Register Access
//---------------------------------------------------------
int TestLocalReg(int boardNumber, int iChan, int iReg, int iMask, int iLoop)
{
	int iTestVal, iOrigVal, iRdVal;
	int i, j, iErr = 0, status;
	CString csErrorString;

	// save original value off
	status = GscSio4LocalRegisterRead(boardNumber, iReg, &iOrigVal);
	if(status)
		return(status);

	for (i=0; i<iLoop;i++) 
	{
		for (j=0; j<32; j++) 
		{
			iTestVal = (iOrigVal & ~iMask) 
				| ((0x55555555l << j) & iMask);
			// write test value and verify			
			status = GscSio4LocalRegisterWrite(boardNumber, iReg, iTestVal);
			if(status)
				return(status);
			status = GscSio4LocalRegisterRead(boardNumber, iReg, &iRdVal);
			if(status)
				return(status);
		
			if ((iRdVal & iMask) != (iTestVal & iMask))
			{
				iErr++;
//				if(iErr == 5)
//				{
//					csErrorString.Format("iRdVal = %X  iTestVal = %X  iMask = %X  reg = %X", 
//										(iRdVal & iMask),
//										(iTestVal & iMask), 
//										iMask, 
//										iReg);
//					AfxMessageBox( csErrorString);
//				}
			}
		}
	 
	}

    // set register back to original value
	status = GscSio4LocalRegisterWrite(boardNumber, iReg, iOrigVal);
	if(status)
		return(status);
	
	return(iErr);
} // end TestLocalReg();


//-----------------------------------------------------------------------
//
//	-SYNC Specific Tests:
//
//-----------------------------------------------------------------------

//---------------------------------------------------------
//	-SYNC Register Tests
//---------------------------------------------------------
int GscSio4_RegisterTestSYNC(int boardIndex, int channelIndex)
{
	int iErrors, iRdValue, status, iLoopCtr;
	iErrors = 0;
	iLoopCtr = 100;
	CString string;

	// Testing BOARD_CTRL_REG
	iErrors += TestLocalReg(boardIndex, channelIndex, BOARD_CONTROL_REG,0x0007l,iLoopCtr);

	// Testing TX_ALMOST
	iErrors += TestLocalReg(boardIndex, channelIndex, (TX_ALMOST_BASE_REG + ((channelIndex - 1) << 4) ),0x7FFF7FFFl,iLoopCtr);

	// Testing RX_ALMOST
	iErrors += TestLocalReg(boardIndex, channelIndex, (RX_ALMOST_BASE_REG + ((channelIndex - 1) << 4) ),0x7FFF7FFFl,iLoopCtr);

	// Testing CONTROL_STATUS_BASE_REG
	iErrors += TestLocalReg(boardIndex, channelIndex, (CONTROL_STATUS_BASE_REG + ((channelIndex - 1) << 4) ),0x1F000000l,iLoopCtr);
	

	// Testing INT_CTRL_REG
	// read int status, invert, pass as mask to Local_Reg_Test
	status = GscSio4LocalRegisterRead(boardIndex, INTERRUPT_STATUS_REG, &iRdValue);
	if(status)
		return(status);
	iRdValue = ~iRdValue;
	iErrors += TestLocalReg(boardIndex, channelIndex, INTERRUPT_CONTROL_REG, iRdValue, iLoopCtr);

	//	Testing PIN_SOURCE_BASE_REG	
	iErrors += TestLocalReg(boardIndex, channelIndex, (PIN_SOURCE_BASE_REG + ((channelIndex - 1) * 4) ),0x7Fl,iLoopCtr);


	return(iErrors);
} // end CSIO4B_TestDlg::GscSio4_RegisterTest(int boardIndex, int channelIndex)

//---------------------------------------------------------
//	-SYNC Loopback Test
//---------------------------------------------------------
int CSIO4B_TestDlg::GscSio4_IntLoopbackTestSYNC(int boardIndex, int channelIndex)
{
	int status = 0x00, index, iErrors;
	int iBytesXfered;
	GSC_SYNC_CONFIG syncConfig;

	#define TEST_BUFFER_SIZE	128			// Size of test transfer
	char TxBuffer[ TEST_BUFFER_SIZE];		// Buffer to hold data to transmit
	char RxBuffer[ TEST_BUFFER_SIZE];		// Buffer to hold received data
	CString str;

	// These variables are used to track the data transfers
	int txId, rxId;

	CListBox *pMainListBox = (CListBox *)GetDlgItem( IDC_MAIN_LIST);

	iErrors = 0;

	// Zero out the Rx buffer
	memset( RxBuffer, 0x00, sizeof(RxBuffer));

	// Fill the Tx buffer with pattern 
	for (index=0; index < TEST_BUFFER_SIZE; index+=4)
	{
		*(TxBuffer + index)		= (char)0xaa;
		*(TxBuffer + (index+1))	= (char)0x55;
		*(TxBuffer + (index+2)) = (char)(index>>10);	// Upper half of index/4
		*(TxBuffer + (index+3)) = (char)(index>>2);		// Lower half of index/4
	}

	// Reset the desired channel. This puts the channel into a known state.
	status = GscSio4ChannelReset( boardIndex, channelIndex);
	if (status)	DisplayErrorMessage(status);

	//	Get default values and set both channels:
	status = GscSio4SyncGetDefaults( &syncConfig );
	if(status) DisplayErrorMessage( status );

	// Channel defaults
	syncConfig.bitRate =			115200;
	syncConfig.encoding =			GSC_ENCODING_NRZ;
	syncConfig.protocol =			GSC_PROTOCOL_RS422_RS485;
	syncConfig.termination =		GSC_TERMINATION_ENABLED;

	// Transmitter defaults
	syncConfig.txStatus =			GSC_ENABLED;
	syncConfig.txCharacterLength =	8;
	syncConfig.txGapLength =		0;
	syncConfig.txClockSource =		GSC_CLOCK_INTERNAL;
	syncConfig.txClockEdge =		GSC_RISING_EDGE;
	syncConfig.txEnvPolarity =		GSC_HIGH_TRUE;
	syncConfig.txIdleCondition =	GSC_IDLE_ALL_0;
	syncConfig.txClockIdleCondition = GSC_IDLE_ACTIVE;
	syncConfig.txMsbLsb =			GSC_MSB_FIRST;
		
	// Receiver defaults
	syncConfig.rxStatus =			GSC_ENABLED;
	syncConfig.rxClockSource =		GSC_CLOCK_EXTERNAL;
	syncConfig.rxClockEdge =		GSC_FALLING_EDGE;
	syncConfig.rxEnvPolarity =		GSC_HIGH_TRUE;
	syncConfig.rxMsbLsb =			GSC_MSB_FIRST;

	// Pin Configuration Variables
	syncConfig.interfaceMode =		GSC_PIN_DTE;
	syncConfig.txDataPinMode =		GSC_PIN_AUTO;
	syncConfig.rxDataPinMode =		GSC_PIN_AUTO;
	syncConfig.txClockPinMode =		GSC_PIN_AUTO;
	syncConfig.rxClockPinMode =		GSC_PIN_AUTO;
	syncConfig.txEnvPinMode =		GSC_PIN_AUTO;
	syncConfig.rxEnvPinMode =		GSC_PIN_AUTO;
	syncConfig.loopbackMode =		GSC_LOOP_EXTERNAL;

	// Setup the Misc defaults
	syncConfig.packetFraming =		GSC_DISABLED;

	//	set channel sync config:
	status = GscSio4SyncSetConfig(boardIndex, channelIndex, syncConfig);
	if(status) DisplayErrorMessage( status );


	// Start processing any recieved data
  	status = GscSio4ChannelReceiveData( boardIndex, channelIndex, RxBuffer, TEST_BUFFER_SIZE, &rxId);
	if (status)	DisplayErrorMessage(status);

	// Initiate the transfer of data
	status = GscSio4ChannelTransmitData( boardIndex, channelIndex, TxBuffer, TEST_BUFFER_SIZE, &txId);
	if (status)	DisplayErrorMessage(status);


	// Wait up to 1 seconds for all of the data to arrive. The last parameter, rxId, 
	// informs the routine of the transfer to wait on. This variable was set during
	// the call to GscSio4ReceiveData(...).
	status = GscSio4ChannelWaitForTransfer( boardIndex, channelIndex, 1000, rxId, &iBytesXfered);
	if (status)	DisplayErrorMessage(status);

	// Verify that the buffers match
	for (index = 0; index < TEST_BUFFER_SIZE; index++)
	{
		if (*(TxBuffer + index) != *(RxBuffer + index)) 
		{
			iErrors++;
			str.Format("Channel %d Data Mismatch at Location 0x%04X\n(Wrote: 0x%02X, Read: 0x%02X)",
				channelIndex,
				index,
				TxBuffer[ index] & 0xff,
				RxBuffer[ index] & 0xff);
			MessageBox(str, "Data Error Detected");
			break;			
		}
	}

	return(iErrors);
}// end int CSIO4B_TestDlg::GscSio4_IntLoopbackTest()


